/**
 * Editor Script for Greyd Button Blocks.
 * 
 * This file is loaded in block editor pages and modifies the editor experience.
 */
( function( wp ) {

	const {
		createElement: el
	} = wp.element;

	const { __ } = wp.i18n;

	const buttonStyles = [
		{
			name: 'prim',
			label: __( 'Primärer Button', 'greyd_blocks' ),
			isDefault: true
		},
		{
			name: 'sec',
			label: __( 'Sekundärer Button', 'greyd_blocks' )
		},
		{
			name: 'trd',
			label: __( 'Alternativer Button', 'greyd_blocks' )
		},
		...(
			greyd.data?.is_greyd_classic
			? [
				{
					name: 'link-prim',
					label: __( 'Link', 'greyd_blocks' )
				},
				{
					name: 'link-sec',
					label: __( 'Sekundärer Link', 'greyd_blocks' )
				}
			]
			: [
				{
					name: 'clear',
					label: __( 'Clear', 'greyd_blocks' )
				}
			]
		)
	];

	// ready/init
	wp.domReady(function () {

		if ( greyd.data?.is_greyd_classic ) {
			buttonStyles.forEach( style => {
				style && wp.blocks.registerBlockStyle( 'core/button', style );
			} )
			wp.blocks.unregisterBlockStyle( 'core/button', 'fill' );
			wp.blocks.unregisterBlockStyle( 'core/button', 'outline' );
		}
	});

	/**
	 * Additional tools/helper functions.
	 * moved from greyd.tools
	 */
	greyd.tools.button = new function() {

	};

	/**
	 * Register Buttons Block (wrapper).
	 */
	wp.blocks.registerBlockType( 'greyd/buttons', {
		title: __('GREYD Buttons', 'greyd_blocks'),
		description: __('Eine Gruppe von Buttons oder Links', 'greyd_blocks'),
		icon: greyd.tools.getBlockIcon('buttons'),
		category: 'greyd-blocks',
		keywords: [ 'button', 'link', 'trigger', 'toggle', 'button', 'btn' ],
		supports: {
			anchor: true,
			align: true,
			spacing: {
				margin: true,
				padding: true
			}
		},
		example: {
			attributes: {
				align: 'center',
			},
			innerBlocks: [
				{
					name: 'greyd/button',
					attributes: {
						content: __( 'Button', 'core' ),
					},
				},
			]
		},
		attributes: {
			align: { type: 'string', default: "" },
		},

		edit: function( props ) {
			return el( 'div', { 
				id: props.attributes.anchor, 
				className: props.className 
			}, el( wp.blockEditor.InnerBlocks, {
				template: [[ 'greyd/button', {} ]],
				// templateLock: false,
				allowedBlocks: [ 'greyd/button' ],
				orientation: 'horizontal'
			} ) );
		},
		save: function( props ) {
			return el( 'div', { 
				id: props.attributes.anchor, 
				className: props.className 
			}, el( wp.blockEditor.InnerBlocks.Content ) );
		},

		transforms: {
			from: [
				{
					type: 'block',
					blocks: [ 'core/buttons' ],
					transform: function ( attributes, innerBlocks ) {
						console.log('convert core buttons to greyd buttons');
						// console.log(attributes);
						// console.log(innerBlocks);
						
						var inner = [];
						for (var i=0; i<innerBlocks.length; i++) {
							// console.log(innerBlocks[i]);
							var inneratts = {};
							if (_.has(innerBlocks[i], 'attributes')) {
								inneratts = greyd.tools.transformDefaultAtts(innerBlocks[i].attributes);
								if (_.has(innerBlocks[i].attributes, 'text')) inneratts.content = innerBlocks[i].attributes.text;
								if (_.has(innerBlocks[i].attributes, 'size')) inneratts.size = innerBlocks[i].attributes.size;
								var icon = { content: '', position: 'after', margin: '10px', size: '100%' };
								if (_.has(innerBlocks[i].attributes, 'icon')) icon.content = innerBlocks[i].attributes.icon;
								if (_.has(innerBlocks[i].attributes, 'icon_align') && innerBlocks[i].attributes.icon_align == 'left') icon.position = 'before';
								if (_.has(innerBlocks[i].attributes, 'icon_margin')) icon.margin = innerBlocks[i].attributes.icon_margin;
								if (_.has(innerBlocks[i].attributes, 'icon_size')) icon.size = innerBlocks[i].attributes.icon_size;
								if (!_.isEmpty(icon)) inneratts.icon = icon;
								if (_.has(innerBlocks[i].attributes, 'url')) {
									var trigger = {
										type: 'link',
										params: { url: innerBlocks[i].attributes.url}
									};
									if (_.has(innerBlocks[i].attributes, 'linkTarget') && innerBlocks[i].attributes.linkTarget == '_blank')
										trigger.params.opensInNewTab = true;
									inneratts.trigger = trigger;
								}

							}
							// console.log(inneratts);
							inner.push(wp.blocks.createBlock(
								'greyd/button',
								inneratts
							));
						}

						return wp.blocks.createBlock(
							'greyd/buttons',
							attributes,
							inner
						);
					},
				},
				{
					type: 'block',
					blocks: [ 'core/heading', 'core/paragraph' ],
					transform: function ( attributes, innerBlocks ) {
						console.log('convert heading/paragraph to greyd buttons');
						// console.log(attributes);
						// console.log(innerBlocks);
						
						var newatts = greyd.tools.transformDefaultAtts(attributes);
						var inner = [];
						var inneratts = {};
						if (_.has(attributes, 'content')) {
							inneratts.content = attributes.content;
						}

						// console.log(inneratts);
						inner.push(wp.blocks.createBlock(
							'greyd/button',
							inneratts
						));

						return wp.blocks.createBlock(
							'greyd/buttons',
							newatts,
							inner
						);
					},
				}

			],
			to: [
				{
					type: 'block',
					blocks: [ 'core/buttons' ],
					transform: function ( attributes, innerBlocks ) {
						console.log('convert greyd buttons to core buttons');
						// console.log(attributes);
						// console.log(innerBlocks);

						var inner = [];
						for (var i=0; i<innerBlocks.length; i++) {
							// console.log(innerBlocks[i]);
							var inneratts = {};
							if (_.has(innerBlocks[i], 'attributes')) {
								inneratts = greyd.tools.transformDefaultAtts(innerBlocks[i].attributes);
								if (_.has(innerBlocks[i].attributes, 'content')) inneratts.text = innerBlocks[i].attributes.content;
								if (_.has(innerBlocks[i].attributes, 'size')) inneratts.size = innerBlocks[i].attributes.size;
								if (_.has(innerBlocks[i].attributes, 'icon')) {
									inneratts.icon = innerBlocks[i].attributes.icon.content;
									inneratts.icon_align = innerBlocks[i].attributes.icon.position == 'before' ? 'left' : '';
									inneratts.icon_margin = innerBlocks[i].attributes.icon.margin;
									inneratts.icon_size = innerBlocks[i].attributes.icon.size;
								}
							}
							// console.log(inneratts);
							inner.push(wp.blocks.createBlock(
								'core/button',
								inneratts
							))
						}

						return wp.blocks.createBlock(
							'core/buttons',
							attributes,
							inner
						);
					},
				}

			]
		}
	} );


	/**
	 * Register Button Block.
	 */
	wp.blocks.registerBlockType( 'greyd/button', {
		title: __('GREYD Button', 'greyd_blocks'),
		description: __('Button/Link mit Trigger Picker', 'greyd_blocks'),
		icon: greyd.tools.getBlockIcon('button'),
		category: 'greyd-blocks',
		parent: [ 'greyd/buttons' ],
		keywords: [ 'link', 'trigger' ],
		styles: buttonStyles,
		supports: {
			anchor: true,
			defaultStylePicker: false
			// color: {
			// 	text: true,
			// 	background: false,
			// 	gradients: false,
			// },
		},
		example: {
			attributes: {
				content: 'Button',
				icon: {
					content: 'arrow_right',
					position: 'after',
					size: '100%',
					margin: '10px'
				}
			},
		},

		attributes: {
			inline_css: { type: 'string' },
			inline_css_id: { type: 'string' },
			greydClass: { type: 'string', default: '' },
			greydStyles: { type: 'object' },
			customStyles: { type: 'object' },
			trigger: { type: 'object' },
			size: { type: 'string', default: '' },
			content: { type: 'string' },
			icon: { type: 'object', properties: {
				content: { type: "string" },
				position: { type: "string" },
				size: { type: "string" },
				margin: { type: "string" },
			}, default: {
				content: '',
				position: 'after',
				size: '100%',
				margin: '10px'
			} },
			custom: { type: 'bool', default: 0 }
		},

		edit: function( props ) {

			props.attributes.greydClass = greyd.tools.getGreydClass(props)

			const extraClass = props.className.indexOf('is-style-link-') === -1 ? 'button' : 'link';

			var classNames = [ extraClass, props.className, props.attributes.greydClass, props.attributes.size ].join(' ');
			var blockProps = { id: props.attributes.anchor, className: classNames };

			// call function to make sure Block is updated when inside a template
			var bp = wp.blockEditor.useBlockProps();

			return [

				// sidebar - settings
				el( wp.blockEditor.InspectorControls, { group: 'settings' }, [
					// icon
					el( greyd.components.ButtonIconControl, {
						value: props.attributes.icon,
						onChange: function(value) {
							props.setAttributes({ icon: value });
						},
					} ),
				] ),

				// sidebar - styles
				el( wp.blockEditor.InspectorControls, { group: 'styles' }, [
					// size
					el( greyd.components.AdvancedPanelBody, {
						title: __( 'Größe', 'greyd_blocks' ),
						holdsChange: !_.isEmpty(props.attributes.size)
					},
						[
							el( greyd.components.ButtonGroupControl, {
								value: props.attributes.size,
								// label: __( 'Größe', 'greyd_blocks' ),
								options: [
									{ value: "is-size-small", label: __( 'klein', 'greyd_blocks' ) },
									{ value: "", label: __( 'normal', 'greyd_blocks' ) },
									{ value: "is-size-big", label: __( 'groß', 'greyd_blocks' ) },
								],
								onChange: function(value) {
									props.setAttributes( { size: value } );
								},
							} ),
						]
					),
					// width
					el( greyd.components.StylingControlPanel, {
						title: __('Breite', 'greyd_blocks'),
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __('Breite', 'greyd_blocks'),
								attribute: "width",
								control: greyd.components.RangeUnitControl,
								max: 500
							}
						]
					} ),

					// custom button
					el( greyd.components.AdvancedPanelBody, {
						title: __( 'Individueller Button', 'greyd_blocks' ),
						initialOpen: true,
						holdsChange: props.attributes.custom ? true : false
					},
						[
							el( wp.components.ToggleControl, {
								label: __( 'Design des Buttons individuell überschreiben', 'greyd_blocks' ),
								checked: props.attributes.custom,
								onChange: function(value) {
									props.setAttributes( { custom: !!value } );
								},
							} ),
						]
					),
					el( greyd.components.CustomButtonStyles, {
						enabled: props.attributes.custom ? true : false,
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// preview
				el( 'div', { ...blockProps }, [
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'before'
					} ),
					el( wp.blockEditor.RichText, {
						format: 'string',
						tagName: 'span',
						style: { flex: '1' },
						value: props.attributes.content,
						placeholder: __( 'Button', 'greyd_blocks' ),
						allowedFormats: [ 'core/bold', 'core/italic', 'core/strikethrough', 'greyd/dtag', 'core/highlight' ],
						onChange: function(value) {
							props.setAttributes( { content: value } );
						},
					} ),
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'after'
					} ),
				] ),
				// normal styles
				el( greyd.components.RenderPreviewStyles, {
					selector: props.attributes.greydClass,
					styles: {
						"": props.attributes.greydStyles,
					}
				} ),
				// custom styles
				!props.attributes.custom ? null : el( greyd.components.RenderPreviewStyles, {
					selector: props.attributes.greydClass,
					styles: {
						"": props.attributes.customStyles,
					},
					important: true
				} )
			];
		},
		save: function( props ) {

			const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';

			return el( wp.element.Fragment, {}, [
				el( 'a', {
					id: props.attributes.anchor,
					role: "trigger", // we replace that trigger-placeholder via render.php
					className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
				}, [
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'before'
					} ),
					(
						_.isEmpty( props.attributes.icon.content )
						? el( wp.blockEditor.RichText.Content, {
							tagName: null,
							value: props.attributes.content
						} )
						: el( wp.blockEditor.RichText.Content, {
							tagName: 'span',
							style: { flex: '1' },
							value: props.attributes.content
						} )
					),
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'after'
					} ),
				] ),
				!props.attributes.custom ? null : el( greyd.components.RenderSavedStyles, {
					selector: props.attributes.greydClass,
					styles: {
						"": props.attributes.customStyles,
					},
					important: true
				} )
			] );
		},

		deprecated: [
			/**
			 * Do not render empty span when content is empty.
			 * @since 1.5.1
			 */
			{
				attributes: {
					dynamic_parent: { type: 'string' }, // dynamic template backend helper
					dynamic_value: { type: 'string' }, // dynamic template frontend helper
					dynamic_fields: { type: 'array' },
					inline_css: { type: 'string' },
					inline_css_id: { type: 'string' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object' },
					customStyles: { type: 'object' },
					trigger: { type: 'object' },
					size: { type: 'string', default: '' },
					content: { type: 'string' },
					icon: { type: 'object', properties: {
						content: { type: "string" },
						position: { type: "string" },
						size: { type: "string" },
						margin: { type: "string" },
					}, default: {
						content: '',
						position: 'after',
						size: '100%',
						margin: '10px'
					} },
					custom: { type: 'bool', default: 0 }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function( props ) {

					const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';
		
					return el( wp.element.Fragment, {}, [
						el( 'a', {
							id: props.attributes.anchor,
							role: "trigger", // we replace that trigger-placeholder via render.php
							className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: props.attributes.content }
							} ),
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'after'
							} ),
						] ),
						!props.attributes.custom ? null : el( greyd.components.RenderSavedStyles, {
							selector: props.attributes.greydClass,
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} )
					] );
				}
			},
			/**
			 * RenderSavedStylesDeprecated
			 * @deprecated since 1.3.7
			 */
			{
				attributes: {
					dynamic_parent: { type: 'string' }, // dynamic template backend helper
					dynamic_value: { type: 'string' }, // dynamic template frontend helper
					dynamic_fields: { type: 'array' },
					inline_css: { type: 'string' },
					inline_css_id: { type: 'string' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object' },
					customStyles: { type: 'object' },
					trigger: { type: 'object' },
					size: { type: 'string', default: '' },
					content: { type: 'string' },
					icon: { type: 'object', properties: {
						content: { type: "string" },
						position: { type: "string" },
						size: { type: "string" },
						margin: { type: "string" },
					}, default: {
						content: '',
						position: 'after',
						size: '100%',
						margin: '10px'
					} },
					custom: { type: 'bool', default: 0 }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function( props ) {

					const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';
		
					return el( wp.element.Fragment, {}, [
						el( 'a', {
							id: props.attributes.anchor,
							role: "trigger", // we replace that trigger-placeholder via render.php
							className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: props.attributes.content }
							} ),
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'after'
							} ),
						] ),
						!props.attributes.custom ? null : el( greyd.components.RenderSavedStylesDeprecated, {
							selector: props.attributes.greydClass,
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} )
					] );
				}
			},
			/**
			 * trigger="placeholder"
			 * @deprecated since 1.3.3
			 */
			{
				attributes: {
					dynamic_parent: { type: 'string' }, // dynamic template backend helper
					dynamic_value: { type: 'string' }, // dynamic template frontend helper
					dynamic_fields: { type: 'array' },
					inline_css: { type: 'string' },
					inline_css_id: { type: 'string' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object' },
					customStyles: { type: 'object' },
					trigger: { type: 'object' },
					size: { type: 'string', default: '' },
					content: { type: 'string' },
					icon: { type: 'object', properties: {
						content: { type: "string" },
						position: { type: "string" },
						size: { type: "string" },
						margin: { type: "string" },
					}, default: {
						content: '',
						position: 'after',
						size: '100%',
						margin: '10px'
					} },
					custom: { type: 'bool', default: 0 }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function( props ) {

					const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';
		
					return el( wp.element.Fragment, {}, [
						el( 'a', {
							id: props.attributes.anchor,
							trigger: "placeholder", // we replace that trigger-placeholder via render.php
							className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: props.attributes.content }
							} ),
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'after'
							} ),
						] ),
						!props.attributes.custom ? null : el( greyd.components.RenderSavedStylesDeprecated, {
							selector: props.attributes.greydClass,
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} )
					] );
				}
			},
			/**
			 * RenderSavedStylesDeprecated
			 * @deprecated since 1.1.2
			 */
			{
				attributes: {
					dynamic_parent: { type: 'string' }, // dynamic template backend helper
					dynamic_value: { type: 'string' }, // dynamic template frontend helper
					dynamic_fields: { type: 'array' },
					inline_css: { type: 'string' },
					inline_css_id: { type: 'string' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object' },
					customStyles: { type: 'object' },
					trigger: { type: 'object' },
					size: { type: 'string', default: '' },
					content: { type: 'string' },
					icon: { type: 'object', properties: {
						content: { type: "string" },
						position: { type: "string" },
						size: { type: "string" },
						margin: { type: "string" },
					}, default: {
						content: '',
						position: 'after',
						size: '100%',
						margin: '10px'
					} },
					custom: { type: 'bool', default: 0 }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function( props ) {

					const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';
		
					return el( wp.element.Fragment, {}, [
						el( 'a', {
							id: props.attributes.anchor,
							trigger: "placeholder",
							className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: props.attributes.content }
							} ),
							el( greyd.components.RenderButtonIcon, {
								value: props.attributes.icon,
								position: 'after'
							} ),
						] ),
						!props.attributes.custom ? null : el( greyd.components.RenderSavedStylesDeprecated, {
							selector: props.attributes.greydClass,
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} )
					] );
				}
			},
			/**
			 * RenderButtonIconDeprecated
			 * @deprecated
			 */
			{
				attributes: {
					dynamic_parent: { type: 'string' }, // dynamic template backend helper
					dynamic_value: { type: 'string' }, // dynamic template frontend helper
					dynamic_fields: { type: 'array' },
					inline_css: { type: 'string' },
					inline_css_id: { type: 'string' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object' },
					customStyles: { type: 'object' },
					trigger: { type: 'object' },
					size: { type: 'string', default: '' },
					content: { type: 'string' },
					icon: { type: 'object', properties: {
						content: { type: "string" },
						position: { type: "string" },
						size: { type: "string" },
						margin: { type: "string" },
					}, default: {
						content: '',
						position: 'after',
						size: '100%',
						margin: '10px'
					} },
					custom: { type: 'bool', default: 0 }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function( props ) {
					const extraClass = _.has(props.attributes, "className") && props.attributes.className.indexOf('is-style-link-') > -1 ? 'link' : 'button';
		
					return el( wp.element.Fragment, {}, [
						el( 'a', {
							id: props.attributes.anchor,
							trigger: "placeholder", // we replace that trigger-placeholder via render.php
							className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size ].join(' ')
						}, [
							el( greyd.components.RenderButtonIconDeprecated, {
								value: props.attributes.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: props.attributes.content }
							} ),
							el( greyd.components.RenderButtonIconDeprecated, {
								value: props.attributes.icon,
								position: 'after'
							} ),
						] ),
						!props.attributes.custom ? null : el( greyd.components.RenderSavedStylesDeprecated, {
							selector: props.attributes.greydClass,
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} )
					] );
				},

			},
		]
	} );


	/**
	 * Register custom attributes to core blocks.
	 * - core/button
	 *
	 * @hook blocks.registerBlockType
	 */
	var registerBlockTypeHook = function(settings, name) {

		if (_.has(settings, 'apiVersion') && settings.apiVersion > 1) {
			// console.log(name);
			// console.log(settings);

			if (name == 'core/button') {
				delete settings.attributes.width;
				settings.attributes.size = { type: 'string', default: '' };
				settings.attributes.min_width = { type: 'string', default: '' };
				// settings.attributes.versal = { type: 'boolean', default: false };
				settings.attributes.icon = { type: 'string', default: '' };
				settings.attributes.icon_align = { type: 'string', default: '' };
				settings.attributes.icon_margin = { type: 'string', default: '10px' };
				settings.attributes.icon_size = { type: 'string', default: '100%' };
				// settings.supports.align = false;
				// console.log(settings);
			}

		}
		return settings;

	}

	wp.hooks.addFilter(
		'blocks.registerBlockType',
		'greyd/hook/button',
		registerBlockTypeHook
	);


	/**
	 * Add custom edit controls to core Blocks.
	 *
	 * @hook editor.BlockEdit
	 */
	var editBlockHook = wp.compose.createHigherOrderComponent( function( BlockEdit ) {

		return function( props ) {

			/**
			 * Extend the core button
			 */
			if (props.name == "core/button") {
				// console.log("add size support to: "+props.name);

				var icon_hidden = "";
				if (props.attributes.icon == "") icon_hidden = " hidden";

				const makeInspectorControls = function() {
					return [
				
						el( wp.components.PanelBody, {
							title: __('Design', 'greyd_blocks'),
							initialOpen: true
						}, [
							el( 'div', {}, [
								el( greyd.components.ButtonGroupControl, {
									label: __('Größe', 'greyd_blocks'),
									value: props.attributes.size,
									options: [
										{ label: __('normal', 'greyd_blocks'), value: 'is-size-normal' },
										{ label: __('groß', 'greyd_blocks'), value: 'is-size-big' },
										{ label: __('klein', 'greyd_blocks'), value: 'is-size-small' },
									],
									onChange: function(value) { 

										// onClear
										if ( isEmpty(value) ) {
											if ( !_.has(props.attributes, 'className') || _.isEmpty(props.attributes.className)) return;
	
											// remove 'is-size-big', 'is-size-small' and 'is-size-normal' from className
											var classNames = props.attributes.className.split(/is-size-big\s*|is-size-small\s*|is-size-normal\s*/g);
											// clean
											classNames = greyd.tools.cleanClassArray(classNames);
											// console.log(classNames);
											props.setAttributes( { size: '', className: classNames.join(' ') } );
										}

										// onChange
										else {
											var classNames = [ value ];
											// saved className
											if (_.has(props.attributes, 'className') && !_.isEmpty(props.attributes.className)) {
												// remove 'is-size-big', 'is-size-small' and 'is-size-normal'
												var oldClasses = props.attributes.className.split(/is-size-big\s*|is-size-small\s*|is-size-normal\s*/g);
												// add all other
												classNames.push( ...oldClasses );
											}
											// clean
											classNames = greyd.tools.cleanClassArray(classNames);
											// console.log(classNames);
											props.setAttributes( { size: value, className: classNames.join(' ') } ); 
										}
									}
								} ),
								el( wp.components.__experimentalUnitControl, {
									label: __('Breite', 'greyd_blocks'),
									className: 'is-edge-layout',
									value: props.attributes.min_width,
									onChange: function(value) { 
										var css_id = (_.has(props.attributes, 'anchor') && props.attributes.anchor != "") ? props.attributes.anchor : 'block-'+props.clientId;
										props.setAttributes( { min_width: value, inline_css_id: css_id } ); 
									},
								} ),
							] ),
						] ),
						el( wp.components.PanelBody, {
							title: __('Icon', 'greyd_blocks'),
							initialOpen: true
						}, [
							el( greyd.components.IconPicker, {
								value: props.attributes.icon,
								// icons: greyd.data.icons,
								onChange: function(value) { 
									var css_id = (_.has(props.attributes, 'anchor') && props.attributes.anchor != "") ? props.attributes.anchor : 'block-'+props.clientId;
									props.setAttributes( { icon: value, inline_css_id: css_id } ); 
								},
							} ),
							el( 'div', { className: "greyd-inspector-wrapper greyd-3"+icon_hidden }, [
								el( greyd.components.ButtonGroupControl, {
									label: __('Position', 'greyd_blocks'),
									value: props.attributes.icon_align,
									options: [
										{ label: __('rechts', 'greyd_blocks'), value: '' },
										{ label: __('links', 'greyd_blocks'), value: 'left' },
									],
									onChange: function(value) { props.setAttributes( { icon_align: value } ); },
								} ),
								el( wp.components.__experimentalUnitControl, {
									label: __('Abstand', 'greyd_blocks'),
									className: 'is-edge-layout',
									value: props.attributes.icon_margin,
									onChange: function(value) { props.setAttributes( { icon_margin: value } ); },
								} ),
								el( wp.components.__experimentalUnitControl, {
									label: __('Größe', 'greyd_blocks'),
									className: 'is-edge-layout',
									value: props.attributes.icon_size,
									onChange: function(value) { props.setAttributes( { icon_size: value } ); },
								} ),
							] ),
						] )
					]
				};
				
				var style = "";
				if (_.has(props.attributes, 'min_width') && props.attributes.min_width != "") {
					var min_width = props.attributes.min_width;
					style += "#block-"+props.clientId+" { width: "+min_width+"; } ";
					style += "#block-"+props.clientId+" > div { width: 100%; } ";
				}
				if (_.has(props.attributes, 'icon') && props.attributes.icon != "") {
					var pos = "after";
					var margin = "left";
					if (_.has(props.attributes, 'icon_align') && props.attributes.icon_align == 'left') {
						pos = "before";
						margin = "right";
					}
					var icon_margin = _.has(props.attributes, 'icon_margin') ? props.attributes.icon_margin : "10px";
					var icon_size = _.has(props.attributes, 'icon_size') ? props.attributes.icon_size : "100%";
					style += "#block-"+props.clientId+" > div:"+pos+" { "+
								"content: '"+greyd.data.icons[props.attributes.icon]['content']+"'; "+
								"font-family: ElegantIcons; "+
								"vertical-align: middle; "+
								"margin-"+margin+": "+icon_margin+"; "+
								"font-size: "+icon_size+"; "+
							"} ";
				}
				
				return el( wp.element.Fragment, { }, [
					// inspector
					el( wp.blockEditor.InspectorControls, { }, [
						// design and icon
						makeInspectorControls()
					] ),
					// style
					(style != "") ? el( 'style', { className: 'greyd_styles' }, style ) : null,
					// original block
					el( BlockEdit, props )
				] );
			}
			
			// return original block
			return el( BlockEdit, props );
		};

	}, 'editBlockHook' );

	wp.hooks.addFilter(
		'editor.BlockEdit',
		'greyd/hook/button/edit',
		editBlockHook
	);

} )( 
	window.wp
);