/**
 * Navigation Extension Editor Script
 * - core/navigation
 * - core/navigation-submenu
 * - core/navigation-link
 * 
 * @since 1.13.0
 * 
 * This file is loaded in block editor pages and modifies the editor experience.
 */
( function( wp ) {

	const {
		createElement: el
	} = wp.element;

	const { __ } = wp.i18n;


	/**
	 * Register custom attributes to core navigation blocks.
	 *
	 * @hook blocks.registerBlockType
	 */
	var registerBlockTypeHook = function(settings, name) {

		if (_.has(settings, 'apiVersion') && settings.apiVersion > 1) {
			// console.log(name);

			if (name == 'core/navigation' ||
				name == 'core/navigation-submenu' ||
				name == 'core/navigation-link') {
				// console.log(settings);
				settings.attributes.custom = { type: 'bool', default: 0 };
				settings.attributes.greydClass = { type: 'string' };
				settings.attributes.customStyles = { type: 'object' };
			}
			if (name == 'core/navigation') {
				// console.log(settings);
				settings.attributes.submenu = { type: 'string', default: '' };
				// settings.attributes.anchoractive = { type: 'bool', default: 0 };
				settings.attributes.anchoractive = { type: 'object', default: {
					enable: 0,
					start: '0%',
					end: '100%',
					multiple: '',
					none: ''
				} };
			}

		}
		return settings;

	}

	wp.hooks.addFilter(
		'blocks.registerBlockType',
		'greyd/hook/navigation',
		registerBlockTypeHook
	);


	/**
	 * Add custom edit controls to core navigation blocks.
	 *
	 * @hook editor.BlockEdit
	 */
	var editBlockHook = wp.compose.createHigherOrderComponent( function( BlockEdit ) {

		return function( props ) {

			/**
			 * Extend core navigation
			 */
			if (props.name == 'core/navigation' ||
				props.name == 'core/navigation-submenu' ||
				props.name == 'core/navigation-link') {

				var extend = true;
				// check site-editor screen
				var urlParams = new URLSearchParams(window.location.search);
				if (
					props.name == 'core/navigation' &&
					urlParams.has('postType') &&
					urlParams.get('postType') == 'wp_navigation'
				) extend = false;

				// don't extend on wp_navigation edit screen
				// core/navigation wrapper block is not saved
				if ( extend ) {

					const makeInspectorControlsStyles = () => {

						// add custom styles component
						return el( wp.blockEditor.InspectorControls, { group: 'styles' }, [
							
							// custom button
							el( greyd.components.AdvancedPanelBody, {
								title: __( 'Individuelles Design', 'greyd_blocks' ),
								initialOpen: true,
								holdsChange: props.attributes.custom ? true : false
							}, [
								el( wp.components.ToggleControl, {
									label: __( 'Design individuell überschreiben', 'greyd_blocks' ),
									checked: props.attributes.custom,
									onChange: function(value) {
										props.setAttributes( { custom: !!value } );
									},
								} ),
							] ),
							el( greyd.components.CustomButtonStyles, {
								enabled: props.attributes.custom ? true : false,
								blockProps: props,
								parentAttr: 'customStyles',
								supportsActive: true
							} )

						] );

					};

					const makeInspectorControls = () => {

						if ( props.name != 'core/navigation' ) return null;

						// disable openSubmenusOnClick and showSubmenuIcon when behaviour is 'show' or 'hide'
						if ( props.attributes.submenu && props.attributes.submenu != '' && props.attributes.submenu != 'toggle' ) {
							if ( props.attributes.openSubmenusOnClick ) props.setAttributes( { openSubmenusOnClick: false } );
							if ( props.attributes.showSubmenuIcon ) props.setAttributes( { showSubmenuIcon: false } );
						}

						// anchoractive
						const options = [
							{
								label: __( '0% – Element an oberer Kante', 'greyd_blocks' ),
								value: '0%'
							},
							{ label: __( '10%', 'greyd_blocks' ), value: '10%' },
							{ label: __( '20%', 'greyd_blocks' ), value: '20%' },
							{ label: __( '30%', 'greyd_blocks' ), value: '30%' },
							{ label: __( '40%', 'greyd_blocks' ), value: '40%' },
							{
								label: __( '50% – Element in der Mitte des Screens', 'greyd_blocks' ),
								value: '50%'
							},
							{ label: __( '60%', 'greyd_blocks' ), value: '60%' },
							{ label: __( '70%', 'greyd_blocks' ), value: '70%' },
							{ label: __( '80%', 'greyd_blocks' ), value: '80%' },
							{ label: __( '90%', 'greyd_blocks' ), value: '90%' },
							{
								label: __( '100% – Element an unterer Kante', 'greyd_blocks' ),
								value: '100%'
							}
						];
						const setAnchoractive = ( key, value ) => {
							var anchoractive = { ...props.attributes.anchoractive };
							anchoractive[key] = value;
							props.setAttributes( { anchoractive: anchoractive } );
						};

						return el( wp.blockEditor.InspectorControls, { }, [
							
							// add submenus behaviour
							el( greyd.components.AdvancedPanelBody, {
								title: __('Submenus', 'greyd_blocks'),
								initialOpen: true,
								holdsChange: props.attributes.submenu != ""
							}, [

								el( greyd.components.ButtonGroupControl, {
									label: __( 'Override Behaviour', 'greyd_blocks' ),
									value: props.attributes.submenu ? props.attributes.submenu : '',
									options: [
										{ value: "", label: __( 'default', 'greyd_blocks' ) },
										{ value: "toggle", label: __( 'toggle', 'greyd_blocks' ) },
										{ value: "show", label: __( 'show', 'greyd_blocks' ) },
										{ value: "hide", label: __( 'hide', 'greyd_blocks' ) },
									],
									onChange: function(value) { 
										props.setAttributes( { submenu: value } );
									},
								} ),
								el( wp.components.Notice, { 
									className: props.attributes.submenu && props.attributes.submenu != '' && props.attributes.submenu != 'toggle' ? '' : 'is-hidden',
									status: 'warning',
									isDismissible: false
								}, [
									el( 'p', {}, __( 'This option overrides the default submenu behaviours "click" and "hover".', 'greyd_blocks' ) ),
									__( 'If set, those submenu behaviours and and their options are disabled.', 'greyd_blocks' ),
								] )

							] ),

							// add active behaviour
							el( greyd.components.AdvancedPanelBody, {
								title: __('Anchor-Active-Style', 'greyd_blocks'),
								initialOpen: true,
								holdsChange: props.attributes.anchoractive?.enable ? true : false
							}, [

								el( wp.components.ToggleControl, {
									label: __('Activate', 'greyd_blocks'),
									checked: props.attributes.anchoractive?.enable ? true : false,
									onChange: (value) => setAnchoractive('enable', value),
								} ),

								props.attributes.anchoractive?.enable ? [

									// help notice
									el( BaseControl, {}, [
										el( wp.components.Notice, { 
											status: 'warning',
											isDismissible: false
										}, [
											__( 'If a menu-item points to an anchor-element of a page, it is highlighted as "active" as long as the anchor-element is in the viewport.', 'greyd_blocks' ),
											/* translation */
											// __( 'Wenn ein Menüpunkt auf einen Anker-Element einer Seite zeigt, so wird er als „active“ hervorgehoben solange sich das Anker-Element im sichtbaren Bereich befindet.', 'greyd_blocks' ),
										] )
									] ),

									// viewport
									el( BaseControl, {
										help: parseInt(props.attributes.anchoractive?.start) > parseInt(props.attributes.anchoractive?.end) ?
											__( 'Viewport Start should be smaller than End!', 'greyd_blocks' ) : '',
									}, [
										el( greyd.components.OptionsControl, {
											style: { marginBottom: '-8px' },
											label: __( 'Viewport start', 'greyd_blocks' ),
											value: props.attributes.anchoractive?.start?? '100%',
											options: options,
											onChange: (value) => setAnchoractive('start', value),
										} ),
										el( greyd.components.OptionsControl, {
											label: __( 'Viewport end', 'greyd_blocks' ),
											value: props.attributes.anchoractive?.end?? '0%',
											options: options,
											onChange: (value) => setAnchoractive('end', value),
										} )
									] ),


									// multiple/none
									el( BaseControl, {}, [
										el( wp.components.RadioControl, {
											label: __( 'Multiple active Elements', 'greyd_blocks' ),
											selected: props.attributes.anchoractive?.multiple?? '',
											options: [
												{ value: "",        label: __( 'Allow multiple active elements', 'greyd_blocks' ) },
												{ value: "closest", label: __( 'Only closest to viewport center', 'greyd_blocks' ) },
												{ value: "latest",  label: __( 'Only latest element in viewport', 'greyd_blocks' ) },
											],
											onChange: (value) => setAnchoractive('multiple', value),
										} ),
										el( wp.components.RadioControl, {
											label: __( 'No active Element', 'greyd_blocks' ),
											selected: props.attributes.anchoractive?.none?? '',
											options: [
												{ value: "",        label: __( 'Allow no active element', 'greyd_blocks' ) },
												{ value: "closest", label: __( 'Highlight closest to viewport center', 'greyd_blocks' ) },
												{ value: "keep",    label: __( 'Keep element active until next', 'greyd_blocks' ) },
											],
											onChange: (value) => setAnchoractive('none', value),
										} )
									] ),
									
								] : [],

							] ),

						] );
					};

					const makeStyles = () => {

						if ( !props.attributes.custom || JSON.stringify(props.attributes.customStyles) == '{}' ) return null;

						props.attributes.greydClass = greyd.tools.getGreydClass(props);

						// split styles (normal and hover)
						var styles = { ...props.attributes.customStyles };
						if ( styles.hover ) delete styles.hover;
						if ( styles.active ) delete styles.active;
						// make selectors
						var selectors = { " .wp-block-navigation-item__content": styles };
						if ( props.attributes.customStyles?.hover ) {
							selectors[" .wp-block-navigation-item__content:hover"] = { ...props.attributes.customStyles.hover };
							selectors["._hover .wp-block-navigation-item__content"] = { ...props.attributes.customStyles.hover };
						}
						// icon
						if ( props.name == 'core/navigation' || props.name == 'core/navigation-submenu' ) {
							if ( props.attributes.customStyles?.color ) {
								selectors[" .wp-block-navigation__submenu-icon svg"] = { stroke: props.attributes.customStyles.color };
							}
							if ( props.attributes.customStyles?.hover?.color ) {
								selectors[" .wp-block-navigation-item__content:hover + .wp-block-navigation__submenu-icon svg"] = { stroke: props.attributes.customStyles.hover.color };
								selectors["._hover .wp-block-navigation-item__content + .wp-block-navigation__submenu-icon svg"] = { stroke: props.attributes.customStyles.hover.color };
							}
						}
						// active
						if ( props.attributes.customStyles?.active ) {
							selectors["._active .wp-block-navigation-item__content"] = { ...props.attributes.customStyles.active };
							// active icon
							if ( props.attributes.customStyles?.active?.color && (props.name == 'core/navigation' || props.name == 'core/navigation-submenu') ) {
								selectors["._active .wp-block-navigation-item__content + .wp-block-navigation__submenu-icon svg"] = { stroke: props.attributes.customStyles.active.color };
							}
						}

						// style preview
						return el( greyd.components.RenderPreviewStyles, {
							selector: 'is-root-container #block-'+props.clientId,
							styles: selectors,
							important: true
						} );

					};

					return el( wp.element.Fragment, { }, [
						// style
						makeStyles(),
						// original block
						el( BlockEdit, props ),
						// inspector
						makeInspectorControls(),
						makeInspectorControlsStyles(),
					] );

				}

			}

			// return original block
			return el( BlockEdit, props );
		};

	}, 'editBlockHook' );

	wp.hooks.addFilter(
		'editor.BlockEdit',
		'greyd/hook/navigation/edit',
		editBlockHook
	);


	/**
	 * Manipulate attributes before edit.
	 * 
	 * @hook editor.BlockListBlock
	 */
	var editBlockListHook = wp.compose.createHigherOrderComponent( function ( BlockListBlock ) {
		return function ( props ) {
			// console.log(BlockListBlock);
			// console.log(props);

			/**
			 * Extend core navigation
			 */
			if ( props.name == 'core/navigation' ) {
				// get classnames
				var classNames = props.className && !_.isEmpty(props.className) ? props.className.split(' ') : [];
				// fix submenu arrows
				if ( !props.attributes.openSubmenusOnClick && classNames.indexOf('open-on-hover-click') == -1 ) {
					classNames.push( 'open-on-hover-click' );
				}
				else if ( props.attributes.openSubmenusOnClick && classNames.indexOf('open-on-hover-click') > -1 ) {
					classNames.splice(classNames.indexOf('open-on-hover-click'), 1);
				}
				if ( props.attributes.showSubmenuIcon === false && classNames.indexOf('open-on-hover-click') > -1 ) {
					classNames.splice(classNames.indexOf('open-on-hover-click'), 1);
				}
				// submenus behaviour
				if ( classNames.indexOf('submenus-show') > -1 ) classNames.splice(classNames.indexOf('submenus-show'), 1);
				if ( classNames.indexOf('submenus-toggle') > -1 ) classNames.splice(classNames.indexOf('submenus-toggle'), 1);
				if ( classNames.indexOf('submenus-hide') > -1 ) classNames.splice(classNames.indexOf('submenus-hide'), 1);
				if ( props.attributes.submenu && props.attributes.submenu != "" ) {
					if ( props.attributes.submenu == 'show' ) classNames.push( 'submenus-show' ); 
					if ( props.attributes.submenu == 'toggle' ) classNames.push( 'submenus-toggle' ); 
					if ( props.attributes.submenu == 'hide' ) classNames.push( 'submenus-hide' ); 
				}
				// set classnames
				props.className = classNames.join(' ');
			}

			return el( BlockListBlock, props );
		};
	}, 'editBlockListHook' );

	wp.hooks.addFilter( 
		'editor.BlockListBlock', 
		'greyd/hook/navigation/list', 
		editBlockListHook 
	);
	
} )( 
	window.wp
);