( function ( wp ) {

	const {
		createElement: el,
		useState
	} = wp.element;

	const { __ } = wp.i18n;

	const {
		PanelBody,
		ToggleControl,
		Button
	} = wp.components;

	const {
		RichText,
		InnerBlocks,
		InspectorControls
	} = wp.blockEditor;

	/**
	 * Register the accordion wrapper.
	 */
	wp.blocks.registerBlockType( 'greyd/tabs', {
		title: __( 'Tabs', 'greyd_blocks' ),
		description: __( 'Organisiere Inhalte in übersichtlichen Registerkarten.', 'greyd_blocks' ),
		icon: greyd.tools.getBlockIcon('tabs'),
		category: 'greyd-blocks',
		keywords: [ 'trigger', 'toggle', 'tabs', 'panelr' ],
		supports: {
			anchor: true,
			align: true,
			defaultStylePicker: false,
			spacing: {
				margin: true,
				padding: true
			}
		},
		styles: [
			{
				name: 'tabs',
				label: __( 'Klassisch', 'greyd_blocks' ),
				isDefault: true
			},
			{
				name: 'chips',
				label: __( 'Chip', 'greyd_blocks' )
			},
			{
				name: 'prim',
				label: __( 'Primary Buttons', 'greyd_blocks' )
			},
			{
				name: 'sec',
				label: __( 'Secondary Button', 'greyd_blocks' )
			},
			{
				name: 'trd',
				label: __( 'Alternative Button', 'greyd_blocks' )
			}
		],
		example: {
			attributes: {
			},
			innerBlocks: [
				{
					name: 'greyd/tab',
					attributes: {
						title: __('Tab 1', 'greyd_blocks'),
						active: true,
					},
					innerBlocks: [
						{
							name: 'core/paragraph',
							attributes: {
								content: __('Organisiere Inhalte in übersichtlichen Registerkarten und ermögliche den einfachen Wechsel zwischen verschiedenen Informationen auf deiner Webseite.', 'greyd_blocks')
							}
						},
					]
				},
				{
					name: 'greyd/tab',
					attributes: {
						title: __('Tab 2', 'greyd_blocks'),
					}
				},
				{
					name: 'greyd/tab',
					attributes: {
						title: __('Tab 3', 'greyd_blocks'),
					}
				}
			]
		},
		attributes: {
			iconPosition: { type: 'string', default: '' },
			greydClass: { type: 'string', default: '' },
			greydStyles: { type: 'object', default: {} },
			customStyles: { type: 'object', default: {} },
			transition: { type: 'string', default: '' },
			stackOnMobile: { type: 'boolen', default: false },

			activeTab: { type: 'string', default: '' }
		},

		edit: function ( props ) {

			props.attributes.greydClass = greyd.tools.getGreydClass( props );

			const {
				className,
				setAttributes,
				attributes: atts
			} = props;

			let extraClass = '';
			if ( className.indexOf( 'is-style-prim' ) > -1 || className.indexOf( 'is-style-sec' ) > -1 || className.indexOf( 'is-style-trd' ) > -1 ) {
				extraClass = 'button';
			}
			const tabClassNames = [ extraClass, atts.size, className.replace( "wp-block-greyd-tabs", "" ).trim(), 'greyd_tab' ].join( ' ' );
			const defaultClass = className.indexOf( 'is-style-' ) > -1 ? '' : 'is-style-tabs';

			const [ mode, setMode ] = useState( "" );
			if ( !props.isSelected && mode != "" ) setMode( "" );

			const container = wp.data.select( "core/block-editor" ).getBlocksByClientId( props.clientId )[ 0 ];
			const tabs = container ? container.innerBlocks : [];

			const isWithinDynamic = _.has(atts, 'dynamic_parent') && atts.dynamic_parent.length;

			if ( !atts.activeTab && tabs.length  && !isWithinDynamic ) {
				wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( tabs[ 0 ].clientId, { active: true } );
			}
			else if ( !isWithinDynamic ) {
				wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( atts.activeTab, { active: true } );
			}

			const ClickThroughWrapper = wp.components.withFocusOutside(
				class extends wp.element.Component {
					render() {
						return el(
							'div',
							omit( clone( this.props ), [ "children" ] ),
							has( this.props, 'children' ) ? this.props.children : null
						);
					}
				}
			);

			return [

				// sidebar - settings
				el( InspectorControls, { group: 'settings' }, [
					// icon
					el( PanelBody, {
						title: __( 'Verhalten', 'greyd_blocks' ),
						initialOpen: true
					}, [

						el( SelectControl, {
							label: __( 'Übergangseffekt', 'greyd_blocks' ),
							value: atts.transition,
							options: [
								{ label: __( 'keine Animation', 'greyd_blocks' ), value: "" },
								{ label: __( 'überblenden (fade)', 'greyd_blocks' ), value: "fade" },
								{ label: __('horizontal verschieben (swipe)', 'greyd_blocks'), value: "horizontal" },
								{ label: __('vertikal verschieben (swipe)', 'greyd_blocks'), value: "vertical" },
							],
							onChange: ( value ) => props.setAttributes( { transition: value } )
						} ),
					] ),
				] ),

				//  sidebar - styles
				el( InspectorControls, { group: 'styles' }, [

					mode == "" ? [
						// elements
						el( PanelBody, { title: __( 'Elemente', 'greyd_blocks' ), initialOpen: true }, [

							el( greyd.components.SectionControl, {
								title: __( 'Tabs', 'greyd_blocks' ),
								onClick: () => setMode( "tabs" )
							} ),
							el( greyd.components.SectionControl, {
								title: __( 'Inhalt', 'greyd_blocks' ),
								onClick: () => setMode( "content" )
							} ),
						] ),
					] : null,

					mode == "tabs" ? [
						el ( 'span', {
							style: { display: 'block', paddingTop: '1rem', borderTop: '1px solid #e0e0e0' },
						}),
						el( greyd.components.SectionControl, {
							title: __( 'Tabs', 'greyd_blocks' ),
							icon: 'arrow-left-alt',
							buttonText: __( 'zurück', 'greyd_blocks' ),
							onClick: () => setMode( "" ),
							isHeader: true
						} ),
						// layout
						el( StylingControlPanel, {
							title: __( 'Layout', 'greyd_blocks' ),
							initialOpen: true,
							supportsResponsive: true,
							blockProps: props,
							controls: [
								{
									label: __( 'Ausrichtung', 'greyd_blocks' ),
									attribute: "--tabs-align-tabs",
									control: greyd.components.ButtonGroupControl,
									options: [
										{ label: __( 'links', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignLeft' ), value: 'flex-start' },
										{ label: __( 'zentriert', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignCenter' ), value: 'center' },
										{ label: __( 'rechts', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignRight' ), value: 'flex-end' },
										{ label: __( 'verteilt', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignSpaceBetween' ), value: 'space-between' },
									]
								},
								{
									label: __( 'Abstand dazwischen', 'greyd_blocks' ),
									attribute: "--tabs-tab-gap",
									control: greyd.components.RangeUnitControl,
								},
							]
						} ),
						// responsive
						el( PanelBody, {
							title: __( 'Responsive Verhalten', 'greyd_blocks' ),
							initialOpen: false,
						}, [
							el( ToggleControl, {
								label: __( 'Auf Mobilgeräten stapeln', 'greyd_blocks' ),
								checked: atts.stackOnMobile,
								onChange: ( value ) => {
									console.log( value );
									props.setAttributes( { stackOnMobile: value } )
								}
							} )
						] ),
						// icon
						el( PanelBody, {
							title: __( 'Icon', 'greyd_blocks' ),
							initialOpen: false,
						}, [
							el( greyd.components.ButtonGroupControl, {
								label: __( 'Position', 'greyd_blocks' ),
								value: atts.iconPosition,
								onChange: value => setAttributes( { iconPosition: value } ),
								options: [
									{ label: __( 'links', 'greyd_blocks' ), value: 'hasiconleft' },
									{ label: __( 'rechts', 'greyd_blocks' ), value: '' },
								]
							} ),
							el( wp.components.Tip, { }, __("Icons kannst du pro Tab individuell hinzufügen.", 'greyd_blocks') )
						] ),
						el( greyd.components.CustomButtonStyles, {
							blockProps: props,
							parentAttr: 'customStyles',
							supportsActive: true
						} )
					] : null,

					mode == "content" ? [
						el ( 'span', {
							style: { display: 'block', paddingTop: '1rem', borderTop: '1px solid #e0e0e0' },
						}),
						el( greyd.components.SectionControl, {
							title: __( 'Inhalt', 'greyd_blocks' ),
							icon: 'arrow-left-alt',
							buttonText: __( 'zurück', 'greyd_blocks' ),
							onClick: () => setMode( "" ),
							isHeader: true
						} ),
						el( greyd.components.StylingControlPanel, {
							title: __( 'Farben', 'greyd_blocks' ),
							initialOpen: true,
							blockProps: props,
							controls: [
								{
									label: __( 'Textfarbe', 'greyd_blocks' ),
									attribute: "--tabs-content-color",
									control: greyd.components.ColorGradientPopupControl,
									mode: 'color',
									preventConvertGradient: true
								},
								{
									label: __( 'Hintergrundfarbe', 'greyd_blocks' ),
									attribute: "--tabs-content-background",
									control: greyd.components.ColorGradientPopupControl,
									preventConvertGradient: true
								},
							]
						} ),
						el( StylingControlPanel, {
							title: __( 'Layout', 'greyd_blocks' ),
							initialOpen: false,
							supportsResponsive: true,
							blockProps: props,
							controls: [
								{
									label: __( 'Breite', 'greyd_blocks' ),
									attribute: "--tabs-content-width",
									control: greyd.components.RangeUnitControl,
									max: 1400
								},
								{
									label: __( 'Ausrichtung', 'greyd_blocks' ),
									attribute: "--tabs-content-align",
									control: greyd.components.ButtonGroupControl,
									options: [
										{ label: __( 'links', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignLeft' ), value: 'flex-start' },
										{ label: __( 'zentriert', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignCenter' ), value: 'center' },
										{ label: __( 'rechts', 'greyd_blocks' ), icon: greyd.tools.getCoreIcon( 'alignRight' ), value: 'flex-end' },
									]
								}
							]
						} ),
						el( StylingControlPanel, {
							title: __( 'Abstände', 'greyd_blocks' ),
							initialOpen: true,
							supportsResponsive: true,
							blockProps: props,
							controls: [
								{
									label: __( 'Padding', 'greyd_blocks' ),
									attribute: "--tabs-content-padding",
									control: greyd.components.DimensionControl,
									type: "string"
								}
							]
						} ),

						// border radius
						el( StylingControlPanel, {
							title: __( 'Eckenradius', 'greyd_blocks' ),
							initialOpen: false,
							blockProps: props,
							controls: [ {
								label: __( 'Eckenradius', 'greyd_blocks' ),
								attribute: "--tabs-content-radius",
								control: greyd.components.DimensionControl,
								labels: {
									"all": __( "alle Ecken", "greyd_blocks" ),
								},
								sides: [ "topLeft", "topRight", "bottomRight", "bottomLeft" ],
								type: "string"
							} ]
						} ),
					] : null
				] ),

				// preview
				el( 'div', {
					id: atts.anchor,
					className: [ className, defaultClass, atts.greydClass ].join( ' ' ),
					"data-transition": atts.transition,
					"data-overflow": atts.stackOnMobile ? "stack" : ""
				}, [
					el( 'div', {
						className: 'tabs',
					}, [
						// select steps
						tabs.length > 0 && tabs.map( ( tab, index ) => {
							const tabAtts = tab.attributes;

							return el( 'div', {
								className: tabClassNames + ( tab.attributes.active ? ' is-active' : '' ),
								onClick: () => {
									tabs.forEach( tab => {
										wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( tab.clientId, { active: false } );
									} );
									wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( tab.clientId, { active: true } );
								}
							}, [
								atts.iconPosition === 'hasiconleft' ? el( 'span', {
									className: tab.attributes.active ? tabAtts.iconActive : tabAtts.iconNormal,
									'aria-hidden': 'true',

									style: {
										cursor: 'pointer',
										'font-size': 'var(--tabs-icon-font-size)'
									}
								} ) : null,
								el( RichText, {
									tagName: 'span',
									className: 'title',
									clientId: tab.clientId,
									value: tabAtts.title,
									onChange: value => {
										wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( tab.clientId, { title: value } );
									},
									placeholder: __( 'Titel', 'greyd_blocks' ),
									style: { flexGrow: '1' }
								} ),
								!atts.iconPosition ? el( 'span', {
									className: tab.attributes.active ? tabAtts.iconActive : tabAtts.iconNormal,
									'aria-hidden': 'true',

									style: {
										cursor: 'pointer',
										'font-size': 'var(--tabs-icon-font-size)'
									}
								} ) : null
							] );
						} ),
						el( 'div', {
							className: tabClassNames,
							onClick: function () {
								const innerCount = wp.data.select( "core/block-editor" ).getBlocksByClientId( props.clientId )[ 0 ].innerBlocks.length;
								let block = wp.blocks.createBlock( "greyd/tab" );
								wp.data.dispatch( "core/block-editor" ).insertBlock( block, innerCount, props.clientId );
							},
						}, "+" )
					] ),
					// custom styles
					el( greyd.components.RenderPreviewStyles, {
						selector: props.attributes.greydClass + " .greyd_tab",
						activeSelector: props.attributes.greydClass + " .greyd_tab.is-active",
						styles: {
							"": props.attributes.customStyles,
						},
						important: true
					} ),

					el( 'div', {
						className: 'panels' + (
							_.has( atts.greydStyles, '--tabs-content-background' ) ? ' has-background-color' : ''
						) + (
							_.has( atts.greydStyles, '--tabs-content-color' ) ? ' has-text-color' : ''
						),
					},[
						el( ClickThroughWrapper, {
							onClick: ( e ) => {
								// console.log(e);
							}
						},
							el( InnerBlocks, {
								allowedBlocks: [ 'greyd/tab' ],
								template: [ [ 'greyd/tab', { active: true, title: 'Tab 1' } ] ],
								renderAppender: false
							} )
						)
					]),

				] ),

				// styles
				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						".wp-block-greyd-tabs": atts.greydStyles,
					}
				} ),
			];
		},

		save: function ( props ) {

			const {
				attributes: atts
			} = props;
			
			const defaultClass = atts.className && atts.className.length && atts.className.indexOf( 'is-style-' ) > -1 ? '' : 'is-style-tabs';

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ defaultClass, atts.greydClass, atts.iconPosition ].join( ' ' ),
				"data-transition": atts.transition,
				...(
					atts.stackOnMobile ? { "data-overflow": "stack" } : {}
				)
			} );

			return el( 'div', blockProps, [

				el( 'div', {
					className: 'tabs',
					role: 'tablist'
				} ),
				el( greyd.components.RenderSavedStyles, {
					selector: props.attributes.greydClass + " .greyd_tab",
					activeSelector: props.attributes.greydClass + " .greyd_tab.is-active",
					styles: {
						"": props.attributes.customStyles,
					},
					important: true
				} ),
				el( 'div', {
					className: 'panels' + (
						_.has( atts.greydStyles, '--tabs-content-background' ) ? ' has-background-color' : ''
					) + (
						_.has( atts.greydStyles, '--tabs-content-color' ) ? ' has-text-color' : ''
					),
				}, [
					el( InnerBlocks.Content, {} ),
				] ),
				el( greyd.components.RenderSavedStyles, {
					selector: atts.greydClass + ".wp-block-greyd-tabs",
					styles: {
						"": atts.greydStyles,
					}
				} )
			] );
		},

		deprecated: [
			/**
			 * Alignment support.
			 */
			{
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				attributes: {
					iconPosition: { type: 'string', default: '' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object', default: {} },
					customStyles: { type: 'object', default: {} },
					transition: { type: 'string', default: '' },
		
					activeTab: { type: 'string', default: '' }
				},
				save: function ( props ) {

					const {
						attributes: atts
					} = props;
					
					const defaultClass = atts.className && atts.className.length && atts.className.indexOf( 'is-style-' ) > -1 ? '' : 'is-style-tabs';
		
					return el( 'div', {
						id: atts.anchor,
						className: [ atts.className, defaultClass, atts.greydClass, atts.iconPosition ].join( ' ' ),
						"data-transition": atts.transition
						// className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size, 'greyd_tab' ].join(' ')
					}, [
		
						el( 'div', {
							className: 'tabs',
							role: 'tablist'
						} ),
						el( greyd.components.RenderSavedStyles, {
							selector: props.attributes.greydClass + " .greyd_tab",
							activeSelector: props.attributes.greydClass + " .greyd_tab.is-active",
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} ),
						el( 'div', {
							className: 'panels' + (
								_.has( atts.greydStyles, '--tabs-content-background' ) ? ' has-background-color' : ''
							) + (
								_.has( atts.greydStyles, '--tabs-content-color' ) ? ' has-text-color' : ''
							),
						}, [
							el( InnerBlocks.Content, {} ),
						] ),
						el( greyd.components.RenderSavedStyles, {
							selector: atts.greydClass + ".wp-block-greyd-tabs",
							styles: {
								"": atts.greydStyles,
							}
						} )
					] );
				}
			},

			/**
			 * add 'has-text-color' and 'has-background-color' classes to the panels
			 * if the color is set in the inspector.
			 * @since 1.7.4
			 */
			{
				attributes: {
					iconPosition: { type: 'string', default: '' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object', default: {} },
					customStyles: { type: 'object', default: {} },
					transition: { type: 'string', default: '' },

					activeTab: { type: 'string', default: '' }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function ( props ) {

					const {
						attributes: atts
					} = props;
					
					const defaultClass = atts.className && atts.className.length && atts.className.indexOf( 'is-style-' ) > -1 ? '' : 'is-style-tabs';
		
					return el( 'div', {
						id: atts.anchor,
						className: [ atts.className, defaultClass, atts.greydClass, atts.iconPosition ].join( ' ' ),
						"data-transition": atts.transition
					}, [
		
						el( 'div', {
							className: 'tabs',
							role: 'tablist'
						} ),
						el( greyd.components.RenderSavedStyles, {
							selector: props.attributes.greydClass + " .greyd_tab",
							activeSelector: props.attributes.greydClass + " .greyd_tab.is-active",
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} ),
						el( 'div', {
							className: 'panels',
						}, [
							el( InnerBlocks.Content, {} ),
						] ),
						el( greyd.components.RenderSavedStyles, {
							selector: atts.greydClass + ".wp-block-greyd-tabs",
							styles: {
								"": atts.greydStyles,
							}
						} )
					] );
				}
			},
			/**
			 * Do not save the word {{greyd_tabs}} unescaped, it will be found by 
			 * the search. Replace the entire <div> instead.
			 * @since 1.7.0
			 */
			{
				attributes: {
					iconPosition: { type: 'string', default: '' },
					greydClass: { type: 'string', default: '' },
					greydStyles: { type: 'object', default: {} },
					customStyles: { type: 'object', default: {} },
					transition: { type: 'string', default: '' },

					activeTab: { type: 'string', default: '' }
				},
				supports: {
					anchor: true,
					defaultStylePicker: false
				},
				save: function ( props ) {

					const {
						attributes: atts
					} = props;
					
					const defaultClass = atts.className && atts.className.length && atts.className.indexOf( 'is-style-' ) > -1 ? '' : 'is-style-tabs';
		
					return el( 'div', {
						id: atts.anchor,
						className: [ atts.className, defaultClass, atts.greydClass, atts.iconPosition ].join( ' ' ),
						"data-transition": atts.transition
						// className: [ extraClass, props.attributes.className, props.attributes.greydClass, props.attributes.size, 'greyd_tab' ].join(' ')
					}, [
		
						el( 'div', {
							className: 'tabs',
							role: 'tablist'
						}, [
							"{{greyd_tabs}}"
						] ),
						el( greyd.components.RenderSavedStyles, {
							selector: props.attributes.greydClass + " .greyd_tab",
							activeSelector: props.attributes.greydClass + " .greyd_tab.is-active",
							styles: {
								"": props.attributes.customStyles,
							},
							important: true
						} ),
						el( 'div', {
							className: "panels",
						},
							el( InnerBlocks.Content, {} ),
						),
						el( greyd.components.RenderSavedStyles, {
							selector: atts.greydClass + ".wp-block-greyd-tabs",
							styles: {
								"": atts.greydStyles,
							}
						} )
					]
					);
				
				}
			}
		]
	} );


	/**
	 * Register the accordion child items.
	 */
	wp.blocks.registerBlockType( 'greyd/tab', {
		title: __( 'Tab', 'greyd_blocks' ),
		description: __( 'Zusammenklappbare Sektion mit Titel, Icon und Inhalten', 'greyd_blocks' ),
		icon: greyd.tools.getBlockIcon('tab'),
		category: 'greyd-blocks',
		parent: [ 'greyd/tabs' ],
		supports: {
			anchor: true,
			className: true
		},
		example: {
			attributes: {
				title: __('Tab 1', 'greyd_blocks'),
				active: true,
			},
			innerBlocks: [
				{
					name: 'core/paragraph',
					attributes: {
						content: __('Organisiere Inhalte in übersichtlichen Registerkarten und ermögliche den einfachen Wechsel zwischen verschiedenen Informationen auf deiner Webseite.', 'greyd_blocks')
					}
				},
			]
		},
		attributes: {
			active: { type: "boolean", default: false },
			title: {
				type: 'string', default: ''
			},
			iconNormal: { type: 'string', default: '' },
			iconActive: { type: 'string', default: '' },
			uniqueId: { type: 'string', default: '' },
		},
		usesContext: [
			'greyd/tab-iconPosition',
		],

		edit: function ( props ) {

			const {
				className,
				setAttributes,
				attributes: atts,
				context
			} = props;

			const iconPosition = context[ 'greyd/tab-iconPosition' ];
			const parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd/tabs" );

			const isWithinDynamic = _.has(atts, 'dynamic_parent') && atts.dynamic_parent.length;

			if ( props.isSelected ) {
				const parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
				const tabs = parent.innerBlocks;
				tabs.forEach( tab => {
					if ( tab.clientId === props.clientId && !isWithinDynamic ) {
						wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( parent.clientId, { activeStep: props.clientId } );
					}
				} );
				// setAttributes( {active: true} );
				// setTimeout(function() {
				// 	wp.data.dispatch('core/block-editor').selectBlock(parentId.clientId);
				// 	$('#block-'+parentId.clientId).focus();
				// }, 0);
			}

			if ( iconPosition !== atts.iconNormal && !isWithinDynamic ) {
				setAttributes( {
					iconPosition: iconPosition,
				} );
			}

			if ( _.isEmpty( atts.uniqueId ) && !isWithinDynamic ) {
				setAttributes( {
					uniqueId: greyd.tools.generateRandomID()
				} );
			}

			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );

			if ( atts.active && !isWithinDynamic ) {
				wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( parentId, { activeTab: props.clientId } );
			}

			return [

				el( InspectorControls, {}, [

					el( PanelBody, {}, [
						el( 'p', { className: "greyd-inspector-help" }, __( 'Das Design dieser Sektion kannst du im übergeordneten Tab Block einstellen.', 'greyd_blocks' ) ),
						el( Button, {
							variant: 'secondary',
							icon: 'visibility',
							onClick: () => {
								wp.data.dispatch( 'core/block-editor' ).selectBlock(
									wp.data.select( 'core/block-editor' ).getBlockParents( props.clientId ).slice( -1 ).pop()
								);
							}
						}, __( 'Tabs', 'greyd_blocks' ) ),
						el( Button, {
							variant: 'tertiary',
							icon: 'no',
							onClick: () => {
								// setTimeout(function() {
								// 	wp.data.dispatch('core/block-editor').selectBlock(parentId);
								// 	$('#block-'+parentId).focus();
								// }, 0);
								wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( parentId, { activeTab: '' } );
								wp.data.dispatch( "core/block-editor" ).removeBlock( props.clientId );
							}
						}, __( 'Tab entfernen', 'greyd_blocks' ) )
					] ),
					// icon
					el( PanelBody, {
						title: __( 'Icon', 'greyd_blocks' ),
						initialOpen: true
					}, [
						el( greyd.components.IconPicker, {
							label: __( 'Normales Icon', 'greyd_blocks' ),
							value: atts.iconNormal,
							onChange: value => {
								setAttributes( { iconNormal: value } );
								setTimeout( function () {
									wp.data.dispatch( 'core/block-editor' ).selectBlock( parentId );
									$( '#block-' + parentId ).focus();
								}, 0 );
							}
						} ),
						el( greyd.components.IconPicker, {
							label: __( 'Aktives Icon', 'greyd_blocks' ),
							value: atts.iconActive,
							onChange: value => {
								setAttributes( { iconActive: value } );
								setTimeout( function () {
									wp.data.dispatch( 'core/block-editor' ).selectBlock( parentId );
									$( '#block-' + parentId ).focus();
								}, 0 );
							}
						} ),
					] )
				] ),

				// preview
				el( 'div', {
					id: "tabpanel_" + atts.uniqueId,
					className: [ className, atts.active ? "is-active" : "" ].join( ' ' )
				}, [
					el( 'div', {
						className: 'wp-block-greyd-tabs__content'
					}, [
						el( InnerBlocks, {
							renderAppender: hasChildBlocks ? InnerBlocks.DefaultBlockAppender : InnerBlocks.ButtonBlockAppender
						} )
					] ),
				] ),

			];
		},
		save: function ( props ) {

			const {
				className,
				attributes: atts,
			} = props;

			return el( Fragment, {}, [
				el( 'div', {
					id: "tabpanel_" + atts.uniqueId,
					// tabindex: "0",
					className: "panel",
					role: 'tabpanel',
					"aria-labelledby": "tab_" + atts.uniqueId
				},
					el( InnerBlocks.Content, {} )
				)
			] );
		},
	} );


} )(
	window.wp
);