<?php
/**
 * Enqueue block editor assets.
 */
namespace greyd\blocks;

if ( !defined( 'ABSPATH' ) ) exit;

new enqueue($config);
class enqueue {

	/**
	 * Holds the plugin config
	 */
	private $config;

	/**
	 * Constructor
	 */
	public function __construct($config) {

		// check if Gutenberg is active.
		if (!function_exists('register_block_type')) return;

		// set config
		$this->config = (object) $config;

		// setup
		if (!function_exists('get_plugin_data')) require_once(ABSPATH . '/wp-admin/includes/plugin.php');
		$this->config->plugin_version = get_plugin_data($this->config->plugin_file)['Version'];
		$this->config->assets_url = plugin_dir_url($this->config->plugin_file).'assets';
		$this->config->assets_path = trailingslashit($this->config->plugin_path).'assets';

		add_action( 'after_setup_theme', array($this, 'add_basic_theme_supports'), 1 );
		add_action( 'init', array($this, 'init') );
	}

	/**
	 * Init the hooks
	 */
	public function init() {
		
		add_action( 'wp_enqueue_scripts', array($this, 'add_frontend_styles'), 90 );

		if ( is_admin() ) {
			add_action( 'admin_enqueue_scripts', array($this, 'add_backend_styles'), 999 );
			add_filter( 'block_editor_settings_all', array($this, 'add_editor_styles'), 99, 2 );
			add_action( 'enqueue_block_assets', array($this, 'register_block_editor_styles') );
			add_action( 'enqueue_block_editor_assets', array($this, 'register_block_editor_scripts') );

			// disable block-directory
			remove_action( 'enqueue_block_editor_assets', 'wp_enqueue_editor_block_directory_assets' );
			remove_action( 'enqueue_block_editor_assets', 'gutenberg_enqueue_block_editor_assets_block_directory' );
		}
	}

	/**
	 * Adds basic theme supports for non-block themes.
	 * @see \wp-includes\theme.php.
	 */
	function add_basic_theme_supports() {
		if ( wp_is_block_theme() ) {
			return;
		}

		add_theme_support( 'post-thumbnails' );
		add_theme_support( 'responsive-embeds' );
		add_theme_support( 'editor-styles' );
		/*
		* Makes block themes support HTML5 by default for the comment block and search form
		* (which use default template functions) and `[caption]` and `[gallery]` shortcodes.
		* Other blocks contain their own HTML5 markup.
		*/
		add_theme_support( 'html5', array( 'comment-form', 'comment-list', 'search-form', 'gallery', 'caption', 'style', 'script' ) );
		add_theme_support( 'automatic-feed-links' );

	}

	/**
	 * =================================================================
	 *                          ENQUEUE STYLES
	 * =================================================================
	 */

	/**
	 * Add frontend block styles
	 * @action wp_enqueue_scripts
	 */
	public function add_frontend_styles() {

		global $wp_styles;
		// debug($wp_styles);

		// unregister wp-editor-classic-layout-styles
		$wp_styles->registered['wp-editor-classic-layout-styles']->src = '';

		// basic frontend blocks styles
		if ( class_exists('\processor') ) {
			wp_register_style(
				'greyd-blocks-frontend-style',
				$this->config->assets_url.'/css/classic/blocks.css',
				array( ),
				$this->config->plugin_version
			);
		}
		else {
			wp_register_style(
				'greyd-blocks-frontend-style',
				$this->config->assets_url.'/css/blocks.css',
				array( ),
				$this->config->plugin_version
			);
		}
		wp_enqueue_style('greyd-blocks-frontend-style');

		// todo: move to Query Feature
		wp_register_style(
			'greyd-query-frontend-style',
			$this->config->assets_url.'/css/move/blocks-query.css',
			array( ),
			$this->config->plugin_version
		);
		wp_enqueue_style('greyd-query-frontend-style');

	}

	public function add_backend_styles() {

		$screen = get_current_screen();
		$disable_preview_helper = false;

		// on edit screens
		if ($screen->base === 'post' && $screen->action !== 'add') {

			// load custom and/or google fonts
			// only in Theme when processor is available
			if (class_exists('processor')) {
				\processor::load_scripts();
			}

			// only for template
			// todo: move to Dynamic Feature
			if ( self::is_greyd_classic() && $screen->post_type === "dynamic_template" ) {

				$post_id	= $_GET['post'];
				$post		= get_post( $post_id );
				$slug		= $post ? $post->post_name : null;

				// enqueue template-specific stylesheets
				if ( strpos($slug, 'footer') === 0 ) {
					wp_process_style('default_admin_css', $this->config->assets_path.'/css/move/editor-footer.css' );
				}
				else if ( strpos($slug, 'main-menu-offcanvas') === 0 ) {
					wp_process_style('default_admin_css', $this->config->assets_path.'/css/move/editor-mainmenuoff.css' );
					$type = get_theme_mod( 'navi_header_off_type', 'offcanvas' );
					if ( $type === 'full' ) {
						echo '<style>.editor-styles-wrapper > .is-root-container.wp-block-post-content { --HEDOFFwidth: 100% !important; }</style>';
					}
				}
				else if ( strpos($slug, 'meta-menu-offcanvas') === 0 ) {
					wp_process_style('default_admin_css', $this->config->assets_path.'/css/move/editor-metamenuoff.css' );
					$type = get_theme_mod( 'navi_header_sec_off_type', 'offcanvas' );
					if ( $type === 'full' ) {
						echo '<style>.editor-styles-wrapper > .is-root-container.wp-block-post-content { --HED2OFFwidth: 100% !important; }</style>';
					}
				}

				// disable preview-helper for all system-templates
				if ( !has_term('dynamic', 'template_type', $post_id) ) {
					$disable_preview_helper = true;
				}
			}

			// remove editor helper
			// todo: move to popup feature
			if ( $screen->post_type === "greyd_popup" ) {
				$disable_preview_helper = true;
			}
		}

		echo '<script>var disablePreviewHelper = '.($disable_preview_helper ? 'true' : 'false').';</script>';
	}

	/**
	 * Add styles to the editor preview
	 */
	public function add_editor_styles($editor_settings, $editor_context) {
		// debug($editor_settings);
		// debug($editor_context);

		if ( ! self::is_greyd_classic() ) {

			// if ( method_exists( '\Greyd\Helper', 'get_file_contents' ) ) {
			// 	$editor_settings['styles'][] = array(
			// 		'css' => \Greyd\Helper::get_file_contents( $this->config->assets_path.'/css/editor-blocks.css' )
			// 	);
			// 	$editor_settings['styles'][] = array(
			// 		'css' => \Greyd\Helper::get_file_contents( $this->config->assets_path.'/css/move/editor-dynamic-blocks.css' )
			// 	);
			// }
			// else {
			// 	$editor_settings['styles'][] = array(
			// 		'css' => file_get_contents( $this->config->assets_path.'/css/editor-blocks.css' )
			// 	);
			// 	$editor_settings['styles'][] = array(
			// 		'css' => file_get_contents( $this->config->assets_path.'/css/move/editor-dynamic-blocks.css' )
			// 	);
			// }

			return $editor_settings;
		}

		// add block pattern with dynamic template
		// todo: move to Query Feature
		$query_template_content = '<!-- wp:query {"query":{"perPage":1,"pages":0,"offset":0,"postType":"post","order":"desc","orderBy":"date","author":"","search":"","exclude":[],"sticky":"","inherit":false}} -->';
		$query_template_content .= '<div class="wp-block-query"><!-- wp:post-template --><!-- wp:greyd/dynamic /--><!-- /wp:post-template --></div>';
		$query_template_content .= '<!-- /wp:query -->';
		$query_template = array(
			"title" => "Template",
			"blockTypes" => array( "core/query" ),
			"categories" => array(),
			"content" => $query_template_content,
			"name" => "core/query-template-posts"
		);
		if ( isset($editor_settings["__experimentalBlockPatterns"]) && is_array($editor_settings["__experimentalBlockPatterns"]) ) {
			array_unshift($editor_settings["__experimentalBlockPatterns"], $query_template);
			$query_default = '<!-- wp:query {"query":{"perPage":1,"pages":0,"offset":0,"postType":"post","order":"desc","orderBy":"date","author":"","search":"","exclude":[],"sticky":"","inherit":false}} -->';
			$query_default .= '<div class="wp-block-query"><!-- wp:post-template --><!-- wp:post-title {"isLink":true} /--><!-- wp:post-excerpt /--><!-- wp:post-date /--><!-- /wp:post-template --></div>';
			$query_default .= '<!-- /wp:query -->';
			$editor_settings["__experimentalBlockPatterns"][1]["content"] = $query_default;
		}

		// Theme Styles in editor
		// todo: refactor/deprecate
		$css = "body:not(.block-editor-block-preview__container) { 
					height: auto !important; 
					overflow-x: unset; 
					padding-bottom: 0px !important; 
				} 
				body.block-editor-block-preview__container { 
					position: relative !important; 
				} 
				body > div { 
					max-width: 100%; 
				} 
				.block-editor-writing-flow { 
					width: 100% 
				}";
		$editor_settings['defaultEditorStyles'][] = array('css' => $css);
		
		// only in Theme when processor is available
		if ( class_exists('processor') ) {
			$editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/root.css'));
			$editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/style.css'));
			// $editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles(get_template_directory_uri().'/assets/css/deprecated/vc_styles.css'));
			// $editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles(get_template_directory_uri().'/assets/css/deprecated/vc_basics.css'));
			// $editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles(get_template_directory_uri().'/assets/css/deprecated/vc_anims.css'));

			$editor_settings['styles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/root.css'));
			$editor_settings['styles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/style.css'));
			$editor_settings['styles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/deprecated/vc_styles.css'));
			$editor_settings['styles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/deprecated/vc_basics.css'));
			// $editor_settings['styles'][] = array('css' => \processor::process_styles(get_template_directory().'/assets/css/deprecated/vc_anims.css'));

			$editor_settings['styles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/classic/editor-blocks.css'));
			$editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/classic/editor-blocks.css'));
			$editor_settings['styles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/classic/editor-blocks.css'));
			// todo: move to Dynamic/Query Feature
			$editor_settings['styles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/move/editor-dynamic-blocks.css'));
			$editor_settings['defaultEditorStyles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/move/editor-dynamic-blocks.css'));
			$editor_settings['styles'][] = array('css' => \processor::process_styles($this->config->assets_path.'/css/move/editor-dynamic-blocks.css'));
		}

		return $editor_settings;
	}

	/**
	 * Register and enqueue all the styles for the editor.
	 * @action enqueue_block_assets
	 */
	public function register_block_editor_styles() {

		if ( ! is_admin() ) return;

		// styles
		wp_register_style(
			'greyd-blocks-style',
			$this->config->assets_url.'/css/editor.css',
			array( ),
			$this->config->plugin_version
		);
		wp_enqueue_style( 'greyd-blocks-style' );

		add_editor_style( $this->config->assets_url.'/css/blocks.css' );

		add_editor_style( $this->config->assets_url.'/css/hover-helper.css' );

		if ( ! self::is_greyd_classic() ) {

			add_editor_style( $this->config->assets_url.'/css/editor-blocks.css' );

			// todo: move to Dynamic Feature
			add_editor_style( $this->config->assets_url.'/css/move/editor-dynamic-blocks.css' );
		}

		// todo: move to Dynamic Feature
		wp_register_style(
			'greyd-dynamic-blocks-style',
			$this->config->assets_url.'/css/move/editor-dynamic.css',
			array( ),
			$this->config->plugin_version
		);
		wp_enqueue_style( 'greyd-dynamic-blocks-style' );

		// todo: move to Query Feature
		if ( defined( 'GREYD_VERSION' ) && version_compare( GREYD_VERSION, '1.6.9', '>' ) ) {
			wp_register_style(
				'greyd-query-frontend-style',
				$this->config->assets_url.'/css/move/blocks-query.css',
				array( ),
				$this->config->plugin_version
			);
			wp_enqueue_style( 'greyd-query-frontend-style' );
		}

	}

	/**
	 * Register and enqueue all the scripts for the editor.
	 * @action enqueue_block_editor_assets
	 */
	public function register_block_editor_scripts() {

		if ( ! is_admin() ) return;

		// tools script
		wp_register_script(
			'greyd-tools',
			$this->config->assets_url.'/js/tools.js',
			array( 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->config->plugin_version
		);
		wp_localize_script('greyd-tools', 'greyd', array( 'data' => $this->get_greyd_data() ));
		wp_enqueue_script('greyd-tools');

		// Custom components
		wp_register_script(
			'greyd-components',
			$this->config->assets_url.'/js/components.js',
			array( 'greyd-tools', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->config->plugin_version
		);
		wp_enqueue_script('greyd-components');

		// editor script
		wp_register_script(
			'greyd-editor',
			$this->config->assets_url.'/js/editor.js',
			array( 'greyd-tools', 'greyd-components', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-data', 'wp-i18n', 'lodash' ),
			$this->config->plugin_version
		);
		wp_enqueue_script('greyd-editor');

		// formats script
		wp_register_script(
			'greyd-formats',
			$this->config->assets_url.'/js/formats.js',
			array( 'greyd-tools', 'greyd-components', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->config->plugin_version
		);
		wp_enqueue_script('greyd-formats');

		// transform script
		wp_register_script(
			'greyd-transform',
			$this->config->assets_url.'/js/transform.js',
			array( 'greyd-tools', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->config->plugin_version
		);
		wp_enqueue_script('greyd-transform');

		// translations
		if ( function_exists( 'wp_set_script_translations' ) ) {
			/**
			 * May be extended to wp_set_script_translations( 'my-handle', 'my-domain',
			 * plugin_dir_path( MY_PLUGIN ) . 'languages' ) ). For details see
			 * https://make.wordpress.org/core/2018/11/09/new-javascript-i18n-support-in-wordpress/
			 */
			wp_set_script_translations( 'greyd-tools', 'greyd_blocks', $this->config->plugin_path."/languages" );
			wp_set_script_translations( 'greyd-components', 'greyd_blocks', $this->config->plugin_path."/languages" );
			wp_set_script_translations( 'greyd-editor', 'greyd_blocks', $this->config->plugin_path."/languages" );
			wp_set_script_translations( 'greyd-formats', 'greyd_blocks', $this->config->plugin_path."/languages" );
			wp_set_script_translations( 'greyd-transform', 'greyd_blocks', $this->config->plugin_path."/languages" );

			if (is_plugin_active('woocommerce/woocommerce.php')) {
				wp_set_script_translations( 'greyd-woocommerce', 'greyd_blocks', $this->config->plugin_path."/languages" );
				wp_set_script_translations( 'greyd-woo-blocks', 'greyd_blocks', $this->config->plugin_path."/languages" );
			}
		}

		// todo: modularize
		if (is_plugin_active('woocommerce/woocommerce.php')) {

			wp_register_script(
				'greyd-woocommerce',
				plugin_dir_url($this->config->plugin_file).'extensions/woocommerce/assets/js/helper.js',
				array('greyd-tools', 'wp-i18n'),
				$this->config->plugin_version
			);
			wp_add_inline_script('greyd-woocommerce', 'greyd.woo = '.json_encode( array(
				'data' => array(
					'wizard_url' => admin_url('edit.php?post_type=dynamic_template&wizard=add'),
					'shop_page_id' => intval(get_option('woocommerce_shop_page_id')),
					'my_account_page_id' => intval(get_option('woocommerce_myaccount_page_id')),
					'cart_page_id' => intval(get_option('woocommerce_cart_page_id')),
					'checkout_page_id' => intval(get_option('woocommerce_checkout_page_id')),
				)
			) ).';', 'before' );
			wp_enqueue_script('greyd-woocommerce');

			wp_register_script(
				'greyd-woo-blocks',
				plugin_dir_url($this->config->plugin_file).'extensions/woocommerce/assets/js/blocks.js',
				array( 'greyd-tools', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash', 'wp-core-data', 'wp-edit-post' ),
				$this->config->plugin_version
			);
			wp_enqueue_script('greyd-woo-blocks');
			
			// translations
			if ( function_exists( 'wp_set_script_translations' ) ) {
				wp_set_script_translations( 'greyd-woocommerce', 'greyd_blocks', $this->config->plugin_path."/languages" );
				wp_set_script_translations( 'greyd-woo-blocks', 'greyd_blocks', $this->config->plugin_path."/languages" );
			}

		}
	}

	/**
	 * Get all the data for JS
	 */
	public function get_greyd_data() {

		// get urls
		$urls = array(
			'home' => esc_url(get_bloginfo('url')),
			'current' => esc_url(get_bloginfo('url')),
			'rss' => esc_url(get_bloginfo('rss2_url')),
			'comments_rss' => esc_url(get_bloginfo('comments_rss2_url')),
		);

		// get language
		$language_default = apply_filters( 'wpml_default_language', NULL );
		$language_post = "";

		// the current post
		$current_post_id = isset($_REQUEST['post']) && !empty($_REQUEST['post']) ? intval($_REQUEST['post']) : null;
		$current_post = (object)array(
			'id' => $current_post_id,
			'name' => "",
			'title' => "",
			'post_type' => get_post_type(),
		);
		if ($current_post->id) {
			// get post
			$post = get_post($current_post->id);
			// make values
			$urls['current'] = esc_url(get_permalink($current_post->id));
			$current_post->name = $post->post_name;
			$current_post->title = $post->post_title;
			// $language_post = \Greyd\Helper::get_post_language($current_post->id);
			$language_post = \greyd\blocks\deprecated\Functions::get_post_language($current_post->id, $language_default);
		}

		// get contents
		// all posttypes, taxes and posts
		$all_post_types = array();
		$all_taxes = array( 'post' => array() );
		$all_posts = array( 'post' => array(), 'page' => array());	

		if (is_plugin_active('woocommerce/woocommerce.php')) {
			$all_taxes['product'] = array();
		}

		if ( method_exists('\Greyd\Posttypes\Posttype_Helper', 'get_dynamic_posttypes' ) ) {
			foreach (\Greyd\Posttypes\Posttype_Helper::get_dynamic_posttypes() as $id => $pt) {
				$all_post_types[$pt['slug']] = $pt;
				$all_taxes[$pt['slug']] = array();
				$all_posts[$pt['slug']] = array();
			}
		}

		if ( class_exists('\Greyd\Helper') ) {
			foreach ($all_taxes as $pt => $value) {
				// get all taxonomies (categories, tags, customtaxes)
				$taxes = \Greyd\Helper::get_all_taxonomies($pt);
				foreach ($taxes as $j => $tax) {
					$terms = \Greyd\Helper::get_all_terms($tax->slug);
					$taxes[$j]->values = $terms;
				}
				$all_taxes[$pt] = $taxes;
				$all_post_types[$pt]['taxes'] = $taxes;
			}
			foreach ($all_posts as $pt => $value) {
				// get all posts
				$posts = \Greyd\Helper::get_all_posts($pt);
				if ($posts) {
					// debug($posts);
					foreach ($posts as $post) {
						if ($post->lang) {
							$all_posts[$pt][] = (object)array( 
								'id' => $post->id, 
								'slug' => $post->slug, 
								'title' => $post->title, 
								'type' => $pt, 
								'lang' => $post->lang
							);
						}
					}
				}
			}
		}

		// get public post types
		$include   = array( 'page', 'post' );
		$exclude   = array( 'attachment' );
		$posttypes = array_keys( get_post_types( array( 'public' => true, 'exclude_from_search' => false ) ) );
		$all_public_posttypes = array_values( array_unique( array_diff( array_merge( $include, $posttypes ), $exclude ) ) );

		// get template_type
		// todo: use filter in Dynamic Feature
		$template_type = "";
		if (
			!empty($current_post->post_type) && 
			$current_post->post_type == 'dynamic_template' && 
			$current_post->id &&
			class_exists('\Greyd\Helper')
		) {
			$template_type = \Greyd\Helper::get_post_taxonomy_terms( $current_post->id, 'template_type' )[0]->slug;
			// debug($template_type);
		}

		// get all templates
		// todo: use filter in Dynamic Feature
		$all_templates = array();
		if ( class_exists('\Greyd\Helper') ) {
			$templates = \Greyd\Helper::get_all_posts('dynamic_template');
			if ($templates) {
				// debug($templates);
				foreach ($templates as $template) {
	
					// only dynamic templates
					if ( 'dynamic' !== \Greyd\Helper::get_post_taxonomy_terms( $template->id, 'template_type' )[0]->slug ) continue;
	
					if ($template->lang) {
						$title = $template->title;
						$categories = \Greyd\Helper::get_post_taxonomy_terms( $template->id, 'template_categories' );
						if ($categories && is_array($categories)) {
							// add categories to title
							$cat_names = array();
							foreach ($categories as $cat) $cat_names[] = $cat->name; 
							$title .= " (".implode(", ", $cat_names).")";
						}
						$all_templates[] = (object) array(
							'id' => $template->id, 
							'slug' => $template->slug, 
							'title' => $title, 
							'type' => 'dynamic_template', 
							'blocks' => has_blocks(get_post_field('post_content', $template->id)),
							'lang' => $template->lang
						);
					}
				}
			}
			foreach (array( 'wp_block', 'wp_template', 'wp_template_part' ) as $type) {
				$templates = \Greyd\Helper::get_all_posts($type);
				if ($templates) {
					$parts = array();
					foreach ($templates as $template) {
						if ($template->lang) {
							$parts[] = (object)array( 
								'id' => $template->id, 
								'slug' => $template->slug, 
								'title' => $template->title, 
								'type' => $type, 
								'blocks' => true,
								'lang' => $template->lang
							);
						}
					}
					$all_templates = array_merge($all_templates, $parts);
				}
			}
		}

		// get all forms
		// todo: use filter in Forms
		$all_forms = false; // 'false' if GREYD.Forms is inactive
		if ( class_exists('\Greyd\Helper') ) {
			if (\Greyd\Helper::is_active_plugin('greyd_tp_forms/init.php')) {
				$all_forms = array(); // 'array()' if GREYD.Forms is active
				$forms = \Greyd\Helper::get_all_posts('tp_forms');
				if ($forms) {
					// debug($forms);
					foreach ($forms as $form) {
						if ($form->lang) {
							$all_forms[] = (object)array( 
								'id' => $form->id, 
								'title' => $form->title, 
								'lang' => $form->lang
							);
						}
					}
				}
			}
		}

		// get all popups
		// todo: use filter in Popups Feature
		$all_popups = array();
		if ( class_exists('\Greyd\Helper') ) {
			$popups = \Greyd\Helper::get_all_posts('greyd_popup');
			if ($popups) {
				// debug($popups);
				foreach ($popups as $popup) {
					if ($popup->lang) {
						$all_popups[] = (object)array( 
							'id' => $popup->id, 
							'slug' => $popup->slug, 
							'title' => $popup->title, 
							'lang' => $popup->lang
						);
					}
				}
			}
		}

		// get sources of all media files
		$media_urls = array();
		$attachments = get_posts(array(
			'post_type' => 'attachment',
			'numberposts' => -1,
			'post_status' => null,
			'post_parent' => null, // any parent
		));
		if ($attachments) {
			// debug($attachments);
			foreach ($attachments as $attachment) {
				// debug($attachment);
				// get language
				// $language = \Greyd\Helper::get_post_language($attachment);
				$language = \greyd\blocks\deprecated\Functions::get_post_language($attachment, $language_default);
				if ($language) {
					$src = "";
					if (strpos($attachment->post_mime_type, 'image/') === 0)
						$src = wp_get_attachment_image_src( $attachment->ID, 'full' )[0];
					else $src = $attachment->guid;
					$media_urls[$attachment->ID] = array( 
						'id' => $attachment->ID, 
						'src' => $src,
						'type' => $attachment->post_mime_type, 
						'title' => $attachment->post_title, 
						'lang' => $language
					);
				}
			}
		}
		// debug($media_urls);

		// get all nav menus
		$nav_menus = array();
		$menus = wp_get_nav_menus();
		if ($menus) {
			// debug($menus);
			foreach ($menus as $menu) {
				$items = array();
				$sub = array();
				foreach (wp_get_nav_menu_items($menu->term_id) as $item) {
					if ($item->menu_item_parent == 0)
						$items[] = array( 'id' => $item->ID, 'title' => $item->title, 'url' => $item->url );
					else
						$sub[$item->menu_item_parent][] = array( 'id' => $item->ID, 'title' => $item->title, 'url' => $item->url );
				}
				for ($i=0; $i<count($items); $i++) {
					if (isset($sub[$items[$i]['id']])) 
						$items[$i]['items'] = $sub[$items[$i]['id']];
				}
				// debug($menu);
				$nav_menus[] = array( 'id' => $menu->term_id, 'slug' => $menu->slug, 'title' => $menu->name, 'items' => $items );
			}
		}
		// debug($nav_menus);

		// FSE navigation menus
		$all_navigations = array();
		if ( class_exists('\Greyd\Helper') ) {
			$navigations = \Greyd\Helper::get_all_posts('wp_navigation');
			if ($navigations) $all_navigations = $navigations;
		}

		// get all blocktypes
		$all_block_types = array();
		$block_types = \WP_Block_Type_Registry::get_instance()->get_all_registered();
		// debug($block_types);
		foreach($block_types as $name => $type) {
			if ($type->api_version < 2 && strpos($name, 'greyd/') !== 0) continue;
			if (!empty($type->parent)) continue;
			if (strpos($name, 'core/post-') > -1) continue;
			if (strpos($name, 'core/site-') > -1) continue;
			if (strpos($name, 'core/query-') > -1) continue;
			// debug($name);
			// debug($type);
			array_push($all_block_types, $name);
		}
		// debug($all_block_types);

		// url params
		$url_params = array();
		if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
			$url_params = \Greyd\Extensions\Cookie_Handler::get_supported_params();
		} else if ( class_exists('\url_handler') ) {
			$url_params = \url_handler::get_params();
		}
		
		// versions
		global $wp_version;
		$versions = array( "wp" => $wp_version );
		// theme
		$theme = is_child_theme() ? wp_get_theme(get_template()) : wp_get_theme();
		$versions[$theme->get_stylesheet()] = $theme->exists() ? $theme->get( 'Version' ) : '0';
		// plugins
		$plugins = class_exists('\Greyd\Helper') ? (object) \Greyd\Helper::active_plugins() : array();
		if ( !function_exists('get_plugin_data') ) require_once ABSPATH.'wp-admin/includes/plugin.php';
		foreach( $plugins as $plugin ) {
			$plugin_data = get_plugin_data( WP_PLUGIN_DIR."/".$plugin );
			$version = isset($plugin_data['Version']) && !empty($plugin_data['Version']) ? $plugin_data['Version'] : '0';
			$versions[$plugin_data['TextDomain']] = $version;
		}
		// debug($versions);
		// url params
		$url_params = array();
		if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
			$url_params = \Greyd\Extensions\Cookie_Handler::get_supported_params();
		} else if ( class_exists('\url_handler') ) {
			$url_params = \url_handler::get_params();
		}

		$data = array(
			// info
			'theme' => $theme->get_stylesheet(),
			'urls' => $urls,
			'icon_url' => $this->config->assets_url.'/icon',
			'language' => array( 'default' => $language_default, 'post' => $language_post ),
			'post_id' => $current_post->id,
			'post_name' => $current_post->name,
			'post_title' => $current_post->title,
			'post_type' => $current_post->post_type,
			'posts_per_page' => get_option('posts_per_page'),
			'template_type' => $template_type, // todo
			// content
			'post_types' => array_values($all_post_types),
			'all_posts' => $all_posts,
			'all_taxes' => $all_taxes,
			'media_urls' => $media_urls,
			'nav_menus' => $nav_menus,
			'navigation_menus' => $all_navigations,
			'all_block_types' => $all_block_types,
			'all_templates' => $all_templates, // todo
			'all_posttypes' => $all_public_posttypes, // todo
			'forms' => $all_forms, // todo
			'popups' => $all_popups, // todo
			// settings
			'settings' => class_exists('\Greyd\Settings') ? \Greyd\Settings::get_setting( array( 'site' ) ) : array(),
			'icons' => class_exists('\Greyd\Icons') ? \Greyd\Icons::get_icons() : icons::get_icons(),
			'url_params' => $url_params,
			'user_roles' => get_editable_roles(),
			'users' => get_users( array( 'fields' => array( 'ID', 'display_name' ) ) ),
			'plugins' => $plugins,
			'versions' => $versions,

			'is_greyd_classic' => self::is_greyd_classic(),
		);

		/**
		 * Filter Blockeditor Data values.
		 * The whole Data Array is added as inline javascript.
		 * It is accessible in the Blockeditor js script under the var greyd.data
		 * 
		 * @filter 'greyd_blocks_editor_data'
		 * 
		 * @param object $data     Original Data Array
		 * 
		 * @return object $data    Data Array with filtered Values
		 */
		return apply_filters( 'greyd_blocks_editor_data', $data );

	}

	/**
	 * Whether the classic 'GREYD.SUITE' Theme is active.
	 * @see \Greyd\Helper::is_greyd_classic()
	 */
	public static function is_greyd_classic() {
		if ( method_exists( '\Greyd\Helper', 'is_greyd_classic' ) ) {
			return \Greyd\Helper::is_greyd_classic();
		}

		// check if GREYD.SUITE is active
		if ( defined( 'GREYD_CLASSIC_VERSION' ) || class_exists("\basics") ) {
			return true;
		}

		// check if GREYD.Theme is active
		if ( defined( 'GREYD_THEME_CONFIG' ) ) {
			return false;
		}

		// check if GREYD.SUITE is installed
		$_current_main_theme = !empty( wp_get_theme()->parent() ) ? wp_get_theme()->parent() : wp_get_theme();
		return strpos( $_current_main_theme->get('Name'), "GREYD.SUITE" ) !== false;
	}

}
