/**
 * Greyd Block Editor Components for Layout Blocks.
 *
 * This file is loaded in the editor.
 */

greyd.components.layout = new function() {

	const el = wp.element.createElement;
	const { __ } = wp.i18n;

	this.HideBlockControl = class extends wp.element.Component {
		constructor() {
			super();
		}
		render() {

			var values = { xs: false, sm: false, md: false, lg: false, ...(this.props.value?? {}) };

			return [
				// label
				this.props.label ? el( 'label', {}, this.props.label ) : null,
				// controls
				el( 'div', { className: 'greyd-inspector-wrapper greyd-icons-inline' }, [
					el( 'div', { className: 'greyd-icon-flex xs' }, [
						el( greyd.components.GreydIcon, {
							icon: 'mobile',
							title: greyd.tools.makeBreakpointTitle("xs") }
						),
						el( wp.components.ToggleControl, {
							checked: values.xs,
							onChange: (value) => { 
								this.props.onChange( { ...values, xs: value } );
							},
						} ),
					] ),
					el( 'div', { className: 'greyd-icon-flex sm' }, [
						el( greyd.components.GreydIcon, {
							icon: 'tablet',
							title: greyd.tools.makeBreakpointTitle("sm") }
						),
						el( wp.components.ToggleControl, {
							checked: values.sm,
							onChange: (value) => { 
								this.props.onChange( { ...values, sm: value } );
							},
						} ),
					] ),
					el( 'div', { className: 'greyd-icon-flex md' }, [
						el( greyd.components.GreydIcon, {
							icon: 'laptop',
							title: greyd.tools.makeBreakpointTitle("md") }
						),
						el( wp.components.ToggleControl, {
							checked: values.md,
							onChange: (value) => { 
								this.props.onChange( { ...values, md: value } );
							},
						} ),
					] ),
					el( 'div', { className: 'greyd-icon-flex lg' }, [
						el( greyd.components.GreydIcon, {
							icon: 'desktop',
							title: greyd.tools.makeBreakpointTitle("lg") }
						),
						el( wp.components.ToggleControl, {
							checked: values.lg,
							onChange: (value) => {
								this.props.onChange( { ...values, lg: value } );
							},
						} ),
					] ),
				] ),
				// help
				this.props.help ? el( 'p', { className: "greyd-inspector-help" }, this.props.help ) : null,
			];
		}
	};

	this.GridControl = class extends wp.element.Component {

		constructor() {
			super();
		}

		render() {
			const {
				label = '',
				min = 1,
				max = 12,
				step = 1,
				defaultValue = '',
				onChange = () => {},
				options = [
					{
						label: __('Erben ↓', 'greyd_blocks'),
						value: ''
					},
					{
						label: __('Automatisch', 'greyd_blocks'),
						value: 'auto'
					}
				]
			} = this.props;

			const value = this.props.value.replace(/(col|offset)-([a-z]{2}-)?/, '');

			// get slider steps
			const count = max - min;
			let steps = [];
			for ( let i=0; i<=count; i++ ) {

				if ( i % step ) continue;

				const num = min + i;
				const isActive = !isNaN(num) && num <= value;
				steps.push( el( 'span', {
					className: 'greyd-grid-control--slider-thumb' + ( isActive ? ' is-active' : '' ),
					title: num,
				} ) )
			}

			return el( 'div', {
				className: 'greyd-grid-control'
			}, [

				( isEmpty(label) ? null : el( greyd.components.AdvancedLabel, {
					label: label + ( value === defaultValue ? '' : ': ' + value ),
					currentValue: value,
					initialValue: defaultValue,
					onClear: () => onChange( '' )
				} ) ),

				el( 'div', {
					className: 'greyd-grid-control--slider'
				}, [
					...steps,
					el( 'input', {
						type: 'range',
						min: min * step,
						max: max,
						step: step,
						value: isNaN(value) ? 0 : value,
						onChange: () => onChange( event.target.value )
					} )
				] ),

				(
					options.length
					? el( 'div', {
						className: 'greyd-grid-control--buttons'
					}, [
						...options.map( (option) => {
							return el( wp.components.Button, {
								className: option.value === value ? 'is-active' : '',
								onClick: () => onChange( option.value )
							}, option.label )
						} )
					] )
					: null
				)


			] )
		}
	}

	//
	// background controls
	this.backgroundInspectorControls = function(props, sets={}) {

		// vars
		if (!_.has(sets, 'overlay')) sets.overlay = true;
		if (!_.has(sets, 'pattern')) sets.pattern = true;
		if (!_.has(sets, 'seperator')) sets.seperator = true;
		var values = greyd.tools.layout.getBackgroundDefaults(sets);
		if (_.has(props.attributes, "background")) {
			values = greyd.tools.getValues(values, props.attributes.background);
		}

		// console.log(values);
		var [parts, setParts ] = wp.element.useState(values);
		// console.log(parts);

		var setBackground = function(val) {
			// console.log(val);
			if (_.has(val.background, 'type')) parts.background = val.background;
			if (_.has(val.background, 'overlay')) parts.overlay = val.background.overlay;
			if (_.has(val.background, 'pattern')) parts.pattern = val.background.pattern;
			if (_.has(val.background, 'seperator')) parts.seperator = val.background.seperator;
			setParts(parts);
			// make background value
			var background = greyd.tools.makeValues(greyd.tools.layout.getBackgroundDefaults(sets), parts);
			// cleaning
			background = greyd.tools.layout.cleanBackgroundvalues(background);
			// console.log(background);
			props.setAttributes( { background: background } );
		}

		return [
			el( greyd.components.AdvancedPanelBody, {
				title: __( 'Hintergrundelement', 'greyd_blocks' ),
				initialOpen: false,
				holdsChange: parts.background.type != ""
			}, [
				greyd.components.layout.backgroundControls( {
					clientId: props.clientId,
					attributes: { background: parts.background },
					setAttributes: setBackground
				} )
			] ),
			sets.overlay && el( greyd.components.AdvancedPanelBody, {
				title: __( 'Überlagerung', 'greyd_blocks' ),
				initialOpen: false,
				// holdsChange: parts.overlay.type != "",
				holdsColors: [
					parts.overlay.type == "color" && { color: parts.overlay.color },
					parts.overlay.type == "gradient" && { color: parts.overlay.gradient },
				]
			}, [
				greyd.components.layout.overlayControls( {
					clientId: props.clientId,
					attributes: { background: { overlay: parts.overlay } },
					setAttributes: setBackground
				} )
			] ),
			sets.pattern && el( greyd.components.AdvancedPanelBody, {
				title: __( 'Muster', 'greyd_blocks' ),
				initialOpen: false,
				holdsChange: parts.pattern.type != ""
			}, [
				greyd.components.layout.patternControls( {
					clientId: props.clientId,
					attributes: { background: { pattern: parts.pattern } },
					setAttributes: setBackground
				} )
			] ),
			sets.seperator && el( greyd.components.AdvancedPanelBody, {
				title: __( 'Trenner', 'greyd_blocks' ),
				initialOpen: false,
				// holdsChange: parts.seperator.type != "",
				holdsColors: [
					parts.seperator.type != "" && { color: parts.seperator.color },
				]
			}, [
				greyd.components.layout.seperatorControls( {
					clientId: props.clientId,
					attributes: { background: { seperator: parts.seperator } },
					setAttributes: setBackground
				} )
			] )
		];
	}

	this.backgroundControls = function(props) {

		var types = [
			{ label: __('✕', 'greyd_blocks'), value: "" },
			{ label: __('Bild', 'greyd_blocks'), value: 'image' },
			{ label: __('Video', 'greyd_blocks'), value: 'video' },
		];
		if (typeof bodymovin !== 'undefined') {
			types.push( { label: __('Animation', 'greyd_blocks'), value: 'animation' } );
		}

		if (props.attributes.background.type == "animation") {
			// console.log("init background anim");
			greyd.lottie.init(['inspector_anim_'+props.clientId, 'anim_'+props.clientId]);
		}

		// if (props.attributes.background.image.id.toString().indexOf('_') === 0) {
		// 	console.log(props.attributes.background.image);
		// 	var url = 'http://greyd_beta.localhost/wp-content/uploads/2020/11/greyd-LogoPh-dark.svg';
		// 	if (props.attributes.background.image.url != url)
		// 		props.setAttributes( { background: { ...props.attributes.background, image: { ...props.attributes.background.image, url: url } } } )
		// }

		return [
			// type
			el( greyd.components.ButtonGroupControl, {
				value: props.attributes.background.type,
				options: types,
				onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, type: value } } ); },
			} ),
			el( 'div', { className: (props.attributes.background.type == "") ? 'hidden' : "" }, [
				// image
				el( 'div', { className: (props.attributes.background.type != "image") ? 'hidden' : "" }, [
					el( greyd.components.DynamicImageControl, {
						clientId: props.clientId,
						value: props.attributes.background.image,
						useIdAsTag: true,
						onChange: (value) => {
							// console.log(value);
							props.setAttributes({
								background: {
									...props.attributes.background,
									image: {
										...props.attributes.background.image,
										...value
									}
								}
							});
						}
					} ),
					el( 'div', { }, [
						// size
						el( wp.components.SelectControl, {
							label: __('Größe', 'greyd_blocks'),
							value: props.attributes.background.image.size,
							options: [
								{ label: __('vollständig ausfüllen (cover)', 'greyd_blocks'), value: 'cover' },
								{ label: __('einpassen (contain)', 'greyd_blocks'), value: 'contain' },
								{ label: __('Originalgröße (initial)', 'greyd_blocks'), value: 'initial' },
							],
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, image: { ...props.attributes.background.image, size: value } } } ); },
						} ),
						// repeat
						el( wp.components.SelectControl, {
							label: __('Wiederholung', 'greyd_blocks'),
							value: props.attributes.background.image.repeat,
							options: [
								{ label: __('nicht wiederholen', 'greyd_blocks'), value: 'no-repeat' },
								{ label: __('vertikal & horizontal', 'greyd_blocks'), value: 'repeat' },
								{ label: __('horizontal', 'greyd_blocks'), value: 'repeat-x' },
								{ label: __('vertikal', 'greyd_blocks'), value: 'repeat-y' },
							],
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, image: { ...props.attributes.background.image, repeat: value } } } ); },
						} ),
						// position
						el( wp.components.SelectControl, {
							label: __('Position', 'greyd_blocks'),
							value: props.attributes.background.image.position,
							options: [
								{ label: __('oben mittig', 'greyd_blocks'), value: "top center" },
								{ label: __('oben links', 'greyd_blocks'), value: 'top left' },
								{ label: __('oben rechts', 'greyd_blocks'), value: 'top right' },
								{ label: __('mittig', 'greyd_blocks'), value: "center center" },
								{ label: __('mittig links', 'greyd_blocks'), value: 'center left' },
								{ label: __('mittig rechts', 'greyd_blocks'), value: 'center right' },
								{ label: __('unten mittig', 'greyd_blocks'), value: "bottom center" },
								{ label: __('unten links', 'greyd_blocks'), value: 'bottom left' },
								{ label: __('unten rechts', 'greyd_blocks'), value: 'bottom right' },
							],
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, image: { ...props.attributes.background.image, position: value } } } ); },
						} ),
					] ),
				] ),
				// video
				el( 'div', { className: (props.attributes.background.type != "video") ? 'hidden' : "" }, [
					// url
					el( wp.components.TextControl, {
						label: __('Video Link', 'greyd_blocks'),
						help: __('YouTube oder Vimeo URL', 'greyd_blocks'),
						value: props.attributes.background.video.url,
						onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, video: { ...props.attributes.background.video, url: value } } } ); },
					} ),
					// aspect
					el( wp.components.SelectControl, {
						label: __('Seitenverhältnis', 'greyd_blocks'),
						value: props.attributes.background.video.aspect,
						options: [
							{ label: __('16/9', 'greyd_blocks'), value: "16/9" },
							{ label: __('21/9', 'greyd_blocks'), value: "21/9" },
							{ label: __('4/3', 'greyd_blocks'), value: "4/3" },
							{ label: __('1/1', 'greyd_blocks'), value: "1/1" },
						],
						onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, video: { ...props.attributes.background.video, aspect: value } } } ); },
					} ),
				] ),
				// animation
				el( 'div', {
					className: (typeof bodymovin === 'undefined' || props.attributes.background.type != "animation") ? 'hidden' : ""
				}, [
					el( wp.components.BaseControl, { }, [
						el( wp.blockEditor.MediaUploadCheck, {
							fallback: el( 'p', {
								className: "greyd-inspector-help"
							}, __('Um die Animation zu bearbeiten, musst zu zum Upload von Medien berechtigt sein.', 'greyd_blocks') )
						}, [
							el( wp.blockEditor.MediaUpload, {
								allowedTypes: 'application/json',
								value: props.attributes.background.anim.id,
								onSelect: function(media) {
									greyd.lottie.set(['inspector_anim_'+props.clientId, 'anim_'+props.clientId], media.url);
									props.setAttributes( { background: { ...props.attributes.background, anim: { ...props.attributes.background.anim, id: media.id, url: media.url } } } );
								},
								render: function(obj) {
									return el( wp.components.Button, {
										className: !props.attributes.background.anim.url ? 'editor-post-featured-image__toggle' : 'editor-post-featured-image__preview',
										onClick: obj.open
									}, !props.attributes.background.anim.url ? __( 'Animation wählen', 'greyd_blocks' ) : el( 'div', {
										id: 'inspector_anim_'+props.clientId,
										className: 'lottie-animation auto',
										"data-anim": 'auto',
										'data-src': props.attributes.background.anim.url
									} ) )
								},
							} ),
							props.attributes.background.anim.url ? el( wp.components.Button, {
								className: "is-link is-destructive",
								onClick: function() { props.setAttributes( { background: { ...props.attributes.background, anim: { ...props.attributes.background.anim, id: false, url: false } } } ) },
							}, __( 'Animation entfernen', 'greyd_blocks' ) ) : ""
						] ),
					] ),
					el( 'div', { className: (!props.attributes.background.anim.url) ? 'hidden' : "" }, [
						// anim
						el( 'span', {}, __('Animation starten', 'greyd_blocks') ),
						el( greyd.components.ButtonGroupControl, {
							value: props.attributes.background.anim.anim,
							options: [
								{ label: __('automatisch', 'greyd_blocks'), value: 'auto' },
								{ label: __('sobald sichtbar', 'greyd_blocks'), value: 'visible' },
							],
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, anim: { ...props.attributes.background.anim, anim: value } } } ); },
						} ),
						// position
						el( wp.components.SelectControl, {
							label: __('Position', 'greyd_blocks'),
							value: props.attributes.background.anim.position,
							options: [
								{ label: __('oben mittig', 'greyd_blocks'), value: "top_center" },
								{ label: __('oben links', 'greyd_blocks'), value: 'top_left' },
								{ label: __('oben rechts', 'greyd_blocks'), value: 'top_right' },
								{ label: __('mittig', 'greyd_blocks'), value: "center_center" },
								{ label: __('mittig links', 'greyd_blocks'), value: 'center_left' },
								{ label: __('mittig rechts', 'greyd_blocks'), value: 'center_right' },
								{ label: __('unten mittig', 'greyd_blocks'), value: "bottom_center" },
								{ label: __('unten links', 'greyd_blocks'), value: 'bottom_left' },
								{ label: __('unten rechts', 'greyd_blocks'), value: 'bottom_right' },
							],
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, anim: { ...props.attributes.background.anim, position: value } } } ); },
						} ),
						// width
						el( greyd.components.RangeUnitControl, {
							label: __('Breite', 'greyd_blocks'),
							value: props.attributes.background.anim.width,
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, anim: { ...props.attributes.background.anim, width: value } } } ); },
						} ),
					] ),
				] ),

				// opacity
				el( wp.components.RangeControl, {
					label: __('Deckkraft', 'greyd_blocks'),
					value: props.attributes.background.opacity,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, opacity: value } } ); },
				} ),
				// scroll
				el( wp.components.SelectControl, {
					label: __('Scroll-Verhalten', 'greyd_blocks'),
					value: props.attributes.background.scroll.type,
					options: [
						{ label: __('mit Inhalt bewegen', 'greyd_blocks'), value: "scroll" },
						{ label: __('fixieren', 'greyd_blocks'), value: 'fixed' },
						{ label: __('Parallax vertikal', 'greyd_blocks'), value: 'vparallax' },
						{ label: __('Parallax horizontal', 'greyd_blocks'), value: 'hparallax' },
					],
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, scroll: { ...props.attributes.background.scroll, type: value } } } ); },
				} ),
				(
					_.has(props.attributes.background.scroll, 'type') && props.attributes.background.scroll.type === 'fixed'
					? el( wp.components.CheckboxControl, {
						label: __('auf mobilen Geräten aktivieren', 'greyd_blocks'),
						help: __( 'Achtung: Auf iOS Geräten können fixierte Hintergründe zu Darstellungsfehlern führen. Bitte teste deine Seite genau, solltest du diese Einstellung aktivieren.', 'greyd_blocks' ),
						checked: props.attributes.background.scroll.parallax_mobile,
						onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, scroll: { ...props.attributes.background.scroll, parallax_mobile: value } } } ); },
					} )
					: null
				),
				(
					_.has(props.attributes.background.scroll, 'type') && props.attributes.background.scroll.type.indexOf('parallax') !== -1
					? el( 'div', { }, [
						el( wp.components.CheckboxControl, {
							label: __('auf mobilen Geräten aktivieren', 'greyd_blocks'),
							checked: props.attributes.background.scroll.parallax_mobile,
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, scroll: { ...props.attributes.background.scroll, parallax_mobile: value } } } ); },
						} ),
						el( wp.components.RangeControl, {
							label: __('Parallax Geschwindigkeit', 'greyd_blocks'),
							help: __('-200% = rückwärts, 0% = fixiert, 200% = doppelt so schnell wie normal', 'greyd_blocks'),
							min: -200,
							max: 200,
							value: props.attributes.background.scroll.parallax,
							onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, scroll: { ...props.attributes.background.scroll, parallax: value } } } ); },
						} ),
					] )
					: null
				)
			] ),
		];

	}

	/**
	 * Get columns overlay controls.
	 * @param {object} props     Block properties
	 * @returns React Element
	 */
	this.overlayControls = function(props) {

		const background = _.has( props.attributes, 'background' ) ? props.attributes.background : {};
		const overlay = _.has( background, 'overlay' ) ? background.overlay : {};
		const {
			type = '',
			color = null,
			gradient = null,
			opacity = 100
		} = overlay;

		return [
			// type
			el( greyd.components.ButtonGroupControl, {
				value: overlay.type,
				options: [
					{ label: __("✕", "greyd_blocks"), value: "" },
					{ label: __("Einfarbig", "greyd_blocks"), value: "color" },
					{ label: __("Farbverlauf", "greyd_blocks"), value: "gradient" },
				],
				onChange: ( val ) => {
					props.setAttributes( { background: { ...background, overlay: {
						...overlay,
						type: val
					} } } );
				}
			} ),
			!_.isEmpty( type ) && el( 'div', { }, [
				// color
				type === 'color' && el( 'div', {}, [
					el( greyd.components.ColorGradientPopupControl, {
						mode: 'color',
						label: __("Farbe", "greyd_blocks"),
						value: color,
						onChange: ( val ) => {
							props.setAttributes( { background: { ...background, overlay: {
								...overlay,
								color: val
							} } } );
						}
					} ),
				] ),
				// gradient
				type === 'gradient' && el( 'div', {}, [
					el( greyd.components.ColorGradientPopupControl, {
						mode: 'gradient',
						label: __("Gradient", "greyd_blocks"),
						value: gradient,
						onChange: ( val ) => {
							props.setAttributes( { background: { ...background, overlay: {
								...overlay,
								gradient: val
							} } } );
						}
					} ),
				] ),
				// opacity
				el( wp.components.RangeControl, {
					label: __('Deckkraft', 'greyd_blocks'),
					value: opacity,
					onChange: ( val ) => {
						props.setAttributes( { background: { ...background, overlay: {
							...overlay,
							opacity: val
						} } } );
					}
				} ),
			] ),
		];

	}

	this.patternControls = function(props) {

		var getstyle = function(key) {
			return { backgroundImage: 'url('+greyd.tools.layout.getPattern(key)+')' };
		}
		var options = [
			{ key: '', name: __('kein Muster', 'greyd_blocks') },
			{ key: 'pattern_01', name: __('kleine Punkte', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_01') },
			{ key: 'pattern_02', name: __('große Punkte', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_02') },
			{ key: 'pattern_03', name: __('Gitterlinien', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_03') },
			{ key: 'pattern_04', name: __('Kreuze', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_04') },
			{ key: 'pattern_05', name: __('horizontale Linien', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_05') },
			{ key: 'pattern_06', name: __('vertikale Linien', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_06') },
			{ key: 'pattern_07', name: __('Quadrate und Punkte', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_07') },
			{ key: 'pattern_08', name: __('kleine Quadrate', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_08') },
			{ key: 'pattern_09', name: __('große Quadrate', 'greyd_blocks'), className: 'custom-select-pattern-type', style: getstyle('pattern_09') },
			{ key: 'pattern_custom', name: __('individuelles Muster', 'greyd_blocks') },
		];
		var type = props.attributes.background.pattern.type;
		for (var i=0; i<options.length; i++) {
			if (options[i].key == type) {
				type = options[i];
				break;
			}
		}

		return [
			// type
			el( wp.components.BaseControl, { }, [
				el( wp.components.CustomSelectControl, {
					options: options,
					value: type,
					onChange: function(value) {
						// console.log(value);
						var val = value.selectedItem.key;
						props.setAttributes( {
							background: {
								...props.attributes.background,
								pattern: {
									...props.attributes.background.pattern,
									type: val
								}
							}
						} );
					},
				} ),
			] ),
			el( 'div', { className: (props.attributes.background.pattern.type == "") ? 'hidden' : "" },
				el( 'div', { className: (props.attributes.background.pattern.type != "pattern_custom") ? 'hidden' : "" },
					// custom
					el( wp.components.BaseControl, { },
						el( wp.blockEditor.MediaUploadCheck, { fallback: el( 'p', { className: "greyd-inspector-help" }, __('Um das Hintergrundbild zu bearbeiten, musst du zum Upload von Medien berechtigt sein.', 'greyd_blocks') ) },
							el( wp.blockEditor.MediaUpload, {
								allowedTypes: 'image/*',
								value: props.attributes.background.pattern.id,
								onSelect: function(value) { props.setAttributes( { background: { ...props.attributes.background, pattern: { ...props.attributes.background.pattern, id: value.id, url: value.url } } } ); },
								render: function(obj) {
									return el( wp.components.Button, {
										className: props.attributes.background.pattern.id == -1 ? 'editor-post-featured-image__toggle' : 'editor-post-featured-image__preview',
										onClick: obj.open
									}, props.attributes.background.pattern.id == -1 ? __( 'Muster wählen', 'greyd_blocks' ) : el( 'img', { src: props.attributes.background.pattern.url } ) )
								},
							} ),
							props.attributes.background.pattern.id !== -1 ? el( wp.components.Button, {
								className: "is-link is-destructive",
								onClick: function() { props.setAttributes( { background: { ...props.attributes.background, pattern: { ...props.attributes.background.pattern, id: -1, url: "" } } } ); },
							}, __( 'Muster entfernen', 'greyd_blocks' ) ) : ""
						),
					),
				),
				// size
				el( greyd.components.RangeUnitControl, {
					label: __('Größe', 'greyd_blocks'),
					value: props.attributes.background.pattern.size,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, pattern: { ...props.attributes.background.pattern, size: value } } } ); },
				} ),
				// opacity
				el( wp.components.RangeControl, {
					label: __('Deckkraft', 'greyd_blocks'),
					value: props.attributes.background.pattern.opacity,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, pattern: { ...props.attributes.background.pattern, opacity: value } } } ); },
				} ),
				// scroll
				el( wp.components.SelectControl, {
					label: __('Scroll Verhalten', 'greyd_blocks'),
					value: props.attributes.background.pattern.scroll,
					options: [
						{ label: __('mit Inhalt bewegen', 'greyd_blocks'), value: "scroll" },
						{ label: __('fixieren', 'greyd_blocks'), value: 'fixed' },
					],
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, pattern: { ...props.attributes.background.pattern, scroll: value } } } ); },
				} ),
			),
		];

	}

	this.seperatorControls = function(props) {

		const getSeperatorName = function(key, title) {
			var svg =  greyd.tools.layout.getSeperator(key, '20px', '#888');
			return el( 'span', { style: { width: '100%', lineHeight: 'initial' } }, svg, el( 'span', {}, title ) );
		}
		let options = [
			{ key: "", name: __("Kein Trennelement", 'greyd_blocks') }
		];
		const seperators = Object.values( greyd.tools.layout.getSeperators() );
		for (var i=0; i<seperators.length; i++) {
			options.push( {
				key: seperators[i].slug,
				name: getSeperatorName(seperators[i].slug, seperators[i].title),
				className: 'custom-select-seperator-type' }
			);
		}

		var type = props.attributes.background.seperator.type;
		for (var i=0; i<options.length; i++) {
			if (options[i].key == type) {
				type = options[i];
				break;
			}
		}

		return [
			// type
			el( wp.components.BaseControl, { }, [
				el( wp.components.CustomSelectControl, {
					className: 'seperatorselect',
					options: options,
					value: type,
					onChange: function(value) {
						// console.log(value);
						var val = value.selectedItem.key;
						props.setAttributes( {
							background: {
								...props.attributes.background,
								seperator: {
									...props.attributes.background.seperator,
									type: val
								}
							}
						} );
					},
				} ),
			] ),
			el( 'div', { className: (props.attributes.background.seperator.type == "") ? 'hidden' : "" }, [
				// color
				el( greyd.components.ColorGradientPopupControl, {
					className: 'single',
					mode: 'color',
					label: __("Farbe", "greyd_blocks"),
					value: props.attributes.background.seperator.color,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, seperator: { ...props.attributes.background.seperator, color: value } } } ); },
				} ),
				// position
				el( wp.components.SelectControl, {
					label: __('Position', 'greyd_blocks'),
					value: props.attributes.background.seperator.position,
					options: [
						{ label: __("oben", 'greyd_blocks'), value: "top" },
						{ label: __("unten", 'greyd_blocks'), value: "bottom" },
						{ label: __("oben & unten", 'greyd_blocks'), value: "top_bottom" },
						{ label: __("oben & unten asymmetrisch", 'greyd_blocks'), value: "top_bottom_asym" },
					],
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, seperator: { ...props.attributes.background.seperator, position: value } } } ); },
				} ),
				// height
				el( greyd.components.RangeUnitControl, {
					label: __('Höhe', 'greyd_blocks'),
					value: props.attributes.background.seperator.height,
					max: 500,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, seperator: { ...props.attributes.background.seperator, height: value } } } ); },
				} ),
				// opacity
				el( wp.components.RangeControl, {
					label: __('Deckkraft', 'greyd_blocks'),
					value: props.attributes.background.seperator.opacity,
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, seperator: { ...props.attributes.background.seperator, opacity: value } } } ); },
				} ),
				// zpos
				el( wp.components.SelectControl, {
					label: __('Ebene (z-Index)', 'greyd_blocks'),
					value: props.attributes.background.seperator.zposition,
					options: [
						{ label: __("zwischen Hintergrund und Inhalt", 'greyd_blocks'), value: "0" },
						{ label: __("über Hintergrund und Inhalt", 'greyd_blocks'), value: "1" },
					],
					onChange: function(value) { props.setAttributes( { background: { ...props.attributes.background, seperator: { ...props.attributes.background.seperator, zposition: value } } } ); },
				} ),
			] ),
		];

	}

	// console.log( "additional layout components: loaded" );

};
