<?php
/**
 * Manage core plugin functionality.
 * 
 * Such as activation, updates, translations, dependencies and more.
 */
namespace greyd\blocks;

if( ! defined( 'ABSPATH' ) ) exit;

new manage($config);
class manage {

	public $config;

	/**
	 * Constructor.
	 */
	public function __construct($config) {
		// set config
		$this->config = (object)$config;
		add_action( 'after_setup_theme', array($this, 'init'), 5 );

		// translations
		add_action( 'plugins_loaded', array($this, 'greyd_textdomain') );

		// display notices
		add_action( 'admin_notices', array($this, 'display_admin_notices') );
		add_action( 'network_admin_notices', array($this, 'display_admin_notices') );

		// set required versions
		// deprecated - remove in next version
		add_filter( 'greyd_theme_required_versions', array($this, 'set_required_theme_version') );
		add_filter( 'greyd_hub_required_versions', array($this, 'set_required_hub_version') );
		add_filter( 'greyd_forms_required_versions', array($this, 'set_required_form_version') );

		// modify user capabilities
		if (is_multisite()) add_filter( 'map_meta_cap', array($this, 'mod_user_capabilities'), 1, 3 );

	}

	/**
	 * Init the class.
	 */
	public function init() {
		// manage updates
		if (class_exists('Puc_v4_Factory') && !$this->is_frozen()) {
			$myUpdateChecker = \Puc_v4_Factory::buildUpdateChecker(
				$this->config->update_file,     // Full path to the update server manifest file.
				$this->config->plugin_file,     // Full path to the main plugin file or functions.php.
				$this->config->plugin_name      // Insert Plugin slug, usually same as directory name.
			);
		}
	}
	
	/**
	 * Register translations.
	 */
	public function greyd_textdomain() {
		load_plugin_textdomain( 'greyd_blocks', false, dirname( plugin_basename(__FILE__) ) . '/languages/' );
	}

	/**
	 * Display admin notices on activation, updates and more.
	 */
	public function display_admin_notices() {

		// show update notice
		if ( $this->plugin_needs_update() && ! ( isset($_GET['action']) && $_GET['action'] === 'do-plugin-upgrade' ) ) {
			echo "<div class='error notice'>
				<p>".sprintf(
					__("Das Plugin %s benötigt eine neuere Version. Bitte %saktualisiere das Plugin%s, um mögliche Fehler zu verhindern.", 'greyd_blocks'),
					"<strong>".$this->config->plugin_name_full."</strong>",
					"<a href='".( is_multisite() ? network_admin_url('update-core.php') : admin_url('update-core.php') )."'>",
					"</a>"
				)."</p>
			</div>";
		}

		// hub plugin not active
		if ( ! defined( 'GREYD_VERSION' ) ) {

			// cache is used to not display the notice in other plugins of the suite
			if ( wp_cache_get( 'greyd_hub_plugin_notice_displayed', 'greyd' ) ) return;

			?>
			<div class="notice notice-warning">
				<div style="padding:12px">
					<p><strong><?php
						_e("Setup Unvollständig", "greyd_blocks" );
					?></strong></p>
					<p><?php echo sprintf(
						__("Das Greyd Blocks Plugin funktioniert nur in Zusammenspiel mit dem %sGREYD Hub Plugin%s. Lade das Plugin herunter und aktiviere es.", "greyd_blocks" ),
						'<strong>',
						'</strong>'
					);
					?></p>
					<p>
						<a href="<?php echo esc_url( admin_url( 'plugins.php' ) ) ?>"><?php echo __("Zu den Plugins", "greyd_blocks"); ?></a>&nbsp;&nbsp;|&nbsp;&nbsp;<a style="color: inherit;" href="https://update.greyd.io/public/plugins/greyd_hub/greyd_tp_management.zip"><?php echo __("Plugin Herunterladen", "greyd_blocks"); ?></a>
					</p>
				</div>
			</div>
			<?php
			wp_cache_set( 'greyd_hub_plugin_notice_displayed', true, 'greyd' );
		}
	}

	/**
	 * Check if this plugin needs an update
	 * 
	 * @return bool Returns true if this plugin needs an update.
	 */
	public function plugin_needs_update() {

		/**
		 * Get required versions
		 * @deprecated
		 * 
		 * @filter greyd_blocks_required_versions
		 */
		$requirements = apply_filters( 'greyd_blocks_required_versions', array() );

		/**
		 * Get required versions of this plugin
		 * 
		 * @filter greyd_versions
		 */
		$versions = apply_filters( 'greyd_versions', array() );
		foreach( $versions as $n => $v ) {
			if ( isset($v['required'][$this->config->plugin_name]) ) {
				$requirements[$n] = $v['required'][$this->config->plugin_name];
			}
		}

		if ( empty($requirements) || !is_array($requirements) ) return false;
        // debug( 'blocks requirements' );
		// debug( $requirements );

		if ( !function_exists('get_plugin_data') ) require_once(ABSPATH.'wp-admin/includes/plugin.php');
		
		$all_true           = true;
		$plugin_data        = get_plugin_data( $this->config->plugin_file );
		$current_version    = isset($plugin_data['Version']) && !empty($plugin_data['Version']) ? $plugin_data['Version'] : '0';
		foreach ( $requirements as $required_version ) {
			if ( $all_true ) {
				$all_true = version_compare( $current_version, $required_version, '>=' );
			}
		}
		return !$all_true;
	}

	/**
	 * Add required GREYD.Theme Version
	 * 
	 * @filter greyd_suite_required_versions
	 */
	public function set_required_theme_version($return) {
		$versions = isset($this->config->required_versions) ? $this->config->required_versions : array();
		$version = isset($versions['greyd_suite']) ? $versions['greyd_suite'] : null;
		if ( $version ) {
			$return[$this->config->plugin_name] = $version;
		}
		return $return;
	}

	/**
	 * Add required GREYD.Hub Version
	 * 
	 * @filter greyd_hub_required_versions
	 */
	public function set_required_hub_version($return) {
		$versions = isset($this->config->required_versions) ? $this->config->required_versions : array();
		$version = isset($versions['greyd_hub']) ? $versions['greyd_hub'] : null;
		if ( $version ) {
			$return[$this->config->plugin_name] = $version;
		}
		return $return;
	}

	/**
	 * Add required GREYD.Forms Version
	 * 
	 * @filter greyd_forms_required_versions
	 */
	public function set_required_form_version($return) {
		$versions = isset($this->config->required_versions) ? $this->config->required_versions : array();
		$version = isset($versions['greyd_forms']) ? $versions['greyd_forms'] : null;
		if ( $version ) {
			$return[$this->config->plugin_name] = $version;
		}
		return $return;
	}

	/**
	 * Enable unfiltered_html capability for Admins and Editors on Multisites.
	 * Needed to save styles in post content
	 *
	 * @param  array  $caps    The user's capabilities.
	 * @param  string $cap     Capability name.
	 * @param  int    $user_id The user ID.
	 * @return array  $caps    The modified user's capabilities.
	 */
	public function mod_user_capabilities($caps, $cap, $user_id) {
		if ($cap === 'unfiltered_html') {
			if (user_can( $user_id, 'administrator' ) || user_can( $user_id, 'editor' )) {
				$caps = array( 'unfiltered_html' );
			}
		}
		return $caps;
	}

	public function is_frozen() {
		$l = get_option('gtpl');
		return $l && !empty($l) && is_array($l) && isset($l['frozen']) && $l['frozen'];
	}
}