/**
 * Form inputs
 */
( function () {

	// return if we're not editing a form
	if ( greyd.data.post_type !== greyd.forms.data.config.plugin_post_type ) return false;

	// basics
	const {
		blocks,
		element,
		i18n,
		hooks,
	} = window.wp;
	const el = element.createElement;
	const { __, _x } = i18n;

	const {
		PanelBody,
		SelectControl,
		RangeControl,
		TextControl,
		__experimentalNumberControl: NumberControl,
		ToggleControl,
		Tip,
		TextareaControl,
		FontSizePicker,
		FormTokenField,
		BaseControl
	} = wp.components;

	const {
		LabelTooltip,
		InputNameControl,
		IconPanelImage
	} = greyd.forms.components;

	const {
		makeInputName,
		sanitizeTitle,
		stripTags,
		getTooltip,
		getValidationPattern
	} = greyd.forms.helper;

	const getGreydClass = (
		typeof greyd.tools.getGreydClass !== 'undefined' ?
			greyd.tools.getGreydClass :
			( blockProps ) => {
				return greyd.tools.generateGreydClass();
			}
	);

	// placeholder to replace when rendering (keep same as in render.php)
	const renderPlaceholder = {
		agb: "<!--agb_text-->",
		id: "___idXXX",
		captcha: "XXXcaptcha_sitekeyXXX",
		tooltip: "{tooltip}",
		enter: "{enter}",
		select: "{select}",
		required: "{required}",
		options: "{options}",
	};

	/**
	 * Always enable the customized preview by default inside forms
	 */
	hooks.addFilter(
		'greyd.previewDefault',
		'greyd/forms/inputs',
		function ( value, post_id, post_type ) {
			return {
				enabled: true,
				maxWidth: "",
				backgroundColor: "",
			};
		}
	);

	/**
	 * Input block
	 */
	blocks.registerBlockType( 'greyd-forms/input', {
		title: __( "Input field", 'greyd_forms' ),
		description: __( "For entering text, email, date, number...", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_input.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		styles: [
			{
				name: 'prim',
				label: __( "primary field", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( "secondary field", 'greyd_forms' )
			},
		],
		attributes: {
			// main
			name: { type: "string", default: "" },

			// input
			type: { type: "string", default: 'text' },
			required: { type: "boolean", default: false },
			placeholder: { type: "string", default: '' },
			autocomplete: { type: "string", default: 'on' },
			correctAnswer: { type: "string", default: ''},
			restrictions: {
				type: 'object', properties: {
					minlength: { type: "number" },
					maxlength: { type: "number" },
					max: { type: "string" },
					min: { type: "string" },
					start: { type: "string" },
					step: { type: "number" },
				}, default: {}
			},
			area: { type: "boolean", default: false },
			rows: { type: "number", default: 5 },

			// label
			label: {
				type: 'string',
				source: 'html',
				selector: 'span',
				default: ''
			},
			labelStyles: { type: "object", default: {} },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// styling
			greydStyles: { type: "object" },
			greydClass: { type: "string", default: "" },
			customStyles: { type: 'object' },
			custom: { type: 'boolean', default: false }
		},
		example: {
			attributes: {
				type: 'email',
				label: __( "email", 'greyd_forms' ),
				placeholder: __( "john@example.com", 'greyd_forms' ),
				required: true,
			}
		},

		edit: function ( props ) {

			// generate attributes
			props.attributes.greydClass = getGreydClass( props );

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( atts.type, atts ) : atts.tooltip.content;

			const parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/multistep-container" );
			const parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
			const quizMode = parent?.attributes.multistepMode == "quiz";

			/**
			 * Convert incorrect address autocomplete names
			 * @since 1.3.9
			 */
			if ( atts.autocomplete == 'adress-level2' ) {
				props.setAttributes( { autocomplete: 'address-level2' } );
			}
			else if ( atts.autocomplete == 'street-adress' ) {
				props.setAttributes( { autocomplete: 'street-address' } );
			}

			return [

				el( InspectorControls, {}, [

					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// Feldtyp
						el( SelectControl, {
							label: __( "Field type", 'greyd_forms' ),
							help: __( "What should the user enter here?", 'greyd_forms' ),
							value: atts.type,
							options: [
								{ label: __( "text", 'greyd_forms' ), value: "text" },
								{ label: __( "email", 'greyd_forms' ), value: "email" },
								{ label: __( "phone number", 'greyd_forms' ), value: "tel" },
								{ label: __( "link", 'greyd_forms' ), value: "url" },
								{ label: __( "number", 'greyd_forms' ), value: "number" },
								{ label: '--------', value: "", disabled: true },
								{ label: __( "time", 'greyd_forms' ), value: "time" },
								{ label: __( "date", 'greyd_forms' ), value: "date" },
								{ label: __( "date & time", 'greyd_forms' ), value: "datetime-local" },
								{ label: __( "week", 'greyd_forms' ), value: "week" },
								{ label: __( "month", 'greyd_forms' ), value: "month" },
							],
							onChange: ( value ) => props.setAttributes( { type: value } )
						} ),

						// Feldname
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: atts.type == 'text' ? 'input' : atts.type,
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						quizMode ? el( wp.components.TextControl, {
							label: __( "Correct answer", 'greyd_forms' ),
							title: __( "Correct answer", 'greyd_forms' ),
							value: atts.correctAnswer,
							onChange: ( value ) => props.setAttributes( { correctAnswer: value } ),
						} ) : null,

						// Autofill
						el( SelectControl, {
							label: __( 'Autofill', 'greyd_forms' ),
							help: __( "Helps the user fill in with automatic suggestions.", 'greyd_forms' ),
							value: atts.autocomplete,
							options: [
								{ label: __( "automatically", 'greyd_forms' ), value: "on" },
								{ label: __( "off", 'greyd_forms' ), value: "off" },
								{ label: '--------', value: "", disabled: true },
								{ label: __( "name", 'greyd_forms' ), value: "name" },
								{ label: __( "first name", 'greyd_forms' ), value: "given-name" },
								{ label: __( "last name", 'greyd_forms' ), value: "family-name" },
								{ label: __( "email", 'greyd_forms' ), value: "email" },
								{ label: __( "phone number", 'greyd_forms' ), value: "tel" },
								{ label: __( "street & house number", 'greyd_forms' ), value: "street-address" },
								{ label: __( "zip code", 'greyd_forms' ), value: "postal-code" },
								{ label: __( "city", 'greyd_forms' ), value: "address-level2" },
								{ label: __( "country", 'greyd_forms' ), value: "country-name" },
								{ label: __( "country abbreviation", 'greyd_forms' ), value: "country" },
								// todo
							],
							onChange: ( value ) => props.setAttributes( { autocomplete: value } )
						} ),

						// Pflichtfeld
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							help: __( "The user must fill in this field.", 'greyd_forms' ),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),
					] ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),

					// BREITE
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [ {
							label: __( "Width", 'greyd_forms' ),
							max: 800,
							attribute: "width",
							control: RangeUnitControl,
						} ]
					} ),

					// HILFESTELLUNG
					el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [

						// // Placeholder
						// el( TextControl, {
						// 	label: __("Placeholder", 'greyd_forms'),
						// 	help: __("Placeholder text in the input field. (default: please select)", 'greyd_forms'),
						// 	value: atts.placeholder,
						// 	onChange: function(value) { props.setAttributes( { placeholder: value } ); },
						// } ),

						// Hilfetext
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),

					// ZEICHEN & SPALTEN
					el( PanelBody, { title: __( "Characters & columns", 'greyd_forms' ), initialOpen: false }, [

						// Zeichenanzahl
						atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ? el( element.Fragment, {}, [
							el( NumberControl, {
								label: __( "Minimum character length", 'greyd_forms' ),
								help: __( "How many characters must be entered at least?", 'greyd_forms' ),
								value: atts.restrictions.minlength,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, minlength: value } } )
							} ),
							el( NumberControl, {
								label: __( "Maximum character length", 'greyd_forms' ),
								help: __( "How many characters can be entered? (default: unlimited)", 'greyd_forms' ),
								value: atts.restrictions.maxlength,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, maxlength: value } } )
							} )
						] ) : null,

						// min, max, start
						( atts.type === "number" ) ? el( element.Fragment, {}, [
							el( NumberControl, {
								label: __( "Start value", 'greyd_forms' ),
								value: atts.restrictions.start,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, start: value } } )
							} ),
							el( NumberControl, {
								label: __( "Minimum value", 'greyd_forms' ),
								value: atts.restrictions.min,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, min: value } } )
							} ),
							el( NumberControl, {
								label: __( "Maximum value", 'greyd_forms' ),
								value: atts.restrictions.max,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, max: value } } )
							} ),
							el( NumberControl, {
								label: __( "Steps", 'greyd_forms' ),
								value: atts.restrictions.step,
								min: 0.1,
								step: 0.1,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, step: value } } )
							} ),
						] ) : null,
						( atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ) ? el( element.Fragment, {}, [
							el( TextControl, {
								label: __( "Start value", 'greyd_forms' ),
								value: atts.restrictions.start,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, start: value } } )
							} ),
							el( TextControl, {
								label: __( "Minimum value", 'greyd_forms' ),
								value: atts.restrictions.min,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, min: value } } )
							} ),
							el( TextControl, {
								label: __( "Maximum value", 'greyd_forms' ),
								value: atts.restrictions.max,
								onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, max: value } } )
							} ),
							el( Tip, {}, (
								__( "For the minimum & maximum values of time data, certain formats must be adhered to:", 'greyd_forms' ) +
								"\n" + __( "time", 'greyd_forms' ) + ": '18:00', " + __( "date", 'greyd_forms' ) + ": '2020-12-31', " + __( "date & time", 'greyd_forms' ) + ": '2020-12-31T18:00', " + __( "week", 'greyd_forms' ) + ": '2020-W52', " + __( "month", 'greyd_forms' ) + ": '2020-12'"
							) ),
						] ) : null,


						// Mehrzeilig?
						( atts.type === "text" ) ? el( element.Fragment, {}, [
							el( ToggleControl, {
								label: __( "Multi-line?", 'greyd_forms' ),
								help: __( "For text input only.", 'greyd_forms' ),
								checked: atts.area,
								onChange: ( value ) => props.setAttributes( { area: value } )
							} ),
							atts.area ? el( RangeControl, {
								label: __( "Number of lines", 'greyd_forms' ),
								help: __( "Number of rows displayed. (default: 5, max. 15)", 'greyd_forms' ),
								value: atts.rows,
								min: 1,
								max: 15,
								onChange: ( value ) => props.setAttributes( { rows: value } )
							} ) : null,
						] ) : null
					] ),

					// custom field
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual field", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: atts.custom
					}, [
						el( ToggleControl, {
							label: __( "Overwrite the design of the field individually", 'greyd_forms' ),
							checked: atts.custom,
							onChange: ( value ) => props.setAttributes( { custom: value } )
						} ),
					] ),
					atts.custom && el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// PREVIEW
				// Input Wrapper
				el( 'div', {
					className: [ "input-wrapper", atts.greydClass ].join( ' ' )
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								value: atts.label,
								onChange: ( value ) => props.setAttributes( { label: value } ),
								placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
							content: tooltipText,
							blockProps: props,
						} ),
					] ) : null,

					// Inner Input Wrapper
					el( 'div', {
						className: "input-inner-wrapper",
					}, [
						// Input Field
						el( wp.blockEditor.RichText, {
							tagName: 'div',
							className: "input " + ( atts.className ?? '' ),
							value: atts.placeholder,
							placeholder: props.isSelected ? __( "Enter a placeholder", 'greyd_forms' ) : __( "enter here", 'greyd_forms' ),
							withoutInteractiveFormatting: true,
							format: 'string',
							allowedFormats: [],
							onChange: ( value ) => props.setAttributes( { placeholder: stripTags( value ) } ),
							style: (
								atts.area && atts.rows > 1
								? {
									height: 'calc(40px + ' + atts.rows + 'em)'
								}
								: {}
							),

						} )
					] )
				] ),
				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						"": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
				// custom styles
				!atts.custom ? null : el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass + " .input",
					styles: {
						"": atts.customStyles,
						".rich-text [data-rich-text-placeholder]:after": { color: 'inherit', opacity: 0.5 },
					},
					important: true
				} )

			];
		},

		save: function ( props ) {

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( atts.type, atts ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
			const renderTootip = tooltipText.trim().length > 0 ? true : null;
			const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( 'div', {
				...blockProps
			}, [
				// Label 
				!atts.label.length ? null : el( 'div', {
					className: "label_wrap",
				}, [
					el( 'label', {
						className: "label",
						for: makeInputName( inputName, renderPlaceholder.id ),
					}, [
						el( wp.blockEditor.RichText.Content, {
							tagName: 'span',
							value: atts.label
						} ),
						// required/optional
						el( 'span', {
							className: !!atts.required ? "requirement-required" : "requirement-optional",
						}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
					] ),

					renderTootip && el( 'span', { className: "forms-tooltip", }, [
						el( 'span', {
							className: "forms-tooltip-toggle",
						} ),
						el( 'span', {
							className: "forms-tooltip-popup",
							id: makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
						}, tooltipText )
					] )
				] ),

				// Inner Input Wrapper
				el( 'div', {
					className: "input-inner-wrapper",
				}, [
					// Input Field
					el( ( atts.area ? 'textarea' : 'input' ), {
						className: "input validation " + ( atts.className ?? '' ),
						type: atts.type === 'url' ? 'text' : atts.type,
						name: inputName,
						id: makeInputName( inputName, renderPlaceholder.id ),
						...( renderTootip ? {
							'aria-describedby': makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
						} : {} ),
						placeholder: atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter,
						autocomplete: atts.autocomplete,
						...( atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ?
							{
								...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
								...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
							} : {}
						),
						...( atts.type === "number" || atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ?
							{
								...atts.restrictions.min ? { min: atts.restrictions.min } : {},
								...atts.restrictions.max ? { max: atts.restrictions.max } : {},
								...atts.restrictions.step ? { step: atts.restrictions.step } : {},
								...atts.restrictions.start ? { value: atts.restrictions.start } : {},
							} : {}
						),
						...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "tel" ?
							{
								...{ pattern: getValidationPattern( atts.type ) },
							} : {}
						),
						required: !!atts.required,
						'data-required': !!atts.required ? "required" : "optional",
						title: renderPlaceholder.required,
						rows: atts.area ? atts.rows : "",
						...(
							atts.correctAnswer && atts.correctAnswer.length > 0 ? {
								"data-correct-answer": atts.correctAnswer?.toLowerCase()
							} : {}
						)

					} ),
					el( 'span', {
						className: "input-field-icon"
					} )
				] ),
			
				el( greyd.components.RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .label": atts.labelStyles,
						...atts.custom ? {
							" .input": atts.customStyles,
							" .input::placeholder": { color: 'inherit', opacity: 0.5 }
						} : {}
					}
				} ),
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/input' ),

		transforms: {
			to: [
				{
					type: 'block',
					blocks: [ 'greyd-forms/upload' ],
					transform: function ( attributes, innerBlocks ) {
						console.log( "convert greyd radiobuttons to greyd dropdown" );

						delete attributes.area;
						delete attributes.autocomplete;
						delete attributes.rows;
						delete attributes.type;

						return blocks.createBlock(
							'greyd-forms/upload',
							attributes
						);
					}
				},
				{
					type: 'block',
					blocks: [ 'greyd-forms/dropdown' ],
					transform: function ( attributes, innerBlocks ) {
						console.log( "convert greyd input to greyd dropdown" );

						delete attributes.area;
						delete attributes.autocomplete;
						delete attributes.rows;
						delete attributes.type;

						return blocks.createBlock(
							'greyd-forms/dropdown',
							attributes
						);
					}
				},
			]
		}
	} );

	/**
	 * Checkbox block
	 */
	blocks.registerBlockType( 'greyd-forms/checkbox', {
		title: __( "checkbox", 'greyd_forms' ),
		description: __( "Single selection field for hooking", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_check.svg" } ),
		category: 'greyd-forms',
		supports: {
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		styles: [
			{
				name: 'checkbox',
				label: __( "checkbox", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'radio',
				label: __( "radio button", 'greyd_forms' ),
			},
			{
				name: 'switch',
				label: __( "iOS switch", 'greyd_forms' )
			},
		],
		attributes: {
			// main
			name: { type: "string", default: "" },
			id: { type: "string", default: "" },

			// input
			content: {
				type: 'string',
				source: 'html',
				selector: 'span',
				default: ''
			},
			useSetting: { type: 'boolean', default: false },
			required: { type: 'boolean', default: false },
			prechecked: { type: 'boolean', default: false },
			// spaceBetween: { type: 'string', default: '10px' },

			// styles
			greydClass: { type: "string", default: "" },
			greydStyles: { type: "object" },
			iconStyles: { type: "object" },
		},
		example: {
			attributes: {
				content: __( "I hereby accept the terms and conditions.", 'greyd_forms' ),
				useSetting: true,
				required: true,
				prechecked: true,
				className: 'is-style-switch'
			},
		},

		edit: function ( props ) {

			// generate attributes
			props.attributes.greydClass = getGreydClass( props );

			// atts
			const atts = props.attributes;

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [
					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// name
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: 'checkbox',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						// required
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							// help: __("The user must fill in this field.", 'greyd_forms'),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),

						// useSetting
						el( ToggleControl, {
							label: __( "Use privacy default text", 'greyd_forms' ),
							checked: atts.useSetting,
							onChange: ( value ) => props.setAttributes( { useSetting: value } )
						} ),

						// checked
						el( ToggleControl, {
							label: __( "Pre-checked on load", 'greyd_forms' ),
							checked: atts.prechecked,
							onChange: ( value ) => props.setAttributes( { prechecked: value } )
						} ),
					] ),

					// text color
					el( StylingControlPanel, {
						title: __( "Text color", 'greyd_forms' ),
						supportsHover: true,
						blockProps: props,
						controls: [
							{
								label: __( "Text color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl
							},
						]
					} ),

					// icon color
					el( StylingControlPanel, {
						title: __( "Icon color", 'greyd_forms' ),
						supportsHover: true,
						supportsActive: true,
						blockProps: props,
						parentAttr: 'iconStyles',
						controls: [
							{
								label: __( "Icon color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl
							},
						]
					} ),

					// typo
					el( StylingControlPanel, {
						title: __( "Typography", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						controls: [
							{
								attribute: "fontSize",
								control: FontSizePicker,
							},
						]
					} ),

					// el( PanelBody, { title: __("Space", 'greyd_forms'), initialOpen: false },
					// 	el( RangeUnitControl, {
					// 		title: __('Abstand zwischen Icon und Text', 'greyd_forms'), 
					// 		units: ["px", "em", "rem"], 
					// 		value: atts.spaceBetween,
					// 		onChange: (value) => props.setAttributes({ spaceBetween: value })
					// 	}, ),
					// ),
				] ),

				// Preview
				el( 'div', {
					className: [ "input-wrapper", atts.greydClass ].join( ' ' )
				}, [

					el( 'div', {
						className: "label checkbox-label",
					}, [
						el( "label", {
							for: makeInputName( atts.name )
						}, [
							el( "input", {
								type: 'checkbox',
								id: makeInputName( atts.name ),
								defaultChecked: atts.prechecked,
								checked: atts.prechecked,
								onChange: () => props.setAttributes( { prechecked: !atts.prechecked } ),
								title: __( "Pre-checked on load", 'greyd_forms' ),
							} ),
							el( "span", {} ),
						] ),
						el( "div", { style: { cursor: atts.useSetting ? 'default' : 'auto' } }, [
							...atts.useSetting ? [
								el( 'span', {
									dangerouslySetInnerHTML: { __html: greyd.forms.data.agb_text },
									title: __( "The privacy standard text is used as content.", 'greyd_forms' ),
								} )
								// el( wp.components.Tooltip, {
								// 	text: __("The privacy standard text is used as content.", 'greyd_forms'),
								// 	position: 'bottom center',
								// }, el( 'span', {}, greyd.forms.data.agb_text ) )
							] : [
								el( wp.blockEditor.RichText, {
									tagName: "span",
									value: atts.content,
									onChange: ( value ) => props.setAttributes( { content: value } ),
									placeholder: __( "Enter a text for the checkbox", 'greyd_forms' ),
									// allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
								} ),
							],
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
					] ),
				] ),

				el( RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						"": atts.greydStyles,
						" input[type=checkbox] + span": atts.iconStyles
					},
					important: true
				} )
			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( 'div', {
				...blockProps
			}, [
				el( 'label', {
					className: "label checkbox-label",
					for: makeInputName( inputName, renderPlaceholder.id ),
				}, [
					el( "input", {
						className: "checkbox_placeholder_value",
						type: "hidden",
						name: inputName,
						value: "false",
					} ),
					el( "input", {
						type: 'checkbox',
						id: makeInputName( inputName, renderPlaceholder.id ),
						name: inputName,
						checked: atts.prechecked,
						value: "true",
						required: !!atts.required,
						'data-required': !!atts.required ? "required" : "optional",
						title: __( "please confirm the checkbox", 'greyd_forms' ),
					} ),

					el( "div", {}, [
						atts.useSetting ? el( 'span', {
							dangerouslySetInnerHTML: {
								__html: renderPlaceholder.agb
							}
						} ) : el( wp.blockEditor.RichText.Content, {
							tagName: 'span',
							value: atts.content
						} ),
						// required/optional
						el( 'span', {
							className: !!atts.required ? "requirement-required" : "requirement-optional",
						}, !!atts.required ? "&nbsp;*" : " " + __( "(optional)", 'greyd_forms' ) ),
					] ),
				] ),
				el( RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						"": atts.greydStyles,
						" input[type=checkbox] + span": atts.iconStyles
					},
					important: true
				} )
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/checkbox' ),
	} );

	/**
	 * Dropdown
	 */
	blocks.registerBlockType( 'greyd-forms/dropdown', {
		title: __( 'Dropdown', 'greyd_forms' ),
		description: __( "Selection from a list", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_dropdown.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		styles: [
			{
				name: 'prim',
				label: __( "Primary dropdown", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( "Secondary dropdown", 'greyd_forms' )
			},
		],
		attributes: {

			// main
			name: { type: "string", default: "" },
			placeholder: { type: "string", default: "" },
			required: { type: "boolean", default: false },

			// options
			options: {
				type: 'array',
				default: []
				/**
				 * @since 1.7.4 Removed default.
				 * this default leads - for some unknown reasons - to validation erros
				 * in the block editor. Somehow the editor sees this as an object and
				 * invalidates the attribute.
				 */
				// default: [
				// 	{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
				// 	{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
				// ]
			},
			inputtype: { type: "string", default: "" },
			multiselect: { type: "boolean", default: false },
			open: { type: "boolean", default: false }, // only used for backend preview

			// label
			label: {
				type: 'string',
				source: 'html',
				selector: 'span',
				default: ''
			},
			labelStyles: { type: "object" },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// styles
			greydClass: { type: "string", default: "" },
			greydStyles: { type: "object" },
			custom: { type: "boolean", default: false },
			customStyles: { type: "object" },
		},
		example: {
			attributes: {
				label: __( 'Dropdown', 'greyd_forms' ),
				required: true,
				open: true
			}
		},

		edit: function ( props ) {

			// generate attributes
			props.attributes.greydClass = getGreydClass( props );

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;
			const defaultOption = { title: '', value: '' };
			const options = atts.options && atts.options.length ? atts.options : [];

			return [

				el( InspectorControls, {}, [

					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// Feldname
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: 'dropdown',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						// Pflichtfeld
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							help: __( "The user must fill in this field.", 'greyd_forms' ),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),

						// Mehrfachauswahl
						el( ToggleControl, {
							label: __( "Enable multiselect", 'greyd_forms' ),
							checked: atts.multiselect,
							onChange: ( value ) => props.setAttributes( { multiselect: value } )
						} ),
					] ),

					// OPTIONEN
					el( PanelBody, {
						title: __( "Options", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( greyd.components.ButtonGroupControl, {
							label: __( "Entry type", 'greyd_forms' ),
							value: atts.inputtype,
							options: [
								{ label: __( "individually", 'greyd_forms' ), value: "" },
								{ label: __( "collectively", 'greyd_forms' ), value: "list" },
								// { label: __("use pre-made list", 'greyd_forms'), value: "predefined" },
							],
							onChange: ( value ) => props.setAttributes( { inputtype: value, open: true } )
						} ),

						atts.inputtype === 'list' ? (
							el( FormTokenField, {
								label: __( "Options", 'greyd_forms' ),
								// help: __("Einzelne Optionen bitte durch Komma trennen. Soll die angezeigte Beschreibung sich vom übermittelten Wert unterscheiden, muss dies mit einem '=' angegeben werden: Wert=Anzeige.", 'greyd_forms'),  // help unfortunatley doesn't work with FormTokenField
								value: options.map( option => {
									return option.title + ( option.value ? '=' + option.value : '' );
								} ),
								onChange: function ( value ) {
									const options = value.map( option => {
										return {
											title: option.split( '=' )[ 0 ],
											value: option.split( '=' )[ 1 ] ?? '',
										};
									} );
									props.setAttributes( { options: [ ...options ] } );
								},
								onFocus: () => props.setAttributes( { open: true } )
							} )
						) : (
							el( 'div', {
								className: 'components-greyd-controlgroup',
							}, [
								...options.map( ( option, i ) => {

									option = { ...defaultOption, ...option };

									const groupComponents = [
										// remove button
										el( wp.components.Button, {
											className: "components-greyd-controlgroup__remove",
											onClick: ( event ) => {
												const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
												const newOptions = [ ...options ];
												newOptions.splice( index, 1 );
												props.setAttributes( { options: [ ...newOptions ], open: true } );
											},
											title: __( "Remove option", 'greyd_forms' )
										}, el( wp.components.Icon, { icon: 'no-alt' } ) ),

										el( 'div', { className: 'flex' }, [
											// title
											el( BaseControl, { style: { marginBottom: 0 }}, [
												el( BaseControl.VisualLabel, {}, __( 'Label', 'greyd_forms' ) ),
												el( 'input', {
													value: option.title,
													className: 'components-text-control__input',
													onInput: ( event ) => {
														const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
														const newOptions = [ ...options ];
														newOptions[ index ].title = event.target.value;
														// console.log( newOptions );
														props.setAttributes( { options: [ ...newOptions ] } );
													},
													onFocus: () => props.setAttributes( { open: true } )
												} ),
											] ),
											// value
											el( BaseControl, {}, [
												el( BaseControl.VisualLabel, {}, __( "Value", 'greyd_forms' ) ),
												el( 'input', {
													value: option.value,
													className: 'components-text-control__input',
													onInput: ( event ) => {
														const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
														const newOptions = [ ...options ];
														newOptions[ index ].value = makeInputName( event.target.value );
														console.log( newOptions );
														props.setAttributes( { options: [ ...newOptions ] } );
													},
													onFocus: () => props.setAttributes( { open: true } ),
													help: __( "If you leave this field empty, the value is generated from the label.", 'greyd_forms' )
												} )
											] ),
										] )
									];

									return el( 'div', {
										className: 'components-greyd-controlgroup__item',
										'data-index': i,
									}, groupComponents );
								} ),
								el( wp.components.Button, {
									title: __( "Add option", 'greyd_forms' ),
									className: 'components-greyd-controlgroup__add' + ( options.length === 0 ? ' group_is_empty' : '' ),
									onClick: ( event ) => props.setAttributes( { options: [ ...options, defaultOption ], open: true } ),
								}, [
									el( wp.components.Icon, { icon: 'plus-alt2' } ),
									options.length === 0 ? el( 'span', {}, __( "Add option", 'greyd_forms' ) ) : null
								] )
							] )
							// el( greyd.forms.components.DynamicOptionsList, {
							// 	blockProps: props,
							// 	value: options,
							// 	onChange: function(options) {
							// 		props.setAttributes( { options: options } ); 
							// 	},
							// 	onClick: function(options) {
							// 		props.setAttributes( { options: options, date: new Date } ); // create new date to trick the edit function to update on onClick-Events. 
							// 	}
							// } )
						)
					] ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),

					// HILFESTELLUNG
					el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [

						// Hilfetext
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),

					// BREITE
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [ {
							label: __( "Width", 'greyd_forms' ),
							max: 800,
							attribute: "width",
							control: RangeUnitControl,
						} ]
					} ),

					// custom field
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual field", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: atts.custom
					}, [
						el( ToggleControl, {
							label: __( "Overwrite the design of the field individually", 'greyd_forms' ),
							checked: atts.custom,
							onChange: ( value ) => props.setAttributes( { custom: value } )
						} ),
					] ),
					atts.custom && el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// Input Wrapper
				el( 'div', {
					className: [
						"input-wrapper",
						atts.greydClass,
						( props.className && props.className.length ? props.className : '' )
					].join( ' ' ),
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								value: atts.label,
								onChange: ( value ) => props.setAttributes( { label: value } ),
								placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
							content: tooltipText,
							blockProps: props,
						} ),
					] ) : null,

					// Input Field
					el( wp.blockEditor.RichText, {
						tagName: 'div',
						className: [
							"select-selected",
							( atts.open ? 'select-arrow-active' : '' ),
						].join( ' ' ),
						value: atts.placeholder,
						placeholder: props.isSelected ? __( "Enter a placeholder", 'greyd_forms' ) : __( "please select", 'greyd_forms' ),
						withoutInteractiveFormatting: true,
						format: 'string',
						allowedFormats: [],
						onChange: ( value ) => props.setAttributes( { placeholder: stripTags( value ) } ),
						style: {
							cursor: 'auto'
						}
					} ),

					// dropdown items
					props.isSelected && atts.open ? el( 'div', {
						className: 'select-items dropdown',
						onClick: () => props.setAttributes( { open: false } ),
					}, options.map( option => {
						return el( 'div', {}, option.title );
					} ) ) : null,

					// button to open & close the dropdown items
					el( 'button', {
						onClick: () => props.setAttributes( { open: !atts.open } ),
						style: {
							position: 'absolute',
							bottom: 'var(--wp--custom--greyd--input--spacing--padding--top, 0.5em)',
							right: 0,
							height: '2em',
							width: '60px',
							background: 'none',
							cursor: 'pointer'
						}
					}, [
						el( 'span', { className: atts.open ? 'arrow_carrot-up' : 'arrow_carrot-down' } )
					] ),
				] ),

				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						".input-wrapper": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
				// custom styles
				!atts.custom ? null : el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass + " > div:not(.label_wrap)",
					styles: {
						"": atts.customStyles,
						".select-items": { boxShadow: 'none' },
						".rich-text [data-rich-text-placeholder]:after": { color: 'inherit', opacity: 0.5 },
					},
					important: true
				} )
			];
		},

		save: function ( props ) {

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
			const renderTootip = tooltipText.trim().length > 0 ? true : null;
			const inputName = _.isEmpty( atts.name ) ? makeInputName( atts.type, renderPlaceholder.id ) : makeInputName( atts.name, renderPlaceholder.id );
			const options = atts.options && atts.options.length ? atts.options : [];

			const coreSpacing = wp.styleEngine.compileCSS( atts.style );

			// we handle multiselects different than normal custom-selects
			const temp = !atts.multiselect ? {
				innerWrapper: 'div',
				innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
				selectAtts: {},
			} : {
				innerWrapper: element.Fragment,
				innerWrapperAtts: {},
				selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
			};

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( element.Fragment, {}, [

				el( 'div', {
					...blockProps
				}, [
					// Label 
					!atts.label.length ? null : el( 'div', {
						className: "label_wrap",
					}, [
						el( 'label', {
							className: "label",
							for: inputName,
						}, [
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: atts.label
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),

						renderTootip && el( 'span', { className: "forms-tooltip", }, [
							el( 'span', {
								className: "forms-tooltip-toggle",
							} ),
							el( 'span', {
								className: "forms-tooltip-popup",
								id: 'tt-' + inputName
							}, tooltipText )
						] )
					] ),

					// Select
					el( temp.innerWrapper, temp.innerWrapperAtts, [
						el( 'select', {
							name: inputName,
							...( renderTootip ? {
								'aria-describedby': 'tt-' + inputName,
							} : {} ),
							required: !!atts.required,
							'data-required': !!atts.required ? "required" : "optional",
							title: renderPlaceholder.options,
							...temp.selectAtts
						}, [
							el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : renderPlaceholder.select ),
							...options.map( option => {
								const title = option.title && option.title.length ? option.title : ( option.value ?? '' );
								const value = option.value && option.value.length ? option.value : sanitizeTitle( option.title );

								return _.isEmpty( value ) ? null : el( 'option', {
									value: makeInputName( value )
								}, stripTags( title ) );
							} )
						] ),
					] )
				] ),

				// Styles
				el( greyd.components.RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .label": atts.labelStyles,
						...atts.custom ? {
							" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
							" .select-items": { boxShadow: 'none' },
						} : {}
					}
				} )
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/dropdown' )
	} );

	/**
	 * Radio Buttons
	 */
	blocks.registerBlockType( 'greyd-forms/radiobuttons', {
		title: __( "Radio buttons", 'greyd_forms' ),
		description: __( "Selection from multiple options", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom'],
				padding: true
			}
		},
		styles: [
			{
				name: 'radio',
				label: __( "radio button", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'checkbox',
				label: __( "checkbox", 'greyd_forms' ),
			},
			{
				name: 'switch',
				label: __( "iOS switch", 'greyd_forms' )
			},
		],
		attributes: {

			// main
			name: { type: "string", default: "" },
			required: { type: "boolean", default: false },
			multiselect: { type: "boolean", default: false },
			variation: { type: 'string', default: '' },
			surveyId: { type: 'string', default: '' },
			surveyTitle: { type: 'string', default: 'Umfrage'},
			

			// label
			label: {
				type: 'string',
				source: 'html',
				selector: 'span',
				default: ''
			},
			labelStyles: { type: "object" },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// styles
			greydClass: { type: "string", default: "" },
			greydStyles: { type: "object", default: {} },
			iconStyles: { type: "object" },
			progressbar: { 
				type: 'object', properties: {
					layout: { type: 'string' },
					verticalAlignment: { type: 'string' },
					horizontalAlignment: { type: 'string' },
				}, default: {
					layout: 'column', 
					verticalAlignment: 'align-start',
					horizontalAlignment: 'flex-start'
				},
			}
		},
		example: {
			attributes: {
				label: __( "Radio buttons", 'greyd_forms' ),
				required: true,
				className: 'is-style-switch'
			},
			innerBlocks: [
				{
					name: 'greyd-forms/radiobutton-item',
					attributes: {
						title: __( 'Option 1', 'greyd_forms' ),
					}
				},
				{
					name: 'greyd-forms/radiobutton-item',
					attributes: {
						title: __( 'Option 2', 'greyd_forms' ),
					}
				},
			]
		},
		providesContext: {
			'greyd-forms/radiobutton-name': 'name',
			'greyd-forms/radiobutton-multiselect': 'multiselect',
			'greyd-forms/radiobutton-required': 'required',
			'greyd-forms/radiobutton-progressbar': 'progressbar',
		},
		variations: [
			{
				name: 'radiobuttons',
				title: __( "Radio buttons", 'greyd_forms' ),
				icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
				scope: [ 'inserter' ],
				isDefault: true,
				innerBlocks: [
					[ 'greyd-forms/radiobutton-item', { variation: '' } ],
					[ 'greyd-forms/radiobutton-item', { variation: '' } ],
				],
				attributes: {
					variation: '',
					surveyId: ''
				},
				isActive: ( blockAttributes, variationAttributes ) => {
					return blockAttributes.variation === variationAttributes.variation;
				}
			},
			{
				name: 'live-survey',
				title: __( "Live survey", 'greyd_forms' ),
				icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
				scope: [ 'inserter' ],
				innerBlocks: [
					[ 'greyd-forms/radiobutton-item', { variation: 'live' } ],
					[ 'greyd-forms/radiobutton-item', { variation: 'live' } ],
				],
				attributes: {
					variation: 'live-survey',
				},
				isActive: ( blockAttributes, variationAttributes ) => {
					return blockAttributes.variation === variationAttributes.variation;
				}
			},
		],
		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;
			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );

			const parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/multistep-container" );
			const parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
			const quizMode = parent?.attributes.multistepMode == "quiz";

			// mode for section setting
			const [mode, setMode ] = wp.element.useState("");
			if (!props.isSelected && mode != "") setMode("");


			const progressbarColors = {
				" .live_survey_progress ": { backgroundColor: atts?.progressbar?.backgroundColor },
				" .live_survey_progress .bar": { backgroundColor: atts?.progressbar?.activeColor }
			};

			if (atts?.variation == 'live-survey' && atts?.surveyId == '') {
				props.setAttributes( { surveyId: greyd.tools.generateRandomID(12) } ) 
			}

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [
					// ALLGEMEIN

					mode == '' ? el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// name
						el( InputNameControl, {
							label: __( "Field name", "greyd_forms" ),
							help: __( "Unique value of the field that is passed with the form.", "greyd_forms" ),
							title: __( "This field is required for the form field to function correctly.", "greyd_forms" ),
							value: atts.name,
							autofillPrefix: 'radio',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						atts.variation == "live-survey" ? el( wp.components.TextControl, {
							label: __( "Survey title", "greyd_forms" ),
							help: __( "Used to display the results in the backend", "greyd_forms" ),
							value: atts.surveyTitle,
							onChange: ( value ) => props.setAttributes( { surveyTitle: value } )
						} ) : null,

						// required
						!quizMode && atts.variation !== "live-survey" ? el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							// help: __("The user must fill in this field.", 'greyd_forms'),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ) : null,

						// Mehrfachauswahl
						!quizMode && atts.variation !== "live-survey" ? el( ToggleControl, {
							label: __( "Enable multiselect", 'greyd_forms' ),
							checked: atts.multiselect,
							onChange: ( value ) => props.setAttributes( { multiselect: value } )
						} ) : null,
						atts.variation == "live-survey" ? el( ToggleControl, {
							label: __( "Text wrap for long texts", "greyd_forms" ),
							checked: atts.textwrap,
							onChange: ( value ) => props.setAttributes( { textwrap: value } )
						} ) : null,
						// to hotspot settings
						atts.variation === "live-survey" ? el ( greyd.components.SectionControl, {
							title: __("Progress display", 'greyd_forms'),
							onClick: () => setMode("progressbar")
						} ) : null
					] ) : null,



					mode == "progressbar" ? [
						el ( greyd.components.SectionControl, {
							title: __("Progress display", 'greyd_forms'),
							icon: 'arrow-left-alt',
							buttonText: __("back", 'greyd_forms'),
							onClick: () => setMode(""),
							isHeader: true
						} ),
						el( PanelBody, { title: __( 'Layout', 'greyd_forms' ), initialOpen: true }, [
							el( greyd.components.ButtonGroupControl, {
								label: __( "Arrangement", 'greyd_forms' ),
								value: atts.progressbar.layout,
								options: [
									{ label: __( "alongside", 'greyd_forms' ), value: 'row' },
									{ label: __( "Below", 'greyd_forms' ), value: 'column' },
								],
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, layout: value } } )
							} ),
							el( greyd.components.ButtonGroupControl, {
								label: __( "Alignment Vertical", 'greyd_forms' ),
								value: atts.progressbar.verticalAlignment,
								options: [
									{ label: __( "Start", 'greyd_forms' ), value: 'align-start' },
									{ label: __( "Center", 'greyd_forms' ), value: 'align-center' },
									{ label: __( "End", 'greyd_forms' ), value: 'align-end' },
								],
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, verticalAlignment: value } } )
							} ),
							el( greyd.components.ButtonGroupControl, {
								label: __( "Alignment Horizontal", 'greyd_forms' ),
								value: atts.progressbar.horizontalAlignment,
								options: [
									{ label: __( "Start", 'greyd_forms' ), value: 'justify-start' },
									{ label: __( "Center", 'greyd_forms' ), value: 'justify-center' },
									{ label: __( "End", 'greyd_forms' ), value: 'justify-end' },
								],
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, horizontalAlignment: value } } )
							} ),
						] ),
						el( PanelBody, { title: __("Space", 'greyd_forms'), initialOpen: false },
							el( RangeUnitControl, {
								title: __("Space between progress bar and option", 'greyd_forms'), 
								units: ["px", "em", "rem"], 
								value: atts.progressbar.spaceBetween,
								onChange: (value) => props.setAttributes( { progressbar: { ...atts.progressbar, spaceBetween: value } } )
							}, ),
						),

						el( PanelBody, { title: __( "Colors", 'greyd_forms' ), initialOpen: false }, [
							el( greyd.components.ColorPopupControl, {
								label: __( "active color", 'greyd_forms' ),
								value: atts.progressbar.activeColor,
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, activeColor: value } } )

							} ),
							el( greyd.components.ColorPopupControl, {
								label: __( "Background color", 'greyd_forms' ),
								value: atts.progressbar.backgroundColor,
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, backgroundColor: value } } )
							} )
						] ),
						el( greyd.components.StylingControlPanel, {
							title: __( "Size", 'greyd_forms' ),
							initialOpen: false,
							supportsResponsive: false,
							blockProps: props,
							parentAttr: 'progressbar',
							controls: [
								{
									label: __( "Height", 'greyd_forms' ),
									max: 500,
									attribute: "height",
									control: RangeUnitControl,
								},
								{
									label: __( "Width", 'greyd_forms' ),
									max: 1200,
									attribute: "width",
									control: RangeUnitControl,
								},
							]
						} ),

						// border-radius
						el( StylingControlPanel, {
							title: __( "Border radius", 'greyd_forms' ),
							initialOpen: false,
							blockProps: props,
							parentAttr: 'progressbar',
							controls: [ {
								label: __( "Border radius", 'greyd_forms' ),
								attribute: "borderRadius",
								control: RangeUnitControl,
							} ]
						} ),
					] : null,

					// Layout
					mode == "" ? [ 
						atts.variation == "" ? el( StylingControlPanel, {
							title: __( 'Layout', 'greyd_forms' ),
							supportsResponsive: true,
							blockProps: props,
							controls: [
								{
									label: __( "Arrangement", 'greyd_forms' ),
									attribute: "display",
									control: greyd.components.ButtonGroupControl,
									options: [
										{ label: __( "below each other", 'greyd_forms' ), value: 'block' },
										{ label: __( "next to each other", 'greyd_forms' ), value: 'inline-block' },
									]
								},
								// {
								// 	label: __('Abstand zwischen den Optionen', 'greyd_forms'),
								// 	attribute: "marginBottom",
								// 	control: RangeUnitControl,
								// 	units: ["px", "em", "rem"], 
								// },
							]
						} ) : null,

						// Optionen
						el( StylingControlPanel, {
							title: __( "Appearance of the options", 'greyd_forms' ),
							supportsHover: true,
							supportsActive: true,
							blockProps: props,
							controls: [
								{
									label: __( "Text color", 'greyd_forms' ),
									attribute: "color",
									control: ColorPopupControl
								},
								{
									label: __( "Font size", 'greyd_forms' ),
									attribute: "fontSize",
									control: RangeUnitControl,
									max: { px: 50, em: 2, rem: 2, '%': 200 }
								},
							]
						} ),

						// icon color
						el( StylingControlPanel, {
							title: __( "Icon color", 'greyd_forms' ),
							supportsHover: true,
							supportsActive: true,
							blockProps: props,
							parentAttr: 'iconStyles',
							controls: [
								{
									label: __( "Icon color", 'greyd_forms' ),
									attribute: "color",
									control: ColorPopupControl
								},
							]
						} ),

						// LABEL
						el( greyd.components.StylingControlPanel, {
							title: __( 'Label', 'greyd_forms' ),
							supportsHover: true,
							parentAttr: 'labelStyles',
							blockProps: props,
							controls: [
								{
									label: __( "Font size", 'greyd_forms' ),
									attribute: "fontSize",
									units: [ "px", "em", "rem" ],
									max: { px: 60, em: 3, rem: 3 },
									control: RangeUnitControl,
								},
								{
									label: __( "Color", 'greyd_forms' ),
									attribute: "color",
									control: ColorPopupControl,
								},
							]
						} ),

						// HILFESTELLUNG
						el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [

							// Hilfetext
							el( TextareaControl, {
								label: __( "Tooltip", 'greyd_forms' ),
								help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
								value: atts.tooltip.content,
								onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
								onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
								onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
							} )
						] ),
					] : null
				] ),

				// PREVIEW
				el( 'div', {
					className: [ "input-wrapper radio_buttons", atts.greydClass ].join( ' ' ),
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'div', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								value: atts.label,
								onChange: ( value ) => props.setAttributes( { label: value } ),
								placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
							content: tooltipText,
							blockProps: props,
						} ),
					] ) : null,

					// inner blocks
					el( InnerBlocks, {
						allowedBlocks: [ 'greyd-forms/radiobutton-item' ],
						template: [
							[ 'greyd-forms/radiobutton-item', { title: __( 'Option 1', 'greyd_forms' ) } ],
							[ 'greyd-forms/radiobutton-item', { title: __( 'Option 2', 'greyd_forms' ) } ],
						],
						orientation: atts.greydStyles.display && atts.greydStyles.display === "inline-block" ? "horizontal" : "vertical",
						...hasChildBlocks ? {} : { renderAppender: wp.blockEditor.InnerBlocks.ButtonBlockAppender }
					} )
				] ),
				el( RenderPreviewStyles, {
					selector: atts.greydClass + " .block-editor-inner-blocks > div > div",
					styles: {
						"": atts.greydStyles,
						" input[type=radio] + span": atts.iconStyles
					}
				} ),
				el( RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						" .label": atts.labelStyles,
					}
				} ),
				el( RenderPreviewStyles, {
					selector: atts.greydClass + " .progress_wrapper",
					styles: {
						" .live_survey_progress": atts.progressbar,
						...progressbarColors
					},
					important: true
				} )
			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
			const renderTootip = tooltipText.trim().length > 0 ? true : null;
			const progressbarColors = {
				" .live_survey_progress ": { backgroundColor: atts?.progressbar?.backgroundColor },
				" .live_survey_progress .bar": { backgroundColor: atts?.progressbar?.activeColor }
			};

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( element.Fragment, {}, [
				// Input Wrapper
				el( 'div', {
					...blockProps,
					...(
						atts?.variation == 'live-survey' ? {
							'data-survey-id': atts?.surveyId,
							'data-survey-title': atts?.surveyTitle
						} : {}
					)
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'label', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: atts.label
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						renderTootip && el( 'span', { className: "forms-tooltip", }, [
							el( 'span', {
								className: "forms-tooltip-toggle",
							} ),
							el( 'span', {
								className: "forms-tooltip-popup",
								id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id )
							}, tooltipText )
						] )
					] ) : null,

					// inner blocks
					atts.multiselect ? el( 'div', {
						className: 'greyd_multiradio',
					}, [
						el( 'input', {
							type: 'text',
							name: makeInputName( atts.name, renderPlaceholder.id ),
							...( renderTootip ? {
								'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
							} : {} ),
							value: '',
							title: renderPlaceholder.options,
							required: !!atts.required,
							'data-required': !!atts.required ? "required" : "optional",
						} ),
						el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
					] ) : el( 'fieldset', {
						title: renderPlaceholder.options,
						required: !!atts.required,
						'data-required': !!atts.required ? "required" : "optional",
					}, [
						el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
					] )
				] ),
				el( RenderSavedStyles, {
					selector: atts.multiselect ? atts.greydClass + " .greyd_multiradio" : atts.greydClass,
					styles: {
						" .option": atts.greydStyles,
						" input[type=radio] + span, .option .icn": atts.iconStyles
					}
				} ),
				el( RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .label": atts.labelStyles,
					}
				} ),

				el( RenderSavedStyles, {
					selector: atts.greydClass + " .progress_wrapper",
					styles: {
						" .live_survey_progress": atts.progressbar,
						...progressbarColors
					},
					important: true
				} )
				
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/radiobuttons' ),

		transforms: {

			from: [
				{
					type: 'block',
					blocks: [ 'greyd-forms/dropdown' ],
					transform: function ( attributes, innerBlocks ) {
						// console.log(attributes);

						var innerAtts = {};
						var inner = [];
						if ( _.has( attributes, 'options' ) ) {
							var options = attributes.options;
						}
						for ( var i = 0; i < options.length; i++ ) {
							innerAtts = {
								title: options[ i ].title,
								value: options[ i ].value
							};

							inner.push( blocks.createBlock(
								'greyd-forms/radiobutton-item',
								innerAtts
							) );

						}
						return blocks.createBlock(
							'greyd-forms/radiobuttons',
							attributes,
							inner
						);
					}
				}
			],
			to: [
				{
					type: 'block',
					blocks: [ 'greyd-forms/dropdown' ],
					transform: function ( attributes, innerBlocks ) {
						console.log( "convert greyd radiobuttons to greyd dropdown" );
						// console.log(innerBlocks);
						var options = [];
						for ( var i = 0; i < innerBlocks.length; i++ ) {
							if ( _.has( innerBlocks[ i ], 'attributes' ) ) {
								innerAtts = innerBlocks[ i ].attributes;
								options.push( { title: innerAtts.title, value: innerAtts.value } );
							}
						}
						attributes.options = options;
						return blocks.createBlock(
							'greyd-forms/dropdown',
							attributes
						);
					}
				},
				{
					type: 'block',
					blocks: [ 'greyd-forms/iconpanels' ],
					transform: function ( attributes, innerBlocks ) {
						console.log( "convert greyd radiobuttons to greyd iconpanels" );

						var newInnerAtts = {};
						var innerAtts = {};
						var inner = [];

						for ( var i = 0; i < innerBlocks.length; i++ ) {

							if ( _.has( innerBlocks[ i ], 'attributes' ) ) {
								innerAtts = innerBlocks[ i ].attributes;

								console.log( innerAtts );
								newInnerAtts = {
									title: innerAtts.title,
									value: innerAtts.value,
									images: {
										normal: { id: -1, url: "" },
										hover: { id: -1, url: "" },
										active: { id: -1, url: "" },
										current: 'normal'
									},
								};
							}

							inner.push( blocks.createBlock(
								'greyd-forms/iconpanel',
								newInnerAtts
							) );

						}
						return blocks.createBlock(
							'greyd-forms/iconpanels',
							attributes,
							inner
						);
					}
				}
			]
		}
	} );

	blocks.registerBlockType( 'greyd-forms/radiobutton-item', {
		title: __( "radio button", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
		category: 'greyd-forms',
		parent: [ 'greyd-forms/radiobuttons' ],
		supports: {
			className: false,
			customClassName: false,
		},
		attributes: {
			
			title: {
				type: 'string',
				// source: 'html',
				// selector: 'span',
				default: ''
			},
			value: { type: 'string', default: '' },
			correctAnswer: { type: 'bool', default: false},
			// parent
			name: { type: 'string', default: '' },
			multiselect: { type: "boolean", default: false },
			required: { type: "boolean", default: false },
			progressbar: { type: "object", default: {} },
			variation: { type: 'string', default: '' },
		},
		example: {
			attributes: {
				title: __( 'Option', 'greyd_forms' ),
			},
		},
		usesContext: [ 'greyd-forms/radiobutton-name', 'greyd-forms/radiobutton-multiselect', 'greyd-forms/radiobutton-required', 'greyd-forms/radiobutton-progressbar' ],
		variations: [
			{
				name: '',
				title: __( "Radio button", 'greyd_forms' ),
				icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
				scope: [ 'inserter' ],
				isDefault: true,
				attributes: {
					variation: ''
				},
				isActive: ( blockAttributes, variationAttributes ) => {
					return blockAttributes.variation === variationAttributes.variation;
				}
			},
			{
				name: 'live',
				title: __( "Radio button with progress bar", 'greyd_forms' ),
				icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_radio.svg" } ),
				scope: [ 'inserter' ],
				attributes: {
					variation: 'live'
				},
				isActive: ( blockAttributes, variationAttributes ) => {
					return blockAttributes.variation === variationAttributes.variation;
				}
			},
		],
		edit: function ( props ) {

			props.setAttributes( { name: props.context[ 'greyd-forms/radiobutton-name' ] } );
			props.setAttributes( { multiselect: props.context[ 'greyd-forms/radiobutton-multiselect' ] } );
			props.setAttributes( { required: props.context[ 'greyd-forms/radiobutton-required' ] } );
			props.setAttributes( { progressbar: props.context[ 'greyd-forms/radiobutton-progressbar' ] } );
			const atts = props.attributes;

			let parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/radiobuttons" );
			let parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
			const isLiveSurvey = parent?.attributes.variation == "live-survey";

			parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/multistep-container" );
			parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
			const quizMode = parent?.attributes.multistepMode == "quiz";

			if ( isLiveSurvey && atts.variation == "") {
				props.setAttributes( { variation: "live" } );
				wp.data.dispatch( 'core/notices' ).createNotice(
					'info', // Can be one of: success, info, warning, error.
					__("The radio button has been converted to a radio button with progress bar, as the parent block uses the variation 'Live survey'.", "greyd_forms"), 
					{
						isDismissible: true, 
					}
				);
			} else if ( !isLiveSurvey && atts.variation == "live" ) {
				props.setAttributes( { variation: "" } );
				wp.data.dispatch( 'core/notices' ).createNotice(
					'info', // Can be one of: success, info, warning, error.
					__("The radio button with progress bar has been converted to a normal radio button, as the parent block uses the normal block variation.", "greyd_forms"), 
					{
						isDismissible: true, 
					}
				);
			}

			// if ( typeof props.attributes.value == "undefined" ) props.setAttributes( { value: 'value_' + props.clientId.substring(0,4) } );

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true },
						el( wp.components.TextControl, {
							label: __( "Value", 'greyd_forms' ),
							value: atts.value,
							autofillPrefix: 'option',
							onChange: ( value ) => props.setAttributes( { value: value } )
						} ),
						quizMode ? el( ToggleControl, {
							label: __( 'Correct Answer', 'greyd_forms' ),
							// help: __('Der Nutzer muss dieses Feld ausfüllen.', 'greyd_forms'),
							checked: !!atts.correctAnswer,
							onChange: ( value ) => props.setAttributes( { correctAnswer: !!value } )
						} ) : null,
					)
				] ),
				atts.variation === "live" ? 
				el( "div", {
					className: ['live_survey_option', 'flex-'+atts.progressbar.verticalAlignment, 'flex-'+atts.progressbar.layout].join(" "),
					style: { gap: atts.progressbar.spaceBetween }
				}, [
					el( "div", {
						className: 'label radio-label',
						for: makeInputName( atts.value )
					}, [
						el( "input", {
							type: 'radio',
							id: makeInputName( atts.value ),
							name: atts.name
						} ),
						el( "span", {} ),
						el( RichText, {
							tagName: 'span',
							className: "option",
							value: atts.title,
							onChange: ( value ) => props.setAttributes( { title: value } ),
							placeholder: props.isSelected ? __( "Enter the text for the option", 'greyd_forms' ) : __( 'Option', 'greyd_forms' ),
							allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
						} ),					
					] ),
					el( 'div', {
						className: ['progress_wrapper', 'flex-'+atts.progressbar.horizontalAlignment ].join(" "),
					}, [
						el( 'div', { className: 'live_survey_progress'},
							el( 'span', { className: 'bar', style: { width: "50%" } } ) 
						)
					] ) 
				] ) : 
				el( "div", {
					className: 'label radio-label',
					for: makeInputName( atts.value )
				}, [
					el( "input", {
						type: 'radio',
						id: makeInputName( atts.value ),
						name: atts.name
					} ),
					el( "span", {} ),
					el( RichText, {
						tagName: 'span',
						className: "option",
						value: atts.title,
						onChange: ( value ) => props.setAttributes( { title: value } ),
						placeholder: props.isSelected ? __( "Enter the text for the option", 'greyd_forms' ) : __( 'Option', 'greyd_forms' ),
						allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
						/**
						 * onSplit
						 * @see 'https://github.com/WordPress/gutenberg/blob/trunk/packages/block-library/src/paragraph/edit.js'
						 * @param {string} value
						 * @param {bool} isOriginal
						 * @returns {blockElement}
						 */
						// onSplit: ( value, isOriginal ) => {
						// 	let newAttributes;

						// 	if ( isOriginal || value ) {
						// 		newAttributes = {
						// 			...props.attributes,
						// 			title: value,
						// 		};
						// 	}

						// 	const block = createBlock( 'greyd-forms/radiobutton-item', newAttributes );

						// 	if ( isOriginal ) {
						// 		block.clientId = clientId;
						// 	}

						// 	return block;
						// },
						// onMerge: wp.data.dispatch('core/block-editor').mergeBlocks,
						// onReplace: wp.data.dispatch('core/block-editor').onReplace,
						// onRemove: wp.data.dispatch('core/block-editor').onRemove,
					} ),					
				] ),
				
				
				// el( 'span', { className:"icn" }, ),
			];
		},
		save: function ( props ) {

			const atts  = props.attributes;
			const value = atts.value.length > 0 ? atts.value : sanitizeTitle( atts.title );
			const id    = makeInputName( sanitizeTitle( value ), renderPlaceholder.id );
			const name  = makeInputName( atts.name, renderPlaceholder.id );

			// multiselect
			if ( atts.multiselect ) {
				return el( 'div', {
					className: 'option',
					onclick: 'window.forms.onMultiRadioClick(this);',
					title: renderPlaceholder.options,
					'data-value': value,
				}, [
					el( "span", {
						className: 'icn'
					} ),
					el( wp.blockEditor.RichText.Content, {
						tagName: 'span',
						value: atts.title
					} ),
				] );
			} else if (atts.variation == "live") {

				return el( 'div', {
					className: ['option', 'live_survey_option', 'flex-'+atts.progressbar.align, 'flex-'+atts.progressbar.layout].join(" "),
					style: { gap: atts.progressbar.spaceBetween }
				}, [
					el( 'label', {
						for: id,
						title: renderPlaceholder.options,
					}, [
						el( "input", {
							type: 'radio',
							id: id,
							value: value,
							name: name,
							required: !!atts.required,
							'data-required': !!atts.required ? "required" : "optional",
				
						} ),
						el( wp.blockEditor.RichText.Content, {
							tagName: 'span',
							value: atts.title
						} ),
					] ),
					el( 'div', {
						className: ['progress_wrapper', 'flex-'+atts.progressbar.alignment ].join(" "),
					}, [
						el( 'div', { className: 'live_survey_progress'},
							el( 'span', { className: 'bar' } ) 
						)
					] ) ]
				);

			}
			// normal
			else {
				return el( 'div', {
					className: 'option'
				}, [
					el( 'label', {
						for: id,
						title: renderPlaceholder.options,
					}, [
						el( "input", {
							type: 'radio',
							id: id,
							value: value,
							name: name,
							required: !!atts.required,
							'data-required': !!atts.required ? "required" : "optional",
							...(
								atts.correctAnswer ? {
									"data-correct-answer": atts.correctAnswer
								} : {}
							)
						} ),
						el( wp.blockEditor.RichText.Content, {
							tagName: 'span',
							value: atts.title
						} ),
					] )
				] );
			}
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/radiobutton-item' )
	} );

	/**
	 * Icon Panels
	 */
	blocks.registerBlockType( 'greyd-forms/iconpanels', {
		title: __( "Image panels", 'greyd_forms' ),
		description: __( "Clickable panels with image and text", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_panels.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		styles: [
			{
				name: 'prim',
				label: __( "primary field", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( "secondary field", 'greyd_forms' )
			},
			{
				name: 'none',
				label: __( "no style", 'greyd_forms' )
			},
		],
		attributes: {

			// main
			name: { type: "string", default: "" },
			required: { type: "boolean", default: false },
			multiselect: { type: "boolean", default: false },
			greydClass: { type: "string", default: "" },

			// label
			label: {
				type: 'string',
				source: 'html',
				selector: 'span',
				default: ''
			},
			labelStyles: { type: "object" },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// wrapper styles
			wrapperStyles: {
				type: "object", properties: {
					flexDirection: { type: "string" },
					alignItems: { type: "string" },
				}, default: {
					flexDirection: "",
					alignItems: "",
				}
			},

			// panel styles
			panelClass: { type: "string", default: "" },
			panelStyles: {
				type: "object", properties: {
					flexDirection: { type: "string" },
					width: { type: "string" },
				}, default: {
					flexDirection: "",
					// width: "auto",
				}
			},

			// image styles
			imgStyles: {
				type: "object", properties: {
					height: { type: "string" },
					margin: { type: "object" },
				}, default: {
					height: "3em",
					margin: { bottom: "0.5em" },
				}
			},
		},
		providesContext: {
			'greyd-forms/iconpanel-name': 'name',
			'greyd-forms/iconpanel-multiselect': 'multiselect',
			'greyd-forms/iconpanel-required': 'required',
			'greyd-forms/iconpanel-panelClass': 'panelClass',
		},


		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;
			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );

			return [

				// Inspector sidebar
				el( InspectorControls, {}, [

					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// name
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: 'image',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						// required
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							// help: __("The user must fill in this field.", 'greyd_forms'),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),

						// Mehrfachauswahl
						el( ToggleControl, {
							label: __( "Enable multiselect", 'greyd_forms' ),
							checked: atts.multiselect,
							onChange: ( value ) => props.setAttributes( { multiselect: value } )
						} ),
					] ),

					// LAYOUT
					el( AdvancedPanelBody, {
						title: __( 'Layout', 'greyd_forms' ),
						initialOpen: true,
						holdsChange: atts.wrapperStyles.flexDirection.length || atts.panelStyles.flexDirection.length
					}, [
						el( greyd.components.ButtonGroupControl, {
							label: __( "Arrangement", 'greyd_forms' ),
							value: atts.wrapperStyles.flexDirection,
							options: [
								{ label: __( "next to each other", 'greyd_forms' ), value: '' },
								{ label: __( "below each other", 'greyd_forms' ), value: 'column' },
							],
							onChange: ( value ) => props.setAttributes( {
								wrapperStyles: {
									flexDirection: value,
									alignItems: value === 'column' ? 'flex-start' : 'center'
								}
							} ),
						} ),
						el( greyd.components.ButtonGroupControl, {
							label: __( "Icon position", 'greyd_forms' ),
							value: atts.panelStyles.flexDirection,
							options: [
								{ label: __( "top", 'greyd_forms' ), value: '' },
								{ label: __( "right", 'greyd_forms' ), value: 'row-reverse' },
								{ label: __( "bottom", 'greyd_forms' ), value: 'column-reverse' },
								{ label: __( "left", 'greyd_forms' ), value: 'row' },
							],
							onChange: ( value ) => {
								let side = 'bottom';
								if ( value === 'row-reverse' ) side = 'left';
								else if ( value === 'column-reverse' ) side = 'top';
								else if ( value === 'row' ) side = 'right';
								props.setAttributes( {
									panelStyles: { ...atts.panelStyles, flexDirection: value },
									imgStyles: { ...atts.imgStyles, margin: 0 },
								} );
							},
						} ),
					] ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),

					// HILFESTELLUNG
					el( AdvancedPanelBody, {
						title: __( "Assistance", 'greyd_forms' ),
						initialOpen: false,
						holdsChange: atts.tooltip.content && atts.tooltip.content.length
					}, [
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),

					// FARBEN
					el( StylingControlPanel, {
						title: __( "Colors", 'greyd_forms' ),
						initialOpen: false,
						supportsHover: true,
						supportsActive: true,
						blockProps: props,
						parentAttr: 'panelStyles',
						classAttr: 'panelClass',
						controls: [
							{
								label: __( "Tile color", 'greyd_forms' ),
								attribute: "backgroundColor",
								control: ColorPopupControl,
							},
							{
								label: __( "Text color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
							{
								label: __( "Opacy", 'greyd_forms' ),
								attribute: "opacity",
								control: RangeUnitControl,
								min: 0,
								max: 100,
								units: [ "%" ]
							},
						]
					} ),

					// GRÖSSEN
					el( AdvancedPanelBody, {
						title: __( "Sizes", 'greyd_forms' ),
						holdsChange: atts.imgStyles.height.length || atts.wrapperStyles.alignItems.length || ( atts.panelStyles.width && atts.panelStyles.width.length ),
						initialOpen: false
					}, [
						el( RangeUnitControl, {
							label: __( "Image height", 'greyd_forms' ),
							value: atts.imgStyles.height,
							min: 0,
							max: { px: 300, em: 10, rem: 10 },
							units: [ "px", "%", "vh", "em", "rem" ],
							onChange: ( value ) => props.setAttributes( { imgStyles: { ...atts.imgStyles, height: value } } ),
						} ),
						el( RangeUnitControl, {
							label: __( "Tile width", 'greyd_forms' ),
							value: atts.panelStyles.width,
							min: 0,
							max: { px: 300, em: 10, rem: 10, '%': 100 },
							units: [ "px", "em", "rem", "%" ],
							onChange: ( value ) => props.setAttributes( { panelStyles: { ...atts.panelStyles, width: value } } )
						} ),
						el( ToggleControl, {
							label: atts.wrapperStyles.flexDirection === 'column' ? __( "Same width", 'greyd_forms' ) : __( "Same height", 'greyd_forms' ),
							help: atts.wrapperStyles.flexDirection === 'column' ? __( "Should the tiles be stretched to full width?", 'greyd_forms' ) : __( "Should the tiles be stretched to the same height?", 'greyd_forms' ),
							checked: atts.wrapperStyles.alignItems === 'stretch',
							onChange: ( value ) => props.setAttributes( {
								wrapperStyles: {
									...atts.wrapperStyles,
									alignItems: value ? 'stretch' : atts.wrapperStyles.flexDirection === 'column' ? 'flex-start' : 'center'
								}
							} )
						} ),
					] ),

					// ABSTÄNDE
					el( StylingControlPanel, {
						title: __( "Spaces", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						parentAttr: 'panelStyles',
						controls: [
							{
								label: __( 'Padding', 'greyd_forms' ),
								attribute: "padding",
								control: DimensionControl,
							},
						]
					} ),

					// TYPO
					el( StylingControlPanel, {
						title: __( "Typography", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						parentAttr: 'panelStyles',
						controls: [
							{
								attribute: "fontSize",
								control: FontSizePicker,
								// fontSizes: greyd.data.fontSizes
							},
						]
					} ),

					// RADIUS
					el( StylingControlPanel, {
						title: __( "Border radius", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						parentAttr: 'panelStyles',
						controls: [ {
							label: __( "Border radius", 'greyd_forms' ),
							attribute: "borderRadius",
							control: DimensionControl,
							labels: {
								"all": __( "all corners", "greyd_forms" ),
							},
							sides: [ "topLeft", "topRight", "bottomRight", "bottomLeft" ],
							type: "string"
						} ]
					} ),

					// BORDER
					el( StylingControlPanel, {
						title: __( "Border", 'greyd_forms' ),
						blockProps: props,
						supportsHover: true,
						supportsActive: true,
						parentAttr: 'panelStyles',
						classAttr: 'panelClass',
						controls: [
							{
								label: __( "Border", 'greyd_forms' ),
								attribute: "border",
								control: BorderControl,
							}
						]
					} ),

				] ),

				el( 'div', {
					className: [
						"input-wrapper",
						atts.greydClass,
						( props.className && props.className.length ? props.className : '' )
					].join( " " )
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								value: atts.label,
								onChange: ( value ) => props.setAttributes( { label: value } ),
								placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
							content: tooltipText,
							blockProps: props,
						} ),
					] ) : null,

					// Panels
					el( 'div', {
						className: "img_pnl_wrapper"
					}, [
						el( InnerBlocks, {
							allowedBlocks: [ 'greyd-forms/iconpanel' ],
							orientation: atts.flexDirection && atts.flexDirection == "column" ? "vertical" : "horizontal",
							template: [
								[ 'greyd-forms/iconpanel', { title: __( 'Option 1', 'greyd_forms' ) } ],
								[ 'greyd-forms/iconpanel', { title: __( 'Option 2', 'greyd_forms' ) } ],
							],
							...hasChildBlocks ? {} : { renderAppender: wp.blockEditor.InnerBlocks.ButtonBlockAppender }
						} )
					] )
				] ),

				// wrapper styles
				el( RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						" .block-editor-inner-blocks > div": atts.wrapperStyles,
						" .label": atts.labelStyles,
					},
					important: true
				} ),
				// panel styles
				el( RenderPreviewStyles, {
					selector: atts.greydClass + " .img_pnl",
					styles: {
						"": atts.panelStyles,
						" img, .editor-post-featured-image__preview": atts.imgStyles,
					},
					important: true
				} ),
			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
			const renderTootip = tooltipText.trim().length > 0 ? true : null;


			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( element.Fragment, {}, [
				el( 'div', {
					...blockProps
				}, [
					// Label 
					!atts.label.length ? null : el( 'div', {
						className: "label_wrap",
					}, [
						el( 'label', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: atts.label
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),

						renderTootip && el( 'span', { className: "forms-tooltip", }, [
							el( 'span', {
								className: "forms-tooltip-toggle",
							} ),
							el( 'span', {
								className: "forms-tooltip-popup",
								id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
							}, tooltipText )
						] )
					] ),

					// Fieldset
					el( atts.multiselect ? element.Fragment : 'fieldset', {
						required: !!atts.required,
						'data-required': !!atts.required ? "required" : "optional",
					}, [

						// Panels
						el( 'div', {
							className: [
								"img_pnl_wrapper",
								( atts.multiselect ? 'greyd_multiradio' : '' )
							].join( ' ' ),
						}, [
							// Control (multiselect)
							atts.multiselect ? el( 'input', {
								type: 'text',
								name: makeInputName( atts.name, renderPlaceholder.id ),
								...( renderTootip ? {
									'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
								} : {} ),
								value: '',
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ) : null,

							// InnerBlocks
							el( InnerBlocks.Content, {
								// allowedBlocks: [ 'greyd-forms/iconpanel' ], 
							} )
						] )
					] )

				] ),

				// wrapper styles
				el( RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .img_pnl_wrapper": atts.wrapperStyles,
						" .label": atts.labelStyles,
					},
					important: true
				} ),
				// panel styles
				el( RenderSavedStyles, {
					selector: atts.greydClass + ' .img_pnl',
					styles: {
						"": { ...atts.panelStyles, active: {} },
						" img": atts.imgStyles,
					},
					important: true
				} ),
				_.has( atts.panelStyles, 'active' ) ? el( RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .img_pnl.selected, input:checked + .img_pnl, input:checked + span + .img_pnl": atts.panelStyles.active,
					},
					important: true
				} ) : null,
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/iconpanels' )
	} );

	blocks.registerBlockType( 'greyd-forms/iconpanel', {
		title: __( "Image panel", 'greyd_forms' ),
		description: __( "Panel with image and text", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_panel.svg" } ),
		category: 'greyd-forms',
		parent: [ 'greyd-forms/iconpanels' ],
		attributes: {

			title: {
				type: 'string',
				// source: 'html',
				// selector: 'span',
				default: ''
			},
			value: { type: 'string', default: '' },

			// parent
			name: { type: 'string', default: '' },
			multiselect: { type: "boolean", default: false },
			required: { type: "boolean", default: false },
			panelClass: { type: 'string', default: '' },

			// image
			images: {
				type: 'object', properties: {
					normal: { type: "object" },
					hover: { type: "object" },
					active: { type: "object" },
					current: { type: 'string' }
				}, default: {
					normal: { id: -1, url: "" },
					hover: { id: -1, url: "" },
					active: { id: -1, url: "" },
					current: 'normal'
				}
			},
		},

		usesContext: [ 'greyd-forms/iconpanel-name', 'greyd-forms/iconpanel-multiselect', 'greyd-forms/iconpanel-required', 'greyd-forms/iconpanel-panelClass' ],

		edit: function ( props ) {

			// basic atts
			props.setAttributes( { name: props.context[ 'greyd-forms/iconpanel-name' ] } );
			props.setAttributes( { multiselect: props.context[ 'greyd-forms/iconpanel-multiselect' ] } );
			props.setAttributes( { required: props.context[ 'greyd-forms/iconpanel-required' ] } );
			props.setAttributes( { panelClass: props.context[ 'greyd-forms/iconpanel-panelClass' ] } );
			const atts = props.attributes;

			// image atts
			const tabs = [
				{ label: __( "Normal", "greyd_forms" ), slug: "normal" },
				{ label: __( "Hover", "greyd_forms" ), slug: "hover" },
				{ label: __( "Active", "greyd_forms" ), slug: "active" }
			];
			const emptyImage = { id: -1, url: '' };
			const currentTab = props.isSelected ? atts.images.current : 'normal';
			const currentImage = _.has( atts.images, currentTab ) ? atts.images[ currentTab ] : emptyImage;
			const currentImageIsset = _.has( currentImage, 'id' ) && currentImage.id > 0;

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true },
						el( wp.components.TextControl, {
							label: __( "Value", 'greyd_forms' ),
							value: atts.value,
							autofillPrefix: 'option',
							onChange: ( value ) => props.setAttributes( { value: value } ),
							help: __( "If you leave this field empty, the value is generated from the label.", 'greyd_forms' )
						} ),
					),
					el( PanelBody, { title: __( "Images", 'greyd_forms' ), initialOpen: true },

						// tabs
						el( "div", {
							className: "greyd_tabs"
						}, tabs.map( tab => {
							return el( "span", {
								className: "tab" + ( currentTab === tab.slug ? " active" : "" ),
								onClick: () => props.setAttributes( { images: { ...atts.images, current: tab.slug } } ),
							}, tab.label );
						} ) ),

						// image control
						el( wp.components.BaseControl, {}, [
							el( wp.blockEditor.MediaUploadCheck, {
								fallback: el( 'p', {
									className: "greyd-inspector-help"
								}, __( "To edit the image, you must be authorized to upload media.", 'greyd_forms' ) )
							}, [
								el( wp.blockEditor.MediaUpload, {
									allowedTypes: 'image/*',
									value: currentImage.id,
									onSelect: ( value ) => props.setAttributes( { images: { ...atts.images, [ currentTab ]: { id: value.id, url: value.url } } } ),
									render: ( obj ) => {
										return el( wp.components.Button, {
											className: currentImage.id === -1 ? 'editor-post-featured-image__toggle' : 'editor-post-featured-image__preview',
											onClick: obj.open
										}, currentImage.id == -1 ? __( "Select image", 'greyd_forms' ) : el( 'img', { src: currentImage.url } ) );
									},
								} ),
								// delete button
								currentImage.id === -1 ? null : el( wp.components.Button, {
									className: "is-link is-destructive",
									onClick: () => props.setAttributes( { images: { ...atts.images, [ currentTab ]: emptyImage } } ),
								}, __( "Remove image", 'greyd_forms' ) )
							] ),
						] )
					),
				] ),

				// preview
				el( 'label', {
					className: 'img_pnl animate_fast _' + currentTab + ( atts.panelClass ? ' ' + atts.panelClass : '' ),
					for: makeInputName( atts.value )
				}, [
					el( wp.blockEditor.MediaUploadCheck, {
						fallback: el( 'p', {
							className: "greyd-inspector-help"
						}, __( "To edit the image, you must be authorized to upload media.", 'greyd_forms' ) )
					}, [
						el( wp.blockEditor.MediaUpload, {
							allowedTypes: 'image/*',
							value: currentImage,
							onSelect: ( value ) => props.setAttributes( { images: { ...atts.images, [ currentTab ]: { id: value.id, url: value.url } } } ),
							render: ( obj ) => {
								return el( 'span', {
									className: currentImageIsset ? 'img_wrap' : 'editor-post-featured-image__preview',
									onClick: obj.open
								}, !currentImageIsset ? null : el( 'img', {
									src: currentImage.url,
									className: 'icon_ghost icon_default icon_hover icon_select'
								} ) );
							},
						} ),
					] ),
					el( RichText, {
						tagName: 'span',
						className: 'title',
						value: atts.title,
						onChange: ( value ) => props.setAttributes( { title: value } ),
						placeholder: __( 'Option', 'greyd_forms' ),
						allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
					} )
				] )
			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const value = makeInputName( atts.value.length > 0 ? atts.value : atts.title );
			const id    = makeInputName( atts.value.length > 0 ? atts.value : atts.title, renderPlaceholder.id );
			const name  = makeInputName( atts.name, renderPlaceholder.id );
			const blockProps = wp.blockEditor.useBlockProps.save();
			
			// build the images
			let image_class = 'icon_select icon_hover';
			let images = [];
			if ( atts.images.active && atts.images.active.id > 0 ) {
				images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast' } ) );
				image_class = 'icon_hover'; // remove select from class
			}
			if ( atts.images.hover && atts.images.hover.id > 0 ) {
				images.push( el( 'img', { src: atts.images.hover.url, className: image_class + ' animate_fast' } ) );
				image_class = ''; // remove hover & select from class
			}
			if ( atts.images.normal && atts.images.normal.id > 0 ) {
				images.push( el( 'img', { src: atts.images.normal.url, className: image_class + ' icon_default animate_fast' } ) );
				images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost' } ) );
			}

			return el( element.Fragment, {}, [

				// input
				atts.multiselect ? null : el( 'input', {
					type: 'radio',
					id: id,
					value: value,
					name: name,
					required: !!atts.required,
					'data-required': !!atts.required ? "required" : "optional",
				} ),

				// clickable label/div
				el( atts.multiselect ? 'div' : 'label', {
					className: 'img_pnl animate_fast option ' + ( _.has(blockProps, 'className') ? blockProps.className : '' ),
					...atts.multiselect ? {
						'data-value': value,
						'onclick': 'window.forms.onMultiRadioClick(this);',
					} : {
						for: id
					}
				}, [
					// images
					el( 'div', { className: 'img_wrap' }, images ),
					// title
					atts.title.length ? el( wp.blockEditor.RichText.Content, {
						tagName: 'span',
						value: atts.title
					} ) : null,
				] )
			] );
		},
		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/iconpanel' )
	} );

	/**
	 * Upload
	 */
	blocks.registerBlockType( 'greyd-forms/upload', {
		title: __( 'Upload', 'greyd_forms' ),
		description: __( "File upload by the user", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_upload.svg" } ),
		category: 'greyd-forms',
		keywords: [ __( 'upload', 'greyd_forms' ), __( 'media', 'greyd_forms' ), __( 'file', 'greyd_forms' ) ],
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		styles: [
			{
				name: 'prim',
				label: __( "primary field", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( "secondary field", 'greyd_forms' )
			},
		],
		attributes: {
			// main
			name: { type: "string", default: '' },
			required: { type: "boolean", default: false },
			placeholder: { type: "string", default: '' },

			// file
			file: {
				type: 'object', properties: {
					type: { type: "string" },
					custom: { type: "string" },
					max: { type: "number" },
				}, default: {
					type: "",
					custom: "",
					max: 2
				}
			},

			// label
			label: {
				type: 'string',
				// source: 'html',
				// selector: 'span',
				default: ''
			},
			labelStyles: { type: "object", default: {} },

			// icon
			icon: {
				type: 'object', properties: {
					content: { type: "string" },
					position: { type: "string" },
					size: { type: "string" },
					margin: { type: "string" },
				}, default: {
					content: 'icon_upload',
					position: 'before',
					size: '100%',
					margin: '10px'
				}
			},

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }
				}, default: {
					content: '',
					visible: false
				}
			},

			// styles
			greydClass: { type: "string", default: "" },
			greydStyles: { type: "object" },
			customStyles: { type: 'object' },
			custom: { type: 'boolean', default: false }
		},
		example: {
			attributes: {
				type: 'email',
				label: __( "File", 'greyd_forms' ),
				placeholder: __( "please select", 'greyd_forms' ),
				required: true,
			}
		},

		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( 'file', atts.file ) : atts.tooltip.content;

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [

					// MAIN
					el( PanelBody, {
						title: __( "General", 'greyd_forms' ),
						initialOpen: true
					}, [
						// name
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: 'upload',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),
						// required
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							// help: __("The user must fill in this field.", 'greyd_forms'),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),
					] ),

					// FILE
					el( PanelBody, {
						title: __( "File settings", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( SelectControl, {
							label: __( "File type", 'greyd_forms' ),
							help: __( "What type of file should be uploaded?", 'greyd_forms' ),
							value: atts.file.type,
							options: [
								{ label: __( "all file types", 'greyd_forms' ), value: "*" },
								{ label: __( "ZIP archive", 'greyd_forms' ), value: ".zip" },
								{ label: __( 'PDF', 'greyd_forms' ), value: "application/pdf" },
								{ label: __( "text file", 'greyd_forms' ), value: "text/*" },
								{ label: __( "audio", 'greyd_forms' ), value: "audio/*" },
								{ label: __( "video", 'greyd_forms' ), value: "video/*" },
								{ label: __( "Image", 'greyd_forms' ), value: "image/*" },
								{ label: __( "other (manual input)", 'greyd_forms' ), value: "other" },
							],
							onChange: ( value ) => props.setAttributes( { file: { ...atts.file, type: value } } ),
						} ),
						atts.file.type == 'other' ? el( TextControl, {
							label: __( "Allowed file formats", 'greyd_forms' ),
							help: __( "Which file formats should be allowed (example: .xlsx, .xls, .doc)?", 'greyd_forms' ),
							value: atts.file.custom,
							onChange: ( value ) => props.setAttributes( { file: { ...atts.file, custom: value } } ),
						} ) : null,
						el( NumberControl, {
							label: __( "Maximum file size (in MB)", 'greyd_forms' ),
							help: __( "Maximum allowed file size in MB (e.g. 1.5)", 'greyd_forms' ),
							value: atts.file.max,
							step: 0.1,
							min: 0,
							max: 25,
							onChange: ( value ) => props.setAttributes( { file: { ...atts.file, max: value } } ),
						} ),
					] ),

					// ICON
					el( greyd.components.ButtonIconControl, {
						value: props.attributes.icon,
						onChange: ( value ) => props.setAttributes( { icon: value } )
					} ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),

					// HILFESTELLUNG
					el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [

						// // Placeholder
						// el( TextControl, {
						// 	label: __("Placeholder", 'greyd_forms'),
						// 	help: __("Placeholder text in the input field. (default: please select)", 'greyd_forms'),
						// 	value: atts.placeholder,
						// 	onChange: function(value) { props.setAttributes( { placeholder: value } ); },
						// } ),

						// Hilfetext
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),

					// BREITE
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [ {
							label: __( "Width", 'greyd_forms' ),
							max: 800,
							attribute: "width",
							control: RangeUnitControl,
						} ]
					} ),

					// CUSTOM
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual field", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: atts.custom
					}, [
						el( ToggleControl, {
							label: __( "Overwrite the design of the field individually", 'greyd_forms' ),
							checked: atts.custom,
							onChange: ( value ) => props.setAttributes( { custom: value } )
						} ),
					] ),
					atts.custom && el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// Preview
				el( 'div', {
					className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
				}, [
					// Label 
					props.isSelected || atts.label.length > 0 ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								value: atts.label,
								onChange: ( value ) => props.setAttributes( { label: value } ),
								placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
							content: tooltipText,
							blockProps: props,
						} ),
					] ) : null,

					// Inner Input Wrapper
					el( 'div', {
						className: "input-inner-wrapper",
					}, [
						// Input Field
						el( 'div', {
							className: 'input ' + ( atts.className ?? '' ),
							style: { display: 'flex' }
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: atts.icon,
								position: 'before'
							} ),
							el( wp.blockEditor.RichText, {
								tagName: 'span',
								value: atts.placeholder,
								placeholder: props.isSelected ? __( "Enter a placeholder", 'greyd_forms' ) : __( "please select", 'greyd_forms' ),
								withoutInteractiveFormatting: true,
								format: 'string',
								allowedFormats: [],
								onChange: ( value ) => props.setAttributes( { placeholder: stripTags( value ) } ),
								style: { flex: '2' },

							} ),
							el( greyd.components.RenderButtonIcon, {
								value: atts.icon,
								position: 'after'
							} ),
						] ),
					] ),
				] ),

				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						".input-wrapper": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
				// custom styles
				!atts.custom ? null : el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass + " .input",
					styles: {
						"": atts.customStyles,
						".rich-text [data-rich-text-placeholder]:after": { color: 'inherit', opacity: 0.5 },
					},
					important: true
				} )
			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( 'file', atts.file ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
			const renderTootip = tooltipText.trim().length > 0 ? true : null;

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( element.Fragment, {}, [

				// Input Wrapper
				el( 'div', {
					...blockProps
				}, [
					// Label 
					!atts.label.length ? null : el( 'div', {
						className: "label_wrap",
					}, [
						el( 'label', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: atts.label
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),

						renderTootip && el( 'span', { className: "forms-tooltip", }, [
							el( 'span', {
								className: "forms-tooltip-toggle",
							} ),
							el( 'span', {
								className: "forms-tooltip-popup",
								id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
							}, tooltipText )
						] )
					] ),

					// Input
					el( "input", {
						type: "file",
						className: "custom-file-upload",
						accept: atts.file.type !== 'other' ? atts.file.type : atts.file.custom,
						name: makeInputName( atts.name, false ),
						...( renderTootip ? {
							'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
						} : {} ),
						title: atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter,
						id: makeInputName( atts.name, renderPlaceholder.id ),
						"data-max": atts.file.size ? atts.file.size : "none",
						required: !!atts.required,
						'data-required': !!atts.required ? "required" : "optional",
					} ),

					el( 'label', {
						for: makeInputName( atts.name, renderPlaceholder.id ),
						"data-text": atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter
					}, [
						el( 'div', {
							className: "input " + ( atts.className ?? '' ),
						}, [
							el( greyd.components.RenderButtonIcon, {
								value: atts.icon,
								position: 'before'
							} ),
							el( 'span', {
								style: { flex: '1' },
								dangerouslySetInnerHTML: { __html: atts.placeholder.length ? atts.placeholder : renderPlaceholder.select }
							} ),
							el( greyd.components.RenderButtonIcon, {
								value: atts.icon,
								position: 'after'
							} ),
						] ),
					] ),
				] ),
				
				el( RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						" .label": atts.labelStyles,
						...atts.custom ? {
							" .input": atts.customStyles,
							" .input::placeholder": { color: 'inherit', opacity: 0.5 }
						} : {}
					}
				} )
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/upload' )
	} );

	/**
	 * Conditions
	 */
	blocks.registerBlockType( 'greyd-forms/condition', {
		title: __( "Conditional fields", 'greyd_forms' ),
		description: __( "Container for interdependent fields that update live.", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_conditional.svg" } ),
		category: 'greyd-forms',
		supports: {
			customClassName: true,
		},
		attributes: {
			operator: { type: 'string', default: 'OR' },
			conditions: { type: 'array', default: [] },
			hideOnload: { type: 'bool', default: true },
			disableReset: { type: 'bool', default: false },
			animation: { type: 'string', default: 'slide' },
		},

		edit: function ( props ) {

			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );
			const defaultCondition = {
				field: '',
				condition: 'not_empty',
				value: '',
			};

			let fields = {
				'': { label: _x( "please select", 'small', 'greyd_forms' ), value: '' },
			};
			for ( const [ key, value ] of Object.entries( greyd.forms.helper.getFormBlocks( [], 'name' ) ) ) {
				fields[ key ] = { value: key, label: key + ' (' + value.replace( 'greyd-forms/', '' ) + ')' };
			};
			const operators = {
				is: { label: _x( "has the value:", 'small', 'greyd_forms' ), value: 'is' },
				has: { label: _x( "contains the value:", 'small', 'greyd_forms' ), value: 'has' },
				is_not: { label: _x( "has not the value:", 'small', 'greyd_forms' ), value: 'is_not' },
				empty: { label: _x( "is empty", 'small', 'greyd_forms' ), value: 'empty' },
				not_empty: { label: _x( "is not empty", 'small', 'greyd_forms' ), value: 'not_empty' },
				greater: { label: _x( "greater (>)", 'small', 'greyd_forms' ), value: 'greater' },
				greater_equal: { label: _x( "greater than or equal to (>=)", 'small', 'greyd_forms' ), value: 'greater_equal' },
				smaller: { label: _x( "smaller (<)", 'small', 'greyd_forms' ), value: 'smaller' },
				smaller_equal: { label: _x( "less than or equal to (<=)", 'small', 'greyd_forms' ), value: 'smaller_equal' },
			};
			const animations = [
				{ label: _x( "slide", 'small', 'greyd_forms' ), value: 'slide' },
				{ label: _x( "fade-in & -out", 'small', 'greyd_forms' ), value: 'fade' },
				{ label: _x( "slide & fade", 'small', 'greyd_forms' ), value: 'slide fade' },
				{ label: _x( "none", 'small', 'greyd_forms' ), value: 'none' },
			];

			return [

				// sidebar
				el( InspectorControls, {}, [
					// operator
					el( PanelBody, {
						title: __( "Relation of the condition", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( greyd.components.ButtonGroupControl, {
							value: props.attributes.operator,
							options: [
								{ value: 'OR', label: __( "OR", 'greyd_forms' ) },
								{ value: 'AND', label: __( "AND", 'greyd_forms' ) },
							],
							onChange: ( value ) => { props.setAttributes( { operator: value } ); },
						} ),
						el( wp.components.Tip, {}, __( "With AND all conditions must be true, with OR at least one.", 'greyd_forms' ) ),
					] ),
					// conditions
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Conditions", 'greyd_forms' ),
						holdsChange: props.attributes.conditions.length > 0,
						initialOpen: true
					}, [
						el( 'div', {
							className: 'components-greyd-controlgroup',
						}, [
							...props.attributes.conditions.map( ( condition, i ) => {

								condition = { ...defaultCondition, ...condition };

								const groupComponents = [];

								// remove
								groupComponents.push( el( wp.components.Button, {
									className: "components-greyd-controlgroup__remove",
									onClick: ( event ) => {
										const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
										const newConditions = [ ...props.attributes.conditions ];
										newConditions.splice( index, 1 );
										props.setAttributes( { conditions: newConditions } );
									},
									title: __( "Remove condition", 'greyd_forms' )
								}, el( wp.components.Icon, { icon: 'no-alt' } ) ) );

								// field
								groupComponents.push( el( greyd.components.OptionsControl, {
									label: __( "Field", 'greyd_forms' ),
									value: condition.field,
									options: Object.values( fields ),
									onChange: ( value, event ) => {
										const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
										const newConditions = [ ...props.attributes.conditions ];
										newConditions[ index ].field = value;
										props.setAttributes( { conditions: newConditions } );
									},
								} ) );

								// operator
								groupComponents.push( el( greyd.components.OptionsControl, {
									label: __( "Condition", 'greyd_forms' ),
									value: condition.condition,
									options: Object.values( operators ),
									onChange: ( value, event ) => {
										const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
										const newConditions = [ ...props.attributes.conditions ];
										newConditions[ index ].condition = value;
										props.setAttributes( { conditions: newConditions } );
									},
								} ) );

								if ( condition.condition !== 'empty' && condition.condition !== 'not_empty' ) {

									// value
									groupComponents.push(
										el( BaseControl, { style: { marginBottom: 0 } }, [
											el( BaseControl.VisualLabel, {}, __( "Value", 'greyd_forms' ) ),
											el( 'input', {
												value: condition.value,
												placeholder: __( "Value", 'greyd_forms' ),
												className: 'components-text-control__input',
												onInput: ( event ) => {
													const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
													const newConditions = [ ...props.attributes.conditions ];
													newConditions[ index ].value = event.target.value;
													props.setAttributes( { conditions: newConditions } );
												},
											} ),
											el( 'p', { className: 'components-tip', style: { fontSize: '12px' } }, __( "For checkboxes, the values 'on' and 'off' apply.", 'greyd_forms' ) )
										] )
									);
								}

								return el( 'div', {
									className: 'components-greyd-controlgroup__item',
									'data-index': i,
								}, groupComponents );
							} ),
							el( wp.components.Button, {
								className: 'components-greyd-controlgroup__add' + ( props.attributes.conditions.length === 0 ? ' group_is_empty' : '' ),
								onClick: ( event ) => {
									props.setAttributes( { conditions: [ ...props.attributes.conditions, defaultCondition ] } );
									// console.log(props.attributes.conditions);
								},
								title: __( "Add condition", 'greyd_forms' )
							}, [
								el( wp.components.Icon, { icon: 'plus-alt2' } ),
								props.attributes.conditions.length === 0 ? el( 'span', {}, __( "Add condition", 'greyd_forms' ) ) : null
							] )
						] )
					] ),
					// Animation
					el( greyd.components.AdvancedPanelBody, {
						title: __( 'Animation', 'greyd_forms' ),
						holdsChange: props.attributes.animation !== 'slide',
						initialOpen: true
					}, [
						el( wp.components.SelectControl, {
							label: __( "Type of animation", 'greyd_forms' ),
							value: props.attributes.animation,
							options: animations,
							onChange: ( value ) => props.setAttributes( { animation: value } ),
						} ),
						el( wp.components.ToggleControl, {
							label: __( "hide at start", 'greyd_forms' ),
							checked: props.attributes.hideOnload,
							onChange: ( value ) => { props.setAttributes( { hideOnload: value } ); },
						} ),
						el( wp.components.ToggleControl, {
							label: __( "Leave containers open after submission?", 'greyd_forms' ),
							checked: props.attributes.disableReset,
							onChange: ( value ) => { props.setAttributes( { disableReset: value } ); },
							help: __( "If enabled, the container is not reset.", 'greyd_forms' ),
						} ),
					] ),
				] ),

				// preview
				el( 'div', { className: props.className + ' preview-info-wrapper' }, [
					el( 'div', { className: 'preview-info-tag flex' }, [
						el( 'img', { className: 'preview-info-icon', src: greyd.forms.data.icon_url + "/forms_conditional.svg" } ),
						el( 'div', { className: 'preview-info-title' },
							el( "ul", {}, props.attributes.conditions.map( ( condition, i ) => {
								condition = { ...defaultCondition, ...condition };
								let prefix = i < 1 ? "" : ( props.attributes.operator === 'AND' ? __( "AND", 'greyd_forms' ) : __( "OR", 'greyd_forms' ) ) + " ";
								let label = _.has( fields, condition.field ) ? fields[ condition.field ].label + " " : condition.field + " <span style='color:red'>(" + __( "Field not found", "greyd_forms" ) + ")</span> ";
								let oprtr = operators[ condition.condition ].label;
								let value = condition.condition === 'empty' || condition.condition === 'not_empty' ? "" : " '" + condition.value + "'";

								return el( "li", { dangerouslySetInnerHTML: { __html: prefix + "<strong>" + label + "</strong>" + oprtr + value + "." } } );
							} ) )
						),
					] ),
					el( 'div', { className: 'preview-info-content' }, [
						el( InnerBlocks, { renderAppender: hasChildBlocks ? InnerBlocks.DefaultBlockAppender : InnerBlocks.ButtonBlockAppender } )
					] )
				] )
			];
		},
		save: function ( props ) {

			let conditions = props.attributes.conditions.map( ( condition, i ) => {
				return '%' + [
					condition.field.replace( /___id.*$/, '' ), // remove id from field name
					condition.condition,
					condition.value
				].join( '%%' ) + '%';
			} );
			conditions = conditions.join( props.attributes.operator === 'AND' ? '&&' : '||' );

			return el( 'div', {
				className: 'condition_wrapper open ' + props.attributes.animation + ( props.attributes.className ? ' ' + props.attributes.className : '' ),
				'data-after': props.attributes.disableReset ? 'hold' : 'reset',
			}, [
				el( 'div', {
					'data-condition': conditions,
					className: 'condition_block ' + ( props.attributes.hideOnload ? 'hide' : 'show' ),
				}, el( InnerBlocks.Content ) )
			] );
		}
	} );

	/**
	 * Google reCAPTCHA
	 */
	blocks.registerBlockType( 'greyd-forms/recaptcha', {
		title: __( 'reCAPTCHA', 'greyd_forms' ),
		description: __( 'Google reCAPTCHA', 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_captcha.svg" } ),
		category: 'greyd-forms',
		attributes: {
			hideBanner: { type: "boolean", default: false },
		},
		example: {
			attributes: {
				hideBanner: false
			},
		},
		edit: function ( props ) {
			return [

				// Inspector sidebar
				el( InspectorControls, {},
					el( PanelBody, {
						title: __( "General", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( ToggleControl, {
							label: __( "Hide banner", 'greyd_forms' ),
							help: el( element.Fragment, {}, [
								el( 'span', {}, __( "By default, the reCAPTCHA watermark is displayed. We do not recommend hiding it, as the watermark is used to represent the privacy policy that applies to the reCAPTCHA. You can find them here: ", 'greyd_forms' ) ),
								el( 'a', { href: 'https://policies.google.com/privacy', target: '_blank' }, __( "Privacy policy", 'greyd_forms' ) ),
								el( element.Fragment, {}, ', ' ),
								el( 'a', { href: 'https://policies.google.com/terms', target: '_blank' }, __( "Terms", 'greyd_forms' ) ),
							] ),
							checked: props.attributes.hideBanner,
							onChange: ( value ) => props.setAttributes( { hideBanner: value } )
						} )
					] )
				),

				// Preview
				el( 'div', { className: props.attributes.className + ' preview-info-wrapper' }, [
					el( 'div', { className: 'preview-info-tag flex' }, [
						el( 'img', { className: 'preview-info-icon', src: greyd.forms.data.icon_url + "/forms_captcha.svg", } ),
						el( 'div', { className: 'preview-info-title' }, [
							el( 'strong', {}, __( "reCAPTCHA banner", 'greyd_forms' ) + ( props.attributes.hideBanner ? " (" + __( "hidden", 'greyd_forms' ) + ")" : "" ) ),
							_.isEmpty( greyd.forms.data.captcha_sitekey ) ? el( 'p', {
								// className: 'message danger',
								style: {
									fontSize: 'smaller',
									lineHeight: 1.4,
									marginBottom: 0,
									color: '#cc1818',
								}
							}, __( "Please enter your Google reCAPTCHA site key via 'Forms > Settings', otherwise the reCAPTCHA cannot be executed.", "greyd_forms" ) ) : null
						] )
					] ),
				] )
			];
		},
		save: function ( props ) {
			return el( 'div', {
				id: "captcha",
				className: ( props.attributes.className ?? '' ) + ( props.attributes.hideBanner ? ' hide' : '' )
			}, [
				el( 'input', {
					type: "hidden",
					name: "reCAPTCHA",
					className: "recaptcha_input",
					id: makeInputName( 'recaptcha', renderPlaceholder.id ),
					value: "not_verfied",
					required: true
				} ),
				el( 'script', {}, "var recaptchaSitekey = '" + renderPlaceholder.captcha + "' ;" ),
				el( 'script', {
					src: "https://www.google.com/recaptcha/api.js?render=" + renderPlaceholder.captcha,
				} ),
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/recaptcha' )
	} );

	/**
	 * Hidden Field
	 */
	blocks.registerBlockType( 'greyd-forms/hidden-field', {
		title: __( "Hidden field", 'greyd_forms' ),
		description: __( "Transmission of hidden values", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_hidden.svg" } ),
		category: 'greyd-forms',

		attributes: {
			name: { type: "string", default: '' },
			value: { type: "string", default: '' },
			isDynamic: { type: "boolean", default: false },
			type: { type: "string", default: '' },
			default: { type: "string", default: '' },
		},
		example: {
			attributes: {
				name: 'name',
				value: 'value'
			},
		},

		edit: function ( props ) {

			const types = {
				'': {
					label: __( "URL parameter", 'greyd_forms' ),
					value: '',
					labelValue: __( "Name of the URL parameter", 'greyd_forms' ),
					help: __( "Example: For the URL www.website.de/?source=google you have to enter 'source' in this field, so that the value of the hidden field becomes 'google'.", 'greyd_forms' ),
				},
				cookie: {
					label: __( "cookie", 'greyd_forms' ),
					value: 'cookie',
					labelValue: __( "Name of the cookie", 'greyd_forms' ),
					help: __( "Enter the unique name of the cookie here.", 'greyd_forms' ),
				},
				url: {
					label: __( 'URL', 'greyd_forms' ),
					value: 'url',
					labelValue: false
				},
				quiz: {
					label: __( 'Quiz', 'greyd_forms' ),
					value: 'quiz',
					// labelValue: __( 'Anzahl richtige Antworten', 'greyd_forms' ),
					// help: __( "Anzahl der richtigen Antworten", 'greyd_forms' ),
				},
			};

			return [
				el( InspectorControls, {}, [

					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// Feldname
						el( InputNameControl, {
							value: props.attributes.name,
							autofillPrefix: 'hidden',
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						//  isDynamic?
						el( ToggleControl, {
							label: __( "dynamic field value", 'greyd_forms' ),
							checked: props.attributes.isDynamic,
							onChange: ( value ) => props.setAttributes( { isDynamic: value } )
						} ),

						// value
						...( props.attributes.isDynamic ? [
							el( greyd.components.ButtonGroupControl, {
								label: __( "Value type", 'greyd_forms' ),
								value: props.attributes.type,
								options: Object.values( types ),
								onChange: ( value ) => props.setAttributes( { type: value } )
							} ),
							types[ props.attributes.type ].labelValue ? el( TextControl, {
								label: types[ props.attributes.type ].labelValue,
								help: types[ props.attributes.type ].help,
								value: props.attributes.value,
								onChange: ( value ) => props.setAttributes( { value: value } )
							} ) : null,
							types[ props.attributes.type ].labelValue ? el( TextControl, {
								label: __( "Default value", 'greyd_forms' ),
								help: __( "The field has this value as soon as the %s is not set.", 'greyd_forms' ).replace( '%s', types[ props.attributes.type ].label ),
								value: props.attributes.default,
								onChange: ( value ) => props.setAttributes( { default: value } )
							} ) : null,
							props.attributes.type == 'quiz' ? el( wp.components.BaseControl, { style: { display: "flex", marginBottom: "20px" } }, 
								el( Tip, {}, __( "This hidden field saves the number of correct answers in the frontend.", 'greyd_forms' ) ) 
							) : null,
						] : [
							el( TextControl, {
								label: __( "Field value", 'greyd_forms' ),
								value: props.attributes.value,
								onChange: ( value ) => props.setAttributes( { value: value } )
							} )
						] )
					] )
				] ),

				// preview
				el( 'div', { className: props.attributes.className + ' preview-info-wrapper' }, [
					el( 'div', { className: 'preview-info-tag flex' }, [
						el( 'img', { className: 'preview-info-icon', src: greyd.forms.data.icon_url + "/forms_hidden.svg", } ),
						el( 'div', {
							className: 'preview-info-title', dangerouslySetInnerHTML: {
								__html: props.attributes.name + ": " + "<strong>" + ( props.attributes.isDynamic ? types[ props.attributes.type ].label + " '" + props.attributes.value + "'" : props.attributes.value ) + "</strong>"
							}
						} )
					] )
				] ),
			];
		},

		save: function ( props ) {
			return el( 'input', {
				type: "hidden",
				name: props.attributes.name,
				value: props.attributes.isDynamic ? props.attributes.default : props.attributes.value,
				...(
					props.attributes.type == 'quiz' ? {
						'data-correct-answers-field': true
					} : {}
				)
			} );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/hidden-field' )
	} );

	/**
	 * Math
	 * @todo After initial blocks release
	 */

	/**
	 * Submit Button
	 */
	blocks.registerBlockType( 'greyd-forms/submitbutton', {
		title: __( "Submit button", 'greyd_forms' ),
		description: __( "Trigger to submit the form", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_button.svg" } ),
		category: 'greyd-forms',
		keywords: [ 'link', 'trigger', 'submit', 'button', 'senden', 'absenden' ],
		styles: [
			{
				name: 'prim',
				label: __( 'Primary', 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( 'Secondary', 'greyd_forms' )
			},
			{
				name: 'trd',
				label: __( 'Alternative', 'greyd_forms' )
			}
		],
		supports: {
			anchor: true,
			align: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},
		example: {
			attributes: {
				content: __( "Submit form", 'greyd_forms' ),
				icon: {
					content: 'arrow_right',
					position: 'after',
					size: '100%',
					margin: '10px'
				},
			},
		},

		attributes: {
			inline_css: { type: 'string' },
			inline_css_id: { type: 'string' },
			greydClass: { type: 'string', default: '' },
			greydStyles: { type: 'object' },
			customStyles: { type: 'object' },
			size: { type: 'string', default: '' },
			content: { type: 'string', default: '' },
			icon: {
				type: 'object', properties: {
					content: { type: "string" },
					position: { type: "string" },
					size: { type: "string" },
					margin: { type: "string" },
				}, default: {
					content: '',
					position: 'after',
					size: '100%',
					margin: '10px'
				}
			},
			custom: { type: 'boolean', default: false }
		},

		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );

			return [
				// sidebar
				el( InspectorControls, {}, [
					// size
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Size", 'greyd_forms' ),
						holdsChange: !isEmpty( props.attributes.size )
					}, [
						el( greyd.components.ButtonGroupControl, {
							value: props.attributes.size,
							// label: __( "Size", 'greyd_forms' ),
							options: [
								{ value: "small", label: __( "small", 'greyd_forms' ) },
								{ value: "", label: __( "default", 'greyd_forms' ) },
								{ value: "big", label: __( "big", 'greyd_forms' ) },
							],
							onChange: function ( value ) {
								props.setAttributes( { size: value } );
							},
						} ),
					] ),
					// width
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __( "Width", 'greyd_forms' ),
								attribute: "width",
								control: greyd.components.RangeUnitControl,
								max: 500
							}
						]
					} ),
					el( greyd.components.ButtonIconControl, {
						value: props.attributes.icon,
						onChange: function ( value ) {
							props.setAttributes( { icon: value } );
						},
					} ),

					// custom button
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual button", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: props.attributes.custom
					}, [
						el( ToggleControl, {
							label: __( "Overwrite the design of the button individually", 'greyd_forms' ),
							checked: props.attributes.custom,
							onChange: function ( value ) {
								props.setAttributes( { custom: !!value } );
							},
						} ),
					] ),
					!props.attributes.custom ? null : el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// preview
				el( 'div', {
					className: [
						"button submitbutton",
						( props.className && props.className.length > 0 ? props.className : '' ),
						props.attributes.greydClass,
						( props.attributes.size && props.attributes.size.length > 0 ? 'is-size-' + props.attributes.size : '' ),
					].join( ' ' ),
				}, [
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'before'
					} ),
					el( wp.blockEditor.RichText, {
						format: 'string',
						tagName: 'span',
						style: { flex: '1' },
						value: props.attributes.content,
						placeholder: __( "submit", 'greyd_forms' ),
						allowedFormats: [ 'core/bold', 'core/italic', 'core/strikethrough', 'greyd/versal' ],
						onChange: function ( value ) {
							props.setAttributes( { content: value } );
						},
					} ),
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'after'
					} ),
				] ),
				
				// styles
				el( greyd.components.RenderPreviewStyles, {
					selector: props.attributes.greydClass + '.submitbutton',
					styles: {
						'': {
							...props.attributes.greydStyles,
							...props.attributes.custom ? props.attributes.customStyles : {}
						},
					}
				} ),
			];
		},
		save: function ( props ) {

			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [
					"button submitbutton",
					props.attributes.greydClass,
					( props.attributes.size && props.attributes.size.length > 0 ? 'is-size-' + props.attributes.size : '' )
				].join( ' ' )
			} );

			return el( element.Fragment, {}, [
				el( "button", {
					...blockProps,
					type: "submit",
					name: "submit",
					title: __( "submit", 'greyd_forms' ),
					"data-text": props.attributes.content/*.replace(/(<([^>]+)>)/gi, "")*/,
				}, [
					// loader & progress bar
					el( 'span', {
						className: "greyd_upload_progress"
					}, [
						el( 'span', { className: "spinner" }, [
							el( 'span', { className: "dot-loader" } ),
							el( 'span', { className: "dot-loader dot-loader--2" } ),
							el( 'span', { className: "dot-loader dot-loader--3" } ),
						] ),
						el( 'span', { className: "bar" } ),
					] ),
					// icon
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'before'
					} ),
					// text
					el( 'span', {
						style: { flex: '1' },
						dangerouslySetInnerHTML: {
							__html: props.attributes.content.length ? props.attributes.content : __( "submit", 'greyd_forms' )
						}
					} ),
					// icon
					el( greyd.components.RenderButtonIcon, {
						value: props.attributes.icon,
						position: 'after'
					} )
				] ),
				el( greyd.components.RenderSavedStyles, {
					selector: props.attributes.greydClass + '.submitbutton',
					styles: {
						'': {
							...props.attributes.greydStyles,
							...props.attributes.custom ? props.attributes.customStyles : {}
						},
					}
				} ),
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/submitbutton' )
	} );


	/**
	 * Multistep
	 */
	blocks.registerBlockType( 'greyd-forms/multistep-container', {
		title: __( "Multistep container", 'greyd_forms' ),
		description: __( "Create a multi step form", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_steps.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			anchor: true,
			// defaultStylePicker: false
		},
		attributes: {
			multistepMode: { type: "string", default: ""},
			transition: { type: "string", default: 'swipe' },
			autostep: { type: "boolean", default: true },
			activeStep: { type: "number", default: 0 },
			progressbar: {
				type: 'object', properties: {
					enable: { type: "boolean" },
					type: { type: "string" },
					align: { type: "string" }
				}, default: {
					enable: true,
					type: "progress",
					align: "center"
				}
			},

			correctAnswers: { type: "object", default: {}},

			// styles
			greydClass: { type: "string", default: "" },
			progressbarStyles: { type: "object", default: {} },
		},
		example: {
			attributes: {
				activeStep: 1
			},
			innerBlocks: [
				{
					name: 'greyd-forms/multistep-step',
					attributes: {
						active: true,
					},
					innerBlocks: [
						{
							name: 'greyd-forms/input',
							attributes: {
								label: __( 'Label', 'greyd_forms' ),
								placeholder: __( "enter here", 'greyd_forms' ),
							}
						}
					]
				}
			]
		},

		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );
			const atts = props.attributes;

			const container = wp.data.select( "core/block-editor" ).getBlocksByClientId( props.clientId )[ 0 ];

			// console.log(container);
			const steps = container ? container.innerBlocks : [];

			// deactivate all steps
			steps.forEach( step => {
				wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( step.clientId, { active: false } );
			} );

			const activeStepIndex = atts.activeStep > 0 ? atts.activeStep - 1 : atts.activeStep;
			const activeStep = container ? container.innerBlocks[ activeStepIndex ] : [];

			// get the visible step numbers
			const maxSteps = 7;

			if ( activeStep !== undefined ) {
				wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( activeStep.clientId, { active: true } );
			}

			// progressbar
			const progressWidth = ( ( 100 / ( steps.length + 1 ) ) * atts.activeStep ) + "%";
			if ( atts.progressbar.type == "pagination" && atts.progressbar.activePaginationIndex == undefined ) {
				props.setAttributes( { progressbar: { ...atts.progressbar, activePaginationIndex: atts.activeStep } } );
			}

			const progressbarColors = {
				" .multistep_pagination > span": { backgroundColor: atts.progressbar.backgroundColor },
				" .multistep_progress": { backgroundColor: atts.progressbar.backgroundColor },
				" .multistep_pagination > span.active": { backgroundColor: atts.progressbar.activeColor },
				" .multistep_progress .bar": { backgroundColor: atts.progressbar.activeColor }
			};

			// mode for progress bar settings
			var [ mode, setMode ] = wp.element.useState( "" );
			if ( !props.isSelected && mode != "" ) setMode( "" );

			return [
				// Inspector sidebar
				el( InspectorControls, {}, [

					( mode == "" ) ? [
						// ALLGEMEIN
						el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

							el( greyd.components.ButtonGroupControl, {
								label: __( "Mode", 'greyd_forms' ),
								value: atts.multistepMode,
								options: [
									{ label: __( 'Normal', 'greyd_forms' ), value: '' },
									{ label: __( 'Quiz', 'greyd_forms' ), value: 'quiz' },
								],
								onChange: ( value ) => props.setAttributes( { multistepMode: value }  )
							} ),

							atts.multistepMode == "quiz" ? el( wp.components.BaseControl, { style: { display: "flex", marginBottom: "20px" } }, 
								el( Tip, {}, __( "Please add only one question per step.", 'greyd_forms' ) ) 
							) : null,

							el( SelectControl, {
								label: __( "Transition", 'greyd_forms' ),
								value: atts.transition,
								options: [
									{ label: __( "swipe horizontally", 'greyd_forms' ), value: "swipe" },
									{ label: __( "move vertically", 'greyd_forms' ), value: "move" },
									{ label: __( "fade", 'greyd_forms' ), value: "fade" },
									{ label: __( "none", 'greyd_forms' ), value: "none" },
								],
								onChange: ( value ) => props.setAttributes( { transition: value } )
							} ),


							atts.multistepMode != "quiz" ? el( ToggleControl, {
								label: __( "go to next step automatically", 'greyd_forms' ),
								help: __( "When all fields of a step have been filled in, the container will automatically go to the next step", 'greyd_forms' ),
								checked: atts.autostep,
								onChange: ( value ) => props.setAttributes( { autostep: value } )
							} ) : null,
						] ),

						// ELEMENTE
						el( PanelBody, { title: __( "Elements", 'greyd_forms' ), initialOpen: true }, [

							// Progressbar
							el( 'div', { className: 'is-flex-space-between' },

								el( wp.components.ToggleControl, {
									label: __( "Progress display", 'greyd_forms' ),
									checked: atts.progressbar.enable,
									onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, enable: value } } )
								} ),
								( atts.progressbar.enable ) ? el( wp.components.Button, {
									icon: 'admin-tools',
									style: { height: '20px' },
									onClick: function () { setMode( "progressbar" ); }
								} ) : el( 'div' ),
							),
						] )
					] : el( wp.components.Button, {
						icon: 'arrow-left-alt',
						style: { height: '20px' },
						onClick: function () { setMode( "" ); }
					}, __( "general settings", 'greyd_forms' ) ),

					// numbers
					( mode == "progressbar" && atts.progressbar.enable ) ? [

						el( PanelBody, { title: __( "Appearance", 'greyd_forms' ), initialOpen: true }, [

							el( greyd.components.ButtonGroupControl, {
								label: __( "Type", 'greyd_forms' ),
								value: atts.progressbar.type,
								options: [
									{ label: __( "Bar", 'greyd_forms' ), value: 'progress' },
									{ label: __( "Show Pagination", 'greyd_forms' ), value: 'pagination' },
								],
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, type: value } } )
							} ),


							el( greyd.components.ButtonGroupControl, {
								label: __( "Alignment", 'greyd_forms' ),
								value: atts.progressbar.align,
								options: [
									{ label: __( "left", 'greyd_forms' ), value: 'start' },
									{ label: __( "center", 'greyd_forms' ), value: 'center' },
									{ label: __( "right", 'greyd_forms' ), value: 'end' },
								],
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, align: value } } )
							} ),


						] ),

						el( PanelBody, { title: __( "Colors", 'greyd_forms' ), initialOpen: false }, [
							el( greyd.components.ColorPopupControl, {
								label: __( "active color", 'greyd_forms' ),
								value: atts.progressbar.activeColor,
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, activeColor: value } } )

							} ),
							el( greyd.components.ColorPopupControl, {
								label: __( "Background color", 'greyd_forms' ),
								value: atts.progressbar.backgroundColor,
								onChange: ( value ) => props.setAttributes( { progressbar: { ...atts.progressbar, backgroundColor: value } } )
							} )
						] ),

						// Größe
						atts.progressbar.type == "pagination" ? [
							el( greyd.components.StylingControlPanel, {
								title: __( "Size", 'greyd_forms' ),
								initialOpen: false,
								supportsResponsive: false,
								blockProps: props,
								parentAttr: 'progressbarStyles',
								controls: [
									{
										label: __( "Height", 'greyd_forms' ),
										max: 50,
										attribute: "height",
										control: RangeUnitControl,
									},
									{
										label: __( "Width", 'greyd_forms' ),
										max: 50,
										attribute: "width",
										control: RangeUnitControl,
									},
								]
							} ),
							el( PanelBody, { title: __( "Space", 'greyd_forms' ), initialOpen: false },
								el( greyd.components.RangeUnitControl, {
									label: __( "space between", 'greyd_forms' ),
									value: atts.progressbarStyles.marginLeft,
									onChange: ( value ) => {

										let doubleMargin = {};
										const progressbarStyles = _.has( atts, "progressbarStyles" ) ? atts.progressbarStyles : {};

										if ( value ) {
											doubleMargin = { marginLeft: value, marginRight: value };
										} else {
											delete atts.progressbarStyles.marginLeft;
											delete atts.progressbarStyles.marginRight;
										}

										props.setAttributes( {
											progressbarStyles: { ...progressbarStyles, ...doubleMargin },
										} );
									}
								} )
							)
						] : (
							el( greyd.components.StylingControlPanel, {
								title: __( "Size", 'greyd_forms' ),
								initialOpen: false,
								supportsResponsive: false,
								blockProps: props,
								parentAttr: 'progressbarStyles',
								controls: [
									{
										label: __( "Height", 'greyd_forms' ),
										max: 100,
										attribute: "height",
										control: RangeUnitControl,
									},
									{
										label: __( "Width", 'greyd_forms' ),
										max: 1200,
										attribute: "width",
										control: RangeUnitControl,
									},
								]
							} )
						),
						// border-radius
						el( StylingControlPanel, {
							title: __( "Border radius", 'greyd_forms' ),
							initialOpen: false,
							blockProps: props,
							parentAttr: 'progressbarStyles',
							controls: [ {
								label: __( "Border radius", 'greyd_forms' ),
								attribute: "borderRadius",
								control: RangeUnitControl,
							} ]
						} ),
					] : []

				] ),

				// preview toolbar
				el( wp.components.Placeholder, { className: "multistep-toolbar" }, [

					el( wp.components.BaseControl, { style: { display: "flex" } }, [

						// no steps yet
						steps.length > 0 ? null : el( 'div', {}, __( "Create the first step.", 'greyd_forms' ) ),

						// select steps
						steps.length < 1 ? null : el( 'div', { style: { display: "flex" } }, [
							el( wp.components.Button, {
								icon: "arrow-left-alt2",
								className: "multistep-button " + ( steps.length < 2 || atts.activeStep < 2 ? "hidden" : "" ),
								onClick: () => props.setAttributes( { activeStep: atts.activeStep - 1 } )
							} ),
							...steps.map( ( block, i ) => {

								const step = i + 1;
								let icon = null;
								let className = "";
								let text = step;

								if ( step == 1 || step == steps.length ) {
									// last and first step are always visible
								} else if ( steps.length > maxSteps ) {

									// we're on the left
									if ( atts.activeStep < maxSteps - 2 ) {
										if ( step < maxSteps - 1 ) {
											// visible
										} else if ( step == maxSteps - 1 ) {
											text = "…";
										} else {
											className = "hidden";
										}
									}
									// we're on the right
									else if ( atts.activeStep > maxSteps - 2 ) {
										if ( step > maxSteps - 3 ) {
											// visible
										} else if ( step == maxSteps - 3 ) {
											text = "…";
										} else {
											className = "hidden";
										}
									}
									// we're in the middle
									else {
										if ( step == atts.activeStep - 2 ) {
											text = "…";
										} else if ( step == atts.activeStep + 2 ) {
											text = "…";
										} else if ( step < atts.activeStep - 2 ) {
											className = "hidden";
										} else if ( step > atts.activeStep + 2 ) {
											className = "hidden";
										}
									}

									// if ( step == 2 && atts.activeStep > 4 ) {
									// 	text = "…";
									// }
									// else if ( step == steps.length -1 && atts.activeStep < 7 ) {
									// 	text = "…";
									// }
								}

								if ( step == atts.activeStep ) {
									className = "is-active";
								}

								return el( wp.components.Button, {
									icon: icon,
									className: "multistep-button " + className,
									onClick: () => props.setAttributes( { activeStep: step } )
								}, text );
							} ),
							el( wp.components.Button, {
								icon: "arrow-right-alt2",
								className: "multistep-button " + ( steps.length < 2 || atts.activeStep >= steps.length ? "hidden" : "" ),
								onClick: () => props.setAttributes( { activeStep: atts.activeStep + 1 } )
							} ),
						] ),

						// add and remove
						el( 'div', {},
							steps.length < 1 ? null : el( wp.components.Button, {
								className: 'is-tertiary',
								onClick: function () {
									const activeStepIndex = atts.activeStep - 1;
									const activeStep = wp.data.select( "core/block-editor" ).getBlocksByClientId( props.clientId )[ 0 ].innerBlocks[ activeStepIndex ];

									if ( activeStep !== undefined ) {
										wp.data.dispatch( "core/block-editor" ).removeBlock( activeStep.clientId );
										// if the active step doesnt change, the edit function won't get triggerd. this is a really hacky way to trigger the edit function
										props.setAttributes( { activeStep: atts.activeStep, date: new Date } );
									}
								},
							}, __( "Remove step", 'greyd_forms' ) ),

							el( wp.components.Button, {
								className: 'is-primary',
								onClick: function () {
									const innerCount = wp.data.select( "core/block-editor" ).getBlocksByClientId( props.clientId )[ 0 ].innerBlocks.length;
									let block = blocks.createBlock( "greyd-forms/multistep-step" );
									wp.data.dispatch( "core/block-editor" ).insertBlock( block, innerCount, props.clientId );
								},
							}, ( steps.length ? __( "new step", 'greyd_forms' ) : __( "add step", 'greyd_forms' ) ) )
						),
					] )
				] ),

				// preview
				el( "div", { className: props.className + " multistep_preview_wrapper" }, [
					// steps
					el( "div", {
						className: [ "multistep_wrapper", atts.greydClass, atts.autostep ? "auto_step" : '' ].join( ' ' ),
						"data-transition": atts.transition
					}, [
						el( InnerBlocks, {
							allowedBlocks: [ 'greyd-forms/multistep-step' ],
							renderAppender: false,
							// template: [
							// 	[ 'greyd-forms/multistep-step', { active: true } ],
							// 	[ 'greyd-forms/multistep-step', { active: false } ],
							// ],
							// ...hasChildBlocks ? {} : { renderAppender: wp.blockEditor.InnerBlocks.ButtonBlockAppender }
						} )
					] ),

					// progress bar
					!atts.progressbar.enable ? null : el( 'div', {
						className: [ "progress_wrapper", "flex-" + atts.progressbar.align, atts.greydClass ].join( ' ' ),
						onClick: () => setMode( "progressbar" )
					}, [
						el( 'div', { className: 'multistep_' + atts.progressbar.type },

							atts.progressbar.type == "progress" ? el( 'span', { className: 'bar', style: { width: progressWidth } } ) :

								steps.map( ( step, index ) => {

									return el( 'span', {
										style: { cursor: "pointer" },
										className: index + 1 == atts.progressbar.activePaginationIndex ? "active" : "",
										onClick: () => {
											props.setAttributes( { activeStep: index + 1, progressbar: { ...atts.progressbar, activePaginationIndex: index + 1 } } );
										}
									} );
								} )
						)
					] )
				] ),

				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass + ".progress_wrapper",
					styles: {
						" .multistep_progress, .multistep_pagination > span": atts.progressbarStyles,
						...progressbarColors
					},
					important: true
				} )

			];
		},

		save: function ( props ) {

			const atts = props.attributes;
			const blockProps = wp.blockEditor.useBlockProps.save();

			const progressbarColors = {
				" .multistep_pagination > span": { backgroundColor: atts.progressbar.backgroundColor },
				" .multistep_progress": { backgroundColor: atts.progressbar.backgroundColor },
				" .multistep_pagination > span.active": { backgroundColor: atts.progressbar.activeColor },
				" .multistep_progress .bar": { backgroundColor: atts.progressbar.activeColor }
			};

			return el( element.Fragment, {}, [
				// steps
				el( "div", {
					...blockProps,
					className: [
						"multistep_wrapper",
						atts.greydClass,
						(atts.autostep ? "auto_step" : ''),
						( _.has(blockProps, 'className') ? blockProps.className : '' )
					].join( ' ' ),
					"data-transition": atts.transition,
					"data-step": 0,
					...(
						atts.multistepMode == "quiz" ? {
							"data-quiz": "true",
							"data-correct-answer-text": __( "Correct answer:", 'greyd_forms' ),
							"data-wrong-answer-text": __( "Wrong answer. The correct answer is:", 'greyd_forms' )
						} : {}
					)
				}, [
					el( InnerBlocks.Content )
				] ),


				// progress bar
				atts.progressbar.enable ? el( 'div', {
					className: [ "progress_wrapper", "flex-" + atts.progressbar.align, atts.greydClass ].join( ' ' ),
				},
					el( 'div', { className: 'multistep_' + atts.progressbar.type }, [

						atts.progressbar.type == "progress" ? el( 'span', { className: 'bar' } ) : ''
					] )
				) : '',

				el( greyd.components.RenderSavedStyles, {
					selector: atts.greydClass + ".progress_wrapper",
					styles: {
						" .multistep_progress, .multistep_pagination > span": atts.progressbarStyles,
						...progressbarColors
						// " .bar": "background-color:"
					},
					important: true
				} )
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/multistep-container' ),
	} );

	blocks.registerBlockType( 'greyd-forms/multistep-step', {
		title: __( "Multistep step", 'greyd_forms' ),
		// description: __('...', 'greyd_forms'),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_steps.svg" } ),
		category: 'greyd-forms',
		parent: [ 'greyd-forms/multistep-container' ],
		attributes: {
			active: { type: "boolean", default: false },
		},

		edit: function ( props ) {

			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );
			const atts = props.attributes;

			const parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/multistep-container" );
			const parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];

			if ( props.isSelected ) {
				// const selected = wp.data.select("core/block-editor").getSelectedBlock();
				const steps = parent.innerBlocks;

				steps.forEach( ( step, index ) => {
					if ( step.clientId === props.clientId ) {
						wp.data.dispatch( 'core/block-editor' ).updateBlockAttributes( parent.clientId, { activeStep: index + 1 } );
					}
				} );
				setTimeout( function () {
					wp.data.dispatch( 'core/block-editor' ).selectBlock( parent.clientId );
					$( '#block-' + parent.clientId ).focus();
				}, 0 );
			}

			const allBlocks = wp.blocks.getBlockTypes().map( block => block.name );
			const disallowedBlocks = [ "greyd-forms/multistep-container", "greyd-forms/multistep-step" ];
			const allowedFormBlocksQuiz = [ "greyd-forms/radiobuttons", "greyd-forms/input", "greyd-forms/submitbutton", "greyd-forms/hidden-field", "greyd-forms/condition" ];

			let allowedBlocks = [];

			if ( parent && parent.attributes.multistepMode == "quiz") {
				allowedBlocks = allBlocks.filter( function ( value, index ) {
					if ( allowedFormBlocksQuiz.includes(value)) return true
					else if ( value.startsWith("greyd-forms/")) return false
					else return disallowedBlocks.indexOf( value ) == -1;
				} );
			} else {
				allowedBlocks = allBlocks.filter( function ( value, index ) {
					return disallowedBlocks.indexOf( value ) == -1;
				} );
			}

			if ( parent && parent.attributes.multistepMode == "quiz" ) {
				let correctAnswer = {};
				const formBlocks = greyd.forms.helper.getFormBlocks(props.innerBlocks);

				Object.keys(formBlocks).forEach( (key, index) => {
					if ( formBlocks[key].name == "greyd-forms/radiobuttons" ) {
						formBlocks[key].innerBlocks.forEach( block => {
							if ( block.attributes.correctAnswer ) {
								correctAnswer[formBlocks[key].attributes.name] = block.attributes.value;
							}
						})
					} else if (formBlocks[key].attributes.correctAnswer) {
						correctAnswer[formBlocks[key].attributes.name] = formBlocks[key].attributes.correctAnswer;
					}
				});
				props.attributes.correctAnswer = correctAnswer;
			}

			return [
				el( "div", {
					className: [ "step", atts.active ? "active" : "" ].join( ' ' ),
				},
					el( InnerBlocks, {
						renderAppender: InnerBlocks.ButtonBlockAppender,
						allowedBlocks: allowedBlocks,
						template: [ [ 'core/paragraph', {} ], [ 'greyd-forms/multistep-button', { align: 'right' } ] ],
					} )
				),
			];
		},

		save: function ( props ) {
			const atts = props.attributes;

			return el( element.Fragment, {}, [
				el( "div", {
					className: [ "step" ].join( ' ' ),
				},
					el( InnerBlocks.Content )
				),
			] );
		}

	} );

	blocks.registerBlockType( 'greyd-forms/multistep-button', {
		title: __( "Next / back button", 'greyd_forms' ),
		description: __( "Navigation between the steps of a multi-level form", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_button.svg" } ),
		category: 'greyd-forms',
		parent: [ 'greyd-forms/multistep-step', 'greyd-forms/condition', 'core/column', 'core/group' ],
		styles: [
			{
				name: 'prim',
				label: __( 'Primary', 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( 'Secondary', 'greyd_forms' )
			},
			{
				name: 'trd',
				label: __( 'Alternative', 'greyd_forms' )
			},
			{
				name: 'link-prim',
				label: __( "link", 'greyd_forms' )
			},
			{
				name: 'link-sec',
				label: __( "Secondary link", 'greyd_forms' )
			},
		],
		supports: {
			anchor: true,
			align: true,
			defaultStylePicker: false
		},
		example: {
			attributes: {
				content: __( "next", 'greyd_forms' ),
				icon: {
					content: 'arrow_right',
					position: 'after',
					size: '100%',
					margin: '10px'
				},
			},
		},

		attributes: {
			inline_css: { type: 'string' },
			inline_css_id: { type: 'string' },
			greydClass: { type: 'string', default: '' },
			greydStyles: { type: 'object' },
			customStyles: { type: 'object' },
			action: { type: 'string', default: 'next_step' },
			stepNumber: { type: 'string' },

			size: { type: 'string', default: '' },
			content: { type: 'string', default: '' },
			icon: {
				type: 'object', properties: {
					content: { type: "string" },
					position: { type: "string" },
					size: { type: "string" },
					margin: { type: "string" },
				}, default: {
					content: '',
					position: 'after',
					size: '100%',
					margin: '10px'
				}
			},
			custom: { type: 'boolean', default: false }
		},

		edit: function ( props ) {

			props.attributes.greydClass = getGreydClass( props );

			const extraClass = props.className.indexOf( 'is-style-link-' ) === -1 ? 'button' : 'link';

			const atts = props.attributes;

			const parentId = wp.data.select( "core/block-editor" ).getBlockParentsByBlockName( props.clientId, "greyd-forms/multistep-container" );
			const parent = wp.data.select( "core/block-editor" ).getBlocksByClientId( parentId )[ 0 ];
			const steps = _.has( parent, "innerBlocks" ) ? parent.innerBlocks : [];

			return [
				// sidebar
				el( InspectorControls, {}, [
					// size
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Action", 'greyd_forms' ),
					},
						[
							el( greyd.components.ButtonGroupControl, {
								value: props.attributes.action,
								options: [
									{ value: "next_step", label: __( "next", 'greyd_forms' ) },
									{ value: "prev_step", label: __( "back", 'greyd_forms' ) },
									{ value: "exact_step", label: __( "individual", 'greyd_forms' ) },
								],
								onChange: function ( value ) {
									props.setAttributes( { action: value } );
								},
							} ),
							atts.action === "exact_step" ? [ el( NumberControl, {
								label: __( "go to step number", 'greyd_forms' ),
								value: atts.stepNumber,
								step: 1,
								min: 1,
								max: steps.length ? steps.length : null,
								onChange: ( value ) => props.setAttributes( { stepNumber: value } ),
							} ),
							el( 'small', {}, __( "Attention: If mandatory fields are skipped, this can cause problems when submitting the form.", 'greyd_forms' ) ) ] : '',
						]
					),
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Size", 'greyd_forms' ),
						holdsChange: !isEmpty( props.attributes.size )
					},
						[
							el( greyd.components.ButtonGroupControl, {
								value: props.attributes.size,
								// label: __( "Size", 'greyd_forms' ),
								options: [
									{ value: "small", label: __( "small", 'greyd_forms' ) },
									{ value: "", label: __( "default", 'greyd_forms' ) },
									{ value: "big", label: __( "big", 'greyd_forms' ) },
								],
								onChange: function ( value ) {
									props.setAttributes( { size: value } );
								},
							} ),
						]
					),
					// width
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __( "Width", 'greyd_forms' ),
								attribute: "width",
								control: greyd.components.RangeUnitControl,
								max: 500
							}
						]
					} ),
					el( greyd.components.ButtonIconControl, {
						value: props.attributes.icon,
						onChange: function ( value ) {
							props.setAttributes( { icon: value } );
						},
					} ),

					// custom button
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual button", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: props.attributes.custom
					},
						[
							el( ToggleControl, {
								label: __( "Overwrite the design of the button individually", 'greyd_forms' ),
								checked: props.attributes.custom,
								onChange: function ( value ) {
									props.setAttributes( { custom: !!value } );
								},
							} ),
						]
					),
					!props.attributes.custom ? null : el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// preview
				el( 'div', {
					className: "input-outer-wrapper input-wrapper" + ( props.attributes.align ? ' flex-' + props.attributes.align : '' )
				}, [
					el( 'span', {
						id: props.attributes.anchor,
						className: [ 'multistep_button', extraClass, props.attributes.greydClass, ( props.attributes.className ?? '' ), props.attributes.size ].join( ' ' ),
					}, [
						el( greyd.components.RenderButtonIcon, {
							value: props.attributes.icon,
							position: 'before'
						} ),
						el( wp.blockEditor.RichText, {
							format: 'string',
							tagName: 'span',
							style: { flex: '1' },
							value: props.attributes.content,
							placeholder: atts.action == "next_step" ? __( "next", 'greyd_forms' ) : atts.action == "prev_step" ? __( "back", 'greyd_forms' ) : __( "to step X", 'greyd_forms' ),
							allowedFormats: [ 'core/bold', 'core/italic', 'core/strikethrough', 'greyd/versal' ],
							onChange: function ( value ) {
								props.setAttributes( { content: value } );
							},
						} ),
						el( greyd.components.RenderButtonIcon, {
							value: props.attributes.icon,
							position: 'after'
						} ),
					] ),
				] ),
				// styles
				el( greyd.components.RenderPreviewStyles, {
					selector: props.attributes.greydClass + '.multistep_button',
					styles: {
						'': {
							...props.attributes.greydStyles,
							...props.attributes.custom ? props.attributes.customStyles : {}
						},
					}
				} ),
			];
		},
		save: function ( props ) {

			const extraClass = _.has( props.attributes, "className" ) && props.attributes.className.indexOf( 'is-style-link-' ) > -1 ? 'link' : 'button';

			return el( element.Fragment, {}, [
				el( 'div', {
					className: "input-outer-wrapper input-wrapper" + ( props.attributes.align ? ' flex-' + props.attributes.align : '' )
				}, [
					el( "span", {
						id: props.attributes.anchor,
						className: [ 'multistep_button', props.attributes.action, extraClass, props.attributes.greydClass, ( props.attributes.className ?? '' ), props.attributes.size ].join( ' ' ),
						"data-text": props.attributes.content/*.replace(/(<([^>]+)>)/gi, "")*/,
						...( props.attributes.action == "exact_step" && { "data-step": props.attributes.stepNumber } ),
					}, [
						// icon
						el( greyd.components.RenderButtonIcon, {
							value: props.attributes.icon,
							position: 'before'
						} ),
						// text
						el( 'span', {
							style: { flex: '1' },
							dangerouslySetInnerHTML: {
								__html: props.attributes.content.length ? props.attributes.content : ( props.attributes.action == "next_step" ? __( "next", 'greyd_forms' ) : props.attributes.action == "prev_step" ? __( "back", 'greyd_forms' ) : __( "to step X", 'greyd_forms' ) ),
							}
						} ),
						// icon
						el( greyd.components.RenderButtonIcon, {
							value: props.attributes.icon,
							position: 'after'
						} )
					] ),
				] ),
				el( greyd.components.RenderSavedStyles, {
					selector: props.attributes.greydClass + '.multistep_button',
					styles: {
						'': {
							...props.attributes.greydStyles,
							...props.attributes.custom ? props.attributes.customStyles : {}
						},
					}
				} ),
			] );
		}

	} );

	/**
	 * Password Field
	 */
	blocks.registerBlockType( 'greyd-forms/password', {
		title: __( "Password", 'greyd_forms' ),
		description: __( "To enter a password", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/block-greyd-password.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false
		},
		styles: [
			{
				name: 'prim',
				label: __( "primary field", 'greyd_forms' ),
				isDefault: true
			},
			{
				name: 'sec',
				label: __( "secondary field", 'greyd_forms' )
			},
		],
		attributes: {
			// main
			name: { type: "string", default: "" },

			confirmPasswordField: {
				type: "bool",
				default: false
			},

			placeholder: { type: "string", default: '' },
			restrictions: {
				type: 'object', properties: {
					minlength: { type: "number" },
					maxlength: { type: "number" },
					max: { type: "string" },
					min: { type: "string" },
					start: { type: "string" },
					step: { type: "number" },
				}, default: {}
			},


			// label
			label: { type: "string", default: __( "Password", 'greyd_forms' ) },
			labelStyles: { type: "object", default: {} },


			// label
			confirmPasswordLabel: { type: "string", default: __( "Confirm password", 'greyd_forms' ) },
			confirmPasswordLabelStyles: { type: "object", default: {} },

			// label for the message 
			confirmPasswordMatching: { type: "string", default: '' },
			confirmPasswordNotMatching: { type: "string", default: '' },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// styling
			greydStyles: { type: "object" },
			greydClass: { type: "string", default: "" },
			customStyles: { type: 'object' },
			custom: { type: 'boolean', default: false }
		},
		example: {
			attributes: {
				label: __( "Password", 'greyd_forms' ),
				placeholder: __( '123456', 'greyd_forms' ),
			}
		},

		edit: function ( props ) {

			// generate attributes
			props.attributes.greydClass = getGreydClass( props );

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( "password", atts ) : atts.tooltip.content;

			return [

				el( InspectorControls, {}, [

					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [

						// Feldname
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: atts.type == 'text' ? 'input' : atts.type,
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						// Pflichtfeld
						el( ToggleControl, {
							label: __( "Confirm password", 'greyd_forms' ),
							help: __( "A second field is rendered with which the user must confirm the password.", 'greyd_forms' ),
							checked: !!atts.confirmPasswordField,
							onChange: ( value ) => props.setAttributes( { confirmPasswordField: !!value } )
						} ),
					] ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),

					// BREITE
					el( greyd.components.StylingControlPanel, {
						title: __( "Width", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [ {
							label: __( "Width", 'greyd_forms' ),
							max: 800,
							attribute: "width",
							control: RangeUnitControl,
						} ]
					} ),

					// HILFESTELLUNG
					el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [

						// Hilfetext
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),

					// Password Restrictions
					el( PanelBody, { title: __( "Password restrictions", 'greyd_forms' ), initialOpen: false }, [
						// Zeichenanzahl
						el( NumberControl, {
							label: __( "Minimum character length", 'greyd_forms' ),
							help: __( "How many characters must be entered at least?", 'greyd_forms' ),
							value: atts.restrictions.minlength,
							onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, minlength: value } } )
						} ),
						el( NumberControl, {
							label: __( "Maximum character length", 'greyd_forms' ),
							help: __( "How many characters can be entered? (default: unlimited)", 'greyd_forms' ),
							value: atts.restrictions.maxlength,
							onChange: ( value ) => props.setAttributes( { restrictions: { ...atts.restrictions, maxlength: value } } )
						} )
					] ),

					// custom field
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Individual field", 'greyd_forms' ),
						initialOpen: true,
						holdsChange: atts.custom
					}, [
						el( ToggleControl, {
							label: __( "Overwrite the design of the field individually", 'greyd_forms' ),
							checked: atts.custom,
							onChange: ( value ) => props.setAttributes( { custom: value } )
						} ),
					] ),
					atts.custom && el( greyd.components.CustomButtonStyles, {
						blockProps: props,
						parentAttr: 'customStyles'
					} )
				] ),

				// PREVIEW
				el( 'div', {
					className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
				}, [
					// Input Wrapper
					el( 'div', {
						className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
					}, [
						// Label 
						( props.isSelected || atts.label.length > 0 ) && el( 'div', {
							className: "label_wrap",
						}, [
							el( 'span', {
								className: "label",
							}, [
								el( wp.blockEditor.RichText, {
									tagName: "span",
									className: "label-content",
									value: atts.label,
									onChange: ( value ) => props.setAttributes( { label: value } ),
									placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
									allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal', 'greyd/text-background' ],
								} ),
								// required/optional
								el( 'span', {
									className: "requirement-required",
								}, " *" ),
							] ),
							// Tooltip
							!tooltipText.length || !tooltipText.trim().length ? null : el( LabelTooltip, {
								content: tooltipText,
								blockProps: props,
							} ),
						] ),

						// Inner Input Wrapper
						el( 'div', {
							className: "input-inner-wrapper",
						}, [
							// Input Field
							el( 'div', {
								className: 'input ' + ( atts.className ?? '' ),
							}, [
								el( wp.blockEditor.RichText, {
									tagName: 'div',
									value: atts.placeholder,
									placeholder: props.isSelected ? __( "Enter a placeholder", 'greyd_forms' ) : __( "Enter password", 'greyd_forms' ),
									withoutInteractiveFormatting: true,
									format: 'string',
									allowedFormats: [],
									onChange: ( value ) => props.setAttributes( { placeholder: stripTags( value ) } ),

								} ),
							] ),
						] ),
					] ),

					// confirm Password
					atts.confirmPasswordField && el( 'div', {
						className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
					}, [
						// Label 
						( props.isSelected || atts.label.length > 0 ) && el( 'div', {
							className: "label_wrap",
						}, [
							el( 'span', {
								className: "label",
							}, [
								el( wp.blockEditor.RichText, {
									tagName: "span",
									className: "label-content",
									value: atts.confirmPasswordLabel,
									onChange: ( value ) => props.setAttributes( { confirmPasswordLabel: value } ),
									placeholder: props.isSelected ? __( "Enter a label", 'greyd_forms' ) : "",
									allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal', 'greyd/text-background' ],
								} ),
								// required/optional
								el( 'span', {
									className: "requirement-required",
								}, " *" ),
							] ),

						] ),

						// Inner Input Wrapper
						el( 'div', {
							className: "input-inner-wrapper",
						}, [
							// Input Field
							el( 'div', {
								className: 'input ' + ( atts.className ?? '' ),
							}, [
								el( wp.blockEditor.RichText, {
									tagName: 'div',
									value: atts.placeholder,
									placeholder: props.isSelected ? __( "Enter a placeholder", 'greyd_forms' ) : __( "Enter password", 'greyd_forms' ),
									withoutInteractiveFormatting: true,
									format: 'string',
									allowedFormats: [],
									onChange: ( value ) => props.setAttributes( { placeholder: stripTags( value ) } ),

								} ),
							] ),
						] ),
					] )
				] ),
				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						".input-wrapper": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
				// custom styles
				atts.custom && el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass + " .input",
					styles: {
						"": atts.customStyles,
						".rich-text [data-rich-text-placeholder]:after": { color: 'inherit', opacity: 0.5 },
					},
					important: true
				} )

			];
		},

		save: function ( props ) {

			// atts
			const atts = props.attributes;
			const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( atts.type, atts ) : atts.tooltip.content;
			const inputName = _.isEmpty( atts.name ) ? "password" : makeInputName( atts.name, false );

			return el( element.Fragment, {}, [

				el( 'div', {
					className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
				}, [

					// Input Wrapper
					el( 'div', {
						className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
					}, [
						// Label 
						!atts.label.length ? null : el( 'div', {
							className: "label_wrap",
						}, [
							el( 'label', {
								className: "label",
								for: makeInputName( inputName, renderPlaceholder.id ),
							}, [
								el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.label
								} ),
								// required/optional
								el( 'span', {
									className: !!atts.required ? "requirement-required" : "requirement-optional",
								}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
							] ),

							!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
								el( 'span', {
									className: "forms-tooltip-toggle",
								} ),
								el( 'span', {
									className: "forms-tooltip-popup"
								}, tooltipText )
							] )
						] ),

						// Inner Input Wrapper
						el( 'div', {
							className: "input-inner-wrapper",
						}, [
							// Input Field
							el( 'input', {
								className: "input validation " + ( atts.className ?? '' ),
								type: 'password',
								name: inputName,
								id: makeInputName( inputName, renderPlaceholder.id ),
								placeholder: atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' ),
								...{
									...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
									...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
								},

								// ...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "phone" ?
								// 	{
								// 		...{ pattern: getValidationPattern(atts.type) },
								// 	} : {}
								// ),
								required: true,
								'data-required': "required",
								title: __( "Please fill out this field", 'greyd_forms' ),

							} ),
							el( 'span', {
								className: "input-field-icon"
							} )
						] ),
					] ),
					// confirm Password
					atts.confirmPasswordField && el( 'div', {
						className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
					}, [
						// Label 
						el( 'div', {
							className: "label_wrap",
						}, [
							el( 'label', {
								className: "label",
								for: makeInputName( inputName, renderPlaceholder.id ),
							}, [
								el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.confirmPasswordLabel
								} ),
								// required/optional
								el( 'span', {
									className: !!atts.required ? "requirement-required" : "requirement-optional",
								}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
							] ),
						] ),

						// Inner Input Wrapper
						el( 'div', {
							className: "input-inner-wrapper",
						}, [
							// Input Field
							el( 'input', {
								className: "input validation confirm-password " + ( atts.className ?? '' ),
								type: 'password',
								id: "confirm-password-" + makeInputName( inputName, renderPlaceholder.id ),
								placeholder: atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' ),

								required: true,
								'data-required': "required",
								title: __( "Please fill out this field", 'greyd_forms' ),

							} ),
							el( 'span', {
								className: "input-field-icon"
							} ),
						] ),
						el( 'span', {
							className: "confirm-password-message",
							'data-label-matching': atts.confirmPasswordMatching.length ? atts.confirmPasswordMatching : __( "is matching", 'greyd_forms' ),
							'data-label-notmatching': atts.confirmPasswordNotMatching.length ? atts.confirmPasswordMatching : __( "is not matching", 'greyd_forms' )
						} )
					] ),
					el( greyd.components.RenderSavedStyles, {
						selector: atts.greydClass,
						styles: {
							" .label": atts.labelStyles,
							...atts.custom ? {
								" .input": atts.customStyles,
								" .input::placeholder": { color: 'inherit', opacity: 0.5 }
							} : {}
						}
					} ),
				] ),
			] );
		},
	} );

	/**
	 * Input block
	 */
	blocks.registerBlockType( 'greyd-forms/range', {
		title: __( "Slider", 'greyd_forms' ),
		description: __( "For entering text, email, date, number...", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_range.svg" } ),
		category: 'greyd-forms',
		supports: {
			align: true,
			customClassName: true,
			defaultStylePicker: false,
			spacing: {
				margin: [ 'top', 'bottom' ],
				padding: true
			}
		},

		attributes: {
			// main
			name: { type: "string", default: "" },

			required: { type: "boolean", default: false },

			min: { type: "float", default: 0 },
			max: { type: "float", default: 100 },
			start: { type: "float", default: 50 },
			step: { type: "number", default: 1 },

			// label
			labelBefore: { type: "string", default: "" },
			labelAfter: { type: "string", default: "" },

			// tooltip
			tooltip: {
				type: 'object', properties: {
					content: { type: 'string' },
					visible: { type: 'boolean' }

				}, default: {
					content: '',
					visible: false
				}
			},

			// styling
			greydStyles: { type: "object" },
			labelStyles: { type: "object" },
			greydClass: { type: "string", default: "" },
		},
		example: {
			attributes: {
				type: 'email',
				label: __( "email", 'greyd_forms' ),
				placeholder: __( "john@example.com", 'greyd_forms' ),
				required: true,
			}
		},

		edit: function ( props ) {

			// generate attributes
			props.attributes.greydClass = getGreydClass( props );

			// atts
			const atts = props.attributes;

			return [

				el( InspectorControls, {}, [

					// ALLGEMEIN
					el( PanelBody, { title: __( "General", 'greyd_forms' ), initialOpen: true }, [
						// Feldname
						el( InputNameControl, {
							value: atts.name,
							autofillPrefix: atts.type == 'text' ? 'input' : atts.type,
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						// Pflichtfeld
						el( ToggleControl, {
							label: __( "Required field", 'greyd_forms' ),
							help: __( "The user must fill in this field.", 'greyd_forms' ),
							checked: !!atts.required,
							onChange: ( value ) => props.setAttributes( { required: !!value } )
						} ),
					] ),
					el( PanelBody, { title: __( "Values", 'greyd_forms' ), initialOpen: true }, [
						el( NumberControl, {
							label: __( "Start value", 'greyd_forms' ),
							value: atts.start,
							onChange: ( value ) => props.setAttributes( { start: value } )
						} ),
						el( NumberControl, {
							label: __( "Minimum value", 'greyd_forms' ),
							value: atts.min,
							onChange: ( value ) => props.setAttributes( { min: value } )
						} ),
						el( NumberControl, {
							label: __( "Maximum value", 'greyd_forms' ),
							value: atts.max,
							onChange: ( value ) => props.setAttributes( { max: value } )
						} ),
						el( NumberControl, {
							label: __( "Steps", 'greyd_forms' ),
							value: atts.step,
							min: 0.1,
							step: 0.1,
							onChange: ( value ) => props.setAttributes( { step: value } )
						} ),
					] ),

					// LABEL
					el( greyd.components.StylingControlPanel, {
						title: __( 'Label', 'greyd_forms' ),
						supportsHover: true,
						parentAttr: 'labelStyles',
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "fontSize",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "color",
								control: ColorPopupControl,
							},
						]
					} ),
					el( greyd.components.StylingControlPanel, {
						title: __( "Value", 'greyd_forms' ),
						supportsHover: true,
						blockProps: props,
						controls: [
							{
								label: __( "Font size", 'greyd_forms' ),
								attribute: "--range-value-font-size",
								units: [ "px", "em", "rem" ],
								max: { px: 60, em: 3, rem: 3 },
								control: RangeUnitControl,
							},
							{
								label: __( "Color", 'greyd_forms' ),
								attribute: "--range-value-color",
								control: ColorPopupControl,
							},
						]
					} ),

					// BAHN
					// Größen
					el( StylingControlPanel, {
						title: __( "Size", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __( "Width", 'greyd_forms' ),
								max: 800,
								attribute: "--rail-width",
								control: RangeUnitControl,
							},
							{
								label: __( "Height", 'greyd_forms' ),
								max: 100,
								attribute: "--rail-height",
								control: RangeUnitControl,
							}
						]
					} ),
					el( StylingControlPanel, {
						title: __( "Colors", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						controls: [
							{
								label: __( "Rail", 'greyd_forms' ),
								attribute: "--rail-color",
								control: ColorPopupControl,
							},
							{
								label: __( "Rail (active)", 'greyd_forms' ),
								attribute: "--rail-color-focus",
								control: ColorPopupControl
							},
							{
								label: __( "Track", 'greyd_forms' ),
								attribute: "--track-color",
								control: ColorPopupControl
							},
							{
								label: __( "Track (active)", 'greyd_forms' ),
								attribute: "--track-color-focus",
								control: ColorPopupControl
							},
							{
								label: __( "Handle color", 'greyd_forms' ),
								attribute: "--thumb-color",
								control: ColorPopupControl,
							},
							{
								label: __( "Handle color (active)", 'greyd_forms' ),
								attribute: "--thumb-color-focus",
								control: ColorPopupControl,
							},
							{
								label: __( "Handle border color", 'greyd_forms' ),
								attribute: "--thumb-border-color",
								control: ColorPopupControl
							},
							{
								label: __( "Handle border color (active)", 'greyd_forms' ),
								attribute: "--thumb-border-color-focus",
								control: ColorPopupControl
							},
							{
								label: __( "Tooltip background color", 'greyd_forms' ),
								attribute: "--range-tooltip-background-color",
								control: ColorPopupControl
							},
							{
								label: __( "Tooltip text color", 'greyd_forms' ),
								attribute: "--range-tooltip-color",
								control: ColorPopupControl
							},
						]
					} ),

					// BAHN
					// Größen
					el( greyd.components.StylingControlPanel, {
						title: __( "Size", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __( "Handle size", 'greyd_forms' ),
								max: 100,
								attribute: "--thumb-size",
								control: RangeUnitControl,
							},
						]
					} ),
					// Größen
					el( greyd.components.StylingControlPanel, {
						title: __( "Tooltip size", 'greyd_forms' ),
						initialOpen: false,
						supportsResponsive: true,
						blockProps: props,
						controls: [
							{
								label: __( "Tooltip font size", 'greyd_forms' ),
								max: 100,
								attribute: "--range-tooltip-fontsize",
								control: RangeUnitControl,
							},
						]
					} ),
					// border-radius
					el( StylingControlPanel, {
						title: __( "Track border radius", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						controls: [ {
							label: __( "Border radius", 'greyd_forms' ),
							attribute: "--rail-borderradius",
							control: DimensionControl,
							labels: {
								"all": __( "all corners", "greyd_forms" ),
							},
							sides: [ "topLeft", "topRight", "bottomRight", "bottomLeft" ],
							type: "string"
						} ]
					} ),
					el( StylingControlPanel, {
						title: __( "Handle border radius", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						controls: [ {
							label: __( "Border radius", 'greyd_forms' ),
							attribute: "--thumb-borderradius",
							control: DimensionControl,
							labels: {
								"all": __( "all corners", "greyd_forms" ),
							},
							sides: [ "topLeft", "topRight", "bottomRight", "bottomLeft" ],
							type: "string"
						} ]
					} ),
					el( StylingControlPanel, {
						title: __( "Tooltip border radius", 'greyd_forms' ),
						initialOpen: false,
						blockProps: props,
						controls: [ {
							label: __( "Border radius", 'greyd_forms' ),
							attribute: "--range-tooltip-borderradius",
							control: DimensionControl,
							labels: {
								"all": __( "all corners", "greyd_forms" ),
							},
							sides: [ "topLeft", "topRight", "bottomRight", "bottomLeft" ],
							type: "string"
						} ]
					} ),

					// HILFESTELLUNG
					el( PanelBody, { title: __( "Assistance", 'greyd_forms' ), initialOpen: false }, [
						// Hilfetext
						el( TextareaControl, {
							label: __( "Tooltip", 'greyd_forms' ),
							help: __( "For certain field types a help text is generated automatically. You can overwrite it here. Remove the text by entering only blanks.", 'greyd_forms' ),
							value: atts.tooltip.content,
							onChange: function ( value ) { props.setAttributes( { tooltip: { ...atts.tooltip, content: value } } ); },
							onFocus: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: true } } ); },
							onBlur: function () { props.setAttributes( { tooltip: { ...atts.tooltip, visible: false } } ); },
						} )
					] ),
				] ),

				// PREVIEW
				el( 'div', {
					className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
				}, [
					// Label 
					( props.isSelected || atts.labelBefore.length || atts.labelAfter.length ) ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						},
							el( wp.blockEditor.RichText, {
								tagName: "span",
								className: "label-content",
								value: atts.labelBefore,
								onChange: ( value ) => props.setAttributes( { labelBefore: value } ),
								placeholder: props.isSelected ? __( "Label before", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal', 'greyd/text-background' ],
							} ),
						),
						el( 'strong', { className: "range_value" }, " value " ),
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText, {
								tagName: "span",
								className: "label-content",
								value: atts.labelAfter,
								onChange: ( value ) => props.setAttributes( { labelAfter: value } ),
								placeholder: props.isSelected ? __( "Label after", 'greyd_forms' ) : "",
								allowedFormats: [ 'core/bold', 'core/italic', 'core/subscript', 'core/superscript', 'greyd/versal', 'greyd/text-background' ],
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						!atts.tooltip.content.length ? null : el( LabelTooltip, {
							content: atts.tooltip.content,
							blockProps: props,
						} ),
					] ) : null,

					// Inner Input Wrapper
					el( 'div', {
						className: "range_control_wrapper",
					}, [
						el( 'span', {
							className: 'rail',
						} ),
						el( 'span', {
							className: 'track',
						} ),
						el( 'span', {
							className: 'thumb_wrapper',
						}, [
							el( 'span', {
								className: 'thumb',
							} ),
							el( 'span', {
								className: 'tooltip',
								style: {
									opacity: props.isSelected ? 1 : 0,
								}
							}, atts.start ),
						] )
					] )
				] ),
				
				el( greyd.components.RenderPreviewStyles, {
					selector: atts.greydClass,
					styles: {
						".input-wrapper": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
			];
		},

		save: function ( props ) {

			// atts
			const atts = props.attributes;
			const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );
			
			const blockProps = wp.blockEditor.useBlockProps.save( {
				className: [ "input-wrapper", atts.greydClass ].join( ' ' )
			} );

			return el( element.Fragment, {}, [

				// Input Wrapper
				el( 'div', {
					...blockProps
				}, [
					// Label 
					( atts.labelBefore.length || atts.labelAfter.length ) ? el( 'div', {
						className: "label_wrap",
					}, [
						el( 'span', {
							className: "label",
						},
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: atts.labelBefore + " "
							} ),
						),
						el( 'strong', { className: 'range_value' } ),
						el( 'span', {
							className: "label",
						}, [
							el( wp.blockEditor.RichText.Content, {
								tagName: 'span',
								value: " " + atts.labelAfter
							} ),
							// required/optional
							el( 'span', {
								className: !!atts.required ? "requirement-required" : "requirement-optional",
							}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
						] ),
						// Tooltip
						atts.tooltip.content.length === 0 ? null : el( 'span', { className: "forms-tooltip", }, [
							el( 'span', {
								className: "forms-tooltip-toggle",
							} ),
							el( 'span', {
								className: "forms-tooltip-popup",
								id: makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
							}, atts.tooltip.content )
						] )

					] ) : null,

					// Inner Input Wrapper
					el( 'div', {
						className: "range_control_wrapper",
					}, [
						el( 'input', {
							type: 'range',
							name: inputName,
							id: makeInputName( inputName, renderPlaceholder.id ),
							min: atts.min,
							max: atts.max,
							step: atts.step,
							value: isNaN(atts.start) ? 50 : atts.start,
							required: !!atts.required,
							'data-required': !!atts.required ? "required" : "optional",
						} ),
						el( 'span', {
							className: 'rail',
						} ),
						el( 'span', {
							className: 'track',
						} ),
						el( 'span', {
							className: 'thumb_wrapper',
						}, [
							el( 'span', {
								className: 'thumb',
							} ),
							el( 'span', {
								className: 'tooltip',
							}, ),
						] )
					] )
				] ),
				
				el( greyd.components.RenderSavedStyles, {
					selector: atts.greydClass,
					styles: {
						".input-wrapper": atts.greydStyles,
						" .label": atts.labelStyles,
					},
				} ),
			] );
		},

		deprecated: greyd.forms.deprecations.getDeprecations( 'greyd-forms/range' ),
	} );

	/**
	 * Math block
	 */
	blocks.registerBlockType( 'greyd-forms/math', {
		title: __( "Mathematical field", 'greyd_forms' ),
		description: __( "Perform complex calculations of static and dynamic values.", 'greyd_forms' ),
		icon: el( 'img', { src: greyd.forms.data.icon_url + "/forms_math.svg" } ),
		category: 'greyd-forms',
		attributes: {
			name: { type: "string", default: "" },
			formula: { type: "string", default: "" },
			displayResult: { type: "bool", default: false },
			displayFormula: { type: "bool", default: false },
			conditionEnable: { type: "bool", default: false },
			conditions: { type: "array", default: [] },
			decimals: { type: "number", default: 0 },
			rounding: { type: "string", default: "auto" },
		},

		edit: function ( props ) {

			const atts = props.attributes;

			/*
				Form Fields
			*/
			const formBlocks = greyd.forms.helper.getFormBlocks();
			const fields = {};

			// filter blocks
			Object.keys( formBlocks ).forEach( ( key, index ) => {
				fields[ key ] = formBlocks[ key ];
			} );

			/*
				Operators
			*/
			const operators = [ '+', '-', '*', '/', '^()', '√()', 'log()', '(', ')' ];

			// cursor position
			const [ pos, setPos ] = wp.element.useState( 0 );

			const setCursorPos = ( e ) => {
				const cursorPos = e.target.selectionStart;
				setPos( cursorPos );
			};

			const defaultCondition = {
				field1: '',
				field2: '',
				result: '',
				condition: 'greater',
			};

			const conditionalOperators = {
				greater: { label: _x( "greater (>)", 'small', 'greyd_forms' ), value: 'greater' },
				greater_equal: { label: _x( "greater than or equal to (>=)", 'small', 'greyd_forms' ), value: 'greater_equal' },
				smaller: { label: _x( "smaller (<)", 'small', 'greyd_forms' ), value: 'smaller' },
				smaller_equal: { label: _x( "less than or equal to (<=)", 'small', 'greyd_forms' ), value: 'smaller_equal' },
				equal: { label: _x( "equals (=)", 'small', 'greyd_forms' ), value: 'equal' },
			};

			return [

				// sidebar
				el( InspectorControls, {}, [
					// operator
					el( PanelBody, {
						title: __( "General", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( InputNameControl, {
							value: atts.name,
							onChange: ( value ) => props.setAttributes( { name: value } )
						} ),

						el( ToggleControl, {
							label: __( "Show result", 'greyd_forms' ),
							help: __( "The result of the formula is displayed in the frontend", 'greyd_forms' ),
							checked: !!atts.displayResult,
							onChange: ( value ) => props.setAttributes( { displayResult: !!value } )
						} ),

						el( ToggleControl, {
							label: __( "Show formula", 'greyd_forms' ),
							help: __( "The formula is displayed in the frontend", 'greyd_forms' ),
							checked: !!atts.displayFormula,
							onChange: ( value ) => props.setAttributes( { displayFormula: !!value } )
						} ),
						el( ToggleControl, {
							label: __( "Change field value depending on condition", 'greyd_forms' ),
							checked: !!atts.conditionEnable,
							onChange: ( value ) => props.setAttributes( { conditionEnable: !!value } )
						} ),

					] ),
					el( PanelBody, {
						title: __( "Rounding & decimal places", 'greyd_forms' ),
						initialOpen: true
					}, [
						el( NumberControl, {
							label: __( "Decimal places", 'greyd_forms' ),
							help: __( "Number of decimal places of the field value. (default: 0)", 'greyd_forms' ),
							value: atts.decimals,
							onChange: ( value ) => props.setAttributes( { decimals: value } )
						} ),
						el( greyd.components.OptionsControl, {
							value: props.attributes.rounding,
							options: [
								{ value: 'auto', label: __( "automatically", 'greyd_forms' ) },
								{ value: 'ceil', label: __( "round up", 'greyd_forms' ) },
								{ value: 'floor', label: __( "round off", 'greyd_forms' ) },
							],
							onChange: ( value ) => { props.setAttributes( { rounding: value } ); },
						} )
					] )
				] ),

				// preview
				el( 'div', { className: props.className + ' preview-info-wrapper' }, [
					el( 'div', { className: 'preview-info-tag flex' }, [
						el( 'img', { className: 'preview-info-icon', src: greyd.forms.data.icon_url + "/forms_math.svg" } ),
						__( "Mathematical field", 'greyd_forms' )
					] ),
					el( 'div', { className: 'preview-info-content' }, [
						el( wp.components.TextareaControl, {
							value: atts.formula,
							onChange: ( value ) => {
								props.setAttributes( { formula: value } );
								var sel = window.getSelection();
								const cursorPos = sel.focusNode.firstChild.selectionStart;
								setPos( cursorPos );
							},
							onFocus: setCursorPos,
							onClick: setCursorPos,
							rows: props.isSelected ? 3 : 1,
							style: { margin: 0 }
						} ),
						props.isSelected && el( 'div', { className: 'preview-info-form-tags flex' }, [
							el( "div", {}, __( "Fields", 'greyd_forms' ) + ": " ),
							el( "div", {}, Object.keys( fields ).map( fieldName => {
								fieldName = `{${ fieldName }}`;

								return el( "span", {
									className: "form_tag",
									value: fieldName,
									onClick: ( e ) => {

										const formula = atts.formula;
										const newFormula = [ formula.slice( 0, pos ), fieldName, formula.slice( pos ) ].join( '' );
										props.setAttributes( { formula: newFormula } );
										const newPos = pos + fieldName.length;
										setPos( newPos );
									}
								}, fieldName );
							} ) )
						] ),
						props.isSelected && el( 'div', { className: 'preview-info-form-tags flex' }, [
							el( "div", {}, __( "Operators", 'greyd_forms' ) + ": " ),
							el( "div", {}, ( operators ).map( operator => {
								return el( "span", {
									className: "form_tag",
									value: operator,
									onClick: ( e ) => {
										const formula = atts.formula;
										const newFormula = [ formula.slice( 0, pos ), operator, formula.slice( pos ) ].join( '' );
										let newPos = 0;
										props.setAttributes( { formula: newFormula } );
										if ( operator.includes( "()" ) ) {
											newPos = pos + operator.length - 1;
										} else {
											newPos = pos + operator.length;
										}

										setPos( newPos );
									}
								}, operator );
							} ) )
						] )
					] ),

					props.isSelected && atts.conditionEnable && el( 'div', {
						className: 'components-greyd-controlgroup preview-info-content greyd-math-condition-wrapper',
					}, [
						...atts.conditions.map( ( condition, i ) => {

							condition = { ...defaultCondition, ...condition };

							const groupComponents = [];
							const firstRow = [];

							// remove
							groupComponents.push( el( wp.components.Button, {
								className: "components-greyd-controlgroup__remove",
								onClick: ( event ) => {
									const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
									const newConditions = [ ...atts.conditions ];
									newConditions.splice( index, 1 );
									props.setAttributes( { conditions: newConditions } );
								},
								title: __( "Remove condition", 'greyd_forms' )
							}, el( wp.components.Icon, { icon: 'no-alt' } ) ) );

							// IF
							firstRow.push(
								el( 'div', { className: 'greyd-math-condition-input flex' }, [
									el( "div", {}, __( "If", 'greyd_forms' ) + ": " ),
									// INPUT
									el( wp.components.TextareaControl, {
										value: condition.field1,
										onChange: ( value, event ) => {

											const sel = window.getSelection();
											const target = sel.focusNode.firstChild;

											const index = parseInt( target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
											const newConditions = [ ...props.attributes.conditions ];
											newConditions[ index ].field1 = value;
											props.setAttributes( { conditions: newConditions } );

											const cursorPos = sel.focusNode.firstChild.selectionStart;
											setPos( cursorPos );
										},
										rows: 2,
										onFocus: setCursorPos,
										onClick: setCursorPos
									} ),
									// FIELDS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Fields", 'greyd_forms' ) + ": " ),
										el( "div", {}, Object.keys( fields ).map( fieldName => {

											fieldName = `{${ fieldName }}`;

											return el( "span", {
												className: "form_tag",
												value: fieldName,
												onClick: ( e ) => {

													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );

													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].field1;

													const newFormula = [ formula.slice( 0, pos ), fieldName, formula.slice( pos ) ].join( '' );

													newConditions[ index ].field1 = newFormula;
													props.setAttributes( { conditions: newConditions } );

													const newPos = pos + fieldName.length;
													setPos( newPos );
												}
											}, fieldName );
										} ) )
									] ),
									// OPERATORS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Operators", 'greyd_forms' ) + ": " ),
										el( "div", {}, ( operators ).map( operator => {
											return el( "span", {
												className: "form_tag",
												value: operator,
												onClick: ( e ) => {
													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].field1;
													const newFormula = [ formula.slice( 0, pos ), operator, formula.slice( pos ) ].join( '' );

													newConditions[ index ].field1 = newFormula;
													props.setAttributes( { conditions: newConditions } );

													let newPos = 0;
													if ( operator.includes( "()" ) ) {
														newPos = pos + operator.length - 1;
													} else {
														newPos = pos + operator.length;
													}

													setPos( newPos );
												}
											}, operator );
										} ) )
									] )
								] )
							);

							// CONDITIONAL OPERATOR
							firstRow.push(
								el( greyd.components.OptionsControl, {
									label: __( "Condition", 'greyd_forms' ),
									value: condition.condition,
									options: Object.values( conditionalOperators ),
									onChange: ( value, event ) => {
										const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
										const newConditions = [ ...atts.conditions ];
										newConditions[ index ].condition = value;
										props.setAttributes( { conditions: newConditions } );
									},
								} )
							);

							// AS
							firstRow.push(
								el( 'div', { className: 'greyd-math-condition-input flex' }, [
									el( "div", {}, __( "Than", 'greyd_forms' ) + ": " ),
									// INPUT
									el( wp.components.TextareaControl, {
										// value: atts.formula,
										value: condition.field2,
										onChange: ( value ) => {
											const sel = window.getSelection();
											const target = sel.focusNode.firstChild;

											const index = parseInt( target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
											const newConditions = [ ...props.attributes.conditions ];
											newConditions[ index ].field2 = value;
											props.setAttributes( { conditions: newConditions } );

											const cursorPos = sel.focusNode.firstChild.selectionStart;
											setPos( cursorPos );
										},
										rows: 2,
										onFocus: setCursorPos,
										onClick: setCursorPos
									} ),
									// FIELDS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Fields", 'greyd_forms' ) + ": " ),
										el( "div", {}, Object.keys( fields ).map( fieldName => {

											fieldName = `{${ fieldName }}`;

											return el( "span", {
												className: "form_tag",
												value: fieldName,
												onClick: ( e ) => {
													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );

													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].field2;
													const newFormula = [ formula.slice( 0, pos ), fieldName, formula.slice( pos ) ].join( '' );

													newConditions[ index ].field2 = newFormula;
													props.setAttributes( { conditions: newConditions } );

													const newPos = pos + fieldName.length;
													setPos( newPos );
												}
											}, fieldName );
										} ) )
									] ),
									// OPERATORS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Operators", 'greyd_forms' ) + ": " ),
										el( "div", {}, ( operators ).map( operator => {
											return el( "span", {
												className: "form_tag",
												value: operator,
												onClick: ( e ) => {
													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].field2;
													const newFormula = [ formula.slice( 0, pos ), operator, formula.slice( pos ) ].join( '' );

													newConditions[ index ].field2 = newFormula;
													props.setAttributes( { conditions: newConditions } );

													let newPos = 0;
													if ( operator.includes( "()" ) ) {
														newPos = pos + operator.length - 1;
													} else {
														newPos = pos + operator.length;
													}

													setPos( newPos );
												}
											}, operator );
										} ) )
									] )
								] )
							);

							groupComponents.push(
								el( 'div', { className: 'preview-info-math-condition-first-row flex' }, firstRow )
							);

							// THEN
							groupComponents.push(
								el( 'div', { className: 'greyd-math-condition-input flex' }, [
									el( "div", {}, __( "Then", 'greyd_forms' ) + ": " ),
									// INPUT
									el( wp.components.TextareaControl, {
										// value: atts.formula,
										value: condition.result,
										onChange: ( value, event ) => {
											const sel = window.getSelection();
											const target = sel.focusNode.firstChild;

											const index = parseInt( target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
											const newConditions = [ ...props.attributes.conditions ];
											newConditions[ index ].result = value;
											props.setAttributes( { conditions: newConditions } );

											const cursorPos = sel.focusNode.firstChild.selectionStart;
											setPos( cursorPos );
										},
										rows: 2,
										onFocus: setCursorPos,
										onClick: setCursorPos
									} ),
									// FIELDS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Fields", 'greyd_forms' ) + ": " ),
										el( "div", { className: "flex" }, Object.keys( fields ).map( fieldName => {

											fieldName = `{${ fieldName }}`;

											return el( "span", {
												className: "form_tag",
												value: fieldName,
												onClick: ( e ) => {
													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );

													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].result;
													const newFormula = [ formula.slice( 0, pos ), fieldName, formula.slice( pos ) ].join( '' );

													newConditions[ index ].result = newFormula;
													props.setAttributes( { conditions: newConditions } );

													const newPos = pos + fieldName.length;
													setPos( newPos );
												}
											}, fieldName );
										} ) )
									] ),
									// OPERATORS
									el( 'div', { className: 'preview-info-form-tags flex' }, [
										el( "div", {}, __( "Operators", 'greyd_forms' ) + ": " ),
										el( "div", { className: "flex" }, ( operators ).map( operator => {
											return el( "span", {
												className: "form_tag",
												value: operator,
												onClick: ( e ) => {
													const index = parseInt( e.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
													const newConditions = [ ...props.attributes.conditions ];
													const formula = newConditions[ index ].result;
													const newFormula = [ formula.slice( 0, pos ), operator, formula.slice( pos ) ].join( '' );

													newConditions[ index ].result = newFormula;
													props.setAttributes( { conditions: newConditions } );

													let newPos = 0;
													if ( operator.includes( "()" ) ) {
														newPos = pos + operator.length - 1;
													} else {
														newPos = pos + operator.length;
													}

													setPos( newPos );
												}
											}, operator );
										} ) )
									] )
								] )
							);

							return el( 'div', {
								className: 'components-greyd-controlgroup__item',
								'data-index': i,
							}, groupComponents );
						} ),
						el( wp.components.Button, {
							className: 'components-greyd-controlgroup__add' + ( atts.conditions.length === 0 ? ' group_is_empty' : '' ),
							onClick: ( event ) => {
								props.setAttributes( { conditions: [ ...atts.conditions, defaultCondition ] } );
								// console.log(atts.conditions);
							},
							title: __( "Add condition", 'greyd_forms' )
						}, [
							el( wp.components.Icon, { icon: 'plus-alt2' } ),
							atts.conditions.length === 0 ? el( 'span', {}, __( "Add condition", 'greyd_forms' ) ) : null
						] )
					] )
				] )
			];
		},
		save: function ( props ) {
			const atts = props.attributes;

			return el( 'div', {
				className: 'input-wrapper math_operation'
			}, [
				el( 'input', {
					type: 'hidden',
					name: atts.name,
					className: [ 'math_field', atts.conditionEnable ? 'has_conditions' : '' ].join( " " ),
					'data-formula': atts.formula,
					'data-decimals': atts.decimals,
					'data-rounding': atts.rounding
				} ),
				el( 'div', {
					className: [ 'display', atts.displayResult ? 'display_result' : '', atts.displayFormula ? 'display_formula' : '' ].join( " " ),
				}, [
					el( 'span', {
						className: 'formula default',
					}, "||displayformula||" ),
					atts.conditions.map( ( condition, index ) => {
						return el( 'span', {
							className: [ 'formula', `cond_${ index }` ].join( " " ),
						}, `||conditionformula_${ index }||` );
					} ),
					el( 'span', {
						className: 'equal_sign',
					}, " = " ),
					el( 'span', {
						className: 'result',
					}, "||result||" ),
				] ),
				el( 'div', {
					className: "math_conditions",
					"aria-hidden": "true",
				}, [
					atts.conditions.map( ( condition, index ) => {
						return el( 'span', {
							'data-field1': `||field_1_${ index }||`,
							'data-field2': `||field_2_${ index }||`,
							'data-value': `||result_${ index }||`,
							'data-operator': condition.condition,
						} );
					} ),
				] )
			] );
		}
	} );

	/**
	 * RichText format
	 */
	wp.richText.registerFormatType( 'greyd-forms/dynamic-form-tag', {
		title: __( "Dynamic form value", 'greyd_forms' ),
		tagName: 'span',
		className: 'form_tag',
		contenteditable: false,
		attributes: {
			tag: 'data-name',
		},
		edit: function ( props ) {

			// first get the current form blocks
			const formBlocks = greyd.forms.helper.getFormBlocks();
			const fields = [];

			Object.keys( formBlocks ).forEach( ( key, index ) => {
				const fieldName = formBlocks[ key ].attributes.name;
				fields.push( { label: fieldName, value: fieldName } );
			} );

			let { tag: formula } = props.activeAttributes;

			const updateFormat = function ( formula ) {
				props.onChange(
					wp.richText.insert( props.value,
						wp.richText.applyFormat(
							wp.richText.create( {
								text: formula,
							} ), {
							type: 'greyd-forms/dynamic-form-tag',
							attributes: {
								tag: formula,
							}
						}, 0, formula.length
						)
					)
				);
			};

			const PopoverContent = () => {
				if ( isEmpty( formula ) ) {
					return fields.length ? fields.map( option => {
						return el( wp.components.Button, {
							onClick: () => {
								updateFormat( option.value );
							}
						}, [
							el( wp.components.Flex, {
								gap: 3,
								align: 'center',
								justify: 'space-between'
							}, [
								el( wp.components.FlexBlock, {}, option.label )
							] )
						] );
					} ) :
						el( wp.components.Tip, {}, [
							el( 'small', {}, __( "Insert at least one input field first.", 'greyd_forms' ) )
						] );
				}
				return [
					el( wp.components.BaseControl, {},
						el( "h4", {}, __( "Dynamic form value", 'greyd_forms' ) ),
						el( greyd.components.OptionsControl, {
							className: 'tagselect',
							value: formula,
							options: fields,
							onChange: function ( value ) {
								updateFormat( value );
							},
						} ),
					),
					el( wp.components.Tip, {}, [
						el( 'small', {}, __( "Here you can select other fields to display their values dynamically in the frontend.", 'greyd_forms' ) )
					] ),
				];
			};
			return [
				el( wp.blockEditor.RichTextToolbarButton, {
					name: 'bold',
					icon: el( 'svg', {
						'width': "24",
						'height': "24",
						'viewBox': "0 0 24 24",
						'xmlns': "http://www.w3.org/2000/svg"
					}, el( "path", {
						fillRule: "evenodd",
						clipRule: "evenodd",
						d: "M12 2.75C6.89137 2.75 2.75 6.89137 2.75 12C2.75 17.1086 6.89137 21.25 12 21.25C17.1086 21.25 21.25 17.1086 21.25 12C21.25 6.89137 17.1086 2.75 12 2.75ZM1.25 12C1.25 6.06294 6.06294 1.25 12 1.25C17.9371 1.25 22.75 6.06294 22.75 12C22.75 17.9371 17.9371 22.75 12 22.75C6.06294 22.75 1.25 17.9371 1.25 12ZM12 6.75C9.1005 6.75 6.75 9.1005 6.75 12C6.75 14.8995 9.1005 17.25 12 17.25C14.8995 17.25 17.25 14.8995 17.25 12C17.25 9.1005 14.8995 6.75 12 6.75ZM5.25 12C5.25 8.27208 8.27208 5.25 12 5.25C15.7279 5.25 18.75 8.27208 18.75 12C18.75 15.7279 15.7279 18.75 12 18.75C8.27208 18.75 5.25 15.7279 5.25 12ZM12 10.75C11.3096 10.75 10.75 11.3096 10.75 12C10.75 12.6904 11.3096 13.25 12 13.25C12.6904 13.25 13.25 12.6904 13.25 12C13.25 11.3096 12.6904 10.75 12 10.75ZM9.25 12C9.25 10.4812 10.4812 9.25 12 9.25C13.5188 9.25 14.75 10.4812 14.75 12C14.75 13.5188 13.5188 14.75 12 14.75C10.4812 14.75 9.25 13.5188 9.25 12Z"
					} ) ),
					title: __( "Dynamic form value", 'greyd_forms' ),
					onClick: function () {
						if ( props.isActive ) {
							props.onChange(
								wp.richText.removeFormat( props.value, 'greyd-forms/dynamic-form-tag' )
							);
						}
						else {
							props.onChange(
								wp.richText.applyFormat( props.value, {
									type: 'greyd-forms/dynamic-form-tag',
									attributes: {
										tag: formula,
									}
								} )
							);
						}
					},
					isActive: props.isActive,
				} ),
				props.isActive && el( wp.components.Popover, {
					className: "components-greyd-format__content",
					...(
						_.has( wp.richText, 'useAnchor' ) ? {
							/**
							 * wp.richText.useAnchor is not supported in Gutenberg
							 * Version prior to 14.0.
							 */
							anchor: wp.richText.useAnchor( {
								editableContentElement: props.contentRef.current,
								value: props.value,
								settings: {
									tagName: 'span',
									className: 'is-button',
									name: 'greyd/trigger'
								}
							} )
						} : {
							anchorRef: isEmpty( props.value ) ? null : wp.richText.useAnchorRef( {
								ref: props.contentRef,
								value: props.value,
								settings: {
									tagName: 'span',
									className: 'is-button',
									name: 'greyd/trigger'
								}
							} )
						}
					)
				}, PopoverContent() )
			];
		}
	} );	
} )();
