/**
 * Class to establish components for Greyd.Forms editor view
 */
greyd.forms.components = new function () {

	const {
		Component,
		createElement: el,
	} = wp.element;
	const { __ } = wp.i18n;

	const {
		sanitizeTitle,
		uniqid
	} = greyd.forms.helper;

	/**
	 * Render the input-field control for the 'name' property.
	 * 
	 * @property {string} value Current value.
	 * @property {string} label Optional label to be displayed.
	 * @property {string} help Optional help-text to be displayed.
	 * ...
	 */
	this.InputNameControl = class extends Component {

		constructor () {
			super();

			this.setConfig = this.setConfig.bind( this );
			this.initValue = this.initValue.bind( this );

			this.config = {
				label: __( "Field name", 'greyd_forms' ),
				help: __( "Unique value of the field that is passed with the form.", 'greyd_forms' ),
				title: __( "This field is required for the form field to function correctly.", 'greyd_forms' ),
				autofillWhenEmpty: true, // whether to set a value when the attribute is empty
				autofillPrefix: 'input',
			};
		}

		componentDidMount() {
			this.setConfig();
			this.initValue();
		}

		/**
		 * Set the class configuration.
		 */
		setConfig() {
			const { config, props } = this;

			// get any user config...
			const userConfig = pick( props, keys( config ) );

			// avoid errors...
			const finalConfig = clone( config );

			assign( finalConfig, userConfig );

			this.config = finalConfig;
		}

		/**
		 * Generate a default value when empty
		 */
		initValue() {
			if ( this.config.autofillWhenEmpty && _.isEmpty( this.props.value ) ) {
				this.props.onChange( uniqid( this.config.autofillPrefix + '_', 6 ) );
			}
		}

		render() {
			return el( wp.components.TextControl, {
				label: this.config.label,
				className: 'input_name_control',
				help: this.config.help,
				title: this.config.title,
				value: this.props.value,
				onChange: ( value ) => this.props.onChange( sanitizeTitle( value ) ),
			} );
		}
	};

	/**
	 * Render a Tooltip for Greyd.Forms Fields.
	 * 
	 * @property {string} content The Content of the Tooltip
	 * @property {string} visible override the state from outside of the component
	 */
	this.LabelTooltip = class extends Component {

		constructor () {
			super();

			this.handleClick = this.handleClick.bind( this );
			this.handleClickOutside = this.handleClickOutside.bind( this );
			this.wrapperRef = React.createRef();

			//..
			this.state = {
				visible: false
			};
		}

		componentDidMount() {
			document.addEventListener( 'mousedown', this.handleClickOutside );
		}

		componentWillUnmount() {
			document.removeEventListener( 'mousedown', this.handleClickOutside );
		}


		handleClickOutside( event ) {

			if ( this.wrapperRef && !this.wrapperRef.current.contains( event.target ) ) {
				this.setState( { visible: false } );
			}
		}


		handleClick() {
			const currentState = this.state.visible;
			this.setState( { visible: !currentState } );
		}

		render() {

			let { content, blockProps } = this.props;

			return el( 'span', {
				className: this.state.visible || blockProps.attributes.tooltip.visible ? "forms-tooltip on" : "forms-tooltip",
				ref: this.wrapperRef,
			}, [
				el( 'span', {
					className: "forms-tooltip-toggle",
					onClick: this.handleClick,
				} ),
				el( 'span', {
					className: "forms-tooltip-popup",
					dangerouslySetInnerHTML: { __html: content }
				} )
			] );
		}
	};
};