/**
 * Class to establish helper functions for Greyd.Forms editor view
 */
greyd.forms.deprecations = new function () {

	// return if we're not editing a form
	if ( greyd.data.post_type !== greyd.forms.data.config.plugin_post_type ) return false;

	// basics
	const {
		element,
		i18n,
	} = window.wp;
	const el = element.createElement;
	const { __, _x } = i18n;

	const {
		makeInputName,
		stripTags,
		getTooltip,
		getValidationPattern
	} = greyd.forms.helper;

	const getGreydClass = (
		typeof greyd.tools.getGreydClass !== 'undefined' ?
			greyd.tools.getGreydClass :
			( blockProps ) => {
				return greyd.tools.generateGreydClass();
			}
	);

	// placeholder to replace when rendering (keep same as in render.php)
	const renderPlaceholder = {
		agb: "<!--agb_text-->",
		id: "___idXXX",
		captcha: "XXXcaptcha_sitekeyXXX",
		tooltip: "{tooltip}",
		enter: "{enter}",
		select: "{select}",
		required: "{required}",
		options: "{options}",
	};

	/**
	 * Get the deprecated block rendering.
	 * @since 1.4.5
	 * 
	 * @param {string} blockName Name of the block.
	 * @returns {array} Array of deprecations
	 */
	this.getDeprecations = function ( blockName ) {

		const deprecations = {
			'greyd-forms/input': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },
			
						// input
						type: { type: "string", default: 'text' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },
						autocomplete: { type: "string", default: 'on' },
						restrictions: {
							type: 'object', properties: {
								minlength: { type: "number" },
								maxlength: { type: "number" },
								max: { type: "string" },
								min: { type: "string" },
								start: { type: "string" },
								step: { type: "number" },
							}, default: {}
						},
						area: { type: "boolean", default: false },
						rows: { type: "number", default: 5 },
			
						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styling
						greydStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},
					save: function ( props ) {
			
						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( atts.type, atts ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );
			
						return el( element.Fragment, {}, [
			
							el( 'div', { 
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
			
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: makeInputName( inputName, renderPlaceholder.id ),
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
			
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
											}, tooltipText )
										] )
									] ),
			
									// Inner Input Wrapper
									el( 'div', {
										className: "input-inner-wrapper",
									}, [
										// Input Field
										el( ( atts.area ? 'textarea' : 'input' ), {
											className: "input validation " + ( atts.className ?? '' ),
											type: atts.type === 'url' ? 'text' : atts.type,
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											...( renderTootip ? {
												'aria-describedby': makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
											} : {} ),
											placeholder: atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter,
											autocomplete: atts.autocomplete,
											...( atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ?
												{
													...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
													...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
												} : {}
											),
											...( atts.type === "number" || atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ?
												{
													...atts.restrictions.min ? { min: atts.restrictions.min } : {},
													...atts.restrictions.max ? { max: atts.restrictions.max } : {},
													...atts.restrictions.step ? { step: atts.restrictions.step } : {},
													...atts.restrictions.start ? { value: atts.restrictions.start } : {},
												} : {}
											),
											...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "tel" ?
												{
													...{ pattern: getValidationPattern( atts.type ) },
												} : {}
											),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.required,
											rows: atts.area ? atts.rows : "",
			
										} ),
										el( 'span', {
											className: "input-field-icon"
										} )
									] ),
								] ),
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .input": atts.customStyles,
											" .input::placeholder": { color: 'inherit', opacity: 0.5 }
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				/**
				 * @since 1.4.5 made tooltips accessible
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },

						// input
						type: { type: "string", default: 'text' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },
						autocomplete: { type: "string", default: 'on' },
						restrictions: {
							type: 'object', properties: {
								minlength: { type: "number" },
								maxlength: { type: "number" },
								max: { type: "string" },
								min: { type: "string" },
								start: { type: "string" },
								step: { type: "number" },
							}, default: {}
						},
						area: { type: "boolean", default: false },
						rows: { type: "number", default: 5 },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styling
						greydStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},

					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( atts.type, atts ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: makeInputName( inputName, renderPlaceholder.id ),
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Inner Input Wrapper
									el( 'div', {
										className: "input-inner-wrapper",
									}, [
										// Input Field
										el( ( atts.area ? 'textarea' : 'input' ), {
											className: "input validation " + ( atts.className ?? '' ),
											type: atts.type === 'url' ? 'text' : atts.type,
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											placeholder: atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter,
											autocomplete: atts.autocomplete,
											...( atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ?
												{
													...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
													...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
												} : {}
											),
											...( atts.type === "number" || atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ?
												{
													...atts.restrictions.min ? { min: atts.restrictions.min } : {},
													...atts.restrictions.max ? { max: atts.restrictions.max } : {},
													...atts.restrictions.step ? { step: atts.restrictions.step } : {},
													...atts.restrictions.start ? { value: atts.restrictions.start } : {},
												} : {}
											),
											...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "tel" ?
												{
													...{ pattern: getValidationPattern( atts.type ) },
												} : {}
											),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.required,
											rows: atts.area ? atts.rows : "",

										} ),
										el( 'span', {
											className: "input-field-icon"
										} )
									] ),
								] ),
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .input": atts.customStyles,
											" .input::placeholder": { color: 'inherit', opacity: 0.5 }
										} : {}
									}
								} ),
							] ),
						] );
					}
				},
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },

						// input
						type: { type: "string", default: 'text' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },
						autocomplete: { type: "string", default: 'on' },
						restrictions: {
							type: 'object', properties: {
								minlength: { type: "number" },
								maxlength: { type: "number" },
								max: { type: "string" },
								min: { type: "string" },
								start: { type: "string" },
								step: { type: "number" },
							}, default: {}
						},
						area: { type: "boolean", default: false },
						rows: { type: "number", default: 5 },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styling
						greydStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},

					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( atts.type, atts ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: makeInputName( inputName, renderPlaceholder.id ),
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Inner Input Wrapper
									el( 'div', {
										className: "input-inner-wrapper",
									}, [
										// Input Field
										el( ( atts.area ? 'textarea' : 'input' ), {
											className: "input validation " + ( atts.className ?? '' ),
											type: atts.type === 'url' ? 'text' : atts.type,
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											placeholder: atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' ),
											autocomplete: atts.autocomplete,
											...( atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ?
												{
													...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
													...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
												} : {}
											),
											...( atts.type === "number" || atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ?
												{
													...atts.restrictions.min ? { min: atts.restrictions.min } : {},
													...atts.restrictions.max ? { max: atts.restrictions.max } : {},
													...atts.restrictions.step ? { step: atts.restrictions.step } : {},
													...atts.restrictions.start ? { value: atts.restrictions.start } : {},
												} : {}
											),
											...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "tel" ?
												{
													...{ pattern: getValidationPattern( atts.type ) },
												} : {}
											),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: __( "Please fill out this field", 'greyd_forms' ),
											rows: atts.area ? atts.rows : "",

										} ),
										el( 'span', {
											className: "input-field-icon"
										} )
									] ),
								] ),
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .input": atts.customStyles,
											" .input::placeholder": { color: 'inherit', opacity: 0.5 }
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				{
					attributes: {
						// main
						name: { type: "string", default: "" },

						// input
						type: { type: "string", default: 'text' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },
						autocomplete: { type: "string", default: 'on' },
						restrictions: {
							type: 'object', properties: {
								minlength: { type: "number" },
								maxlength: { type: "number" },
								max: { type: "string" },
								min: { type: "string" },
								start: { type: "string" },
								step: { type: "number" },
							}, default: {}
						},
						area: { type: "boolean", default: false },
						rows: { type: "number", default: 5 },

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object", default: {} },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styling
						greydStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( atts.type, atts ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: makeInputName( inputName, renderPlaceholder.id ),
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Inner Input Wrapper
									el( 'div', {
										className: "input-inner-wrapper",
									}, [
										// Input Field
										el( ( atts.area ? 'textarea' : 'input' ), {
											className: "input validation " + ( atts.className ?? '' ),
											type: atts.type,
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											placeholder: atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' ),
											autocomplete: atts.autocomplete,
											...( atts.type === "text" || atts.type === "email" || atts.type === "tel" || atts.type === "url" ?
												{
													...atts.restrictions.minlength ? { min: atts.restrictions.minlength } : {},
													...atts.restrictions.maxlength ? { max: atts.restrictions.maxlength } : {},
												} : {}
											),
											...( atts.type === "number" || atts.type === "time" || atts.type === "date" || atts.type === "datetime-local" || atts.type === "week" || atts.type === "month" ?
												{
													...atts.restrictions.min ? { min: atts.restrictions.min } : {},
													...atts.restrictions.max ? { max: atts.restrictions.max } : {},
													...atts.restrictions.step ? { step: atts.restrictions.step } : {},
													...atts.restrictions.start ? { value: atts.restrictions.start } : {},
												} : {}
											),
											...( atts.type === "email" || atts.type === "url" || atts.type === "date" || atts.type === "phone" ?
												{
													...{ pattern: getValidationPattern( atts.type ) },
												} : {}
											),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: __( "Please fill out this field", 'greyd_forms' ),
											rows: atts.area ? atts.rows : "",

										} ),
										el( 'span', {
											className: "input-field-icon"
										} )
									] ),
								] ),
								/**
								 * RenderSavedStylesDeprecated
								 * @deprecated since 1.1.2
								 */
								el( greyd.components.RenderSavedStylesDeprecated, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .input": atts.customStyles,
											" .input::placeholder": { color: 'inherit', opacity: 0.5 }
										} : {}
									}
								} ),
							] ),
						] );
					}
				}
			],
			'greyd-forms/checkbox': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },
						id: { type: "string", default: "" },
			
						// input
						content: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						useSetting: { type: 'boolean', default: false },
						required: { type: 'boolean', default: false },
						prechecked: { type: 'boolean', default: false },
						// spaceBetween: { type: 'string', default: '10px' },
			
						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						iconStyles: { type: "object" },
					},
					save: function ( props ) {
			
						const atts = props.attributes;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );
			
						return el( 'div', { className: [ "input-wrapper", atts.greydClass, atts.className ].join( ' ' ) }, [
							el( 'label', {
								className: "label checkbox-label",
								for: makeInputName( inputName, renderPlaceholder.id ),
							}, [
								el( "input", {
									className: "checkbox_placeholder_value",
									type: "hidden",
									name: inputName,
									value: "false",
								} ),
								el( "input", {
									type: 'checkbox',
									id: makeInputName( inputName, renderPlaceholder.id ),
									name: inputName,
									checked: atts.prechecked,
									value: "true",
									required: !!atts.required,
									'data-required': !!atts.required ? "required" : "optional",
									title: __( "please confirm the checkbox", 'greyd_forms' ),
								} ),
			
								el( "div", {}, [
									atts.useSetting ? el( 'span', {
										dangerouslySetInnerHTML: {
											__html: renderPlaceholder.agb
										}
									} ) : el( wp.blockEditor.RichText.Content, {
										tagName: 'span',
										value: atts.content
									} ),
									// required/optional
									el( 'span', {
										className: !!atts.required ? "requirement-required" : "requirement-optional",
									}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									"": atts.greydStyles,
									" input[type=checkbox] + span": atts.iconStyles
								},
								important: true
							} )
						] );
					},
				},
				{
					attributes: {
						// main
						name: { type: "string", default: "" },
						id: { type: "string", default: "" },

						// input
						content: { type: "string", default: '' },
						useSetting: { type: 'boolean', default: false },
						required: { type: 'boolean', default: false },
						prechecked: { type: 'boolean', default: false },
						// spaceBetween: { type: 'string', default: '10px' },

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						iconStyles: { type: "object" },
					},
					save: function ( props ) {

						const atts = props.attributes;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );

						return el( 'div', { className: [ "input-wrapper", atts.greydClass, atts.className ].join( ' ' ) }, [
							el( 'label', {
								className: "label checkbox-label",
								for: makeInputName( inputName, renderPlaceholder.id ),
							}, [
								el( "input", {
									className: "checkbox_placeholder_value",
									type: "hidden",
									name: inputName,
									value: "false",
								} ),
								el( "input", {
									type: 'checkbox',
									id: makeInputName( inputName, renderPlaceholder.id ),
									name: inputName,
									checked: atts.prechecked,
									value: "true",
									required: !!atts.required,
									'data-required': !!atts.required ? "required" : "optional",
									title: __( "please confirm the checkbox", 'greyd_forms' ),
								} ),

								el( "div", {}, [
									atts.useSetting ? el( 'span', {
										dangerouslySetInnerHTML: {
											__html: renderPlaceholder.agb
										}
									} ) : el( wp.blockEditor.RichText.Content, {
										tagName: 'span',
										value: atts.content
									} ),
									// required/optional
									el( 'span', {
										className: !!atts.required ? "requirement-required" : "requirement-optional",
									}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
								] ),
							] ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									"": atts.greydStyles,
									" input[type=checkbox] + span": atts.iconStyles
								},
								important: true
							} )
						] );
					}
				}
			],
			'greyd-forms/dropdown': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
			
						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },
			
						// options
						options: {
							type: 'array',
							default: []
							/**
							 * @since 1.7.4 Removed default.
							 * this default leads - for some unknown reasons - to validation erros
							 * in the block editor. Somehow the editor sees this as an object and
							 * invalidates the attribute.
							 */
							// default: [
							// 	{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
							// 	{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							// ]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview
			
						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},
					save: function ( props ) {
			
						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
						const inputName = _.isEmpty( atts.name ) ? makeInputName( atts.type, renderPlaceholder.id ) : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];
			
						const coreSpacing = wp.styleEngine.compileCSS( atts.style );
			
						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};
			
						return el( element.Fragment, {}, [
			
							el( 'div', { 
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ),
								style: coreSpacing 
							}, [
			
								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
			
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: 'tt-' + inputName
											}, tooltipText )
										] )
									] ),
			
									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											...( renderTootip ? {
												'aria-describedby': 'tt-' + inputName,
											} : {} ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.options,
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : renderPlaceholder.select ),
											...options.map( option => {
												const title = option.title && option.title.length ? option.title : ( option.value ?? '' );
												const value = option.value && option.value.length ? option.value : sanitizeTitle( option.title );
			
												return _.isEmpty( value ) ? null : el( 'option', {
													value: makeInputName( value )
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),
			
								// Styles
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },

						// options
						options: {
							type: 'array',
							/**
							 * @since 1.7.4 Removed default.
							 * this default leads - for some unknown reasons - to validation erros
							 * in the block editor. Somehow the editor sees this as an object and
							 * invalidates the attribute.
							 */
							// default: [
							// 	{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
							// 	{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							// ]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
						const inputName = _.isEmpty( atts.name ) ? makeInputName( atts.type, renderPlaceholder.id ) : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];

						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: 'tt-' + inputName
											}, tooltipText )
										] )
									] ),

									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											...( renderTootip ? {
												'aria-describedby': 'tt-' + inputName,
											} : {} ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.options,
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : renderPlaceholder.select ),
											...options.map( option => {
												const title = option.title ?? ( option.value ?? '' );
												const value = option.value ?? option.title;

												return _.isEmpty( value ) ? null : el( 'option', {
													value: makeInputName( value )
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),

								// Styles
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				/**
				 * @since 1.4.5 made tooltips accessible
				 */
				{
					attributes: {
						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },

						// options
						options: {
							type: 'array', default: [
								{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
								{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];

						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.options,
											tabindex: -1,
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : renderPlaceholder.select ),
											...options.map( option => {
												const title = option.title ?? ( option.value ?? '' );
												const value = option.value ?? option.title;

												return _.isEmpty( value ) ? null : el( 'option', {
													value: makeInputName( value )
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),

								// Styles
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				{
					attributes: {
						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },

						// options
						options: {
							type: 'array', default: [
								{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
								{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];

						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: __( "Please select an option", 'greyd_forms' ),
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : __( "please select", 'greyd_forms' ) ),
											...options.map( option => {
												const title = option.title ?? ( option.value ?? '' );
												const value = option.value ?? option.title;

												return _.isEmpty( value ) ? null : el( 'option', {
													value: makeInputName( value )
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),

								// Styles
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					},
				},
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },

						// options
						options: {
							type: 'array', default: [
								{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
								{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];

						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: __( "Please select an option", 'greyd_forms' ),
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : __( "please select", 'greyd_forms' ) ),
											...options.map( option => {
												const title = option.title ?? ( option.value ?? '' );
												const value = option.value ?? option.title;

												return _.isEmpty( value ) ? null : el( 'option', {
													value: value
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),
								/**
								 * RenderSavedStylesDeprecated
								 * @deprecated since 1.1.2
								 */
								el( greyd.components.RenderSavedStylesDeprecated, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					}
				},
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						placeholder: { type: "string", default: "" },
						required: { type: "boolean", default: false },

						// options
						options: {
							type: 'array', default: [
								{ title: __( 'Option 1', 'greyd_forms' ), value: __( 'option_1', 'greyd_forms' ) },
								{ title: __( 'Option 2', 'greyd_forms' ), value: __( 'option_2', 'greyd_forms' ) },
							]
						},
						inputtype: { type: "string", default: "" },
						multiselect: { type: "boolean", default: false },
						open: { type: "boolean", default: false }, // only used for backend preview

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						custom: { type: "boolean", default: false },
						customStyles: { type: "object" },
					},

					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, renderPlaceholder.id );
						const options = atts.options && atts.options.length ? atts.options : [];

						// we handle multiselects different than normal custom-selects
						const temp = !atts.multiselect ? {
							innerWrapper: 'div',
							innerWrapperAtts: { className: 'custom-select ' + ( atts.className ?? '' ) },
							selectAtts: {},
						} : {
							innerWrapper: element.Fragment,
							innerWrapperAtts: {},
							selectAtts: { multiple: true, className: ( atts.className ?? '' ) },
						};

						return el( element.Fragment, {}, [

							el( 'div', { className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ) }, [

								// Input Wrapper
								el( 'div', {
									className: "input-wrapper " + atts.greydClass
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
											for: inputName,
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Select
									el( temp.innerWrapper, temp.innerWrapperAtts, [
										el( 'select', {
											name: inputName,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
											title: renderPlaceholder.options,
											...temp.selectAtts
										}, [
											el( 'option', { value: '' }, atts.placeholder.length ? atts.placeholder : renderPlaceholder.select ),
											...options.map( option => {
												const title = option.title ?? ( option.value ?? '' );
												const value = option.value ?? option.title;

												return _.isEmpty( value ) ? null : el( 'option', {
													value: makeInputName( value )
												}, stripTags( title ) );
											} )
										] ),
									] )
								] ),

								// Styles
								el( greyd.components.RenderSavedStyles, {
									selector: atts.greydClass,
									styles: {
										" .label": atts.labelStyles,
										...atts.custom ? {
											" .select-selected, .input, .dropdown, .select-items": atts.customStyles,
											" .select-items": { boxShadow: 'none' },
										} : {}
									}
								} ),
							] ),
						] );
					},
				}
			],
			'greyd-forms/radiobuttons': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
			
						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
			
						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},

					save: function ( props ) {
			
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
			
						const coreSpacing = wp.styleEngine.compileCSS( atts.style );
			
						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ),
								style: coreSpacing
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id )
											}, tooltipText )
										] )
									] ) : null,
			
									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											...( renderTootip ? {
												'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
											} : {} ),
											value: '',
											title: renderPlaceholder.options,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: renderPlaceholder.options,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.multiselect ? atts.greydClass + " .greyd_multiradio" : atts.greydClass,
								styles: {
									" .option": atts.greydStyles,
									" input[type=radio] + span, .option .icn": atts.iconStyles
								}
							} ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					},
				},
				/**
				 * @since 1.7.0 inline-style for fieldset
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
			
						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
			
						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id )
											}, tooltipText )
										] )
									] ) : null,
			
									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											...( renderTootip ? {
												'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
											} : {} ),
											value: '',
											title: renderPlaceholder.options,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: renderPlaceholder.options,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.multiselect ? atts.greydClass + " .greyd_multiradio" : atts.greydClass + " .option",
								styles: {
									" .option": atts.greydStyles,
									" input[type=radio] + span, .option .icn": atts.iconStyles
								}
							} ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					},
				},
				/**
				 * @since 1.4.5 made tooltips accessible
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},

					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ) : null,

									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											value: '',
											title: renderPlaceholder.options,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: renderPlaceholder.options,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.multiselect ? atts.greydClass + " .greyd_multiradio" : atts.greydClass + " .option",
								styles: {
									" .option": atts.greydStyles,
									" input[type=radio] + span, .option .icn": atts.iconStyles
								}
							} ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					},
				},
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},

					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ) : null,

									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											value: '',
											title: __( "Please select an option", 'greyd_forms' ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: __( "Please select an option", 'greyd_forms' ),
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass + " .option",
								styles: {
									"": atts.greydStyles,
									" input[type=radio] + span, .icn": atts.iconStyles
								}
							} ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					},
				},
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'span', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ) : null,

									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											value: '',
											title: __( "Please select an option", 'greyd_forms' ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: __( "Please select an option", 'greyd_forms' ),
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass + " .option",
								styles: {
									"": atts.greydStyles,
									" input[type=radio] + span, .icn": atts.iconStyles
								}
							} ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					},
				},
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'span', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ) : null,

									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											value: '',
											title: __( "Please select an option", 'greyd_forms' ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: __( "Please select an option", 'greyd_forms' ),
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							/**
							 * RenderSavedStylesDeprecated
							 * @deprecated since 1.1.2
							 */
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass + " .option",
								styles: {
									"": atts.greydStyles,
									" input[type=radio] + span, .icn": atts.iconStyles
								}
							} ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					}
				},
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object", default: {} },
						iconStyles: { type: "object" },
					},
					providesContext: {
						'greyd-forms/radiobutton-name': 'name',
						'greyd-forms/radiobutton-multiselect': 'multiselect',
						'greyd-forms/radiobutton-required': 'required',
					},

					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "radio_buttons input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( ' ' ),
								}, [
									// Label 
									props.isSelected || atts.label.length > 0 ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'span', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ) : null,

									// inner blocks
									atts.multiselect ? el( 'div', {
										className: 'greyd_multiradio',
									}, [
										el( 'input', {
											type: 'text',
											name: makeInputName( atts.name, renderPlaceholder.id ),
											value: '',
											title: __( "Please select an option", 'greyd_forms' ),
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] ) : el( 'fieldset', {
										title: __( "Please select an option", 'greyd_forms' ),
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
										el( InnerBlocks.Content, { allowedBlocks: [ 'greyd-forms/radiobutton-item' ] } )
									] )
								] ),
							] ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass + " .option",
								styles: {
									"": atts.greydStyles,
									" input[type=radio] + span, .icn": atts.iconStyles
								}
							} ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
								}
							} )
						] );
					}
				}
			],
			'greyd-forms/radiobutton-item': [
				/**
				 * @since 1.7.4
				 */
				{
					supports: {
						className: false,
						customClassName: false,
					},
					attributes: {
						title: {
							type: 'string',
							// source: 'html',
							// selector: 'span',
							default: ''
						},
						value: { type: 'string', default: '' },
			
						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
					},
					usesContext: [ 'greyd-forms/radiobutton-name', 'greyd-forms/radiobutton-multiselect', 'greyd-forms/radiobutton-required' ],
					save: function ( props ) {

						const atts = props.attributes;
						const value = atts.value.length > 0 ? atts.value : atts.title;
						const id = makeInputName( greyd.forms.helper.sanitizeTitle( value ), renderPlaceholder.id );
						const name = makeInputName( atts.name, renderPlaceholder.id );
			
						// multiselect
						if ( atts.multiselect ) {
							return el( 'div', {
								className: 'option',
								onclick: 'window.forms.onMultiRadioClick(this);',
								title: renderPlaceholder.options,
								'data-value': value,
							}, [
								el( "span", {
									className: 'icn'
								} ),
								el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ),
							] );
						}
						// normal
						else {
							return el( 'div', {
								className: 'option'
							}, [
								el( 'label', {
									for: id,
									title: renderPlaceholder.options,
								}, [
									el( "input", {
										type: 'radio',
										id: id,
										value: value,
										name: name,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),
									el( wp.blockEditor.RichText.Content, {
										tagName: 'span',
										value: atts.title
									} ),
								] )
							] );
						}
					}
				},
				/**
				 * @since 1.3.9
				 */
				{
					attributes: {
						title: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						value: { type: 'string', default: '' },

						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
					},
					supports: {
						className: false,
						customClassName: false,
					},
					usesContext: [ 'greyd-forms/radiobutton-name', 'greyd-forms/radiobutton-multiselect', 'greyd-forms/radiobutton-required' ],
					save: function ( props ) {

						const atts = props.attributes;
						const value = makeInputName( atts.value.length > 0 ? atts.value : atts.title );
						const name = makeInputName( atts.name, renderPlaceholder.id );

						// multiselect
						if ( atts.multiselect ) {
							return el( 'div', {
								className: 'option',
								onclick: 'window.forms.onMultiRadioClick(this);',
								title: __( "Please select an option", 'greyd_forms' ),
								'data-value': value,
							}, [
								el( "span", {
									className: 'icn'
								} ),
								el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ),
							] );
						}
						// normal
						else {
							return el( 'div', {
								className: 'option'
							}, [
								el( 'label', {
									for: value,
									title: __( "Please select an option", 'greyd_forms' ),
								}, [
									el( "input", {
										type: 'radio',
										id: value,
										value: value,
										name: name,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),
									el( wp.blockEditor.RichText.Content, {
										tagName: 'span',
										value: atts.title
									} ),
								] )
							] );
						}
					},
				},
				{
					attributes: {
						title: { type: 'string', default: '' },
						value: { type: 'string', default: '' },

						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
					},
					supports: {
						className: false,
						customClassName: false,
					},
					usesContext: [ 'greyd-forms/radiobutton-name', 'greyd-forms/radiobutton-multiselect', 'greyd-forms/radiobutton-required' ],
					save: function ( props ) {

						const atts = props.attributes;
						const value = ( atts.value.length > 0 ? atts.value : atts.title );
						const name = makeInputName( atts.name, renderPlaceholder.id );

						// multiselect
						if ( atts.multiselect ) {
							return el( 'div', {
								className: 'option',
								onclick: 'window.forms.onMultiRadioClick(this);',
								title: __( "Please select an option", 'greyd_forms' ),
								'data-value': value,
							}, [
								el( "span", {
									className: 'icn'
								} ),
								el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ),
							] );
						}
						// normal
						else {
							return el( 'div', {
								className: 'option'
							}, [
								el( 'label', {
									for: value,
									title: __( "Please select an option", 'greyd_forms' ),
								}, [
									el( "input", {
										type: 'radio',
										id: value,
										value: value,
										name: name,
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),
									el( wp.blockEditor.RichText.Content, {
										tagName: 'span',
										value: atts.title
									} ),
								] )
							] );
						}
					}
				}
			],
			'greyd-forms/iconpanels': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
			
						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
						greydClass: { type: "string", default: "" },
			
						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// wrapper styles
						wrapperStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								alignItems: { type: "string" },
							}, default: {
								flexDirection: "",
								alignItems: "",
							}
						},
			
						// panel styles
						panelClass: { type: "string", default: "" },
						panelStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								width: { type: "string" },
							}, default: {
								flexDirection: "",
								// width: "auto",
							}
						},
			
						// image styles
						imgStyles: {
							type: "object", properties: {
								height: { type: "string" },
								margin: { type: "object" },
							}, default: {
								height: "3em",
								margin: { bottom: "0.5em" },
							}
						},
					},
					save: function ( props ) {
			
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
			
						const coreSpacing = wp.styleEngine.compileCSS( atts.style );
			
						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" ),
								style: coreSpacing
							}, [
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( " " )
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
			
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
											}, tooltipText )
										] )
									] ),
			
									// Fieldset
									el( atts.multiselect ? element.Fragment : 'fieldset', {
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [
			
										// Panels
										el( 'div', {
											className: "greyd_icon_panels img_pnl_wrapper" + ( atts.multiselect ? ' greyd_multiradio' : '' ),
										}, [
											// Control (multiselect)
											atts.multiselect ? el( 'input', {
												type: 'text',
												name: makeInputName( atts.name, renderPlaceholder.id ),
												...( renderTootip ? {
													'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
												} : {} ),
												value: '',
												required: !!atts.required,
												'data-required': !!atts.required ? "required" : "optional",
											} ) : null,
			
											// InnerBlocks
											el( InnerBlocks.Content, {
												// allowedBlocks: [ 'greyd-forms/iconpanel' ], 
											} )
										] )
									] )
			
								] ),
							] ),
			
							// wrapper styles
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl_wrapper": atts.wrapperStyles,
									" .label": atts.labelStyles,
								},
								important: true
							} ),
							// panel styles
							el( RenderSavedStyles, {
								selector: atts.greydClass + ' .img_pnl',
								styles: {
									"": { ...atts.panelStyles, active: {} },
									" img": atts.imgStyles,
								},
								important: true
							} ),
							_.has( atts.panelStyles, 'active' ) ? el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl.selected, input:checked + .img_pnl, input:checked + span + .img_pnl": atts.panelStyles.active,
								},
								important: true
							} ) : null,
						] );
					},
				},
				/**
				 * @since 1.4.5 made tooltips accessible
				 */
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
						greydClass: { type: "string", default: "" },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// wrapper styles
						wrapperStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								alignItems: { type: "string" },
							}, default: {
								flexDirection: "",
								alignItems: "",
							}
						},

						// panel styles
						panelClass: { type: "string", default: "" },
						panelStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								width: { type: "string" },
							}, default: {
								flexDirection: "",
								// width: "auto",
							}
						},

						// image styles
						imgStyles: {
							type: "object", properties: {
								height: { type: "string" },
								margin: { type: "object" },
							}, default: {
								height: "3em",
								margin: { bottom: "0.5em" },
							}
						},
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? ( getTooltip( 'multiselect' ).length ? renderPlaceholder.tooltip : '' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( " " )
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Fieldset
									el( atts.multiselect ? element.Fragment : 'fieldset', {
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [

										// Panels
										el( 'div', {
											className: "greyd_icon_panels img_pnl_wrapper" + ( atts.multiselect ? ' greyd_multiradio' : '' ),
										}, [
											// Control (multiselect)
											atts.multiselect ? el( 'input', {
												type: 'text',
												name: makeInputName( atts.name, renderPlaceholder.id ),
												value: '',
												required: !!atts.required,
												'data-required': !!atts.required ? "required" : "optional",
											} ) : null,

											// InnerBlocks
											el( InnerBlocks.Content, {
												// allowedBlocks: [ 'greyd-forms/iconpanel' ], 
											} )
										] )
									] )

								] ),
							] ),

							// wrapper styles
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl_wrapper": atts.wrapperStyles,
									" .label": atts.labelStyles,
								},
								important: true
							} ),
							// panel styles
							el( RenderSavedStyles, {
								selector: atts.greydClass + ' .img_pnl',
								styles: {
									"": { ...atts.panelStyles, active: {} },
									" img": atts.imgStyles,
								},
								important: true
							} ),
							_.has( atts.panelStyles, 'active' ) ? el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl.selected, input:checked + .img_pnl, input:checked + span + .img_pnl": atts.panelStyles.active,
								},
								important: true
							} ) : null,
						] );
					}
				},
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
						greydClass: { type: "string", default: "" },

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// wrapper styles
						wrapperStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								alignItems: { type: "string" },
							}, default: {
								flexDirection: "",
								alignItems: "",
							}
						},

						// panel styles
						panelClass: { type: "string", default: "" },
						panelStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								width: { type: "string" },
							}, default: {
								flexDirection: "",
								// width: "auto",
							}
						},

						// image styles
						imgStyles: {
							type: "object", properties: {
								height: { type: "string" },
								margin: { type: "object" },
							}, default: {
								height: "3em",
								margin: { bottom: "0.5em" },
							}
						},
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( " " )
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Fieldset
									el( atts.multiselect ? element.Fragment : 'fieldset', {
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [

										// Panels
										el( 'div', {
											className: "greyd_icon_panels img_pnl_wrapper" + ( atts.multiselect ? ' greyd_multiradio' : '' ),
										}, [
											// Control (multiselect)
											atts.multiselect ? el( 'input', {
												type: 'text',
												name: makeInputName( atts.name, renderPlaceholder.id ),
												value: '',
												required: !!atts.required,
												'data-required': !!atts.required ? "required" : "optional",
											} ) : null,

											// InnerBlocks
											el( InnerBlocks.Content, {
												// allowedBlocks: [ 'greyd-forms/iconpanel' ], 
											} )
										] )
									] )

								] ),
							] ),

							// wrapper styles
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl_wrapper": atts.wrapperStyles,
									" .label": atts.labelStyles,
								},
								important: true
							} ),
							// panel styles
							el( RenderSavedStyles, {
								selector: atts.greydClass + ' .img_pnl',
								styles: {
									"": { ...atts.panelStyles, active: {} },
									" img": atts.imgStyles,
								},
								important: true
							} ),
							_.has( atts.panelStyles, 'active' ) ? el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl.selected, input:checked + .img_pnl, input:checked + span + .img_pnl": atts.panelStyles.active,
								},
								important: true
							} ) : null,
						] );
					}
				},
				{
					attributes: {

						// main
						name: { type: "string", default: "" },
						required: { type: "boolean", default: false },
						multiselect: { type: "boolean", default: false },
						greydClass: { type: "string", default: "" },

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object" },

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }

							}, default: {
								content: '',
								visible: false
							}
						},

						// wrapper styles
						wrapperStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								alignItems: { type: "string" },
							}, default: {
								flexDirection: "",
								alignItems: "",
							}
						},

						// panel styles
						panelClass: { type: "string", default: "" },
						panelStyles: {
							type: "object", properties: {
								flexDirection: { type: "string" },
								width: { type: "string" },
							}, default: {
								flexDirection: "",
								// width: "auto",
							}
						},

						// image styles
						imgStyles: {
							type: "object", properties: {
								height: { type: "string" },
								margin: { type: "object" },
							}, default: {
								height: "3em",
								margin: { bottom: "0.5em" },
							}
						},
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( atts.multiselect ? getTooltip( 'multiselect' ) : '' ) : atts.tooltip.content;

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper" + ( atts.align && atts.align.length > 0 ? " flex-" + atts.align : "" )
							}, [
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass, ( atts.className ?? '' ) ].join( " " )
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Fieldset
									el( atts.multiselect ? element.Fragment : 'fieldset', {
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									}, [

										// Panels
										el( 'div', {
											className: "greyd_icon_panels img_pnl_wrapper" + ( atts.multiselect ? ' greyd_multiradio' : '' ),
										}, [
											// Control (multiselect)
											atts.multiselect ? el( 'input', {
												type: 'text',
												name: makeInputName( atts.name, renderPlaceholder.id ),
												value: '',
												required: !!atts.required,
												'data-required': !!atts.required ? "required" : "optional",
											} ) : null,

											// InnerBlocks
											el( InnerBlocks.Content, {
												// allowedBlocks: [ 'greyd-forms/iconpanel' ], 
											} )
										] )
									] )

								] ),
							] ),

							// wrapper styles
							/**
							 * RenderSavedStylesDeprecated
							 * @deprecated since 1.1.2
							 */
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl_wrapper": atts.wrapperStyles,
									" .label": atts.labelStyles,
								},
								important: true
							} ),
							// panel styles
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass + ' .img_pnl',
								styles: {
									"": { ...atts.panelStyles, active: {} },
									" img": atts.imgStyles,
								},
								important: true
							} ),
							_.has( atts.panelStyles, 'active' ) ? el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									" .img_pnl.selected, input:checked + .img_pnl, input:checked + span + .img_pnl": atts.panelStyles.active,
								},
								important: true
							} ) : null,
						] );
					}
				}
			],
			'greyd-forms/iconpanel': [
				{
					attributes: {

						title: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						value: { type: 'string', default: '' },
			
						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
						panelClass: { type: 'string', default: '' },
			
						// image
						images: {
							type: 'object', properties: {
								normal: { type: "object" },
								hover: { type: "object" },
								active: { type: "object" },
								current: { type: 'string' }
							}, default: {
								normal: { id: -1, url: "" },
								hover: { id: -1, url: "" },
								active: { id: -1, url: "" },
								current: 'normal'
							}
						},
					},
			
					usesContext: [ 'greyd-forms/iconpanel-name', 'greyd-forms/iconpanel-multiselect', 'greyd-forms/iconpanel-required', 'greyd-forms/iconpanel-panelClass' ],
			
					save: function ( props ) {
			
						const atts = props.attributes;
						const value = makeInputName( atts.value.length > 0 ? atts.value : atts.title );
						const id    = makeInputName( atts.value.length > 0 ? atts.value : atts.title, renderPlaceholder.id );
						const name  = makeInputName( atts.name, renderPlaceholder.id );
						const blockProps = wp.blockEditor.useBlockProps.save();
						
						// build the images
						let image_class = 'icon_select icon_hover';
						let images = [];
						if ( atts.images.active && atts.images.active.id > 0 ) {
							images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast' } ) );
							image_class = 'icon_hover'; // remove select from class
						}
						if ( atts.images.hover && atts.images.hover.id > 0 ) {
							images.push( el( 'img', { src: atts.images.hover.url, className: image_class + ' animate_fast' } ) );
							image_class = ''; // remove hover & select from class
						}
						if ( atts.images.normal && atts.images.normal.id > 0 ) {
							images.push( el( 'img', { src: atts.images.normal.url, className: image_class + ' icon_default animate_fast' } ) );
							images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost' } ) );
						}
			
						return el( element.Fragment, {}, [
			
							// input
							atts.multiselect ? null : el( 'input', {
								type: 'radio',
								id: id,
								value: value,
								name: name,
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ),
			
							// clickable label/div
							el( atts.multiselect ? 'div' : 'label', {
								className: 'greyd_icon_panel img_pnl animate_fast option ' + ( _.has(blockProps, 'className') ? blockProps.className : '' ),
								...atts.multiselect ? {
									'data-value': value,
									'onclick': 'window.forms.onMultiRadioClick(this);',
								} : {
									for: id
								}
							}, [
								// images
								el( 'div', { className: 'img_wrap' }, images ),
								// title
								atts.title.length ? el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ) : null,
							] )
						] );
					}
				},
				/**
				 * @since 1.6.8 className was not rendered
				 */
				{
					attributes: {

						title: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						value: { type: 'string', default: '' },
			
						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
						panelClass: { type: 'string', default: '' },
			
						// image
						images: { type: 'object', properties: {
							normal: { type: "object" },
							hover: { type: "object" },
							active: { type: "object" },
							current: { type: 'string' }
						}, default: {
							normal: { id: -1, url: "" },
							hover: { id: -1, url: "" },
							active:{ id: -1, url: "" },
							current: 'normal'
						} },
					},
					save: function( props ) {

						const atts  = props.attributes;
						const value = makeInputName( atts.value.length > 0 ? atts.value : atts.title );
						const id    = makeInputName( atts.value.length > 0 ? atts.value : atts.title, renderPlaceholder.id );
						const name  = makeInputName( atts.name, renderPlaceholder.id );
						
						// build the images
						let image_class = 'icon_select icon_hover';
						let images = [];
						if ( atts.images.active && atts.images.active.id > 0 ) {
							images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast'} ) );
							image_class = 'icon_hover'; // remove select from class
						}
						if ( atts.images.hover && atts.images.hover.id > 0 ) {
							images.push( el( 'img', { src: atts.images.hover.url, className: image_class+' animate_fast'} ) );
							image_class = ''; // remove hover & select from class
						}
						if ( atts.images.normal && atts.images.normal.id > 0 ) {
							images.push( el( 'img', { src: atts.images.normal.url, className: image_class+' icon_default animate_fast'} ) );
							images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost'} ) );
						}
						
						return el( element.Fragment, {}, [
			
							// input
							atts.multiselect ? null : el( 'input', {
								type: 'radio',
								id: id,
								value: value,
								name: name,
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ),
			
							// clickable label/div
							el( atts.multiselect ? 'div' : 'label', {
								className: 'greyd_icon_panel img_pnl animate_fast option',
								...atts.multiselect ? {
									'data-value': value,
									'onclick': 'window.forms.onMultiRadioClick(this);',
								} : {
									for: id
								}
							}, [
								// images
								el( 'div', { className: 'img_wrap' }, images ),
								// title
								atts.title.length ? el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ) : null,
							] )
						] );
					}
				},
				/**
				 * @since 1.4.5
				 */
				{
					migrate: function ( attributes ) {
						console.log( attributes );
						if ( attributes.title.match( /<span class="img_wrap">(.*?)<\/span>/g ) ) {
							attributes.title = attributes.title.replace( /<span class="img_wrap">(.*?)<\/span>/g, '' );
						}
						if ( attributes.title.match( /<span>(.*?)<\/span>/g ) ) {
							attributes.title = attributes.title.replace( /<span>(.*?)<\/span>/g, '$1' );
						}
						if ( attributes.title.match( /<div>(.*?)<\/div>/g ) ) {
							attributes.title = attributes.title.replace( /<div>(.*?)<\/div>/g, '$1' );
						}
						console.log( attributes );
						return attributes;
					},
					attributes: {
						title: {
							type: 'string',
							source: 'html',
							selector: 'div',
							default: ''
						},
						value: { type: 'string', default: '' },
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
						panelClass: { type: 'string', default: '' },

						images: {
							type: 'object', properties: {
								normal: { type: "object" },
								hover: { type: "object" },
								active: { type: "object" },
								current: { type: 'string' }
							}, default: {
								normal: { id: -1, url: "" },
								hover: { id: -1, url: "" },
								active: { id: -1, url: "" },
								current: 'normal'
							}
						},
					},
					save: function ( props ) {

						const atts = props.attributes;

						let title = atts.title;
						if ( title.match( /<span class="img_wrap">(.*?)<\/span>/g ) ) {
							title = title.replace( /<span class="img_wrap">(.*?)<\/span>/g, '' );
						}
						if ( title.match( /<span>(.*?)<\/span>/g ) ) {
							title = title.replace( /<span>(.*?)<\/span>/g, '$1' );
						}
						if ( title.match( /<div>(.*?)<\/div>/g ) ) {
							title = title.replace( /<div>(.*?)<\/div>/g, '$1' );
						}

						const value = makeInputName( atts.value.length > 0 ? atts.value : title );
						const id = makeInputName( atts.value.length > 0 ? atts.value : title, renderPlaceholder.id );
						const name = makeInputName( atts.name, renderPlaceholder.id );

						// build the images
						let image_class = 'icon_select icon_hover';
						let images = [];
						if ( atts.images.active && atts.images.active.id > 0 ) {
							images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast' } ) );
							image_class = 'icon_hover'; // remove select from class
						}
						if ( atts.images.hover && atts.images.hover.id > 0 ) {
							images.push( el( 'img', { src: atts.images.hover.url, className: image_class + ' animate_fast' } ) );
							image_class = ''; // remove hover & select from class
						}
						if ( atts.images.normal && atts.images.normal.id > 0 ) {
							images.push( el( 'img', { src: atts.images.normal.url, className: image_class + ' icon_default animate_fast' } ) );
							images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost' } ) );
						}

						return el( element.Fragment, {}, [

							// input
							atts.multiselect ? null : el( 'input', {
								type: 'radio',
								id: id,
								value: value,
								name: name,
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ),

							// clickable label/div
							el( atts.multiselect ? 'div' : 'label', {
								className: 'greyd_icon_panel img_pnl animate_fast option',
								...atts.multiselect ? {
									'data-value': value,
									'onclick': 'window.forms.onMultiRadioClick(this);',
								} : {
									for: id
								}
							}, [
								// images
								el( 'span', { className: 'img_wrap' }, images ),
								// title
								title.length ? el( wp.blockEditor.RichText.Content, {
									tagName: 'div',
									value: title
								} ) : null,
							] )
						] );
					},
				},
				{
					attributes: {
						title: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						value: { type: 'string', default: '' },

						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
						panelClass: { type: 'string', default: '' },

						// image
						images: {
							type: 'object', properties: {
								normal: { type: "object" },
								hover: { type: "object" },
								active: { type: "object" },
								current: { type: 'string' }
							}, default: {
								normal: { id: -1, url: "" },
								hover: { id: -1, url: "" },
								active: { id: -1, url: "" },
								current: 'normal'
							}
						},
					},
					save: function ( props ) {

						const atts = props.attributes;
						const value = makeInputName( atts.value.length > 0 ? atts.value : atts.title );
						const id = makeInputName( atts.value.length > 0 ? atts.value : atts.title, renderPlaceholder.id );
						const name = makeInputName( atts.name, renderPlaceholder.id );

						// build the images
						let image_class = 'icon_select icon_hover';
						let images = [];
						if ( atts.images.active && atts.images.active.id > 0 ) {
							images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast' } ) );
							image_class = 'icon_hover'; // remove select from class
						}
						if ( atts.images.hover && atts.images.hover.id > 0 ) {
							images.push( el( 'img', { src: atts.images.hover.url, className: image_class + ' animate_fast' } ) );
							image_class = ''; // remove hover & select from class
						}
						if ( atts.images.normal && atts.images.normal.id > 0 ) {
							images.push( el( 'img', { src: atts.images.normal.url, className: image_class + ' icon_default animate_fast' } ) );
							images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost' } ) );
						}

						return el( element.Fragment, {}, [

							// input
							atts.multiselect ? null : el( 'input', {
								type: 'radio',
								id: id,
								value: value,
								name: name,
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ),

							// clickable label/div
							el( atts.multiselect ? 'div' : 'label', {
								className: 'greyd_icon_panel img_pnl animate_fast option',
								...atts.multiselect ? {
									'data-value': value,
									'onclick': 'window.forms.onMultiRadioClick(this);',
								} : {
									for: id
								}
							}, [
								// images
								el( 'span', { className: 'img_wrap' }, images ),
								// title
								atts.title.length ? el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ) : null,
							] )
						] );
					}
				},
				{
					attributes: {

						title: { type: 'string', default: '' },
						value: { type: 'string', default: '' },

						// parent
						name: { type: 'string', default: '' },
						multiselect: { type: "boolean", default: false },
						required: { type: "boolean", default: false },
						panelClass: { type: 'string', default: '' },

						// image
						images: {
							type: 'object', properties: {
								normal: { type: "object" },
								hover: { type: "object" },
								active: { type: "object" },
								current: { type: 'string' }
							}, default: {
								normal: { id: -1, url: "" },
								hover: { id: -1, url: "" },
								active: { id: -1, url: "" },
								current: 'normal'
							}
						},
					},
					usesContext: [ 'greyd-forms/iconpanel-name', 'greyd-forms/iconpanel-multiselect', 'greyd-forms/iconpanel-required', 'greyd-forms/iconpanel-panelClass' ],
					save: function ( props ) {

						const atts = props.attributes;
						const value = ( atts.value.length > 0 ? atts.value : atts.title );
						const id = makeInputName( atts.value.length > 0 ? atts.value : atts.title, renderPlaceholder.id );
						const name = makeInputName( atts.name, renderPlaceholder.id );

						// build the images
						let image_class = 'icon_select icon_hover';
						let images = [];
						if ( atts.images.active && atts.images.active.id > 0 ) {
							images.push( el( 'img', { src: atts.images.active.url, className: 'icon_select animate_fast' } ) );
							image_class = 'icon_hover'; // remove select from class
						}
						if ( atts.images.hover && atts.images.hover.id > 0 ) {
							images.push( el( 'img', { src: atts.images.hover.url, className: image_class + ' animate_fast' } ) );
							image_class = ''; // remove hover & select from class
						}
						if ( atts.images.normal && atts.images.normal.id > 0 ) {
							images.push( el( 'img', { src: atts.images.normal.url, className: image_class + ' icon_default animate_fast' } ) );
							images.push( el( 'img', { src: atts.images.normal.url, className: 'icon_ghost' } ) );
						}

						return el( element.Fragment, {}, [

							// input
							atts.multiselect ? null : el( 'input', {
								type: 'radio',
								id: id,
								value: value,
								name: name,
								required: !!atts.required,
								'data-required': !!atts.required ? "required" : "optional",
							} ),

							// clickable label/div
							el( atts.multiselect ? 'div' : 'label', {
								className: 'greyd_icon_panel img_pnl animate_fast option',
								...atts.multiselect ? {
									'data-value': value,
									'onclick': 'window.forms.onMultiRadioClick(this);',
								} : {
									for: id
								}
							}, [
								// images
								el( 'span', { className: 'img_wrap' }, images ),
								// title
								atts.title.length ? el( wp.blockEditor.RichText.Content, {
									tagName: 'span',
									value: atts.title
								} ) : null,
							] )
						] );
					}
				}
			],
			'greyd-forms/upload': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: '' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },
			
						// file
						file: {
							type: 'object', properties: {
								type: { type: "string" },
								custom: { type: "string" },
								max: { type: "number" },
							}, default: {
								type: "",
								custom: "",
								max: 2
							}
						},
			
						// label
						label: {
							type: 'string',
							// source: 'html',
							// selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },
			
						// icon
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: 'icon_upload',
								position: 'before',
								size: '100%',
								margin: '10px'
							}
						},
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},

					save: function ( props ) {
			
						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( 'file', atts.file ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
						const renderTootip = tooltipText.trim().length > 0 ? true : null;
						const blockProps = wp.blockEditor.useBlockProps.save();
			
						const coreSpacing = wp.styleEngine.compileCSS( atts.style );
			
						return el( element.Fragment, {}, [
			
							el( "div", {
								className: "input-outer-wrapper" + ( atts.align ? " flex-" + atts.align : '' ),
								style: coreSpacing
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
			
										renderTootip && el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
											}, tooltipText )
										] )
									] ),
			
									// Input
									el( "input", {
										type: "file",
										className: "custom-file-upload",
										accept: atts.file.type !== 'other' ? atts.file.type : atts.file.custom,
										name: makeInputName( atts.name, false ),
										...( renderTootip ? {
											'aria-describedby': makeInputName( 'tt-' + atts.name, renderPlaceholder.id ),
										} : {} ),
										id: makeInputName( atts.name, renderPlaceholder.id ),
										"data-max": atts.file.size ? atts.file.size : "none",
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),
			
									el( 'label', {
										for: makeInputName( atts.name, renderPlaceholder.id ),
										"data-text": atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter
									}, [
										el( 'div', {
											className: "input " + ( atts.className ?? '' ),
										}, [
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'before'
											} ),
											el( 'span', {
												style: { flex: '1' },
												dangerouslySetInnerHTML: { __html: atts.placeholder.length ? atts.placeholder : renderPlaceholder.select }
											} ),
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'after'
											} ),
										] ),
									] ),
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
									...atts.custom ? {
										" .input": atts.customStyles,
										" .input::placeholder": { color: 'inherit', opacity: 0.5 }
									} : {}
								}
							} )
						] );
					}
				},
				/**
				 * @since 1.4.5 made tooltips accessible
				 */
				{
					attributes: {
						// main
						name: { type: "string", default: '' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },

						// file
						file: {
							type: 'object', properties: {
								type: { type: "string" },
								custom: { type: "string" },
								max: { type: "number" },
							}, default: {
								type: "",
								custom: "",
								max: 2
							}
						},

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },

						// icon
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: 'icon_upload',
								position: 'before',
								size: '100%',
								margin: '10px'
							}
						},

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? ( getTooltip( 'file', atts.file ).length ? renderPlaceholder.tooltip : '' ) : atts.tooltip.content;
						const blockProps = wp.blockEditor.useBlockProps.save();

						return el( element.Fragment, {}, [

							el( "div", {
								className: "input-outer-wrapper" + ( atts.align ? " flex-" + atts.align : '' ),
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Input
									el( "input", {
										type: "file",
										className: "custom-file-upload",
										accept: atts.file.type !== 'other' ? atts.file.type : atts.file.custom,
										name: makeInputName( atts.name, false ),
										id: makeInputName( atts.name, renderPlaceholder.id ),
										"data-max": atts.file.size ? atts.file.size : "none",
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),

									el( 'label', {
										for: makeInputName( atts.name, renderPlaceholder.id ),
										"data-text": atts.placeholder.length ? atts.placeholder : renderPlaceholder.enter
									}, [
										el( 'div', {
											className: "input " + ( atts.className ?? '' ),
										}, [
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'before'
											} ),
											el( 'span', {
												style: { flex: '1' },
												dangerouslySetInnerHTML: { __html: atts.placeholder.length ? atts.placeholder : renderPlaceholder.select }
											} ),
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'after'
											} ),
										] ),
									] ),
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
									...atts.custom ? {
										" .input": atts.customStyles,
										" .input::placeholder": { color: 'inherit', opacity: 0.5 }
									} : {}
								}
							} )
						] );
					}
				},
				{
					attributes: {
						// main
						name: { type: "string", default: '' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },

						// file
						file: {
							type: 'object', properties: {
								type: { type: "string" },
								custom: { type: "string" },
								max: { type: "number" },
							}, default: {
								type: "",
								custom: "",
								max: 2
							}
						},

						// label
						label: {
							type: 'string',
							source: 'html',
							selector: 'span',
							default: ''
						},
						labelStyles: { type: "object", default: {} },

						// icon
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: 'icon_upload',
								position: 'before',
								size: '100%',
								margin: '10px'
							}
						},

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( 'file', atts.file ) : atts.tooltip.content;
						const blockProps = wp.blockEditor.useBlockProps.save();

						return el( element.Fragment, {}, [

							el( "div", {
								className: "input-outer-wrapper" + ( atts.align ? " flex-" + atts.align : '' ),
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Input
									el( "input", {
										type: "file",
										className: "custom-file-upload",
										accept: atts.file.type !== 'other' ? atts.file.type : atts.file.custom,
										name: makeInputName( atts.name, false ),
										id: makeInputName( atts.name, renderPlaceholder.id ),
										"data-max": atts.file.size ? atts.file.size : "none",
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),

									el( 'label', {
										for: makeInputName( atts.name, renderPlaceholder.id ),
										"data-text": atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' )
									}, [
										el( 'div', {
											className: "input " + ( atts.className ?? '' ),
										}, [
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'before'
											} ),
											el( 'span', {
												style: { flex: '1' },
												dangerouslySetInnerHTML: { __html: atts.placeholder.length ? atts.placeholder : __( "please select", 'greyd_forms' ) }
											} ),
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'after'
											} ),
										] ),
									] ),
								] ),
							] ),
							el( RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
									...atts.custom ? {
										" .input": atts.customStyles,
										" .input::placeholder": { color: 'inherit', opacity: 0.5 }
									} : {}
								}
							} )
						] );
					}
				},
				{
					attributes: {
						// main
						name: { type: "string", default: '' },
						required: { type: "boolean", default: false },
						placeholder: { type: "string", default: '' },

						// file
						file: {
							type: 'object', properties: {
								type: { type: "string" },
								custom: { type: "string" },
								max: { type: "number" },
							}, default: {
								type: "",
								custom: "",
								max: 2
							}
						},

						// label
						label: { type: "string", default: "" },
						labelStyles: { type: "object", default: {} },

						// icon
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: 'icon_upload',
								position: 'before',
								size: '100%',
								margin: '10px'
							}
						},

						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
							}, default: {
								content: '',
								visible: false
							}
						},

						// styles
						greydClass: { type: "string", default: "" },
						greydStyles: { type: "object" },
						customStyles: { type: 'object' },
						custom: { type: 'boolean', default: false }
					},
					save: function ( props ) {

						const atts = props.attributes;
						const tooltipText = atts.tooltip.content.length === 0 ? getTooltip( 'file', atts.file ) : atts.tooltip.content;
						const blockProps = wp.blockEditor.useBlockProps.save();

						return el( element.Fragment, {}, [

							el( "div", {
								className: "input-outer-wrapper" + ( atts.align ? " flex-" + atts.align : '' ),
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									!atts.label.length ? null : el( 'div', {
										className: "label_wrap",
									}, [
										el( 'label', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.label
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),

										!tooltipText.length || !tooltipText.trim().length ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup"
											}, tooltipText )
										] )
									] ),

									// Input
									el( "input", {
										type: "file",
										className: "custom-file-upload",
										accept: atts.file.type !== 'other' ? atts.file.type : atts.file.custom,
										name: atts.name,
										id: makeInputName( atts.name, renderPlaceholder.id ),
										"data-max": atts.file.size ? atts.file.size : "none",
										required: !!atts.required,
										'data-required': !!atts.required ? "required" : "optional",
									} ),

									el( 'label', {
										for: makeInputName( atts.name, renderPlaceholder.id ),
										"data-text": atts.placeholder.length ? atts.placeholder : __( "enter here", 'greyd_forms' )
									}, [
										el( 'div', {
											className: "input " + ( atts.className ?? '' ),
										}, [
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'before'
											} ),
											el( 'span', {
												style: { flex: '1' },
												dangerouslySetInnerHTML: { __html: atts.placeholder.length ? atts.placeholder : __( "please select", 'greyd_forms' ) }
											} ),
											el( greyd.components.RenderButtonIcon, {
												value: atts.icon,
												position: 'after'
											} ),
										] ),
									] ),
								] ),
							] ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: atts.greydClass,
								styles: {
									" .label": atts.labelStyles,
									...atts.custom ? {
										" .input": atts.customStyles,
										" .input::placeholder": { color: 'inherit', opacity: 0.5 }
									} : {}
								}
							} )
						] );
					}
				}
			],
			'greyd-forms/recaptcha': [
				{
					attributes: {
						hideBanner: { type: "boolean", default: false },
					},
					save: function ( props ) {
						return el( 'div', {
							id: "captcha",
							className: ( props.attributes.className ?? '' ) + ( props.attributes.hideBanner ? ' hide' : '' )
						}, [
							el( 'input', {
								type: "hidden",
								name: "reCAPTCHA",
								className: "recaptcha_input",
								id: makeInputName( 'recaptcha', renderPlaceholder.id ),
								value: "not_verfied",
								required: true
							} ),
							el( 'script', {
								src: "https://www.google.com/recaptcha/api.js?render='" + renderPlaceholder.captcha + "'",
							} ),
							el( 'script', {}, "var recaptchaSitekey = '" + renderPlaceholder.captcha + "' ;" )
						] );
					}
				}
			],
			'greyd-forms/submitbutton': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						anchor: true,
						align: true,
						defaultStylePicker: false
					},
					attributes: {
						inline_css: { type: 'string' },
						inline_css_id: { type: 'string' },
						greydClass: { type: 'string', default: '' },
						greydStyles: { type: 'object' },
						customStyles: { type: 'object' },
						size: { type: 'string', default: '' },
						content: { type: 'string', default: '' },
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: '',
								position: 'after',
								size: '100%',
								margin: '10px'
							}
						},
						custom: { type: 'boolean', default: false }
					},
					save: function ( props ) {
			
						const extraClass = _.has( props.attributes, "className" ) && props.attributes.className.indexOf( 'is-style-link-' ) > -1 ? 'link' : 'button';
						const coreSpacing = wp.styleEngine.compileCSS( props.attributes.style );
			
						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper input-wrapper" + ( props.attributes.align ? ' flex-' + props.attributes.align : '' ),
								style: coreSpacing
							}, [
								el( "button", {
									id: props.attributes.anchor,
									className: [ 'submitbutton', extraClass, props.attributes.greydClass, ( props.attributes.className ?? '' ), props.attributes.size ].join( ' ' ),
									type: "submit",
									name: "submit",
									title: __( "submit", 'greyd_forms' ),
									"data-text": props.attributes.content/*.replace(/(<([^>]+)>)/gi, "")*/,
								}, [
									// loader & progress bar
									el( 'span', {
										className: "greyd_upload_progress"
									}, [
										el( 'span', { className: "spinner" }, [
											el( 'span', { className: "dot-loader" } ),
											el( 'span', { className: "dot-loader dot-loader--2" } ),
											el( 'span', { className: "dot-loader dot-loader--3" } ),
										] ),
										el( 'span', { className: "bar" } ),
									] ),
									// icon
									el( greyd.components.RenderButtonIcon, {
										value: props.attributes.icon,
										position: 'before'
									} ),
									// text
									el( 'span', {
										style: { flex: '1' },
										dangerouslySetInnerHTML: {
											__html: props.attributes.content.length ? props.attributes.content : __( "submit", 'greyd_forms' )
										}
									} ),
									// icon
									el( greyd.components.RenderButtonIcon, {
										value: props.attributes.icon,
										position: 'after'
									} )
								] ),
							] ),
							el( greyd.components.RenderSavedStyles, {
								selector: props.attributes.greydClass + '.submitbutton',
								styles: {
									'': {
										...props.attributes.greydStyles,
										...props.attributes.custom ? props.attributes.customStyles : {}
									},
								}
							} ),
						] );
					}
				},
				/**
				 * RenderSavedStylesDeprecated
				 * @deprecated since 1.1.2
				 */
				{
					attributes: {
						inline_css: { type: 'string' },
						inline_css_id: { type: 'string' },
						greydClass: { type: 'string', default: '' },
						greydStyles: { type: 'object' },
						customStyles: { type: 'object' },
						size: { type: 'string', default: '' },
						content: { type: 'string', default: '' },
						icon: {
							type: 'object', properties: {
								content: { type: "string" },
								position: { type: "string" },
								size: { type: "string" },
								margin: { type: "string" },
							}, default: {
								content: '',
								position: 'after',
								size: '100%',
								margin: '10px'
							}
						},
						custom: { type: 'boolean', default: false }
					},
					save: function ( props ) {

						const extraClass = _.has( props.attributes, "className" ) && props.attributes.className.indexOf( 'is-style-link-' ) > -1 ? 'link' : 'button';

						return el( element.Fragment, {}, [
							el( 'div', {
								className: "input-outer-wrapper input-wrapper" + ( props.attributes.align ? ' flex-' + props.attributes.align : '' )
							}, [
								el( "button", {
									id: props.attributes.anchor,
									className: [ 'submitbutton', extraClass, props.attributes.greydClass, ( props.attributes.className ?? '' ), props.attributes.size ].join( ' ' ),
									type: "submit",
									name: "submit",
									title: __( "submit", 'greyd_forms' ),
									"data-text": props.attributes.content/*.replace(/(<([^>]+)>)/gi, "")*/,
								}, [
									// loader & progress bar
									el( 'span', {
										className: "greyd_upload_progress"
									}, [
										el( 'span', { className: "spinner" }, [
											el( 'span', { className: "dot-loader" } ),
											el( 'span', { className: "dot-loader dot-loader--2" } ),
											el( 'span', { className: "dot-loader dot-loader--3" } ),
										] ),
										el( 'span', { className: "bar" } ),
									] ),
									// icon
									el( greyd.components.RenderButtonIcon, {
										value: props.attributes.icon,
										position: 'before'
									} ),
									// text
									el( 'span', {
										style: { flex: '1' },
										dangerouslySetInnerHTML: {
											__html: props.attributes.content.length ? props.attributes.content : __( "submit", 'greyd_forms' )
										}
									} ),
									// icon
									el( greyd.components.RenderButtonIcon, {
										value: props.attributes.icon,
										position: 'after'
									} )
								] ),
							] ),
							el( greyd.components.RenderSavedStylesDeprecated, {
								selector: props.attributes.greydClass + '.submitbutton',
								styles: {
									'': {
										...props.attributes.greydStyles,
										...props.attributes.custom ? props.attributes.customStyles : {}
									},
								}
							} ),
						] );
					}
				}
			],
			'greyd-forms/hidden-field': [
				/**
				 * Default value for dynamic hidden fields
				 * @since 1.6.3
				 */
				{
					attributes: {
						name: { type: "string", default: '' },
						value: { type: "string", default: '' },
						isDynamic: { type: "boolean", default: false },
						type: { type: "string", default: '' },
					},
					save: function ( props ) {
						return el( 'input', {
							type: "hidden",
							name: props.attributes.name,
							value: props.attributes.value,
						} );
					}
				}
			],
			'greyd-forms/multistep-container': [
				{
					supports: {
						align: true,
						customClassName: true
					},
					attributes: {
						transition: { type: "string", default: 'swipe' },
						autostep: { type: "boolean", default: true },
						activeStep: { type: "number", default: 0 },
						progressbar: {
							type: 'object', properties: {
								enable: { type: "boolean" },
								type: { type: "string" },
								align: { type: "string" }
							}, default: {
								enable: true,
								type: "progress",
								align: "center"
							}
						},
			
						// styles
						greydClass: { type: "string", default: "" },
						progressbarStyles: { type: "object", default: {} },
					},
					save: function ( props ) {

						const atts = props.attributes;
			
						const progressbarColors = {
							" .multistep_pagination > span": { backgroundColor: atts.progressbar.backgroundColor },
							" .multistep_progress": { backgroundColor: atts.progressbar.backgroundColor },
							" .multistep_pagination > span.active": { backgroundColor: atts.progressbar.activeColor },
							" .multistep_progress .bar": { backgroundColor: atts.progressbar.activeColor }
						};
			
						return el( element.Fragment, {}, [
							// steps
							el( "div", {
								className: [ "multistep_wrapper", atts.greydClass, atts.autostep ? "auto_step" : '' ].join( ' ' ),
								"data-transition": atts.transition,
								"data-step": 0,
							},
								el( InnerBlocks.Content )
							),
			
			
							// progress bar
							atts.progressbar.enable ? el( 'div', {
								className: [ "progress_wrapper", "flex-" + atts.progressbar.align, atts.greydClass ].join( ' ' ),
							},
								el( 'div', { className: 'multistep_' + atts.progressbar.type },
			
									atts.progressbar.type == "progress" ? el( 'span', { className: 'bar' } ) :
			
										props.innerBlocks.map( ( step, index ) => {
											return el( 'span', {
											} );
										} )
								)
							) : '',
			
							el( greyd.components.RenderSavedStyles, {
								selector: atts.greydClass + ".progress_wrapper",
								styles: {
									" .multistep_progress, .multistep_pagination > span": atts.progressbarStyles,
									...progressbarColors
									// " .bar": "background-color:"
								},
								important: true
							} )
						] );
					}
				}
			],
			'greyd-forms/range': [
				/**
				 * Updated core supports including margin & padding
				 * @since 1.14.0
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },
			
						required: { type: "boolean", default: false },
			
						min: { type: "float", default: 0 },
						max: { type: "float", default: 100 },
						start: { type: "float", default: 50 },
						step: { type: "number", default: 1 },
			
						// label
						labelBefore: { type: "string", default: "" },
						labelAfter: { type: "string", default: "" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styling
						greydStyles: { type: "object" },
						labelStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
					},
					save: function ( props ) {
			
						// atts
						const atts = props.attributes;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );
						const coreSpacing = wp.styleEngine.compileCSS( atts.style );
			
						return el( element.Fragment, {}, [
			
							el( 'div', {
								className: [ "input-outer-wrapper", atts.align && atts.align.length > 0 ? "flex-" + atts.align : "", props.className ].join( " " ),
								style: coreSpacing
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									( atts.labelBefore.length || atts.labelAfter.length ) ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'span', {
											className: "label",
										},
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.labelBefore + " "
											} ),
										),
										el( 'strong', { className: 'range_value' } ),
										el( 'span', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: " " + atts.labelAfter
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										atts.tooltip.content.length === 0 ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
											}, atts.tooltip.content )
										] )
			
									] ) : null,
			
									// Inner Input Wrapper
									el( 'div', {
										className: "range_control_wrapper",
									}, [
										el( 'input', {
											type: 'range',
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											min: atts.min,
											max: atts.max,
											step: atts.step,
											value: isNaN(atts.start) ? 50 : atts.start,
											required: !!atts.required,
											'data-required': !!atts.required ? "required" : "optional",
										} ),
										el( 'span', {
											className: 'rail',
										} ),
										el( 'span', {
											className: 'track',
										} ),
										el( 'span', {
											className: 'thumb_wrapper',
										}, [
											el( 'span', {
												className: 'thumb',
											} ),
											el( 'span', {
												className: 'tooltip',
											}, ),
										] )
									] )
								] )
							] ),
							el( greyd.components.RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									".input-wrapper": atts.greydStyles,
									" .label": atts.labelStyles,
								},
							} ),
						] );
					},
				},
				/**
				 * @since 1.7.0 required attribute was not set
				 */
				{
					supports: {
						align: true,
						customClassName: true,
						defaultStylePicker: false
					},
					attributes: {
						// main
						name: { type: "string", default: "" },
			
						required: { type: "boolean", default: false },
			
						min: { type: "number", default: 0 },
						max: { type: "number", default: 100 },
						start: { type: "number", default: 50 },
						step: { type: "number", default: 1 },
			
						// label
						labelBefore: { type: "string", default: "" },
						labelAfter: { type: "string", default: "" },
			
						// tooltip
						tooltip: {
							type: 'object', properties: {
								content: { type: 'string' },
								visible: { type: 'boolean' }
			
							}, default: {
								content: '',
								visible: false
							}
						},
			
						// styling
						greydStyles: { type: "object" },
						labelStyles: { type: "object" },
						greydClass: { type: "string", default: "" },
					},
					save: function ( props ) {

						// atts
						const atts = props.attributes;
						const inputName = _.isEmpty( atts.name ) ? atts.type : makeInputName( atts.name, false );
			
						return el( element.Fragment, {}, [
			
							el( 'div', {
								className: [ "input-outer-wrapper", atts.align && atts.align.length > 0 ? "flex-" + atts.align : "", props.className ].join( " " )
							}, [
								// Input Wrapper
								el( 'div', {
									className: [ "input-wrapper", atts.greydClass ].join( ' ' ),
								}, [
									// Label 
									( atts.labelBefore.length || atts.labelAfter.length ) ? el( 'div', {
										className: "label_wrap",
									}, [
										el( 'span', {
											className: "label",
										},
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: atts.labelBefore + " "
											} ),
										),
										el( 'strong', { className: 'range_value' } ),
										el( 'span', {
											className: "label",
										}, [
											el( wp.blockEditor.RichText.Content, {
												tagName: 'span',
												value: " " + atts.labelAfter
											} ),
											// required/optional
											el( 'span', {
												className: !!atts.required ? "requirement-required" : "requirement-optional",
											}, !!atts.required ? " *" : " " + __( "(optional)", 'greyd_forms' ) ),
										] ),
										// Tooltip
										atts.tooltip.content.length === 0 ? null : el( 'span', { className: "forms-tooltip", }, [
											el( 'span', {
												className: "forms-tooltip-toggle",
											} ),
											el( 'span', {
												className: "forms-tooltip-popup",
												id: makeInputName( 'tt-' + inputName, renderPlaceholder.id ),
											}, atts.tooltip.content )
										] )
			
									] ) : null,
			
									// Inner Input Wrapper
									el( 'div', {
										className: "range_control_wrapper",
									}, [
										el( 'input', {
											type: 'range',
											name: inputName,
											id: makeInputName( inputName, renderPlaceholder.id ),
											min: atts.min,
											max: atts.max,
											step: atts.step,
											value: atts.start ? atts.start : {}
										} ),
										el( 'span', {
											className: 'rail',
										} ),
										el( 'span', {
											className: 'track',
										} ),
										el( 'span', {
											className: 'thumb_wrapper',
										}, [
											el( 'span', {
												className: 'thumb',
											} ),
											el( 'span', {
												className: 'tooltip',
											}, ),
										] )
									] )
								] )
							] ),
							el( greyd.components.RenderSavedStyles, {
								selector: atts.greydClass,
								styles: {
									".input-wrapper": atts.greydStyles,
									" .label": atts.labelStyles,
								},
							} ),
						] );
					}
				}
			]
		};

		return deprecations[ blockName ] ?? [];
	};
};