/*
	Comments: Javascript File for Conditional Fields (CF)
*/

document.addEventListener( 'DOMContentLoaded', function () {
	// jQuery(function() {
	CF.init();
} );

var CF = new function () {

	/**
	 * Hold all dependant fields
	 */
	this.listeners = [];

	/**
	 * ID prefix
	 */
	this.id_pre = '___id';

	/**
	 * Field types
	 */
	this.types = {
		text: [ 'text', 'email', 'tel', 'url', 'number', 'time', 'date', 'datetime-local', 'week', 'month' ],
	};

	/**
	 * Init the class
	 */
	this.init = function () {
		if ( typeof $ === 'undefined' ) $ = jQuery;

		// add listeners
		this.addListeners();
		this.__deprecatedAddListeners();

		// add events
		if ( this.listeners.length ) {
			$.each( this.listeners, function () {

				var form = $( this ).closest( 'form' );
				var name = $( this ).attr( 'name' );

				// on change
				$( this ).on( "change input", function () {
					CF.inputChanged( name, form );
					CF.__deprecatedInputChanged( name, form );
				} );
				// on load
				CF.inputChanged( name, form );
				CF.__deprecatedInputChanged( name, form );
			} );
		}

		// on form reset
		$( "form.greyd_form" ).on( "reset", function ( e ) {
			CF.onFormReset( $( this ) );
			CF.__deprecatedClearConditions( $( this ) );
		} );
	};

	/**
	 * First we add all dependant fields to CF.listeners
	 */
	this.addListeners = function () {

		$( 'form.greyd_form' ).each( function () {
			var form = $( this );
			form.find( '.condition_block' ).each( function () {

				var conditionData = CF.getConditionData( $( this ) );
				if ( conditionData && conditionData.conditions ) {

					conditionData.conditions.forEach( function ( condition ) {
						CF.pushToListeners( condition.name, form );
					} );
				}
			} );
		} );
	};

	/**
	 * Push a field to the CF.listeners array.
	 * @param {string} name Name of the field
	 * @param {object} form jQuery object of the wrapping form
	 */
	this.pushToListeners = function ( name, form ) {
		// if (greydFormSettings.wp_debug) console.log(name);
		var input = forms.getFieldByName( name, form );
		if ( input.length > 0 && !CF.arrayContainsInput( input, CF.listeners ) ) {
			CF.listeners.push( input );
			// if (greydFormSettings.wp_debug) console.log(input);
		}
	};

	/**
	 * Trigger Change Event
	 */
	this.inputChanged = function ( name, form ) {
		// if (greydFormSettings.wp_debug) console.log( 'Input „'+name+'“ changed' );

		name = forms.getCleanName( name );

		form.find( '.condition_block[data-condition*="%' + name + '%"]' ).each( function () {
			CF.checkConditionInputs( $( this ), form );
		} );
	};

	/**
	 * Check all depending fields of a container for matches
	 * @param {object} container 
	 * @param {object} form 
	 * @returns 
	 */
	this.checkConditionInputs = function ( container, form ) {

		var conditionData = CF.getConditionData( container );
		if ( !conditionData ) return;

		var isMatching = conditionData.operator === 'and' ? true : false;

		conditionData.conditions.forEach( function ( condition ) {
			var currentValue = forms.getVal( forms.getFieldByName( condition.name, form ) );

			var conditionIsMatching = CF.matchCondition(
				conditionData.operator,
				currentValue,
				condition.operator,
				condition.value
			);
			if ( greydFormSettings.wp_debug ) {
				// console.log( conditionIsMatching ? '--> YES it does' : '--> NO it doesn’t' );
			}

			if ( conditionData.operator === 'and' ) {
				if ( conditionIsMatching ) {
					// we just continue checking the other conditions, because they all need to match...
				} else {
					isMatching = false;
					return false;
				}
			}
			else {
				if ( conditionIsMatching ) {
					isMatching = true;
					return false;
				} else {
					// we just continue checking the other conditions, until at least one is matching...
				}
			}
		} );

		this.showOrHideCondition( container, isMatching );
	};

	/**
	 * Show or hide a condition and animate the wrapper
	 * @param {object} container 
	 * @param {bool} show 
	 */
	this.showOrHideCondition = function ( container, show ) {

		if ( typeof show === 'undefined' ) show = true;
		var wrapper = container.parent( '.condition_wrapper' );
		var animate = wrapper.hasClass( 'slide' );

		// store height for animation
		if ( animate ) {
			var oldHeight = container.height();
			wrapper.css( 'height', oldHeight );
		}

		if ( show ) CF.showConditionField( container );
		else CF.hideConditionField( container );

		// animate container
		if ( animate ) {

			var newHeight = show ? container.height() : 0;

			let parentWrapper = wrapper.parents(".condition_wrapper");

			if ( parentWrapper.length && show ) {
				parentWrapper.css( {
					height: "auto"
				} );
			}

			var heightDiff = newHeight - oldHeight;
			if ( heightDiff !== 0 ) {

				// remove class 'open' during animation for clean overflows
				wrapper.removeClass( 'open' );

				// set the height (animated via css)
				wrapper.css( {
					height: newHeight
				} );

				// add class 'open' again, to have visible overflows
				setTimeout( function () {
					container.parent().addClass( 'open' );
				}, 400 );

				// trigger event 'heightChange', used for multistep containers
				wrapper.closest( 'form' ).trigger( 'heightChange', [ wrapper, heightDiff ] );
			}
		}
	};

	/**
	 * Display a single condiiton
	 * @param {object} field 
	 */
	this.showConditionField = function ( field ) {

		field.removeClass( 'hide' ).addClass( 'show' );

		setTimeout( function () {
			field.css( 'opacity', 1 );
		}, 0 );

		// set required attributes
		field.find( 'input, select, textarea' ).each( function () {
			$( this ).attr( 'novalue', false ); // remove no-send data
			if ( $( this ).data( 'required' ) === 'required' ) {
				$( this ).prop( 'required', true );
			}
			forms.changeDependingFormTags( this );
		} );
	};

	/**
	 * Hide a single condition
	 * @param {object} field 
	 */
	this.hideConditionField = function ( field ) {
		field.removeClass( 'show' ).addClass( 'hide' ).css( 'opacity', 0 );
		// remove required attributes
		field.find( 'input, select, textarea' ).each( function () {
			$( this ).attr( 'novalue', true ); // add no-send data
			if ( $( this ).prop( 'required' ) ) $( this ).prop( 'required', false );
		} );
	};

	/**
	 * Reset all containers when a form is reseted
	 * @param {object} form 
	 */
	this.onFormReset = function ( form ) {
		var containers = form.find( '.condition_wrapper:not([data-after="hold"]) .condition_block' );
		containers.each( function () {
			CF.checkConditionInputs( $( this ), $( this ).closest( 'form' ) );
		} );
	};

	/**
	 * =================================================================
	 *                          HELPERS
	 * =================================================================
	 */

	/**
	 * Check if a single condition matches
	 * 
	 * @param {string} operator 'or'|'and'
	 * @param {mixed} value Current value
	 * @param {string} type 'is'|'has'|'empty'|...
	 * @param {mixed} needed Value to compare to
	 * @returns bool Whether or not the condition is matching.
	 */
	this.matchCondition = function ( operator, value, type, needed ) {
		operator = operator && operator.length ? operator : 'or';
		value = typeof value === 'undefined' ? '' : value;
		type = type && type.length ? type : 'is';
		needed = typeof needed === 'undefined' ? '' : needed;

		if ( typeof value === "string" ) {
			value = value.replace( /\s|_/g, "" ).toLowerCase();
		}
		if ( typeof needed === "string" ) {
			needed = needed.replace( /\s|_/g, "" ).toLowerCase();
		}

		if ( greydFormSettings.wp_debug ) {
			// console.log( 'Check if value "'+value+'" ('+(typeof value)+') '+type+' "'+needed+'" ('+(typeof needed)+')' );
		}

		if ( operator === 'or' ) {
			switch ( type ) {
				case "is":
					if ( value == needed ) return true;
					break;
				case "has":
					if ( String( value ).indexOf( needed ) >= 0 ) return true;
					break;
				case "empty":
					if ( !String( value ).length || value === 'off' ) return true;
					break;
				case "not_empty":
					if ( String( value ).length && value !== 'off' ) return true;
					break;
				case "is_not":
					if ( value !== needed ) return true;
					break;
				case "greater":
					if ( parseFloat( value ) > parseFloat( needed ) ) return true;
					break;
				case "greater_equal":
					if ( parseFloat( value ) >= parseFloat( needed ) ) return true;
					break;
				case "smaller":
					if ( parseFloat( value ) < parseFloat( needed ) ) return true;
					break;
				case "smaller_equal":
					if ( parseFloat( value ) <= parseFloat( needed ) ) return true;
					break;
			}
			return false;
		}
		else if ( operator === 'and' ) {
			switch ( type ) {
				case "is":
					if ( value != needed ) return false;
					break;
				case "has":
					if ( String( value ).indexOf( needed ) === -1 ) return false;
					break;
				case "empty":
					if ( String( value ).length && value !== 'off' ) return false;
					break;
				case "not_empty":
					if ( !String( value ).length || value === 'off' ) return false;
					break;
				case "is_not":
					if ( value === needed ) return false;
					break;
				case "greater":
					if ( parseFloat( value ) <= parseFloat( needed ) ) return false;
					break;
				case "greater_equal":
					if ( parseFloat( value ) < parseFloat( needed ) ) return false;
					break;
				case "smaller":
					if ( parseFloat( value ) >= parseFloat( needed ) ) return false;
					break;
				case "smaller_equal":
					if ( parseFloat( value ) > parseFloat( needed ) ) return false;
					break;
			}
			return true;
		}
		return false;
	};

	/**
	 * Check if an array contains an input
	 * @param {object} obj jQuery object of the input
	 * @param {array} array the array to crawl
	 * @returns 
	 */
	this.arrayContainsInput = function ( obj, array ) {
		var i;
		for ( i = 0; i < array.length; i++ ) {
			if ( array[ i ][ 0 ] === obj[ 0 ] && array[ i ].context === obj.context ) {
				return true;
			}
		}
		return false;
	};

	/**
	 * Get the condition data as an object
	 * @param {object} container jQuery Object of the condition-container
	 * @returns object|bool
	 */
	this.getConditionData = function ( container ) {
		var conditionData = container.data( 'condition' );
		if ( !conditionData || conditionData.length === 0 ) return false;

		var conditions = [], operator = '';
		if ( conditionData.indexOf( '||' ) > -1 ) {
			operator = 'or';
			conditions = conditionData.split( '||' );
		}
		else if ( conditionData.indexOf( '&&' ) > -1 ) {
			operator = 'and';
			conditions = conditionData.split( '&&' );
		}
		else {
			conditions = [ conditionData ];
		}

		conditions.forEach( function ( condition ) {
			var fieldname = [ 0 ];
		} );

		return {
			operator: operator,
			conditions: conditions.map( function ( condition ) {
				var split = condition.replace( /(^%|%$)/g, '' ).split( '%%' );
				return {
					name: split[ 0 ],
					operator: split[ 1 ],
					value: split[ 2 ] ?? ''
				};
			} )
		};
	};

	/**
	 * Reset all conditional field listeners
	 * @param {DOMElement} form 
	 */
	this.reset = function ( form ) {
		this.listeners = [];
		this.init();
	};

	/**
	 * =================================================================
	 *                          DEPRECATED
	 *                          @since 1.0 Blocks Version
	 * =================================================================
	 */

	/**
	 * First we add all dependant fields to CF.listeners
	 */
	this.__deprecatedAddListeners = function () {
		$( "form.greyd_form" ).each( function () {
			if ( $( this ).find( 'div.conditional_fields' ).length !== 0 ) {
				// form has condition in it
				var form = $( this );
				$( this ).find( 'div.conditional_fields' ).each( function () {
					var fieldname = $( this ).data( 'field' ).replace( /\#/g, '' );
					if ( fieldname !== undefined ) {
						// if multiple
						if ( fieldname.indexOf( '&' ) !== -1 ) {
							fieldname = fieldname.split( '&' );
							$.each( fieldname, function ( i, str ) {
								CF.pushToListeners( this, form );
							} );
						} else {
							CF.pushToListeners( fieldname, form );
						}
					}
				} );
			}
		} );
	};

	/**
	 * Trigger Change Event
	 */
	this.__deprecatedInputChanged = function ( name, form ) {
		// // get correct name
		// if (name && name.indexOf(CF.id_pre) !== -1)
		//     name = name.substring(0, name.indexOf(CF.id_pre));

		var containers = form.find( '.conditional_fields' );
		if ( containers.length ) {
			containers.each( function () {
				CF.__deprecatedCheckFields( $( this ), form );
			} );
		}
	};

	/**
	 * Check all fields in container for matches
	 * @param {object} container 
	 * @param {object} form 
	 * @returns 
	 */
	this.__deprecatedCheckFields = function ( container, form ) {

		// attributes
		var fields = container.find( '.condition' );
		var names = container.data( 'field' ).replace( /\#/g, '' );
		var inputs = [];
		var matching_fields = [];

		// if multiple
		if ( names.indexOf( '&' ) !== -1 ) {
			names = names.split( '&' );
		} else {
			names = [ names ];
		}

		// find inputs by names
		$.each( names, function ( i, name ) {

			var value;
			var input = form.find( '*[name^="' + name + '"]' );

			if ( input.length > 0 ) {

				value = forms.getVal( input );

				// clean value
				if ( typeof value === 'undefined' ) {
					value = '';
				}
				if ( typeof value === "string" && value.indexOf( CF.id_pre ) !== -1 ) {
					value = value.substring( 0, value.indexOf( CF.id_pre ) );
				}

				if ( input.attr( 'novalue' ) === 'true' ) value = '';

				// push to array
				inputs.push( { name: name, value: value } );
			}
		} );

		if ( !inputs.length ) return false;

		// fallback for not defined names
		if ( typeof names[ 1 ] === 'undefined' ) names[ 1 ] = names[ 0 ];
		if ( typeof names[ 2 ] === 'undefined' ) names[ 2 ] = names[ 1 ];

		fields.each( function () {
			var field = $( this );
			var operator = '';
			var conditions = $( this ).data( 'value' );
			var match = false;

			// get all conditions as array
			if ( conditions.indexOf( '||' ) !== -1 ) {
				conditions = conditions.split( '||' );
				operator = 'or';
			} else if ( conditions.indexOf( '&&' ) !== -1 ) {
				conditions = conditions.split( '&&' );
				operator = 'and';
			}

			if ( operator !== '' ) // default-field has no value, therefore operator is not defined
				match = CF.__deprecatedMatchField( operator, conditions, inputs, names );

			if ( match && !CF.arrayContainsInput( field, matching_fields ) ) matching_fields.push( field );
		} );

		// Update all containers to new fields
		CF.__deprecatedUpdateFields( matching_fields, container );
	};

	/**
	 * Check whether a specific field matches with it's conditions
	 * returns true if matches
	 */
	this.__deprecatedMatchField = function ( operator, conditions, inputs, names ) {
		operator = operator.length === 0 ? 'or' : operator;
		// default match bool
		var matches = operator === 'and' ? true : false;

		$.each( conditions, function ( i, condition ) {
			if ( !this.length ) return false;

			// get single condition as array
			var con = condition.split( '%' );

			// get condition value by name
			var result = inputs.filter( function ( obj ) {
				return obj.name === names[ con[ 0 ] ];
			} );

			match = CF.matchCondition( operator, result[ 0 ].value, con[ 1 ], con[ 2 ] );
			// if (greydFormSettings.wp_debug) console.log('condition '+i+': ' + match);

			// return if matching is successfull
			if ( operator === 'or' && match === true ) matches = true;
			if ( operator === 'and' && match === false ) matches = false;
		} );
		return matches;
	};

	this.__deprecatedUpdateFields = function ( fields, container ) {
		var matchIDs = [];
		var animate = container.parent().hasClass( 'slide' );

		// store height for animation
		if ( animate ) {
			var oldHeight = container.height();
			container.parent().css( 'height', oldHeight );
		}

		// get all matched_fields ids
		if ( fields.length > 0 ) {
			$.each( fields, function ( i, field ) {
				var id = field.attr( 'id' );
				matchIDs.push( id );
			} );
		}

		// hide all shown fields that are not matched
		container.find( '.condition.show' ).each( function () {
			var id = $( this ).attr( 'id' );
			if ( !matchIDs.indexOf( id ) !== -1 )
				CF.hideConditionField( $( this ) );
		} );

		// show matching fields
		if ( fields.length > 0 ) {
			$.each( fields, function ( i, field ) {
				if ( !field.hasClass( 'show' ) )
					CF.showConditionField( field );
			} );
		}
		// or show default
		else {
			CF.hideConditionField( container.find( '.condition:not(.default)' ) ); // for init
			var field = container.find( '.condition.default' );
			CF.showConditionField( field );
		}

		// animate container
		if ( animate ) {

			var newHeight = container.height();
			var heightDiff = newHeight - oldHeight;
			if ( heightDiff !== 0 ) {

				// remove class 'open' during animation for clean overflows
				container.parent().removeClass( 'open' );

				// set the height (animated via css)
				container.parent().css( {
					height: newHeight
				} );

				// add class 'open' again, to have visible overflows
				setTimeout( function () {
					container.parent().addClass( 'open' );
				}, 400 );

				// trigger event 'heightChange', used for multistep containers
				container.closest( 'form' ).trigger( 'heightChange', [ container, heightDiff ] );
			}
		}
	};

	// Clear or Reset Inputs
	this.__deprecatedClearConditions = function ( form ) {
		var containers = form.find( '.conditional_fields:not([data-after="hold"])' );
		if ( containers.length ) {
			containers.each( function () {
				CF.__deprecatedUpdateFields( [], $( this ) );
			} );
		}
	};
};