/*
	Comments: Javascript File for Plugin Frontend
*/

document.addEventListener( 'DOMContentLoaded', function () {
	// jQuery(function () {
	forms.init();

	window.onload = function () {
		console.log( "onload" );
		window.addEventListener( "beforeunload", function ( e ) {
			if ( forms.formSubmitted ) {
				return undefined;
			}
		} );
	};

} );

var forms = new function () {

	this.ajax_url;
	this.formSubmitted = false;

	this.init = function () {
		if ( typeof $ === 'undefined' ) $ = jQuery;

		// logging
		if ( greydFormSettings.wp_debug ) console.log( 'Greyd.Forms default script: loaded' );

		this.addEvents();
	};


	/**
	 * =================================================================
	 *                          ADD EVENTS
	 * =================================================================
	 */
	this.addEvents = function () {

		// VALIDATIONS
		// On Focus In
		$( "form.greyd_form input, form.greyd_form textarea" ).on( "focusin focus", function () {
			$( this ).addClass( 'input-focus' ).removeClass( 'input-unfocus invalid' );
		} );
		// On Focus Out
		$( "form.greyd_form input, form.greyd_form textarea" ).on( "focusout", function () {

			$( this ).removeClass( 'input-focus' );

			// If needs to be validated
			if ( $( this ).hasClass( 'validation' ) ) {
				if ( $( this ).val() !== '' || $( this ).prop( 'required' ) ) {
					$( this ).addClass( 'input-unfocus' );
				}
			}
			// If empty & required
			else if ( $( this ).prop( 'required' ) ) {
				if ( $( this ).val() === '' ) {
					$( this ).addClass( 'input-unfocus invalid' );
				} else {
					$( this ).removeClass( 'input-unfocus invalid' );
				}
			}
		} );
		// On submit button click
		$( "form.greyd_form .submitbutton" ).on( "click", function () {
			$( this ).closest( "form.greyd_form" ).addClass( "validate" );
		} );
		// Remove colored state
		$( ".input-inner-wrapper .input-field-icon" ).on( "click", function () {
			$( this ).siblings().removeClass( 'input-unfocus invalid' );
		} );

		// TOOLTIPS
		this.initFormTooltips();

		// UPLOAD
		if ( $( 'input[type="file"].custom-file-upload' ).length > 0 ) {
			this.initUpload();
		}

		// RECAPTCHA
		this.maybeExecuteCaptcha();
		if ( $( 'form.greyd_form' ).find( '#captcha.hide' ).length !== 0 ) {
			$( 'body' ).addClass( 'hideCaptcha' );
		}

		// SUBMIT
		$( "form.greyd_form" ).on( "submit", function ( e ) {
			e.preventDefault();

			/**
			 * Check if we have to do a recaptcha validation first.
			 * Otherwise the handler will be called directly.
			 */
			forms.maybeExecuteCaptcha( forms.handleForm, [ e, $( this ) ] );
		} );

		// RESET
		$( "form.greyd_form" ).on( "reset", function ( e ) {
			forms.formReset( $( this ) );
		} );

		// ERROR
		$( "form.greyd_form" ).on( "error", function ( e, atts ) {
			var html = typeof atts.html !== 'undefined' ? atts.html : '';
			var log = typeof atts.log !== 'undefined' ? atts.log : '';
			forms.formError( $( this ), html, log );
		} );

		// RANGE
		if ( $( '.range_control_wrapper' ).length > 0 ) {
			this.initCustomRange();
		}

		// MATH FIELDS
		if ( $( '.math_field' ).length > 0 ) {
			this.initMathFields();
		}


		if ( document.querySelectorAll( 'input[type="password"]' ) && document.querySelectorAll( 'input[type="password"].confirm-password' ) ) {
			this.initConfirmPasswordField();
		}

		// DYNAMIC FORM TAGS
		if ( $( '.form_tag' ).length > 0 ) {
			this.initFormTags();
		}

		// Set ajax url
		forms.setAjaxUrl();

	};


	/**
	 * =================================================================
	 *                          HANDLER
	 * =================================================================
	 */

	/**
	 * Handle Form
	 */
	this.handleForm = function ( event, form ) {

		var block_send = form.data( 'send' );
		if ( block_send == "prevent" ) {
			event.preventDefault();
			return false;
		}

		var id = form.data( 'id' );
		var nonce = form.data( 'nonce' );
		var mode = form.data( 'form-mode' );
		var message_wrapper = form.siblings( ".after-message" );
		var message = message_wrapper.children( ".message" );
		var popup = form.siblings( ".after-popup" );
		var spinner = form.find( '.greyd_upload_progress .spinner' );
		var progress_bar = form.find( '.greyd_upload_progress .bar' );
		var button = form.find( ".submitbutton" );

		// hide message
		message_wrapper.css( { 'height': 0, "opacity": 0 }, 300 );
		message_wrapper.siblings( 'pre' ).remove();

		// change button
		button.prop( 'disabled', true );
		button.children( 'span:not(.greyd_upload_progress)' ).css( 'opacity', 0 );
		spinner.css( 'opacity', 1 );

		// Get form data array
		var formValues = form.serializeArray();
		var cleanFormValues = [];
		[ ...formValues ].forEach( function ( item ) {

			// get the value
			const inputVal = String( forms.getVal( form.find( '*[name="' + item.name + '"]' ) ) );

			// compare the value to the input value
			if ( inputVal != item.value ) {
				item.value = inputVal;
			}

			// clear uniqid
			item.name = forms.getCleanName( item.name );
			item.value = forms.getCleanName( item.value );

			// push to final array
			cleanFormValues.push( item );
		} );

		var encodedFormValues = $.param( Object.values( cleanFormValues ) );

		if ( greydFormSettings.wp_debug ) {
			console.log( 'original form values:', formValues, 'clean form values:', cleanFormValues, 'encoded form values:', encodedFormValues );
			// return false;
		}

		// save to FormData object
		var form_data = new FormData();
		form_data.append( 'action', 'handle_form_data' ); // important to make valid request
		form_data.append( 'data', encodedFormValues );
		form_data.append( 'post_id', id );
		form_data.append( 'nonce', nonce );
		form_data.append( 'form_mode', mode );

		// insert files
		var hasfile = false;
		form.find( 'input[type=file]' ).each( function () {
			hasfile = true;
			var inputName = forms.getCleanName( $( this ).attr( "name" ) );
			var fileToUpload = $( this )[ 0 ].files[ 0 ];
			//console.log(fileToUpload);
			form_data.append( inputName, fileToUpload );
		} );

		// send ajax request to handle.php
		forms.xhr;
		$.ajax( {
			type: 'POST',
			url: forms.ajax_url, // set via setAjaxUrl();
			data: form_data,
			processData: false,
			contentType: false,
			cache: false,

			xhr: function () {
				var xhr = new window.XMLHttpRequest();

				// Progress Bar
				xhr.upload.addEventListener( "progress", function ( event ) {
					if ( event.lengthComputable ) {
						//console.log('filesize: '+event.total);
						var percentComplete = event.loaded / event.total;
						percentComplete = parseInt( percentComplete * 100 );

						progress_bar.hide();
						if ( hasfile )
							progress_bar.show();
						progress_bar.width( percentComplete + '%' );

						// console.log(percentComplete);
						if ( percentComplete === 100 ) {
							// console.log("Upload successfull");
							setTimeout( function () {
								progress_bar.hide();
								progress_bar.width( '0px' );
							}, 400 );
						}
					}
				}, false );

				xhr.addEventListener( "load", function ( event ) {
					if ( greydFormSettings.wp_debug ) console.log( event );

					// log full response with all debugs to console
					if ( greydFormSettings.wp_debug ) console.log( event.target.response );

					// SERVER ERROR
					if ( event.target.status !== 200 ) {
						form.trigger( "error", { log: "Error in handle.php" } );

						// NO SERVER ERROR
					} else {

						if ( !event.target.response ) {
							form.trigger( "error", { log: "No response set" } );
							return false;
						}

						var response;
						try {
							response = JSON.parse( event.target.response.split( "greyd_form_response::" )[ 1 ] );
						} catch ( e ) {
							console.error( e );
							console.log( event.target.response );
							form.trigger( "error", { log: "Error parsing JSON response" } );
							return;
						}

						// ERROR
						if ( response.response == 'ERROR' ) {
							// get html
							var html = response.errors ? response.errors : "";
							html = Array.isArray( html ) ? ( html.length > 1 ? "<ul><li>" + html.join( "</li><li>" ) + "</li></ul>" : html.join( "" ) ) : html;
							// get log (for debug mode)
							var log = typeof response.errors === 'object' ? response.errors : response;
							log = response.errors && typeof response.errors.body !== "undefined" ? response.errors.body : log;
							// trigger error
							form.trigger( "error", { html: html, log: log } );

							// SUCCESS
						} else if ( response.response == 'SUCCESS' ) {

							forms.formSubmitted = true;

							/**
							 * Reset form message to not display previous errors.
							 * @since 1.6.0
							 */
							message.html( '' ).removeClass( 'danger' ).addClass( 'success' );

							// Action after
							if ( response.message == 'popup' || response.message == 'fullscreen' ) {
								popup.addClass( 'in' );
								if ( response.message == 'fullscreen' ) $( 'body' ).addClass( 'no-scroll' );
								popup.animate( { opacity: 1 }, 300 );

							} else if ( response.message.indexOf( 'page' ) !== -1 ) {
								var link = response.message.substr( response.message.indexOf( "=" ) + 1 );
								window.location.href = link;

							} else {
								message.html( response.message );
							}

							// reset form
							form.trigger( "reset", response );

							// ERROR
						} else {
							form.trigger( "error", { log: "Error in handle.php. Activate WP_DEBUG and see console for details." } );
						}
					}

				}, false );

				xhr.addEventListener( "error", function ( event ) {
					form.trigger( "error", { log: "Invalid Ajax Call to: " + greydFormSettings.ajaxurl } );
				}, false );

				return xhr;
			}
		} );
	};

	/**
	 * Default form response handling.
	 * This function is called after every form handling - whether it was successfull or not.
	 * 
	 * @param {object} form The form object.
	 */
	this.afterFormHandled = function ( form ) {
		var message_wrapper = form.siblings( ".after-message" );
		var message = message_wrapper.children( ".message" );
		var spinner = form.find( '.greyd_upload_progress .spinner' );
		var progress_bar = form.find( '.greyd_upload_progress .bar' );
		var button = form.find( ".submitbutton" );

		// toggle message
		if ( message.html() !== "" ) {
			var height = message.outerHeight() + parseInt( message.css( "margin-top" ) ) + parseInt( message.css( "margin-bottom" ) );
			message_wrapper.css( { 'height': height, "opacity": 1 }, 300 );
		} else {
			message_wrapper.css( { 'height': 0, "opacity": 0 }, 300 );
		}

		// reset recaptcha
		forms.clearCaptcha();

		forms.formSubmitted = true;

		// reset button
		setTimeout( function () {
			button.children( 'span:not(.greyd_upload_progress)' ).css( 'opacity', 1 );
			button.prop( 'disabled', false );
			spinner.css( 'opacity', 0 );
			progress_bar.hide();
			progress_bar.width( '0px' );
		}, 150 );
	};

	/**
	 * Whenever there is an error handling the form, this function is triggered.
	 * 
	 * @param {object} form The form object.
	 * @param {array|object|string} html Content to be appended to the message.
	 * @param {array|object|string} log Gets logged to the console or appended below the form, if debug mode is enabled.
	 */
	this.formError = function ( form, html, log ) {

		// objects
		var message_wrapper = form.siblings( ".after-message" );

		// log event
		// if ( typeof event.target !== 'undefined' ) {
		//     var statuscode = typeof event.target.status !== 'undefined' ? event.target.status : '';
		//     var statustext = typeof event.target.statusText !== 'undefined' ? event.target.statusText : '';
		//     if (statuscode.length > 0 && statustext.length > 0) {
		//         var pre = 'Error (statuscode: ' + statuscode + ', statustext: ' + statustext + "). Look below for details.";
		//         if (greydFormSettings.wp_debug) message_wrapper.after('<pre>'+pre+'</pre>');
		//         console.log(pre);
		//     }
		// }
		console.log( event );

		// extra log
		if ( typeof log !== 'undefined' ) {
			log = typeof log === 'object' || Array.isArray( log ) ? JSON.stringify( log, null, 4 ) : log;
			if ( log.length > 0 ) {
				// if (greydFormSettings.wp_debug) message_wrapper.after('<pre>'+log+'</pre>');
				console.log( log );
			}
		}

		// message content
		var message = message_wrapper.children( ".message" );
		var content = greydFormSettings.default_error;
		if ( typeof html === 'object' ) {
			content = JSON.stringify( html, null, 4 );
		}
		else if ( Array.isArray( html ) ) {
			if ( html.length > 1 ) {
				content = "<p style='margin-top:0;'>" + greydFormSettings.errors + "</p>";
				content += "<ul>";
				$.each( html, function ( key, val ) {
					content += "<li>" + val + "</li>";
				} );
				content += "</ul>";
			} else {
				content = html[ 0 ];
			}
		}
		else if ( typeof html !== 'undefined' && html.length > 0 ) {
			content = html;
		}

		message.html( content ).removeClass( 'success' ).addClass( 'danger' );
		forms.afterFormHandled( form );
	};

	/**
	 * Reset form
	 * 
	 * @param {object} form The form object.
	 */
	this.formReset = function ( form ) {

		forms.afterFormHandled( form );

		form.removeClass( 'validate' );

		// reset inputs
		$( 'form.greyd_form' ).find( 'input, textarea, select' ).each( function () {

			// $(this).val('').trigger('change');

			// default
			$( this ).removeClass( 'input-focus input-unfocus valid invalid' );
			// upload
			if ( $( this ).attr( 'type' ) == 'file' ) {
				forms.resetUpload( $( this ) );
			}
			// multiselects
			else if ( $( this ).parent().hasClass( 'greyd_multiselect' ) ) {
				$( this ).parent().find( '.input .tag' ).each( function () {
					$( this ).trigger( "click" );
				} );
			} else if ( $( this ).parent().hasClass( 'greyd_multiradio' ) ) {
				$( this ).parent().find( '.option.selected' ).removeClass( 'selected' );
			}
		} );

		greyd.input.closeAll( null );

		// trigger events for other JS classes
		if ( typeof MSF !== 'undefined' ) MSF.reset( form );
		if ( typeof CF !== 'undefined' ) CF.reset( form );
	};

	/**
	 * Execute recaptcha if necessary or directly call a callback.
	 * 
	 * @param {function} callback   Will be executed in even if recaptcha is undefined.
	 * @param {array} args          Arguments to be applied to the function.
	 */
	this.maybeExecuteCaptcha = function ( callback, args ) {

		var sitekey = typeof recaptchaSitekey !== 'undefined' ? recaptchaSitekey : null;

		if ( greydFormSettings.wp_debug ) console.log( "recaptcha sitekey: " + sitekey );

		// revalidate recaptcha
		if ( sitekey && !forms.captchaValid ) {
			grecaptcha.ready( function () {
				grecaptcha.execute( sitekey, { action: 'homepage' } ).then( function ( token ) {

					// set the class var to true
					forms.captchaValid = true;

					// append token to every recaptcha
					$( "input.recaptcha_input" ).each( function () {
						$( this ).val( token );
					} );
					console.log( 'reCAPTCHA token set: ' + token );

					// set timeout to invalidate the class var
					clearTimeout( forms.captchaTimeout );
					forms.captchaTimeout = setTimeout(
						function () { forms.clearCaptcha(); },
						1000 * 60 * 1.9 // valid for 2 minutes (ms * s * min)
					);

					// callback
					if ( typeof callback === "function" ) {
						callback.apply( null, args );
					}
				} );
			} );
		}
		// ...or call callback directly
		else if ( typeof callback === "function" ) {
			callback.apply( null, args );
		}
	};

	/**
	 * reCAPTCHA token is currently valid
	 */
	this.captchaValid = false;

	/**
	 * reCAPTCHA timeout for invalidation
	 */
	this.captchaTimeout = null;

	/**
	 * Clear the recaptcha tokens
	 */
	this.clearCaptcha = function () {
		forms.captchaValid = false;
		clearTimeout( forms.captchaTimeout );
	};


	/**
	 * =================================================================
	 *                          FRONTEND FEATURES
	 * =================================================================
	 */

	/**
	 * Init form tooltips
	 */
	this.initFormTooltips = function () {

		if ( $( '.forms-tooltip' ).length == 0 ) return;
		if ( typeof custom_inputs !== 'undefined' && typeof custom_inputs.initTooltips !== 'undefined' ) return;

		var activeTooltip = null;

		$( document ).on( "click", () => {
			if ( activeTooltip ) {
				activeTooltip.removeClass( 'on' );
				activeTooltip = null;
			}
		} );

		// toggle tooltip
		$( ".forms-tooltip .forms-tooltip-toggle" ).on( "click", ( e ) => {
			console.log( e );
			let tooltip = $( e.target ).parent();
			if ( tooltip.hasClass( 'on' ) ) {
				if ( activeTooltip ) {
					activeTooltip.removeClass( 'on' );
				}
				activeTooltip = null;
			} else {
				if ( activeTooltip ) {
					activeTooltip.removeClass( 'on' );
				}
				activeTooltip = tooltip;
			}
			tooltip.toggleClass( 'on' );
			e.stopPropagation();
		} );
	};

	/**
	 * Custom File Upload
	 */
	this.initUpload = function () {

		// vars
		var input = $( 'input[type="file"].custom-file-upload' );
		var label = input.next();
		var labeltext, labelval;

		// trigger input
		label.on( 'click', function () {

			// manually trigger the click event on the file input
			if ( !input.attr( 'id' ) ) $( this ).prev().trigger( "click" );

			labeltext = $( this ).find( '.input span[style="flex:1"]' );
			$( this ).prev().removeClass( 'input-unfocus valid invalid' ).addClass( 'input-focus' );
		} );

		// on trigger
		input.on( 'change', function ( e ) {

			var filename = $( this ).val().replace( /C:\\fakepath\\/i, '' );

			// file chosen
			if ( filename ) {

				// set label text
				if ( labeltext ) labeltext.text( filename );

				// validate filesize
				$( this ).removeClass( 'input-focus' ).addClass( 'input-unfocus' );
				var maxsize = $( this ).data( 'max' );
				if ( maxsize != 'none' )
					maxsize = parseFloat( maxsize.toString().replace( ',', '.' ) ) * 1000 * 1000;

				if ( maxsize == 'none' || this.files[ 0 ].size <= maxsize )
					$( this ).removeClass( 'invalid' ).addClass( 'valid' );
				else
					$( this ).removeClass( 'valid' ).addClass( 'invalid' );
			}
			// no file chosen or aborted
			else {
				forms.resetUpload( $( this ) );
			}

		} );

		// Firefox bug fix
		input.on( 'focus', function () {
			$( this ).addClass( 'has-focus' );
		} );
		input.on( 'blur', function () {
			$( this ).removeClass( 'has-focus' );
		} );
	};
	this.resetUpload = function ( input ) {
		var label = input.next();
		var labeltext = label.find( '.input span[style="flex:1"]' );
		var labelval = label.data( 'text' );
		labeltext.text( labelval );
		input.removeClass( 'valid invalid input-focus input-unfocus' );
	};

	/**
	 * Multiselect Radio Buttons
	 */
	this.onMultiRadioClick = function ( elem ) {
		$( elem ).toggleClass( 'selected' );
		var input = $( elem ).closest( '.greyd_multiradio' ).find( 'input' );
		forms.toggleMultiRadioValue( input, $( elem ).data( 'value' ) );
	};
	this.toggleMultiRadioValue = function ( input, value = "" ) {
		var oldval = input.val();
		var newval = oldval.indexOf( value ) === -1 ? oldval + "," + value : oldval.replace( value, "" );
		// cleanup double commas or commas at the end or beginning of the string
		newval = newval.replace( /,+/, "," ).replace( /^,|,$/, "" );

		input.val( newval );
		input.trigger( "change" );
		if ( greydFormSettings.wp_debug ) console.log( "New multiradio value: " + newval );
	};

	/**
	 * Custom Range Slider
	 */
	this.initCustomRange = function () {
		var timeout;

		$( '.range_control_wrapper' ).each( function () {

			var input = $( this ).children( "input[type='range']" );

			function onChange( e ) {

				input = typeof e === "undefined" ? input : $( this );

				var wrapper = input.parent();
				var track = wrapper.find( ".track" );
				var thumb = wrapper.find( ".thumb_wrapper" );
				var ttip = wrapper.find( ".tooltip" );
				var label = wrapper.siblings( ".label_wrap" ).find( ".range_value" );
				var min = parseFloat( input.attr( "min" ) );
				var max = parseFloat( input.attr( "max" ) );
				var val = parseFloat( input.val() );
				var offset = 100 / ( ( max - min ) / ( val - min ) );

				// set validity
				var isRequired = input[0].hasAttribute('required');
				if ( isRequired ) {
					if ( input.val() == '0' || input.val() == '' ) {
						input[0].setCustomValidity( 'Cannot be zero' );
					} else if ( input.val() > max ) {
						input[0].setCustomValidity( 'Cannot be greater than ' + max );
					} else if ( input.val() < min ) {
						input[0].setCustomValidity( 'Cannot be less than ' + min );
					} else {
						input[0].setCustomValidity( '' );
					}
				}

				label.text( val );
				ttip.text( val );

				track.css( "width", "calc(10px + " + offset + "%)" );
				thumb.css( "left", offset + "%" );

				// safari and mobile devices don't support 'focus' event
				if (
					typeof e !== "undefined" && // only on real change event
					( $( "html" ).hasClass( "safari" ) || $( window ).width() <= 1000 )
				) {
					$( ".range_control_wrapper input[type='range']" ).removeClass( "input-focus" );
					input.addClass( "input-focus" );
					clearTimeout( timeout );
					timeout = setTimeout( function () {
						input.removeClass( "input-focus" );
					}, 1200 );
				}
			}

			input.on( "input change", onChange );
			onChange();
		} );
	};

	/**
	 * Math Operation Fields
	 */
	this.initMathFields = function () {
		$( '.math_field' ).each( function () {
			forms.updateMathField( $( this ) );
		} );
	};

	this.updateMathField = function ( math_field ) {

		var formula = math_field.length ? math_field.data( "formula" ) : false;
		var form = math_field.closest( "form.greyd_form" );
		var cond_num = -1;
		const rounding = math_field.length ? math_field[ 0 ].getAttribute( "data-rounding" ) : false;

		if ( greydFormSettings.wp_debug ) var name = forms.getCleanName( math_field.attr( "name" ) );

		// check for conditions
		if ( math_field.hasClass( "has_conditions" ) ) {
			var i = 0;
			math_field.siblings( ".math_conditions" ).children().each( function () {

				var value1 = forms.getFormulaValue( $( this ).data( "field1" ), form );
				if ( isNaN( value1 ) ) return;

				var value2 = forms.getFormulaValue( $( this ).data( "field2" ), form );
				if ( isNaN( value2 ) ) return;

				var operator = $( this ).data( "operator" );

				if ( greydFormSettings.wp_debug ) console.log( String( value1 ) + " " + operator + " " + String( value2 ) );

				if ( CF.matchCondition( 'or', value1, operator, value2 ) ) {
					cond_num = i;
					formula = $( this ).data( "value" );
					return false;
				}
				i++;
			} );

			if ( greydFormSettings.wp_debug ) {
				if ( cond_num === -1 ) {
					console.log( "No condition hit for math field '" + name + "'" );
				}
				else {
					console.log( "Condition number " + cond_num + " hit for math field '" + name + "' with formula '" + formula + "'" );
				}
			}
		}

		// let's calculate!
		var value = forms.getFormulaValue( formula, form );
		if ( isNaN( value ) ) {
			math_field.addClass( "error" );
			return;
		}
		else if ( math_field.hasClass( "error" ) ) {
			math_field.removeClass( "error" );
		}

		// round the value
		var decimals = math_field.data( "decimals" ) ?? "0";
		// rounding
		switch ( rounding ) {
			case "ceil":
				value = +( Math.ceil( value + "e+" + decimals ) + "e-" + decimals );
			case "floor":
				value = +( Math.floor( value + "e+" + decimals ) + "e-" + decimals );
			case "auto":
				value = +( Math.round( value + "e+" + decimals ) + "e-" + decimals );
		}

		value = value.toFixed( decimals );

		// update the field
		math_field.val( value );

		if ( greydFormSettings.wp_debug ) console.log( "Math field '" + name + "' value updated to '" + value + "' (" + typeof value + ")" );

		math_field.trigger( "change" );

		// update the displayed formulas
		var display = math_field.siblings( ".display" );
		if ( display.length ) {
			display.attr( "data-condition", String( cond_num ) );
		}
	};

	this.initConfirmPasswordField = function () {

		const confirmPasswordFields = document.querySelectorAll( 'input[type="password"].confirm-password' );

		confirmPasswordFields.forEach( confirmField => {
			//get closest (true) password field

			const form = confirmField.closest( "form.greyd_form" );
			const submit = form.querySelector( "button[name='submit']" );
			const confirmPasswordMessage = confirmField.closest( '.input-wrapper' ).querySelector( 'span.confirm-password-message' );
			const passwordField = confirmField.closest( ".input-outer-wrapper" ).querySelector( 'input[type="password"]:not(.confirm-password)' );
			const labelMatching = confirmPasswordMessage.getAttribute( 'data-label-matching' );
			const labelNotMatching = confirmPasswordMessage.getAttribute( 'data-label-notmatching' );

			confirmField.addEventListener( "keyup", function () {
				if ( this.value.length > 0 ) {
					if ( this.value == passwordField.value ) {
						confirmPasswordMessage.style.color = "green";
						confirmPasswordMessage.innerHTML = labelMatching;
						"matching";
						submit.disabled = false;
						confirmField.setCustomValidity( "" );
					} else {
						confirmPasswordMessage.style.color = "red";
						confirmPasswordMessage.innerHTML = labelNotMatching;
						submit.disabled = true;
						confirmField.setCustomValidity( "Invalid field." );
					}
				}
			} );
		} );
	};

	/**
	 * Calculate a formula with field placeholders etc.
	 * @param {string} formula  Formula to evaluate
	 * @param {object} form     Greyd.Form
	 * @returns {float|NaN} NaN on error, float on success.
	 */
	this.getFormulaValue = function ( formula, form ) {

		if ( !formula ) return false;

		var fields = String( formula ).match( /{([A-Za-z0-9\-\_\[\]]+?)}/g );

		if ( fields ) {
			fields.forEach( function ( field_name ) {

				field_name = field_name.replace( /[{}]/g, "" );
				var field = forms.getFieldByName( field_name, form );
				var val = 0;

				if ( field.length ) {
					val = forms.getVal( field );

					// convert numeric values to float
					if ( typeof val === "string" && val.match( /^[.,0-9]+$/ ) ) {
						val = parseFloat( val );
					}
				}

				// convert to float if not already
				if ( !forms.isFloat( val ) && !forms.isInt( val ) ) {
					val = parseFloat( String( val ).replaceAll( /[^-,0-9]/g, '' ) );
				}

				// if (greydFormSettings.wp_debug) console.log( field_name+": "+val+" ("+typeof val+")" );

				val = isNaN( val ) ? 0 : val;
				formula = formula.replace( "{" + field_name + "}", String( val ) );
			} );
		}

		result = forms.calculateFormula( formula );

		return result;
	};

	/**
	 * Calculate a formula value
	 * @param {string} formula Formula to evaluate
	 * @returns {mixed}
	 */
	this.calculateFormula = function ( formula ) {

		if ( typeof formula !== "string" || formula.length === 0 ) return formula;

		var regex = "";

		// replace whitespace & commas
		formula = formula.replace( /\s+/g, "" ).replace( /\,/, "." );

		// only math chars
		formula = formula.replace( /[^\(\d\.\-\+\*\/\^\√log\)]/g, "" );

		// convert '(4*2)^(2+1)' to 'Math.pow(4*2, 2+1)'
		regex = /([\d.]+|\([^\(\)]+?\))\^(\d+|\([^\(\)]+?\))+?/g;
		formula = formula.replace( regex, function ( match, base, power ) {
			return "Math.pow(" + base + ", " + power + ")";
		} );

		// convert 'log(2' to 'Math.log('
		regex = /log\(/g;
		formula = formula.replace( regex, "Math.log(" );

		// convert '√(' to 'Math.sqrt('
		regex = /\√\(/g;
		formula = formula.replace( regex, "Math.sqrt(" );

		if ( greydFormSettings.wp_debug ) console.log( formula );

		// operate via eval()
		var value = 0;
		try {
			value = eval( formula );
		} catch ( e ) {
			console.error( e );
			return NaN;
		}
		return value;
	};

	/**
	 * Init Dynamic Form Tags
	 */
	this.initFormTags = function () {

		$( "form.greyd_form input, form.greyd_form textarea, form.greyd_form select" ).each( function () {
			$( this ).on( "change input", function () {
				forms.changeDependingFormTags( this );
			} );
		} );
		setTimeout( function () {
			$( "form.greyd_form input, form.greyd_form textarea, form.greyd_form select" ).each( function () {
				forms.changeDependingFormTags( this );
			} );
		}, 100 );
	};

	/**
	 * Change all form tags depending on a single field
	 * @param {object} elem Input element
	 */
	this.changeDependingFormTags = function ( elem ) {

		if ( $( elem ).attr( "novalue" ) === "true" ) return;

		var name = forms.getCleanName( $( elem ).attr( "name" ) );
		var value = forms.getDisplayVal( $( elem ) );

		// update depending form tag
		var tags = $( "form.greyd_form .form_tag[data-name='" + name + "']" );
		if ( tags.length ) {
			tags.each( function () {

				$( this ).text( value );

				// update math fields
				if ( $( this ).parent( "span" ).hasClass( "formula" ) || $( this ).parent().parent( "span" ).hasClass( "formula" ) ) {
					forms.updateMathField( $( this ).closest( ".math_operation" ).find( ".math_field" ) );
				}
			} );
		}

		// // find depending math fields
		// var math_fields = $(elem).closest("form.greyd_form").find(".math_field[data-formula*='{"+name+"}']");
		// if ( math_fields.length ) {
		//     math_fields.each(function() {
		//         forms.updateMathField( math_fields );
		//     });
		// }

		// find depending conditons
		var cond_wrapper = $( elem ).closest( "form.greyd_form" ).find( ".math_conditions" );
		if ( cond_wrapper.length ) {
			cond_wrapper.each( function () {
				var cond_items = $( this ).find( "span[data-field1*='{" + name + "}'], span[data-field2*='{" + name + "}'], span[data-value*='{" + name + "}']" );
				if ( cond_items.length ) {
					forms.updateMathField( $( this ).siblings( ".math_field" ) );
				}
			} );
		}
	};

	this.isFloat = function ( n ) {
		return n === +n && n !== ( n | 0 );
	};

	this.isInt = function ( n ) {
		return n === +n && n === ( n | 0 );
	};

	/**
	 * =================================================================
	 *                          HELPERS
	 * =================================================================
	 */

	/**
	 * Find a input by it's name
	 * @param {string} name Name of the input
	 * @param {object} form jQuery object of the form to crawl
	 */
	this.getFieldByName = function ( name, form ) {
		return form.find( "[name='" + name + "'], [name^='" + name + "___id']" );
	};

	/**
	 * Get cleaned up name for a string value
	 * @param {mixed} string Name of the input
	 * @returns mixed
	 */
	this.getCleanName = function ( string ) {

		// remove suffix '__id12345'
		var pos = typeof string === "string" ? string.indexOf( '___id' ) : -1;
		if ( pos !== -1 ) {
			string = string.substring( 0, pos );
		}
		return string;
	};

	/**
	 * Get the value of an input field
	 * @param {mixed} input jQuery object or the value
	 * @returns mixed
	 */
	this.getVal = function ( input ) {
		var value;

		// not an object
		if ( typeof input !== "object" ) {
			value = input;
		}
		// jQuery object
		else if ( input.length > 0 ) {

			var type = input.attr( 'type' );
			if ( input.hasClass( 'checkbox_placeholder_value' ) ) {
				type = 'checkbox';
			}

			// get value
			if ( type === 'radio' ) {
				value = input.filter( ':checked' ).val();
			}
			else if ( type === 'checkbox' ) {
				value = input.filter( ':not(.checkbox_placeholder_value)' ).first().prop( "checked" ) ? 'on' : 'off';
			}
			else {
				if ( input.length > 1 ) {
					input = input.filter( ':not([novalue="true"])' ).first();
				}
				value = input.length ? input.val() : null;
			}
		}

		return forms.getCleanName( value );
	};

	/**
	 * Get a nice looking display var
	 * @param {object} input Input element (jQuery object)
	 * @returns {string}
	 */
	this.getDisplayVal = function ( input ) {

		let value = forms.getVal( input );

		// convert numeric values to float
		if ( typeof value === "string" && value.match( /^[.,0-9]+$/ ) ) {
			value = parseFloat( value );
		}

		if ( decimals = input.data( "decimals" ) ) {
			value = typeof value === "string" ? "" : parseFloat( value ).toFixed( decimals );
		}
		else if ( typeof value === "undefined" || value === null ) {
			value = '';
		}
		else {
			value = String( value );
		}

		// set points as commas and add points as thousands
		if ( value.indexOf( "." ) === -1 ) {
			// 140000 --> 140.000
			value = value.replace( /\B(?=(\d{3})+(?!\d))/g, "." );
		}
		else {
			// 140000,50 --> 140.000,50
			value = value.split( "." );
			value = value[ 0 ].replace( /\B(?=(\d{3})+(?!\d))/g, "." ) + "," + value[ 1 ];
		}

		return value;
	};

	/**
	 * WPML Multidomain Mapping
	 *  
	 * this function prevents a known bug, where the admin-ajax is called from the root language domain
	 * it matches the ajax-url given by the core function "admin_url( 'admin-ajax.php' )" via enqueue.php
	 * and replaces the domain ending (tld) of the ajax-url with the current window domain ending
	 *  
	 * Example:
	 * Multidomain mapping is set to greyd.io (DE - root) & greyd.com (EN).
	 * In some setups the admin-ajax returns 'greyd.io/wp-admin/admin-ajax.php' no matter the current domain location.
	 *
	 * While this function prevents that behaviour it does not actually fix the error.
	 */
	this.setAjaxUrl = function () {

		function getUrlEnding( url ) {
			var match = url.match( /(\.)([a-z]*?)(\/)/ );
			if ( match ) return match[ 2 ];
			else return false;
		}
		function replaceUrlEnding( url, end ) {
			return url.replace( /(\.)([a-z]*?)(\/)/, '.' + end + '/' );
		}

		var ajax_url = greydFormSettings.ajaxurl;
		if ( greydFormSettings.wp_debug ) console.log( "WPML Setup = " + { 0: "Off", 1: "Directory", 2: "Different Domains", 3: "URL-Parameter" }[ greydFormSettings.wpml_setup ] );
		if ( greydFormSettings.wpml_setup === '2' ) {
			// get current url
			var url = window.location.href;
			var tld = getUrlEnding( url );
			var ajax_tld = getUrlEnding( ajax_url );
			if ( tld && tld !== ajax_tld ) ajax_url = replaceUrlEnding( ajax_url, tld );
			// if (greydFormSettings.wp_debug) console.log( "AJAX-URL was changed from '"+greydFormSettings.ajaxurl+"' to '"+ajax_url+"'." );
		}
		forms.ajax_url = ajax_url;
	};
};
