/*
	Comments: Javascript File for Live Surveys Frontend
*/

document.addEventListener( 'DOMContentLoaded', function () {
	LiveSurvey.init();
} );

const LiveSurvey = new function () {
	this.forms = [];
	this.surveys = [];
	this.debug = true;

	/**
	 * Init the class
	 */
	this.init = function () {
		this.forms = document.querySelectorAll( "form.greyd_form" );

		if ( this.forms.length === 0 ) return;

		this.forms.forEach( form => {
			let surveys = form.querySelectorAll( "[data-survey-id]" );

			surveys.forEach( survey => {
				const surveyId = survey.getAttribute( 'data-survey-id' );
				if ( survey ) {
					this.surveys.push( survey );

					let cookie = this.getCookie( `greyd_survey_participated_${surveyId}` ) ;
					cookie = cookie ? JSON.parse( cookie ) : false;
			
					if ( cookie ) {
						this.updateSurvey( form, surveyId, false, false );
						const option = survey.querySelector( `fieldset input[value='${cookie.value}']` )
						if ( option) option.checked = true;
						survey.querySelector( "fieldset" ).setAttribute( "disabled", true );
					}
				}
			} );
		} );

		// add listeners
		this.addListeners();
	};

	this.addListeners = function () {

		this.surveys.forEach( ( survey ) => {
			const form = survey.closest( "form.greyd_form" );
			const surveyId = survey.getAttribute( 'data-survey-id' );

			// update survey on click
			survey.getElementsByTagName( "fieldset" )[ 0 ].addEventListener( "change", ( e ) => {
				this.updateSurvey( form, surveyId, e );
				e.target.closest( "fieldset" ).setAttribute( "disabled", true );
				const cookie = {
					surveyId: surveyId,
					value: e.target.value
				};
				
				// set cookie
				this.setCookie( `greyd_survey_participated_${surveyId}`, cookie, 30 );
			} );
		} );
	};

	this.fetchSurveyResult = async ( form, surveyId ) => {
		
		const formID = form.getAttribute( "data-id" );
		const nonce = form.getAttribute( "data-nonce" );
		const uid = form.getAttribute( "id" );

		const body = new FormData();

		const data = {
			form_id: formID,
			uid: uid,
			survey_id: surveyId
		};

		// append the necessary data for the admin ajax call
		body.append( 'action', 'send_survey' );
		body.append( 'nonce', nonce );
		body.append( 'data', JSON.stringify( data ) );

		if ( !forms.ajax_url ) forms.setAjaxUrl();

		return await fetch( forms.ajax_url, {
			method: 'POST',
			credentials: 'same-origin',
			body: body,
		} );
	};

	this.updateSurvey = ( form, surveyId, event, sendToBackend = true ) => {
		console.log("start loading...");
		const survey = form.querySelector( `[data-survey-id='${ surveyId }']` );
		survey.setAttribute( "disabled", true );
		survey.style.opacity = 0.7;
		this.fetchSurveyResult( form, surveyId )
			.then( ( response ) => response.json() )
			.then( ( data ) => {
				if ( data && data.success ) {
					if ( LiveSurvey.debug ) console.log( data );

					const surveyData = data.data.survey_data;
					let totalSubmissions = data.data.total_submissions;
					const surveyWrapper = document.querySelector( `[data-survey-id='${ surveyId }']` );
					const surveyAnswerValue = event?.target?.value;
					const newSurveyData = [];
					const surveyTitle = surveyWrapper.getAttribute( "data-survey-title" );
				
					survey.setAttribute( "disabled", false );
					survey.style.opacity = 1;

					if ( surveyData ) {
						surveyData.forEach( datapoint => {

							// if the option is the clicked one we update our survey data
							if ( datapoint.value === surveyAnswerValue ) {
								datapoint.survey_result++;
								totalSubmissions++;
							};

							const option = surveyWrapper.querySelector( `input[value='${ datapoint.value }']` );
							const optionWrapper = option.closest( ".option" );
							const progressWrapper = optionWrapper.querySelector( ".progress_wrapper" );
							const bar = progressWrapper.querySelector( ".bar" );
							bar.style.width = 0;
							bar.style.width = `${ Math.round( ( datapoint.survey_result / totalSubmissions ) * 100 ) }%`;
						} );
						return {
							surveyId: surveyId,
							options: surveyData,
							totalSubmissions: totalSubmissions,
							surveyTitle: surveyTitle
						};
					} else {

						const targetOption = event.target;

						if ( !event.target ) return;

						const targetOptionWrapper = targetOption.closest( ".option" );
						const progressWrapper = targetOptionWrapper.querySelector( ".progress_wrapper" );
						const bar = progressWrapper.querySelector( ".bar" );

						bar.style.width = ` ${ 100 }%`;

						const allOptions = surveyWrapper.querySelectorAll( "input" );
						totalSubmissions = 1;

						allOptions.forEach( ( option ) => {
							let surveyResult = targetOption == option ? 1 : 0;
							newSurveyData.push( {
								'survey_result': surveyResult,
								value: option.value
							} );
						} );

						return {
							surveyId: surveyId,
							options: newSurveyData,
							totalSubmissions: totalSubmissions,
							surveyTitle
						};
					}
				}
			} )
			.then( ( finalData ) => {
				// send final data to backend
				if ( sendToBackend ) {
					this.sendSurveyData( form, finalData );
				}
			} )
			.catch( ( error ) => {
				if ( LiveSurvey.debug ) console.error( error );
				return false;
			} );
	};

	this.sendSurveyData = async ( form, finalData ) => {
		const formID = form.getAttribute( "data-id" );
		const nonce = form.getAttribute( "data-nonce" );
		const uid = form.getAttribute( "id" );

		const body = new FormData();

		const data = {
			form_id: formID,
			uid: uid,
			surveyData: finalData,
		};

		// append the necessary data for the admin ajax call
		body.append( 'action', 'receive_survey' );
		body.append( 'nonce', nonce );
		body.append( 'data', JSON.stringify( data ) );

		if ( !forms.ajax_url ) forms.setAjaxUrl();

		return await fetch( forms.ajax_url, {
			method: 'POST',
			credentials: 'same-origin',
			body: body,
		} );
	};

	this.setCookie = ( cname, cvalue, exdays ) => {
		const d = new Date();
		d.setTime( d.getTime() + ( exdays * 24 * 60 * 60 * 1000 ) );
		let expires = "expires=" + d.toUTCString();
		document.cookie = cname + "=" + JSON.stringify(cvalue) + ";" + expires + ";path=/";
	};

	this.getCookie = ( cname ) => {
		let name = cname + "=";
		let ca = document.cookie.split( ';' );
		for ( let i = 0; i < ca.length; i++ ) {
			let c = ca[ i ];
			while ( c.charAt( 0 ) == ' ' ) {
				c = c.substring( 1 );
			}
			if ( c.indexOf( name ) == 0 ) {
				return c.substring( name.length, c.length );
			}
		}
		return "";
	};
};