
var greyd = greyd || {};
greyd.forms = greyd.forms || {};

/**
 * Multistep forms module.
 */
greyd.forms.multistep = new function () {

	/**
	 * Holds all the multistep forms.
	 * Keyed by the element id.
	 * 
	 * @type {object}
	 */
	this.elements = [];

	/**
	 * Holds the last hash to determine if a hash change is a multistep form change.
	 */
	this.lastHash = '';

	/**
	 * Initialize the module.
	 */
	this.init = function () {

		if ( typeof $ === 'undefined' ) $ = jQuery;
		if ( $( '.multistep_wrapper' ).length == 0 ) return;

		/**
		 * Multistep form class.
		 */
		class Greyd_Multistep {
			constructor ( wrapper ) {

				// setup class properties
				this.wrapper = $( wrapper );

				this.form = this.wrapper.closest( '.greyd_form' );
				this.id = this.wrapper.attr( 'id' ) ?? this.form.attr( 'id' );
				this.progressBar = this.wrapper.next( '.progress_wrapper' )?.children();
				this.steps = this.wrapper.find( '.step' );
				this.currentStepIndex = 0;
				this.config = {
					autoStep: this.wrapper.hasClass( 'auto_step' ),
					transition: this.wrapper.data( 'transition' ),
				};

				this.quizMode = this.wrapper[ 0 ].getAttribute( 'data-quiz' ) == "true";
				this.correctAnswersField = this.form[ 0 ].querySelector( 'input[data-correct-answers-field=true]' );
				if ( this.correctAnswersField ) this.correctAnswersField.value = 0;

				this.wait = false;
				this.currentCorrectAnswer = null;
				this.selectedAnswer = null;

				// bind functions
				this.updateStep.bind( this );
				this.animateToStep.bind( this );
				this.currentStepInputsValid.bind( this );
				this.updateProgress.bind( this );
				this.scrollIntoViewport.bind( this );

				// add events
				this.wrapper.on( 'nextStep', this.nextStep.bind( this ) );
				this.wrapper.on( 'prevStep', this.prevStep.bind( this ) );
				this.wrapper.find( '.next_step' ).on( 'click', () => this.wrapper.trigger( 'nextStep' ) );
				this.wrapper.find( '.prev_step' ).on( 'click', () => this.wrapper.trigger( 'prevStep' ) );
				this.wrapper.find( '.exact_step' ).on( 'click', ( e ) => {
					// get data-step and convert to int
					const step = parseInt( $( e.currentTarget ).data( 'step' ), 10 );
					this.updateStep( step - 1 );
				} );

				// initial state
				this.steps.eq( this.currentStepIndex ).addClass( 'active' );
				this.updateProgress( 0 );

				// setup auto step events
				if ( this.config.autoStep || this.quizMode ) {
					this.setupAutoStepEvents.bind( this );
					this.setupAutoStepEvents();
				}

				// handle submit by 'Enter' key
				this.wrapper.on( 'keypress', function ( e ) {

					var code = e.keyCode || e.which;
					if ( code != 13 || $( ':focus' ).is( "textarea" ) ) return;

					// on the last step, submit the form
					if ( this.currentStepIndex === this.steps.length - 1 ) {
						return;
					}

					// try to go to the next step
					e.preventDefault();
					$( ':focus' ).blur();
					this.wrapper.trigger( 'nextStep' );
				} );

				/**
				 * Fill up progress bar after submit.
				 * @since 2.0.0
				 */
				this.form.on( "reset", () => {
					this.updateProgress( this.steps.length );
				} );
			}

			/**
			 * Go to the next step.
			 */
			nextStep() {
				if ( this.quizMode ) {

					this.wait = !this.wait;

					if ( !this.wait ) {
						this.resetMessagebox();
						this.currentCorrectAnswer = null;
						this.selectedAnswer.value = '';
					}

					var isAnswerCorrect = false;
					if ( this.selectedAnswer ) {
						isAnswerCorrect = this.checkForCorrectAnswer();
					}
				}

				// check if current step is valid
				if ( !this.currentStepInputsValid() ) {
					this.form.find( 'button[type="submit"]' ).trigger( "click" );
					return;
				}

				if ( !this.wait ) this.updateStep( this.currentStepIndex + 1 );

				if ( this.quizMode & this.wait ) this.showQuizResults( isAnswerCorrect );
			}

			/**
			 * Go to the previous step.
			 */
			prevStep() {
				this.updateStep( this.currentStepIndex - 1 );
			}

			/**
			 * Update the current step.
			 * @param {number} index
			 */
			updateStep( index ) {

				if (
					index < 0
					|| index > this.steps.length - 1
					|| index === this.currentStepIndex
				) {
					return;
				}

				this.steps.removeClass( 'active' );
				this.steps.eq( index ).addClass( 'active' );
				this.animateToStep( index );
				this.updateProgress( index );

				this.currentStepIndex = index;

				this.scrollIntoViewport();

				// update url hash
				window.location.hash = this.id + '-step-' + ( index + 1 );
			}

			/**
			 * Check if all required fields in the current step are valid.
			 * @returns {boolean} true if all fields in the current step are valid
			 */
			currentStepInputsValid() {
				var isValid = true;

				const currentStep = this.steps.eq( this.currentStepIndex );

				currentStep.find( 'input[required], select[required], textarea[required]' ).each( function () {
					var inputElement = this;
					var isInputValid = inputElement.checkValidity();

					if ( !isInputValid ) {
						isValid = false;
						return false; // Exit the loop early, no need to check other fields
					}
				} );

				return isValid;
			}

			/**
			 * Animate the step transition.
			 * @param {number} index
			 */
			animateToStep( index ) {
				if ( this.config.transition !== 'swipe' ) return;

				this.wrapper.animate(
					{ now: index },
					{
						duration: 300,
						step: function ( now ) {
							$( this ).css( {
								"-webkit-transform": 'translateX(' + ( -now * 100 ) + '%)',
								"-ms-transform": 'translateX(' + ( -now * 100 ) + '%)',
								"transform": 'translateX(' + ( -now * 100 ) + '%)',
							} );
						}
					}
				);
			}

			/**
			 * Update the progress bar.
			 */
			updateProgress( index ) {
				// set progress bar
				if ( this.progressBar.hasClass( 'multistep_progress' ) ) {
					var bar = this.progressBar.children();
					if ( bar.length > 0 ) {
						var x = ( index + .5 ) / this.steps.length;
						bar.css( { "width": x * 100 + "%" } );
					}
				}
				// set pagination
				else if ( this.progressBar.hasClass( 'multistep_pagination' ) ) {
					this.progressBar.find( "span:nth-child(n+" + ( index + 2 ) + ")" ).removeClass( 'active' );
					this.progressBar.find( "span:nth-child(-n+" + ( index + 1 ) + ")" ).addClass( 'active' );
				}
			}

			/**
			 * Scroll the wrapper into the viewport if it's out of it.
			 */
			scrollIntoViewport() {
				// Get the top offset of the element relative to the viewport
				var elementTop = this.wrapper[ 0 ].getBoundingClientRect().top;

				// Check if the element top is out of the viewport
				var isOutOfViewport = elementTop < 0 || elementTop > window.innerHeight;

				if ( isOutOfViewport ) {
					console.log( 'scroll', elementTop, window.innerHeight );
					this.wrapper[ 0 ].scrollIntoView();
				}
			}

			/**
			 * Setup auto-step events.
			 */
			setupAutoStepEvents() {
				this.wrapper.on( 'change', 'input, select, textarea', ( e ) => {
					if ( this.quizMode ) {
						this.selectedAnswer = e.target;
					} else {
						if ( this.currentStepIndex < this.steps.length - 1 && this.currentStepInputsValid() ) {
							this.wrapper.trigger( 'nextStep' );
						}
					}
				} );
			}

			showQuizResults( isAnswerCorrect ) {
				if ( !this.quizMode ) return false;

				const formsWrapper = this.form[ 0 ]?.parentNode;
				const messageWrapper = formsWrapper?.querySelector( ".after-message" );
				const message = messageWrapper?.querySelector( ".message" );

				const correctAnswerMessage = this.wrapper[ 0 ]?.getAttribute( "data-correct-answer-text" );
				const wrongAnswerMessage = this.wrapper[ 0 ]?.getAttribute( "data-wrong-answer-text" );
				const correctAnswer = this.currentCorrectAnswer.getAttribute( "data-correct-answer" ) == "true" ? this.currentCorrectAnswer.value : this.currentCorrectAnswer.getAttribute( "data-correct-answer" );

				if ( isAnswerCorrect ) {
					this.selectedAnswer?.parentNode.classList.add( "answer-correct" );
					message.innerHTML = correctAnswerMessage + " " + correctAnswer;
					message.classList.add( 'success' );
					if ( this.correctAnswersField ) {
						const correctAnswersCount = parseInt( this.correctAnswersField.value );
						this.correctAnswersField.value = correctAnswersCount + 1;

						// trigger change to update dynamic form tags
						this.correctAnswersField.dispatchEvent( new Event( 'change' ) );
					}
				} else {
					this.selectedAnswer.parentNode.classList.add( "answer-false" );
					this.currentCorrectAnswer?.parentNode.classList.add( "answer-correct" );
					message.innerHTML = wrongAnswerMessage + " " + correctAnswer;
					message.classList.add( 'danger' );
				}

				const height = message.offsetHeight + parseInt( window.getComputedStyle( message ).marginTop ) + parseInt( window.getComputedStyle( message ).marginBottom );
				messageWrapper.style.height = `${ height }px`;
				messageWrapper.style.opacity = 1;
			}

			checkForCorrectAnswer() {
				this.currentCorrectAnswer = this.form[ 0 ].querySelector( '.step.active [data-correct-answer=true]' );
				let textAnswer = '';
				if ( !this.currentCorrectAnswer ) {
					this.currentCorrectAnswer = this.form[ 0 ].querySelector( '.step.active [data-correct-answer]' );
					textAnswer = this.currentCorrectAnswer.getAttribute( 'data-correct-answer' );
				}

				if ( this.selectedAnswer.value === "" ) return false;

				return ( /true/ ).test( this.selectedAnswer.getAttribute( 'data-correct-answer' ) ) || textAnswer === this.selectedAnswer.value.toLowerCase();
			}
			resetMessagebox() {
				const formsWrapper = this.form[ 0 ]?.parentNode;
				const messageWrapper = formsWrapper?.querySelector( ".after-message" );
				const message = messageWrapper?.querySelector( ".message" );
				message.classList.remove( 'success' );
				message.classList.remove( 'danger' );
				message.innerHTML = "";
				messageWrapper.style.height = "0";
				messageWrapper.style.opacity = "0";
			}
		}

		// init
		$( '.multistep_wrapper' ).each( function () {
			const id = this.getAttribute( 'id' ) ?? this.closest( '.greyd_form' ).getAttribute( 'id' );

			greyd.forms.multistep.elements[ id ] = new Greyd_Multistep( this );
		} );

		/**
		 * Bind step-change to hashchange event.
		 * 
		 * This allows to go back to the last step when the user clicks the back button.
		 */
		$( window ).on( 'hashchange', function ( e ) {

			let hash = window.location.hash.split( '-step-' );

			// update step if hash contains valid step
			if ( hash.length === 2 ) {

				const id = hash[ 0 ].replace( '#', '' );
				const step = parseInt( hash[ 1 ], 10 );

				if ( greyd.forms.multistep.elements[ id ] ) {
					greyd.forms.multistep.elements[ id ].updateStep( step - 1 );
				}
			}
			// if hash is empty and lastHash is not empty, go back to the first step
			else if ( hash.length < 2 && greyd.forms.multistep.lastHash !== '' ) {
				hash = greyd.forms.multistep.lastHash.split( '-step-' );
				console.log( 'lastHash', hash );
				if ( hash.length === 2 ) {

					const id = hash[ 0 ].replace( '#', '' );

					if ( greyd.forms.multistep.elements[ id ] ) {
						greyd.forms.multistep.elements[ id ].updateStep( 0 );
					}
				}
			}

			greyd.forms.multistep.lastHash = window.location.hash;
		} );

	};
};

document.addEventListener( 'DOMContentLoaded', function () {
	// jQuery(function() {
	greyd.forms.multistep.init();
} );
