/*
	Comments: Javascript File for Multi-Step Forms (MSF)
*/

document.addEventListener( 'DOMContentLoaded', function () {
	// jQuery(function() {
	MSF.init();
} );

var MSF = new function () {

	this.lastTitle = '';
	this.requiredInputs = new Object();

	this.mode = null;

	this.init = function () {
		if ( typeof $ === 'undefined' ) $ = jQuery;

		// early escape
		if ( $( '.multistep_wrapper, .greyd_form.multistep' ).length == 0 ) return;

		this.addEvents();
	};

	this.addEvents = function () {

		$( '.multistep_wrapper, .greyd_form.multistep' ).each( function () {

			// initial setup
			var form = $( this ).hasClass( 'greyd_form' ) ? $( this ) : $( this ).closest( '.greyd_form' );
			var id = form.attr( 'id' );
			var autostep = $( this ).hasClass( 'auto_step' ) || form.hasClass( 'auto_step' );

			MSF.setup( form );
			var autostep = MSF.mode == "vc" ? form.hasClass( 'auto_step' ) : form.find( ".multistep_wrapper" ).hasClass( 'auto_step' );

			// onchange events
			form.find( 'input, select, textarea' ).each( function () {
				$( this ).on( 'change', function ( e ) {


					var required = autostep ? true : $( this ).attr( 'required' );
					var empty = ( this.value && this.value !== '' && this.value !== undefined && this.value !== 'false' ) ? false : true;
					var valid = $( this )[ 0 ].validity.valid;

					if ( required ) {
						if ( !valid )
							MSF.addInput( $( this ), id );
						else
							MSF.removeInput( $( this ), id );
					} else {
						if ( empty || valid )
							MSF.removeInput( $( this ), id );
						else
							MSF.addInput( $( this ), id );
					}

					// move if all inputs set on autostep form
					if ( autostep && MSF.requiredInputs[ id ].length === 0 ) {
						$( '.multistep_button' ).off( 'click' );
						MSF.move( form );
					}
				} );
			} );
		} );

		// bind step-change to hashchange event
		$( window ).on( 'hashchange', function ( e ) {

			var hash = window.location.hash.split( '_step' );
			var title = hash[ 0 ].replace( '#', '' );
			var step = parseInt( hash[ 1 ], 10 );

			if ( title.length === 0 ) { // if hash is empty -> reset
				title = MSF.lastTitle;
				step = 0;
			} else {
				MSF.lastTitle = title;
			}

			// go to step
			$( 'form.greyd_form[data-title="' + title + '"]' ).each( function () {
				MSF.step( $( this ), step );
			} );
		} );

		// disable enter submit
		$( '.multistep_wrapper, .greyd_form.multistep' ).on( 'keypress', function ( e ) {
			var code = e.keyCode || e.which;

			var form = $( this ).hasClass( 'greyd_form' ) ? $( this ) : $( this ).closest( '.greyd_form' );
			var id = form.attr( 'id' );
			var autostep = $( this ).hasClass( 'auto_step' ) || form.hasClass( 'auto_step' );

			if ( code != 13 || $( ':focus' ).is( "textarea" ) ) return;

			// auto-step
			if ( autostep ) {
				e.preventDefault();
				$( ':focus' ).trigger( "change" );
				return false;
			}
			// submit-step
			else if (
				form.find( 'section.vc_section.active' ).next().hasClass( 'vc_section' ) ||
				form.find( '.step.active' ).next().hasClass( 'step' )
			) {

				// remove name from array
				if ( $( ':focus' )[ 0 ].validity.valid ) {
					MSF.removeInput( $( ':focus' ), id );
				} else {
					MSF.addInput( $( ':focus' ), id );
				}
				// move if all inputs set
				if ( MSF.requiredInputs[ id ].length === 0 ) {
					e.preventDefault();
					$( ':focus' ).blur();
					MSF.move( form );
					return false;
				}
			}
		} );

		// next / prev button
		$( '.multistep_button' ).on( 'click', MSF.multistepButtonEvent );
	};

	this.multistepButtonEvent = function ( e ) {

		e.preventDefault();

		var id = $( this ).data( 'form' );
		if ( typeof id === 'undefined' || !id || id.length === 0 ) {
			var form = $( this ).closest( 'form.greyd_form' );
			var id = form.attr( "id" );
		} else {
			var form = $( 'form#' + id );
		}

		// next
		if ( $( this ).hasClass( 'next_step' ) ) {
			if ( MSF.requiredInputs[ id ].length === 0 ) {
				MSF.move( form );
			} else {
				form.find( 'button[type="submit"]' ).trigger( "click" );
			}

			// prev
		} else if ( $( this ).hasClass( 'prev_step' ) ) {
			MSF.move( form, -1 );

			// exact
		} else if ( $( this ).hasClass( 'exact_step' ) ) {

			var step = $( this ).data( 'step' );
			var currentStep = form.find( '.multistep_wrapper' ).data( 'step' );
			var title = form.data( 'title' );

			// step is an ID
			if ( isNaN( step ) ) {
				var stepObj = form.find( step );
				step = stepObj.length ? stepObj.index() : -1;
			} else {
				step -= 1;
			}

			// return if moving forward and not all inputs filled out
			if ( currentStep < step && MSF.requiredInputs[ id ].length > 0 ) {
				form.find( 'button[type="submit"]' ).trigger( "click" );
				return false;
			}

			if ( step === -1 ) {
				MSF.move( form );
			} else {
				window.location.hash = title + '_step' + step;
			}
		}

		return false;
	};

	// initial setup of multistep form
	this.setup = function ( form ) {
		if ( typeof form === undefined || typeof mode === undefined ) return;

		// vars
		var title = form.data( 'title' );
		var wrapper = form.find( '.multistep_wrapper' );

		let section = '';
		let lastSection = '';

		/**
		 * @deprecated VC form
		 */
		if ( form.hasClass( 'multistep' ) ) {
			section = wrapper.find( 'section.vc_section:first-of-type' );
			lastSection = wrapper.find( 'section.vc_section:last-of-type' );
		} else {
			section = wrapper.find( '.step:first-of-type' );
			lastSection = wrapper.find( '.step:last-of-type' );
		}

		var progress = form.parent().find( '.progress_wrapper' ).children();

		// set active section
		section.addClass( 'active' );

		// init height & anim
		setTimeout( function () {
			form.addClass( 'anim_init' );
		}, 0 );

		// set progress bar
		if ( progress.hasClass( 'multistep_progress' ) ) {
			var bar = progress.children();
			if ( bar.length > 0 ) {
				var x = ( section.index() + .5 ) / ( lastSection.index() + 1 );
				bar.css( { "width": x * 100 + "%" } );
			}

			// set pagination
		} else if ( progress.hasClass( 'multistep_pagination' ) ) {
			progress.children().first().addClass( 'active' );
		}

		// setup necessary inputs
		MSF.inputs( section, form );

		// bind to event
		// event is triggered when height of condition is changed
		form.on( 'heightChange', function ( event, elem, diff ) {
			diff = diff.length === 0 ? 0 : diff;

			//if (greydFormSettings.wp_debug) console.log('form-height changed by: '+diff);
			if ( typeof elem !== undefined && diff !== 0 ) {

				// if changed condition is inside current wrapper
				// --> resize wrapper
				if ( elem.closest( '.vc_section' ).hasClass( 'active' ) ) {
					wrapper.css( { "height": wrapper.height() + diff } );
				}
			}
		} );
	};

	// move in direction
	this.move = function ( form, direction ) {
		if ( typeof form === undefined ) return;

		direction = typeof direction === 'undefined' || direction.length === 0 ? 1 : direction;

		var title = form.data( 'title' );
		var wrapper = form.find( '.multistep_wrapper' );

		var newStep = wrapper.data( 'step' ) + direction;

		if (
			wrapper.find( 'section.vc_section:nth-of-type(' + ( newStep + 1 ) + ')' ).length
			|| wrapper.find( '.step:nth-of-type(' + ( newStep + 1 ) + ')' ).length
		) {
			let newHash = title + '_step' + newStep;
			if ( window.location.hash === '#' + newHash ) {
				// go to step
				MSF.step( $( 'form.greyd_form[data-title="' + title + '"]' ), newStep );
			} else {
				window.location.hash = newHash;
			}
		}

		$( '.multistep_button' ).off().on( 'click', MSF.multistepButtonEvent );
	};

	// go to step
	this.step = function ( form, i, speed ) {
		if ( typeof form === undefined ) return;

		i = typeof i === 'undefined' || i.length === 0 ? 0 : i;
		speed = typeof speed === 'undefined' || speed.length === 0 ? 300 : speed;


		var wrapper = form.find( '.multistep_wrapper' );

		/**
		 * @deprecated VC form
		 */
		if ( form.hasClass( 'multistep' ) ) {
			var section = wrapper.find( 'section.vc_section:nth-of-type(' + ( i + 1 ) + ')' );
			var lastSection = wrapper.find( 'section.vc_section:last-of-type' );
		} else {
			var section = wrapper.find( '.step:nth-of-type(' + ( i + 1 ) + ')' );
			var lastSection = wrapper.find( '.step:last-of-type' );
		}

		var progress = form.parent().find( '.progress_wrapper' ).children();

		if ( !section.length ) return;

		// set height first to enable clean transition
		wrapper.css( { "height": wrapper.height() } );

		// set active section
		wrapper.data( 'step', i );
		if ( form.hasClass( 'multistep' ) ) {
			wrapper.find( '.vc_section.active' ).removeClass( 'active' );
		} else {
			wrapper.find( '.step.active' ).removeClass( 'active' );
		}
		section.addClass( 'active' );

		// get active inputs
		MSF.inputs( section, form );

		// resize wrapper
		var height = 0;
		section.children().each( function () {
			if ( this.classList.contains( "greyd-styles" ) ) return;
			height = height + $( this ).outerHeight( true );
		} );
		wrapper.css( { "height": height } );

		// set progress bar
		if ( progress.hasClass( 'multistep_progress' ) ) {
			var bar = progress.children();
			if ( bar.length > 0 ) {
				var x = ( i + .5 ) / ( lastSection.index() + 1 );
				bar.css( { "width": x * 100 + "%" } );
			}
			// set pagination
		} else if ( progress.hasClass( 'multistep_pagination' ) ) {
			progress.find( "span:nth-child(n+" + ( i + 2 ) + ")" ).removeClass( 'active' );
			progress.find( "span:nth-child(-n+" + ( i + 1 ) + ")" ).addClass( 'active' );
		}

		// Animate to new step (swipe only)
		if ( form.data( 'transition' ) === 'swipe' || wrapper.data( 'transition' ) === 'swipe' ) MSF.animate( wrapper, i, speed );

		if ( greydFormSettings.wp_debug ) console.log( 'Navigated multistep to step: ' + i );
	};

	// get all input-names as array
	this.inputs = function ( section, form ) {
		if ( typeof section === undefined || typeof form === undefined ) return;

		var id = form.attr( 'id' );
		var isAutoStep = form.hasClass( 'auto_step' ) || section.parent().hasClass( 'auto_step' );


		MSF.requiredInputs[ id ] = [];

		section.find( 'input, select, textarea' ).each( function () {
			var isRequired = !!$( this ).attr( 'required' );
			var isEmpty = ( this.value && this.value !== '' && this.value !== undefined && this.value !== 'false' ) ? false : true;
			var isValid = $( this )[ 0 ].validity.valid;

			// add the field if required or using autostep
			if ( isAutoStep || isRequired ) {
				if ( isEmpty || !isValid ) {
					MSF.addInput( $( this ), id );
				}
			}
			// else if (!required && !empty && !valid) {
			//     MSF.addInput($(this), id);
			// }
		} );

		if ( greydFormSettings.wp_debug ) console.log( "Required inputs of the current step:", MSF.requiredInputs );
	};

	this.removeInput = function ( input, form_id ) {
		if ( typeof input === undefined || typeof form_id === undefined ) return;

		var name = input.attr( 'name' );
		var index = MSF.requiredInputs[ form_id ].indexOf( name );
		if ( index !== -1 ) {
			MSF.requiredInputs[ form_id ].splice( index, 1 );
			// if (greydFormSettings.wp_debug) console.log('removed input '+name);
		}
	};

	this.addInput = function ( input, form_id ) {
		if ( typeof input === undefined || typeof form_id === undefined ) return;

		var name = input.attr( 'name' );
		if ( MSF.requiredInputs[ form_id ].indexOf( name ) === -1 && input.attr( 'novalue' ) !== 'true' ) {
			MSF.requiredInputs[ form_id ].push( name );
			// if (greydFormSettings.wp_debug) console.log('added input '+name);
		}
	};

	// reset form
	this.reset = function ( form ) {
		if ( typeof form === undefined ) return;
		if ( !form.hasClass( 'multistep' ) && form.find( ".multistep_wrapper" ).length === 0 ) return;

		var step = 0;
		var title = form.data( 'title' );
		// var bar     = form.parent().find('.progress_wrapper').find('.multistep_progress, .multistep_pagination');

		window.location.hash = title + '_step' + step;
		// if (bar.length > 0) bar.hide();
	};

	this.animate = function ( wrapper, step, speed, style ) {
		if ( typeof wrapper === undefined ) return;

		step = typeof step === 'undefined' || step.length === 0 ? 0 : step;
		speed = typeof speed === 'undefined' || speed.length === 0 ? 300 : speed;
		style = typeof style === 'undefined' || style.length === 0 ? 'swipe' : style;

		if ( style === 'swipe' ) {
			wrapper.animate(
				{ now: step },
				{
					duration: speed,
					step: function ( now ) {
						$( this ).css( {
							"-webkit-transform": 'translateX(' + ( -now * 100 ) + '%)',
							"-ms-transform": 'translateX(' + ( -now * 100 ) + '%)',
							"transform": 'translateX(' + ( -now * 100 ) + '%)',
						} );
					}
				}
			);
		}
	};

};