<?php
/*
	Comments: Plugin helper
*/
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new VC_Helper();

class VC_Helper {

	/*
	------------------------------

			GET PREDEFINED LISTS
			for dropdowns & radios

	------------------------------
	*/
	public static function get_dropdown_array( $type, $pre = array(), $after = array() ) {
		if ( $type === 'countrycodes_all' || $type === 'countrynames_all' ) {
			$return = self::get_country_codes_all();
		} elseif ( $type === 'currencies' ) {
			$return = self::get_currencies();
		} elseif ( $type === 'currencies_all' ) {
			$return = self::get_currencies_all();
		} elseif ( $type === 'languages' ) {
			$return = self::get_languages();
		} elseif ( $type === 'languages_all' ) {
			$return = self::get_languages_all();
		} else {
			$return = self::get_country_codes();
		}
		$return = array_merge( (array) $pre, (array) $return, (array) $after );
		return $return;
	}

	public static function get_country_codes() {
		return array(
			'EG' => __( "Egypt" ),
			'AL' => __( "Albania" ),
			'DZ' => __( "Algeria" ),
			'AR' => __( "Argentina" ),
			'AM' => __( "Armenia" ),
			'AU' => __( "Australia" ),
			'BE' => __( "Belgium" ),
			'BO' => __( "Bolivia" ),
			'BA' => __( "Bosnia and Herzegovina" ),
			'BR' => __( "Brazil" ),
			'BG' => __( "Bulgaria" ),
			'CL' => __( 'Chile' ),
			'CN' => __( 'China' ),
			'DK' => __( "Denmark" ),
			'DE' => __( "Germany" ),
			'EE' => __( "Estonia" ),
			'FI' => __( "Finland" ),
			'FR' => __( "France" ),
			'GE' => __( "Georgia" ),
			'GR' => __( "Greece" ),
			'GB' => __( "United Kingdom" ),
			'IS' => __( "Iceland" ),
			'IN' => __( "India" ),
			'IE' => __( "Ireland" ),
			'IL' => __( 'Israel' ),
			'IT' => __( "Italy" ),
			'JP' => __( 'Japan' ),
			'CA' => __( "Canada" ),
			'CO' => __( "Colombia" ),
			'KR' => __( 'Korea' ),
			'HR' => __( "Croatia" ),
			'LV' => __( "Latvia" ),
			'LI' => __( 'Liechtenstein' ),
			'LT' => __( "Lithuania" ),
			'LU' => __( "Luxembourg" ),
			'MA' => __( "Morocco" ),
			'MX' => __( "Mexico" ),
			'MC' => __( 'Monaco' ),
			'ME' => __( 'Montenegro' ),
			'NZ' => __( "New Zealand" ),
			'NL' => __( "Netherlands" ),
			'NO' => __( "Norway" ),
			'AT' => __( "Austria" ),
			'PL' => __( "Poland" ),
			'PT' => __( 'Portugal' ),
			'RO' => __( "Romania" ),
			'RU' => __( "Russia" ),
			'SM' => __( 'San Marino' ),
			'SA' => __( "Saudi Arabia" ),
			'SE' => __( "Sweden" ),
			'CH' => __( "Switzerland" ),
			'RS' => __( "Serbia" ),
			'SK' => __( "Slovakia" ),
			'SI' => __( "Slovenia" ),
			'ES' => __( "Spain" ),
			'ZA' => __( "South Africa" ),
			'CZ' => __( "Czech Republic" ),
			'TN' => __( "Tunisia" ),
			'TR' => __( "Turkey" ),
			'UA' => __( 'Ukraine' ),
			'HU' => __( "Hungary" ),
			'US' => __( "United States" ),
		);
	}

	public static function get_country_codes_all() {
		return array(
			'AF' => __( 'Afghanistan' ),
			'AX' => __( 'Åland' ),
			'AL' => __( "Albania" ),
			'DZ' => __( "Algeria" ),
			'AS' => __( 'Amerikanisch-Samoa' ),
			'AD' => __( 'Andorra' ),
			'AO' => __( 'Angola' ),
			'AI' => __( 'Anguilla' ),
			'AQ' => __( 'Antarktis' ),
			'AG' => __( 'Antigua und Barbuda' ),
			'AR' => __( "Argentina" ),
			'AM' => __( "Armenia" ),
			'AW' => __( 'Aruba' ),
			'AU' => __( "Australia" ),
			'AT' => __( "Austria" ),
			'AZ' => __( 'Aserbaidschan' ),
			'BS' => __( 'Bahamas' ),
			'BH' => __( 'Bahrain' ),
			'BD' => __( 'Bangladesch' ),
			'BB' => __( 'Barbados' ),
			'BY' => __( 'Weißrussland' ),
			'BE' => __( "Belgium" ),
			'BZ' => __( 'Belize' ),
			'BJ' => __( 'Benin' ),
			'BM' => __( 'Bermuda' ),
			'BT' => __( 'Bhutan' ),
			'BO' => __( "Bolivia" ),
			'BQ' => __( 'Bonaire, Sint Eustatius und Saba' ),
			'BA' => __( 'Bosnien und Herzegowina' ),
			'BW' => __( 'Botswana' ),
			'BV' => __( 'Bouvetinsel' ),
			'BR' => __( 'Brasilien' ),
			'IO' => __( 'Britisches Territorium im Indischen Ozean' ),
			'VG' => __( 'Britische Jungferninseln' ),
			'BN' => __( 'Brunei' ),
			'BG' => __( "Bulgaria" ),
			'BF' => __( 'Burkina Faso' ),
			'BI' => __( 'Burundi' ),
			'KH' => __( 'Kambodscha' ),
			'CM' => __( 'Kamerun' ),
			'CA' => __( "Canada" ),
			'CV' => __( 'Kap Verde' ),
			'KY' => __( 'Kaimaninseln' ),
			'CF' => __( 'Zentralafrikanische Republik' ),
			'TD' => __( 'Tschad' ),
			'CL' => __( 'Chile' ),
			'CN' => __( 'China' ),
			'CX' => __( 'Weihnachtsinsel' ),
			'CC' => __( 'Kokosinseln' ),
			'CO' => __( "Colombia" ),
			'KM' => __( 'Komoren' ),
			'CK' => __( 'Cookinseln' ),
			'CR' => __( 'Costa Rica' ),
			'HR' => __( "Croatia" ),
			'CU' => __( 'Kuba' ),
			'CW' => __( 'Curacao' ),
			'CY' => __( 'Zypern' ),
			'CZ' => __( 'Tschechische Republik' ),
			'CD' => __( 'Demokratische Republik Kongo' ),
			'DK' => __( "Denmark" ),
			'DJ' => __( 'Dschibuti' ),
			'DM' => __( 'Dominica' ),
			'DO' => __( 'Dominikanische Republik' ),
			'TL' => __( 'Osttimor' ),
			'EC' => __( 'Ecuador' ),
			'EG' => __( "Egypt" ),
			'SV' => __( 'El Salvador' ),
			'GQ' => __( 'Äquatorialguinea' ),
			'ER' => __( 'Eritrea' ),
			'EE' => __( "Estonia" ),
			'ET' => __( 'Äthiopien' ),
			'FK' => __( 'Falkland-Inseln' ),
			'FO' => __( 'Färöer-Inseln' ),
			'FJ' => __( 'Fidschi' ),
			'FI' => __( "Finland" ),
			'FR' => __( "France" ),
			'GF' => __( 'Französisch-Guayana' ),
			'PF' => __( 'Französisch-Polynesien' ),
			'TF' => __( 'Französische Süd- und Antarktisgebiete' ),
			'GA' => __( 'Gabun' ),
			'GM' => __( 'Gambia' ),
			'GE' => __( "Georgia" ),
			'DE' => __( "Germany" ),
			'GH' => __( 'Ghana' ),
			'GI' => __( 'Gibraltar' ),
			'GR' => __( "Greece" ),
			'GL' => __( 'Grönland' ),
			'GD' => __( 'Grenada' ),
			'GP' => __( 'Guadeloupe' ),
			'GU' => __( 'Guam' ),
			'GT' => __( 'Guatemala' ),
			'GG' => __( 'Guernsey' ),
			'GN' => __( 'Guinea' ),
			'GW' => __( 'Guinea-Bissau' ),
			'GY' => __( 'Guyana' ),
			'HT' => __( 'Haiti' ),
			'HM' => __( 'Heard und McDonaldinseln' ),
			'HN' => __( 'Honduras' ),
			'HK' => __( 'Hongkong' ),
			'HU' => __( "Hungary" ),
			'IS' => __( "Iceland" ),
			'IN' => __( "India" ),
			'ID' => __( 'Indonesien' ),
			'IR' => __( 'Iran' ),
			'IQ' => __( 'Irak' ),
			'IE' => __( "Ireland" ),
			'IM' => __( 'Isle of Man' ),
			'IL' => __( 'Israel' ),
			'IT' => __( "Italy" ),
			'CI' => __( 'Elfenbeinküste' ),
			'JM' => __( 'Jamaika' ),
			'JP' => __( 'Japan' ),
			'JE' => __( 'Jersey' ),
			'JO' => __( 'Jordanien' ),
			'KZ' => __( 'Kasachstan' ),
			'KE' => __( 'Kenia' ),
			'KI' => __( 'Kiribati' ),
			'XK' => __( 'Kosovo' ),
			'KW' => __( 'Kuwait' ),
			'KG' => __( 'Kirgisistan' ),
			'LA' => __( 'Laos' ),
			'LV' => __( "Latvia" ),
			'LB' => __( 'Libanon' ),
			'LS' => __( 'Lesotho' ),
			'LR' => __( 'Liberia' ),
			'LY' => __( 'Libyen' ),
			'LI' => __( 'Liechtenstein' ),
			'LT' => __( 'Litauen' ),
			'LU' => __( "Luxembourg" ),
			'MO' => __( 'Macao' ),
			'MK' => __( 'Mazedonien' ),
			'MG' => __( 'Madagaskar' ),
			'MW' => __( 'Malawi' ),
			'MY' => __( 'Malaysia' ),
			'MV' => __( 'Malediven' ),
			'ML' => __( 'Mali' ),
			'MT' => __( 'Malta' ),
			'MH' => __( 'Marshallinseln' ),
			'MQ' => __( 'Martinique' ),
			'MR' => __( 'Mauretanien' ),
			'MU' => __( 'Mauritius' ),
			'YT' => __( 'Mayotte' ),
			'MX' => __( "Mexico" ),
			'FM' => __( 'Mikronesien' ),
			'MD' => __( 'Moldawien' ),
			'MC' => __( 'Monaco' ),
			'MN' => __( 'Mongolei' ),
			'ME' => __( 'Montenegro' ),
			'MS' => __( 'Montserrat' ),
			'MA' => __( "Morocco" ),
			'MZ' => __( 'Mosambik' ),
			'MM' => __( 'Myanmar' ),
			'NA' => __( 'Namibia' ),
			'NR' => __( 'Nauru' ),
			'NP' => __( 'Nepal' ),
			'NL' => __( "Netherlands" ),
			'AN' => __( 'Niederländische Antillen' ),
			'NC' => __( 'Neukaledonien' ),
			'NZ' => __( "New Zealand" ),
			'NI' => __( 'Nicaragua' ),
			'NE' => __( 'Niger' ),
			'NG' => __( 'Nigeria' ),
			'NU' => __( 'Niue' ),
			'NF' => __( 'Norfolkinsel' ),
			'KP' => __( 'Nordkorea' ),
			'MP' => __( 'Nördliche Marianen' ),
			'NO' => __( "Norway" ),
			'OM' => __( 'Oman' ),
			'PK' => __( 'Pakistan' ),
			'PW' => __( 'Palau' ),
			'PS' => __( 'Palästinensische Autonomiegebiete' ),
			'PA' => __( 'Panama' ),
			'PG' => __( 'Papua-Neuguinea' ),
			'PY' => __( 'Paraguay' ),
			'PE' => __( 'Peru' ),
			'PH' => __( 'Philippinen' ),
			'PN' => __( 'Pitcairninseln' ),
			'PL' => __( "Poland" ),
			'PT' => __( 'Portugal' ),
			'PR' => __( 'Puerto Rico' ),
			'QA' => __( 'Katar' ),
			'CG' => __( 'Republik Kongo' ),
			'RE' => __( 'Réunion' ),
			'RO' => __( "Romania" ),
			'RU' => __( "Russia" ),
			'RW' => __( 'Ruanda' ),
			'BL' => __( 'Saint-Barthélemy' ),
			'SH' => __( 'St. Helena' ),
			'KN' => __( 'St. Kitts und Nevis' ),
			'LC' => __( 'St. Lucia' ),
			'MF' => __( 'St. Martin' ),
			'PM' => __( 'Saint-Pierre und Miquelon' ),
			'VC' => __( 'St. Vincent und die Grenadinen' ),
			'WS' => __( 'Samoa' ),
			'SM' => __( 'San Marino' ),
			'ST' => __( 'São Tomé und Príncipe' ),
			'SA' => __( 'Saudi-Arabien' ),
			'SN' => __( 'Senegal' ),
			'RS' => __( "Serbia" ),
			'CS' => __( 'Serbien und Montenegro' ),
			'SC' => __( 'Seychellen' ),
			'SL' => __( 'Sierra Leone' ),
			'SG' => __( 'Singapur' ),
			'SX' => __( 'Sint Maarten' ),
			'SK' => __( 'Slowakei' ),
			'SI' => __( 'Slowenien' ),
			'SB' => __( 'Salomon-Inseln' ),
			'SO' => __( 'Somalia' ),
			'ZA' => __( 'Südafrika' ),
			'GS' => __( 'Südgeorgien und die Südlichen Sandwichinseln' ),
			'KR' => __( 'Südkorea' ),
			'SS' => __( 'Südsudan' ),
			'ES' => __( "Spain" ),
			'LK' => __( 'Sri Lanka' ),
			'SD' => __( 'Sudan' ),
			'SR' => __( 'Suriname' ),
			'SJ' => __( 'Svalbard und Jan Mayen' ),
			'SZ' => __( 'Swasiland' ),
			'SE' => __( "Sweden" ),
			'CH' => __( "Switzerland" ),
			'SY' => __( 'Syrien' ),
			'TW' => __( 'Taiwan' ),
			'TJ' => __( 'Tadschikistan' ),
			'TZ' => __( 'Tansania' ),
			'TH' => __( 'Thailand' ),
			'TG' => __( 'Togo' ),
			'TK' => __( 'Tokelau' ),
			'TO' => __( 'Tonga' ),
			'TT' => __( 'Trinidad und Tobago' ),
			'TN' => __( "Tunisia" ),
			'TR' => __( "Turkey" ),
			'TM' => __( 'Turkmenistan' ),
			'TC' => __( 'Turks- und Caicosinseln' ),
			'TV' => __( 'Tuvalu' ),
			'VI' => __( 'Amerikanische Jungferninseln' ),
			'UG' => __( 'Uganda' ),
			'UA' => __( 'Ukraine' ),
			'AE' => __( 'Vereinte Arabische Emirate' ),
			'GB' => __( "United Kingdom" ),
			'US' => __( 'Vereinigte Staaten von Amerika (USA)' ),
			'UM' => __( 'United States Minor Outlying Islands' ),
			'UY' => __( 'Uruguay' ),
			'UZ' => __( 'Usbekistan' ),
			'VU' => __( 'Vanuatu' ),
			'VA' => __( 'Vatikanstadt' ),
			'VE' => __( 'Venezuela' ),
			'VN' => __( 'Vietnam' ),
			'WF' => __( 'Wallis und Futuna' ),
			'EH' => __( 'Westsahara' ),
			'YE' => __( 'Jemen' ),
			'ZM' => __( 'Sambia' ),
			'ZW' => __( 'Simbabwe' ),
		);
	}

	public static function get_currencies() {
		return array(
			'EUR' => __( '€ (Euro)' ),
			'USD' => __( '$ (US-Dollar)' ),
			'AUD' => __( '$ (Australischer Dollar)' ),
			'BRL' => __( 'Ré (Brasilianisches Real)' ),
			'CAD' => __( 'C$ (Kanadischer Dollar)' ),
			'CZK' => __( 'Kč (Tschechische Krone)' ),
			'DKK' => __( 'kr (Dänische Krone)' ),
			'HKD' => __( '$ (Hongkong-Dollar)' ),
			'HUF' => __( 'Ft (Ungarischer Forint)' ),
			'ILS' => __( '₪ (Israelischer Neuer Sheqel)' ),
			'INR' => __( '₹ (Indische Rupie)' ),
			'JPY' => __( '¥ (Japanischer Yen)' ),
			'MYR' => __( 'RM (Malaysischer Ringgit)' ),
			'MXN' => __( 'Mex$ (Mexikanischer Peso)' ),
			'NOK' => __( 'kr (Norwegische Krone)' ),
			'NZD' => __( '$ (Neuseeland-Dollar)' ),
			'PHP' => __( '₱ (Philippinischer Peso)' ),
			'PLN' => __( 'zł (Polnischer Zloty)' ),
			'GBP' => __( '£ (Britischer Pfund)' ),
			'SEK' => __( 'kr (Schwedische Krone)' ),
			'CHF' => __( 'Fr (Schweizer Franken)' ),
			'TWD' => __( '$ (Taiwan New Dollar)' ),
			'THB' => __( '฿ (Thailändischer Baht)' ),
			'TRY' => __( '₺ (Türkische Lira)' ),
		);
	}

	public static function get_currencies_all() {
		return array(
			'ALL' => __( 'Albania Lek' ),
			'AFN' => __( 'Afghanistan Afghani' ),
			'ARS' => __( 'Argentina Peso' ),
			'AWG' => __( 'Aruba Guilder' ),
			'AUD' => __( 'Australia Dollar' ),
			'AZN' => __( 'Azerbaijan New Manat' ),
			'BSD' => __( 'Bahamas Dollar' ),
			'BBD' => __( 'Barbados Dollar' ),
			'BDT' => __( 'Bangladeshi taka' ),
			'BYR' => __( 'Belarus Ruble' ),
			'BZD' => __( 'Belize Dollar' ),
			'BMD' => __( 'Bermuda Dollar' ),
			'BOB' => __( 'Bolivia Boliviano' ),
			'BAM' => __( 'Bosnia and Herzegovina Convertible Marka' ),
			'BWP' => __( 'Botswana Pula' ),
			'BGN' => __( 'Bulgaria Lev' ),
			'BRL' => __( 'Brazil Real' ),
			'BND' => __( 'Brunei Darussalam Dollar' ),
			'KHR' => __( 'Cambodia Riel' ),
			'CAD' => __( 'Canada Dollar' ),
			'KYD' => __( 'Cayman Islands Dollar' ),
			'CLP' => __( 'Chile Peso' ),
			'CNY' => __( 'China Yuan Renminbi' ),
			'COP' => __( 'Colombia Peso' ),
			'CRC' => __( 'Costa Rica Colon' ),
			'HRK' => __( 'Croatia Kuna' ),
			'CUP' => __( 'Cuba Peso' ),
			'CZK' => __( 'Czech Republic Koruna' ),
			'DKK' => __( 'Denmark Krone' ),
			'DOP' => __( 'Dominican Republic Peso' ),
			'XCD' => __( 'East Caribbean Dollar' ),
			'EGP' => __( 'Egypt Pound' ),
			'SVC' => __( 'El Salvador Colon' ),
			'EEK' => __( 'Estonia Kroon' ),
			'EUR' => __( 'Euro Member Countries' ),
			'FKP' => __( 'Falkland Islands (Malvinas) Pound' ),
			'FJD' => __( 'Fiji Dollar' ),
			'GHC' => __( 'Ghana Cedis' ),
			'GIP' => __( 'Gibraltar Pound' ),
			'GTQ' => __( 'Guatemala Quetzal' ),
			'GGP' => __( 'Guernsey Pound' ),
			'GYD' => __( 'Guyana Dollar' ),
			'HNL' => __( 'Honduras Lempira' ),
			'HKD' => __( 'Hong Kong Dollar' ),
			'HUF' => __( 'Hungary Forint' ),
			'ISK' => __( 'Iceland Krona' ),
			'INR' => __( 'India Rupee' ),
			'IDR' => __( 'Indonesia Rupiah' ),
			'IRR' => __( 'Iran Rial' ),
			'IMP' => __( 'Isle of Man Pound' ),
			'ILS' => __( 'Israel Shekel' ),
			'JMD' => __( 'Jamaica Dollar' ),
			'JPY' => __( 'Japan Yen' ),
			'JEP' => __( 'Jersey Pound' ),
			'KZT' => __( 'Kazakhstan Tenge' ),
			'KPW' => __( 'Korea (North) Won' ),
			'KRW' => __( 'Korea (South) Won' ),
			'KGS' => __( 'Kyrgyzstan Som' ),
			'LAK' => __( 'Laos Kip' ),
			'LVL' => __( 'Latvia Lat' ),
			'LBP' => __( 'Lebanon Pound' ),
			'LRD' => __( 'Liberia Dollar' ),
			'LTL' => __( 'Lithuania Litas' ),
			'MKD' => __( 'Macedonia Denar' ),
			'MYR' => __( 'Malaysia Ringgit' ),
			'MUR' => __( 'Mauritius Rupee' ),
			'MXN' => __( 'Mexico Peso' ),
			'MNT' => __( 'Mongolia Tughrik' ),
			'MZN' => __( 'Mozambique Metical' ),
			'NAD' => __( 'Namibia Dollar' ),
			'NPR' => __( 'Nepal Rupee' ),
			'ANG' => __( 'Netherlands Antilles Guilder' ),
			'NZD' => __( 'New Zealand Dollar' ),
			'NIO' => __( 'Nicaragua Cordoba' ),
			'NGN' => __( 'Nigeria Naira' ),
			'NOK' => __( 'Norway Krone' ),
			'OMR' => __( 'Oman Rial' ),
			'PKR' => __( 'Pakistan Rupee' ),
			'PAB' => __( 'Panama Balboa' ),
			'PYG' => __( 'Paraguay Guarani' ),
			'PEN' => __( 'Peru Nuevo Sol' ),
			'PHP' => __( 'Philippines Peso' ),
			'PLN' => __( 'Poland Zloty' ),
			'QAR' => __( 'Qatar Riyal' ),
			'RON' => __( 'Romania New Leu' ),
			'RUB' => __( 'Russia Ruble' ),
			'SHP' => __( 'Saint Helena Pound' ),
			'SAR' => __( 'Saudi Arabia Riyal' ),
			'RSD' => __( 'Serbia Dinar' ),
			'SCR' => __( 'Seychelles Rupee' ),
			'SGD' => __( 'Singapore Dollar' ),
			'SBD' => __( 'Solomon Islands Dollar' ),
			'SOS' => __( 'Somalia Shilling' ),
			'ZAR' => __( 'South Africa Rand' ),
			'LKR' => __( 'Sri Lanka Rupee' ),
			'SEK' => __( 'Sweden Krona' ),
			'CHF' => __( 'Switzerland Franc' ),
			'SRD' => __( 'Suriname Dollar' ),
			'SYP' => __( 'Syria Pound' ),
			'TWD' => __( 'Taiwan New Dollar' ),
			'THB' => __( 'Thailand Baht' ),
			'TTD' => __( 'Trinidad and Tobago Dollar' ),
			'TRY' => __( 'Turkey Lira' ),
			'TRL' => __( 'Turkey Lira' ),
			'TVD' => __( 'Tuvalu Dollar' ),
			'UAH' => __( 'Ukraine Hryvna' ),
			'GBP' => __( 'United Kingdom Pound' ),
			'USD' => __( 'United States Dollar' ),
			'UYU' => __( 'Uruguay Peso' ),
			'UZS' => __( 'Uzbekistan Som' ),
			'VEF' => __( 'Venezuela Bolivar' ),
			'VND' => __( 'Viet Nam Dong' ),
			'YER' => __( 'Yemen Rial' ),
			'ZWD' => __( 'Zimbabwe Dollar' ),
		);
	}

	public static function get_languages() {
		return array(
			'sq' => __( 'Albanisch' ),
			'ar' => __( 'Arabisch' ),
			'zh' => __( 'Chinesisch' ),
			'da' => __( 'Dänisch' ),
			'de' => __( 'Deutsch' ),
			'en' => __( 'Englisch' ),
			'fi' => __( 'Finnisch' ),
			'fr' => __( 'Französisch' ),
			'el' => __( 'Griechisch' ),
			'is' => __( 'Isländisch' ),
			'it' => __( 'Italienisch' ),
			'ja' => __( 'Japanisch' ),
			'ko' => __( 'Koreanisch' ),
			'hr' => __( 'Kroatisch' ),
			'lv' => __( 'Lettisch' ),
			'lt' => __( 'Litauisch' ),
			'nl' => __( 'Niederländisch' ),
			'no' => __( 'Norwegisch' ),
			'pl' => __( 'Polnisch' ),
			'pt' => __( 'Portugisisch' ),
			'rm' => __( 'Rumänisch' ),
			'ru' => __( 'Russisch' ),
			'sv' => __( 'Schwedisch' ),
			'sr' => __( 'Serbisch' ),
			'sk' => __( 'Slovakisch' ),
			'sl' => __( 'Slovenisch' ),
			'es' => __( 'Spanisch' ),
			'cs' => __( 'Tschechisch' ),
			'tr' => __( 'Türkisch' ),
			'uk' => __( 'Ukrainisch' ),
			'hu' => __( 'Ungarisch' ),
		);
	}

	public static function get_languages_all() {
		return array(
			'ab' => __( 'Abkhazian' ),
			'aa' => __( 'Afar' ),
			'af' => __( 'Afrikaans' ),
			'ak' => __( 'Akan' ),
			'sq' => __( 'Albanian' ),
			'am' => __( 'Amharic' ),
			'ar' => __( 'Arabic' ),
			'an' => __( 'Aragonese' ),
			'hy' => __( 'Armenian' ),
			'as' => __( 'Assamese' ),
			'av' => __( 'Avaric' ),
			'ae' => __( 'Avestan' ),
			'ay' => __( 'Aymara' ),
			'az' => __( 'Azerbaijani' ),
			'bm' => __( 'Bambara' ),
			'ba' => __( 'Bashkir' ),
			'eu' => __( 'Basque' ),
			'be' => __( 'Belarusian' ),
			'bn' => __( 'Bengali' ),
			'bh' => __( 'Bihari languages' ),
			'bi' => __( 'Bislama' ),
			'bs' => __( 'Bosnian' ),
			'br' => __( 'Breton' ),
			'bg' => __( 'Bulgarian' ),
			'my' => __( 'Burmese' ),
			'ca' => __( 'Catalan, Valencian' ),
			'km' => __( 'Central Khmer' ),
			'ch' => __( 'Chamorro' ),
			'ce' => __( 'Chechen' ),
			'ny' => __( 'Chichewa, Chewa, Nyanja' ),
			'zh' => __( 'Chinese' ),
			'cu' => __( 'Church Slavonic, Old Bulgarian, Old Church Slavonic' ),
			'cv' => __( 'Chuvash' ),
			'kw' => __( 'Cornish' ),
			'co' => __( 'Corsican' ),
			'cr' => __( 'Cree' ),
			'hr' => __( 'Croatian' ),
			'cs' => __( 'Czech' ),
			'da' => __( 'Danish' ),
			'dv' => __( 'Divehi, Dhivehi, Maldivian' ),
			'nl' => __( 'Dutch, Flemish' ),
			'dz' => __( 'Dzongkha' ),
			'en' => __( 'English' ),
			'eo' => __( 'Esperanto' ),
			'et' => __( 'Estonian' ),
			'ee' => __( 'Ewe' ),
			'fo' => __( 'Faroese' ),
			'fj' => __( 'Fijian' ),
			'fi' => __( 'Finnish' ),
			'fr' => __( 'French' ),
			'ff' => __( 'Fulah' ),
			'gd' => __( 'Gaelic, Scottish Gaelic' ),
			'gl' => __( 'Galician' ),
			'lg' => __( 'Ganda' ),
			'ka' => __( 'Georgian' ),
			'de' => __( 'German' ),
			'ki' => __( 'Gikuyu, Kikuyu' ),
			'el' => __( 'Greek (Modern)' ),
			'kl' => __( 'Greenlandic, Kalaallisut' ),
			'gn' => __( 'Guarani' ),
			'gu' => __( 'Gujarati' ),
			'ht' => __( 'Haitian, Haitian Creole' ),
			'ha' => __( 'Hausa' ),
			'he' => __( 'Hebrew' ),
			'hz' => __( 'Herero' ),
			'hi' => __( 'Hindi' ),
			'ho' => __( 'Hiri Motu' ),
			'hu' => __( 'Hungarian' ),
			'is' => __( 'Icelandic' ),
			'io' => __( 'Ido' ),
			'ig' => __( 'Igbo' ),
			'id' => __( 'Indonesian' ),
			'ia' => __( 'Interlingua (International Auxiliary Language Association)' ),
			'ie' => __( 'Interlingue' ),
			'iu' => __( 'Inuktitut' ),
			'ik' => __( 'Inupiaq' ),
			'ga' => __( 'Irish' ),
			'it' => __( 'Italian' ),
			'ja' => __( 'Japanese' ),
			'jv' => __( 'Javanese' ),
			'kn' => __( 'Kannada' ),
			'kr' => __( 'Kanuri' ),
			'ks' => __( 'Kashmiri' ),
			'kk' => __( 'Kazakh' ),
			'rw' => __( 'Kinyarwanda' ),
			'kv' => __( 'Komi' ),
			'kg' => __( 'Kongo' ),
			'ko' => __( 'Korean' ),
			'kj' => __( 'Kwanyama, Kuanyama' ),
			'ku' => __( 'Kurdish' ),
			'ky' => __( 'Kyrgyz' ),
			'lo' => __( 'Lao' ),
			'la' => __( 'Latin' ),
			'lv' => __( 'Latvian' ),
			'lb' => __( 'Letzeburgesch, Luxembourgish' ),
			'li' => __( 'Limburgish, Limburgan, Limburger' ),
			'ln' => __( 'Lingala' ),
			'lt' => __( 'Lithuanian' ),
			'lu' => __( 'Luba-Katanga' ),
			'mk' => __( 'Macedonian' ),
			'mg' => __( 'Malagasy' ),
			'ms' => __( 'Malay' ),
			'ml' => __( 'Malayalam' ),
			'mt' => __( 'Maltese' ),
			'gv' => __( 'Manx' ),
			'mi' => __( 'Maori' ),
			'mr' => __( 'Marathi' ),
			'mh' => __( 'Marshallese' ),
			'ro' => __( 'Moldovan, Moldavian, Romanian' ),
			'mn' => __( 'Mongolian' ),
			'na' => __( 'Nauru' ),
			'nv' => __( 'Navajo, Navaho' ),
			'nd' => __( 'Northern Ndebele' ),
			'ng' => __( 'Ndonga' ),
			'ne' => __( 'Nepali' ),
			'se' => __( 'Northern Sami' ),
			'no' => __( 'Norwegian' ),
			'nb' => __( 'Norwegian Bokmål' ),
			'nn' => __( 'Norwegian Nynorsk' ),
			'ii' => __( 'Nuosu, Sichuan Yi' ),
			'oc' => __( 'Occitan (post 1500)' ),
			'oj' => __( 'Ojibwa' ),
			'or' => __( 'Oriya' ),
			'om' => __( 'Oromo' ),
			'os' => __( 'Ossetian, Ossetic' ),
			'pi' => __( 'Pali' ),
			'pa' => __( 'Panjabi, Punjabi' ),
			'ps' => __( 'Pashto, Pushto' ),
			'fa' => __( 'Persian' ),
			'pl' => __( 'Polish' ),
			'pt' => __( 'Portuguese' ),
			'qu' => __( 'Quechua' ),
			'rm' => __( 'Romansh' ),
			'rn' => __( 'Rundi' ),
			'ru' => __( 'Russian' ),
			'sm' => __( 'Samoan' ),
			'sg' => __( 'Sango' ),
			'sa' => __( 'Sanskrit' ),
			'sc' => __( 'Sardinian' ),
			'sr' => __( 'Serbian' ),
			'sn' => __( 'Shona' ),
			'sd' => __( 'Sindhi' ),
			'si' => __( 'Sinhala, Sinhalese' ),
			'sk' => __( 'Slovak' ),
			'sl' => __( 'Slovenian' ),
			'so' => __( 'Somali' ),
			'st' => __( 'Sotho, Southern' ),
			'nr' => __( 'South Ndebele' ),
			'es' => __( 'Spanish, Castilian' ),
			'su' => __( 'Sundanese' ),
			'sw' => __( 'Swahili' ),
			'ss' => __( 'Swati' ),
			'sv' => __( 'Swedish' ),
			'tl' => __( 'Tagalog' ),
			'ty' => __( 'Tahitian' ),
			'tg' => __( 'Tajik' ),
			'ta' => __( 'Tamil' ),
			'tt' => __( 'Tatar' ),
			'te' => __( 'Telugu' ),
			'th' => __( 'Thai' ),
			'bo' => __( 'Tibetan' ),
			'ti' => __( 'Tigrinya' ),
			'to' => __( 'Tonga (Tonga Islands)' ),
			'ts' => __( 'Tsonga' ),
			'tn' => __( 'Tswana' ),
			'tr' => __( 'Turkish' ),
			'tk' => __( 'Turkmen' ),
			'tw' => __( 'Twi' ),
			'ug' => __( 'Uighur, Uyghur' ),
			'uk' => __( 'Ukrainian' ),
			'ur' => __( 'Urdu' ),
			'uz' => __( 'Uzbek' ),
			've' => __( 'Venda' ),
			'vi' => __( 'Vietnamese' ),
			'vo' => __( 'Volap_k' ),
			'wa' => __( 'Walloon' ),
			'cy' => __( 'Welsh' ),
			'fy' => __( 'Western Frisian' ),
			'wo' => __( 'Wolof' ),
			'xh' => __( 'Xhosa' ),
			'yi' => __( 'Yiddish' ),
			'yo' => __( 'Yoruba' ),
			'za' => __( 'Zhuang, Chuang' ),
			'zu' => __( 'Zulu' ),
		);
	}

	/*
	------------------------------

			DEPRECATED

	------------------------------
	*/

	/**
	 * Get all input names of a form
	 *
	 * @deprecated 1.0
	 * @param string $post_content
	 * @return array
	 */
	public static function get_all_input_names( $post_content ) {

		$all_names  = array();
		$all_inputs = self::get_all_between( $post_content, '[vc_form_', ']' );
		$excluded   = array_flip( array( 'button', 'button_steps', 'icon_panel', 'condition', 'conditional' ) );

		foreach ( (array) $all_inputs as $input ) {
			// get name & type
			$type = self::get_input_type( $input );
			$name = self::get_input_name( $input, $type );

			if ( $name && ! empty( $name ) && ! isset( $excluded[ $type ] ) ) {
				$all_names[ $name ] = $name;
			}
		}
		return $all_names;
	}

	/**
	 * Get all inputs of a form
	 * including types, restrictions, filetype, list...
	 *
	 * @deprecated 1.0
	 * @param string $post_content
	 * @return array
	 */
	public static function get_all_inputs( $post_content ) {

		$inputs = array();
		// Get all inputs
		$all_inputs = self::get_all_between( $post_content, '[vc_form_', ']' );

		foreach ( $all_inputs as $input ) {
			// get name & type
			$type = self::get_input_type( $input );
			$name = self::get_input_name( $input, $type );

			if ( $type == 'dropdown' || $type == 'radio' ) {

				$inputtype      = self::get_string_between( $input, 'inputtype="', '"' );
				$options_string = self::get_string_between( $input, 'options="', '"' );
				$list           = self::get_string_between( $input, 'predefined_lists="', '"' );
				$options        = self::build_options_array( $inputtype, $options_string, $list );
				// $options = [ 'key' => 'val' ];
				$inputs[ $name ] = array(
					'type'    => $type,
					'options' => $options,
				);

			} elseif ( $type == 'upload' ) {

				$size       = self::get_string_between( $input, ' max="', '"' );
				$filetype   = self::get_string_between( $input, ' filetype="', '"' );
				$customtype = $filetype == 'other' ? self::get_string_between( $input, ' custom_filetype="', '"' ) : '';

				$inputs[ $name ] = array(
					'type'         => $type,
					'restrictions' => array(
						'size'     => $size,
						'filetype' => $filetype,
						'custom'   => $customtype,
					),
				);

			} elseif ( $type == 'range' ) {

				$min = VC_Helper::str2int( self::get_string_between( $input, ' min="', '"' ) );
				$max = VC_Helper::str2int( self::get_string_between( $input, ' max="', '"' ) );

				$inputs[ $name ] = array(
					'type'         => $type,
					'restrictions' => array(
						'min' => empty( $min ) ? 0 : $min,
						'max' => empty( $max ) ? 100 : $max,
					),
				);

			} elseif ( $type == 'checkbox' ) {

				$is_agb = self::get_string_between( $input, ' is_agb="', '"' ) === 'agb' ? true : false;
				$text   = '';

				// get text
				if ( $is_agb === true ) {
					$text = strip_tags( get_option( 'agb_text', __( "I hereby accept the terms and conditions." ) ) ); // get global agb text
				} else {
					// get text of checkbox differently, as it's a content and therefore between brackets
					$all_checkboxes = self::get_all_between( $post_content, '[vc_form_checkbox', '/vc_form_checkbox]' );
					foreach ( $all_checkboxes as $_i => $_input ) {
						$_name = self::get_string_between( $_input, ' name="', '"' );
						// get checkbox text
						if ( $_name === $name ) {
							$text = strip_tags( self::get_string_between( $_input, ']', '[' ) );
						}
					}
				}
				$inputs[ $name ] = array(
					'type' => $type,
					'text' => $text,
					'agb'  => $is_agb,
				);

			} elseif ( $type !== 'button' && $type !== 'button_steps' && $type !== 'icon_panel' && $type !== 'condition' && $type !== 'conditional' ) {
				$inputs[ $name ] = array(
					'type' => $type,
				);
				// MAXLENGTH CHECK
				$maxlength = self::get_string_between( $input, ' maxlength="', '"' );
				// apply filter to modify valid maxlength
				$maxlength = apply_filters( 'formhelper_modify_maxlength', $maxlength, $name, null );
				// check if maxlength is not greater than absolut max
				$maxmax = (int) esc_attr( get_option( 'maxlength_inputs', 1000 ) );
				if ( ! empty( $maxlength ) && (int) $maxlength < $maxmax ) {
					$inputs[ $name ]['maxlength'] = (int) $maxlength;
				} else {
					$inputs[ $name ]['maxlength'] = $maxmax;
				}

				// MINLENGTH CHECK
				$minlength = self::get_string_between( $input, ' minlength="', '"' );
				// apply filter to modify valid minlength
				$minlength = apply_filters( 'formhelper_modify_minlength', $minlength, $name, null );
				if ( ! empty( $minlength ) ) {
					$inputs[ $name ]['minlength'] = $minlength;
				}
			}
		}
		return $inputs;
	}

	/**
	 * Get names of all required inputs in a form
	 *
	 * @deprecated 1.0
	 * @param string $post_content
	 */
	public static function get_all_required_fields( $post_content ) {

		$requirements = array();
		$all_inputs   = self::get_all_between( $post_content, '[vc_form_', ']' );

		foreach ( $all_inputs as $key => $val ) {
			if ( strpos( $val, 'required="true"' ) !== false || strpos( $val, 'captcha' ) !== false ) {
				$type = self::get_input_type( $val );
				$name = self::get_input_name( $val, $type );
				array_push( $requirements, $name );
			}
		}
		// Get all conditions
		$all_conditions = self::get_all_between( $post_content, '[vc_form_conditional', '[/vc_form_conditional]' );
		foreach ( $all_conditions as $key => $val ) {
			// get all inputs within
			$condition_inputs = self::get_all_between( $val, '[vc_form_', ']' );
			foreach ( $condition_inputs as $key => $val ) {
				if ( strpos( $val, 'required="true"' ) !== false || strpos( $val, 'captcha' ) !== false ) {
					// unset required inputs within condition
					$type = self::get_input_type( $val );
					$name = self::get_input_name( $val, $type );
					if ( ( $key = array_search( $name, $requirements ) ) !== false ) {
						unset( $requirements[ $key ] );
					}
				}
			}
		}
		return $requirements;
	}

	/**
	 * Searches a string for all occurences between 2 substrings
	 *
	 * @return array
	 */
	public static function get_all_between( $content, $start, $end ) {

		$all_between = array();

		while ( strpos( $content, $start ) !== false ) {
			// get first occurence between start and end
			$output = self::get_string_between( $content, $start, $end );
			// push it to array
			array_push( $all_between, $output );
			// get its position
			$pos = strpos( $content, $start ) + strlen( $start ) + strlen( $output );
			// shorten string until substring end
			$content = substr( $content, $pos );
		}

		return $all_between;
	}

	/**
	 * Get a string between 2 substrings
	 *
	 * @param string $content   Content to query.
	 * @param string $start     Start of the string.
	 * @param string $end       End of the string.
	 *
	 * @return string
	 */
	public static function get_string_between( $content, $start, $end ) {
		$r = explode( $start, $content );
		if ( isset( $r[1] ) ) {
			$r = explode( $end, $r[1] );
			return $r[0];
		}
		return '';
	}

	/**
	 * converts string to number
	 */
	public static function str2int( $str = '' ) {
		// $return = floatval(str_replace(',', '.', str_replace('.', '', $str))); // delete points
		$return = floatval( str_replace( ',', '.', $str ) );
		return $return;
	}

	/**
	 * Get the type from a stringified input
	 *
	 * @deprecated 1.0
	 */
	public static function get_input_type( $input ) {

		$type = strtok( $input, ' ' );
		if ( $type == 'input' ) {
			$type = self::get_string_between( $input, ' type="', '"' );
			if ( empty( $type ) ) {
				$type = 'text';
			}
		}
		return $type;
	}

	/**
	 * Evaluates & returns name from stringified input
	 *
	 * @deprecated 1.0
	 */
	public static function get_input_name( $input, $type ) {

		$name = $type == 'captcha' ? 'reCAPTCHA' : str_replace( ' ', '_', self::get_string_between( $input, ' name="', '"' ) );
		if ( empty( $name ) ) {
			$label = self::get_string_between( $input, ' label="', '"' );
			if ( ! empty( $label ) ) {
				$name = $type . '_' . str_replace( ' ', '_', strtolower( $label ) );
			} else {
				$placeholder = self::get_string_between( $input, ' placeholder="', '"' );
				if ( ! empty( $placeholder ) ) {
					$name = $type . '_' . str_replace( ' ', '_', strtolower( $placeholder ) );
				} else {
					$options = self::get_string_between( $input, ' options="', '"' );
					if ( ! empty( $options ) ) {
						$name = $type . '_' . str_replace( ' ', '_', strtolower( $options ) );
					}
				}
			}
		}
		if ( ! empty( $name ) ) {
			return Helper::make_input_name( $name, false );
		}
	}

	/**
	 * @since 0.7.7.8 (Theme Version)
	 * @deprecated 1.0
	 */
	public static function get_vc_custom_css( $content ) {
		if ( method_exists( '\vc\helper', 'get_vc_custom_css' ) ) {
			return \vc\helper::get_vc_custom_css( $content );
		}
		$return    = '';
		$regex_css = '/css="(.+?)"/';
		preg_match_all( $regex_css, $content, $css );
		if ( is_array( $css ) && isset( $css[1] ) && is_array( $css[1] ) ) {
			foreach ( $css[1] as $key ) {
				$return .= $key;
			}
		}
		return $return;
	}

	/**
	 * build options from vc input attributes
	 *
	 * @deprecated 1.0
	 */
	public static function build_options_array( $type = '', $input = '', $list = '' ) {
		$return = array();

		// options as textarea
		if ( $type === 'list' ) {
			$options = explode( ',', $input );
			foreach ( (array) $options as $option ) {
				$split = explode( '=', trim( $option ) );
				$val   = $split[0];
				$show  = isset( $split[1] ) && ! empty( $split[1] ) ? $split[1] : $val;
				$val   = empty( $val ) ? $show : $val;
				if ( ! empty( $val ) && ! empty( $show ) ) {
					$return[ Helper::make_input_name( $val, false ) ] = $show;
				}
			}
		}
		// predefined list
		elseif ( $type === 'predefined' ) {
			$options = explode( ',', $input );
			if ( empty( $list ) ) {
				return array();
			}
			$predefined_options = self::get_dropdown_array( $list );

			// if checkbox "all" is set
			if ( isset( array_flip( $options )['all'] ) ) {
				$options = array_flip( (array) $predefined_options );
			}

			foreach ( (array) $options as $val ) {
				$show = $val;
				if ( $list === 'currencies' || $list === 'countrynames' || $list === 'countrynames_all' || $list === 'languages' || $list === 'languages_all' ) {
					$show = $predefined_options[ $val ];
				}
				if ( ! empty( $val ) && ! empty( $show ) ) {
					$return[ Helper::make_input_name( $val, false ) ] = $show;
				}
			}
		}
		// options as vc_params_group
		else {
			$options = function_exists( 'vc_param_group_parse_atts' ) ? vc_param_group_parse_atts( $input ) : self::get_vc_input_options( $input );
			foreach ( (array) $options as $option ) {
				$val  = isset( $option['value'] ) ? $option['value'] : '';
				$show = isset( $option['show'] ) ? $option['show'] : $val;
				$val  = empty( $val ) ? $show : $val;
				if ( ! empty( $val ) && ! empty( $show ) ) {
					$return[ Helper::make_input_name( $val, false ) ] = $show;
				}
			}
		}
		return $return;
	}

	/**
	 * Get selected options from 'vc_params_group'
	 *
	 * @deprecated 1.0
	 * @return array all checked values
	 */
	public static function get_vc_option_checked_values( $input ) {
		$return  = array();
		$options = function_exists( 'vc_param_group_parse_atts' ) ? vc_param_group_parse_atts( $input ) : self::get_vc_input_options( $input );
		foreach ( (array) $options as $option ) {
			$val  = isset( $option['value'] ) ? $option['value'] : '';
			$show = isset( $option['show'] ) ? $option['show'] : $val;
			$val  = empty( $val ) ? $show : $val;
			if ( isset( $option['selected'] ) && $option['selected'] === 'selected' ) {
				$return[] = Helper::make_input_name( $val, false );
			}
		}
		return $return;
	}

	/**
	 * get vc formatted options as array
	 *
	 * @deprecated 1.0
	 */
	public static function get_vc_input_options( $string ) {
		$options = json_decode( urldecode( $string ), true );
		foreach ( $options as $k => $v ) {
			$options[ $k ] = array_values( $v )[0];
		}
		return $options;
	}

	/**
	 * @deprecated 1.4
	 * @see \vc\helper::get_color()
	 */
	public static function get_color( $atts, $pre, $post = '', $default = '' ) {

		if ( method_exists( '\vc\helper', 'get_color' ) ) {
			return \vc\helper::get_color( $atts, $pre, $post, $default );
		}

		// debug($atts);
		if ( isset( $atts[ $pre . 'colorselect' . $post ] ) && $atts[ $pre . 'colorselect' . $post ] != '' ) {
			$color = isset( $atts[ $pre . 'colorselect' . $post ] ) ? esc_attr( $atts[ $pre . 'colorselect' . $post ] ) : '';
		} else {
			$color = isset( $atts[ $pre . 'color' . $post ] ) ? esc_attr( $atts[ $pre . 'color' . $post ] ) : $default;
			if ( $color == 'color_custom' ) {
				$color = isset( $atts[ $pre . 'ccolor' . $post ] ) ? esc_attr( $atts[ $pre . 'ccolor' . $post ] ) : '';
			}
		}
		if ( strpos( $color, 'color_' ) === 0 ) {
			$color = 'var(--' . str_replace( '_', '', $color ) . ')';
		}

		return $color;
	}

	/**
	 * @deprecated 1.4
	 * @see \vc\helper::make_custom_element()
	 */
	public static function make_custom_element( $atts, $pre, $css = '' ) {
		if ( method_exists( '\vc\helper', 'make_custom_element' ) ) {
			return \vc\helper::make_custom_element( $atts, $pre, $css );
		}
		return null;
	}

	/**
	 * @deprecated 1.4
	 * @see \vc\helper::make_custom_icon()
	 */
	public static function make_custom_icon( $atts, $pre, $content ) {
		if ( method_exists( '\vc\helper', 'make_custom_icon' ) ) {
			return \vc\helper::make_custom_icon( $atts, $pre, $content );
		}
		return null;
	}
}
