<?php
/**
 * Register shortcodes
 *
 * @deprecated since greyd_blocks
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Shortcode();
class Shortcode {

	/**
	 * Shortcode prefix
	 */
	public $pre = 'vc_form_';

	/**
	 * Hold the post-ID of the current form
	 */
	public static $post_id;

	/**
	 * Holds the frontend ID of the current form
	 */
	public static $form_id;

	/**
	 * Holds the meta info of the current form
	 */
	public static $post_meta = array();

	/**
	 * RegEx Rules for validation
	 */
	public static $patterns = array(
		'email' => "pattern=\"\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*\"",
		'url'   => "pattern=\"(?:http(s)?:\/\/)?[\w.-]+(?:\.[\w\.-]+)+[\w\-\._~:/?#[\]@!\$&'\(\)\*\+,;=.]+\"",
		'date'  => "pattern='[0-9]{4}-[0-9]{2}-[0-9]{2}'", // Fallback for unsupported browsers
		'phone' => "pattern='[(]?[+]?[\s0-9]{0,4}[(]?[0-9]{1,4}[)]?[-\s/0-9]{4,16}'",
	);

	/**
	 * Field names
	 */
	public static $field_names = array();

	/**
	 * Placeholder Wordings
	 */
	public static $placeholder = array();


	/**
	 * =================================================================
	 *                          SETUP
	 * =================================================================
	 */

	/**
	 * Construct the class
	 */
	public function __construct() {

		// init rendering
		add_action( 'after_setup_theme', array( $this, 'init' ) );

		if ( ! class_exists( 'Greyd\Forms\VC_Helper' ) ) {
			require_once __DIR__ . '/helper.php';
		}
	}

	/**
	 * Add actions & filter
	 */
	public function init() {

		if ( is_admin() ) {
			return;
		}

		// init wordings
		$this->init_wordings();

		/**
		 * define form module
		 */
		add_shortcode( 'vc_form', array( $this, 'render_form' ) );
		// add errors
		add_action( 'formshortcode_errors_before_form', array( $this, 'mail_errors' ), 10, 1 );

		/**
		 * define modules inside form
		 */
		add_shortcode( $this->pre . 'input', array( $this, 'render_input' ) );
		add_shortcode( $this->pre . 'button', array( $this, 'render_button' ) );
		add_shortcode( $this->pre . 'checkbox', array( $this, 'render_checkbox' ) );
		add_shortcode( $this->pre . 'dropdown', array( $this, 'render_dropdown' ) );
		add_shortcode( $this->pre . 'radio', array( $this, 'render_radio' ) );
		add_shortcode( $this->pre . 'info', array( $this, 'render_info' ) );
		add_shortcode( $this->pre . 'upload', array( $this, 'render_upload' ) );
		add_shortcode( $this->pre . 'captcha', array( $this, 'render_captcha' ) );
		add_shortcode( $this->pre . 'hidden', array( $this, 'render_hidden' ) );

		add_shortcode( $this->pre . 'icon_panels', array( $this, 'render_icon_panels' ) );
		add_shortcode( $this->pre . 'icon_panel', array( $this, 'render_icon_panel' ) );

		add_shortcode( $this->pre . 'conditional', array( $this, 'render_conditional_fields' ) );
		add_shortcode( $this->pre . 'condition', array( $this, 'render_condition' ) );

		add_shortcode( $this->pre . 'button_steps', array( $this, 'render_button_steps' ) );

		add_shortcode( $this->pre . 'range', array( $this, 'render_range' ) );
		add_shortcode( $this->pre . 'math', array( $this, 'render_math' ) );

		/**
		 * add dynamic form tags
		 */
		add_filter( 'formshortcode_raw_content', array( $this, 'convert_form_tags' ), 10, 2 );
	}

	/**
	 * Modify the form content and convert all dynamic form tags
	 *
	 * @param string $content   Post content of the form.
	 * @param object $post      The WP_Post object.
	 *
	 * @return string $content
	 */
	public function convert_form_tags( $content, $post ) {
		// debug($content);
		return Helper::build_form_tags( $content );
	}

	public function init_wordings() {

		$fields            = array(
			// defaults
			'input'       => __( "the input field", 'greyd_forms' ),
			'checkbox'    => __( "the checkbox", 'greyd_forms' ),
			'dropdown'    => __( "the dropdown", 'greyd_forms' ),
			'radio'       => __( "the radio buttons", 'greyd_forms' ),
			'upload'      => __( "the upload field", 'greyd_forms' ),
			'icon_panels' => __( "the image panels", 'greyd_forms' ),
			'range'       => __( "the slider", 'greyd_forms' ),
		);
		self::$field_names = $fields;

		$placeholder = array(
			'optional'             => __( '(optional)', 'greyd_forms' ),
			// defaults
			'placeholder'          => __( "enter here", 'greyd_forms' ),
			'placeholder_select'   => __( "please select", 'greyd_forms' ),
			'submit'               => __( "submit", 'greyd_forms' ),
			'required_after'       => __( "Mandatory", 'greyd_forms' ),
			'agb'                  => __( "I hereby accept the terms and conditions.", 'greyd_forms' ),
			'title'                => __( "Please fill out this field", 'greyd_forms' ),
			'title_select'         => __( "Please select an option", 'greyd_forms' ),
			'title_check'          => __( "Please mark the checkbox", 'greyd_forms' ),
			'title_file'           => _x( "upload a file", 'small', 'greyd_forms' ),
			'next'                 => __( "next", 'greyd_forms' ),
			'prev'                 => __( "back", 'greyd_forms' ),
			'step'                 => __( "to step %s", 'greyd_forms' ),
			'title_next'           => _x( "go to next step", 'small', 'greyd_forms' ),
			'title_prev'           => _x( "go to last step", 'small', 'greyd_forms' ),
			'title_step'           => _x( "go to step %s", 'small', 'greyd_forms' ),

			// alerts
			'no_form'              => __( "No form selected.", 'greyd_forms' ),
			'no_form_found'        => __( "No form with the ID „%s“ found.", 'greyd_forms' ),
			'trash'                => __( "The form was moved to trash.", 'greyd_forms' ),
			'no_name'              => __( "Please select a name for %s.", 'greyd_forms' ),
			'no_content'           => __( "**Please enter text**", 'greyd_forms' ),
			'no_option'            => __( "**no selection added**", 'greyd_forms' ),
			'no_options'           => __( "Please set options for %s.", 'greyd_forms' ),
			'no_template'          => __( "No templated with the name \"%s\" found", 'greyd_forms' ),
			'no_email'             => __( "Please set the field for the user’s email, otherwise the confirmation email cannot be sent.", 'greyd_forms' ),
			'no_email_verify'      => __( "Please set the field for the user’s email, otherwise the verification email cannot be sent.", 'greyd_forms' ),
			'no_recaptcha_sitekey' => __( "Please enter your Google reCAPTCHA site key via 'Forms > Settings', otherwise the reCAPTCHA cannot be executed.", 'greyd_forms' ),

			// tooltips
			'tt_email'             => __( "The email address should contain an '@'character and a valid domain with a period.", 'greyd_forms' ),
			'tt_url'               => __( "The URL should contain a domain name after a period.", 'greyd_forms' ),
			'tt_tel'               => __( "The phone number can only consist of numbers and the following characters: + ( ) - / ", 'greyd_forms' ),
			'tt_number'            => __( "The input must be a number.", 'greyd_forms' ),
			'tt_number_min'        => __( "The number must have a value of minimum %d.", 'greyd_forms' ),
			'tt_number_max'        => __( "The number must have a value of maximum %d.", 'greyd_forms' ),
			'tt_number_minmax'     => __( "The number must be between the values %1\$d and %2\$d.", 'greyd_forms' ),
			'tt_multiselect'       => __( "Multiselect possible.", 'greyd_forms' ),
			// file tooltips
			'tt_img'               => __( "The file must be an image.", 'greyd_forms' ),
			'tt_zip'               => __( "The file must be a ZIP archive.", 'greyd_forms' ),
			'tt_pdf'               => __( "The file must be a PDF.", 'greyd_forms' ),
			'tt_text'              => __( "The file must be a text file.", 'greyd_forms' ),
			'tt_video'             => __( "The file must have a valid video format.", 'greyd_forms' ),
			'tt_audio'             => __( "The file must have a valid audio format.", 'greyd_forms' ),
			'tt_other'             => __( "The file must have one of the following formats: '%s'.", 'greyd_forms' ),
			'tt_size'              => __( " It must not be bigger than %s MB.", 'greyd_forms' ),
			'tt_size_only'         => __( "The file must not be bigger than %s MB.", 'greyd_forms' ),
		);

		self::$placeholder = $placeholder;
	}


	/**
	 * =================================================================
	 *                          FORM MODULE
	 * =================================================================
	 */

	/**
	 * Form
	 *
	 * @shortcode 'vc_form'
	 */
	public static function render_form( $atts, $content = '' ) {

		// debug($atts);
		// debug(Helper::get_language_code());

		$post_id = isset( $atts['id'] ) ? intval( $atts['id'] ) : null;
		if ( ! $post_id || empty( $post_id ) ) {
			// return Helper::make_message(self::$placeholder["no_form"], 'info');
			return false;
		}

		/**
		 * Filter post id before inserting the form.
		 *
		 * @since 1.3.6
		 */
		$post_id = Helper::filter_post_id( $post_id, 'tp_forms' );

		// Attributes
		self::$post_id = $post_id;
		self::$form_id = uniqid();
		$post          = get_post( $post_id );
		if ( ! isset( $post ) ) {
			// return Helper::make_message(sprintf(self::$placeholder["no_form_found"], $post_id), 'info');
			return false;
		} elseif ( $post->post_status === 'trash' ) {
			// return Helper::make_message(self::$placeholder["trash"], 'info');
			return false;
		}
		$title  = urlencode( preg_replace( '/\s/', '', esc_html( get_the_title( $post_id ) ) ) );
		$nonce  = wp_create_nonce( 'handle_form_data_' . $post_id );
		$errors = array();

		// Styling Options
		$block_send     = get_post_meta( $post_id, 'block_send', true ) ? true : false;
		$required_class = 'requirement-style__' . ( $block_send ? 'none' : get_post_meta( $post_id, 'required_style', true ) );
		$required_class = empty( $required_class ) ? 'required' : $required_class;
		$hide_tooltips  = get_post_meta( $post_id, 'hide_tooltips', true );
		$ignore_gutter  = get_post_meta( $post_id, 'ignore_gutter', true );
		$gutter         = $ignore_gutter === 'ignore_gutter' ? '' : 'force_gutter';

		$form_mode          = get_post_meta( $post_id, 'form_mode', true );
		$registration_meta  = get_post_meta( $post_id, 'user_registration', true );
		$selected_role_name = isset( $registration_meta['role'] ) ? $registration_meta['role'] : null;

		// Multistep
		$multistep = get_post_meta( $post_id, 'multistep', true );
		if ( $multistep ) {
			$multistep_style      = get_post_meta( $post_id, 'multistep_style', true );
			$multistep_style      = empty( $multistep_style ) ? 'submit_step' : $multistep_style;
			$multistep_transition = get_post_meta( $post_id, 'multistep_transition', true );
			$multistep_transition = ! empty( $multistep_transition ) ? "data-transition='$multistep_transition'" : "data-transition='swipe'";
			$same_stepheight      = get_post_meta( $post_id, 'same_stepheight', true );
		} else {
			$multistep_style = $multistep_transition = $same_stepheight = '';
		}

		/**
		 * Set class meta options
		 */
		self::$post_meta = array(
			'hide_tooltips' => $hide_tooltips,
			'ignore_gutter' => $ignore_gutter,
			'multistep'     => $multistep,
			'block_send'    => $block_send,
			'form_mode'     => $form_mode,
		);

		// Get action after
		$after_form     = get_post_meta( $post_id, 'after_form', true );
		$popup_bg_color = get_post_meta( $post_id, 'popup_bg_color', true );

		// Get hidden fields
		$hidden_fields = (array) apply_filters( 'formshortcode_add_hidden_fields', array(), $post_id );

		// fix CSS
		$vc_css = VC_Helper::get_vc_custom_css( $post->post_content );

		// css classes
		$classes = trim(
			preg_replace(
				'/\s+/',
				' ',
				implode(
					' ',
					apply_filters(
						'greyd_forms_css_classes',
						array(
							'title'           => '„' . $title . '“',
							'required_class'  => $required_class,
							'hide_tooltips'   => $hide_tooltips,
							'is_multistep'    => $multistep,
							'multistep_style' => $multistep_style,
						),
						$post->post_content,
						$post_id
					)
				)
			)
		);

		// start render
		ob_start();

		// don't render the userdata update form if the user has no permission (role)
		if ( ! is_admin() && $form_mode === 'update_user' ) {
			$current_user = wp_get_current_user();
			if ( ! $current_user || ! in_array( $selected_role_name, (array) $current_user->roles ) ) {
				return;
			}
		}

		// Add error checks before form creation
		do_action( 'formshortcode_errors_before_form', $post_id );

		// render custom css
		if ( method_exists( '\enqueue', 'add_custom_style' ) ) {
			\enqueue::add_custom_style( $vc_css );
		} else {
			echo "<style>$vc_css</style>";
		}

		// Wrappers
		echo "<div class='forms_wrapper $ignore_gutter'>";
		echo "<form
			class='greyd_form $classes' name='form_$post_id' id='" . self::$form_id . "'
			data-id='$post_id' data-form-mode='{$form_mode}' data-title='$title' data-nonce='$nonce' data-send='" . ( $block_send ? 'prevent' : 'enable' ) . "'
			method='POST' enctype='multipart/form-data' $multistep_transition
		>";

		/**
		 * Render something inside a form (start).
		 *
		 * @param int $form_post_id     WP_Post ID of the form
		 */
		do_action( 'greyd_forms_render_form_start', $post_id );

		// Multistep (Wrapper & format Content)
		if ( $multistep ) {
			echo "<div class='multistep_wrapper $same_stepheight' data-step='0'>";

			// delete content not in sections
			$new_content    = '';
			$default_button = '[vc_row][vc_column][vc_form_button_steps alignment="right" icon="arrow_right"][/vc_column][/vc_row]';
			$all_steps      = VC_Helper::get_all_between( $post->post_content, '[vc_section', '[/vc_section]' );
			// debug($all_steps);
			if ( count( $all_steps ) > 0 ) {
				foreach ( $all_steps as $key => $val ) {

					// Insert Next-Buttons
					if ( $multistep_style === 'submit_step' ) {
						// only insert if not defined and not an last step
						if ( strpos( $val, '[vc_form_button_steps' ) === false && $key < ( count( $all_steps ) - 1 ) ) {
							$val .= $default_button;
						}
					}
					$new_content .= '[vc_section' . $val . '[/vc_section]';
				}
				$post->post_content = $new_content;
			}
		}

		// Content
		if ( has_blocks( $post->post_content ) ) {
			echo do_blocks( apply_filters( 'formblocks_raw_content', $post->post_content, $post ) );
		} else {
			echo do_shortcode( apply_filters( 'formshortcode_raw_content', $post->post_content, $post ) );
		}

		// End of inner Wrappers
		if ( $multistep ) {
			echo '</div>';
		}

		// add hidden fields
		if ( ! $block_send && count( $hidden_fields ) > 0 ) {
			foreach ( $hidden_fields as $name => $val ) {
				echo "<input type='hidden' name='$name' value='$val'></input>";
			}
			// debug($hidden_fields);
		}

		/**
		 * Render something inside a form (end).
		 *
		 * @param int $form_post_id     WP_Post ID of the form
		 */
		do_action( 'greyd_forms_render_form_end', $post_id );

		echo '</form>';

		// Required Text after
		if ( ! $block_send ) {
			if ( $required_class === 'required' || $required_class === 'both' ) {
				$required_text  = get_post_meta( $post_id, 'required_text', true );
				$required_align = get_post_meta( $post_id, 'required_align', true );
				$required_text  = ! empty( $required_text ) ? $required_text : self::$placeholder['required_after'];
				if ( $required_align !== 'hidden' ) {
					echo "<div class='flex-$required_align $gutter'><label class='requirement_after'><span class='requ_star'>* </span>" . $required_text . '</label></div>';
				}
			}
		}

		// Progress Bar (Multistep only))
		if ( $multistep ) {
			echo "<div class='$gutter'>";
			self::render_progress_bar( $post_id, $all_steps );
			echo '</div>';
		}

		if ( ! $block_send ) {

			// Message Container
			echo "<div class='after-message animate_fast $gutter'><div class='message'></div></div>";

			// Pop-up
			if ( $after_form == 'popup' || $after_form == 'fullscreen' ) {
				$popup = get_posts(
					array(
						'name'      => 'form_popup_' . $post_id,
						'post_type' => 'dynamic_template',
					)
				);
				if ( $popup ) {
					$popup = $popup[0];
					echo "<div class='after-popup $after_form' " . ( ! empty( $popup_bg_color ) ? "style='background-color:" . $popup_bg_color . ";'" : '' ) . '>' . (
						has_blocks( $popup->post_content ) ? do_blocks( $popup->post_content ) : do_shortcode( $popup->post_content ) ) .
					'</div>';
				} else {
					array_push( $errors, sprintf( self::$placeholder['no_template'], '<strong>form_popup_' . $post_id . '</strong>' ) );
				}
			}

			foreach ( $errors as $error ) {
				echo Helper::make_message( $error, 'danger' );
			}
		}

		// End of outer Wrapper
		echo '</div>';

		// Return
		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Check if mail settings are correct
	 */
	public function mail_errors( $post_id ) {

		if ( self::$post_meta['block_send'] ) {
			return;
		}

		$verify_user          = get_post_meta( $post_id, 'verify_user', true ) == 'verify_user' ? true : false;
		$verify_mail_to_field = get_post_meta( $post_id, 'verify_email', true );
		$user_gets_mail       = get_post_meta( $post_id, 'user_gets_mail', true );
		$mail_to_field        = get_post_meta( $post_id, 'user_email', true );

		// no verify email
		if ( $verify_user && Helper::is_select_empty( $verify_mail_to_field ) ) {
			echo Helper::make_message( self::$placeholder['no_email_verify'], 'danger' );
		}
		if ( $user_gets_mail && Helper::is_select_empty( $mail_to_field ) && Helper::is_select_empty( $verify_mail_to_field ) ) {
			echo Helper::make_message( self::$placeholder['no_email'], 'danger' );
		}

	}

	/**
	 * Render Progress Bar
	 */
	public static function render_progress_bar( $post_id, $all_steps ) {
		$progress_bar = get_post_meta( $post_id, 'progress_bar', true );
		if ( $progress_bar === 'progress' || $progress_bar === 'pagination' ) {

			$inline_style = '';

			// get height
			$progress_height = get_post_meta( $post_id, 'progress_height', true );
			$progress_height = ! empty( $progress_height ) ? 'height:' . Helper::force_units( $progress_height ) . ';' : '';
			// get radius
			$progress_radius = get_post_meta( $post_id, 'progress_radius', true );
			$progress_radius = ! empty( $progress_radius ) ? 'border-radius:' . Helper::force_units( $progress_radius ) . ';' : '';
			// active color
			$progress_color = get_post_meta( $post_id, 'progress_color', true );
			$progress_color = ! empty( $progress_color ) ? 'background-color:' . $progress_color . ';' : '';
			// bg color
			$progress_bar_color = get_post_meta( $post_id, 'progress_bar_color', true );
			$progress_bar_color = ! empty( $progress_bar_color ) ? 'background-color:' . $progress_bar_color . ' !important;' : '';

			// alignment wrapper
			$progress_align = get_post_meta( $post_id, 'progress_align', true );
			echo "<div class='progress_wrapper flex-$progress_align'>";

			// PAGINATION
			if ( $progress_bar === 'pagination' ) {
				// get width
				$progress_width = get_post_meta( $post_id, 'pagination_width', true );
				$progress_width = ! empty( $progress_width ) ? 'width:' . Helper::force_units( $progress_width ) . ';' : '';
				// get margin
				$progress_margin = get_post_meta( $post_id, 'progress_margin', true );
				$progress_margin = ! empty( $progress_margin ) ? 'margin:0 ' . Helper::force_units( $progress_margin ) . ';' : '';

				$progress_style = $progress_height . $progress_radius . $progress_color . $progress_width . $progress_margin;
				$inline_style   = empty( $progress_bar_color ) ? '' : 'form[name="form_' . $post_id . '"] ~ div .multistep_pagination > span.active {' . $progress_bar_color . '}';

				echo "<div class='multistep_$progress_bar'>";
				foreach ( (array) $all_steps as $key => $val ) {
					echo "<span style='$progress_style' class=''></span>";
				}
				echo '</div>';

				// PROGRESS BAR
			} elseif ( $progress_bar === 'progress' ) {
				// get width
				$progress_width = get_post_meta( $post_id, 'progress_width', true );
				$progress_width = ! empty( $progress_width ) ? 'width:' . $progress_width . ';' : '';

				$progress_style = $progress_height . $progress_radius . $progress_color . $progress_width;

				echo "<div class='multistep_$progress_bar' style='$progress_style'><span class='bar' style='$progress_bar_color'></span></div>";
			}

			// end of alignment wrapper
			echo '</div>';

			// render custom css
			if ( method_exists( '\enqueue', 'add_custom_style' ) ) {
				\enqueue::add_custom_style( $inline_style );
			} else {
				echo "<style>$inline_style</style>";
			}
		}
	}


	/**
	 * =================================================================
	 *                          INPUT MODULES
	 * =================================================================
	 */

	/**
	 * Input field
	 *
	 * @shortcode 'vc_form_input'
	 */
	public function render_input( $atts, $content ) {

		// debug($atts);

		// modify attributes
		$atts = apply_filters( 'formshortcode_modify_input_atts', $atts, self::$post_id );

		// Attributes
		$type         = ( isset( $atts['type'] ) && ! empty( $atts['type'] ) ) ? $atts['type'] : 'text';
		$label        = isset( $atts['label'] ) ? $atts['label'] : '';
		$placeholder  = isset( $atts['placeholder'] ) ? $atts['placeholder'] : self::$placeholder['placeholder'];
		$required     = isset( $atts['required'] ) ? 'required' : 'optional';
		$css_class    = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$autocomplete = ( isset( $atts['autocomplete'] ) && ! empty( $atts['autocomplete'] ) ) ? 'autocomplete="' . $atts['autocomplete'] . '"' : 'autocomplete="' . $type . '"';
		$title        = 'title="' . self::$placeholder['title'] . '"';

		// Attributes new
		$style     = isset( $atts['style'] ) ? $atts['style'] : '';
		$width     = isset( $atts['width'] ) ? 'width:' . Helper::force_units( $atts['width'], array( 'px', '%' ) ) . ';' : '';
		$alignment = isset( $atts['alignment'] ) ? $atts['alignment'] : 'left';
		// custom input
		$css = ''; // $width;
		if ( $style == 'custom' ) {
			$style = '';
			$css   = VC_Helper::make_custom_element( $atts, 'input', $css );
		}

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = $type . '_' . str_replace( ' ', '_', strtolower( $label ) );
			} elseif ( isset( $atts['placeholder'] ) ) {
				$name = $type . '_' . str_replace( ' ', '_', strtolower( $placeholder ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['input'] ), 'danger' );
			}
		}
		// convert names & set unique id
		$name = Helper::make_input_name( $name, false );
		$id   = Helper::make_input_name( $name );

		// Hilfetext & Beschränkungen
		$pattern    = isset( $atts['pattern'] ) ? $atts['pattern'] : '';
		$min        = $max = $steps = $unit = $tooltip = '';
		$startvalue = '';

		if ( $type == 'email' ) {
			$tooltip = self::$placeholder['tt_email'];
			$pattern = self::$patterns['email'];

		} elseif ( $type == 'url' ) {
			$type    = 'text';
			$tooltip = self::$placeholder['tt_url'];
			$pattern = self::$patterns['url'];

		} elseif ( $type == 'tel' ) {
			$tooltip = self::$placeholder['tt_tel'];
			$pattern = self::$patterns['phone'];

		} elseif ( $type == 'number' || $type == 'time' || $type == 'date' || $type == 'datetime-local' || $type == 'week' || $type == 'month' ) {

			// start value for numbers only
			$startvalue = isset( $atts['start'] ) ? "value='" . VC_Helper::str2int( $atts['start'] ) . "'" : '';

			if ( $type == 'number' ) {
				$tooltip = self::$placeholder['tt_number'];

				if ( isset( $atts['min'] ) ) {
					$min     = VC_Helper::str2int( $atts['min'] );
					$tooltip = sprintf( self::$placeholder['tt_number_min'], $min );
				}
				if ( isset( $atts['max'] ) ) {
					$max     = VC_Helper::str2int( $atts['max'] );
					$tooltip = sprintf( self::$placeholder['tt_number_max'], $max );
				}
				if ( isset( $max ) && isset( $min ) ) {
					$tooltip = sprintf( self::$placeholder['tt_number_minmax'], $min, $max );
				}

				$steps = isset( $atts['steps'] ) ? "step='" . VC_Helper::str2int( $atts['steps'] ) . "'" : '';

			} else {
				$min = isset( $atts['min'] ) ? $atts['min'] : '';
				$max = isset( $atts['max'] ) ? $atts['max'] : '';
			}
			$min = ! empty( $min ) ? "min='" . $min . "'" : '';
			$max = ! empty( $max ) ? "max='" . $max . "'" : '';
		}

		if ( isset( $atts['tooltip'] ) && ! empty( trim( $atts['tooltip'] ) ) ) {
			$tooltip = trim( $atts['tooltip'] );
		}

		$minlength = isset( $atts['minlength'] ) ? "minlength='" . (int) $atts['minlength'] . "'" : '';
		$maxmax    = esc_attr( get_option( 'maxlength_inputs', 1000 ) );
		$maxlength = isset( $atts['maxlength'] ) && ( (int) $atts['maxlength'] < (int) $maxmax ) ? (int) $atts['maxlength'] : (int) $maxmax;
		$maxlength = "maxlength='" . $maxlength . "'";

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// check if rendered
		$render_field = apply_filters( 'formshortcode_render_input', true, $atts['name'], self::$post_id );
		if ( $render_field !== true ) {
			return $render_field;
		}

		// Render
		ob_start();

		// Wrapper
		echo "<div class='input-outer-wrapper flex-$alignment'>";
		echo "<div class='input-wrapper $css_class' style='$width'>";

		// Label
		if ( ! empty( $label ) ) {

			echo "<div class='label_wrap'>";

			$font_size = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
			echo "<label class='label' for='$name' data-text='$label' $font_size>$label<span class='requirement-$required'>$required_text</span></label>";

			// Tooltip
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup'>$tooltip</span>" .
					 '</span>';
				$style .= ' validation';
			}
			echo '</div>';

		} else {
			$placeholder .= $required_text;
		}

		// Inner Wrapper
		echo "<div class='input-inner-wrapper'>";

		// Textarea
		if ( $type == 'text' && ( isset( $atts['area'] ) && $atts['area'] ) ) {
			$rows = isset( $atts['rows'] ) ? $atts['rows'] : 5;
			$rows = $rows > 15 ? 15 : $rows;
			echo "<textarea class='$style' style='$css' id='$id' name='$name' placeholder='$placeholder' rows='$rows' $minlength $maxlength $title $required data-required='$required'></textarea>";

			// Input
		} else {
			echo "<input class='$style' style='$css' type='$type' id='$id' name='$name' placeholder='$placeholder' $autocomplete $pattern $minlength $maxlength $min $max $steps $unit $required $title $startvalue data-required='$required'>" .
				"<span class='input-field-icon'></span>"; // Feedback Icon
		}

		/*
		 Fehlermeldung
		echo "<span class='input-popup error-popup'>".
				"<span></span>". // Icon
				"<span>$error</span>". // Text
			"</span>";*/

		// End of Wrappers
		echo '</div></div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Submit button
	 *
	 * @shortcode 'vc_form_button'
	 */
	public function render_button( $atts, $content ) {

		// debug($atts);
		if ( isset( self::$post_meta['block_send'] ) && self::$post_meta['block_send'] ) {
			return '';
		}

		// Attributes
		$type = isset( $atts['type'] ) ? $atts['type'] : '';
		$size = ( isset( $atts['size'] ) && ! empty( $atts['size'] ) ) ? str_replace( '_', '', $atts['size'] ) : '';
		// $width       = isset($atts['width']) ? "style='width:".$atts['width'].";'" : null;
		$width       = ( isset( $atts['width'] ) && ! empty( $atts['width'] ) ) ? 'width:' . Helper::force_units( $atts['width'], array( 'px', '%' ) ) . ';max-width:100%;' : '';
		$inner_width = empty( $width ) ? '' : 'width:100%;max-width:100%;';
		// $default      = $type=='reset' ? 'reset' : 'senden';
		$value        = isset( $atts['text'] ) ? $atts['text'] : self::$placeholder['submit'];
		$value        = nl2br( str_replace( array( '<br />', '<br>' ), '', $value ) );
		$default_text = $value;
		$alignment    = isset( $atts['alignment'] ) ? $atts['alignment'] : 'left';
		// $css_class      = vc_shortcode_custom_css_class($atts['css']);

		// custom button
		$css = $inner_width; // $width;
		if ( $type == 'custom' ) {
			$type = '';
			$css  = VC_Helper::make_custom_element( $atts, 'btn', $css );
		}
		// make link content & icon
		$value = VC_Helper::make_custom_icon( $atts, '', $value );

		// Render
		ob_start();

		echo "<div class='input-outer-wrapper flex-$alignment'>";
		echo "<div class='input-wrapper' style='$width'>";

		echo "<button class='button submitbutton $type $size' style='$css' type='submit' name='submit' title='" . self::$placeholder['submit'] . "' data-text='$default_text'>" .
				$value .
				"<span class='greyd_upload_progress' style='$inner_width'>" .
					"<span class='spinner'>" .
						"<span class='dot-loader'></span>" .
						"<span class='dot-loader dot-loader--2'></span>" .
						"<span class='dot-loader dot-loader--3'></span>" .
					'</span>' .
					"<span class='bar'></span>" .
				'</span>' .
			'</button>';

		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Checkbox
	 *
	 * @shortcode 'vc_form_checkbox'
	 */
	public function render_checkbox( $atts, $content ) {

		// debug($atts);

		// Attributes
		$content     = isset( $content ) ? $content : self::$placeholder['no_content'];
		$required    = isset( $atts['required'] ) ? 'required' : 'optional';
		$prechecked  = isset( $atts['prechecked'] ) ? 'checked' : '';
		$style       = isset( $atts['style'] ) ? preg_replace( '/_/', '', $atts['style'] ) : 'checkbox';
		$name_origin = isset( $atts['name'] ) ? $atts['name'] : '';
		$title       = "title='" . self::$placeholder['title_check'] . "'";

		$textcolor = isset( $atts['textcolor'] ) ? 'color:' . VC_Helper::get_color( $atts, 'text' ) . ';' : '';

		$marginleft = isset( $atts['marginleft'] ) ? 'margin-left:' . Helper::force_units( $atts['marginleft'], array( 'px', '%' ) ) . ';' : '';
		$fontsize   = isset( $atts['fontsize'] ) ? 'font-size:' . Helper::force_units( $atts['fontsize'], array( 'px', 'em' ) ) . ';' : '';
		$lineheight = isset( $atts['lineheight'] ) ? 'line-height:' . Helper::force_units( $atts['lineheight'], array( 'px', 'em', '%' ) ) . ';' : '';

		$is_agb = isset( $atts['is_agb'] ) ? $atts['is_agb'] : 'default';
		$is_agb = $is_agb === 'agb' ? true : false; // convert to bool
		$is_agb = apply_filters( 'formshortcode_checkbox_is_agb', $is_agb, $name_origin, self::$post_id ); // apply filter

		$css_class = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';

		// define name
		if ( ! empty( $name_origin ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['checkbox'] ), 'danger' );
		}
		$name = Helper::make_input_name( $name );

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// iOS Switch Style
		$inline_style = '';
		if ( $style === 'switch' ) {
			$switch_color          = VC_Helper::get_color( $atts, 'switch_', '', 'color_22' );
			$switch_color_selected = VC_Helper::get_color( $atts, 'switch_', '_selected', 'color_13' );
			if ( ! empty( $switch_color ) ) {
				$inline_style .= "input.switch[name='$name'] + span {color: " . $switch_color . ';}';
			}
			if ( ! empty( $switch_color_selected ) ) {
				$inline_style .= "input.switch[name='$name']:checked + span {color: " . $switch_color_selected . ';}';
			}
		}

		// Render
		ob_start();

		// Manipulate Content if is_agb
		if ( $is_agb ) {
			$content = html_entity_decode( esc_attr( get_option( 'agb_text', __( "I hereby accept the terms and conditions.", 'greyd_forms' ) ) ) );

			// Add Actions for manipulation of checkbox (SFMC Integration)
			do_action( 'formshortcode_before_checkbox', $name_origin, $content );
			$content = apply_filters( 'formshortcode_checkbox_content', $content );
		}

		// render custom css
		if ( method_exists( '\enqueue', 'add_custom_style' ) ) {
			\enqueue::add_custom_style( $inline_style );
		} else {
			echo "<style>$inline_style</style>";
		}

		echo "<div class='input-wrapper'>";
			echo "<label class='checkbox-label $css_class' for='$name' style='$textcolor'>";
				echo "<input type='hidden' name='$name' class='checkbox_placeholder_value' value='false'>";
				echo "<input type='checkbox' name='$name' id='$name' value='true' class='$style' $required data-required='$required' $prechecked $title>";
				echo "<div style='$marginleft $fontsize $lineheight'>$content<span class='requirement-$required'>$required_text</span></div>";
			echo '</label>';
		echo '</div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Dropdown
	 *
	 * @shortcode 'vc_form_dropdown'
	 */
	public function render_dropdown( $atts, $content ) {

		// debug($atts);

		// Attributes
		$label       = isset( $atts['label'] ) ? $atts['label'] : null;
		$placeholder = isset( $atts['placeholder'] ) ? $atts['placeholder'] : self::$placeholder['placeholder_select'];
		$required    = isset( $atts['required'] ) ? 'required' : 'optional';
		$multiselect = isset( $atts['multiselect'] ) ? true : false;
		$css_class   = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$title       = self::$placeholder['title_select'];

		// Options
		$options = $this->get_options( $atts );
		if ( ! is_array( $options ) || count( $options ) === 0 ) {
			Helper::make_message( sprintf( self::$placeholder['no_options'], self::$field_names['dropdown'] ), 'danger' );
		}

		// Attributes new
		$style     = isset( $atts['style'] ) ? $atts['style'] : '';
		$width     = isset( $atts['width'] ) ? 'width:' . Helper::force_units( $atts['width'], array( 'px', '%' ) ) . ';' : '';
		$alignment = isset( $atts['alignment'] ) ? $atts['alignment'] : 'left';

		// custom dropdown
		$css = ''; // $width;
		if ( $style == 'custom' ) {
			$style = '';
			$css   = VC_Helper::make_custom_element( $atts, 'dropdown', $css );
		}

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = 'dropdown_' . str_replace( ' ', '_', strtolower( $label ) );
			} elseif ( isset( $atts['placeholder'] ) ) {
				$name = 'dropdown_' . str_replace( ' ', '_', strtolower( $placeholder ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['dropdown'] ), 'danger' );
			}
		}
		$name = Helper::make_input_name( $name );

		// get values
		$values = isset( $atts['options'] ) ? VC_Helper::get_vc_option_checked_values( $atts['options'] ) : array();

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// Render
		ob_start();

		echo "<div class='input-outer-wrapper flex-$alignment'>";
		echo "<div class='input-wrapper custom-select $style $css_class' style='$width'>";

		// Label
		if ( ! empty( $label ) ) {

			echo "<div class='label_wrap'>";

			$font_size = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
			echo "<label class='label' for='$name' $font_size>$label<span class='requirement-$required'>$required_text</span></label>";

			// Tooltip
			$tooltip = isset( $atts['tooltip'] ) ? esc_attr( trim( $atts['tooltip'] ) ) : ( $multiselect ? self::$placeholder['tt_multiselect'] : '' );
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup'>$tooltip</span>" .
					 '</span>';
			}
			echo '</div>';

		} else {
			$placeholder .= $required_text;
		}

		if ( $multiselect ) {
			echo Helper::render_multiselect(
				$name,
				$options,
				array(
					'value'    => implode( ',', $values ),
					'holder'   => $placeholder,
					'style'    => $css,
					'required' => $required === 'required' ? true : false,
					'title'    => $title,
					'atts'     => array( "data-required='$required'" ),
				)
			);
		} else {
			$value = isset( $values[0] ) ? $values[0] : '';
			echo "<select style='$css' name='$name' $required data-required='$required' title='$title' >";
			echo "<option value=''>$placeholder</option>";
			foreach ( $options as $val => $show ) {
				$selected = $value == $val ? 'selected' : '';
				echo "<option value='$val' $selected>$show</option>";
			}
			echo '</select>';
		}

		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Radio Buttons
	 *
	 * @shortcode 'vc_form_radio'
	 */
	public function render_radio( $atts, $content ) {

		// debug($atts);

		// Attributes
		$label       = isset( $atts['label'] ) ? $atts['label'] : null;
		$required    = isset( $atts['required'] ) ? 'required' : 'optional';
		$multiselect = isset( $atts['multiselect'] ) ? true : false;
		$style       = isset( $atts['style'] ) ? preg_replace( '/_/', '', $atts['style'] ) : '';
		$layout      = ( isset( $atts['layout'] ) && $atts['layout'] == 'inline' ) ? "class='inline'" : '';
		$css_class   = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$title       = "title='" . self::$placeholder['title_select'] . "'";
		$font_size   = isset( $atts['options_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['options_font_size'], array( 'px', 'em' ) ) . ";line-height:100%;'" : '';

		// Options
		$options = $this->get_options( $atts );
		if ( ! is_array( $options ) || count( $options ) === 0 ) {
			Helper::make_message( sprintf( self::$placeholder['no_options'], self::$field_names['radio'] ), 'danger' );
		}

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = 'radio_' . str_replace( ' ', '_', strtolower( $label ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['radio'] ), 'danger' );
			}
		}
		$name = Helper::make_input_name( $name );

		// get values
		$values = isset( $atts['options'] ) ? VC_Helper::get_vc_option_checked_values( $atts['options'] ) : array();

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// iOS Switch Style
		$inline_style = '';
		if ( $style === 'switch' ) {
			$selector       = "input[name='$name']";
			$switch_color   = VC_Helper::get_color( $atts, 'switch_', '', 'color_22' );
			$switch_color_s = VC_Helper::get_color( $atts, 'switch_', '_selected', 'color_13' );

			if ( ! empty( $switch_color_s ) ) {
				$selector_s    = $multiselect ? $selector . ' ~ div .option.selected .icn.switch' : $selector . '.switch:checked + span';
				$inline_style .= $selector_s . ' { color: ' . $switch_color_s . '; }';
			}
			if ( ! empty( $switch_color ) ) {
				$selector     .= $multiselect ? ' ~ div .option .icn.switch' : '.switch + span';
				$inline_style .= $selector . ' { color: ' . $switch_color . '; }';
			}
		}

		// Render
		ob_start();

		// render custom css
		if ( method_exists( '\enqueue', 'add_custom_style' ) ) {
			\enqueue::add_custom_style( $inline_style );
		} else {
			echo "<style>$inline_style</style>";
		}

		echo "<div class='input-outer-wrapper'>";
		echo "<div class='input-wrapper'>";

		// Label
		if ( ! empty( $label ) ) {

			echo "<div class='label_wrap'>";

			$lbl_font_size = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
			echo "<label class='label' for='$name' $lbl_font_size>$label<span class='requirement-$required'>$required_text</span></label>";

			// Tooltip
			$tooltip = isset( $atts['tooltip'] ) ? esc_attr( trim( $atts['tooltip'] ) ) : ( $multiselect ? self::$placeholder['tt_multiselect'] : '' );
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup'>$tooltip</span>" .
					 '</span>';
			}
			echo '</div>';
		}

		if ( $multiselect ) {
			echo "<div class='greyd_multiradio'>";
			echo "<input type='text' name='$name' $required data-required='$required' $title value='" . implode( ',', $values ) . "' >";

			foreach ( $options as $val => $text ) {
				$checked = isset( array_flip( $values )[ $val ] ) ? 'selected' : '';
				echo "<div $layout $font_size><div class='option $css_class $checked' onclick='window.forms.onMultiRadioClick(this);' $title data-value='$val'><span class='icn $style'></span>$text</div></div>";
			}

			echo '</div>';
		} else {
			echo "<fieldset $title $required data-required='$required'>";

			foreach ( $options as $val => $text ) {
				$checked = isset( $values[0] ) && $val == $values[0] ? 'checked' : '';
				echo "<div $layout $font_size><label for='$val' class='$css_class' $title><input type='radio' id='$val' $checked value='$val' name='$name' class='$style' $required data-required='$required'>$text</label></div>";
			}
			echo '</fieldset>';
		}

		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Upload
	 *
	 * @shortcode 'vc_form_upload'
	 */
	public function render_upload( $atts, $content ) {

		// debug($atts);

		// Attributes
		$label        = isset( $atts['label'] ) ? $atts['label'] : null;
		$placeholder  = isset( $atts['placeholder'] ) ? $atts['placeholder'] : self::$placeholder['placeholder_select'];
		$default_text = $placeholder;
		$required     = isset( $atts['required'] ) ? 'required' : 'optional';
		$css_class    = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$filesize     = isset( $atts['max'] ) ? trim( str_replace( array( 'MB', 'mb', 'Mb', 'Megabyte', 'megabyte' ), '', $atts['max'] ) ) : '';
		$layout       = isset( $atts['layout'] ) ? 'input ' . $atts['layout'] : 'input';
		$width        = isset( $atts['width'] ) ? 'width: ' . str_replace( ',', '.', Helper::force_units( $atts['width'], array( 'px', '%', 'em' ) ) ) . ';' : '';
		$height       = isset( $atts['height'] ) ? 'height:' . str_replace( ',', '.', Helper::force_units( $atts['height'], array( 'px', '%', 'em' ) ) ) . ';' : '';
		$title        = "title='" . self::$placeholder['title_file'] . "'";

		// Attributes new
		$style     = isset( $atts['style'] ) ? $atts['style'] : '';
		$alignment = isset( $atts['alignment'] ) ? $atts['alignment'] : 'left';
		// custom input
		$css = ''; // $width;
		if ( $style == 'custom' ) {
			$style = '';
			$css   = VC_Helper::make_custom_element( $atts, 'input', $css );
		}

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = 'upload_' . str_replace( ' ', '_', strtolower( $label ) );
			} elseif ( isset( $atts['placeholder'] ) ) {
				$name = 'upload_' . str_replace( ' ', '_', strtolower( $placeholder ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['upload'] ), 'danger' );
			}
		}
		$name = Helper::make_input_name( $name );

		// Filetype
		$filetype = isset( $atts['filetype'] ) ? $atts['filetype'] : '*';
		if ( $filetype == 'other' ) {
			$filetype = $atts['custom_filetype'];
		}

		// Hilfetext
		$tooltip = '';
		if ( isset( $atts['tooltip'] ) && ! empty( trim( $atts['tooltip'] ) ) ) {
			$tooltip = $atts['tooltip'];
		} else {
			if ( $filetype == 'application/pdf' ) {
				$tooltip = self::$placeholder['tt_pdf'];

			} elseif ( $filetype == '.zip' ) {
				$tooltip = self::$placeholder['tt_zip'];

			} elseif ( $filetype == 'image/*' ) {
				$tooltip = self::$placeholder['tt_img'];

			} elseif ( $filetype == 'text/*' ) {
				$tooltip = self::$placeholder['tt_text'];

			} elseif ( $filetype == 'audio/*' ) {
				$tooltip = self::$placeholder['tt_audio'];

			} elseif ( $filetype == 'video/*' ) {
				$tooltip = self::$placeholder['tt_video'];

			} elseif ( $filetype !== '*' ) {
				$tooltip = sprintf( self::$placeholder['tt_other'], $filetype );
			}

			// add filesize to tooltip
			if ( ! empty( $filesize ) ) {
				$tooltip = $tooltip . sprintf( self::$placeholder['tt_size'], $filesize );
				if ( $filetype == '*' ) {
					$tooltip = sprintf( self::$placeholder['tt_size_only'], $filesize );
				}
			}
		}

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// Render
		ob_start();

		echo "<div class='input-outer-wrapper flex-$alignment'>";
		echo "<div class='input-wrapper' style='$width'>";

		// Label
		if ( ! empty( $label ) ) {

			echo "<div class='label_wrap'>";

			$font_size = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
			echo "<label class='label' $font_size>$label<span class='requirement-$required'>$required_text</span></label>";

			// Tooltip
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup'>$tooltip</span>" .
					 '</span>';
			}
			echo '</div>';

		} else {
			$placeholder .= $required_text;
		}

		// add icon to placeholder html
		$placeholder = VC_Helper::make_custom_icon( $atts, '', $placeholder );

		echo "<input type='file' name='$name' accept='$filetype' style='$width' class='custom-file-upload' data-max='" . ( empty( $filesize ) ? 'none' : VC_Helper::str2int( $filesize ) ) . "' $required data-required='$required'>";
		echo "<label for='$name' style='$width' data-text='$default_text'><div class='$layout $style $css_class' style='$css $height' $title>$placeholder</div></label>";

		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * reCAPTCHA
	 *
	 * @shortcode 'vc_form_captcha'
	 */
	public function render_captcha( $atts, $content ) {

		// debug($atts);

		$sitekey = isset( $atts['sitekey'] ) ? esc_attr( $atts['sitekey'] ) : get_option( 'captcha_sitekey', '' );
		$hide    = isset( $atts['hide'] ) ? 'hide' : '';
		$id      = Helper::make_input_name( 'recaptcha' );

		// Render
		ob_start();

		if ( empty( $sitekey ) ) {
			echo Helper::make_message( self::$placeholder['no_recaptcha_sitekey'], 'danger' );
		} else {
			echo '<div id="captcha" class="' . $hide . '">
				<input type="hidden" name="reCAPTCHA" class="recaptcha_input" id="' . $id . '" value="not_verified" required>
				<script src="https://www.google.com/recaptcha/api.js?render=' . $sitekey . '"></script>
				<script>var recaptchaSitekey = "' . $sitekey . '";</script>
			</div>';
		}

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Hidden Fields
	 *
	 * @shortcode 'vc_form_hidden'
	 */
	public function render_hidden( $atts, $content ) {
		// debug($atts);

		// attributes
		$name  = isset( $atts['name'] ) ? Helper::make_input_name( $atts['name'], false ) : null;
		$type  = isset( $atts['type'] ) ? esc_attr( $atts['type'] ) : 'static';
		$value = '';

		// dynamic value
		if ( $type === 'dynamic' ) {

			// attributes
			$dyn_type = isset( $atts['dyn_type'] ) ? $atts['dyn_type'] : 'urlparam';
			$dyn_name = isset( $atts[ 'value_' . $dyn_type ] ) ? $atts[ 'value_' . $dyn_type ] : '';

			// cookie
			if ( $dyn_type === 'cookie' ) {
				$values = array();
				if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
					$values = \Greyd\Extensions\Cookie_Handler::get_all_cookie_values();
				} elseif ( class_exists( '\url_handler' ) ) {
					$values = \url_handler::get_all_cookie_values();
				}
			}
			// url
			elseif ( $dyn_type === 'url' ) {
				global $wp;
				$values = array( $dyn_name => home_url( $wp->request ) );
			}
			// urlparam
			else {
				$values = array();
				if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
					$values = \Greyd\Extensions\Cookie_Handler::get_all_url_values();
				} elseif ( class_exists( '\url_handler' ) ) {
					$values = \url_handler::get_all_url_values();
				}
			}

			// value
			$value = isset( $values[ $dyn_name ] ) ? $values[ $dyn_name ] : '';
		}

		// static value
		else {
			$value = isset( $atts['value'] ) ? $atts['value'] : '';
		}

		// return
		return "<input type='hidden' name='$name' value='$value'>";
	}

	/**
	 * Icon panels
	 *
	 * @shortcode 'vc_form_icon_panels'
	 */
	public function render_icon_panels( $atts, $content ) {

		// debug($atts);

		// Attributes
		$name         = isset( $atts['name'] ) ? $atts['name'] : null;
		$label        = isset( $atts['label'] ) ? $atts['label'] : null;
		$required     = isset( $atts['required'] ) ? 'required' : 'optional';
		$classes      = array( 'greyd_icon_panels' );
		$classes[]    = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$classes[]    = isset( $atts['flex_direction'] ) ? $atts['flex_direction'] : 'flex-row';
		$classes[]    = isset( $atts['justify_content'] ) && ! empty( $atts['justify_content'] ) ? $atts['justify_content'] : 'flex-justify-start';
		$classes[]    = isset( $atts['same_height'] ) ? 'flex-align-stretch' : '';
		$responsive   = isset( $atts['responsive'] ) ? $atts['responsive'] : '';
		$shadow_class = isset( $atts['box_shadow'] ) ? 'has_shadows' : '';
		$multiselect  = isset( $atts['multiselect'] ) ? true : false;

		if ( $multiselect ) {
			$classes[] = 'greyd_multiradio';
		}

		// Calculate Margins & Paddings
		// to get clean overflow behaviour
		$between      = isset( $atts['between'] ) ? Helper::force_units( $atts['between'], array( 'px', 'em' ) ) : '0px';
		$b_val        = strval( (int) filter_var( $between, FILTER_SANITIZE_NUMBER_INT ) / 2 );
		$b_unit       = preg_replace( '/[0-9]+/', '', $between );
		$between      = 'margin: ' . $b_val . $b_unit . '; ';
		$outer_margin = 'margin-left: -' . $b_val . $b_unit . '; margin-right: -' . $b_val . $b_unit . '; margin-top: -' . $b_val . $b_unit . '; ';
		$outer_width  = 'width: calc(' . $b_val . $b_unit . ' + ' . $b_val . $b_unit . ' + 100%); ';

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', $atts['name'] );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = 'icon_panel_' . str_replace( ' ', '_', strtolower( $label ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['icon_panels'] ), 'danger' );
			}
		}
		$name = Helper::make_input_name( $name );

		// define values to be collected within nested shortcodes
		$values = array();

		// Required text
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		/**
		 * PASS ATTRIBUTES TO NESTED ELEMENTS
		 *
		 * In order to access all attributes inside the inner elements,
		 * if set, the attributes are written into the $(is set)content variable
		 */

		// 1. define nested attributes
		$nested_atts = array(
			'font_size',
			'panel_flex_direction',
			'align_items',
			'align_items2',
			'icon_size',
			'between_inner',
			'font_size',
			'font_weight',
			'line_height',
			'multiselect',
		);

		// 2. get all nested shortcodes
		preg_match_all( '/\[(.*?)\]/', $content, $codes, PREG_PATTERN_ORDER );
		// debug($codes);
		$new_content = '';

		// 3. pass panel attributes to '$new_content' & collect values
		foreach ( $codes[1] as $i => $text ) {

			// collect the value
			if ( strpos( $text, 'selected="selected"' ) !== false ) {
				$value = Helper::get_string_between( $text, 'name="', '"' );
				if ( ! empty( $value ) ) {
					$values[] = Helper::make_input_name( $value, false );
				}
			}

			// mod the shortcode
			$new_content .= '[' . $text . ' parent="' . $name . '"';
			foreach ( $nested_atts as $a ) {
				$new_content .= isset( $atts[ $a ] ) ? ' ' . $a . '="' . $atts[ $a ] . '"' : '';
			}
				$new_content .= ' required="' . $required . '"';
				$new_content .= ' between="' . $between . '"';
			$new_content     .= ']';
		}
		// debug($new_content);

		// force Border unit to pixel
		foreach ( array( 'border_top', 'border_left', 'border_right', 'border_bottom' ) as $attr ) {
			if ( isset( $atts[ $attr ] ) && strpos( $atts[ $attr ], 'px' ) === false ) {
				$atts[ $attr ] .= 'px';
			}
		}

		// Styles
		$styles   = $hover = $select = array();
		$styles[] = isset( $atts['padding_y'] ) ? 'padding-top: ' . $atts['padding_y'] . '; padding-bottom: ' . $atts['padding_y'] . ';' : '';
		$styles[] = isset( $atts['padding_x'] ) ? 'padding-left: ' . $atts['padding_x'] . '; padding-right: ' . $atts['padding_x'] . ';' : '';
		$styles[] = isset( $atts['max_width'] ) ? 'max-width: ' . $atts['max_width'] . ';' : '';
		$styles[] = isset( $atts['border_radius'] ) ? 'border-radius: ' . $atts['border_radius'] . ';' : '';
		if ( isset( $atts['border_active'] ) ) {
			$styles[] = isset( $atts['border_style'] ) ? 'border-style: ' . $atts['border_style'] . ';' : 'border-style: solid;';
			$styles[] = isset( $atts['border_top'] ) ? 'border-top-width: ' . $atts['border_top'] . ';' : '';
			$styles[] = isset( $atts['border_left'] ) ? 'border-left-width: ' . $atts['border_left'] . ';' : '';
			$styles[] = isset( $atts['border_right'] ) ? 'border-right-width: ' . $atts['border_right'] . ';' : '';
			$styles[] = isset( $atts['border_bottom'] ) ? 'border-bottom-width: ' . $atts['border_bottom'] . ';' : '';
		}
		// colors
		$styles[] = isset( $atts['bgcolor'] ) ? 'background-color: ' . VC_Helper::get_color( $atts, 'bg', '' ) . ';' : '';
		$styles[] = isset( $atts['textcolor'] ) ? 'color: ' . VC_Helper::get_color( $atts, 'text', '' ) . ';' : '';
		$styles[] = isset( $atts['bordercolor'] ) ? 'border-color: ' . VC_Helper::get_color( $atts, 'border', '' ) . ';' : 'border-color: transparent;';
		$styles[] = isset( $atts['opacity'] ) ? 'opacity: ' . $atts['opacity'] . ';' : 'opacity:1;';
		// hover colors
		$hover[] = isset( $atts['bgcolor_hover'] ) ? 'background-color: ' . VC_Helper::get_color( $atts, 'bg', '_hover' ) . ';' : '';
		$hover[] = isset( $atts['textcolor_hover'] ) ? 'color: ' . VC_Helper::get_color( $atts, 'text', '_hover' ) . ';' : '';
		$hover[] = isset( $atts['bordercolor_hover'] ) ? 'border-color: ' . VC_Helper::get_color( $atts, 'border', '_hover' ) . ';' : '';
		$hover[] = isset( $atts['opacity_hover'] ) ? 'opacity: ' . $atts['opacity_hover'] . ';' : 'opacity:1;';
		// select colors
		$select[] = isset( $atts['bgcolor_select'] ) ? 'background-color: ' . VC_Helper::get_color( $atts, 'bg', '_select' ) . ';' : '';
		$select[] = isset( $atts['textcolor_select'] ) ? 'color: ' . VC_Helper::get_color( $atts, 'text', '_select' ) . ';' : '';
		$select[] = isset( $atts['bordercolor_select'] ) ? 'border-color: ' . VC_Helper::get_color( $atts, 'border', '_select' ) . ';' : '';
		$select[] = isset( $atts['opacity_select'] ) ? 'opacity: ' . $atts['opacity_select'] . ';' : 'opacity:1;';

		// Build custom css
		$inline_style = "
			#$name input ~ label { " . ( implode( ' ', $styles ) ) . " }
			#$name input:hover + label, #$name input:hover + span + label { " . ( implode( ' ', $hover ) ) . " }
			#$name input:checked + label, #$name input:checked + span + label { " . ( implode( ' ', $select ) ) . ' }
		';
		if ( $multiselect ) {
			$inline_style = "
				#$name .option { " . ( implode( ' ', $styles ) ) . " }
				#$name .option:hover { " . ( implode( ' ', $hover ) ) . " }
				#$name .option.selected { " . ( implode( ' ', $select ) ) . ' }
			';
		}

		// Render
		ob_start();

		// custom css
		if ( true && method_exists( '\enqueue', 'add_custom_style' ) ) {
			$result = \enqueue::add_custom_style( $inline_style );
		} else {
			echo "<style>$inline_style</style>";
		}

		// input wrapper
		echo "<div class='input-wrapper icon_panels $responsive $shadow_class'>";

		// label
		if ( ! empty( $label ) ) {

			echo "<div class='label_wrap'>";

			$font_size = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
			echo "<label class='label' $font_size>$label<span class='requirement-$required'>$required_text</span></label>";

			// Tooltip
			$tooltip = isset( $atts['tooltip'] ) ? esc_attr( trim( $atts['tooltip'] ) ) : ( $multiselect ? self::$placeholder['tt_multiselect'] : '' );
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup'>$tooltip</span>" .
					 '</span>';
			}
			echo '</div>';
		}

		// fieldset
		if ( ! $multiselect ) {
			echo "<fieldset $required data-required='$required'>";
		}

		// panel
		echo "<div id='$name' class='" . implode( ' ', $classes ) . "' style='$outer_margin $outer_width'>";

		// hidden input
		if ( $multiselect ) {
			echo "<input type='text' name='$name' value='" . implode( ',', $values ) . "' $required data-required='$required'>";
		}

		// content
		echo do_shortcode( $new_content );

		// close panel
		echo '</div>';

		// close fieldset
		if ( ! $multiselect ) {
			echo '</fieldset>';
		}

		echo '</div>'; // close input wrapper

		$r = ob_get_contents();
		ob_end_clean();
		return $r;

	}

	/**
	 * Icon panel
	 *
	 * @shortcode 'vc_form_icon_panel'
	 */
	public function render_icon_panel( $atts, $content ) {

		// debug($atts);

		// Attributes
		$icon        = isset( $atts['icon'] ) ? wp_get_attachment_url( $atts['icon'] ) : '';
		$icon_hover  = isset( $atts['icon_hover'] ) ? wp_get_attachment_url( $atts['icon_hover'] ) : '';
		$icon_select = isset( $atts['icon_select'] ) ? wp_get_attachment_url( $atts['icon_select'] ) : '';
		$label       = isset( $atts['label'] ) ? $atts['label'] : '';

		$classes   = array( 'greyd_icon_panel', 'animate_fast' );
		$classes[] = isset( $atts['icon_panel_class'] ) ? $atts['icon_panel_class'] : '';
		$classes[] = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$title     = "title='" . self::$placeholder['title_select'] . "'";

		// nested attributes from parent
		$parent         = isset( $atts['parent'] ) ? $atts['parent'] : '';
		$flex_direction = isset( $atts['panel_flex_direction'] ) ? $atts['panel_flex_direction'] : 'flex-column';
		$align_items    = isset( $atts['align_items'] ) ? $atts['align_items'] : ( isset( $atts['align_items2'] ) ? $atts['align_items2'] : 'flex-align-center' );
		$between        = isset( $atts['between'] ) ? $atts['between'] : '';
		$required       = isset( $atts['required'] ) ? $atts['required'] : '';
		$icon_size      = isset( $atts['icon_size'] ) ? Helper::force_units( $atts['icon_size'], array( 'px', 'em', '%' ) ) : 'auto';
		$between_inner  = isset( $atts['between_inner'] ) ? Helper::force_units( $atts['between_inner'], array( 'px', 'em' ) ) : '0px';
		$classes[]      = isset( $atts['box_shadow'] ) ? 'shadow' : '';
		$multiselect    = isset( $atts['multiselect'] ) ? true : false;

		// name & value
		$_name    = isset( $atts['name'] ) ? $atts['name'] : '';
		$name     = Helper::make_input_name( $_name );
		$value    = Helper::make_input_name( $_name, false );
		$selected = isset( $atts['selected'] ) && $atts['selected'] === 'selected' ? 'checked' : '';

		// values depending on flex-direction
		if ( $flex_direction == 'flex-column' ) { // Icon oben
			$icon_size       = "height: $icon_size; width: auto; ";
			$between_inner   = "margin-bottom: $between_inner; ";
			$justify_content = 'flex-justify-start';
			if ( strpos( $align_items, 'flex-align-start' ) !== false ) {
				$text_align = 'left';
			} elseif ( strpos( $align_items, 'flex-align-end' ) !== false ) {
				$text_align = 'right';
			}
		} elseif ( $flex_direction == 'flex-column-reverse' ) { // Icon unten
			$icon_size       = "height: $icon_size; width: auto; ";
			$between_inner   = "margin-top: $between_inner; ";
			$justify_content = 'flex-justify-end';
			if ( strpos( $align_items, 'flex-align-start' ) !== false ) {
				$text_align = 'left';
			} elseif ( strpos( $align_items, 'flex-align-end' ) !== false ) {
				$text_align = 'right';
			}
		} elseif ( $flex_direction == 'flex-row' ) { // Icon links
			$icon_size       = "width: $icon_size; ";
			$between_inner   = "margin-right: $between_inner; ";
			$justify_content = 'flex-justify-start';
			$text_align      = 'left';

		} elseif ( $flex_direction == 'flex-row-reverse' ) { // Icon rechts
			$icon_size       = "width: $icon_size; ";
			$between_inner   = "margin-left: $between_inner; ";
			$justify_content = 'flex-justify-end';
			$text_align      = 'left';

		}
		$text_align = isset( $text_align ) ? 'text-align: ' . $text_align . ';' : 'text-align: center;';

		// text styles
		$text_styles  = isset( $atts['font_size'] ) ? 'font-size: ' . $atts['font_size'] . ';' : '';
		$text_styles .= isset( $atts['font_weight'] ) ? 'font-weight: ' . $atts['font_weight'] . ';' : '';
		$text_styles .= isset( $atts['line_height'] ) ? 'line-height: ' . $atts['line_height'] . ';' : '';

		// Render
		ob_start();

		// input
		if ( ! $multiselect ) {
			echo "<input type='radio' id='$name' value='$value' name='$parent' $selected $required data-required='$required'>";
		}

		// wrapper
		$tag  = 'label';
		$data = "for='$name'";
		if ( $multiselect ) {
			$tag       = 'div';
			$data      = "data-value='$value' onclick='window.forms.onMultiRadioClick(this);'";
			$classes[] = 'option';
			if ( ! empty( $selected ) ) {
				$classes[] = 'selected';
			}
		}
		echo "<$tag $data class='" . implode( ' ', $classes ) . " $flex_direction $align_items $justify_content' style='$between $text_align' $title>";

		// set icons
		$img_class = 'icon_select icon_hover';
		echo "<span style='position: relative; $between_inner'>";
		$icon_ghost = '';
		if ( ! empty( $icon_select ) ) {
			$icon_ghost = $icon_select;
			echo "<img src='$icon_select' style='$icon_size' class='icon_select animate_fast'>";
			$img_class = 'icon_hover'; // remove select from class
		}
		if ( ! empty( $icon_hover ) ) {
			$icon_ghost = $icon_hover;
			echo "<img src='$icon_hover' style='$icon_size' class='$img_class animate_fast'>";
			$img_class = ''; // remove hover & select from class
		}
		if ( ! empty( $icon ) ) {
			$icon_ghost = $icon;
			echo "<img src='$icon' style='$icon_size' class='icon_default $img_class animate_fast'>";
		}
		if ( ! empty( $icon_ghost ) ) {
			echo "<img src='$icon_ghost' style='$icon_size' class='icon_ghost'>";
		}
		echo '</span>';

		// set label
		if ( ! empty( $label ) ) {
			echo "<span style='$text_styles'>$label</span>";
		}

		// end of wrapper
		echo "</$tag>";

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Conditional Fields
	 *
	 * @shortcode 'vc_form_conditional'
	 */
	public function render_conditional_fields( $atts, $content ) {

		// debug($atts);

		$title       = isset( $atts['title'] ) ? $atts['title'] : null;
		$anim        = isset( $atts['anim'] ) ? $atts['anim'] : 'slide';
		$reset_after = isset( $atts['reset_after'] ) ? "data-after='hold'" : '';

		$fields = array(
			isset( $atts['dependant_field'] ) ? Helper::make_input_name( $atts['dependant_field'], false ) : '',
			isset( $atts['dependant_field2'] ) ? Helper::make_input_name( $atts['dependant_field2'], false ) : '',
			isset( $atts['dependant_field3'] ) ? Helper::make_input_name( $atts['dependant_field3'], false ) : '',
		);

		// Render
		ob_start();
		echo "<div class='condition_outer_container $anim open'>";
			echo "<div class='conditional_fields' title='$title' data-field='#" . implode( '&#', $fields ) . "#' $reset_after>";
				echo do_shortcode( $content );
			echo '</div>';
		echo '</div>';

		$r = ob_get_contents();
		ob_end_clean();
		return $r;

	}

	/**
	 * Condition
	 *
	 * @shortcode 'vc_form_condition'
	 */
	public function render_condition( $atts, $content ) {

		// debug($atts);

		// Attributes
		$title         = isset( $atts['title'] ) ? $atts['title'] : null;
		$id            = 'condition_' . uniqid();
		$default_class = ( isset( $atts['default_view'] ) && $atts['default_view'] !== 'hide' ) ? 'default show' : '';
		$data_value    = '';

		if ( $default_class === '' ) {
			$values  = array();
			$type    = isset( $atts['condition_type'] ) ? $atts['condition_type'] : '||';
			$options = isset( $atts['options'] ) ? vc_param_group_parse_atts( $atts['options'] ) : array();
			if ( is_array( $options ) && count( $options ) > 0 ) {
				foreach ( $options as $option ) {
					$values[] = implode(
						'%',
						array(
							isset( $option['field'] ) ? $option['field'] : 0,
							isset( $option['condition'] ) ? $option['condition'] : 'is',
							( isset( $option['value'] ) ? strtolower( $option['value'] ) : '' ),
						)
					);
				}
			}
			$data_value = implode( $type, $values ) . $type;
		}

		// Render
		ob_start();

		echo "<div class='condition $default_class' data-value='$data_value' id='$id'>";
			echo do_shortcode( $content );
		echo '</div>';

		$r = ob_get_contents();
		ob_end_clean();
		return $r;

	}

	/**
	 * Range
	 *
	 * @shortcode 'vc_form_range'
	 * @since 0.8.8
	 */
	public function render_range( $atts, $content ) {
		// debug( $atts );

		// basic vars
		$align     = isset( $atts['align'] ) ? esc_attr( $atts['align'] ) : 'left';
		$required  = isset( $atts['required'] ) ? 'required' : 'optional';
		$css_class = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';
		$required  = isset( $atts['required'] ) ? 'required' : 'optional';

		// min & max
		$min   = isset( $atts['min'] ) ? VC_Helper::str2int( $atts['min'] ) : 0;
		$max   = isset( $atts['max'] ) ? VC_Helper::str2int( $atts['max'] ) : 100;
		$start = isset( $atts['start'] ) ? VC_Helper::str2int( $atts['start'] ) : null;
		$step  = isset( $atts['step'] ) ? VC_Helper::str2int( $atts['step'] ) : 1;

		// label
		$label         = isset( $atts['label'] ) ? Helper::make_sup( esc_attr( $atts['label'] ) ) : '';
		$suffix        = isset( $atts['suffix'] ) ? Helper::make_sup( esc_attr( $atts['suffix'] ) ) : '';
		$font_size     = isset( $atts['label_font_size'] ) ? "style='font-size:" . Helper::force_units( $atts['label_font_size'], array( 'px', 'em' ) ) . ";'" : '';
		$tooltip       = isset( $atts['tooltip'] ) && ! empty( $atts['tooltip'] ) ? esc_attr( $atts['tooltip'] ) : '';
		$required_text = $required === 'required' ? '&nbsp;*' : ' ' . self::$placeholder['optional'];

		// define name
		if ( isset( $atts['name'] ) ) {
			$name = str_replace( ' ', '_', esc_attr( $atts['name'] ) );
		} else {
			if ( isset( $atts['label'] ) ) {
				$name = $type . '_' . str_replace( ' ', '_', strtolower( $label ) );
			} else {
				return Helper::make_message( sprintf( self::$placeholder['no_name'], self::$field_names['range'] ), 'danger' );
			}
		}
		// convert names & set unique id
		$name = Helper::make_input_name( $name, false );
		$id   = Helper::make_input_name( $name );

		// styles
		$stylesheet = "
			#$id .rail {
				" . ( ! isset( $atts['rail_font_size'] ) ? '' :
					'font-size: ' . Helper::force_units( $atts['rail_font_size'], array( 'px', 'em' ) ) . ';'
				) . '
				' . ( ! isset( $atts['rail_border_radius'] ) ? '' :
					'border-radius: ' . Helper::force_units( $atts['rail_border_radius'], array( 'px', '%' ) ) . ';'
				) . '
				' . ( ! isset( $atts['rail_background_color'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['rail_background_color'] ) . ');'
				) . "
			}
			#$id input[type='range'].input-focus ~ .rail {
				" . ( ! isset( $atts['rail_background_color_focus'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['rail_background_color_focus'] ) . ');'
				) . "
			}
			#$id .track {
				" . ( ! isset( $atts['rail_font_size'] ) ? '' :
					'font-size: ' . Helper::force_units( $atts['rail_font_size'], array( 'px', 'em' ) ) . ';'
				) . '
				' . ( ! isset( $atts['rail_border_radius'] ) ? '' :
					'border-radius: ' . Helper::force_units( $atts['rail_border_radius'], array( 'px', '%' ) ) . ';'
				) . '
				' . ( ! isset( $atts['track_background_color'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['track_background_color'] ) . ');'
				) . "
			}
			#$id input[type='range'].input-focus ~ .track {
				" . ( ! isset( $atts['track_background_color_focus'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['track_background_color_focus'] ) . ');'
				) . "
			}
			#$id .thumb_wrapper {
				" . ( ! isset( $atts['thumb_wrapper_font_size'] ) ? '' :
					'font-size: ' . Helper::force_units( $atts['thumb_wrapper_font_size'], array( 'px', 'em' ) ) . ';'
				) . "
			}
			#$id .thumb {
				" . ( ! isset( $atts['thumb_border_radius'] ) ? '' :
					'border-radius: ' . Helper::force_units( $atts['thumb_border_radius'], array( 'px', '%' ) ) . ';'
				) . '
				' . ( ! isset( $atts['thumb_background_color'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['thumb_background_color'] ) . ');'
				) . '
				' . ( ! isset( $atts['thumb_color'] ) ? '' :
					'color: var(--' . str_replace( array( '_', '-' ), '', $atts['thumb_color'] ) . ');'
				) . "
			}
			#$id input[type='range'].input-focus ~ .thumb_wrapper .thumb {
				" . ( ! isset( $atts['thumb_background_color_focus'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['thumb_background_color_focus'] ) . ');'
				) . '
				' . ( ! isset( $atts['thumb_color_focus'] ) ? '' :
					'color: var(--' . str_replace( array( '_', '-' ), '', $atts['thumb_color_focus'] ) . ');'
				) . "
			}
			#$id .tooltip {
				" . ( ! isset( $atts['tooltip_font_size'] ) ? '' :
					'font-size: ' . Helper::force_units( $atts['tooltip_font_size'], array( 'px', 'em' ) ) . ';'
				) . '
				' . ( ! isset( $atts['tooltip_border_radius'] ) ? '' :
					'border-radius: ' . Helper::force_units( $atts['tooltip_border_radius'], array( 'px', '%' ) ) . ';'
				) . '
				' . ( ! isset( $atts['tooltip_background_color'] ) ? '' :
					'background-color: var(--' . str_replace( array( '_', '-' ), '', $atts['tooltip_background_color'] ) . ');'
				) . '
				' . ( ! isset( $atts['tooltip_color'] ) ? '' :
					'color: var(--' . str_replace( array( '_', '-' ), '', $atts['tooltip_color'] ) . ');'
				) . '
			}
		';

		// replace unnecessary whitespace
		$stylesheet = preg_replace( '/(^\s|\s+|\s$)/', ' ', $stylesheet );

		// only render when at least one css-property isset
		if ( preg_match( '/[a-z-]+: [^:;]+;/', $stylesheet ) ) {
			if ( method_exists( '\enqueue', 'add_custom_style' ) ) {
				\enqueue::add_custom_style( $stylesheet );
			}
		}

		// Render
		ob_start();

		// Wrapper
		echo "<div class='input-outer-wrapper flex-$align'>";
		echo "<div class='input-wrapper $css_class'>";

		// Label
		if ( ! empty( $label ) || ! empty( $suffix ) || ! empty( $tooltip ) ) {
			echo "<div class='label_wrap'>";

			echo "<label class='label' for='$name' data-text='$label' $font_size>" .
				( empty( $label ) ? '' : "$label:&nbsp;" ) .
				"<strong class='range_value'>" . ( $start === null ? '__' : strval( $start ) ) . '</strong>' .
				"&nbsp;$suffix" .
				"<span class='requirement-$required'>$required_text</span>
			</label>";

			// tooltip
			if ( ! empty( $tooltip ) ) {
				echo "<span class='forms-tooltip'>" .
						"<span class='forms-tooltip-toggle'></span>" .
						"<span class='forms-tooltip-popup' role='tooltip'>$tooltip</span>" .
					'</span>';
			}

			echo '</div>';
		}

		// Custom Control
		echo "<div class='range_control_wrapper' id='$id'>
			<input
				type='range'
				name='$name'
				min='$min'
				max='$max'
				step='$step'
				" . ( $start === null ? '' : "value='$start'" ) . "
			/>
			<span class='rail' aria-hidden='true'></span>
			<span class='track' aria-hidden='true'></span>
			<span class='thumb_wrapper'>
				<span class='thumb' aria-hidden='true'></span>
				<span class='tooltip' aria-hidden='false' role='tooltip'>$start</span>
			</span>
		</div>";

		// End of Wrappers
		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}

	/**
	 * Math operation control
	 *
	 * @shortcode 'vc_form_math'
	 * @since 0.8.8
	 */
	public function render_math( $atts, $content ) {
		// debug($atts);

		$name     = isset( $atts['name'] ) ? Helper::make_input_name( $atts['name'], false ) : null;
		$color    = isset( $atts['color'] ) ? esc_attr( $atts['color'] ) : '';
		$decimals = isset( $atts['decimals'] ) ? intval( esc_attr( $atts['decimals'] ) ) : 0;
		$style    = '';
		$classes  = array();
		if ( isset( $atts['display'] ) ) {
			$classes[] = str_replace( ',', ' ', esc_attr( $atts['display'] ) );
		}
		if ( isset( $atts['tag'] ) ) {
			$classes[] = esc_attr( $atts['tag'] );
		}
		if ( isset( $atts['align'] ) ) {
			$classes[] = esc_attr( $atts['align'] );
		}

		// style
		if ( strpos( $color, 'color_' ) !== false ) {
			$classes[] = 'has-' . str_replace( '_', '-', $color );
			$color     = '';
		} elseif ( ! empty( $color ) ) {
			$style .= "color: $color;";
		}

		// get formula
		$formula        = isset( $atts['value'] ) ? $atts['value'] : '';
		$realformula    = Helper::get_formula( $formula );
		$displayformula = Helper::display_formula( $formula );

		// conditions
		$has_conditions = isset( $atts['enable_conditions'] ) ? true : false;
		if ( $has_conditions ) {

			$cond_class    = 'has_conditions';
			$cond_elements = array();
			$cond_formulas = array();

			// get condition atts
			$condition_atts = isset( $atts['conditions'] ) ? $atts['conditions'] : '';
			$condition_atts = function_exists( 'vc_param_group_parse_atts' ) ? vc_param_group_parse_atts( $condition_atts ) : VC_Helper::get_vc_input_options( $condition_atts );

			$condition_atts = array_reverse( $condition_atts );

			// loop through conditions
			foreach ( $condition_atts as $i => $condition ) {

				$condition = array_merge(
					array(
						'field1'   => '',
						'operator' => 'greater',
						'field2'   => '',
						'value'    => '',
					),
					$condition
				);

				// formula for frontend
				$cond_formulas[] = "<span class='formula cond_$i'>" . Helper::display_formula( Helper::build_form_tags( $condition['value'] ) ) . '</span>';

				// elements for calculation
				$cond_elements[] = "<span
					data-field1='" . Helper::get_formula( $condition['field1'] ) . "'
					data-operator='" . $condition['operator'] . "'
					data-field2='" . Helper::get_formula( $condition['field2'] ) . "'
					data-value='" . Helper::get_formula( $condition['value'] ) . "'
				></span>";
			}

			// build html elements
			$cond_elements = "<div class='math_conditions' aria-hidden='true'>
				" . implode( '', $cond_elements ) . '
			</div>';
			$cond_formulas = implode( '', $cond_formulas );
		} else {
			$cond_elements = $cond_class = $cond_formulas = '';
		}

		ob_start();

		echo "<div class='input-wrapper math_operation'>";

		echo "<input type='hidden' name='$name' class='math_field $cond_class' data-formula='$realformula' data-decimals='$decimals' />";

		/**
		 * Display formula, conditionsal formulas & result
		 * we always render it to use it to trigger the math_field change via JS
		 * we just hide it depending on the attribute 'display'
		 */
		echo "<div class='display " . implode( ' ', $classes ) . "' style='$style' data-condition='-1'>
			<span class='formula default'>$displayformula</span>
			$cond_formulas
			<span class='equal_sign'> = </span>
			<span class='result'>" . Helper::build_form_tags( '{' . $name . '}' ) . '</span>
		</div>';

		echo $cond_elements;

		echo "<div class='notice'>" . Helper::make_message(
			sprintf( __( "The mathematical calculation for the field \"%s\" could not be performed.", 'greyd_forms' ), $name ),
			'danger'
		) . '</div>';

		echo '</div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}

	/**
	 * Button steps (multistep forms)
	 *
	 * @shortcode 'vc_form_button_steps'
	 */
	public function render_button_steps( $atts, $content ) {

		// debug($atts);

		// Attributes
		$action      = isset( $atts['action'] ) ? $atts['action'] : 'next_step';
		$placeholder = $action === 'prev_step' ? self::$placeholder['prev'] : self::$placeholder['next'];
		$title       = $action === 'prev_step' ? self::$placeholder['title_prev'] : self::$placeholder['title_next'];
		$step_number = '';
		if ( $action === 'exact_step' ) {
			$step_number = isset( $atts['step_number'] ) ? esc_attr( $atts['step_number'] ) : 2;
			$step_number = is_numeric( $step_number ) ? intval( $step_number ) : '#' . str_replace( '#', '', esc_attr( $step_number ) );
			$placeholder = sprintf( self::$placeholder['step'], $step_number );
			$title       = sprintf( self::$placeholder['title_step'], $step_number );
			$step_number = "data-step='" . $step_number . "'";
		}
		$title        = "title='" . $title . "'";
		$value        = isset( $atts['text'] ) ? $atts['text'] : $placeholder;
		$value        = nl2br( str_replace( array( '<br />', '<br>' ), '', $value ) );
		$default_text = $value;

		$alignment = isset( $atts['alignment'] ) ? $atts['alignment'] : 'left';
		$width     = ( isset( $atts['width'] ) && ! empty( $atts['width'] ) ) ? 'width:' . Helper::force_units( $atts['width'], array( 'px', '%' ) ) . ';max-width:100%;' : '';
		$size      = ( isset( $atts['size'] ) && ! empty( $atts['size'] ) ) ? str_replace( '_', '', $atts['size'] ) : '';
		$type      = isset( $atts['type'] ) ? str_replace( '_', '', $atts['type'] ) : 'button';

		// custom button
		$css = empty( $width ) ? '' : 'width:100%;';
		if ( $type == 'custom' ) {
			$type = 'button';
			$css  = VC_Helper::make_custom_element( $atts, 'btn', $css );
		}
		// make link content
		$value = VC_Helper::make_custom_icon( $atts, '', $value );

		// Render
		ob_start();
		if ( strpos( $type, 'link' ) !== false ) {
			echo "<div class='flex-$alignment'><div style='$width'>";
		} else {
			echo "<div class='input-outer-wrapper flex-$alignment'>";
			echo "<div class='input-wrapper' style='$width'>";
		}

		echo "<span class='multistep_button $type $action $size' data-form='" . self::$form_id . "' data-text='$default_text' $step_number style='$css' $title>" . $value . '</span>';

		echo '</div></div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Info notice
	 *
	 * @shortcode 'vc_form_info'
	 */
	public function render_info( $atts, $content ) {
		// debug($atts);

		// Attributes
		$icon       = isset( $atts['icon'] ) ? $atts['icon'] : '';
		$url        = wp_get_attachment_url( $icon );
		$textcolor  = isset( $atts['textcolor'] ) ? "style='color:" . VC_Helper::get_color( $atts, 'text', '' ) . ";'" : '';
		$iconwidth  = isset( $atts['iconwidth'] ) ? $atts['iconwidth'] : '20px';
		$iconmargin = isset( $atts['iconmargin'] ) ? $atts['iconmargin'] : '8px';
		$css_class  = isset( $atts['css'] ) ? vc_shortcode_custom_css_class( $atts['css'] ) : '';

		// Render
		ob_start();

		echo "<div class='input-wrapper info-hinweis'>";

		if ( ! empty( $icon ) ) {
			echo "<img src='$url' style='width:$iconwidth;margin-right:$iconmargin'>";
		}

		if ( ! empty( $content ) ) {
			echo "<p $textcolor class='$css_class'>$content</p>";
		}

		echo '</div>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}


	/**
	 * =================================================================
	 *                          HELPER
	 * =================================================================
	 */

	/**
	 * Get options for dropdowns & radios
	 */
	public function get_options( $atts ) {
		$list = $input = '';
		$type = isset( $atts['inputtype'] ) ? $atts['inputtype'] : '';

		if ( $type === 'list' ) {
			$input = isset( $atts['list_options'] ) ? $atts['list_options'] : '';
		} elseif ( $type === 'predefined' ) {
			$list  = isset( $atts['predefined_lists'] ) ? $atts['predefined_lists'] : '';
			$input = isset( $atts[ $list . '_options' ] ) ? $atts[ $list . '_options' ] : '';
		} else {
			$input = isset( $atts['options'] ) ? $atts['options'] : '';
		}

		return VC_Helper::build_options_array( $type, $input, $list );
	}
}
