<?php
/*
  Comments: Plugin File for Enqueue Scripts
*/
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Enqueue( $config );
class Enqueue {

	/**
	 * Hold the plugin config
	 *
	 * @var object
	 */
	private $config;

	/**
	 * Hold the plugin url
	 *
	 * @var string
	 */
	private $plugin_url;

	/**
	 * Hold the blocks assets url
	 *
	 * @var string
	 */
	private $assets_url;

	/**
	 * Hold the current plugin version
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Class constructor
	 */
	public function __construct( $config ) {

		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . '/wp-admin/includes/plugin.php';
		}

		// class vars
		$this->config     = (object) $config;
		$this->plugin_url = plugin_dir_url( $this->config->plugin_file );
		$this->assets_url = $this->plugin_url . 'assets';
		$this->version    = get_plugin_data( $this->config->plugin_file )['Version'];

		// frontend
		add_action( 'init', array( $this, 'register_frontend_styles' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_styles' ), 11 );
		add_action( 'wp_enqueue_scripts', array( $this, 'register_frontend_scripts' ), 11 );

		if ( is_admin() ) {

			// admin pages
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ), 100 );
	
			// block editor
			add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_editor_scripts' ) );
			add_action( 'enqueue_block_editor_assets', array( $this, 'register_block_scripts' ), 94 );
			add_action( 'enqueue_block_editor_assets', array( $this, 'register_block_styles' ) );
			add_filter( 'block_editor_settings_all', array( $this, 'enqueue_editor_processor_styles' ), 99, 2 );
		}
	}

	/**
	 * Register stylesheets in the frontend.
	 * 
	 * We only register the stylesheets here. They are enqueued conditionally
	 * because they are registered as 'style' using the
	 * register_block_type() function.
	 * 
	 * @see /inc/register-blocks.php for details
	 */
	public function register_frontend_styles() {

		/**
		 * we do not register the stylesheets in the classic suite,
		 * because the stylesheets are processed by the processor.
		 * @see enqueue_frontend_styles() below for details
		 */
		if ( Helper::is_greyd_classic() ) return;
			
		wp_register_style(
			'greyd-forms-main-style',
			$this->assets_url . '/css/public.css',
			null,
			$this->version
		);
	}

	/**
	 * Enqueue stylesheets in the frontend.
	 */
	public function enqueue_frontend_styles() {

		// default style
		if ( function_exists( '\wp_process_style' ) ) {
			wp_process_style(
				'default_style',
				plugin_dir_path( $this->config->plugin_file ) . 'assets/css/deprecated/default.css'
			);
		}
	}

	/**
	 * Register scripts in the frontend.
	 *
	 * We only register the scripts here. They are enqueued conditionally
	 * because they are registered as 'script' or 'style' using the
	 * register_block_type() function.
	 */
	public function register_frontend_scripts() {

		// Main Script
		wp_register_script(
			'greyd-forms-main-script',
			$this->assets_url . '/js/public-default.js',
			array( 'jquery' ),
			$this->version,
			true
		);
		$is_wpml_configured = apply_filters( 'wpml_setting', 0, 'language_negotiation_type' );
		wp_localize_script(
			'greyd-forms-main-script',
			'greydFormSettings',
			array(
				'ajaxurl'       => admin_url( 'admin-ajax.php' ),
				'default_error' => Helper::get_default_error(),
				'errors'        => __( "Please check the following fields:", 'greyd_forms' ),
				'wpml_setup'    => $is_wpml_configured,
				'wp_debug'      => defined( 'WP_DEBUG' ) && constant( 'WP_DEBUG' ),
			)
		);

		// Condition Script
		wp_register_script(
			'greyd-forms-condition-script',
			$this->assets_url . '/js/public-conditions.js',
			array( 'jquery' ),
			$this->version,
			true
		);

		// Multistep Script
		wp_register_script(
			'greyd-forms-multistep-script',
			$this->assets_url . '/js/public-multistep.js',
			array( 'jquery' ),
			$this->version,
			true
		);

		// Multistep Script
		wp_register_script(
			'greyd-forms-live-survey-script',
			$this->assets_url . '/js/public-live-survey.js',
			$this->version,
			true
		);
	}

	/**
	 * Enqueue backend scripts and styles.
	 */
	public function enqueue_admin_scripts() {

		$screen   = get_current_screen();
		$posttype = get_post_type() ? get_post_type() : $screen->post_type;

		if ( $posttype !== $this->config->plugin_post_type && $posttype !== $this->config->plugin_post_type . '_entry' ) {
			return;
		}

		// default admin style
		wp_register_style(
			'greyd-forms-admin-style',
			$this->assets_url . '/css/admin.css',
			null,
			$this->version
		);
		wp_enqueue_style( 'greyd-forms-admin-style' );

		// Backend Scripts
		wp_register_script(
			'greyd-forms-admin-script',
			$this->assets_url . '/js/admin.js',
			array( 'jquery', 'wp-color-picker' ),
			$this->version
		);
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'greyd-forms-admin-script' );
	}

	/**
	 * Add general component scripts for the block editor.
	 */
	public function enqueue_editor_scripts() {

		// inside forms
		if ( get_post_type() != $this->config->plugin_post_type ) {
			return;
		}

		// tools script
		wp_register_script(
			'greyd-forms-helper',
			$this->assets_url . '/js/editor-helper.js',
			array( 'greyd-tools', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->version
		);
		wp_enqueue_script( 'greyd-forms-helper' );

		// establish greyd.forms subclass
		wp_add_inline_script(
			'greyd-forms-helper',
			'greyd.forms = ' . json_encode(
				array(
					'data' => array(
						'config'          => $this->config,
						'plugin_url'      => $this->plugin_url,
						'assets_url'      => $this->assets_url,
						'icon_url'        => $this->plugin_url . 'assets/icon',
						'version'         => $this->version,
						'captcha_sitekey' => get_option( 'captcha_sitekey' ),
						'agb_text'        => html_entity_decode( esc_attr( get_option( 'agb_text', __( "I hereby accept the terms and conditions.", 'greyd_forms' ) ) ) ),
					),
				)
			) . ';',
			'before'
		);

		// components script
		wp_register_script(
			'greyd-forms-components',
			$this->assets_url . '/js/editor-components.js',
			array( 'greyd-tools', 'greyd-forms-helper', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->version
		);
		wp_enqueue_script( 'greyd-forms-components' );

		// deprecations script
		wp_register_script(
			'greyd-forms-deprecations',
			$this->assets_url . '/js/editor-deprecations.js',
			array( 'greyd-tools', 'greyd-forms-helper', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash' ),
			$this->version
		);
		wp_enqueue_script( 'greyd-forms-deprecations' );

		// translations
		if ( function_exists( 'wp_set_script_translations' ) ) {
			/**
			 * May be extended to wp_set_script_translations( 'my-handle', 'my-domain',
			 * plugin_dir_path( MY_PLUGIN ) . 'languages' ) ). For details see
			 * https://make.wordpress.org/core/2018/11/09/new-javascript-i18n-support-in-wordpress/
			 */
			wp_set_script_translations( 'greyd-form-deprecations', 'greyd_forms', $this->config->plugin_path . '/languages' );
		}
	}

	/**
	 * Register the scripts that define all the blocks.
	 */
	public function register_block_scripts() {

		// register the form script
		wp_register_script(
			'greyd-form-block',
			$this->assets_url . '/js/editor-blocks-form.js',
			array( 'greyd-tools', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash', 'wp-core-data', 'wp-edit-post' ),
			$this->version
		);

		// register the script for all inputs
		wp_register_script(
			'greyd-form-inputs',
			$this->assets_url . '/js/editor-blocks-inputs.js',
			array( 'greyd-tools', 'greyd-forms-deprecations', 'greyd-forms-components', 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'lodash', 'wp-core-data', 'wp-edit-post' ),
			$this->version
		);

		// translations
		if ( function_exists( 'wp_set_script_translations' ) ) {
			/**
			 * May be extended to wp_set_script_translations( 'my-handle', 'my-domain',
			 * plugin_dir_path( MY_PLUGIN ) . 'languages' ) ). For details see
			 * https://make.wordpress.org/core/2018/11/09/new-javascript-i18n-support-in-wordpress/
			 */
			wp_set_script_translations( 'greyd-form-block', 'greyd_forms', $this->config->plugin_path . '/languages' );
			wp_set_script_translations( 'greyd-form-inputs', 'greyd_forms', $this->config->plugin_path . '/languages' );
		}
	}

	/**
	 * Register stylesheets to the block editor.
	 * Those stylesheets will also be registered in the frontend.
	 * @see https://developer.wordpress.org/reference/hooks/enqueue_block_assets/
	 */
	public function register_block_styles() {

		if ( ! is_admin() ) return;
		
		wp_register_style(
			'greyd-forms-editor-form-style',
			$this->assets_url . '/css/editor-blocks-form.css',
			null,
			$this->version
		);
		
		wp_register_style(
			'greyd-forms-editor-input-style',
			$this->assets_url . '/css/editor-blocks-inputs.css',
			null,
			$this->version
		);

		// // we enqueue the sub stylesheets conditionally
		// // otherwise they interfere with each other.
		// if ( !class_exists( '\processor' ) ) {

		// 	add_editor_style( $this->assets_url . '/css/public.css' );

		// 	$screen = get_current_screen();
			
		// 	if ( isset( $screen->post_type ) && $screen->post_type === $this->config->plugin_post_type ) {
		// 		add_editor_style( $this->assets_url . '/css/editor-blocks-inputs.css' );
		// 	} else {
		// 		add_editor_style( 'greyd-forms-editor-form-style' );
		// 	}
		// }
	}

	/**
	 * Process stylesheets to the block editor.
	 * @deprecated since FSE
	 */
	public function enqueue_editor_processor_styles( $editor_settings, $editor_context ) {

		// classic SUITE
		if ( class_exists( '\processor' ) ) {

			$plugin_path                              = plugin_dir_path( $this->config->plugin_file );
			$editor_settings['styles'][]              = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/default.css' ) );
			$editor_settings['defaultEditorStyles'][] = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/default.css' ) );

			// when editing form posts
			if ( isset( $editor_context->post->post_type ) && $editor_context->post->post_type === $this->config->plugin_post_type ) {
				$editor_settings['styles'][]              = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/edit-forms.css' ) );
				$editor_settings['defaultEditorStyles'][] = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/edit-forms.css' ) );
			} else {
				$editor_settings['styles'][]              = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/edit.css' ) );
				$editor_settings['defaultEditorStyles'][] = array( 'css' => \processor::process_styles( $plugin_path . 'assets/css/deprecated/edit.css' ) );
			}
		}

		return $editor_settings;
	}
}
