<?php
/*
	Comments: Plugin File for Entry Registration
*/
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Entry( $config );
class Entry {

	private $config;
	private $cpt;

	public function __construct( $config ) {

		// set config
		$this->config = (object) $config;
		$this->cpt    = $this->config->plugin_post_type . '_entry';

		add_action( 'init', array( $this, 'add_entry_post_type' ) );
		add_filter( 'register_post_type_args', array( $this, 'register_post_type_args' ), 99, 2 );

		// Add Columns to entries
		add_filter( 'manage_edit-' . $this->cpt . '_columns', array( $this, 'set_custom_columns' ) );
		add_action( 'manage_' . $this->cpt . '_posts_custom_column', array( $this, 'add_custom_column' ), 10, 2 );

		// Add Meta Box
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
		add_action( 'save_post', array( $this, 'on_save_post' ), 20, 2 );

		// Add Search Restrictions
		add_action( 'restrict_manage_posts', array( $this, 'add_filter_dropdowns' ), 20, 2 );
		add_filter( 'parse_query', array( $this, 'filter_entries' ), 20, 2 );

		// delete actions
		add_action( 'before_delete_post', array( $this, 'on_delete_entry' ), 10, 2 );

		// automatic entry deletion
		add_filter( 'cron_schedules', array( $this, 'add_schedule_intervals' ) );
		add_action( 'greyd_forms_delete_entries', array( $this, 'delete_entries' ) );

		// metabox
		add_action( 'render_metabox_entry_delete', array( $this, 'render_entry_delete_metabox' ), 10, 2 );
		add_action( 'formmeta_save', array( $this, 'save_entry_delete_meta' ) );
	}

	// add custom entry post type
	public function add_entry_post_type() {
		// make some vars
		$post_type = $this->config->plugin_post_type . '_entry';

		$name     = __( "Entries", 'greyd_forms' );
		$singular = __( "Entry", 'greyd_forms' );

		// define new post type
		$post_type_labels    = array(
			'name'               => $name,
			'singular_name'      => $singular,
			'menu_name'          => $name,
			'name_admin_bar'     => $singular,
			'add_new'            => sprintf( __( "New %s", 'greyd_forms' ), $singular ),
			'add_new_item'       => sprintf( __( "Create %s", 'greyd_forms' ), $singular ),
			'new_item'           => sprintf( __( "New %s", 'greyd_forms' ), $singular ),
			'edit_item'          => sprintf( __( "Edit %s", 'greyd_forms' ), $singular ),
			'view_item'          => sprintf( __( "Show %s", 'greyd_forms' ), $singular ),
			'all_items'          => sprintf( __( '%s', 'greyd_forms' ), $name ),
			'search_items'       => sprintf( __( "Search %s", 'greyd_forms' ), $singular ),
			'parent_item_colon'  => sprintf( __( "Parent %s", 'greyd_forms' ), $singular ),
			'not_found'          => sprintf( __( "No %s found", 'greyd_forms' ), $name ),
			'not_found_in_trash' => sprintf( __( "No %s found in the the trash", 'greyd_forms' ), $singular ),
		);
		$post_type_arguments = array(
			'labels'                => $post_type_labels,
			'description'           => __( "Description", 'greyd_forms' ),
			'public'                => false,
			'exclude_from_search'   => true,
			'publicly_queryable'    => false,
			'show_ui'               => true,
			'show_in_menu'          => 'edit.php?post_type=' . $this->config->plugin_post_type,
			'show_in_nav_menus'     => false,
			'show_in_admin_bar'     => false,
			'show_in_rest'          => false,
			'query_var'             => true,
			'rewrite'               => false,
			'capabilities'          => array(
				'edit_post'          => 'edit_others_pages',
				'read_post'          => 'edit_others_pages',
				'delete_post'        => 'delete_pages',
				'edit_posts'         => 'edit_others_pages',
				'edit_others_posts'  => 'edit_others_pages',
				'publish_posts'      => 'publish_books',
				'read_private_posts' => 'read_private_pages',
				'create_posts'       => 'edit_others_pages',
			),
			'has_archive'           => false,
			'supports'              => array( 'title' ),
			'admin_column_sortable' => true,
			'admin_column_filter'   => true,
		);

		/**
		 * Apply filter whether to save entries
		 */
		$save_entries = apply_filters( 'formcpt_save_entries', true );

		// register post_type and taxonony
		if ( $save_entries ) {
			register_post_type( $post_type, $post_type_arguments );
		}
	}

	/**
	 * When the current user is not allowed to edit forms,
	 * show the entry post type in the menu. Still it is only
	 * possible to view the entries when the user has the
	 * correct capabilities.
	 * 
	 * @since 1.7.0
	 * @see https://developer.wordpress.org/reference/functions/register_post_type/#capabilities
	 *
	 * @param array  $args       Array of arguments for registering a post type.
	 * @param string $post_type post_type slug.
	 * 
	 * @return array $args
	 */
	public function register_post_type_args( $args, $post_type ) {

		if ( $post_type === 'tp_forms_entry' ) {
			if ( ! current_user_can( "edit_tp_forms" ) && ! current_user_can( "edit_pages" ) ) {
				$args['show_in_menu'] = true;
				$args['labels']['menu_name'] = "Formulareinträge";
				$args['menu_icon'] = 'dashicons-email-alt';
			}
		}
	
		return $args;;
	}

	/**
	 * =================================================================
	 *                          COLUMNS
	 * =================================================================
	 */

	public function set_custom_columns( $columns ) {

		$columns['entry_state'] = __( 'Status', 'greyd_forms' );
		$columns['entry_data']  = __( "Inputs", 'greyd_forms' );

		// add date column at last position
		unset( $columns['date'] );
		$columns['date'] = __( 'Date' );

		// debug( $columns );

		return $columns;
	}

	public function add_custom_column( $column, $id ) {
		switch ( $column ) {

			case 'entry_state':
				$state = Helper::get_entry_state( $id );
				echo self::get_current_state( $state );
				break;

			case 'entry_data':
				$result = '';

				$entry_form_data = Helper::get_entry_form_data( $id );

				if ( $entry_form_data && ( is_array( $entry_form_data ) || is_object( $entry_form_data ) ) ) {

					$content = array_slice( (array) $entry_form_data, 0, 3 );
					foreach ( $content as $key => $val ) {
						$result = $result . $val . '<br>';
					}
				}

				echo $result;
				break;
		}
	}

	/**
	 * =================================================================
	 *                          META BOX
	 * =================================================================
	 */

	public function add_meta_box() {
		add_meta_box(
			'vc_entry_info', // ID
			__( "Registered data", 'greyd_forms' ), // Title
			array( $this, 'render_meta_box' ), // Callback
			$this->cpt, // CPT name
			'normal', // advanced = at bottom of page
			'default' // Priority
			// array $callback_args = null // Arguments passed to the callback
		);
	}

	/**
	 * Render the entry meta box
	 */
	public function render_meta_box( $post ) {

		$post_id   = $post->ID;
		$form_data = Helper::get_entry_form_data( $post_id );
		$uploads   = Helper::get_entry_uploads( $post_id );

		// AKTUELLER STATUS
		$state = Helper::get_entry_state( $post_id );
		echo '<h2>' . __( "Current state", 'greyd_forms' ) . '</h2>';
		echo self::get_current_state( $state, 'large' );

		// ANGABEN DES NUTZERS
		echo '<h2>' . __( "User information", 'greyd_forms' ) . '</h2>';

		echo "<table class='greyd_table'>";

		if ( ! empty( $form_data ) ) {
			echo '<tr>';
				echo '<th>';
					echo __( "Field name", 'greyd_forms' );
				echo '</th>';
				echo '<th>';
					echo __( "Input", 'greyd_forms' );
				echo '</th>';
			echo '</tr>';
			foreach ( $form_data as $key => $val ) {

				// display checkbox value
				if ( strpos( $val, '[on]' ) !== false ) {
					$val = preg_replace( '/\[on\]/', '☑', $val );
				} elseif ( strpos( $val, '[off]' ) !== false ) {
					$val = preg_replace( '/\[off\]/', '∅', $val );
				}
				// deprecated
				if ( $val == 'on' || $val == 'true' ) {
					$val = __( "☑ Box checked", 'greyd_forms' );
				} elseif ( $val == 'off' || $val == 'false' ) {
					$val = __( "∅ Box unchecked", 'greyd_forms' );
				}

				echo '<tr>';
					echo '<td>';
						echo $key;
					echo '</td>';
					echo '<td>';
				if ( isset( $uploads[ $key ] ) ) {
					echo Helper::get_upload_link_element( $uploads[ $key ] );
				} else {
					echo nl2br( $val );
				}
					echo '</td>';
				echo '</tr>';

			}
		} elseif ( ! empty( $post->post_content ) ) {
			echo '<tr>';
				echo "<td colspan ='2'>";
					echo "<span class='greyd_info_box danger'>" . __( "There was an error converting the values.", 'greyd_forms' ) . '</span>';
				echo '</td>';
			echo '</tr>';
			echo '<tr>';
				echo '<td>';
					echo __( "raw inputs:", 'greyd_forms' );
				echo '</td>';
				echo '<td>';
					echo "<span style='hyphens:auto;word-break:break-all;'>" . $post->post_content . '</span>';
				echo '</td>';
			echo '</tr>';
		}

		echo '</table>';

		// STATUS
		// debug($state);

		echo '<h2>' . __( "Events", 'greyd_forms' ) . '</h2>';

		echo "<table class='greyd_table'>
                <tr>
                    <th>" .
						__( "Time", 'greyd_forms' ) .
					'</th>
                    <th>' .
						__( "Event", 'greyd_forms' ) .
					'</th>
                </tr>';
				$default_states = array(
					'created'     => __( "Entry created", 'greyd_forms' ),
					'verify_send' => __( "The verification mail was successfully sent.", 'greyd_forms' ),
					'no_doi'      => __( "no double opt-in", 'greyd_forms' ),
					'verified'    => __( "Identity verified (opt-in)", 'greyd_forms' ),
					'bounced'     => __( "Permission withdrawn (opt-out)", 'greyd_forms' ),
				);
				$logs           = isset( $state['log'] ) ? (array) $state['log'] : array();
				if ( count( $logs ) > 0 ) {
					foreach ( $logs as $log ) {
						$time = isset( $log['timestamp'] ) ? date_i18n( 'd. F Y - H:i', strtotime( $log['timestamp'] ) ) : "<span style='opacity:.5;'>" . __( "not stated", 'greyd_forms' ) . '</span>';
						$type = isset( $log['type'] ) ? ( $log['type'] === 'error' ? 'danger' : $log['type'] ) : '';
						$text = isset( $log['text'] ) ? strval( $log['text'] ) : '';
						/*
						 *  Salesforce logs a whole html page
						 *  with scripts & functions that reload the page
						 *  therefore these parts need to be excluded
						 */
						$text = preg_match( "/\n/", $text ) ? preg_replace( "/\n.+/", '', $text ) : $text; // remove everything after linebreak
						$text = isset( $default_states[ $text ] ) ? $default_states[ $text ] : strip_tags( $text ); // match if default text exists, else strip tags
						echo "<tr><td>$time</td><td><div class='greyd_info_box $type'>" . $text . '</div></td></tr>';
					}
				}
				echo '</table>';

				/**
				 * Redo Optin action.
		 *
				 * @since 1.4.5
				 */
				$buttontext = __( "Trigger follow-up actions manually", 'greyd_forms' );
				if ( Helper::check_if_entry_is( $post_id, 'verified' ) ) {
					$buttontext = __( "Retrigger follow-up actions", 'greyd_forms' );
				} elseif ( Helper::check_if_entry_is( $post_id, 'bounced' ) ) {
					$buttontext = '';
				}

				if ( ! empty( $buttontext ) ) {
					echo "<input type='hidden' id='redooptin' name='redooptin' value='false'>"; // set to 'true' on button click
					echo "<button class='button' onclick='redoOptin(event)' style='margin-bottom: 1em'>{$buttontext}</button>";
					echo Helper::render_info_box(
						array(
							'text' => __( "When triggering the follow-up actions, mails are sent to admin(s) and contacts and, for example, interfaces are addressed again. If you want to prevent this, you can temporarily disable the corresponding actions in the form.", 'greyd_forms' ),
						)
					);
					?>
			<script>
				const redoOptin = ( e ) => {
					e.preventDefault();
					$( '#redooptin' ).val('true');
					console.log( $( '#redooptin' ).val() )
					$( 'form#post' ).submit();
				}
			</script>
					<?php
				}

				// GENERIERTE INFORMATIONEN

				$form_id   = get_post_meta( $post->ID, 'tp_form_id', true );
				$title     = get_the_title( $form_id );
				$edit_link = get_edit_post_link( $form_id );
				$host      = get_post_meta( $post->ID, 'tp_host', true );
				$host      = empty( $host ) ? __( "no IP address stored", 'greyd_forms' ) : $host;
				$token     = get_post_meta( $post->ID, 'tp_token', true );
				$token     = empty( $token ) ? __( "no token generated", 'greyd_forms' ) : $token;

				// Header
				echo '<h2>' . __( "Metadata", 'greyd_forms' ) . '</h2>';

				echo "<table class='greyd_table vertical'>";
				// Welches Formular?
				echo '<tr>';
				echo '<th>' .
					__( "Form", 'greyd_forms' ) .
					'<br><small>' . __( "Which form has been completed?", 'greyd_forms' ) . '</small>' .
				 '</th>';
				echo '<td>';
				echo "<a href='" . $edit_link . "' target='_blank'>" . $title . '</a>';
				echo '</td>';
				echo '</tr>';

				// Host IP
				echo '<tr>';
				echo '<th>' .
					__( "IP address", 'greyd_forms' ) .
					'<br><small>' . __( "IP address of the user", 'greyd_forms' ) . '</small>' .
				 '</th>';
				echo '<td>';
				echo "<p>$host</p>";
				echo '</td>';
				echo '</tr>';

				// Token
				echo '<tr>';
				echo '<th>' .
					__( "User token", 'greyd_forms' ) .
					'<br><small>' . __( "Generated automatically", 'greyd_forms' ) . '</small>' .
				 '</th>';
				echo '<td>';
				echo "<p>$token</p>";
				echo '</td>';
				echo '</tr>';

				// OptIn Link
				$render_optin = Helper::check_if_entry_is( $post->ID, 'bounced' ) || Helper::check_if_entry_is( $post->ID, 'verified' ) || Helper::check_if_entry_is( $post->ID, 'no_doi' ) ? false : true;
				$render_optin = apply_filters( 'greyd_forms_entry_render_optin', $render_optin, $post->ID );
				if ( $render_optin ) {
					$optin_link = Helper::build_optin_link( $form_id, array(), $token );
					preg_match( "/href='([^']+?)'/", $optin_link, $matches );
					if ( $matches && isset( $matches[1] ) ) {
						$optin_link = $matches[1];
					}
					echo '<tr>';
						echo '<th>' .
						__( "Opt-in link", 'greyd_forms' ) .
						'<br><small>' . __( "With this link, the user can authenticate himself.", 'greyd_forms' ) . '</small>' .
					 '</th>';
						echo '<td>';
					echo "<p class='copy-link' id='copy_link_1' title='" . __( "copy to the clipboard", 'greyd_forms' ) . "' data-text='$optin_link' data-message='✓ " . __( "Copied to clipboard", 'greyd_forms' ) . "'>" . $optin_link . '</p>';
						echo '</td>';
					echo '</tr>';
				}

				// OptOut Link
				$render_optout = Helper::check_if_entry_is( $post->ID, 'bounced' ) ? false : true;
				$render_optout = apply_filters( 'greyd_forms_entry_render_optout', $render_optout, $post->ID );
				if ( $render_optout ) {
					$optout_link = Helper::build_optout_link( $form_id, array(), $token );
					preg_match( "/href='([^']+?)'/", $optout_link, $matches );
					if ( $matches && isset( $matches[1] ) ) {
						$optout_link = $matches[1];
					}
					echo '<tr>';
						echo '<th>' .
						__( "Opt-out link", 'greyd_forms' ) .
						'<br><small>' . __( "With this link, the user can object to the verification.", 'greyd_forms' ) . '</small>' .
					 '</th>';
						echo '<td>';
					echo "<p class='copy-link' title='" . __( "copy to the clipboard", 'greyd_forms' ) . "' id='copy_link_2' data-text='$optout_link' data-message='✓ " . __( "Copied to clipboard", 'greyd_forms' ) . "'>" . $optout_link . '</p>' .
						'<br><small><i>' . __( "Copy a link by clicking on it", 'greyd_forms' ) . '</i></small>';
						echo '</td>';
					echo '</tr>';
				}

				echo '</table>';

				if ( $render_optout ) {
					do_action( 'greyd_forms_render_optout_warning' );
				}

				// Notizen
				$entry_note = get_post_meta( $post->ID, 'entry_note', true );
				echo '<h2>' .
					__( "Notes", 'greyd_forms' ) .
				 '</h2>';
				echo "<textarea type='text' name='entry_note' placeholder='" . __( "Enter your notes here.", 'greyd_forms' ) . "' rows='7'>$entry_note</textarea>";

	}

	public function on_save_post( $post_id, $post ) {

		if ( $post->post_type != $this->cpt ) {
			return;
		}
		if ( ! is_admin() ) {
			return;
		}
		if ( empty( $_POST ) ) {
			return;
		}

		if ( isset( $_POST['entry_note'] ) ) {
			$entry_note = update_post_meta( $post_id, 'entry_note', esc_attr( $_POST['entry_note'] ) );
		}

		if ( isset( $_POST['redooptin'] ) && $_POST['redooptin'] == 'true' ) {
			$this->trigger_optin_by_entry_id( $post_id );
		}
	}

	/**
	 * =================================================================
	 *                          FILTER
	 * =================================================================
	 */

	public function add_filter_dropdowns( $post_type ) {
		if ( $post_type != $this->cpt ) {
			return;
		}

		// add a dropdown with all forms
		$selected = isset( $_GET['form'] ) ? (int) esc_attr( $_GET['form'] ) : '';
		$forms    = get_posts(
			array(
				'post_type'   => $this->config->plugin_post_type,
				'numberposts' => -1,
			)
		);
		echo "<select name='form'>
            <option value=''>" . sprintf( __( "All %s", 'greyd_forms' ), __( "Forms", 'greyd_forms' ) ) . '</option>';

		if ( is_array( $forms ) ) {
			foreach ( $forms as $form ) {
				$id      = $form->ID;
				$title   = $form->post_title;
				$sel     = $id == $selected ? 'selected' : '';
				$entries = Helper::get_entries_by_form_id( $id );
				$count   = $entries ? '(' . count( $entries ) . ')' : '(0)';
				echo "<option value='$id' $sel>$title $count</option>";
			}
		}
		echo '</select>';
	}

	public function filter_entries( $query ) {
		if ( ! $query->is_main_query() || ! is_admin() ) {
			return;
		}

		global $pagenow;
		$post_type = isset( $_GET['post_type'] ) ? esc_attr( $_GET['post_type'] ) : null;

		if ( $pagenow == 'edit.php' && $post_type == $this->cpt ) {

			// filter by form
			$form = isset( $_GET['form'] ) ? (int) esc_attr( $_GET['form'] ) : null;
			if ( $form ) {
				$query->query_vars['meta_key']   = 'tp_form_id';
				$query->query_vars['meta_value'] = $form;
			}
		}
	}

	/**
	 * Called whenever a post is deleted via action hook 'before_delete_post'
	 *
	 * @since 0.9.1
	 *
	 * @param int     $post_id
	 * @param WP_Post $post
	 */
	public function on_delete_entry( $post_id, $post ) {

		$uploads = Helper::get_entry_uploads( $post_id );
		if ( ! is_array( $uploads ) ) {
			return false;
		}

		foreach ( $uploads as $key => $value ) {

			/**
			 * This is just for backward compatiblity reasons.
			 *
			 * Prior to version 0.9.1 uploads were saved as a string
			 * in the post-meta of the entry. Now they are saved as
			 * an array holding more information.
			 *
			 * @deprecated 0.9.1
			 */
			if ( is_string( $value ) ) {
				$path = '/' . str_replace( '//', '/', Helper::get_string_between( $value, 'wp-content/uploads/', "' " ) );
			}
			/**
			 * Uploaded files are saved as an array in the 'tp_uploads' post meta:
			 *
			 *      @property string name   Name of the file.
			 *      @property string path   Relative path from the uploads/ dir
			 *      @property string type   Filetype.
			 *
			 * @see Handler::save_form_as_entry()
			 */
			else {
				$path = isset( $value['path'] ) ? $value['path'] : '';
			}
			$file = wp_upload_dir()['basedir'] . $path;
			return unlink( $file );
		}
	}

	/**
	 * =================================================================
	 *                          HELPER
	 * =================================================================
	 */

	 /**
	  * Calculate the current state of an entry
	  */
	public static function get_current_state( $state, $class = '' ) {
		if ( ! isset( $state ) || empty( $state ) ) {
			return false;
		}

		$default_states = array(
			'created'     => '~ ' . __( "pending", 'greyd_forms' ),
			'verify_send' => '~ ' . __( "pending", 'greyd_forms' ),
			'no_doi'      => __( "no double opt-in", 'greyd_forms' ),
			'verified'    => '✓ ' . __( "verified", 'greyd_forms' ),
			'bounced'     => '∅ ' . __( "objected", 'greyd_forms' ),
			'success'     => '✓ ' . __( "sent", 'greyd_forms' ),
			'danger'      => '✕ ' . __( "Error(s)", 'greyd_forms' ),
		);
		$type           = isset( $state['current']['type'] ) ? ( $state['current']['type'] === 'error' ? 'danger' : $state['current']['type'] ) : '';
		$text           = isset( $state['current']['text'] ) ? $state['current']['text'] : '';
		if ( $text === 'created' || $text === 'verify_send' ) {
			$type = '';
		}
		if ( isset( $default_states[ $text ] ) ) {
			$text = $default_states[ $text ];
		} elseif ( isset( $default_states[ $type ] ) ) {
			$text = $default_states[ $type ];
		}
		return "<div class='greyd_info_box $type $class'>$text</div>";
	}


	/**
	 * =================================================================
	 *                          AUTOMATIC ENTRY DELETION
	 * =================================================================
	 */


	public function render_entry_delete_metabox( $post ) {
		$pre = 'entry_delete';

		$entry_delete_settings = get_post_meta( $post->ID, $pre, true );

		// add filters here
		$filters = array(
			'unverified' => array(
				'title' => __( 'unverified', 'greyd_forms' ),
				'value' => null,
			),
			'withdrawn'  => array(
				'title' => __( 'withdrawn', 'greyd_forms' ),
				'value' => null,
			),
			'error'      => array(
				'title' => __( 'error', 'greyd_forms' ),
				'value' => null,
			),
		);

		if ( isset( $entry_delete_settings['filter'] ) && is_array( $entry_delete_settings['filter'] ) ) {
			foreach ( $entry_delete_settings['filter'] as $saved_filter => $saved_value ) {
				if ( ! isset( $filters[ $saved_filter ] ) ) {
					continue;
				}
				$filters[ $saved_filter ]['value'] = isset( $saved_value ) && $saved_value === 'on' ? 'checked' : '';
			}
		}

		$debug_mail           = isset( $entry_delete_settings['debug_mail'] ) ? $entry_delete_settings['debug_mail'] : '';
		$debug_mail_from      = isset( $entry_delete_settings['debug_mail_from'] ) ? $entry_delete_settings['debug_mail_from'] : '';
		$debug_mail_from_name = isset( $entry_delete_settings['debug_mail_from_name'] ) ? $entry_delete_settings['debug_mail_from_name'] : '';
		$entry_age            = isset( $entry_delete_settings['age'] ) ? $entry_delete_settings['age'] : '';
		$sschedule            = isset( $entry_delete_settings['schedule'] ) ? $entry_delete_settings['schedule'] : '';  // selected schedule

		$schedules = array(
			'daily'   => __( "daily", 'greyd_forms' ),
			'weekly'  => __( "weekly", 'greyd_forms' ),
			'monthly' => __( "monthly", 'greyd_forms' ),
		);

		// start rendering
		echo "<table class='form_meta_box'>";

		// Header
		echo '<tr>' .
				'<th colspan=2><div>' . __( "Automatic deletion of form entries", 'greyd_forms' ) . '</div></th>' .
			'</tr>';

		echo '<tr>';
			echo '<td>' . __( "To be deleted entries", 'greyd_forms' ) . '</td>';

			echo '<td>';

		foreach ( $filters as $filter => $details ) {
			echo "<label for='" . $pre . "[filter][{$filter}]'><input type='checkbox' name='" . $pre . "[filter][{$filter}]' id='" . $pre . "[filter][{$filter}]' {$details['value']} data-text='' >&nbsp;" . $details['title'] . '</label><br>';
		}

			echo '</td>';
		echo '</tr>';

		echo '<tr>';
			echo '<td>' .
			__( "Review period", 'greyd_forms' ) .
			'<small>' . __( "At what interval should the entries be deleted?", 'greyd_forms' ) . '</small>' .
			'</td>';
			echo '<td>';
			echo "<select name='" . $pre . "[schedule]'>";
		foreach ( $schedules as $value => $title ) {
			$s = $value == $sschedule ? 'selected="selected"' : '';
			echo "<option value='$value' $s> $title </option>";
		}
			echo '</select><br>';

			echo '</td>';
		echo '</tr>';

		echo '<tr>';
			echo '<td>' .
				__( "Age", 'greyd_forms' ) .
				'<small>' . __( "Age of the entries to be deleted in days", 'greyd_forms' ) . '</small>' .
				'</td>';
			echo '<td>';
				echo "<input type='number' name='" . $pre . "[age]' value='$entry_age' placeholder='15' step='1' min='1'>";
			echo '</td>';

		echo '</tr>';

		// mail
		echo '<tr>';
			echo '<td>' .
					__( "Admin email", 'greyd_forms' ) .
					'<small>' . __( "Log of the deletion (together with all deleted data) is sent to", 'greyd_forms' ) . '</small>' .
				'</td>';
			echo '<td>';
				echo "<textarea type='text' name='" . $pre . "[debug_mail]' placeholder='example@mail.de, other@mail.de' rows='1' class='no-margin'>$debug_mail</textarea>";
				echo "<i class='info'>" . __( "Please separate multiple email addresses with a comma followed by an empty space.", 'greyd_forms' ) . '</i>';
			echo '</td>';
		echo '</tr>';

		echo '<tr>';
			echo '<td>' .
			__( "Sender address", 'greyd_forms' ) .
			'<small>' . __( "Log of the deletion (together with all deleted data) is sent from", 'greyd_forms' ) . '</small>' .
			'</td>';

			echo '<td>';
				echo "<input type='email' name='" . $pre . "[debug_mail_from]' placeholder='example@mail.de' value='{$debug_mail_from}' class='no-margin'></input>";
			echo '</td>';
		echo '</tr>';

		echo '<tr>';
		echo '<td>' .
		__( "Sender name", 'greyd_forms' ) .
		'</td>';
		echo '<td>';
			echo "<input type='text' name='" . $pre . "[debug_mail_from_name]' placeholder='Admin' value='{$debug_mail_from_name}'  class='no-margin'></input>";
		echo '</td>';
		echo '</tr>';
		echo '</table>';
	}

	// callback on post save
	public function save_entry_delete_meta( $post_args ) {

		$data    = isset( $post_args['data'] ) ? $post_args['data'] : array();
		$form_id = isset( $post_args['post_id'] ) ? $post_args['post_id'] : array();

		$entry_delete_settings = isset( $data['entry_delete'] ) ? $data['entry_delete'] : array();

		update_post_meta( $form_id, 'entry_delete', $entry_delete_settings );

		if ( ! isset( $entry_delete_settings ) || ! isset( $form_id ) ) {
			return;
		}

		$this->create_scheduled_event( $form_id, $entry_delete_settings['schedule'] );
	}


	/**
	 * Add custom intervals for wp_schedule_event()
	 * called via filter 'cron_schedules'
	 */
	public function add_schedule_intervals( $schedules ) {

		// TODO: maybe check here if is already created

		// add a 'weekly' interval
		$schedules['weekly'] = array(
			'interval' => 604800,
			'display'  => __( 'weekly', 'greyd_forms' ),
		);
		// add a 'monthly' interval
		$schedules['monthly'] = array(
			'interval' => 2635200,
			'display'  => __( 'monthly', 'greyd_forms' ),
		);

		return $schedules;
	}

	/**
	 * Schedule events when option is changed.
	 * TODO: maybe check for changed meta in form posts
	 */
	public function create_scheduled_event( $form_id, $schedule ) {

		$event = isset( $schedule ) ? $schedule : null;

		if ( empty( $event ) || ! isset( $form_id ) ) {
			return;
		}

		$hook = 'greyd_forms_delete_entries';

		$args = array(
			'form_id' => $form_id,
		);

		$scheduled_event = wp_get_scheduled_event( $hook, $args );
		// clear event if it already exists --> only one event per form_id
		if ( $scheduled_event && isset( $scheduled_event->args['form_id'] ) && $scheduled_event->args['form_id'] == $form_id ) {
			wp_clear_scheduled_hook( $hook, $args );
		}

		wp_schedule_event( current_time( 'timestamp' ), $event, $hook, $args );
	}


	public function delete_entries( $form_id ) {

		if ( ! isset( $form_id ) || $form_id === 0 ) {
			return;
		}

		$settings = get_post_meta( $form_id, 'entry_delete' )[0];

		$current_time = current_time( 'timestamp' );

		$age_in_days    = isset( $settings['age'] ) ? intval( $settings['age'] ) : 0;
		$age_in_seconds = $age_in_days * 24 * 60 * 60;

		$log                  = array();
		$debug_mail           = isset( $settings['debug_mail'] ) ? $settings['debug_mail'] : false;
		$debug_mail_from      = isset( $settings['debug_mail_from'] ) ? $settings['debug_mail_from'] : false;
		$debug_mail_from_name = isset( $settings['debug_mail_from_name'] ) ? $settings['debug_mail_from_name'] : false;

		if ( $debug_mail ) {
			$log['age']             = $age_in_days;
			$log['schedule']        = $settings['schedule'];
			$log['current_time']    = current_time( 'timestamp' );
			$log['deleted_entries'] = array();
		}

		// states
		$filter_states = is_array( $settings['filter'] ) ? array_keys( $settings['filter'] ) : array();

		// get entries
		$entries = Helper::get_entries_by_form_id( $form_id );

		if ( count( $entries ) > 0 ) {
			foreach ( $entries as $entry ) {

				$state     = Helper::get_entry_state( $entry );
				$timestamp = strtotime( $state['current']['timestamp'] );
				$time_diff = $current_time - $timestamp; // time difference in seconds

				if ( isset( $state['current']['type'] ) && in_array( $state['current']['type'], $filter_states ) && $time_diff >= $age_in_seconds ) {
					wp_delete_post( $entry->ID );
					array_push( $log['deleted_entries'], $state['current'] );
				}
			}
		}
		$time_after_deletion = current_time( 'timestamp' );

		if ( $debug_mail ) {
			$this->send_debug_email( $form_id, $log, $debug_mail, $debug_mail_from, $debug_mail_from_name );
		}
	}

	/**
	 * Send the email with all the debug logs to an admin.
	 */
	public function send_debug_email( $form_id, $log, $debug_mail, $debug_mail_from, $debug_mail_from_name ) {

		if ( empty( $debug_mail ) || empty( $log ) || empty( $form_id ) ) {
			return;
		}

		// get basic mail info
		$subject = sprintf( __( 'Automatische Löschung von Formular-Einträgen der Seite %s wurde abgeschlossen', 'greyd_hub' ), get_site_url() );

		if ( empty( $debug_mail_from ) ) {
			$sitename = wp_parse_url( network_home_url(), PHP_URL_HOST );
			if ( 'www.' === substr( $sitename, 0, 4 ) ) {
				$sitename = substr( $sitename, 4 );
			}
			$mail_from = "wordpress@{$sitename}";
		}

		$log_string = '';

		$current_time    = $log['current_time'];
		$deleted_entries = $log['deleted_entries'];

		foreach ( $deleted_entries as $entry ) {
			$log_string .= '<li>';
			$log_string .= '<b>' . __( 'Info:', 'greyd_hub' ) . ' ' . $entry['text'] . '</b><br>';
			$log_string .= __( 'Zeitpunkt:', 'greyd_hub' ) . ' ' . $entry['timestamp'] . '<br>';
			$log_string .= __( 'Status:', 'greyd_hub' ) . ' ' . $entry['type'] . '<br>';
			$log_string .= '</li><br><br>';
		}

		// mail content
		$mail_content = '
            <p>' . $subject . '</p><br><br>
            <h3>' . sprintf( __( 'Folgende Einträge wurden für das Formular mit der ID %s gelöscht:', 'greyd_hub' ), $form_id ) . '</h3>
            <ul>' .
			$log_string .
			"</ul>
            <br><a href='" . wp_login_url() . "'>" . __( "login to WordPress", 'greyd_hub' ) . '</a>
        ';

		// on localhost we display the logs directly because mails are usually not supported.
		if ( strpos( explode( '://', get_site_url(), 2 )[1], 'localhost:' ) === 0 && is_admin() ) {
			echo "<div style='margin-left:180px;border-left:2px solid currentColor;padding-left:20px;'>" . $mail_content . '</div>';
			return;
		}

		$headers = "From: $debug_mail_from_name <$debug_mail_from>\r\n" .
			"MIME-Version: 1.0\r\n" .
			"Content-Type: text/html; charset=UTF-8\r\n";

		add_filter( 'wp_mail_content_type', array( $this, 'wpdocs_set_html_mail_content_type' ) );
		add_action( 'wp_mail_failed', array( $this, 'display_mail_error' ), 10, 1 );

		// Send Mails to every address
		foreach ( explode( ',', $debug_mail ) as $address ) {
			$return = wp_mail( trim( $address ), $subject, $mail_content, $headers );
		}

		remove_filter( 'wp_mail_content_type', array( $this, 'wpdocs_set_html_mail_content_type' ) );
		remove_action( 'wp_mail_failed', array( $this, 'display_mail_error' ) );

		return $return;
	}

	/**
	 * Display an email error
	 */
	function display_mail_error( $wp_error ) {
		var_error_log( $wp_error );
	}

	/**
	 * set mails to html
	 *
	 * @source (see comments): https://developer.wordpress.org/reference/functions/wp_mail/
	 */
	public function wpdocs_set_html_mail_content_type() {
		return 'text/html';
	}

	/**
	 * Manually trigger the OPT-IN of an entry.
	 *
	 * @since 1.4.5
	 *
	 * @param int $entry_post_id WP_Post ID of the entry.
	 * @return bool Whether the optin was successfull
	 */
	public function trigger_optin_by_entry_id( $entry_post_id ) {

		$token = get_post_meta( $entry_post_id, 'tp_token', true );
		if ( empty( $token ) ) {
			return false;
		}

		return Handler::do_actions_after_optin( $token );
	}
}
