<?php
/**
 * Register all blocks necessary for Greyd.Forms.
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Honeypot();

class Honeypot {

	/**
	 * Holds the name of the hidden checkbox input
	 *
	 * @var string
	 */
	const CHECKBOX_NAME = 'contact_me_by_fax_only';


	/**
	 * Holds the name of the hidden text input
	 *
	 * @var string
	 */
	const INPUT_NAME = 'set_your_password';

	/**
	 * Constructor
	 */
	public function __construct() {

		// render the inputs
		add_action( 'greyd_forms_render_form_start', array( $this, 'render_honeypot_inputs' ) );

		// allow the inputs as form fields
		add_filter( 'formhandler_allow_fields', array( $this, 'allow_honeypot_inputs' ), 10, 2 );

		// validate the honeypot inputs
		add_filter( 'greyd_forms_handler_errors', array( $this, 'handle_honeypot_inputs' ), 10, 3 );

		// remove the inputs after validation
		add_filter( 'greyd_forms_validated_data', array( $this, 'remove_honeypot_inputs' ), 10, 3 );
	}

	/**
	 * Render the inputs
	 *
	 * @param int $form_post_id     WP_Post ID of the form
	 */
	public function render_honeypot_inputs( $form_post_id ) {
		?>
		<div style="visibility: hidden; position: absolute; z-index: -1; height: 0.1px; width: 0.1px; margin: 0; padding: 0;">
			<label for="<?php echo self::CHECKBOX_NAME; ?>">
				<input
					type="checkbox"
					class="input validation"
					name="<?php echo self::CHECKBOX_NAME; ?>"
					id="<?php echo self::CHECKBOX_NAME; ?>"
					value="true"
					aria-hidden="true"
					tabindex="-1"
					autocomplete="off"
				>
				Contact me by fax only
			</label>
			<input
				type="text"
				class="input validation"
				name="<?php echo self::INPUT_NAME; ?>"
				aria-hidden="true"
				tabindex="-1"
				autocomplete="off"
			>
		</div>
		<?php
	}

	/**
	 * Allow hidden fields for backend validation.
	 *
	 * @see handle.php
	 *
	 * @param array $allowed_inputs     Allowed fields of the form.
	 * @param int   $form_post_id         WP_Post ID of the form.
	 *
	 * @return array
	 */
	public function allow_honeypot_inputs( $allowed_inputs, $form_post_id ) {
		return array_merge(
			(array) $allowed_inputs,
			array(
				self::CHECKBOX_NAME => true,
				self::INPUT_NAME    => true,
			)
		);
	}

	/**
	 * Check for the Spambot Honeypot.
	 *
	 * @param array                                  $errors   Current form errors.
	 * @param array                                  $data     Validated form data.
	 * @param array int       WP_Post ID of the form.
	 */
	public function handle_honeypot_inputs( $errors, $data, $form_post_id ) {

		$checkbox_value   = isset( $data[ self::CHECKBOX_NAME ] ) ? $data[ self::CHECKBOX_NAME ] : false;
		$text_input_value = isset( $data[ self::INPUT_NAME ] ) ? $data[ self::INPUT_NAME ] : false;

		// everything is fine
		if ( $checkbox_value === false || empty( $checkbox_value ) || $text_input_value === false || empty( $text_input_value ) ) {
			return $errors;
		}

		$errors[] = __( 'Conspicuous behavior was observed on the site. Bot activity was detected.', 'greyd_forms' );

		return $errors;
	}

	/**
	 * Remove honeypot inputs afterwards.
	 *
	 * @param array                                  $data     Validated form data.
	 * @param array int       WP_Post ID of the form.
	 */
	public function remove_honeypot_inputs( $data, $form_post_id ) {

		if ( isset( $data[ self::CHECKBOX_NAME ] ) ) {
			unset( $data[ self::CHECKBOX_NAME ] );
		}

		if ( isset( $data[ self::INPUT_NAME ] ) ) {
			unset( $data[ self::INPUT_NAME ] );
		}

		return $data;
	}
}
