/**
 * Greyd.Forms Statistics frontend script for analyzing form interactions
 */

document.addEventListener( 'DOMContentLoaded', function () {
	Interactions.init();
} );

var Interactions = new function () {

	this.firstInteractionComplete = false;
	this.completedInteractions = [];
	this.currentForm = null;
	this.areSent = false;
	this.debug = false;

	this.init = function () {

		if (
			typeof document.getElementsByClassName( "greyd_form" ) === 'undefined'
			&& document.getElementById( "greyd_form" ) === 'undefined'
		) {
			return; // early escape if no forms are found
		}

		this.addEvents();
	};

	/**
	 * Listen for a user starting an interaction
	 */
	this.addEvents = function () {

		if ( !Interactions.areSent ) {
			document.addEventListener( 'mousedown', Interactions.checkForFirstInteraction );
		}
	};

	/**
	 * Check if the user starts interacting
	 */
	this.checkForFirstInteraction = function ( e ) {

		Interactions.areSent = false;

		const form = e.target.closest( ".greyd_form" );

		if ( typeof form == "undefined" || !form ) return;

		// safe the current form 
		Interactions.currentForm = form;

		let targetElement = e.target;
		let targetElementName = targetElement.tagName.toLowerCase();

		// fix for radio buttons or checkboxes if someone clicks on the span element
		if ( targetElementName == "span" ) {
			let el = targetElement.parentNode.querySelector( 'input' );

			if ( typeof el === "undefined" || !el ) {
				const label = targetElement.closest( 'label' );

				if ( typeof label !== "undefined" && label ) {
					el = label.querySelector( "input[type='checkbox']" );
				}
			}
			if ( typeof el !== "undefined" && el ) {
				targetElement = el;
				targetElementName = targetElement.tagName.toLowerCase();
			}
		}

		// fix for select dropdowns
		if ( targetElementName == "div" && targetElement.classList.contains( 'select-selected' ) ) {
			let el = targetElement.parentNode.querySelector( 'select' );

			if ( typeof el !== "undefined" && el ) {
				targetElement = el;
				targetElementName = targetElement.tagName.toLowerCase();
			}
		}

		// check for the input fields
		if ( targetElementName == "input" || targetElementName == "textarea" || targetElementName == "select" ) {

			// user clicked in the form -> so start the interaction
			if ( Interactions.debug ) console.log( "User started interacting" );

			// remove the event listener for first interactions
			document.removeEventListener( 'mousedown', Interactions.checkForFirstInteraction );

			// add event
			[ 'focusin', 'focusout', 'mousedown', 'keyup', 'ontouchstart' ].forEach( evt =>
				form.addEventListener( evt, Interactions.trackInteractions, false )
			);

			// if the user leaves the site without submitting the form, the data gets send to the backend via the beforeunload event
			window.addEventListener( 'beforeunload', ( event ) => {
				Interactions.sendInteractions( form );

				// we need to return nothing to hide the standard dialog box
				return undefined;
			} );

		}
	};

	/**
	 * tracks interactions of the user with the form
	 */
	this.trackInteractions = function ( e ) {

		const form = this;

		if ( typeof form == "undefined" || !form ) return;

		const key = e.keyCode || e.charCode;

		let field = {
			ref: e.target,
			tagName: e.target.tagName.toLowerCase(),
			type: e.target.getAttribute( "type" ),
			// for later use
			microInteraction: {
				click: null,
				char: null,
				delete: null
			},
		};

		// fix for radio buttons or checkboxes if someone clicks on the span element
		if ( field.tagName == "span" ) {
			let el = field.ref.parentNode.querySelector( 'input' );
			if ( typeof el == "undefined" || !el ) {
				const label = field.ref.closest( 'label' );

				if ( typeof label !== "undefined" && label ) {
					el = label.querySelector( "input[type='checkbox']" );
				}
			}
			if ( typeof el !== "undefined" && el ) {
				field.ref = el;
				field.tagName = field.ref.tagName.toLowerCase();
			}
		}

		// fix for select dropdowns
		if ( field.tagName == "div" && field.ref.parentNode.classList.contains( 'select-items' ) ) {

			let el = field.ref.parentNode.parentNode.querySelector( 'select' );
			if ( typeof el !== "undefined" && el ) {
				field.ref = el;
				field.tagName = field.ref.tagName.toLowerCase();
			}
		}

		// Logging and debug mode for all interactions for differnt fields
		if ( e.type == "focusin" ) {
			if ( Interactions.debug ) console.log( "User beginns interacting with field: " + field.ref.getAttribute( "name" ) );
		}
		if ( field.tagName == "input" || field.tagName == "textarea" ) {
			// check which type of interaction the user is executing
			if ( e.type == "keyup" && ( key == 8 || key == 46 ) ) {
				if ( Interactions.debug ) console.log( "User deletes Chars in Field: " + field.ref.getAttribute( "name" ) );

			} else if ( e.type == "keyup" ) {
				if ( Interactions.debug ) console.log( "User types in Field: " + field.ref.getAttribute( "name" ) );
			}
		}
		else if ( field.tagName == "select" ) {
			if ( Interactions.debug ) console.log( "User interacts with dropdown: " + field.ref.getAttribute( "name" ) );
			Interactions.saveInteraction( field.ref );

		}
		// if user interacts with a button, we safe the interaction and send it to the backend
		else if ( field.tagName == "button" && field.ref.getAttribute( 'type' ) == "submit" ) {
			Interactions.saveInteraction( field.ref );
			Interactions.sendInteractions( form );
		}
		else {
			if ( Interactions.debug ) console.log( "User clicked in the form, but no form field" );
		}
		if ( e.type == "focusout" && !Interactions.areSent ) {
			if ( Interactions.debug ) console.log( "User stops interacting with field: " + field.ref.getAttribute( "name" ) );
			if ( Interactions.debug ) console.log( "Saving Data" );
			Interactions.saveInteraction( field.ref );
		} else if ( e.type == "focusout" && Interactions.areSent ) {
			if ( Interactions.debug ) console.log( "Interactions sent..." );
		}
	};

	/**
	 * saves the current interaction to the array 'completedInteractions'
	 */
	this.saveInteraction = function ( el ) {
		if ( Interactions.areSent ) return;

		const interaction = {
			id: el.getAttribute( 'id' ),
			type: el.getAttribute( 'type' ),
			name: el.getAttribute( 'name' ),
			required: el.hasAttribute( 'required' ),
			timestamp: Date.now(),
		};
		Interactions.completedInteractions.push( interaction );

		if ( Interactions.debug ) console.log( "Saved Interaction" );
		if ( Interactions.debug ) console.log( "____________________________________________" );
	};

	/**
	 * sends the interactions to the backend
	 */
	this.sendInteractions = function ( form ) {

		if ( Interactions.areSent ) return;

		const formID = form.getAttribute( "data-id" );
		const nonce = form.getAttribute( "data-nonce" );
		const uid = form.getAttribute( "id" );

		if ( Interactions.debug ) console.log( "Sending Interaction for Form: " + formID + " to backend" );

		const body = new FormData();

		const data = {
			form_id: formID,
			uid: uid,
			interactions: Interactions.getCompletedInteractions()
		};

		// append the necessary data for the admin ajax call
		body.append( 'action', 'receive_interactions' );
		body.append( 'nonce', nonce );
		body.append( 'data', JSON.stringify( data ) );

		// send all raw data to the backend using the forms nonce 
		fetch( forms.ajax_url, {
			method: 'POST',
			credentials: 'same-origin',
			body: body,
		} )
			.then( ( response ) => response.json() )
			.then( ( data ) => {
				if ( data ) {
					if ( Interactions.debug ) console.log( data );
				}
			} )
			.catch( ( error ) => {
				if ( Interactions.debug ) console.error( error );
			} );

		Interactions.areSent = true;

		return;
	};

	/**
	 * getter for completed interactions
	 */
	this.getCompletedInteractions = function () {
		return Interactions.completedInteractions;
	};
};
