<?php
/**
 * Greyd.Forms Statistics main file
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Stats( $config );
class Stats {

	/**
	 * Holds the plugin config.
	 */
	private $config;

	/**
	 * Holds the custom table name.
	 */
	private $table_name = 'greyd_forms_interactions';

	/**
	 * Init the class.
	 */
	public function __construct( $config ) {

		$this->config = (object) $config;

		// add the page
		add_action( 'admin_menu', array( $this, 'add_statistics_page' ) );
		// add_action( 'admin_init', array( $this, 'trigger_export' ), 1 );

		// register ajax request
		add_action( 'wp_ajax_nopriv_receive_interactions', array( $this, 'ajax_save_interaction' ) );
		add_action( 'wp_ajax_receive_interactions', array( $this, 'ajax_save_interaction' ) );
		// add_action( 'wp_ajax_send_interaction_data', array( $this, 'ajax_get_interactions' ) );
	}

	/**
	 * Add the admin page.
	 */
	public function add_statistics_page() {
		add_submenu_page(
			'edit.php?post_type=' . $this->config->plugin_post_type, // Parent Slug
			__( 'Greyd.Forms', 'greyd_forms' ) . ' › ' . __( "Stats", 'greyd_forms' ), // Page Title
			__( "Stats", 'greyd_forms' ), // Menu Title
			'administrator', // capability
			'statistics', // Menu slug
			array( $this, 'render_statistics_page' ), // callback function
			3
		);
	}

	/**
	 * Render the admin page
	 */
	public function render_statistics_page() {

		$this->maybe_add_table();

		$form_id    = isset( $_GET['select_form'] ) ? intval( $_GET['select_form'] ) : null;
		$forms      = $this->get_all_forms();
		$form_title = isset( $forms[ $form_id ] ) ? $forms[ $form_id ] : '';
		$stats      = $this->get_interaction_stats( $form_id );
		// debug( $stats );

		if ( $stats['interactions_this_week'] == 0 && $stats['interactions_last_week'] == 0 ) {
			$interaction_trend = '▶ 0%';
		} elseif ( $stats['interactions_last_week'] == 0 ) {
			$interaction_trend = '▲ 100%';
		} else {
			$interaction_trend = round( $stats['interactions_this_week'] / max( $stats['interactions_last_week'], 1 ), 2 );
			$interaction_trend = ( $interaction_trend > 1 ? '▲ ' : '▼ ' ) . abs( $interaction_trend - 1 ) * 100 . '%';
		}

		if ( $stats['started_forms_this_week'] == 0 && $stats['started_forms_last_week'] == 0 ) {
			$started_forms_trend = '▶ 0%';
		} elseif ( $stats['started_forms_last_week'] == 0 ) {
			$started_forms_trend = '▲ 100%';
		} else {
			$started_forms_trend = round( $stats['started_forms_this_week'] / max( $stats['started_forms_last_week'], 1 ), 2 );
			$started_forms_trend = ( $started_forms_trend > 1 ? '▲ ' : '▼ ' ) . abs( $started_forms_trend - 1 ) * 100 . '%';
		}

		if ( $stats['completed_forms_this_week'] == 0 && $stats['completed_forms_last_week'] == 0 ) {
			$completed_forms_trend = '▶ 0%';
		} elseif ( $stats['completed_forms_last_week'] == 0 ) {
			$completed_forms_trend = '▲ 100%';
		} else {
			$completed_forms_trend = round( $stats['completed_forms_this_week'] / max( $stats['completed_forms_last_week'], 1 ), 2 );
			$completed_forms_trend = ( $completed_forms_trend > 1 ? '▲ ' : '▼ ' ) . abs( $completed_forms_trend - 1 ) * 100 . '%';
		}

		if ( $stats['completed_forms'] == 0 || $stats['started_forms'] == 0 ) {
			$conversion = '▶ 0%';
		} else {
			$conversion = $stats['completed_forms'] / max( $stats['started_forms'], 1 );
			$conversion = round( $conversion, 2 ) * 100 . '%';
		}

		if ( $stats['completed_forms_this_week'] == 0 || $stats['started_forms_this_week'] == 0 ) {
			$conversion_this_week = 0;
		} else {
			$conversion_this_week = $stats['completed_forms_this_week'] / max( $stats['started_forms_this_week'], 1 );
		}

		if ( $stats['completed_forms_last_week'] == 0 || $stats['started_forms_last_week'] == 0 ) {
			$conversion_last_week = 0;
		} else {
			$conversion_last_week = $stats['completed_forms_last_week'] / max( $stats['started_forms_last_week'], 1 );
		}

		if ( $conversion_this_week == 0 && $conversion_last_week == 0 ) {
			$conversion_trend = '▶ 0%';
		} elseif ( $conversion_last_week == 0 ) {
			$conversion_trend = '▲ 100%';
		} else {
			$conversion_trend = round( $conversion_this_week / max( $conversion_last_week, 1 ), 2 );
			$conversion_trend = ( $conversion_trend > 1 ? '▲ ' : '▼ ' ) . abs( $conversion_trend - 1 ) * 100 . '%';
		}
		$conversion_this_week = round( $conversion_this_week, 2 ) * 100 . '%';

		?>
		<div class='wrap'>

			<h1><?php echo empty( $form_id ) ? __( "Statistics of all forms", 'greyd_forms' ) : sprintf( __( "Statistics of the form %s", 'greyd_forms' ), '<a href="' . get_edit_post_link( $form_id ) . '">' . $form_title . '</a>' ); ?></h1>
			<?php settings_errors(); ?>

			<div id="greyd_statistics">

				<form method="get" class="form_select">
					<input type="hidden" name="post_type" value="<?php echo $this->config->plugin_post_type; ?>"/>
					<input type="hidden" name="page" value="statistics"/>
					<?php
					if ( count( $forms ) > 0 ) {

						echo '<select id="select_form" class="select_form" name="select_form">';
						echo '<option value="">' . __( "All forms", 'greyd_forms' ) . '</option>';
						foreach ( $forms as $id => $title ) {
							$selected = $id == $form_id ? 'selected="selected"' : '';
							echo "<option value=\"{$id}\" {$selected}>{$title}</option>";
						}
						echo '</select>';

					} else {
						echo Helper::render_info_box(
							array(
								'style' => 'blue',
								'text'  => __( "no form created yet", 'greyd_forms' ),
							)
						);
					}
					?>
					<!-- <div id="load_statistics" class="button button-primary"><?php echo __( "Load statistics", 'greyd_forms' ); ?></div> -->
					<button type="submit" name="submit" id="submit" class="button" value="export"><?php echo __( "Filter statistics", 'greyd_forms' ); ?></button>
					<!-- <button type="submit" name="submit" id="submit" class="button" value="export"><?php echo __( "Download raw data as CSV file", 'greyd_forms' ); ?></button> -->
					<!-- <input type="hidden" name="greyd_statistics_export" value="true"/> -->
				</form>

				<?php if ( $stats['interactions'] ) : ?>

					<h3><?php echo __( "Key figures of the last 7 days", 'greyd_forms' ); ?></h3>
					<div class="count--wrapper">
						<div class='interactions'>
							<data class='count--value'><?php echo $stats['interactions_this_week']; ?></data>
							<p class='count--label'><?php _e( "Interactions", 'greyd_forms' ); ?></p>
							<p class='count--percent'><data value="<?php echo $interaction_trend; ?>"></data> <?php _e( "compared to the previous week", 'greyd_forms' ); ?></p>
							<small class='count--total'><?php _e( "Total:", 'greyd_forms' ); ?> <data value="<?php echo $stats['interactions']; ?>"></data></small>
						</div>
						<div class='started-forms'>
							<data class='count--value'><?php echo $stats['started_forms_this_week']; ?></data>
							<p class='count--label'><?php _e( "Clicked forms", 'greyd_forms' ); ?></p>
							<p class='count--percent'><data value="<?php echo $started_forms_trend; ?>"></data> <?php _e( "compared to the previous week", 'greyd_forms' ); ?></p>
							<small class='count--total'><?php _e( "Total:", 'greyd_forms' ); ?> <data value="<?php echo $stats['started_forms']; ?>"></data></small>
						</div>
						<div class='completed-forms'>
							<data class='count--value'><?php echo $stats['completed_forms_this_week']; ?></data>
							<p class='count--label'><?php _e( "Completed forms", 'greyd_forms' ); ?></p>
							<p class='count--percent'><data value="<?php echo $completed_forms_trend; ?>"></data> <?php _e( "compared to the previous week", 'greyd_forms' ); ?></p>
							<small class='count--total'><?php _e( "Total:", 'greyd_forms' ); ?> <data value="<?php echo $stats['completed_forms']; ?>"></data></small>
						</div>
						<div class='conversion'>
							<data class='count--value'><?php echo $conversion_this_week; ?></data>
							<p class='count--label'><?php _e( 'Conversion Rate', 'greyd_forms' ); ?></p>
							<p class='count--percent'><data value="<?php echo $conversion_trend; ?>"></data> <?php _e( "compared to the previous week", 'greyd_forms' ); ?></p>
							<small class='count--total'><?php _e( "Total:", 'greyd_forms' ); ?> <data value="<?php echo $conversion; ?>"></data></small>
						</div>
					</div>

					<h3><?php _e( "Stats", 'greyd_forms' ); ?></h3>

					<?php
					if ( ! isset( $stats['interactions_by_field'] ) ) {
						echo Helper::render_info_box(
							array(
								'style' => 'blue',
								'text'  => __( "Select a form to see the interactions per field.", 'greyd_forms' ),
							)
						);
					}
					?>

					<div id="greyd_statistics_wrapper">

						<?php if ( isset( $stats['interactions_by_field'] ) ) : ?>
							<div class="greyd_statistic">
								<canvas
									id="interactions_by_field"
									data-source="
									<?php
									echo $this->prepare_data_for_canvas(
										array(
											__( "Interactions", 'greyd_forms' ) => $stats['interactions_by_field'],
										)
									);
									?>
									"
									data-title="<?php _e( "Interactions per field", 'greyd_forms' ); ?>"
									data-label-x="<?php _e( "Field names", 'greyd_forms' ); ?>"
									data-label-y="<?php _e( "Number of interactions", 'greyd_forms' ); ?>"
								></canvas>
							</div>
						<?php endif; ?>
						
						<div class="greyd_statistic">
							<canvas
								id="forms_by_day"
								data-source="
								<?php
								echo $this->prepare_data_for_canvas(
									array(
										__( "started", 'greyd_forms' ) => $stats['started_forms_by_day'],
										__( "submitted", 'greyd_forms' ) => $stats['completed_forms_by_day'],
									)
								);
								?>
								"
								data-title="<?php _e( "Number of calls (started → submitted)", 'greyd_forms' ); ?>"
								data-label-x="<?php _e( "date", 'greyd_forms' ); ?>"
								data-label-y="<?php _e( "Number of calls", 'greyd_forms' ); ?>"
							></canvas>
						</div>

						<div class="greyd_statistic">
							<canvas
								id="interactions_by_day"
								data-source="
								<?php
								echo $this->prepare_data_for_canvas(
									array(
										__( "Interactions per day", 'greyd_forms' ) => $stats['interactions_by_day'],
									)
								);
								?>
								"
								data-title="<?php _e( "Sum of all interactions", 'greyd_forms' ); ?>"
								data-label-x="<?php _e( "date", 'greyd_forms' ); ?>"
								data-label-y="<?php _e( "Number of interactions", 'greyd_forms' ); ?>"
							></canvas>
						</div>

					</div>

					<?php
				endif;

				if ( ! $stats['interactions'] ) {
					echo Helper::render_info_box(
						array(
							'style' => 'red',
							'text'  => (
								$form_id
								? __( "This form has no recorded interactions yet. Please select another form.", 'greyd_forms' )
								: __( "There are no recorded interactions yet. Please wait a while or interact with a form and submit it.", 'greyd_forms' )
							),
						)
					);
				}
				?>

			</div>

		</div>
		<?php
	}

	/**
	 * Prepare data for JS graph canvas.
	 *
	 * @param array $data Data keyed by x-value
	 *
	 * @return string JSON encoded array with x- & y-values
	 */
	public function prepare_data_for_canvas( $allData ) {
		foreach ( $allData as $label => $data ) {
			$json = array();
			foreach ( $data as $key => $value ) {
				$json[] = array(
					'x' => $key,
					'y' => $value,
				);
			}
			$allData[ $label ] = $json;
		}
		return esc_attr( json_encode( $allData ) );
	}

	/**
	 * Handle frontend ajax 'receive_interactions'.
	 *
	 * @see assets/js/frontend.js
	 */
	public function ajax_save_interaction() {

		// format data
		$json = str_replace( '\"', '"', $_POST['data'] );
		$data = json_decode( preg_replace( '/[\x00-\x1F\x80-\xFF]/', '', $json ), true );

		if ( json_last_error() ) {
			wp_die( 'Invalid JSON provided!' );
		}

		$uid              = isset( $data['uid'] ) ? $data['uid'] : uniqid();
		$form_id          = isset( $data['form_id'] ) ? intval( $data['form_id'] ) : null;
		$interaction_data = isset( $data['interactions'] ) ? $data['interactions'] : array();

		$result = $this->save_interaction(
			$uid,
			$form_id,
			$interaction_data
		);

		if ( ! $result ) {
			wp_die( 'Error saving data' );
		} else {
			wp_die( $result );
		}
	}

	/**
	 * =================================================================
	 *                          Helper
	 * =================================================================
	 */

	/**
	 * Add our own custom table if it doesn't exist.
	 *
	 * @return bool
	 */
	public function maybe_add_table() {

		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();
		$table_name      = $wpdb->prefix . $this->table_name;

		// return if table exists
		$query = $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table_name ) );
		if ( $wpdb->get_var( $query ) == $table_name ) {
			return false;
		}

		$sql = "CREATE TABLE `{$table_name}` (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			uid varchar(255) NOT NULL,
			form_id varchar(255) NOT NULL,
			interaction_data LONGTEXT,
			timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
			PRIMARY KEY (id)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );

		return true;
	}

	/**
	 * Save an interaction.
	 *
	 * @param string $uid              Unique ID of the frontend form.
	 * @param string $form_id          WP_Post ID of the form.
	 * @param array  $interaction_data  All the interaction data.
	 *
	 * @return int|false               The number of rows inserted, or false on error.
	 */
	public function save_interaction( string $uid, $form_id, $interaction_data ) {

		$this->maybe_add_table();

		global $wpdb;
		$table_name = $wpdb->prefix . $this->table_name;

		return $wpdb->insert(
			$table_name,
			array(
				'uid'              => esc_sql( $uid ),
				'form_id'          => esc_sql( $form_id ),
				'interaction_data' => maybe_serialize( $interaction_data ),
			)
		);
	}

	/**
	 * Retrieve interactions from DB.
	 *
	 * @param string $form_id   Optional WP_Post form ID to filter by.
	 *
	 * @return array
	 */
	public function get_interactions( $form_id = null ) {

		global $wpdb;
		$table_name = $wpdb->prefix . $this->table_name;

		if ( empty( $form_id ) ) {
			$results = $wpdb->get_results(
				"SELECT * FROM {$table_name}"
			);
		} else {
			$results = $wpdb->get_results(
				$wpdb->prepare( "SELECT * FROM {$table_name} WHERE form_id=%d", $form_id )
			);
		}

		if ( empty( $results ) ) {
			return array();
		}

		$interactions = array();

		foreach ( $results as $result ) {

			$interaction_data = maybe_unserialize( $result->interaction_data );

			if ( ! is_array( $interaction_data ) ) {
				continue;
			}

			foreach ( $interaction_data as $data ) {
				$interaction = array_merge(
					$data,
					array(
						'uid'     => $result->uid,
						'form_id' => $result->form_id,
						'date'    => $result->timestamp,
					)
				);

				if ( isset( $interaction['name'] ) && strpos( $interaction['name'], '___id' ) !== false ) {
					$interaction['name'] = explode( '___id', $interaction['name'], 2 )[0];
				}

				$interactions[] = $interaction;
			}
		}

		return $interactions;
	}

	/**
	 * Get computed interaction stats.
	 *
	 * @param string $form_id   Optional WP_Post form ID to filter by.
	 *
	 * @return array
	 */
	public function get_interaction_stats( $form_id = null ) {

		$interactions = $this->get_interactions( $form_id );

		// filtered arrays
		$stats = array(
			'interactions'              => $interactions,
			'interactions_this_week'    => array(),
			'interactions_last_week'    => array(),
			'interactions_by_day'       => array(),
			'started_forms'             => array(),
			'started_forms_this_week'   => array(),
			'started_forms_last_week'   => array(),
			'started_forms_by_day'      => array(),
			'completed_forms'           => array(),
			'completed_forms_this_week' => array(),
			'completed_forms_last_week' => array(),
			'completed_forms_by_day'    => array(),
		);

		// this stat only makes sense, when we select a specific field
		if ( ! empty( $form_id ) ) {
			$stats['interactions_by_field'] = array();
		}

		foreach ( $interactions as $interaction ) {

			$is_this_week = strtotime( $interaction['date'] ) > strtotime( '-7 days' );
			$is_last_week = strtotime( $interaction['date'] ) < strtotime( '-7 days' ) && strtotime( $interaction['date'] ) > strtotime( '-14 days' );

			$day = date( 'Y-m-d', strtotime( $interaction['date'] ) );

			// interactions by week
			if ( $is_this_week ) {
				$stats['interactions_this_week'][] = $interaction;
			} elseif ( $is_last_week ) {
				$stats['interactions_last_week'][] = $interaction;
			}

			// interactions by day
			if ( ! isset( $stats['interactions_by_day'][ $day ] ) ) {
				$stats['interactions_by_day'][ $day ] = array();
			}
			$stats['interactions_by_day'][ $day ][] = $interaction;

			// started forms
			if ( ! isset( $stats['started_forms'][ $interaction['uid'] ] ) ) {

				$stats['started_forms'][ $interaction['uid'] ] = $interaction;
				if ( $is_this_week ) {
					$stats['started_forms_this_week'][ $interaction['uid'] ] = $interaction;
				} elseif ( $is_last_week ) {
					$stats['started_forms_last_week'][ $interaction['uid'] ] = $interaction;
				}

				// ...by day
				if ( ! isset( $stats['started_forms_by_day'][ $day ] ) ) {
					$stats['started_forms_by_day'][ $day ] = array();
				}
				$stats['started_forms_by_day'][ $day ][ $interaction['uid'] ] = $interaction;
			}

			// completed forms
			if ( $interaction['type'] === 'submit' ) {

				$stats['completed_forms'][ $interaction['uid'] ] = $interaction;
				if ( $is_this_week ) {
					$stats['completed_forms_this_week'][ $interaction['uid'] ] = $interaction;
				} elseif ( $is_last_week ) {
					$stats['completed_forms_last_week'][ $interaction['uid'] ] = $interaction;
				}

				// ...by day
				if ( ! isset( $stats['completed_forms_by_day'][ $day ] ) ) {
					$stats['completed_forms_by_day'][ $day ] = array();
				}
				$stats['completed_forms_by_day'][ $day ][ $interaction['uid'] ] = $interaction;
			}

			// interactions by field
			if ( ! empty( $form_id ) ) {

				if ( ! isset( $stats['interactions_by_field'][ $interaction['name'] ] ) ) {
					$stats['interactions_by_field'][ $interaction['name'] ] = array();
				}

				$stats['interactions_by_field'][ $interaction['name'] ][] = $interaction;
			}
		}

		// convert to counts
		foreach ( $stats as $key => $ints ) {

			if ( strpos( $key, 'by_day' ) !== false ) {
				foreach ( $ints as $day => $_ints ) {
					$stats[ $key ][ $day ] = count( $_ints );
				}
			} elseif ( strpos( $key, 'by_field' ) !== false ) {

				$inputs        = array_merge( $this->get_all_input_names( $form_id ), array( 'submit' => 'submit' ) );
				$stats[ $key ] = array();
				foreach ( $inputs as $name ) {
					$stats[ $key ][ $name ] = isset( $ints[ $name ] ) ? count( $ints[ $name ] ) : 0;
				}
			} else {
				$stats[ $key ] = count( $ints );
			}
		}

		return $stats;
	}

	/**
	 * Get all input names of a form
	 *
	 * @param mixed $post_id_or_content
	 *
	 * @return array
	 */
	public function get_all_input_names( $post_id_or_content ) {
		if ( class_exists( '\Greyd\Forms\Helper' ) ) {
			return \Greyd\Forms\Helper::get_all_input_names( $post_id_or_content );
		}
		return array();
	}

	/**
	 * Get all greyd.forms
	 *
	 * @return array Post-titles keyed by post-ID
	 */
	public function get_all_forms( array $prepend = array() ) {
		if ( class_exists( '\Greyd\Forms\Helper' ) ) {
			return \Greyd\Forms\Helper::get_all_forms( $prepend );
		}
		return array();
	}

	/**
	 * Download an array as a CSV file.
	 *
	 * @param array  $array
	 * @param string $filename
	 */
	public function download_array_as_csv( $array, $filename = '' ) {
		if ( class_exists( '\Greyd\Forms\export' ) ) {
			return \Greyd\Forms\export::download_array_as_csv( $array, $filename );
		}
		return null;
	}


	/**
	 * =================================================================
	 *                          DEPRECATED
	 * =================================================================
	 */

	/**
	 * Handle backend ajax 'send_interaction_data'
	 *
	 * @see assets/js/backend.js
	 */
	public function ajax_get_interactions() {
		$form_id = isset( $_POST['form_id'] ) ? intval( $_POST['form_id'] ) : null;
		$data    = $this->get_interaction_stats( $form_id );
		wp_send_json( $data );
	}

	public function trigger_export() {
		if ( isset( $_POST ) && isset( $_POST['greyd_statistics_export'] ) && isset( $_POST['select_form'] ) ) {
			$this->export_raw_data_by_form( $_POST['select_form'] );
		}
	}

	public function export_raw_data_by_form( $form_id ) {
		if ( empty( $form_id ) ) {
			return false;
		}

		/**
		 * get all entries by form
		 */
		$interaction_data = $this->get_interaction_data( $form_id );

		// reorder data
		$raw_data = $this->format_interaction_data_for_download( $interaction_data['raw_data'] );

		if ( empty( $raw_data ) || ! is_array( $raw_data ) ) {
			return false;
		}

		$keys = array( array_keys( $raw_data[0] ) );

		$value_rows = array();

		foreach ( $raw_data as $data_entry ) {
			array_push( $value_rows, array_values( $data_entry ) );
		}

		$export = array_merge( $keys, $value_rows );

		/**
		 * get all the data
		 */
		$file_name = preg_replace( '/\s+/', '_', get_the_title( $form_id ) ) . '-' . __( "Stats", 'greyd_forms' ) . '-' . date( 'y_m_d' );

		$this->download_array_as_csv( $export, $file_name );
	}
}
