<?php
/**
 * Greyd.Forms Surveys main file
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Survey( $config );
class Survey {

	/**
	 * Holds the plugin config.
	 */
	private $config;

	/**
	 * Holds the custom table name.
	 */
	private $table_name = 'greyd_forms_surveys';

	/**
	 * Init the class.
	 */
	public function __construct( $config ) {

		$this->config = (object) $config;

		add_action( 'admin_menu', array( $this, 'add_surveys_page' ) );

		// register ajax request
		add_action( 'wp_ajax_nopriv_receive_survey', array( $this, 'ajax_save_survey' ) );
		add_action( 'wp_ajax_receive_survey', array( $this, 'ajax_save_survey' ) );

		add_action( 'wp_ajax_nopriv_send_survey', array( $this, 'ajax_send_survey' ) );
		add_action( 'wp_ajax_send_survey', array( $this, 'ajax_send_survey' ) );
	}

	public function add_surveys_page() {
		add_submenu_page(
			'edit.php?post_type=' . $this->config->plugin_post_type, // Parent Slug
			__( 'Greyd.Forms', 'greyd_forms' ) . ' › ' . __( "Surveys", 'greyd_forms' ), // Page Title
			__( "Surveys", 'greyd_forms' ), // Menu Title
			'administrator', // capability
			'surveys', // Menu slug
			array( $this, 'render_surveys_page' ), // callback function
			3
		);
	}

	public function render_surveys_page() {

		$this->maybe_add_table();

		$form_id = isset( $_GET['select_form'] ) ? intval( $_GET['select_form'] ) : null;
		$forms   = $this->get_all_forms();

		$forms_with_surveys = array();

		$form_title = isset( $forms[ $form_id ] ) ? $forms[ $form_id ] : '';
		$survey_id  = isset( $_GET['select_survey'] ) ? intval( $_GET['select_survey'] ) : null;

		$surveys = array();

		if ( $form_id ) {
			$surveys = $this->get_all_surveys_by_form( $form_id );
		}

		foreach ( $forms as $id => $title ) {
			if ( count( $this->get_all_surveys_by_form( $id ) ) !== 0 ) {
				$forms_with_surveys[ $id ] = $title;
			}
		}

		?>
		<div class='wrap'>

			<h1><?php echo empty( $form_id ) ? __( "Surveys", 'greyd_forms' ) : sprintf( __( "Surveys of the form %s", 'greyd_forms' ), '<a href="' . get_edit_post_link( $form_id ) . '">' . $form_title . '</a>' ); ?></h1>
			<?php settings_errors(); ?>

			<div id="greyd_surveys">

				<form method="get" class="form_select">
					<input type="hidden" name="post_type" value="<?php echo $this->config->plugin_post_type; ?>"/>
					<input type="hidden" name="page" value="surveys"/>
					<?php
					if ( count( $forms_with_surveys ) > 0 ) {

						echo '<select id="select_form" class="select_form" name="select_form">';
						echo '<option value="">' . __( "please select", 'greyd_forms' ) . '</option>';
						foreach ( $forms_with_surveys as $id => $title ) {
							// $has_survey = count( $this->get_all_surveys_by_form( $id ) ) !== 0;
							$selected = $id == $form_id ? 'selected="selected"' : '';
							echo "<option value=\"{$id}\" {$selected}>{$title}</option>";
						}
						echo '</select>';
						echo '<button type="submit" name="submit" id="submit" class="button" value="export">' . __( "Select form", 'greyd_forms' ) . '</button>';
					} elseif ( count( $forms ) === 0 ) {
						echo Helper::render_info_box(
							array(
								'style' => 'blue',
								'text'  => __( "no form created yet", 'greyd_forms' ),
							)
						);
					} else {
						echo Helper::render_info_box(
							array(
								'style' => 'blue',
								'text'  => __( "No survey data has been collected yet", 'greyd_forms' ),
							)
						);
					}
					?>
					
				</form>

				<div id="greyd_surveys_wrapper">
					<?php if ( isset( $surveys ) && count( $surveys ) > 0 ) : ?>
			
						<?php foreach ( $surveys as $survey_id => $survey ) : ?>
							<h3><?php echo $survey['survey_title']; ?></h3>
							<div class="survey_row">
								<div class="greyd_survey">
									<canvas
										id="survey_result"
										data-source="
										<?php
										echo $this->prepare_data_for_canvas( $survey['survey_data'] );
										?>
										"
									></canvas>
								</div>
								<div class="count--wrapper">
									<div class='survey_results_total'>
										<p class='count--label'><?php _e( "Number of participants in the survey", 'greyd_forms' ); ?></p>
										<data class='count--value'><?php echo $survey['total_submissions']; ?></data>
									</div>
								</div>
							</div>
					
						<?php endforeach; ?>
					<?php endif; ?>
				</div>
			</div>

		</div>
		<?php
	}

	/**
	 * Handle frontend ajax 'receive_survey'.
	 *
	 * @see assets/js/frontend.js
	 */
	public function ajax_save_survey() {

		// format data
		$json    = str_replace( '\"', '"', $_POST['data'] );
		$rawdata = json_decode( preg_replace( '/[\x00-\x1F\x80-\xFF]/', '', $json ), true );

		if ( json_last_error() ) {
			wp_die( 'Invalid JSON provided!' );
		}

		$uid     = isset( $rawdata['uid'] ) ? $rawdata['uid'] : uniqid();
		$form_id = isset( $rawdata['form_id'] ) ? intval( $rawdata['form_id'] ) : null;
		$data    = isset( $rawdata['surveyData'] ) ? $rawdata['surveyData'] : array();

		$survey_id    = isset( $data['surveyId'] ) ? $data['surveyId'] : array();
		$survey_title = isset( $data['surveyTitle'] ) ? $data['surveyTitle'] : __( "Survey", 'greyd_forms' );
		$options      = isset( $data['options'] ) ? $data['options'] : array();

		$total_submissions = isset( $data['totalSubmissions'] ) ? $data['totalSubmissions'] : 0;

		$result = $this->save_survey(
			$uid,
			$form_id,
			$options,
			$survey_id,
			$total_submissions,
			$survey_title
		);

		if ( ! $result ) {
			wp_die( 'Error saving data' );
		} else {
			wp_die( $result );
		}
	}

	/**
	 * Handle frontend ajax 'send_survey'.
	 *
	 * @see assets/js/frontend.js
	 */
	public function ajax_send_survey() {

		// format data
		$json = str_replace( '\"', '"', $_POST['data'] );
		$data = json_decode( preg_replace( '/[\x00-\x1F\x80-\xFF]/', '', $json ), true );

		if ( json_last_error() ) {
			wp_die( 'Invalid JSON provided!' );
		}

		$uid       = isset( $data['uid'] ) ? $data['uid'] : uniqid();
		$form_id   = isset( $data['form_id'] ) ? intval( $data['form_id'] ) : null;
		$survey_id = isset( $data['survey_id'] ) ? $data['survey_id'] : array();

		$result = $this->get_survey( $form_id, $survey_id );

		if ( ! $result ) {
			wp_die( 'Error sending data' );
		} else {
			wp_send_json_success( $result, 200 );
		}
	}

	/**
	 * =================================================================
	 *                          Helper
	 * =================================================================
	 */

	/**
	 * Add our own custom table if it doesn't exist.
	 *
	 * @return bool
	 */
	public function maybe_add_table() {

		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();
		$table_name      = $wpdb->prefix . $this->table_name;

		// return if table exists
		$query = $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table_name ) );

		if ( $wpdb->get_var( $query ) == $table_name ) {
			return false;
		}

		$sql = "CREATE TABLE `{$table_name}` (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			uid VARCHAR(255) NOT NULL,
			form_id VARCHAR(255) NOT NULL,
			survey_data LONGTEXT,
			survey_id VARCHAR(12),
			survey_title TEXT,
			total_submissions int NOT NULL,
			timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
			PRIMARY KEY (id),
			UNIQUE (survey_id)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );

		return true;
	}

	/**
	 * Save an interaction.
	 *
	 * @param string $uid              Unique ID of the frontend form.
	 * @param string $form_id          WP_Post ID of the form.
	 * @param array  $interaction_data  All the interaction data.
	 *
	 * @return int|false               The number of rows inserted, or false on error.
	 */
	public function save_survey( string $uid, $form_id, $survey_data, $survey_id, $total_submissions, $survey_title ) {

		$this->maybe_add_table();

		global $wpdb;
		$table_name = $wpdb->prefix . $this->table_name;

		return $wpdb->replace(
			$table_name,
			array(
				'uid'               => esc_sql( $uid ),
				'form_id'           => esc_sql( $form_id ),
				'survey_data'       => maybe_serialize( $survey_data ),
				'survey_id'         => esc_sql( $survey_id ),
				'total_submissions' => esc_sql( $total_submissions ),
				'survey_title'      => esc_sql( $survey_title ),
			),
		);
	}

	/**
	 * Retrieve survey from DB.
	 *
	 * @param string $form_id   WP_Post form ID
	 * @param string $survey_id string survey ID
	 *
	 * @return array
	 */
	public function get_survey( $form_id, $survey_id ) {

		global $wpdb;
		$table_name = $wpdb->prefix . $this->table_name;

		$this->maybe_add_table();

		$result = $wpdb->get_row(
			$wpdb->prepare( "SELECT * FROM {$table_name} WHERE form_id=%d AND survey_id=%s", $form_id, $survey_id )
		);

		if ( empty( $result ) ) {
			return array(
				'form_id'           => $form_id,
				'survey_id'         => $survey_id,
				'total_submissions' => 0,
			);
		}

		$survey = array(
			'form_id'           => (int) $result->form_id,
			'survey_id'         => $result->survey_id,
			'survey_data'       => maybe_unserialize( $result->survey_data ),
			'total_submissions' => (int) $result->total_submissions,
			'survey_title'      => $result->survey_title,
		);

		return $survey;
	}

	/**
	 * Retrieve survey from DB.
	 *
	 * @param string $form_id   WP_Post form ID
	 * @param string $survey_id string survey ID
	 *
	 * @return array
	 */
	public function get_all_surveys_by_form( $form_id ) {

		global $wpdb;
		$table_name = $wpdb->prefix . $this->table_name;

		$this->maybe_add_table();

		$results = $wpdb->get_results(
			$wpdb->prepare( "SELECT * FROM {$table_name} WHERE form_id=%d", $form_id )
		);

		if ( empty( $results ) ) {
			return array();
		}

		$surveys = array();

		foreach ( $results as $result ) {
			$surveys[ $result->survey_id ] = array(
				'form_id'           => (int) $result->form_id,
				'survey_id'         => $result->survey_id,
				'survey_data'       => maybe_unserialize( $result->survey_data ),
				'total_submissions' => (int) $result->total_submissions,
				'survey_title'      => $result->survey_title,
			);
		}

		return $surveys;
	}

	/**
	 * Get all greyd.forms
	 *
	 * @return array Post-titles keyed by post-ID
	 */
	public function get_all_forms( array $prepend = array() ) {
		if ( class_exists( '\Greyd\Forms\Helper' ) ) {
			return \Greyd\Forms\Helper::get_all_forms( $prepend );
		}
		return array();
	}

	/**
	 * Prepare data for JS graph canvas.
	 *
	 * @param array $data Data keyed by x-value
	 *
	 * @return string JSON encoded array with x- & y-values
	 */
	public function prepare_data_for_canvas( $allData ) {
		$newData = array();
		foreach ( $allData as $data ) {
			$json                      = array();
			$newData[ $data['value'] ] = $data['survey_result'];
		}
		return esc_attr( json_encode( $newData ) );
	}
}
