<?php
/**
 *  Core file for handling form-data
 *
 *  this file receives all the form data send via default.js
 *  it then validates and checks all values according to the backend setup
 *
 *  the listed functions are categorized
 *  the categories appear in a chronological order:
 *      - register parameters for later handling
 *      - handle form data and respond to ajax-call
 *      - define and send action after
 *      - send opt-in mail (optional)
 *      - handle opt-in & opt-out (optional)
 *      - send registration mails (to user & admin)
 *      - trigger additional actions, such as API-calls and interfaces
 *
 *  also there are different action hooks for developers to call functions from within your own plugin
 *  just search for '@action'
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Handler( $config );

class Handler {

	/**
	 * Holds the plugin config.
	 */
	public static $config;

	/**
	 * Contruct
	 */
	public function __construct( $config ) {

		// set config
		self::$config = (object) $config;

		// add url-params & cookies
		add_filter( 'greyd_add_url_params', array( $this, 'add_url_params' ) );
		add_filter( 'greyd_cookies_supported_values', array( $this, 'add_url_params' ) );
		add_filter( 'formshortcode_add_hidden_fields', array( $this, 'add_hidden_fields' ) );
		add_filter( 'formhandler_allow_fields', array( $this, 'allow_hidden_fields' ), 10, 2 );

		// register ajax request
		add_action( 'wp_ajax_nopriv_handle_form_data', array( $this, 'handle_form_data' ) );
		add_action( 'wp_ajax_handle_form_data', array( $this, 'handle_form_data' ) );

		// send response
		add_action( 'formhandler_error', array( $this, 'send_error' ), 10, 1 );
		add_action( 'formhandler_success', array( $this, 'send_success' ), 10, 1 );
		add_action( 'formhandler_log', array( $this, 'log_debug' ), 10, 2 );

		// handle links from within mails
		add_action( 'parse_request', array( $this, 'handle_mail_link' ), 1 );
		// show wp_mail() errors
		add_action( 'wp_mail_failed', array( $this, 'on_mail_error' ), 10, 1 );
	}



	/*
	=======================================================================

								URL PARAMS

	=======================================================================
	*/

	/**
	 * Add params to Cookie_Handler
	 */
	public static function add_url_params( $input = array() ) {
		$params = array(
			array(
				'nicename' => __( "Form opt-in", 'greyd_forms' ),
				'name'     => 'optin',
				'url'      => 'optin',
				'cookie'   => 'greyd_forms_optin',
			),
			array(
				'nicename' => __( "Form opt-out", 'greyd_forms' ),
				'name'     => 'optout',
				'url'      => 'optout',
				'cookie'   => 'greyd_forms_optout',
			),
			array(
				'nicename' => __( "Form ID (Post ID)", 'greyd_forms' ),
				'name'     => 'form',
				'url'      => 'form',
				'cookie'   => 'greyd_forms_form_id',
			),
		);
		return array_merge( $input, $params );
	}

	/**
	 * Add hidden fields to form
	 */
	public static function add_hidden_fields( $input ) {
		$fields = array();
		if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
			$fields = \Greyd\Extensions\Cookie_Handler::get_cookie_values();
		} elseif ( class_exists( '\url_handler' ) ) {
			$fields = \url_handler::get_cookie_values();
		}
		unset( $fields['optin'] );
		unset( $fields['optout'] );
		unset( $fields['form'] );
		return array_merge( $input, $fields );
	}

	/**
	 * Allow hidden fields for handler
	 */
	public static function allow_hidden_fields( $input, $post_id ) {
		$fields = array();
		if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
			$fields = \Greyd\Extensions\Cookie_Handler::get_supported_params();
		} elseif ( class_exists( '\url_handler' ) ) {
			$fields = \url_handler::get_params();
		}
		$allowed_fields = array();
		foreach ( $fields as $i => $param ) {
			if ( $param['name'] === 'optin' || $param['name'] === 'optout' ) {
				continue;
			}
			$allowed_fields[ $param['name'] ] = true;
		}
		return array_merge( (array) $input, $allowed_fields );
	}


	/*
	=======================================================================

								HANDLE FORMS

	=======================================================================
	*/
	// Vars
	public static $rules     = array(
		'url'   => '/((http|https)\:\/\/)?[a-zA-Z0-9\.\/\?\:@\-_=#]+\.([a-zA-Z0-9\&\.\/\?\:@\-_=#])*/',
		'date'  => '/^[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])$/',
		'time'  => '/^(?:2[0-3]|[01][0-9]):[0-5][0-9]$/',
		'phone' => '/^(\+?\(?[\d\s\+\(\-\–\/\)]{4,10}\)?[\d\s\-\–\/]{1,16})$/',
		'email' => "/^(\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*)$/",
	);
	public static $post_data = null;
	public static $form_id;
	public static $token;
	public static $entry_id;

	/**
	 * Backend validation of all forms
	 */
	public static function handle_form_data() {
		// do_action('formhandler_error', $_POST ); // debug

		/**
		 * Build data array
		 */
		parse_str( $_POST['data'], $form_data );
		$file_data = isset( $_FILES ) ? $_FILES : array();
		$data      = array();
		$raw_data  = array_merge( $form_data, $file_data );
		do_action( 'formhandler_log', $raw_data, 'raw form data' );

		// return if form is empty
		if ( empty( $form_data ) && empty( $file_data ) ) {
			do_action( 'formhandler_error', __( "The submitted form is empty.", 'greyd_forms' ) );
		}

		// Vars
		$errors        = array();
		$post_id       = $_POST['post_id'];
		self::$form_id = $post_id;
		$form_mode     = filter_input( INPUT_POST, 'form_mode' );
		$host          = Helper::get_client_host();
		$block_send    = get_post_meta( $post_id, 'block_send', true ) == 'block_send' ? true : false;

		/**
		 * Return if submitting is blocked
		 */
		if ( $block_send ) {
			do_action( 'formhandler_error', __( "This form cannot be submitted. Submitting is prevented by the settings.", 'greyd_forms' ) );
		}

		// /**
		// * Verify the nonce
		// */
		// if ( check_ajax_referer( 'handle_form_data_' .$post_id, 'nonce', false ) === false ) {

		// **
		// * Send the admin an error email.
		// * @since 1.4.1
		// */
		// ob_start();
		// echo sprintf(
		// "<p>The validation of a Greyd.Form has failed because a WordPress NONCE could not be verified. You can find the entire request below. See the link %s for details.</p>",
		// "<a href='https://developer.wordpress.org/reference/functions/check_ajax_referer/'>check_ajax_referer()</a>"
		// );
		// debug( $_REQUEST );
		// $message = ob_get_contents();
		// ob_end_clean();
		// self::send_admin_error_mail( $post_id, $message );

		// do_action('formhandler_error', sprintf(__("Das Formular konnte nicht validiert werden. Bitte wenden Sie sich an den Administrator. (Post-ID: %s)", 'greyd_forms'), $post_id) );
		// }

		/**
		 * Get all inputs
		 */
		$expected_inputs = Helper::get_all_inputs( $post_id );
		do_action( 'formhandler_log', $expected_inputs, 'expected inputs of the form' );

		/**
		 * Filter & sanitize the $raw_data and push it to $data
		 */
		foreach ( $raw_data as $name => $val ) {

			$expected_input = isset( $expected_inputs[ $name ] ) ? $expected_inputs[ $name ] : array();
			$type           = isset( $expected_input['type'] ) ? $expected_input['type'] : 'unknown';

			// check if input is part of expected inputs
			if ( empty( $expected_input ) ) {

				/**
				 * Filter to allow custom fields in forms
				 *
				 * @filter 'formhandler_allow_fields'
				 *
				 * @param array $exceptions     Inputs to be allowed, keyed by their name.
				 * @param int   $post_id        Post ID of the form.
				 */
				$exceptions = apply_filters( 'formhandler_allow_fields', array(), $post_id );

				if ( ! $exceptions[ $name ] ) {
					$errors[] = sprintf( __( "The field '%s' is not permitted.", 'greyd_forms' ), $name );
				}
			}

			// we do not need to verify empty values.
			if ( empty( $val ) ) {
				continue;
			}

			// check type
			switch ( $type ) {
				case 'number':
					if ( ! is_numeric( $val ) ) {
						$errors[] = sprintf( __( "The value for '%s' must be numeric.", 'greyd_forms' ), $name );
					} else {
						strval( $val );
					}
					break;

				case 'email':
					if ( ! preg_match( self::$rules['email'], $val ) ) {
						$errors[] = sprintf( __( "'%s' is not a valid email.", 'greyd_forms' ), $val );
					}
					break;

				case 'url':
					if ( ! preg_match( self::$rules['url'], $val ) ) {
						$errors[] = sprintf( __( "'%s' is not a valid URL.", 'greyd_forms' ), $val );
					}
					break;

				case 'date':
					if ( ! preg_match( self::$rules['date'], $val ) ) {
						$errors[] = sprintf( __( "'%s' is no valid date.", 'greyd_forms' ), $val );
					}
					break;

				case 'time':
					if ( ! preg_match( self::$rules['time'], $val ) ) {
						$errors[] = sprintf( __( "'%s' is no valid time.", 'greyd_forms' ), $val );
					}
					break;

				case 'tel':
					$val = (string) $val;
					if ( ! preg_match( self::$rules['phone'], $val ) ) {
						$errors[] = sprintf( __( "'%s' is no valid phone number.", 'greyd_forms' ), $val );
					}
					break;

				case 'range':
					if ( ! is_numeric( $val ) ) {
						$errors[] = sprintf( __( "The value for '%s' must be numeric.", 'greyd_forms' ), $name );
					} else {
						$num_val      = floatval( $val );
						$restrictions = $expected_input['restrictions'];
						if ( $num_val < $restrictions['min'] ) {
							$errors[] = sprintf( __( "The value for '%1\$s' must be at least %2\$s.", 'greyd_forms' ), $name, $restrictions['min'] );
						} elseif ( $num_val > $restrictions['max'] ) {
							$errors[] = sprintf( __( "The value for '%1\$s' must not be greater than %2\$s.", 'greyd_forms' ), $name, $restrictions['max'] );
						}
					}
					break;

				case 'radio':
				case 'dropdown':
					$val     = strip_tags( $val );
					$values  = explode( ',', $val );
					$options = $expected_input['options'];

					foreach ( (array) $values as $k => $_val ) {
						if ( !isset( $options[$_val] ) && !in_array( $_val, $options ) ) {
							do_action( 'formhandler_log', $options, 'options for ' . $name );
							$errors[] = sprintf( __( "'%1\$s' is not a valid input for '%2\$s'.", 'greyd_forms' ), $_val, $name );
						}
					}
					break;

				case 'checkbox':
					if ( $val === 'true' ) {
						$val = '[on] ';
					} elseif ( $val === 'on' ) {
						$val = '[on] ';
					} elseif ( $val === 'false' ) {
						$val = '[off] ';
					} elseif ( $val === 'off' ) {
						$val = '[off] ';
					} else {
						$errors[] = sprintf( __( "The value for '%1\$s' must be 'on' or 'off'. (is: '%2\$s')", 'greyd_forms' ), $name, $val );
					}

					// add checkbox text
					if ( $val == '[on] ' || $val == '[off] ' ) {
						$val .= isset( $expected_input['text'] ) ? $expected_input['text'] : '';
					}

					break;

				case 'upload':
					// file info
					$file     = $file_data[ $name ];
					$val      = $file['name'];
					$filesize = $file['size'] / ( 1000 * 1000 );
					$tmp_path = $file['tmp_name'];

					// restrictions
					$restrictions = $expected_input['restrictions'];
					$maxsize      = floatval( str_replace( ',', '.', $restrictions['size'] ) );
					$filetype     = mime_content_type( $tmp_path );
					$extension    = pathinfo( $file['name'], PATHINFO_EXTENSION );
					$req_filetype = $restrictions['filetype'] !== 'other' ? '/' . str_replace( array( '/', '*' ), array( '\/', '.*' ), $restrictions['filetype'] ) . '/' : $restrictions['custom'];

					// check for errors
					if ( $file['error'] > 0 ) {
						$file_errors = array(
							0 => __( "The file has been uploaded successfuly.", 'greyd_forms' ),
							1 => __( "The uploaded file exceeds the allowed file limit of the server.", 'greyd_forms' ),
							2 => __( "The uploaded file exceeds the allowed file size of the HTML form.", 'greyd_forms' ),
							3 => __( "The uploaded file was uploaded only partially.", 'greyd_forms' ),
							4 => __( "No file was uploaded.", 'greyd_forms' ),
							6 => __( "Missing a temporary folder.", 'greyd_forms' ),
							7 => __( "Failed to save the file.", 'greyd_forms' ),
							8 => __( "The file was stopped during the upload process.", 'greyd_forms' ),
						);
						$errors[]    = $file_errors[ $file['error'] ];
					}

					// check if filesize not too big
					if ( ! empty( $maxsize ) && $filesize > $maxsize ) {
						$errors[] = sprintf( __( "The file must not be larger than %1\$s MB. (File size: %2\$s MB)", 'greyd_forms' ), $maxsize, round( $filesize, 1 ) );
					}

					// check filetype
					if ( preg_match( $req_filetype, $filetype ) ) {
						// mimetype is valid
					} elseif ( strpos( $req_filetype, $extension ) !== false ) {
						// extension is valid
					} else {
						$errors[] = sprintf( __( "The selected file (%1\$s) does not match the specified file format (%2\$s).", 'greyd_forms' ), $val, $req_filetype );
					}

					break;
				case 'password':
					// TODO: check password strength and restrictions here
					// if (!preg_match(self::$rules['password'], $val)) $errors[] = sprinf(__("'%s' is no valid password.", 'greyd_forms'), $val);
					break;

				default: // text
					if ( ! is_string( $val ) ) {
						$errors[] = sprintf( __( "'%s' is no valid text format.", 'greyd_forms' ), $val );
					}
			}

			// check max- & minlength
			$maxlength = isset( $expected_input['maxlength'] ) ? $expected_input['maxlength'] : null;
			$minlength = isset( $expected_input['minlength'] ) ? $expected_input['minlength'] : null;
			$strlength = strlen( $val );
			if ( ! empty( $maxlength ) ) {
				if ( $strlength > $maxlength ) {
					$errors[] = sprintf(
						__( "Your input for the field '%1\$s' is too long, it must not exceed %2\$d characters.", 'greyd_forms' ),
						$name,
						$maxlength
					);
				}
			}
			if ( ! empty( $minlength ) ) {
				if ( $strlength < $minlength ) {
					$errors[] = sprintf(
						__( "Your input for the field '%1\$s' is too short, it must be at least %2\$d characters.", 'greyd_forms' ),
						$name,
						$minlength
					);
				}
			}

			// sanitize
			$val           = preg_replace( '{\\\}', '', $val );
			$data[ $name ] = htmlspecialchars( $val );
		}

		/**
		 * @filter greyd_forms_required_fields
		 *
		 * @param array  $required_fields   Array of required field names.
		 * @param int    $post_id           WP_Post ID of the form.
		 * @param array  $data              Validated form data.
		 *
		 * @return array $required_fields
		 */
		$requirements = apply_filters( 'greyd_forms_required_fields', Helper::get_all_required_fields( $post_id ), $post_id, $data );

		do_action( 'formhandler_log', $requirements, 'required fields of the form' );

		foreach ( $requirements as $k => $required_field ) {
			if (
				! isset( $data[ $required_field ] ) || // not set
				$data[ $required_field ] === false || // false
				$data[ $required_field ] === '' || // empty string
				$data[ $required_field ] === array() || // empty array
				substr( $data[ $required_field ], 0, 6 ) === '[off] ' // unchecked checkbox
			) {
				$errors[] = sprintf( __( "The field '%s' is mandatory.", 'greyd_forms' ), $required_field );
			}
		}

		// debug
		// do_action('formhandler_error', $data );

		/**
		 * Check Google reCAPTCHA score
		 */
		if ( isset( $data['reCAPTCHA'] ) ) {
			$threshold = get_option( 'captcha_thres' );
			$threshold = ! empty( $threshold ) ? floatval( $threshold ) : 0.5;

			$captcha_response = self::check_google_reCAPTCHA( $data['reCAPTCHA'] );

			// response is a score
			if ( is_float( $captcha_response ) ) {
				// check if score is high enough
				if ( floatval( $captcha_response ) >= $threshold ) {
					$data['reCAPTCHA'] = sprintf( __( "Successfull (Score: %s of 1)", 'greyd_forms' ), strval( $captcha_response ) ); // wird nicht angezeigt
				} else {
					$errors[] = sprintf(
						__( "Conspicuous behaviour on the site has been discovered. reCAPTCHA could not rate you as safe (evaluation: %s of 1)", 'greyd_forms' ),
						$captcha_response
					);
				}
				// there was an error
			} else {
				$captcha_errors = '';
				foreach ( (array) $captcha_response as $code ) {
					$captcha_errors .= $captcha_errors === '' ? $code : ', ' . $code;
				}
				$errors[] = sprintf(
					__( "The reCAPTCHA validation was not successful (description: „%s“). Please try again or contact the administrator.", 'greyd_forms' ),
					$captcha_errors
				);
			}
		}

		/**
		 * @filter greyd_forms_handler_errors
		 *
		 * @param array $errors   Current form errors.
		 * @param array $data     Validated form data.
		 * @param int   $post_id  WP_Post ID of the form.
		 *
		 * @return array $errors
		 */
		$errors = apply_filters( 'greyd_forms_handler_errors', $errors, $data, $post_id );

		/**
		 * Exit if any errors occured up to this point.
		 */
		if ( count( $errors ) > 0 ) {
			do_action( 'formhandler_error', $errors );
		}

		/**
		 * @filter greyd_forms_validated_data
		 *
		 * @param array  $data              Validated form data.
		 * @param int    $post_id           WP_Post ID of the form.
		 *
		 * @return array $data
		 */
		$data = apply_filters( 'greyd_forms_validated_data', $data, $post_id );

		/**
		 * @action formhandler_before_entry
		 *
		 * @param array  $data      Validated form data.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param string $host      User IP address.
		 */
		do_action( 'formhandler_before_entry', $post_id, $data, $host, $form_mode );

		/**
		 *  GENERATE TOKEN
		 *  unique by:
		 *  form + IP + submitted data
		 *  --> no data can be submitted twice
		 */
		$user_data = $data;
		if ( isset( $user_data['reCAPTCHA'] ) ) {
			unset( $user_data['reCAPTCHA'] ); // create token without captcha (could be different every time)
		}
		$user_data   = json_encode( $user_data, JSON_UNESCAPED_UNICODE );
		$token       = wp_create_nonce( 'greyd_tp_forms_' . $post_id . $host . $user_data );
		self::$token = $token;
		// die($token);

		do_action( 'formhandler_log', $token, 'Generated token' );

		// Check if entry already exists
		$exists = Helper::get_entry_by_token( $token );
		if ( $exists ) {

			/**
			 * Enable dulicate entries.
			 *
			 * @since 1.6.0
			 *
			 * @filter 'greyd_forms_enable_duplicate_entries'
			 *
			 * @param bool   $enabled   Whether this is enabled (default: false).
			 * @param array  $data      Validated form data.
			 * @param int    $post_id   WP_Post ID of the form.
			 * @param string $host      User IP address.
			 * @param string $token     Generated form token.
			 */
			$enable_duplicate_entries = apply_filters( 'greyd_forms_enable_duplicate_entries', false, intval( $post_id ), (array) $data, strval( $host ), strval( $token ) );

			if ( ! $enable_duplicate_entries ) {
				do_action(
					'formhandler_error',
					__( "You have already submitted this request. Please try again or contact the administrator.", 'greyd_forms' )
				);
			}
		}
		// Save as entry-post
		if ( ! $exists ) {
			$result = self::save_form_as_entry( $post_id, $data, $host, $token, $file_data );
			if ( $result !== true ) {

				/**
				 * Send the admin an error email.
				 *
				 * @since 1.4.1
				 */
				ob_start();
				echo __( "The submitted form could not be saved in the database. Please try again or contact the administrator", 'greyd_forms' );
				debug( $_REQUEST );
				$message = ob_get_contents();
				ob_end_clean();
				self::send_admin_error_mail( $post_id, $message );

				do_action(
					'formhandler_error',
					__( "The submitted form could not be saved in the database. Please try again or contact the administrator", 'greyd_forms' )
				);
			} else {
				Helper::log_entry_state( self::$entry_id, 'created', 'success' );
			}
		}
		// --> post_id of entry is saved as self::$entry_id

		/**
		 * @action formhandler_after_entry
		 *
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $host      User IP address.
		 * @param string $form_mode Form mode.
		 */
		do_action( 'formhandler_after_entry', $post_id, $data, $host, $form_mode );

		// check if 2-opt-in is activated
		$verify_user = get_post_meta( $post_id, 'verify_user', true ) == 'verify_user' ? true : false;

		// on: send optin-mail
		if ( $verify_user ) {
			$result = self::send_optin_mail( $post_id, $data, $token );
			// do_action('formhandler_error', $result ); // debug
			if ( $result !== true ) {
				$msg = __( "The verification mail could not be sent.", 'greyd_forms' );
				if ( is_wp_error( $result ) ) {
					$msg .= ' ' . $result->get_error_message();
				} elseif ( is_string( $return ) ) {
					$msg .= ' ' . $return;
				}
				Helper::log_entry_state( self::$entry_id, $msg );

				/**
				 * Send the admin an error email.
				 *
				 * @since 1.4.1
				 */
				self::send_admin_error_mail( $post_id, $msg );

				do_action( 'formhandler_error', $msg );
			} else {
				Helper::log_entry_state( self::$entry_id, 'verify_send', 'success' );
			}
		}
		// off: verify directly
		else {
			$result = self::optin( $token, true );
			if ( $result !== true ) {

				/**
				 * Send the admin an error email.
				 *
				 * @since 1.4.1
				 */
				self::send_admin_error_mail( $post_id, __( "The confirmation mail could not be sent.", 'greyd_forms' ) );

				do_action( 'formhandler_error', __( "The confirmation mail could not be sent.", 'greyd_forms' ) );
			}
		}

		/**
		 * Perform an action after the mails were sent before the user
		 * is verified.
		 *
		 * @action formhandler_after_mails
		 *
		 * If you want to call a function after the user is verified,
		 * use the action @see greyd_forms_action_after_optin for this. Find
		 * more details her: self::optin().
		 *
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $host      User IP address.
		 */
		do_action( 'formhandler_after_mails', $post_id, $data, $host, $form_mode );

		// Give back result to frontend (default.js)
		if ( $result == true ) {
			do_action( 'formhandler_success', array( 'post_id' => $post_id ) );
		} else {
			do_action( 'formhandler_error', $result );
		}
	}

	// die and send error
	public static function send_error( $message = '' ) {
		if ( $message === '' ) {
			$message = Helper::get_default_error();
		}
		$response             = array();
		$response['response'] = 'ERROR';
		$response['errors']   = $message;
		die( 'greyd_form_response::' . json_encode( $response ) );
	}

	// exit and send success
	public static function send_success( $args = array(
		'post_id' => '',
		'message' => '',
	) ) {

		if ( isset( $args['post_id'] ) ) {
			$args['message'] = self::get_action_after( $args['post_id'] );
		}
		$message = $args['message'] === '' || ! isset( $args['message'] ) ? __( "Thank you very much. Your data was submitted successfully.", 'greyd_forms' ) : $args['message'];

		$response             = (array) $args;
		$response['response'] = 'SUCCESS';
		$response['message']  = $message;
		exit( 'greyd_form_response::' . json_encode( $response ) );
	}

	// validate Google reCAPTCHA
	public static function check_google_reCAPTCHA( $val ) {
		$secret = get_option( 'captcha_secret', '' );

		$post     = wp_safe_remote_post(
			'https://www.google.com/recaptcha/api/siteverify',
			array(
				'body' => array(
					'secret'   => $secret,
					'response' => $val,
					'remoteip' => $_SERVER['REMOTE_ADDR'],
				),
			)
		);
		$response = json_decode( $post['body'], true );
		// debug
		// do_action('formhandler_error', $response );

		if ( $response['success'] == false ) {
			return $response['error-codes'];
		} else {
			return $response['score'];
		}
	}

	/**
	 * Log a debug message to the frontend console if WP_DEBUG is true
	 *
	 * @action 'formhandler_log'
	 */
	public static function log_debug( $log_this, $description = '' ) {
		if ( ! WP_DEBUG ) {
			return;
		}

		echo "------\n\r";
		if ( ! empty( $description ) ) {
			echo strtoupper( $description ) . ":\n\r\n\r";
		}
		print_r( $log_this );
		echo "\n\r\n\r";
	}


	/*
	=======================================================================

								ACTON AFTER

	=======================================================================
	*/

	/**
	 * Save form data as posttype entry
	 */
	public static function save_form_as_entry( $post_id, $form_data, $host, $token, $file_data ) {

		/**
		 * Save form_data as post_content
		 *
		 * @deprecated 0.9.1
		 */
		// foreach((array)$form_data as $key => $val) {
		// $form_data[$key] = preg_replace('/\t/', "\\\\t", preg_replace('/\r/', "\\\\r", preg_replace('/\n/', "\\\\n", $val)));
		// }
		// // $form_data = array_map('utf8_encode', $form_data);
		// $content = json_encode($form_data, JSON_UNESCAPED_UNICODE);

		// insert the post
		$entry_id = wp_insert_post(
			array(
				'post_title'  => get_the_title( $post_id ),
				'post_type'   => self::$config->plugin_post_type . '_entry',
				'post_status' => 'publish',
			// 'post_content'    => $content, // @depecated 0.9.1
			)
		);

		if ( is_wp_error( $entry_id ) ) {
			// there was an error in the post insertion --> return the error
			return $entry_id->get_error_message();
		}

		// save entry id in class var
		self::$entry_id = $entry_id;

		// handle user uploads
		if ( ! empty( $file_data ) ) {
			// do_action('formhandler_error', $file_data );

			$uploads    = array();
			$upload_dir = wp_upload_dir();

			// basic upload folder "/uploads/greyd_forms"
			$upload_path = $upload_dir['basedir'] . '/greyd_forms';
			$upload_url  = $upload_dir['baseurl'] . '/greyd_forms';
			if ( ! file_exists( $upload_path ) ) {
				mkdir( $upload_path );
			}

			// form based upload folder
			$upload_path .= '/form_' . $post_id;
			$upload_url  .= '/form_' . $post_id;
			if ( ! file_exists( $upload_path ) ) {
				mkdir( $upload_path );
			}

			foreach ( $file_data as $key => $file_info ) {

				$filetype       = $file_info['type'];
				$orig_file_name = $file_info['name'];
				$temp_file_name = $file_info['tmp_name'];
				$new_file_name  = preg_replace( '/[\s\.]+/', '_', pathinfo( $orig_file_name, PATHINFO_FILENAME ) ) . '__' . uniqid() . '.' . pathinfo( $orig_file_name, PATHINFO_EXTENSION );
				$new_file_path  = $upload_path . '/' . $new_file_name;
				$new_file_url   = $upload_url . '/' . $new_file_name;

				// move the file to the uploads folder
				$uploaded = move_uploaded_file( $temp_file_name, $new_file_path );

				if ( $uploaded ) {

					// replace in data meta array
					if ( isset( $form_data[ $key ] ) ) {
						$form_data[ $key ] = $new_file_url;
					}

					// save in uploads meta array
					$uploads[ $key ] = array(
						'name' => $new_file_name,
						'path' => explode( $upload_dir['baseurl'], $new_file_url )[1],
						'type' => $filetype,
					);
				}
			}

			update_post_meta( $entry_id, 'tp_uploads', $uploads );
		}

		// Update post meta
		update_post_meta( $entry_id, 'tp_form_id', $post_id );
		update_post_meta( $entry_id, 'tp_host', $host );
		update_post_meta( $entry_id, 'tp_token', $token );

		/**
		 * Save form data in meta 'entry_form_data'
		 *
		 * @since 0.9.1
		 */
		update_post_meta( $entry_id, 'entry_form_data', $form_data );

		return true;
	}

	// return action after
	public static function get_action_after( $post_id ) {
		$return = '';

		$after = get_post_meta( $post_id, 'after_form', true );
		if ( $after == 'popup' || $after == 'fullscreen' ) {
			// Pop-up
			$return = $after;

		} elseif ( $after == 'page' ) {
			// Other Page
			$jump_to = get_post_meta( $post_id, 'jump_to', true );
			if ( $jump_to == '--' ) {
				$link = '#';
			} elseif ( $jump_to == 'STARTPAGE' ) {
				$link = '/';
			} elseif ( $jump_to == 'LINK' ) {
				$link = get_post_meta( $post_id, 'jump_to_link', true );
			} else {
				$link = get_permalink( $jump_to );
			}

			$return = $after . '=' . $link;

		} else {
			if ( ! empty( get_post_meta( $post_id, 'success_message', true ) ) ) {
				$return = get_post_meta( $post_id, 'success_message', true );
			}
		}
		return $return;
	}


	/*
	=======================================================================

								OPT-IN MAIL

	=======================================================================
	*/

	/**
	 * Send 2-opt-in mail to user
	 */
	public static function send_optin_mail( $post_id, $data, $token ) {

		$mail_to_field = get_post_meta( $post_id, 'verify_email', true );
		$mail_to_field = Helper::is_select_empty( $mail_to_field ) ? get_post_meta( $post_id, 'user_email', true ) : $mail_to_field;
		$mail_to       = isset( $data[ $mail_to_field ] ) ? $data[ $mail_to_field ] : null;

		/**
		 * @filter greyd_forms_optin_mail_to
		 *
		 * @param string $mail_to   Full mail mail_to.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_to
		 */
		$mail_to = apply_filters( 'greyd_forms_optin_mail_to', $mail_to, $post_id, $data, $token );

		if ( empty( $mail_to ) ) {
			return false;
		}

		// content
		$content = get_post_meta( $post_id, 'verify_mail_content', true );
		if ( strpos( $content, '[opt_in_link]' ) === false ) {
			$opt_in_link = '<br><br>' . Helper::build_optin_link( $post_id, $data, $token );
		} else {
			$opt_in_link = '';
		}
		if ( ! empty( $content ) ) {
			$content = Helper::replace_form_data_tags( $content, $data, $post_id, $token );
			$content = html_entity_decode( nl2br( $content ) ) . $opt_in_link;
		} else {
			$content = __( "Thank you very much.<br><br>Please click on the following link to confirm your identity:", 'greyd_forms' ) . $opt_in_link;
		}

		/**
		 * @filter greyd_forms_optin_mail_content
		 *
		 * @param string $content   Full mail HTML content.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $content
		 */
		$content = apply_filters( 'greyd_forms_optin_mail_content', $content, $post_id, $data, $entry_id );

		// subject
		$subject = get_post_meta( $post_id, 'verify_mail_subject', true );
		$subject = ! empty( $subject ) ? Helper::replace_form_data_tags( $subject, $data, $post_id, $token ) : __( "Verify your identity", 'greyd_forms' );

		/**
		 * @filter greyd_forms_optin_mail_subject
		 *
		 * @param string $subject   Full mail subject.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $subject
		 */
		$subject = apply_filters( 'greyd_forms_optin_mail_subject', $subject, $post_id, $data, $entry_id );

		// mail from
		$mail_from      = Helper::replace_form_data_tags( get_post_meta( $post_id, 'verify_mail_from', true ), $data, $post_id, $token );
		$mail_from_name = Helper::replace_form_data_tags( get_post_meta( $post_id, 'verify_mail_from_name', true ), $data, $post_id, $token );
		if ( empty( $mail_from ) ) {
			$sitename = wp_parse_url( network_home_url(), PHP_URL_HOST );
			if ( 'www.' === substr( $sitename, 0, 4 ) ) {
				$sitename = substr( $sitename, 4 );
			}
			// if (filter_var($sitename, FILTER_VALIDATE_URL) === false) {
			// $sitename = "wordpress.org";
			// }
			$mail_from = "wordpress@{$sitename}";
		}

		/**
		 * @filter greyd_forms_optin_mail_from
		 *
		 * @param string $mail_from Full mail mail_from.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_from
		 */
		$mail_from = apply_filters( 'greyd_forms_optin_mail_from', $mail_from, $post_id, $data, $token );

		/**
		 * @filter greyd_forms_optin_mail_from_name
		 *
		 * @param string $mail_from_name    Full mail mail_from_name.
		 * @param int    $post_id           WP_Post ID of the form.
		 * @param array  $data              Validated form data.
		 * @param string $token             Entry token
		 *
		 * @return string $mail_from_name
		 */
		$mail_from_name = apply_filters( 'greyd_forms_optin_mail_from_name', $mail_from_name, $post_id, $data, $token );

		// headers
		$headers = array();
		if ( empty( $mail_from_name ) ) {
			$headers[] = "From: $mail_from";
		} else {
			$headers[] = "From: $mail_from_name <$mail_from>\r\n";
		}

		/**
		 * @filter greyd_forms_optin_mail_headers
		 * @see https://developer.wordpress.org/reference/functions/wp_mail/
		 *
		 * @param array  $headers   Full mail headers.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return array $headers
		 */
		$headers = apply_filters( 'greyd_forms_optin_mail_headers', $headers, $post_id, $data, $token );

		// send Mail
		$return = self::greyd_mail( $mail_to, $subject, $content, $headers );

		return $return;
	}


	/*
	=======================================================================

								OPT-IN & -OUT

	=======================================================================
	*/
	/**
	 * log state of optin & optout
	 *
	 * false-found     |   no entry with this token found
	 * false-out       |   cannot verify because state is already optout (optin only)
	 * true-already    |   entry found but action has been done already
	 * true-just       |   entry found and action was usccessfull
	 */
	public static $opt_state;

	/**
	 * Handle url-params from mail link
	 */
	public static function handle_mail_link() {

		/**
		 * Don't allow bots to verify the identity of a user.
		 */
		if ( Helper::is_bot() ) {
			return false;
		}

		// early exit
		if ( ! isset( $_GET['token'] ) && ! isset( $_GET['optin'] ) && ! isset( $_GET['optout'] ) ) {
			return false;
		}

		// opt in
		$token = isset( $_GET['token'] ) ? $_GET['token'] : ( isset( $_GET['optin'] ) ? $_GET['optin'] : '' );
		if ( isset( $token ) && ! empty( $token ) ) {
			self::optin( $token, false );
			// modify value of url-param
			if ( ! empty( self::$opt_state ) ) {
				add_filter(
					'greyd_set_url_param-optin',
					function() {
						return self::$opt_state;
					}
				);
			}
		}

		// opt out
		$optout = isset( $_GET['optout'] ) ? $_GET['optout'] : '';
		if ( ! empty( $optout ) ) {
			self::optout( $optout );
			// modify value of url-param
			if ( ! empty( self::$opt_state ) ) {
				add_filter(
					'greyd_set_url_param-optout',
					function() {
						return self::$opt_state;
					}
				);
			}
		}

		// redirect after
		if ( isset( $_GET['forms_redirect'] ) ) {
			wp_redirect( urldecode( $_GET['forms_redirect'] ) );
			exit;
		}
	}

	/**
	 * Opt-in
	 */
	public static function optin( $token, $direct = false ) {

		// get entry
		$entry_id = Helper::get_entry_by_token( $token );
		if ( ! $entry_id ) {
			self::$opt_state = 'false-found';
			return false; // early exit
		}
		self::$entry_id = $entry_id;

		// if it is already verified or aborted
		if ( Helper::check_if_entry_is( $entry_id, 'bounced' ) ) {
			self::$opt_state = 'false-out';
			return false; // early exit
		}
		if ( Helper::check_if_entry_is( $entry_id, 'verified' ) ) {
			self::$opt_state = 'true-already';
			return false; // early exit
		}

		// update verified time in postmeta
		if ( ! $direct ) {
			Helper::log_entry_state( $entry_id, 'verified', 'success' );
		} else {
			Helper::log_entry_state( $entry_id, 'no_doi', 'info' );
		}
		self::$opt_state = 'true-just';

		return self::do_actions_after_optin( $token, $entry_id );
	}

	/**
	 * Do all the actions after the doi opt-in.
	 *
	 * @since 1.4.5
	 *
	 * @param string $token     Entry token (required).
	 * @param int    $entry_id     Entry ID (optional).
	 * @return bool
	 */
	public static function do_actions_after_optin( $token, $entry_id = 0 ) {

		$return = true;

		// get entry
		if ( ! $entry_id ) {
			$entry_id = Helper::get_entry_by_token( $token );
			if ( ! $entry_id ) {
				return false; // early exit
			}
		}
		self::$entry_id = $entry_id;

		// get all the data
		$entry_data = Helper::get_entry_data_by_token( $token );
		if ( empty( $entry_data ) || ! is_array( $entry_data ) ) {
			return false;
		}

		$form_id       = $entry_data['form_id'];
		self::$form_id = $form_id;
		$form_data     = $entry_data['data'];

		// send registration mail to user
		if ( $entry_data['user_gets_mail'] ) {
			$result = self::send_registration_mail( $form_id, $form_data, $token );
			if ( $result !== true ) {
				Helper::log_entry_state( $entry_id, __( "The confirmation mail could not be sent.", 'greyd_forms' ) );
				$return = false; // return error to frontend
			} else {
				Helper::log_entry_state( $entry_id, __( "The confirmation mail was successfully sent.", 'greyd_forms' ), 'success', false );
			}
		}

		// send admin mails
		$result = self::send_admin_mail( $form_id, $form_data, $token, $entry_id );
		if ( $result !== true ) {
			Helper::log_entry_state( $entry_id, __( "The admin mail(s) could not be sent.", 'greyd_forms' ), 'error', false );
		} else {
			Helper::log_entry_state( $entry_id, __( "The admin mail(s) were successfully sent.", 'greyd_forms' ), 'success', false );
		}

		/**
		 * Perform an action after the form was verified.
		 *
		 * This hook is a good way to trigger your own actions after the doi is verified.
		 * That's especially usefull for custom API-calls.
		 *
		 * @action greyd_forms_action_after_optin
		 *
		 * @param array  $form_data    Validated form data.
		 * @param int    $form_id      WP_Post ID of the form.
		 * @param int    $entry_id     WP_Post ID of the entry.
		 * @param array  $entry_data   Entire data of the entry.
		 */
		do_action( 'greyd_forms_action_after_optin', (array) $form_data, intval( $form_id ), intval( $entry_id ), (array) $entry_data );

		/** @deprecated since 1.6.0 */
		do_action( 'formhandler_after_doi', $token );

		return $return;
	}

	/**
	 * Opt-out
	 */
	public static function optout( $token ) {

		$entry_id = Helper::get_entry_by_token( $token );
		if ( ! $entry_id ) {
			self::$opt_state = 'false-found';
			return false; // early exit
		}
		self::$entry_id = $entry_id;

		// $aborted_time = get_post_meta($entry_id, 'abort_time', true);
		if ( Helper::check_if_entry_is( $entry_id, 'bounced' ) ) {
			self::$opt_state = 'true-already';
			return false; // early exit
		} else {
			// update_post_meta($entry_id, 'abort_time', current_time( 'mysql' ) );
			Helper::log_entry_state( $entry_id, 'bounced' );
			self::$opt_state = 'true-just';
		}

		// enable api functions
		do_action( 'formhandler_after_optout', $entry_id );

		return true;
	}


	/*
	=======================================================================

								REGISTRATION MAILS

	=======================================================================
	*/

	/**
	 * Send registration mail to user
	 */
	public static function send_registration_mail( $post_id, $data, $token ) {

		$mail_to_field = get_post_meta( $post_id, 'user_email', true );
		$mail_to_field = Helper::is_select_empty( $mail_to_field ) ? get_post_meta( $post_id, 'verify_email', true ) : $mail_to_field;
		$mail_to       = isset( $data[ $mail_to_field ] ) ? $data[ $mail_to_field ] : null;

		/**
		 * @filter greyd_forms_registration_mail_to
		 *
		 * @param string $mail_to   Full mail mail_to.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_to
		 */
		$mail_to = apply_filters( 'greyd_forms_registration_mail_to', $mail_to, $post_id, $data, $token );

		if ( empty( $mail_to ) ) {
			return false; // early exit
		}

		// build content
		$content      = get_post_meta( $post_id, 'user_mail_content', true );
		$send_opt_out = get_post_meta( $post_id, 'send_opt_out', true ) == 'send_opt_out' ? true : false;
		if ( $send_opt_out && strpos( $content, '[opt_out_link]' ) === false ) {
			$opt_out_link = '<br><br>' . Helper::build_optout_link( $post_id, $data, $token );
		} else {
			$opt_out_link = '';
		}
		if ( ! empty( $content ) ) {
			$content = Helper::replace_form_data_tags( $content, $data, $post_id, $token );
			$content = html_entity_decode( nl2br( $content ) ) . $opt_out_link;
		} else {
			$content = __( "Thank you for submitting your request!", 'greyd_forms' ) . $opt_out_link;
		}

		/**
		 * @filter greyd_forms_registration_mail_content
		 *
		 * @param string $content   Full mail HTML content.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $content
		 */
		$content = apply_filters( 'greyd_forms_registration_mail_content', $content, $post_id, $data, $token );

		// set subject
		$subject = get_post_meta( $post_id, 'user_mail_subject', true );
		$subject = ! empty( $subject ) ? Helper::replace_form_data_tags( $subject, $data, $post_id, $token ) : __( "Thank you very much", 'greyd_forms' );

		/**
		 * @filter greyd_forms_registration_mail_subject
		 *
		 * @param string $subject   Full mail subject.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $subject
		 */
		$subject = apply_filters( 'greyd_forms_registration_mail_subject', $subject, $post_id, $data, $token );

		// mail from
		$mail_from      = Helper::replace_form_data_tags( get_post_meta( $post_id, 'user_mail_from', true ), $data, $post_id, $token );
		$mail_from_name = Helper::replace_form_data_tags( get_post_meta( $post_id, 'user_mail_from_name', true ), $data, $post_id, $token );
		if ( empty( $mail_from ) ) {
			$sitename = wp_parse_url( network_home_url(), PHP_URL_HOST );
			if ( 'www.' === substr( $sitename, 0, 4 ) ) {
				$sitename = substr( $sitename, 4 );
			}
			// if (filter_var($sitename, FILTER_VALIDATE_URL) === false) {
			// $sitename = "wordpress.org";
			// }
			$mail_from = "wordpress@{$sitename}";
		}

		/**
		 * @filter greyd_forms_registration_mail_from
		 *
		 * @param string $mail_from Full mail mail_from.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_from
		 */
		$mail_from = apply_filters( 'greyd_forms_registration_mail_from', $mail_from, $post_id, $data, $token );

		/**
		 * @filter greyd_forms_registration_mail_from_name
		 *
		 * @param string $mail_from_name    Full mail mail_from_name.
		 * @param int    $post_id           WP_Post ID of the form.
		 * @param array  $data              Validated form data.
		 * @param string $token             Entry token
		 *
		 * @return string $mail_from_name
		 */
		$mail_from_name = apply_filters( 'greyd_forms_registration_mail_from_name', $mail_from_name, $post_id, $data, $token );

		// set headers
		$headers = array();
		if ( empty( $mail_from_name ) ) {
			$headers[] = "From: $mail_from";
		} else {
			$headers[] = "From: $mail_from_name <$mail_from>\r\n";
		}

		/**
		 * @filter greyd_forms_registration_mail_headers
		 * @see https://developer.wordpress.org/reference/functions/wp_mail/
		 *
		 * @param array  $headers   Full mail headers.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return array $headers
		 */
		$headers = apply_filters( 'greyd_forms_registration_mail_headers', $headers, $post_id, $data, $token );

		// send Mail
		$return = self::greyd_mail( $mail_to, $subject, $content, $headers );

		return $return;
	}

	/**
	 * Send admin mail
	 */
	public static function send_admin_mail( $post_id, $data, $token, $entry_id ) {

		$mail_to = get_post_meta( $post_id, 'mail_to', true );
		$mail_to = Helper::replace_form_data_tags( $mail_to, $data, $post_id, $token );

		/**
		 * @filter greyd_forms_admin_mail_to
		 *
		 * @param string $mail_to   Full mail mail_to.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_to
		 */
		$mail_to = apply_filters( 'greyd_forms_admin_mail_to', $mail_to, $post_id, $data, $token );

		if ( empty( $mail_to ) ) {
			return true; // If no mail is filled out -> return
		}

		// set mail content
		$mail_title = sprintf(
			__( "On your WordPress page \"%1\$s\" the form \"%2\$s\" was filled in:", 'greyd_forms' ),
			get_bloginfo( 'name' ),
			get_the_title( $post_id )
		) . '<br><br>';
		$mail_note  = get_post_meta( $post_id, 'mail_note', true );
		$mail_note  = ! empty( $mail_note ) ? '<i>' . html_entity_decode( nl2br( get_post_meta( $post_id, 'mail_note', true ) ) ) . '</i><br><br>' : '';
		$links      = "<br><br><a href='" . wp_login_url() . "'>" . __( "login to WordPress", 'greyd_forms' ) . '</a>';
		$links     .= " | <a href='" . get_edit_post_link( $entry_id ) . "'>" . __( "go to entry", 'greyd_forms' ) . '</a>';
		$content    = Helper::formdata2html( $data, $entry_id );
		$content    = $mail_title . $mail_note . $content . $links;

		/**
		 * @filter greyd_forms_admin_mail_content
		 *
		 * @param string $content   Full mail HTML content.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $content
		 */
		$content = apply_filters( 'greyd_forms_admin_mail_content', $content, $post_id, $data, $token );

		// set subject
		$subject = get_post_meta( $post_id, 'subject', true );
		$subject = ! empty( $subject ) ? Helper::replace_form_data_tags( $subject, $data, $post_id, $token ) : __( "A form was submitted", 'greyd_forms' );

		/**
		 * @filter greyd_forms_admin_mail_subject
		 *
		 * @param string $subject   Full mail subject.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $subject
		 */
		$subject = apply_filters( 'greyd_forms_admin_mail_subject', $subject, $post_id, $data, $token );

		// set mail from
		$mail_from      = Helper::replace_form_data_tags( get_post_meta( $post_id, 'mail_from', true ), $data, $post_id, $token );
		$mail_from_name = Helper::replace_form_data_tags( get_post_meta( $post_id, 'mail_from_name', true ), $data, $post_id, $token );
		if ( empty( $mail_from ) ) {
			$sitename = wp_parse_url( network_home_url(), PHP_URL_HOST );
			if ( 'www.' === substr( $sitename, 0, 4 ) ) {
				$sitename = substr( $sitename, 4 );
			}
			// if (filter_var($sitename, FILTER_VALIDATE_URL) === false) {
			// $sitename = "wordpress.org";
			// }
			$mail_from = "wordpress@{$sitename}";
		}

		/**
		 * @filter greyd_forms_admin_mail_from
		 *
		 * @param string $mail_from Full mail mail_from.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return string $mail_from
		 */
		$mail_from = apply_filters( 'greyd_forms_admin_mail_from', $mail_from, $post_id, $data, $token );

		/**
		 * @filter greyd_forms_admin_mail_from_name
		 *
		 * @param string $mail_from_name    Full mail mail_from_name.
		 * @param int    $post_id           WP_Post ID of the form.
		 * @param array  $data              Validated form data.
		 * @param string $token             Entry token
		 *
		 * @return string $mail_from_name
		 */
		$mail_from_name = apply_filters( 'greyd_forms_admin_mail_from_name', $mail_from_name, $post_id, $data, $token );

		// set headers
		$headers = array();
		if ( empty( $mail_from_name ) ) {
			$headers[] = "From: $mail_from";
		} else {
			$headers[] = "From: $mail_from_name <$mail_from>\r\n";
		}

		/**
		 * @filter greyd_forms_admin_mail_headers
		 * @see https://developer.wordpress.org/reference/functions/wp_mail/
		 *
		 * @param array  $headers   Full mail headers.
		 * @param int    $post_id   WP_Post ID of the form.
		 * @param array  $data      Validated form data.
		 * @param string $token     Entry token
		 *
		 * @return array $headers
		 */
		$headers = apply_filters( 'greyd_forms_admin_mail_headers', $headers, $post_id, $data, $token );

		// send Mail
		$return = self::greyd_mail( $mail_to, $subject, $content, $headers );

		return $return;
	}


	/*
	=======================================================================

								MAIL HANDLING

	=======================================================================
	*/

	/**
	 * Holds all mail errors
	 */
	public static $mail_errors = array();

	/**
	 * Set mail charset to UTF-8
	 */
	public static function utf8() {
		return 'UTF-8';
	}

	/**
	 * on wp_mail() error event
	 */
	public static function on_mail_error( $wp_error ) {
		self::$mail_errors[] = $wp_error;
	}

	/**
	 * Sends an email, similar to PHP's mail function.
	 * @see https://developer.wordpress.org/reference/functions/wp_mail/
	 *
	 * @param string|string[] $to          Array or comma-separated list of email addresses to send message.
	 * @param string          $subject     Email subject.
	 * @param string          $message     Message contents.
	 * @param string|string[] $headers     Optional. Additional headers.
	 * @param string|string[] $attachments Optional. Paths to files to attach.
	 * @return bool Whether the email was sent successfully.
	 */
	public static function greyd_mail( $to, $subject, $message, $headers = '', $attachments = array() ) {

		$content = self::create_mail_template( $subject, $message );

		// set filter
		add_filter( 'wp_mail_content_type', array( 'Greyd\Forms\Helper', 'wpdocs_set_html_mail_content_type' ) );
		add_filter( 'wp_mail_charset', array( 'Greyd\Forms\Handler', 'utf8' ) );
		add_action( 'wp_mail_failed', array( 'Greyd\Forms\Handler', 'on_mail_error' ) );

		// send Mail
		$return = wp_mail( $to, $subject, $content, $headers, $attachments );

		// Reset content-type to avoid conflicts -- https://core.trac.wordpress.org/ticket/23578
		remove_filter( 'wp_mail_content_type', array( 'Greyd\Forms\Helper', 'wpdocs_set_html_mail_content_type' ) );
		remove_filter( 'wp_mail_charset', array( 'Greyd\Forms\Handler', 'utf8' ) );
		add_action( 'wp_mail_failed', array( 'Greyd\Forms\Handler', 'on_mail_error' ) );

		if ( count( self::$mail_errors ) ) {
			return self::$mail_errors[0];
		}

		return $return;
	}

	/**
	 * Get mail template and replace placeholders.
	 * @see https://github.com/leemunroe/responsive-html-email-template
	 */
	public static function create_mail_template( $title, $body, $footer=null ) {
		
		// default footer
		$sitetitle = get_bloginfo('name');
		if ( ! $footer && !empty($sitetitle) ) {
			$footer = '<a href="'.get_home_url().'" style="color: #999999; font-size: 12px; text-align: center; text-decoration: none;">'.$sitetitle.'</a>';
			$blogdecription = get_bloginfo('description');
			if ( ! empty( $blogdecription ) ) {
				$footer .= '  |  '.$blogdecription;
			}
		}

		return '<!doctype html>
		<html>
			<head>
				<meta name="viewport" content="width=device-width, initial-scale=1.0">
				<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
				<title>Simple Transactional Email</title>
				<style>
				@media only screen and (max-width: 620px) {
					table.body h1 {
						font-size: 28px !important;
						margin-bottom: 10px !important;
					}
				
					table.body p,
					table.body ul,
					table.body ol,
					table.body td,
					table.body span,
					table.body a {
						font-size: 16px !important;
					}
				
					table.body .wrapper,
					table.body .article {
						padding: 10px !important;
					}
				
					table.body .content {
						padding: 0 !important;
					}
				
					table.body .container {
						padding: 0 !important;
						width: 100% !important;
					}
				
					table.body .main {
						border-left-width: 0 !important;
						border-radius: 0 !important;
						border-right-width: 0 !important;
					}
				
					table.body .btn table {
						width: 100% !important;
					}
				
					table.body .btn a {
						width: 100% !important;
					}
				
					table.body .img-responsive {
						height: auto !important;
						max-width: 100% !important;
						width: auto !important;
					}
				}
				
				@media all {
					.ExternalClass {
						width: 100%;
					}
				
					.ExternalClass,
					.ExternalClass p,
					.ExternalClass span,
					.ExternalClass font,
					.ExternalClass td,
					.ExternalClass div {
						line-height: 100%;
					}
				
					.apple-link a {
						color: inherit !important;
						font-family: inherit !important;
						font-size: inherit !important;
						font-weight: inherit !important;
						line-height: inherit !important;
						text-decoration: none !important;
					}
				
					#MessageViewBody a {
						color: inherit;
						text-decoration: none;
						font-size: inherit;
						font-family: inherit;
						font-weight: inherit;
						line-height: inherit;
					}
				
					.btn-primary table td:hover {
						background-color: #34495e !important;
					}
				
					.btn-primary a:hover {
						background-color: #34495e !important;
						border-color: #34495e !important;
					}
				}
				</style>
			</head>
			<body style="background-color: #f6f6f6; font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; -webkit-font-smoothing: antialiased; font-size: 16px; line-height: 1.4; margin: 0; padding: 0; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%;">
				<table role="presentation" border="0" cellpadding="0" cellspacing="0" class="body" style="border-collapse: separate; mso-table-lspace: 0pt; mso-table-rspace: 0pt; background-color: #f6f6f6; width: 100%;" width="100%" bgcolor="#f6f6f6">
				<tr>
					<td style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; font-size: 16px; vertical-align: top;" valign="top">&nbsp;</td>
					<td class="container" style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; font-size: 16px; vertical-align: top; display: block; max-width: 580px; padding: 10px; width: 580px; margin: 0 auto;" width="580" valign="top">
						<div class="content" style="box-sizing: border-box; display: block; margin: 0 auto; max-width: 580px; padding: 24px;">
				
							<!-- START TITLE -->
							<div class="title" style="clear: both; margin-top: 10px; text-align: center; width: 100%;">
								<table role="presentation" border="0" cellpadding="0" cellspacing="0" style="border-collapse: separate; mso-table-lspace: 0pt; mso-table-rspace: 0pt; width: 100%;" width="100%">
									<tr>
										<td class="content-block" style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; vertical-align: top; padding-bottom: 10px; padding-top: 10px; color: #1E1E1E; font-size: 32px; text-align: center;" valign="top" align="center"><h1 style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; color: #1E1E1E; font-size: 32px; text-align: center;">'.$title.'</h1></td>
									</tr>
								</table>
							</div>
							<!-- END TITLE -->
				
							<!-- START CENTERED WHITE CONTAINER -->
							<table role="presentation" class="main" style="border-collapse: separate; mso-table-lspace: 0pt; mso-table-rspace: 0pt; background: #ffffff; border-radius: 3px; color: #1E1E1E; width: 100%;" width="100%">
				
								<!-- START MAIN CONTENT AREA -->
								<tr>
									<td class="wrapper" style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; font-size: 16px; color: #1E1E1E; vertical-align: top; box-sizing: border-box; padding: 20px;" valign="top">
									<table role="presentation" border="0" cellpadding="0" cellspacing="0" style="border-collapse: separate; mso-table-lspace: 0pt; mso-table-rspace: 0pt; width: 100%;" width="100%">
										<tr>
										<td style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; font-size: 16px; color: #1E1E1E; vertical-align: top;" valign="top">' . $body . '</td>
										</tr>
									</table>
									</td>
								</tr>
								<!-- END MAIN CONTENT AREA -->

							</table>
							<!-- END CENTERED WHITE CONTAINER -->
				
							<!-- START FOOTER -->
							<div class="footer" style="clear: both; margin-top: 10px; text-align: center; width: 100%;">
								<table role="presentation" border="0" cellpadding="0" cellspacing="0" style="border-collapse: separate; mso-table-lspace: 0pt; mso-table-rspace: 0pt; width: 100%;" width="100%">
									<tr>
										<td class="content-block powered-by" style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; vertical-align: top; padding-bottom: 10px; padding-top: 10px; color: #999999; font-size: 12px; text-align: center;" valign="top" align="center">'.$footer.'</td>
									</tr>
								</table>
							</div>
							<!-- END FOOTER -->
				
						</div>
					</td>
					<td style="font-family: DM Sans, Inter, Helvetica, -apple-system, BlinkMacSystemFont, Roboto, Arial, sans-serif; font-size: 14px; vertical-align: top;" valign="top">&nbsp;</td>
				</tr>
				</table>
			</body>
		</html>';
	}

	/**
	 * Send an admin email because an error ocurred
	 *
	 * @param int $post_id  The form post ID.
	 * @param int $message  Content of the email.
	 */
	public static function send_admin_error_mail( $post_id, $message, $entry_id = null ) {

		$mail_to = get_post_meta( $post_id, 'mail_to', true );
		if ( empty( $mail_to ) ) {
			$mail_to = get_bloginfo( 'admin_email' );
		}

		if ( empty( $mail_to ) ) {
			return true;
		}

		$subject = __( "There were errors filling out a form.", 'greyd_forms' );

		// add headline
		$mail_title = sprintf(
			__( "On your WordPress site \"%1\$s\" there were errors while filling out the form \"%2\$s\".", 'greyd_forms' ),
			get_bloginfo( 'name' ),
			get_the_title( $post_id )
		);

		// links to WordPress
		$links  = "<br><br><a href='" . wp_login_url() . "'>" . __( "login to WordPress", 'greyd_forms' ) . '</a>';
		$links .= " | <a href='" . get_edit_post_link( $entry_id ) . "'>" . __( "go to entry", 'greyd_forms' ) . '</a>';

		// mail content
		$mail_content = $mail_title . '<br><br>' . $message . $links;

		// set filter
		add_filter( 'wp_mail_content_type', array( 'Greyd\Forms\Helper', 'wpdocs_set_html_mail_content_type' ) );
		add_filter( 'wp_mail_charset', array( 'Greyd\Forms\Handler', 'utf8' ) );
		add_action( 'wp_mail_failed', array( 'Greyd\Forms\Handler', 'on_mail_error' ) );

		// send Mail
		$return = self::greyd_mail( $mail_to, $subject, $mail_content );

		// Reset content-type to avoid conflicts -- https://core.trac.wordpress.org/ticket/23578
		remove_filter( 'wp_mail_content_type', array( 'Greyd\Forms\Helper', 'wpdocs_set_html_mail_content_type' ) );
		remove_filter( 'wp_mail_charset', array( 'Greyd\Forms\Handler', 'utf8' ) );
		remove_filter( 'wp_mail_failed', array( 'Greyd\Forms\Handler', 'on_mail_error' ) );

		return $return;
	}
}
