<?php
/*
	Comments: Plugin helper
*/
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Helper( $config );

class Helper {

	public static $config;

	public function __construct( $config ) {
		self::$config = (object) $config;
	}
	/**
	 * Check if Greyd.Suite Theme is installed and active.
	 */
	public static function is_greyd_classic() {

		if ( method_exists( '\Greyd\Helper', 'is_greyd_classic' ) ) {
			return \Greyd\Helper::is_greyd_classic();
		}

		if ( method_exists( '\Greyd\Helper', 'is_greyd_suite' ) ) {
			return \Greyd\Helper::is_greyd_suite();
		}

		// check if Greyd.Suite is active
		if ( defined( 'GREYD_CLASSIC_VERSION' ) || class_exists( "\basics" ) ) {
			return true;
		}

		// check if Greyd.Theme is active
		if ( defined( 'GREYD_THEME_CONFIG' ) ) {
			return false;
		}

		// check if Greyd.Suite is installed
		$_current_main_theme = ! empty( wp_get_theme()->parent() ) ? wp_get_theme()->parent() : wp_get_theme();
		return strpos( $_current_main_theme->get( 'Name' ), 'GREYD.SUITE' ) !== false;
	}

	/**
	 * @deprecated 1.6
	 * @see greyd_tp_managment/functions.php
	 *
	 * @param string $name      name of the input
	 * @param array  $options   all options as $value => $label
	 * @param array  $args      optional arguments for the element (value, placeholder, classes...)
	 *
	 * @return string html element
	 */
	public static function is_greyd_blocks() {

		if ( function_exists( 'is_greyd_blocks' ) ) {
			return is_greyd_blocks();
		}

		if ( function_exists( '__is_greyd_blocks' ) ) {
			return __is_greyd_blocks();
		}

		if ( function_exists( 'get_is_greyd_blocks' ) ) {
			return get_is_greyd_blocks();
		}

		/**
		 * Support for constant override.
		 *
		 * @since 1.6.0
		 */
		if ( defined( 'IS_GREYD_BLOCKS' ) && constant( 'IS_GREYD_BLOCKS' ) ) {
			return true;
		}

		if ( get_option( 'greyd_gutenberg' ) === true || get_option( 'greyd_gutenberg' ) == '1' ) {
			return true;
		}

		$settings = (array) get_option( 'settings_site_greyd_tp', array() );
		if ( isset( $settings['builder'] ) && $settings['builder'] === 'gg-gg' ) {
			return true;
		}

		return false;
	}


	/*
	------------------------------

			GET BACKEND DATA OF FORM

	------------------------------
	*/

	/**
	 * Get all greyd.forms
	 *
	 * @return array Post-titles keyed by post-ID
	 */
	public static function get_all_forms( array $prepend = array() ) {
		$return = array();

		if ( isset( $prepend ) ) {
			$return = (array) $prepend;
		}

		$forms = get_posts(
			array(
				'post_type'      => self::$config->plugin_post_type,
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			// 'suppress_filters' => false
			)
		);

		if ( is_array( $forms ) && ! empty( $forms ) ) {
			foreach ( $forms as $form ) {
				$return[ $form->ID ] = $form->post_title;
			}
		}
		return $return;
	}

	/**
	 * Get all input names of a form
	 *
	 * @param mixed $post_id_or_content
	 *
	 * @return array
	 */
	public static function get_all_input_names( $post_id_or_content ) {

		$post_content = self::get_post_content( $post_id_or_content );

		// vc
		if ( ! has_blocks( $post_content ) ) {

			if ( ! class_exists( 'Greyd\Forms\VC_Helper' ) ) {
				require_once __DIR__ . '/deprecated/vc/helper.php';
			}
			return VC_Helper::get_all_input_names( $post_content );
		}

		$all_names = self::get_form_blocks( parse_blocks( $post_content ), array( 'attrs', 'name' ) );

		return $all_names;
	}

	/**
	 * Get all inputs of a form
	 * including types, restrictions, filetype, list...
	 *
	 * @param mixed $post_id_or_content
	 * @return array
	 */
	public static function get_all_inputs( $post_id_or_content ) {

		$post_content = self::get_post_content( $post_id_or_content );

		if ( ! has_blocks( $post_content ) ) {

			if ( ! class_exists( 'Greyd\Forms\VC_Helper' ) ) {
				require_once __DIR__ . '/deprecated/vc/helper.php';
			}
			return VC_Helper::get_all_inputs( $post_content );
		}

		$inputs = array();
		$blocks = self::get_form_blocks( parse_blocks( $post_content ) );
		// debug($blocks);

		foreach ( $blocks as $name => $block ) {

			$block_name = str_replace( 'greyd-forms/', '', $block['blockName'] );
			$atts       = $block['attrs'];

			// default
			$input = array();

			switch ( $block_name ) {
				case 'input':
					$input = array(
						'type'  => isset( $atts['type'] ) ? $atts['type'] : 'text',
						'label' => isset( $atts['label'] ) ? $atts['label'] : '',
					);
					if ( isset( $atts['minlength'] ) ) {
						$input['restrictions']['minlength'] = apply_filters( 'formhelper_modify_minlength', $atts['minlength'], $name );
					}
					break;
				case 'checkbox':
					$input         = array(
						'type' => 'checkbox',
						'agb'  => isset( $atts['useSetting'] ) && $atts['useSetting'],
					);

					// get text
					if ( $atts['useSetting'] ) {
						$input['text'] = strip_tags( get_option( 'agb_text', __( "I hereby accept the terms and conditions.", 'greyd_forms' ) ) );
					} else if ( isset( $atts['content'] ) ){
						$input['text'] = strip_tags( $atts['content'] );
					} else if ( isset( $block['innerHTML'] ) ){
						$input['text'] = preg_match( '/<input type="checkbox".+?<div><span>(.+?)<\/span><\/div>/', $block['innerHTML'], $matches ) ? strip_tags( $matches[1] ) : '';
					}
					break;
				case 'dropdown':
					$options = isset( $atts['options'] ) ? $atts['options'] : array();
					$input   = array(
						'type'    => 'dropdown',
						'options' => self::get_options_from_attribute( $options )
					);
					break;
				case 'radiobuttons':
					$inner_blocks = isset( $block['innerBlocks'] ) ? $block['innerBlocks'] : array();
					$input        = array(
						'type'    => 'radio',
						'options' => self::get_options_from_inner_blocks( $inner_blocks ),
					);
					break;
				case 'iconpanels':
					$inner_blocks = isset( $block['innerBlocks'] ) ? $block['innerBlocks'] : array();
					$input        = array(
						'type'    => 'radio',
						'options' => self::get_options_from_inner_blocks( $inner_blocks ),
					);
					break;
				case 'upload':
					$input = array(
						'type'         => 'upload',
						'restrictions' => array(
							'size'     => isset( $atts['file']['max'] ) ? $atts['file']['max'] : '',
							'filetype' => isset( $atts['file']['type'] ) ? $atts['file']['type'] : '*',
							'custom'   => isset( $atts['file']['custom'] ) ? $atts['file']['custom'] : '',
						),
					);
					break;
				case 'hidden-field':
					break;
				case 'password':
					$input = array(
						'type'  => 'password',
						'label' => isset( $atts['label'] ) ? $atts['label'] : '',
					);
					break;
				case 'range':
				case 'math':
					$input = array(
						'type'  => 'number',
						'label' => isset( $atts['label'] ) ? $atts['label'] : '',
					);
					break;
			}

			// required
			if ( $block_name == 'recaptcha' ) {
				$input['required'] = true;
			} elseif ( isset( $block['parentBlocks'] ) && isset( array_flip( $block['parentBlocks'] )['greyd-forms/condition'] ) ) {
				$input['required'] = false; // don't require inputs inside conditional-contents
			} else {
				$input['required'] = isset( $atts['required'] ) && $atts['required'];
			}

			// cut maxlength to option 'maxlength_inputs'
			$maxlength = apply_filters( 'formhelper_modify_minlength', isset( $atts['maxlength'] ) ? $atts['maxlength'] : 1000, $name );
			$maxmax    = (int) esc_attr( get_option( 'maxlength_inputs', 1000 ) );
			if ( empty( $maxlength ) || (int) $maxlength < $maxmax ) {
				$maxlength = $maxmax;
			}
			$input['maxlength'] = $maxlength;

			// push to array
			$inputs[ $name ] = $input;
		}

		return $inputs;
	}

	/**
	 * Get names of all required inputs in a form
	 *
	 * @param mixed $post_id_or_content
	 */
	public static function get_all_required_fields( $post_id_or_content ) {

		$post_content = self::get_post_content( $post_id_or_content );

		if ( ! has_blocks( $post_content ) ) {
			if ( ! class_exists( 'Greyd\Forms\VC_Helper' ) ) {
				require_once __DIR__ . '/deprecated/vc/helper.php';
			}
			return VC_Helper::get_all_required_fields( $post_content );
		}

		$required_names = array();
		$inputs         = self::get_all_inputs( $post_content );

		foreach ( $inputs as $name => $atts ) {
			if ( isset( $atts['required'] ) && $atts['required'] ) {
				$required_names[] = $name;
			}
		}

		return $required_names;
	}

	/**
	 * Get all options of an input like a dropdown.
	 * @since 1.7.4
	 * 
	 * @param array $options_attr
	 * 
	 * @return array
	 */
	public static function get_options_from_attribute( $options_attr ) {

		if ( ! is_array( $options_attr ) ) {
			return array();
		}

		$options = array();

		foreach ( $options_attr as $option ) {
			$title = isset( $option['title'] ) ? esc_attr( $option['title'] ) : null;
			$value = isset( $option['value'] ) ? esc_attr( $option['value'] ) : null;

			if ( empty( $value ) && ! empty( $title ) ) {
				$value = sanitize_title( $title );
			}
			if ( empty( $title ) && ! empty( $value ) ) {
				$title = $value;
			}

			if ( ! empty( $value ) ) {
				$options[ $value ] = $title;
			}
		}

		return $options;
	}

	/**
	 * Get all options of an input like a radiobuttons or iconpanels.
	 * @since 1.7.4
	 * 
	 * @param array $innerBlocks
	 * 
	 * @return array
	 */
	public static function get_options_from_inner_blocks( $innerBlocks ) {

		if ( ! is_array( $innerBlocks ) ) {
			return array();
		}

		// debug( $innerBlocks );

		$options = array();

		foreach ( $innerBlocks as $option ) {
			$title = isset( $option['attrs']['title'] ) ? esc_attr( $option['attrs']['title'] ) : null;
			$value = isset( $option['attrs']['value'] ) ? esc_attr( $option['attrs']['value'] ) : null;

			if ( empty( $value ) && ! empty( $title ) ) {
				$value = sanitize_title( $title );
			}
			if ( empty( $title ) && ! empty( $value ) ) {
				$title = $value;
			}

			if ( ! empty( $value ) ) {
				$options[ $value ] = $title;
			}
		}

		return $options;
	}

	/*
	------------------------------

			GET BACKEND DATA OF ENTRY

	------------------------------
	*/

	/**
	 * Search for a Entry-Post by token value
	 *
	 * @return int WP_Post ID of the entry
	 */
	public static function get_entry_by_token( $token ) {

		$meta_key = 'tp_token';

		// Query
		$query = new \WP_Query(
			array(
				'posts_per_page' => 1,
				'post_type'      => self::$config->plugin_post_type . '_entry',
				'meta_key'       => $meta_key,
				'meta_value'     => $token,
			)
		);
		$post  = $query->posts;
		// $post = $wpdb->get_results("SELECT * FROM `".$wpdb->postmeta."` WHERE meta_key='".esc_sql($meta_key)."' AND meta_value='".esc_sql($token)."'"); // deprecated

		if ( is_array( $post ) && ! empty( $post ) && isset( $post[0] ) ) {
			$post = $post[0];
		}
		return is_object( $post ) ? $post->ID : false;
	}

	/**
	 * Get all necessary data and metadata stored in entry
	 */
	public static function get_entry_data_by_token( $token ) {
		$entry_id = self::get_entry_by_token( $token );
		return $entry_id ? self::get_entry_data( $entry_id ) : false;
	}

	/**
	 * Get data of entry
	 *
	 * @param int $entry_id
	 *
	 * @return array|false Array on success, false on failure.
	 */
	public static function get_entry_data( $entry_id ) {

		$form_id = get_post_meta( $entry_id, 'tp_form_id', true );
		if ( empty( $form_id ) ) {
			return false;
		}

		return array(
			'form_id'        => $form_id,
			'entry_id'       => $entry_id,
			'data'           => self::get_entry_form_data( $entry_id ),
			'user_gets_mail' => get_post_meta( $form_id, 'user_gets_mail', true ) == 'user_gets_mail' ? true : false,
			'host'           => get_post_meta( $entry_id, 'tp_host', true ),
			'created'        => get_the_date( 'Y-m-d H:i:s', $entry_id ),
		);
	}

	/**
	 * Get form_data of entry
	 *
	 * @since 0.9.1
	 *
	 * @param int $entry_id
	 * @return array|null
	 */
	public static function get_entry_form_data( $entry_id ) {

		/**
		 * form data saved in meta 'entry_form_data'
		 *
		 * @since 0.9.1
		 */
		$form_data = get_post_meta( $entry_id, 'entry_form_data', true );
		/**
		 * Fallback: form data saved in post_content
		 */
		if ( empty( $form_data ) ) {
			$post      = get_post( $entry_id );
			$form_data = json_decode( $post->post_content, true );
		}
		return $form_data;
	}

	/**
	 * Get uploads array of entry
	 *
	 * @since 0.9.1
	 *
	 * @param int $entry_id
	 * @return array|null
	 */
	public static function get_entry_uploads( $entry_id ) {
		$uploads = get_post_meta( $entry_id, 'tp_uploads', true );
		/**
		 * Form uploads saved as encoded json
		 *
		 * @deprecated 0.9.1
		 */
		if ( ! empty( $uploads ) && ! is_array( $uploads ) ) {
			$uploads = json_decode( $uploads, true );
		}
		return $uploads;
	}

	public static function get_entries_by_form_id( $form_id = '' ) {
		if ( ! isset( $form_id ) || empty( $form_id ) ) {
			return false;
		}

		$return   = array();
		$meta_key = 'tp_form_id';

		// Query
		$query = new \WP_Query(
			array(
				'posts_per_page' => -1,
				'post_type'      => self::$config->plugin_post_type . '_entry',
				'meta_key'       => $meta_key,
				'meta_value'     => $form_id,
			)
		);
		$posts = $query->posts;

		if ( is_array( $posts ) ) {
			foreach ( $posts as $post ) {
				if ( $post->post_status !== 'trash' ) {
					$return[] = $post->ID;
				}
			}
		}
		return count( $return ) > 0 ? $return : false;
	}

	public static function get_formatted_entry_data( $entry_id = '', $only_include = '' ) {
		if ( empty( $entry_id ) ) {
			return false;
		}

		$return = array();

		// + data & uploads
		if ( $only_include !== 'meta' ) {
			$data    = self::get_entry_form_data( $entry_id );
			$uploads = self::get_entry_uploads( $entry_id );
			foreach ( (array) $data as $name => $value ) {
				if ( $name === 'form' ) {
					continue;
				}
				if ( empty( $uploads[ $name ] ) ) {
					$return[ $name ] = $value;
				} else {
					$return[ $name ] = self::get_upload_link_element( $uploads[ $value ] );
				}
			}
		}

		// + meta infos
		if ( $only_include !== 'data' ) {
			$meta_keys = array(
				'tp_form_id' => 'FORM_ID',
				'tp_token'   => 'TOKEN',
				'tp_host'    => 'IP_ADRRESS',
				'entry_note' => 'ADMIN_NOTE',
			);
			foreach ( $meta_keys as $meta => $name ) {
				$value = get_post_meta( $entry_id, $meta, true );
				if ( ! empty( $value ) ) {
					$return[ $name ] = $value;
				}
			}

			// + states:
			/*
				submitted
				verified
				aborted
			*/
			$state = self::get_entry_state( $entry_id );
			$logs  = isset( $state['log'] ) ? (array) $state['log'] : array();
			if ( count( $logs ) > 0 ) {
				foreach ( $logs as $log ) {
					$text = isset( $log['text'] ) ? strval( $log['text'] ) : '';
					$time = isset( $log['timestamp'] ) ? $log['timestamp'] : __( "not stated", 'greyd_forms' );
					if ( $text === 'created' || $text === 'verified' || $text === 'bounced' ) {
						$return[ strtoupper( $text ) ] = $time;
					}
				}
			}
		}

		return $return;
	}

	/**
	 * Get entry state:
	 *   [
	 *       'current' => [
	 *           'type' => info | success | error,
	 *           'text' => {{message}} | created | verified | bounced | no_doi
	 *       ],
	 *       'log' => [
	 *           [
	 *               'timestamp' => {{time}},
	 *               'type'      => info | success | error,
	 *               'text'      => {{message}}
	 *           ],
	 *           ...
	 *       ]
	 *   ];
	 */
	public static function get_entry_state( $entry_id ) {

		$option = get_post_meta( $entry_id, 'greyd_forms_entry_state', true );

		if ( ! is_array( $option ) || count( $option ) === 0 ) {
			$legacy = self::convert_old_entry_state( $entry_id );
		}
		$state = isset( $legacy ) ? $legacy : $option;
		return $state;
	}

	/**
	 * Log new state to entry
	 *
	 * @param int    $entry_id  The post ID of the entry.
	 * @param string $text      Text to display.
	 * @param string $type      Class to be displayed to the greyd_info_box.
	 *                          Common types: error, info, success, warning.
	 * @param bool   $major     If true, this overwrites the current state of the entry.
	 *
	 * @return int|bool         See function update_post_meta() for return details.
	 */
	public static function log_entry_state( $entry_id, $text = '', $type = 'error', $major = true ) {
		if ( ! isset( $entry_id ) || empty( $entry_id ) || empty( $text ) ) {
			return false;
		}

		$state = self::get_entry_state( $entry_id );

		$log   = isset( $state['log'] ) ? (array) $state['log'] : array();
		$new   = array(
			'timestamp' => current_time( 'mysql' ),
			'type'      => $type,
			'text'      => $text,
		);
		$log[] = $new;
		if ( $major ) {
			$state['current'] = $new;
		}
		$state['log'] = $log;
		return update_post_meta( $entry_id, 'greyd_forms_entry_state', $state );
	}

	public static function check_if_entry_is( $entry_id = '', $text = 'bounced' ) {
		if ( ! isset( $entry_id ) || empty( $entry_id ) || empty( $text ) ) {
			return false;
		}

		$state = self::get_entry_state( $entry_id );
		$logs  = isset( $state['log'] ) ? (array) $state['log'] : array();
		foreach ( $logs as $log ) {
			if ( isset( $log['text'] ) && $log['text'] === $text ) {
				return true;
			}
		}
		return false;
	}

	public static function convert_old_entry_state( $entry_id ) {
		$log = array();

		// get initial states
		$created = get_post_meta( $entry_id, 'tp_time', true );
		if ( ! empty( $created ) ) {
			$log[] = array(
				'timestamp' => $created,
				'type'      => 'success',
				'text'      => 'created',
			);
		}
		$verified = get_post_meta( $entry_id, 'verify_time', true );
		$no_doi   = get_post_meta( $entry_id, 'no_doi', true );
		if ( ! empty( $verified ) && $no_doi ) {
			$log[] = array(
				'timestamp' => $verified,
				'type'      => 'info',
				'text'      => 'no_doi',
			);
		} elseif ( ! empty( $verified ) ) {
			$log[] = array(
				'timestamp' => $verified,
				'type'      => 'success',
				'text'      => 'verified',
			);
		}

		// get form errors (admin or verify email)
		$errors = get_post_meta( $entry_id, 'handle_errors', true );
		if ( is_array( $errors ) && count( $errors ) > 0 ) {
			foreach ( $errors as $text ) {
				$log[] = array(
					'type' => 'error',
					'text' => $text,
				);
			}
		}

		// get interface errors
		$interface_state = get_post_meta( $entry_id, \Greyd\Forms\Interfaces\Greyd_Forms_Interface::ENTRY, true );
		if ( is_array( $interface_state ) && count( $interface_state ) > 0 ) {
			foreach ( $interface_state as $interface => $response ) {
				$name = \Greyd\Forms\Interfaces\Greyd_Forms_Interface::get_config( $interface, 'name' );
				if ( $name === false ) {
					continue;
				}
				$class = $response === true ? 'success' : 'danger';
				if ( $response === true ) {
					$log[] = array(
						'type' => 'success',
						'text' => sprintf( '✓ ' . __( "Successfully sent to %s", 'greyd_forms' ), $name ),
					);
				} else {
					$text = sprintf( '∅ ' . __( "Failed to send to %s", 'greyd_forms' ), $name );

					if ( is_string( $response ) && ! empty( $response ) ) {
						$text .= $response;
						$log[] = array(
							'type' => 'error',
							'text' => $text . ': ' . $response,
						);
					} elseif ( is_array( $response ) ) {
						if ( count( $response ) > 1 ) {
							foreach ( $response as $resp ) {
								if ( is_string( $resp ) ) {
									$log[] = array(
										'type' => 'error',
										'text' => $text . ': ' . $resp,
									);
								}
							}
						} elseif ( count( $response ) === 1 ) {
							$log[] = array(
								'type' => 'error',
								'text' => $text . ': ' . array_values( $response )[0],
							);
						}
					}
				}
			}
		}

		$bounced = get_post_meta( $entry_id, 'abort_time', true );
		if ( ! empty( $bounced ) ) {
			$log[] = array(
				'timestamp' => $bounced,
				'type'      => 'error',
				'text'      => 'bounced',
			);
		}

		$state = array(
			'current' => end( $log ),
			'log'     => $log,
		);

		delete_post_meta( $entry_id, 'tp_time' );
		delete_post_meta( $entry_id, 'verify_time' );
		delete_post_meta( $entry_id, 'no_doi' );
		delete_post_meta( $entry_id, 'handle_errors' );
		delete_post_meta( $entry_id, 'abort_time' );
		update_post_meta( $entry_id, 'greyd_forms_entry_state', $state );

		return $state;
	}

	/**
	 * Get the HTML link element to an uploaded file.
	 *
	 * @since 0.9.1
	 *
	 * @param array|string $value   @since 0.9.1: File info saved as array [ name, path, file ]
	 *                              @deprecated 0.9.1: HTML element as string.
	 * @return string HTML element
	 */
	public static function get_upload_link_element( $value ) {
		if ( is_string( $value ) ) {
			return $value;
		} else {
			$name = isset( $value['name'] ) ? $value['name'] : '';
			$path = isset( $value['path'] ) ? $value['path'] : '';
			$type = isset( $value['type'] ) ? $value['type'] : '';
			$url  = wp_upload_dir()['baseurl'] . $path;

			if ( strpos( $type, 'image' ) !== false ) {
				return "<a href='$url' target='_blank' title='" . __( "Open image in seperate tab", 'greyd_forms' ) . "' class='uploaded_image_link'><img src='$url' title='" . sprintf(
					__( "Open image in separate tab.", 'greyd_forms' ),
					$name
				) . "'></a>";
			} else {
				return "<a href='$url' target='_blank' title='" . __( "Open file in separate tab", 'greyd_forms' ) . "'>$name</a>";
			}
		}
	}

	/*
	------------------------------

			HELPERS

	------------------------------
	*/

	/**
	 * Get post content from id or post-content
	 *
	 * @param mixed $post_id_or_content
	 *
	 * @return string
	 */
	public static function get_post_content( $post_id_or_content ) {

		if ( is_numeric( $post_id_or_content ) || is_object( $post_id_or_content ) ) {
			$post = get_post( $post_id_or_content );

			if ( ! $post ) {
				return '';
			}

			return $post->post_content;
		} else {
			return $post_id_or_content;
		}
	}

	/**
	 * Retrieve all form blocks of the current post or a blocks list.
	 *
	 * @since 1.0
	 *
	 * @param array  $blocks      Parsed blocks.
	 * @param string $prop      Property to retrieve. Leave empty for whole object.
	 * @param array  $parents    Log all parent blocks.
	 *
	 * @return array
	 */
	public static function get_form_blocks( $blocks, $prop = null, $parents = array() ) {
		// debug($blocks);
		if ( ! is_array( $blocks ) ) {
			return array();
		}

		$collected_blocks    = array();
		$allowed_form_blocks = array_flip(
			array(
				'greyd-forms/input',
				'greyd-forms/checkbox',
				'greyd-forms/dropdown',
				'greyd-forms/radiobuttons',
				'greyd-forms/iconpanels',
				'greyd-forms/upload',
				'greyd-forms/hidden-field',
				'greyd-forms/recaptcha',
				'greyd-forms/password',
				'greyd-forms/range',
				'greyd-forms/math',
			)
		);

		foreach ( $blocks as $block ) {

			$block_name = isset( $block['blockName'] ) ? $block['blockName'] : '';

			// loop through inner blocks
			$inner_blocks = isset( $block['innerBlocks'] ) ? $block['innerBlocks'] : null;
			if ( $inner_blocks ) {
				$collected_blocks = array_merge( $collected_blocks, self::get_form_blocks( $inner_blocks, $prop, array_merge( $parents, array( $block_name ) ) ) );
			}

			// this is a form field
			if ( isset( $allowed_form_blocks[ $block_name ] ) ) {

				// get input name
				if ( $block_name === 'greyd-forms/recaptcha' ) {
					$input_name = 'reCAPTCHA';
				} else {
					$input_name = isset( $block['attrs'] ) && isset( $block['attrs']['name'] ) ? self::make_input_name( $block['attrs']['name'], false ) : null;
				}

				if ( ! empty( $input_name ) ) {

					// get whole block (default)
					if ( empty( $prop ) ) {
						$collected_blocks[ $input_name ] = array_merge( $block, array( 'parentBlocks' => $parents ) );
					}
					// get property
					elseif ( is_string( $prop ) ) {
						$collected_blocks[ $input_name ] = isset( $block[ $prop ] ) ? $block[ $prop ] : null;
					}
					// get nested property
					elseif ( is_array( $prop ) ) {
						$collected_blocks[ $input_name ] = isset( $block[ $prop[0] ] ) && isset( $block[ $prop[0] ][ $prop[1] ] ) ? $block[ $prop[0] ][ $prop[1] ] : null;
					}
				}
			}
		}
		return $collected_blocks;
	}

	/**
	 * Get a string between 2 substrings
	 *
	 * @param string $content   Content to query.
	 * @param string $start     Start of the string.
	 * @param string $end       End of the string.
	 *
	 * @return string
	 */
	public static function get_string_between( $content, $start, $end ) {
		$r = explode( $start, $content );
		if ( isset( $r[1] ) ) {
			$r = explode( $end, $r[1] );
			return $r[0];
		}
		return '';
	}

	/**
	 * makes names of input fields unique
	 */
	public static function make_input_name( $name, $uniq = true ) {
		$name = preg_replace( '/[^A-Za-z0-9\-\_\[\]]/', '', preg_replace( '/\s/', '_', trim( $name ) ) );
		if ( $uniq ) {
			$name .= '___id' . uniqid();
		}
		return $name;
	}

	/**
	 * Converts form-data into html lines for admin mail
	 */
	public static function formdata2html( $form_data, $entry_id = null ) {
		ob_start();

		// get uploads
		if ( ! empty( $entry_id ) ) {
			$uploads = self::get_entry_uploads( $entry_id );
		}

		foreach ( $form_data as $key => $val ) {
			if ( isset( $uploads ) && isset( $uploads[ $key ] ) ) {
				echo $key . ': ' . self::get_upload_link_element( $uploads[ $key ] ) . '<br>';
			} elseif ( $key !== 'submit' ) {
				echo $key . ": <b>$val</b><br>";
			}
		}
		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}

	/**
	 * Get Client HOST IP
	 */
	public static function get_client_host() {
		return isset( $_SERVER['REMOTE_ADDR'] ) ? $_SERVER['REMOTE_ADDR'] : 'IP COULD NOT BE RESOLVED';
	}

	/**
	 * Get current language.
	 * @supports:
	 * - Polylang
	 * - WPML
	 * - Default WP
	 */
	public static function get_language_code() {

		// polylang support
		if ( function_exists('pll_current_language') ) {
			return pll_current_language();
		}

		// wpml support
		if ( defined('ICL_LANGUAGE_CODE') ) {
			return ICL_LANGUAGE_CODE;
		}

		return explode('_', get_locale(), 2)[0];
	}

	/**
	 * Whether we are in a REST REQUEST. Similar to is_admin().
	 */
	public static function is_rest_request() {
		return defined( 'REST_REQUEST' ) && REST_REQUEST;
	}

	/**
	 * Filter WP_Post ID based on various factors:
	 * - See if WPML translation exists.
	 *
	 * @since 1.3.6
	 *
	 * @param int    $post_id       WP_Post ID.
	 * @param string $post_type  Post type, defaults to 'post'.
	 *
	 * @return int
	 */
	public static function filter_post_id( $post_id, $post_type = 'post' ) {

		/**
		 * WPML: filter Post ID and look for tranlated version.
		 *
		 * @see https://wpml.org/wpml-hook/wpml_object_id/
		 */
		$filtered_id = apply_filters( 'wpml_object_id', $post_id, $post_type, null, null );
		if ( $filtered_id && $filtered_id != $post_id ) {
			$post_id = $filtered_id;
		}

		return intval( $post_id );
	}

	/**
	 * returns Default-Error of Forms (Server Error in scripts)
	 */
	public static function get_default_error() {
		return __( "The form could not be sent. Please check your internet connection and try again or contact the administrator.", 'greyd_forms' );
	}

	/**
	 *  converts values according to possible units:
	 *
	 *  @param $value:      (string) input value to be corrected
	 *  @param $units:      (array) all possible units
	 *  @param $default:    (string) if none of the units are found, this unit is attached
	 */
	public static function force_units( $value = '', $units = array( 'px' ), $default = 'px' ) {
		if ( empty( $value ) ) {
			return $value;
		}

		// make units array
		if ( ! is_array( $units ) ) {
			$units = array( $units );
		}
		// if only 1 unit is defined, set as default
		if ( sizeof( $units ) === 1 ) {
			$default = $units[0];
		}

		// search for matches
		$has_unit = false;
		foreach ( $units as $unit ) {
			if ( strpos( $value, strval( $unit ) ) !== false ) {
				$has_unit = true;
			}
		}
		// no match -> add default
		if ( ! $has_unit ) {
			$value .= $default; }
		return $value;
	}

	/**
	 * set mails to html
	 *
	 * @source (see comments): https://developer.wordpress.org/reference/functions/wp_mail/
	 */
	public static function wpdocs_set_html_mail_content_type() {
		return 'text/html';
	}

	/**
	 * Add params from @class \url_handler to link
	 */
	public static function add_url_params_to_link( $href, $data = array() ) {
		$params = array();
		if ( class_exists( '\Greyd\Extensions\Cookie_Handler' ) ) {
			$params = \Greyd\Extensions\Cookie_Handler::get_supported_params();
		} elseif ( class_exists( '\url_handler' ) ) {
			$params = \url_handler::get_params();
		}
		if ( is_array( $params ) && count( $params ) > 0 ) {
			foreach ( $params as $i => $param ) {
				if ( isset( $data[ $param['name'] ] ) ) {
					if ( $param['name'] === 'optin' || $param['name'] === 'optout' || $param['name'] === 'form' ) {
						continue;
					}
					$href = add_query_arg( $param['name'], $data[ $param['name'] ], $href );
				}
			}
		}
		return $href;
	}

	/**
	 * check if select is empty
	 */
	public static function is_select_empty( $value ) {
		return (
				empty( $value ) || trim( $value ) === '' ||
				$value === 'bitte wählen' || $value === 'bitte auswählen' ||
				$value === 'select' || $value === 'please select' || $value === 'choose' || $value === 'please choose' ||
				$value === 'none' || $value === '--'
		);
	}

	/**
	 * render admin notice
	 */
	public static function render_admin_notice( $message, $type = 'updated', $echo = true ) {
		$return = '<div class="notice notice-' . $type . ' is-dismissible">
                <p>' . strval( $message ) . '</p>
            </div>';
		if ( $echo ) {
			echo $return;
		}
		return $return;
	}

	/**
	 * Render a frontend message box
	 */
	public static function make_message( $msg, $mode = 'info' ) {
		if ( $mode != 'info' && $mode != 'success' && $mode != 'danger' ) {
			$mode = 'info';
		}
		return "<div class='message " . $mode . "'>" . $msg . '</div>';
	}

	/**
	 * Sideload & include library from external url.
	 *
	 * @param array $args       Array of arguments:
	 *
	 *      @property string path   Path where the library should be.
	 *      @property string file   File to be included.
	 *      @property string url    Url from where to download the library.
	 *
	 * @return bool
	 */
	public static function include_library( $args ) {
		if (
			! is_array( $args ) ||
			! isset( $args['path'] ) ||
			! isset( $args['file'] ) ||
			! isset( $args['url'] )
		) {
			return false;
		}
		// debug($args);
		if ( ! file_exists( $args['path'] . $args['file'] ) ) {
			// debug('lib not found - sideloading lib ...');
			if ( ! is_dir( $args['path'] ) ) {
				mkdir( $args['path'], 0755, true );
			}
			$tmpfile = $args['path'] . '/tmp_file.zip';
			if ( ! copy( $args['url'], $tmpfile ) ) {
				// debug("unable to copy ".$args['url']." ...");
				return false;
			}
			$zip = new \ZipArchive();
			if ( $zip->open( $tmpfile ) === false ) {
				// debug("unable to open zip ...");
				return false;
			}
			// extract, close and delete zip
			$zip->extractTo( $args['path'] );
			$zip->close();
			unlink( $tmpfile );
		}
		if ( ! file_exists( $args['path'] . $args['file'] ) ) {
			// debug("unable to sideload ".$args['url']." ...");
			return false;
		}
		// debug('lib found - including lib ...');
		require_once $args['path'] . $args['file'];
		return true;
	}

	/**
	 * Build form tag for frontend
	 * {input_name} --> "<span class='form_tag' data-name='input_name'></span>"
	 */
	public static function build_form_tags( string $input ) {
		return preg_replace( '/{([A-Za-z0-9\-\_\[\]]+?)}/', "<span class='form_tag' data-name='$1'></span>", $input );
	}

	/**
	 * Revert form tag for frontend
	 * "<span class='form_tag' data-name='input_name'></span>" --> {input_name}
	 */
	public static function revert_form_tags( string $input ) {
		return preg_replace( "/<span class=(\'|\")form_tag(\'|\")\sdata-name=(\'|\")([A-Za-z0-9\-\_\[\]]+?)(\'|\")><\/span>/", '{$4}', $input );
	}

	/**
	 * Revert form tag for frontend
	 * "<span class='form_tag' data-name='input_name'></span>" --> {input_name}
	 */
	public static function revert_block_form_tags( string $input ) {
		return trim( preg_replace( "/<span data-name=(\'|\")([A-Za-z0-9\-\_\[\]]+?)(\'|\")\sclass=(\'|\")form_tag(\'|\")><\/span>/", '{$2}', $input ) );
	}

	/**
	 * Get real formula for math operation fields
	 */
	public static function get_formula( $formula ) {
		return str_replace( ',', '.', preg_replace( '/[^0-9.,\/\*\+\-\^\√\(\{\}\(\)A-Za-z\_\[\]]/', '', self::revert_form_tags( $formula ) ) );
	}

	public static function display_formula( $formula ) {

		// replace '/' with '÷'
		$formula = preg_replace( '/([^<])\/([^>])/', '$1÷$2', $formula );

		// replace '*' with 'x'
		$formula = str_replace( array( '*' ), array( 'x' ), $formula );

		// format √(___)
		$formula = preg_replace( '/\√\(([^\(\)]+?)\)/', "&radic;<span style='text-decoration:overline;'> $1 </span>", $formula );

		// format 4^(2 + 1)
		$formula = self::make_sup( $formula );

		return $formula;
	}

	public static function make_sup( $text ) {
		return preg_replace_callback(
			'/\^(([^\(\)\s]+?)|\(([^\(\)]+?)\))/',
			function( $m ) {
				$replacement = ! empty( $m[2] ) ? $m[2] : ( ! empty( $m[3] ) ? $m[3] : $m[1] );
				return "<sup>{$replacement}</sup>";
			},
			$text
		);
	}

	/**
	 * Replace data-tags with field values for after actions
	 *
	 * @param string $subject   Content where the tags should be replaced
	 * @param array  $data       Form data with field values
	 */
	public static function replace_form_data_tags( $subject, $data, $post_id, $token = '' ) {

		$data =  is_array($data) ? array_merge(
			$data,
			array(
				'opt_in_link'       => self::build_optin_link( $post_id, $data, $token ),
				'opt_out_link'      => self::build_optout_link( $post_id, $data, $token ),
				'registration_link' => self::build_optin_link( $post_id, $data, $token ),
			)
		) : array();

		foreach ( $data as $key => $val ) {

			/**
			 * Filter form values before replacement inside mail contents etc.
			 *
			 * @filter greyd_forms_filter_data_tag_value
			 *
			 * @param string $val       Representative value of the field.
			 * @param string $key       Name of the field. Example: tag [user_email] has the name 'user_email'.
			 * @param string $subject   Content to be modified, usually the mail content.
			 * @param int    $post_id   WP_Post ID of the form.
			 *
			 * @return string
			 */
			$val = apply_filters( 'greyd_forms_filter_data_tag_value', $val, $key, $subject, (int) $post_id );

			$subject = str_replace( '[' . $key . ']', $val, $subject );
		}

		return $subject;
	}

	/**
	 * Build the option link
	 *
	 * @param string $post_id   WP_Post ID
	 * @param array  $data       Form data with field values
	 * @param string $token     Entry token
	 *
	 * @return string           <a href="...">...</a>
	 */
	public static function build_optin_link( $post_id, $data, $token = '', $user_id = '' ) {

		if ( self::$opt_in_link ) {
			return self::$opt_in_link;
		}

		// get url dynamically
		$meta = get_post_meta( $post_id, 'verify_link', true );
		$url  = self::get_url_from_post_meta( $meta );

		$args = array(
			'optin' => $token,
			'form'  => $post_id,
		);

		if ( isset( $user_id ) ) {
			$args['user_id'] = $user_id;
		}

		$href = add_query_arg( $args, self::add_url_params_to_link( $url, $data ) );

		// atts
		$link_name  = get_post_meta( $post_id, 'verify_link_name', true );
		$link_name  = ! empty( $link_name ) ? $link_name : _x( "verify email", 'small', 'greyd_forms' );
		$link_color = get_post_meta( $post_id, 'verify_link_color', true );
		$link_color = ! empty( $link_color ) ? "style='color:$link_color;'" : '';

		self::$opt_in_link = "<a $link_color href='$href' rel='nofollow' color><span $link_color><font $link_color>$link_name</font></span></a>";
		return self::$opt_in_link;
	}

	public static $opt_in_link = null;

	/**
	 * Build optout link
	 *
	 * @param string $post_id   WP_Post ID
	 * @param array  $data       Form data with field values
	 * @param string $token     Entry token
	 *
	 * @return string           <a href="...">...</a>
	 */
	public static function build_optout_link( $post_id, $data, $token = '' ) {

		if ( self::$opt_out_link ) {
			return self::$opt_out_link;
		}

		// href
		$meta = get_post_meta( $post_id, 'opt_out_link', true );
		$url  = self::get_url_from_post_meta( $meta );

		$href = add_query_arg(
			array(
				'optout' => $token,
				'form'   => $post_id,
			),
			self::add_url_params_to_link( $url, $data )
		);

		// atts
		$link_name  = get_post_meta( $post_id, 'opt_out_link_name', true );
		$link_name  = ! empty( $link_name ) ? $link_name : _x( "withdraw permission", 'small', 'greyd_forms' );
		$link_color = get_post_meta( $post_id, 'opt_out_link_color', true );
		$link_color = ! empty( $link_color ) ? "style='color:$link_color;'" : '';

		self::$opt_out_link = "<a $link_color href='$href' rel='nofollow' color><span $link_color><font $link_color>$link_name</font></span></a>";
		return self::$opt_out_link;
	}
	public static $opt_out_link = null;

	/**
	 * Convert an url meta value (Page ID or Custom URL) into a valid url on this page
	 */
	public static function get_url_from_post_meta( $meta_value ) {

		$url = null;

		// default: home url
		if ( empty( $meta_value ) || $meta_value == '--' || $meta_value == 'home' ) {

		}
		// selected page-id
		elseif ( is_numeric( $meta_value ) ) {
			$url = get_permalink( $meta_value );
		}
		// custom url
		else {
			$page_id = get_page_by_path( $meta_value );
			if ( $page_id ) {
				$url = get_permalink( $page_id );
			}
			if ( empty( $url ) ) {
				$url = get_home_url();

				// add redirect to url
				if ( filter_var( $url, FILTER_VALIDATE_URL ) !== false ) {
					$url = add_query_arg(
						array(
							'forms_redirect' => esc_url( $meta_value ),
						),
						$url
					);
				}
			}
		}

		// fallback
		if ( empty( $url ) || filter_var( $url, FILTER_VALIDATE_URL ) === false ) {
			$url = get_home_url();
		}

		return $url;
	}

	/**
	 * Whether the current user agent is a bot.
	 *
	 * @see https://github.com/fabiomb/is_bot
	 * @since 1.3.7
	 *
	 * @return bool
	 */
	public static function is_bot() {

		$agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? strval( $_SERVER['HTTP_USER_AGENT'] ) : '';

		/**
		 * Empty user agents are almost always robots.
		 *
		 * @see https://security.stackexchange.com/questions/195076/can-i-assume-that-an-empty-user-agent-will-always-be-a-robot
		 */
		if ( empty( $agent ) ) {
			return true;
		}

		$bots = array(
			'Googlebot',
			'Baiduspider',
			'ia_archiver',
			'R6_FeedFetcher',
			'NetcraftSurveyAgent',
			'Sogou web spider',
			'bingbot',
			'Yahoo! Slurp',
			'facebookexternalhit',
			'PrintfulBot',
			'msnbot',
			'Twitterbot',
			'UnwindFetchor',
			'urlresolver',
			'Butterfly',
			'TweetmemeBot',
			'PaperLiBot',
			'MJ12bot',
			'AhrefsBot',
			'Exabot',
			'Ezooms',
			'YandexBot',
			'SearchmetricsBot',
			'picsearch',
			'TweetedTimes Bot',
			'QuerySeekerSpider',
			'ShowyouBot',
			'woriobot',
			'merlinkbot',
			'BazQuxBot',
			'Kraken',
			'SISTRIX Crawler',
			'R6_CommentReader',
			'magpie-crawler',
			'GrapeshotCrawler',
			'PercolateCrawler',
			'MaxPointCrawler',
			'R6_FeedFetcher',
			'NetSeer crawler',
			'grokkit-crawler',
			'SMXCrawler',
			'PulseCrawler',
			'Y!J-BRW',
			'80legs',
			'Mediapartners-Google',
			'Spinn3r',
			'InAGist',
			'Python-urllib',
			'NING',
			'TencentTraveler',
			'Feedfetcher-Google',
			'mon.itor.us',
			'spbot',
			'Feedly',
			'bitlybot',
			'ADmantX',
			'Niki-Bot',
			'Pinterest',
			'python-requests',
			'DotBot',
			'HTTP_Request2',
			'linkdexbot',
			'A6-Indexer',
			'Baiduspider',
			'TwitterFeed',
			'Microsoft Office',
			'Pingdom',
			'BTWebClient',
			'KatBot',
			'SiteCheck',
			'proximic',
			'Sleuth',
			'Abonti',
			'(BOT for JCE)',
			'Baidu',
			'Tiny Tiny RSS',
			'newsblur',
			'updown_tester',
			'linkdex',
			'baidu',
			'searchmetrics',
			'genieo',
			'majestic12',
			'spinn3r',
			'profound',
			'domainappender',
			'VegeBot',
			'terrykyleseoagency.com',
			'CommonCrawler Node',
			'AdlesseBot',
			'metauri.com',
			'libwww-perl',
			'rogerbot-crawler',
			'MegaIndex.ru',
			'ltx71',
			'Qwantify',
			'Traackr.com',
			'Re-Animator Bot',
			'Pcore-HTTP',
			'BoardReader',
			'omgili',
			'okhttp',
			'CCBot',
			'Java/1.8',
			'semrush.com',
			'feedbot',
			'CommonCrawler',
			'AdlesseBot',
			'MetaURI',
			'ibwww-perl',
			'rogerbot',
			'MegaIndex',
			'BLEXBot',
			'FlipboardProxy',
			'techinfo@ubermetrics-technologies.com',
			'trendictionbot',
			'Mediatoolkitbot',
			'trendiction',
			'ubermetrics',
			'ScooperBot',
			'TrendsmapResolver',
			'Nuzzel',
			'Go-http-client',
			'Applebot',
			'LivelapBot',
			'GroupHigh',
			'SemrushBot',
			'ltx71',
			'commoncrawl',
			'istellabot',
			'DomainCrawler',
			'cs.daum.net',
			'StormCrawler',
			'GarlikCrawler',
			'The Knowledge AI',
			'getstream.io/winds',
			'YisouSpider',
			'archive.org_bot',
			'semantic-visions.com',
			'FemtosearchBot',
			'360Spider',
			'linkfluence.com',
			'glutenfreepleasure.com',
			'Gluten Free Crawler',
			'YaK/1.0',
			'Cliqzbot',
			'app.hypefactors.com',
			'axios',
			'semantic-visions.com',
			'webdatastats.com',
			'schmorp.de',
			'SEOkicks',
			'DuckDuckBot',
			'DuckDuckGo-Favicons-Bot',
			'Barkrowler',
			'ZoominfoBot',
			'Linguee Bot',
			'Mail.RU_Bot',
			'OnalyticaBot',
			'Linguee Bot',
			'admantx-adform',
			'Buck/2.2',
			'Barkrowler',
			'Zombiebot',
			'Nutch',
			'SemanticScholarBot',
			'Jetslide',
			'scalaj-http',
			'XoviBot',
			'sysomos.com',
			'PocketParser',
			'newspaper',
			'serpstatbot',
			'MetaJobBot',
			'SeznamBot/3.2',
			'VelenPublicWebCrawler/1.0',
			'WordPress.com mShots',
			'adscanner',
			'BacklinkCrawler',
			'netEstate NE Crawler',
			'Astute SRM',
			'GigablastOpenSource/1.0',
			'DomainStatsBot',
			'Winds: Open Source RSS & Podcast',
			'dlvr.it',
			'BehloolBot',
			'7Siters',
			'AwarioSmartBot',
			'Apache-HttpClient/5',
			'Seekport Crawler',
			'AHC/2.1',
			'eCairn-Grabber',
			'mediawords bot',
			'PHP-Curl-Class',
			'Scrapy',
			'curl/7',
			'Blackboard',
			'NetNewsWire',
			'node-fetch',
			'admantx',
			'metadataparser',
			'Domains Project',
			'SerendeputyBot',
			'Moreover',
			'DuckDuckGo',
			'monitoring-plugins',
			'Selfoss',
			'Adsbot',
			'acebookexternalhit',
			'SpiderLing',
			'Cocolyzebot',
			'AhrefsBot',
			'TTD-Content',
			'superfeedr',
			'Twingly',
			'Google-Apps-Scrip',
			'LinkpadBot',
			'CensysInspect',
			'Reeder',
			'tweetedtimes',
			'Amazonbot',
			'MauiBot',
			'Symfony BrowserKit',
			'DataForSeoBot',
			'GoogleProducer',
			'TinEye-bot-live',
			'sindresorhus/got',
			'CriteoBot',
			'Down/5',
			'Yahoo Ad monitoring',
			'MetaInspector',
			'PetalBot',
			'MetadataScraper',
			'Cloudflare SpeedTest',
			'CriteoBot',
			'aiohttp',
			'AppEngine-Google',
			'heritrix',
			'sqlmap',
			'Buck',
			'MJ12bot',
			'wp_is_mobile',
			'SerendeputyBot',
			'01h4x.com',
			'404checker',
			'404enemy',
			'AIBOT',
			'ALittle Client',
			'ASPSeek',
			'Aboundex',
			'Acunetix',
			'AfD-Verbotsverfahren',
			'AiHitBot',
			'Aipbot',
			'Alexibot',
			'AllSubmitter',
			'Alligator',
			'AlphaBot',
			'Anarchie',
			'Anarchy',
			'Anarchy99',
			'Ankit',
			'Anthill',
			'Apexoo',
			'Aspiegel',
			'Asterias',
			'Atomseobot',
			'Attach',
			'AwarioRssBot',
			'BBBike',
			'BDCbot',
			'BDFetch',
			'BackDoorBot',
			'BackStreet',
			'BackWeb',
			'Backlink-Ceck',
			'BacklinkCrawler',
			'Badass',
			'Bandit',
			'Barkrowler',
			'BatchFTP',
			'Battleztar Bazinga',
			'BetaBot',
			'Bigfoot',
			'Bitacle',
			'BlackWidow',
			'Black Hole',
			'Blackboard',
			'Blow',
			'BlowFish',
			'Boardreader',
			'Bolt',
			'BotALot',
			'Brandprotect',
			'Brandwatch',
			'Buck',
			'Buddy',
			'BuiltBotTough',
			'BuiltWith',
			'Bullseye',
			'BunnySlippers',
			'BuzzSumo',
			'CATExplorador',
			'CCBot',
			'CODE87',
			'CSHttp',
			'Calculon',
			'CazoodleBot',
			'Cegbfeieh',
			'CensysInspect',
			'CheTeam',
			'CheeseBot',
			'CherryPicker',
			'ChinaClaw',
			'Chlooe',
			'Citoid',
			'Claritybot',
			'Cliqzbot',
			'Cloud mapping',
			'Cocolyzebot',
			'Cogentbot',
			'Collector',
			'Copier',
			'CopyRightCheck',
			'Copyscape',
			'Cosmos',
			'Craftbot',
			'Crawling at Home Project',
			'CrazyWebCrawler',
			'Crescent',
			'CrunchBot',
			'Curious',
			'Custo',
			'CyotekWebCopy',
			'DBLBot',
			'DIIbot',
			'DSearch',
			'DTS Agent',
			'DataCha0s',
			'DatabaseDriverMysqli',
			'Demon',
			'Deusu',
			'Devil',
			'Digincore',
			'DigitalPebble',
			'Dirbuster',
			'Disco',
			'Discobot',
			'Discoverybot',
			'Dispatch',
			'DittoSpyder',
			'DnBCrawler-Analytics',
			'DnyzBot',
			'DomCopBot',
			'DomainAppender',
			'DomainCrawler',
			'DomainSigmaCrawler',
			'DomainStatsBot',
			'Domains Project',
			'Dotbot',
			'Download Wonder',
			'Dragonfly',
			'Drip',
			'ECCP/1.0',
			'EMail Siphon',
			'EMail Wolf',
			'EasyDL',
			'Ebingbong',
			'Ecxi',
			'EirGrabber',
			'EroCrawler',
			'Evil',
			'Exabot',
			'Express WebPictures',
			'ExtLinksBot',
			'Extractor',
			'ExtractorPro',
			'Extreme Picture Finder',
			'EyeNetIE',
			'Ezooms',
			'FDM',
			'FHscan',
			'FemtosearchBot',
			'Fimap',
			'Firefox/7.0',
			'FlashGet',
			'Flunky',
			'Foobot',
			'Freeuploader',
			'FrontPage',
			'Fuzz',
			'FyberSpider',
			'Fyrebot',
			'G-i-g-a-b-o-t',
			'GT::WWW',
			'GalaxyBot',
			'Genieo',
			'GermCrawler',
			'GetRight',
			'GetWeb',
			'Getintent',
			'Gigabot',
			'Go!Zilla',
			'Go-Ahead-Got-It',
			'GoZilla',
			'Gotit',
			'GrabNet',
			'Grabber',
			'Grafula',
			'GrapeFX',
			'GrapeshotCrawler',
			'GridBot',
			'HEADMasterSEO',
			'HMView',
			'HTMLparser',
			'HTTP::Lite',
			'HTTrack',
			'Haansoft',
			'HaosouSpider',
			'Harvest',
			'Havij',
			'Heritrix',
			'Hloader',
			'HonoluluBot',
			'Humanlinks',
			'HybridBot',
			'IDBTE4M',
			'IDBot',
			'IRLbot',
			'Iblog',
			'Id-search',
			'IlseBot',
			'Image Fetch',
			'Image Sucker',
			'IndeedBot',
			'Indy Library',
			'InfoNaviRobot',
			'InfoTekies',
			'Intelliseek',
			'InterGET',
			'InternetSeer',
			'Internet Ninja',
			'Iria',
			'Iskanie',
			'IstellaBot',
			'JOC Web Spider',
			'JamesBOT',
			'Jbrofuzz',
			'JennyBot',
			'JetCar',
			'Jetty',
			'JikeSpider',
			'Joomla',
			'Jorgee',
			'JustView',
			'Jyxobot',
			'Kenjin Spider',
			'Keybot Translation-Search-Machine',
			'Keyword Density',
			'Kinza',
			'Kozmosbot',
			'LNSpiderguy',
			'LWP::Simple',
			'Lanshanbot',
			'Larbin',
			'Leap',
			'LeechFTP',
			'LeechGet',
			'LexiBot',
			'Lftp',
			'LibWeb',
			'Libwhisker',
			'LieBaoFast',
			'Lightspeedsystems',
			'Likse',
			'LinkScan',
			'LinkWalker',
			'Linkbot',
			'LinkextractorPro',
			'LinkpadBot',
			'LinksManager',
			'LinqiaMetadataDownloaderBot',
			'LinqiaRSSBot',
			'LinqiaScrapeBot',
			'Lipperhey',
			'Lipperhey Spider',
			'Litemage_walker',
			'Lmspider',
			'Ltx71',
			'MFC_Tear_Sample',
			'MIDown tool',
			'MIIxpc',
			'MJ12bot',
			'MQQBrowser',
			'MSFrontPage',
			'MSIECrawler',
			'MTRobot',
			'Mag-Net',
			'Magnet',
			'Mail.RU_Bot',
			'Majestic-SEO',
			'Majestic12',
			'Majestic SEO',
			'MarkMonitor',
			'MarkWatch',
			'Mass Downloader',
			'Masscan',
			'Mata Hari',
			'MauiBot',
			'Mb2345Browser',
			'MeanPath Bot',
			'Meanpathbot',
			'Mediatoolkitbot',
			'MegaIndex.ru',
			'Metauri',
			'MicroMessenger',
			'Microsoft Data Access',
			'Microsoft URL Control',
			'Minefield',
			'Mister PiX',
			'Moblie Safari',
			'Mojeek',
			'Mojolicious',
			'MolokaiBot',
			'Morfeus Fucking Scanner',
			'Mozlila',
			'Mr.4x3',
			'Msrabot',
			'Musobot',
			'NICErsPRO',
			'NPbot',
			'Name Intelligence',
			'Nameprotect',
			'Navroad',
			'NearSite',
			'Needle',
			'Nessus',
			'NetAnts',
			'NetLyzer',
			'NetMechanic',
			'NetSpider',
			'NetZIP',
			'Net Vampire',
			'Netcraft',
			'Nettrack',
			'Netvibes',
			'NextGenSearchBot',
			'Nibbler',
			'Niki-bot',
			'Nikto',
			'NimbleCrawler',
			'Nimbostratus',
			'Ninja',
			'Nmap',
			'Not',
			'Nuclei',
			'Nutch',
			'Octopus',
			'Offline Explorer',
			'Offline Navigator',
			'OnCrawl',
			'OpenLinkProfiler',
			'OpenVAS',
			'Openfind',
			'Openvas',
			'OrangeBot',
			'OrangeSpider',
			'OutclicksBot',
			'OutfoxBot',
			'PECL::HTTP',
			'PHPCrawl',
			'POE-Component-Client-HTTP',
			'PageAnalyzer',
			'PageGrabber',
			'PageScorer',
			'PageThing.com',
			'Page Analyzer',
			'Pandalytics',
			'Panscient',
			'Papa Foto',
			'Pavuk',
			'PeoplePal',
			'Petalbot',
			'Pi-Monster',
			'Picscout',
			'Picsearch',
			'PictureFinder',
			'Piepmatz',
			'Pimonster',
			'Pixray',
			'PleaseCrawl',
			'Pockey',
			'ProPowerBot',
			'ProWebWalker',
			'Probethenet',
			'Psbot',
			'Pu_iN',
			'Pump',
			'PxBroker',
			'PyCurl',
			'QueryN Metasearch',
			'Quick-Crawler',
			'RSSingBot',
			'RankActive',
			'RankActiveLinkBot',
			'RankFlex',
			'RankingBot',
			'RankingBot2',
			'Rankivabot',
			'RankurBot',
			'Re-re',
			'ReGet',
			'RealDownload',
			'Reaper',
			'RebelMouse',
			'Recorder',
			'RedesScrapy',
			'RepoMonkey',
			'Ripper',
			'RocketCrawler',
			'Rogerbot',
			'SBIder',
			'SEOkicks',
			'SEOkicks-Robot',
			'SEOlyticsCrawler',
			'SEOprofiler',
			'SEOstats',
			'SISTRIX',
			'SMTBot',
			'SalesIntelligent',
			'ScanAlert',
			'Scanbot',
			'ScoutJet',
			'Scrapy',
			'Screaming',
			'ScreenerBot',
			'ScrepyBot',
			'Searchestate',
			'SearchmetricsBot',
			'Seekport',
			'SemanticJuice',
			'Semrush',
			'SemrushBot',
			'SentiBot',
			'SeoSiteCheckup',
			'SeobilityBot',
			'Seomoz',
			'Shodan',
			'Siphon',
			'SiteCheckerBotCrawler',
			'SiteExplorer',
			'SiteLockSpider',
			'SiteSnagger',
			'SiteSucker',
			'Site Sucker',
			'Sitebeam',
			'Siteimprove',
			'Sitevigil',
			'SlySearch',
			'SmartDownload',
			'Snake',
			'Snapbot',
			'Snoopy',
			'SocialRankIOBot',
			'Sociscraper',
			'Sogou web spider',
			'Sosospider',
			'Sottopop',
			'SpaceBison',
			'Spammen',
			'SpankBot',
			'Spanner',
			'Spbot',
			'Spinn3r',
			'SputnikBot',
			'Sqlmap',
			'Sqlworm',
			'Sqworm',
			'Steeler',
			'Stripper',
			'Sucker',
			'Sucuri',
			'SuperBot',
			'SuperHTTP',
			'Surfbot',
			'SurveyBot',
			'Suzuran',
			'Swiftbot',
			'Szukacz',
			'T0PHackTeam',
			'T8Abot',
			'Teleport',
			'TeleportPro',
			'Telesoft',
			'Telesphoreo',
			'Telesphorep',
			'TheNomad',
			'The Intraformant',
			'Thumbor',
			'TightTwatBot',
			'Titan',
			'Toata',
			'Toweyabot',
			'Tracemyfile',
			'Trendiction',
			'Trendictionbot',
			'True_Robot',
			'Turingos',
			'Turnitin',
			'TurnitinBot',
			'TwengaBot',
			'Twice',
			'Typhoeus',
			'URLy.Warning',
			'URLy Warning',
			'UnisterBot',
			'Upflow',
			'V-BOT',
			'VB Project',
			'VCI',
			'Vacuum',
			'Vagabondo',
			'VelenPublicWebCrawler',
			'VeriCiteCrawler',
			'VidibleScraper',
			'Virusdie',
			'VoidEYE',
			'Voil',
			'Voltron',
			'WASALive-Bot',
			'WBSearchBot',
			'WEBDAV',
			'WISENutbot',
			'WPScan',
			'WWW-Collector-E',
			'WWW-Mechanize',
			'WWW::Mechanize',
			'WWWOFFLE',
			'Wallpapers',
			'Wallpapers/3.0',
			'WallpapersHD',
			'WeSEE',
			'WebAuto',
			'WebBandit',
			'WebCollage',
			'WebCopier',
			'WebEnhancer',
			'WebFetch',
			'WebFuck',
			'WebGo IS',
			'WebImageCollector',
			'WebLeacher',
			'WebPix',
			'WebReaper',
			'WebSauger',
			'WebStripper',
			'WebSucker',
			'WebWhacker',
			'WebZIP',
			'Web Auto',
			'Web Collage',
			'Web Enhancer',
			'Web Fetch',
			'Web Fuck',
			'Web Pix',
			'Web Sauger',
			'Web Sucker',
			'Webalta',
			'WebmasterWorldForumBot',
			'Webshag',
			'WebsiteExtractor',
			'WebsiteQuester',
			'Website Quester',
			'Webster',
			'Whack',
			'Whacker',
			'Whatweb',
			'Who.is Bot',
			'Widow',
			'WinHTTrack',
			'WiseGuys Robot',
			'Wonderbot',
			'Woobot',
			'Wotbox',
			'Wprecon',
			'Xaldon WebSpider',
			'Xaldon_WebSpider',
			'Xenu',
			'YoudaoBot',
			'Zade',
			'Zauba',
			'Zermelo',
			'Zeus',
			'Zitebot',
			'ZmEu',
			'ZoomBot',
			'ZoominfoBot',
			'ZumBot',
			'ZyBorg',
			'adscanner',
			'archive.org_bot',
			'arquivo-web-crawler',
			'arquivo.pt',
			'autoemailspider',
			'backlink-check',
			'cah.io.community',
			'check1.exe',
			'clark-crawler',
			'coccocbot',
			'cognitiveseo',
			'com.plumanalytics',
			'crawl.sogou.com',
			'crawler.feedback',
			'crawler4j',
			'dataforseo.com',
			'demandbase-bot',
			'domainsproject.org',
			'eCatch',
			'evc-batch',
			'facebookscraper',
			'gopher',
			'heritrix',
			'instabid',
			'internetVista monitor',
			'ips-agent',
			'isitwp.com',
			'iubenda-radar',
			'linkdexbot',
			'lwp-request',
			'lwp-trivial',
			'magpie-crawler',
			'meanpathbot',
			'mediawords',
			'muhstik-scan',
			'netEstate NE Crawler',
			'oBot',
			'page scorer',
			'pcBrowser',
			'plumanalytics',
			'polaris version',
			'probe-image-size',
			'ripz',
			's1z.ru',
			'satoristudio.net',
			'scalaj-http',
			'scan.lol',
			'seobility',
			'seocompany.store',
			'seoscanners',
			'seostar',
			'serpstatbot',
			'sexsearcher',
			'sitechecker.pro',
			'siteripz',
			'sogouspider',
			'sp_auditbot',
			'spyfu',
			'sysscan',
			'tAkeOut',
			'trendiction.com',
			'trendiction.de',
			'ubermetrics-technologies.com',
			'voyagerx.com',
			'webgains-bot',
			'webmeup-crawler',
			'webpros.com',
			'webprosbot',
			'x09Mozilla',
			'x22Mozilla',
			'xpymep1.exe',
			'zauba.io',
			'zgrab',
			'petalsearch',
			'protopage',
			'Miniflux',
			'Feeder',
			'Semanticbot',
			'ImageFetcher',
		);

		foreach ( $bots as $b ) {
			if ( stripos( $agent, $b ) !== false ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * This function insures compatibility accross different
	 * Greyd.Suite versions. Can be removed in future versions.
	 *
	 * @see \Greyd\Helper
	 * @since 1.4.5
	 */
	public static function render_info_box( $atts = array(), $echo = false ) {
		if ( method_exists( '\Greyd\Helper', 'render_info_box' ) ) {
			return \Greyd\Helper::render_info_box( $atts, $echo );
		} elseif ( method_exists( '\vc\helper', 'render_info_box' ) ) {
			return \vc\helper::render_info_box( $atts, $echo );
		}
	}

	/**
	 * @deprecated 1.4
	 * @see \vc\helper::render_multiselect()
	 *
	 * @param string $name      name of the input
	 * @param array  $options   all options as $value => $label
	 * @param array  $args      optional arguments for the element (value, placeholder, classes...)
	 *
	 * @return string html element
	 */
	public static function render_multiselect( $name, $options, $args = array() ) {

		if ( method_exists( '\Greyd\Helper', 'render_multiselect' ) ) {
			return \Greyd\Helper::render_multiselect( $name, $options, $args );
		}

		if ( method_exists( '\vc\helper', 'render_multiselect' ) ) {
			return \vc\helper::render_multiselect( $name, $options, $args );
		}

		if ( ! is_array( $options ) ) {
			return;
		}

		// make args
		$default_args = array(
			'value'    => '',
			'id'       => '',
			'class'    => '',
			'required' => false,
			'title'    => '',
		);
		$args         = array_merge( $default_args, (array) $args );

		// atts
		$value    = $args['value'];
		$values   = explode( ',', $args['value'] );
		$id       = ! empty( $args['id'] ) ? "id='" . $args['id'] . "'" : '';
		$class    = $args['class'];
		$required = $args['required'] ? 'required' : '';
		$title    = ! empty( $args['title'] ) ? "title='" . $args['title'] . "'" : '';

		ob_start();

		// select
		echo "<select multiple $id name='$name' value='$value' class='$class' $required $title>";
		foreach ( $options as $value => $label ) {
			echo '<option ' . ( isset( array_flip( $values )[ $value ] ) ? 'selected' : '' ) . "' value='" . trim( $value ) . "'>" . trim( $label ) . '</option>';
		}
		echo '</select>';

		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}
}
