<?php
/**
 * Register all blocks necessary for Greyd.Forms.
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Blocks( $config );
class Blocks {

	/**
	 * Hold the plugin config
	 *
	 * @var object
	 */
	private $config;

	/**
	 * Hold the plugin url
	 *
	 * @var string
	 */
	private $plugin_url;

	/**
	 * Hold the blocks assets url
	 *
	 * @var string
	 */
	private $assets_url;

	/**
	 * Hold the current plugin version
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Constructor
	 */
	public function __construct( $config ) {

		// check if block editor is active
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		// class vars
		$this->config = (object) $config;

		// register the blocks
		add_action( 'init', array( $this, 'register_blocks' ), 95 );
		add_filter( 'block_categories_all', array( $this, 'add_form_category' ), 20, 2 );

		// only allow specific blocks inside forms
		// add_filter( 'allowed_block_types_all', array( $this, 'filter_allowed_blocks' ), 98, 2 );
		add_filter( 'wp_kses_allowed_html', array( $this, 'filter_allowed_html_tags' ), 98, 2 );
	}

	/**
	 * Register the blocks
	 */
	public function register_blocks() {

		// debug( wp_kses_allowed_html( 'post' ) );

		// register the form block
		register_block_type(
			'greyd/form',
			array(
				'supports'        => array(
					'anchor' => true,
				),
				'attributes'      => array(
					'anchor'         => array( 'type' => 'string' ),
					'dynamic_parent' => array( 'type' => 'string' ), // dynamic template backend helper
					'dynamic_value'  => array( 'type' => 'string' ), // dynamic template frontend helper
					'dynamic_fields' => array( 'type' => 'array' ),
					'inline_css'     => array(
						'type'    => 'string',
						'default' => '',
					),
					'inline_css_id'  => array(
						'type'    => 'string',
						'default' => '',
					),
					'id'             => array(
						'type'    => 'string',
						'default' => '',
					),
				),
				'editor_script'   => 'greyd-form-block',
				'render_callback' => array( '\Greyd\Forms\Render', 'render_form' ),
				'script'          => 'greyd-forms-main-script',
				'style'           => Helper::is_greyd_classic() ? '' : 'greyd-forms-main-style',
				'editor_style'    => Helper::is_greyd_classic() ? '' : 'greyd-forms-editor-form-style'

			)
		);

		$register_args = array(
			'editor_script'       => 'greyd-form-inputs',
			'style'               => Helper::is_greyd_classic() || is_admin() ? '' : 'greyd-forms-main-style',
			'editor_style'        => Helper::is_greyd_classic() ? '' : 'greyd-forms-editor-input-style'

		);

		/**
		 * Register the fields
		 */
		register_block_type( 'greyd-forms/input', $register_args );
		register_block_type( 'greyd-forms/checkbox', $register_args );
		register_block_type( 'greyd-forms/recaptcha', $register_args );
		register_block_type( 'greyd-forms/hidden-field', $register_args );
		register_block_type( 'greyd-forms/radiobuttons', array_merge( 
			$register_args,
			array(
				'script' => 'greyd-forms-live-survey-script'
			) 
		) );
		register_block_type( 'greyd-forms/radiobutton-item', $register_args );
		register_block_type( 'greyd-forms/submitbutton', $register_args );
		register_block_type( 'greyd-forms/upload', $register_args );
		register_block_type( 'greyd-forms/dropdown', $register_args );
		register_block_type( 'greyd-forms/iconpanels', $register_args );
		register_block_type( 'greyd-forms/iconpanel', $register_args );
		register_block_type( 'greyd-forms/condition', array_merge(
			$register_args,
			array(
				'script' => 'greyd-forms-condition-script'
			)
		) );
		register_block_type( 'greyd-forms/multistep-container', array_merge(
			$register_args,
			array(
				'script' => 'greyd-forms-multistep-script'
			)
		)  );
		register_block_type( 'greyd-forms/multistep-step', $register_args );
		register_block_type( 'greyd-forms/multistep-button', $register_args );
		register_block_type( 'greyd-forms/password', $register_args );
		register_block_type( 'greyd-forms/range', $register_args );
		register_block_type( 'greyd-forms/math', $register_args );
	}

	/**
	 * Add a new category for the form blocks.
	 */
	public function add_form_category( $categories, $post ) {
		return array_merge(
			array(
				array(
					'slug'  => 'greyd-forms',
					'title' => 'Greyd.Forms',
				),
			),
			$categories
		);
	}

	/**
	 * Filters the allowed block types for forms.
	 * @deprecated since 1.7.3
	 *
	 * @param string[]                $allowed_block_types                  Array of block type slugs, or boolean to enable/disable all.
	 *                                                                      Default true (all registered block types supported).
	 * @param WP_Block_Editor_Context $block_editor_context  The current block editor context.
	 *
	 * @return string[] $allowed_block_types
	 */
	public function filter_allowed_blocks( $allowed_blocks, $block_editor_context ) {

		if (
			$block_editor_context &&
			is_object( $block_editor_context ) &&
			isset( $block_editor_context->post ) &&
			is_object( $block_editor_context->post ) &&
			isset( $block_editor_context->post->post_type ) &&
			$block_editor_context->post->post_type == $this->config->plugin_post_type
		) {
			$allowed_blocks = array(

				// wp
				'core/group',
				'core/columns',
				'core/column',
				'core/spacer',
				'core/image',
				'core/paragraph',
				'core/heading',
				'core/list',
				'core/cover',
				'core/media-text',
				'core/quote',
				'core/file',
				'core/separator',

				// greyd_blocks
				'greyd/list',
				'greyd/list-item',
				'greyd/box',

				// greyd_forms
				'greyd-forms/input',
				'greyd-forms/checkbox',
				'greyd-forms/recaptcha',
				'greyd-forms/hidden-field',
				'greyd-forms/radiobuttons',
				'greyd-forms/radiobutton-item',
				'greyd-forms/submitbutton',
				'greyd-forms/upload',
				'greyd-forms/dropdown',
				'greyd-forms/iconpanels',
				'greyd-forms/iconpanel',
				'greyd-forms/condition',
				'greyd-forms/multistep-container',
				'greyd-forms/multistep-step',
				'greyd-forms/multistep-button',
				'greyd-forms/password',
				'greyd-forms/range',
				'greyd-forms/math',
				// ... add all the other form inputs
			);
		}

		/**
		 * Filters the allowed block types for forms.
		 *
		 * @filter greyd_forms_filter_allowed_blocks
		 *
		 * @param bool|string[] $allowed_block_types             Array of block type slugs, or boolean to enable/disable all.
		 * @param WP_Block_Editor_Context $block_editor_context  The current block editor context.
		 *
		 * @return string[] $allowed_block_types
		 */
		return apply_filters( 'greyd_forms_filter_allowed_blocks', $allowed_blocks, $block_editor_context );
	}

	/**
	 * Filters the HTML tags that are allowed inside forms.
	 *
	 * HTML tags and attribute names are case-insensitive in HTML but must be
	 * added to the KSES allow list in lowercase. An item added to the allow list
	 * in upper or mixed case will not recognized as permitted by KSES.
	 *
	 * @param array[] $html    Allowed HTML tags.
	 * @param string  $context Context name.
	 */
	public function filter_allowed_html_tags( $html, $context ) {

		if ( $context !== 'post' ) {
			return $html;
		}

		$default_attributes = array(
			'id'            => true,
			'class'         => true,
			'type'          => true,
			'name'          => true,
			'value'         => true,
			'required'      => true,
			'autocomplete'  => true,
			'placeholder'   => true,
			'accept'        => true,
			'title'         => true,
			'min'           => true,
			'max'           => true,
			'step'          => true,
			'start'         => true,
			'minlength'     => true,
			'maxlength'     => true,
			'rows'          => true,
			'data-required' => true,
			'data-max'      => true,
			'pattern'       => true,
		);

		$html_tags = array(
			'form'     => array(
				'id'      => true,
				'class'   => true,
				'name'    => true,
				'method'  => true,
				'enctype' => true,
				'data-*'  => true,
			),
			'input'    => $default_attributes,
			'textarea' => $default_attributes,
			'select'   => $default_attributes,
			'option'   => array(
				'value' => true,
			),
			'fieldset' => array(
				'required'      => true,
				'title'         => true,
				'data-required' => true,
			),
		);

		foreach ( $html_tags as $tag => $allowed_attributes ) {

			if ( ! isset( $html[ $tag ] ) ) {
				$html[ $tag ] = $allowed_attributes;
			} else {
				$html[ $tag ] = array_merge( $html[ $tag ], $allowed_attributes );
			}
		}

		/**
		 * Filters the HTML tags that are allowed inside forms.
		 *
		 * @filter greyd_forms_filter_allowed_html_tags
		 *
		 * @param array[] $html    Allowed HTML tags.
		 * @param string  $context Context name.
		 *
		 * @return array[] $html   Allowed HTML tags.
		 */
		return apply_filters( 'greyd_forms_filter_allowed_html_tags', $html, $context );
	}
}
