<?php
/**
 * Block render callbacks for the frontend.
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Render( $config );
class Render {

	/**
	 * Holds the plugin config.
	 */
	private $config;

	/**
	 * Class constructor.
	 */
	public function __construct( $config ) {

		// check if Gutenberg is active.
		if ( is_admin() ) {
			return;
		}

		// set config
		$this->config = (object) $config;

		// display errors before form
		add_action( 'greyd_forms_display_before', array( $this, 'display_form_errors' ), 10, 1 );

		// hook block rendering
		add_filter( 'render_block', array( $this, 'render_block' ), 10, 2 );
	}

	/**
	 * Form render callback.
	 */
	public static function render_form( $atts ) {

		$post_id = isset( $atts['id'] ) ? intval( $atts['id'] ) : null;
		if ( ! $post_id ) {
			return false;
		}

		/**
		 * Filter post id before inserting the form.
		 *
		 * @since 1.3.6
		 */
		$post_id = Helper::filter_post_id( $post_id, 'tp_forms' );

		/**
		 * Get the post object.
		 */
		$post = get_post( $post_id );
		if ( ! $post || ! ( $post instanceof \WP_Post ) || $post->post_status !== 'publish' ) {
			return false;
		}

		/**
		 * Render shortcode form
		 *
		 * @deprecated since greyd_blocks
		 */
		if ( ! has_blocks( $post->post_content ) ) {
			return Shortcode::render_form( $atts, '' );
		}

		// get attributes
		$form_id = wp_unique_id( 'greyd-form-' );
		$title   = urlencode( preg_replace( '/\s/', '', esc_html( get_the_title( $post_id ) ) ) );
		$nonce   = wp_create_nonce( 'handle_form_data_' . $post_id );

		// get meta
		$block_send     = get_post_meta( $post_id, 'block_send', true );
		$required_class = 'requirement-style__' . ( $block_send ? 'none' : get_post_meta( $post_id, 'required_style', true ) );
		$required_class = empty( $required_class ) ? 'required' : $required_class;
		$hide_tooltips  = get_post_meta( $post_id, 'hide_tooltips', true );
		$after_form     = get_post_meta( $post_id, 'after_form', true );
		$popup_bg_color = get_post_meta( $post_id, 'popup_bg_color', true );

		$form_mode          = get_post_meta( $post_id, 'form_mode', true );
		$registration_meta  = get_post_meta( $post_id, 'user_registration', true );
		$registration_meta  = json_decode( json_encode( $registration_meta ), true );
		$selected_role_name = isset( $registration_meta['role'] ) ? $registration_meta['role'] : null;

		// Don't render the userdata update form if the user has no permission (role)
		// Apply only in the frontend, not in the admin area and not in the rest api
		if ( 'update_user' === $form_mode && ! is_admin() && ! Helper::is_rest_request() ) {
			$current_user = wp_get_current_user();

			// user is not logged in
			if ( ! $current_user ) {
				return '';
			}
			// user is logged in but doesn't have the selected role
			elseif ( ! empty( $selected_role_name ) && ! in_array( $selected_role_name, (array) $current_user->roles ) ) {
				return '';
			}
		}

		// get hidden fields
		$hidden_fields = (array) apply_filters( 'formshortcode_add_hidden_fields', array(), $post_id );

		// build css class names
		$classes = apply_filters(
			'greyd_forms_css_classes',
			array(
				'title'          => '„' . $title . '“',
				'required_class' => $required_class,
				'hide_tooltips'  => $hide_tooltips,
			),
			$post->post_content,
			$post_id
		);
		$classes = trim( preg_replace( '/\s+/', ' ', implode( ' ', $classes ) ) );

		// start rendering
		ob_start();

		/**
		 * Display contents before form.
		 *
		 * @action greyd_forms_display_before
		 * @since 1.2.2
		 * @see formshortcode_errors_before_form
		 */
		do_action( 'greyd_forms_display_before', $post_id );

		// wrapper
		echo "<div class='forms_wrapper'>";

		// skip link
		echo "<a href='#after-{$form_id}' class='greyd_skip_link'>" . __( "Skip form", 'greyd_forms' ) . '</a>';
		// echo "<div id='before-{$form_id}' class='greyd_skip_target'>".__("Form skipped", 'greyd_forms')."</div>";

		// form
		echo "<form
			class='greyd_form $classes' name='form_$post_id' id='$form_id'
			data-id='$post_id' data-form-mode='{$form_mode}' data-title='$title' data-nonce='$nonce' data-send='" . ( $block_send ? 'prevent' : 'enable' ) . "'
			method='POST' enctype='multipart/form-data'
		>";

		// Render content
		echo do_blocks(
			/**
			 * Filter form content to be rendered.
			 *
			 * @filter formblocks_raw_content
			 */
			apply_filters( 'formblocks_raw_content', $post->post_content, $post )
		);

		// add hidden fields
		if ( ! $block_send && count( $hidden_fields ) > 0 ) {
			foreach ( $hidden_fields as $name => $val ) {
				echo "<input type='hidden' name='$name' value='$val'></input>";
			}
		}

		// end of form
		echo '</form>';

		if ( ! $block_send ) {

			// display required label after
			if ( $required_class === 'required' || $required_class === 'both' ) {
				$required_text  = get_post_meta( $post_id, 'required_text', true );
				$required_align = get_post_meta( $post_id, 'required_align', true );
				$required_text  = ! empty( $required_text ) ? $required_text : __( "Mandatory", 'greyd_forms' );
				if ( $required_align !== 'hidden' ) {
					echo "<div class='flex-$required_align $gutter'><label class='requirement_after'><span class='requ_star'>* </span>" . $required_text . '</label></div>';
				}
			}

			// message after
			echo "<div class='after-message animate_fast'><div class='message'></div></div>";

			// optional popup after
			if ( $after_form == 'popup' || $after_form == 'fullscreen' ) {
				$popup = get_posts(
					array(
						'name'      => 'form_popup_' . $post_id,
						'post_type' => 'dynamic_template',
					)
				);
				if ( $popup ) {
					$popup = $popup[0];
					echo "<div class='after-popup $after_form' " . ( ! empty( $popup_bg_color ) ? "style='background-color:" . $popup_bg_color . ";'" : '' ) . '>' . (
						has_blocks( $popup->post_content ) ? do_blocks( $popup->post_content ) : do_shortcode( $popup->post_content ) ) .
					'</div>';
				} else {
					\Greyd\Forms\Helper::make_message(
						sprintf(
							__( "No templated with the name \"%s\" found", 'greyd_forms' ),
							'<strong>form_popup_' . $post_id . '</strong>'
						),
						'danger'
					);
				}
			}
		}

		// skip link
		// echo "<a href='#before-{$form_id}' class='greyd_skip_link'>".__("Skip form", 'greyd_forms')."</a>";
		echo "<div id='after-{$form_id}' class='greyd_skip_target'>" . __( "Form skipped", 'greyd_forms' ) . '</div>';

		// end of wrapper
		echo '</div>';

		/**
		 * Display contents after form.
		 *
		 * @action greyd_forms_display_after
		 * @since 1.2.2
		 */
		do_action( 'greyd_forms_display_after', $post_id );

		// Return
		$return = ob_get_contents();
		ob_end_clean();
		return $return;

	}

	/**
	 * Display errors if mail settings are not correct.
	 *
	 * @param int $post_id
	 */
	public function display_form_errors( $post_id ) {

		if ( get_post_meta( $post_id, 'block_send', true ) ) {
			return;
		}

		$verify_user          = get_post_meta( $post_id, 'verify_user', true ) == 'verify_user' ? true : false;
		$verify_mail_to_field = get_post_meta( $post_id, 'verify_email', true );
		$user_gets_mail       = get_post_meta( $post_id, 'user_gets_mail', true );
		$mail_to_field        = get_post_meta( $post_id, 'user_email', true );

		// no verify email
		if ( $verify_user && Helper::is_select_empty( $verify_mail_to_field ) ) {
			echo \Greyd\Forms\Helper::make_message( __( "Please set the field for the user’s email, otherwise the verification email cannot be sent.", 'greyd_forms' ), 'danger' );
		}
		if ( $user_gets_mail && Helper::is_select_empty( $mail_to_field ) && Helper::is_select_empty( $verify_mail_to_field ) ) {
			echo \Greyd\Forms\Helper::make_message( __( "Please set the field for the user’s email, otherwise the confirmation email cannot be sent.", 'greyd_forms' ), 'danger' );
		}
	}

	/**
	 * Hook block rendering
	 * https://developer.wordpress.org/reference/hooks/render_block/
	 *
	 * @param string $block_content     pre-rendered Block Content
	 * @param object $block             parsed Block
	 *
	 * @return string $block_content    altered Block Content
	 */
	public function render_block( $block_content, $block ) {

		// placeholder to replace (keep same as in inputs.js)
		$render_placeholder = array(
			'agb'      => '<!--agb_text-->',
			'id'       => '___idXXX',
			'captcha'  => 'XXXcaptcha_sitekeyXXX',
			'tooltip'  => '{tooltip}',
			'enter'    => '{enter}',
			'select'   => '{select}',
			'required' => '{required}',
			'options'  => '{options}',
		);

		// Hidden field
		if ( $block['blockName'] === 'greyd-forms/hidden-field' ) {

			if ( isset( $block['attrs']['isDynamic'] ) && $block['attrs']['isDynamic'] ) {
				$type = isset( $block['attrs']['type'] ) ? $block['attrs']['type'] : '';
				$name = isset( $block['attrs']['value'] ) ? $block['attrs']['value'] : '';

				if ( $type === 'cookie' ) {
					$values = class_exists( '\Greyd\Extensions\Cookie_Handler' ) ? \Greyd\Extensions\Cookie_Handler::get_all_cookie_values() : \url_handler::get_all_cookie_values();
				} elseif ( $type === 'url' ) {
					global $wp;
					$values = array( $name => home_url( $wp->request ) );
				} else {
					$values = class_exists( '\Greyd\Extensions\Cookie_Handler' ) ? \Greyd\Extensions\Cookie_Handler::get_all_url_values() : \url_handler::get_all_url_values();
				}

				$default = isset( $block['attrs']['default'] ) ? $block['attrs']['default'] : '';
				$value   = isset( $values[ $name ] ) ? $values[ $name ] : $default;

				$block_content = preg_replace( '/value="([^"]*)"/', "value='{$value}'", $block_content );
			}
		}

		// Input field
		elseif ( $block['blockName'] === 'greyd-forms/input' ) {

			/**
			 * Convert incorrect address autocomplete names
			 *
			 * @since 1.3.9
			 * can be removed in future version
			 */
			if ( preg_match( '/(autocomplete="adress-level2"|autocomplete="street-adress")/', $block_content ) ) {
				$block_content = str_replace(
					array(
						'autocomplete="adress-level2"',
						'autocomplete="street-adress"',
					),
					array(
						'autocomplete="address-level2"',
						'autocomplete="street-address"',
					),
					$block_content
				);
			}
		}

		// Checkbox
		elseif ( $block['blockName'] === 'greyd-forms/checkbox' ) {

			if ( isset( $block['attrs']['useSetting'] ) && $block['attrs']['useSetting'] ) {
				$block_content = str_replace(
					$render_placeholder['agb'],
					html_entity_decode( esc_attr( get_option( 'agb_text', __( "I hereby accept the terms and conditions.", 'greyd_forms' ) ) ) ),
					$block_content
				);
			}
		}

		// reCAPTCHA
		elseif ( $block['blockName'] === 'greyd-forms/recaptcha' ) {
			// debug( esc_attr( $block_content ) );
			$sitekey = get_option( 'captcha_sitekey' );
			if ( empty( $sitekey ) ) {
				$block_content = "<div class='message danger'>" . __( "Please enter your Google reCAPTCHA site key via 'Forms > Settings', otherwise the reCAPTCHA cannot be executed.", 'greyd_forms' ) . '</div>';
			} else {
				$block_content = str_replace( $render_placeholder['captcha'], $sitekey, $block_content );
			}
		}

		// set tooltip
		if (
			$block['blockName'] === 'greyd-forms/input'
			|| $block['blockName'] === 'greyd-forms/dropdown'
			|| $block['blockName'] === 'greyd-forms/radiobuttons'
			|| $block['blockName'] === 'greyd-forms/iconpanels'
			|| $block['blockName'] === 'greyd-forms/upload'
		) {

			$tooltip = '';

			switch ( $block['blockName'] ) {

				case 'greyd-forms/input':
					$type    = isset( $block['attrs']['type'] ) ? esc_attr( $block['attrs']['type'] ) : 'text';
					$tooltip = $this->get_tooltip( $type, $block['attrs'] );
					break;

				case 'greyd-forms/dropdown':
				case 'greyd-forms/radiobuttons':
				case 'greyd-forms/iconpanels':
					$tooltip = $this->get_tooltip( 'multiselect' );
					break;

				case 'greyd-forms/upload':
					$file    = isset( $block['attrs']['file'] ) ? $block['attrs']['file'] : array();
					$tooltip = $this->get_tooltip( 'file', $file );
					break;
			}

			$block_content = str_replace(
				array(
					$render_placeholder['tooltip'],
					$render_placeholder['enter'],
					$render_placeholder['select'],
					$render_placeholder['required'],
					$render_placeholder['options'],
				),
				array(
					$tooltip,
					__( "enter here", 'greyd_forms' ),
					__( "please select", 'greyd_forms' ),
					__( "Please fill out this field", 'greyd_forms' ),
					__( "Please select an option", 'greyd_forms' ),
				),
				$block_content
			);
		}

		/**
		 * Set unique option id for radio buttons.
		 *
		 * @since 1.4.0
		 */
		if ( $block['blockName'] === 'greyd-forms/radiobutton-item' ) {
			$uniqid        = uniqid( '___id' );
			$block_content = preg_replace( '/(<label[^>]+for="[^"]+|<input[^>]+id="[^"]+)___id[^"]+"/m', '$1' . $uniqid . '"', $block_content );
		}

		/**
		 * Render math form fields.
		 *
		 * @since 1.5.0
		 */
		if ( $block['blockName'] === 'greyd-forms/math' ) {

			$formula    = isset( $block['attrs']['formula'] ) ? $block['attrs']['formula'] : '';
			$name       = isset( $block['attrs']['name'] ) ? $block['attrs']['name'] : '';
			$conditions = isset( $block['attrs']['conditions'] ) ? $block['attrs']['conditions'] : array();

			$realformula    = Helper::get_formula( Helper::build_form_tags( $formula ) );
			$displayformula = Helper::display_formula( Helper::build_form_tags( $formula ) );
			$result         = Helper::build_form_tags( '{' . $name . '}' );
			$block_content  = str_replace( '||displayformula||', $displayformula, $block_content );
			$block_content  = str_replace( '||result||', $result, $block_content );

			foreach ( $conditions as $index => $condition ) {
				$conditionformula = Helper::display_formula( Helper::build_form_tags( $condition['result'] ) );
				$block_content    = str_replace( "||conditionformula_{$index}||", $conditionformula, $block_content );

				$field1        = Helper::get_formula( Helper::build_form_tags( $condition['field1'] ) );
				$block_content = str_replace( "||field_1_{$index}||", $field1, $block_content );
				$field2        = Helper::get_formula( Helper::build_form_tags( $condition['field2'] ) );
				$block_content = str_replace( "||field_2_{$index}||", $field2, $block_content );
				$result        = Helper::get_formula( Helper::build_form_tags( $condition['result'] ) );
				$block_content = str_replace( "||result_{$index}||", $result, $block_content );
			}
		}

		/**
		 * Filter multistep wrapper
		 */
		if ( $block['blockName'] === 'greyd-forms/multistep-container' ) {

			// add id to multistep wrapper
			$block_content = preg_replace(
				'/<div class="multistep_wrapper/',
				'<div id='.wp_unique_id( 'multistep-' ).' class="multistep_wrapper',
				$block_content
			);

			
			if (
				isset($block['attrs']['progressbar'])
				&& isset($block['attrs']['progressbar']['enable'])
				&& $block['attrs']['progressbar']['enable']
				&& isset($block['attrs']['progressbar']['type'])
				&& $block['attrs']['progressbar']['type'] === 'pagination'
				&& isset($block['innerBlocks'])
			) {
				// make sure pagination is set correctly
				$block_content = str_replace(
					'<div class="multistep_pagination"></div>',
					'<div class="multistep_pagination">'.str_repeat( '<span></span>', count($block['innerBlocks']) ).'</div>',
					$block_content
				);
			}
		}


		/**
		 * Replace validation regex patterns.
		 */
		if ( $block['blockName'] === 'greyd-forms/input' && isset($block['attrs']['type']) ) {

			$field_type = isset($block['attrs']['type']) ? $block['attrs']['type'] : '';

			if ( $field_type === 'email' ) {
				$block_content = preg_replace(
					'/pattern="[^"]+"/',
					'pattern="\w+([\-+\.]\w+)*@\w+([\-\.]\w+)*.\w+([\-\.]\w+)*"',
					$block_content
				);
			} else if ( $field_type === 'tel' ) {
				$block_content = preg_replace(
					'/pattern="[^"]+"/',
					'pattern="\(?\+?[\s0-9]{0,4}\(?[0-9]{1,4}\)?[\-\s\/0-9]{4,16}"',
					$block_content
				);
			} else if ( $field_type === 'url' ) {
				$block_content = preg_replace(
					'/pattern="[^"]+"/',
					'pattern="(http(s)?:\/\/)?[\w\.\-]+(\.[\w\.\-]+)+[\w\-\.\_\~\:\/\?\#\[\]\@\!\$\&\'\(\)\*\+\,\;\=\.]+"',
					$block_content
				);
			} else if ( $field_type === 'date' ) {
				$block_content = preg_replace(
					'/pattern="[^"]+"/',
					'pattern="[0-9]{4}\-[0-9]{2}\-[0-9]{2}"',
					$block_content
				);
			}
		}

		if ( $block['blockName'] === 'greyd/form' ) {

			/**
			 * Replace the uniqid.
			 * Each form rendered on the page gets their own uniqid.
			 */
			$uniqid        = uniqid( '___id' );
			$block_content = str_replace( $render_placeholder['id'], $uniqid, $block_content );
		}

		return $block_content;
	}

	/**
	 * Get the tooltip default text depending on the input type.
	 *
	 * @param string $type  Type of the input field.
	 * @param object $atts  Block attributes to include in the text.
	 *
	 * @return string       Default tooltip text.
	 */
	public function get_tooltip( $type, $atts = array() ) {

		$tooltip  = '';
		$tooltips = array(
			'email'       => __( "The email address should contain an '@'character and a valid domain with a period.", 'greyd_forms' ),
			'url'         => __( "The URL should contain a domain name after a period.", 'greyd_forms' ),
			'tel'         => __( "The phone number can only consist of numbers and the following characters: + ( ) - / ", 'greyd_forms' ),
			'number'      => array(
				'default' => __( "The input must be a number.", 'greyd_forms' ),
				'min'     => __( "The number must have a value of minimum %d.", 'greyd_forms' ),
				'max'     => __( "The number must have a value of maximum %d.", 'greyd_forms' ),
				'minmax'  => __( "The number must be between the values %$1d and %$2d.", 'greyd_forms' ),
			),
			'multiselect' => __( "Multiselect possible.", 'greyd_forms' ),
			'file'        => array(
				'default'         => __( "The file must have one of the following formats: '%s'.", 'greyd_forms' ),
				'image/*'         => __( "The file must be an image.", 'greyd_forms' ),
				'.zip'            => __( "The file must be a ZIP archive.", 'greyd_forms' ),
				'application/pdf' => __( "The file must be a PDF.", 'greyd_forms' ),
				'text/*'          => __( "The file must be a text file.", 'greyd_forms' ),
				'video/*'         => __( "The file must have a valid video format.", 'greyd_forms' ),
				'audio/*'         => __( "The file must have a valid audio format.", 'greyd_forms' ),
				'size'            => __( " It must not be bigger than %s MB.", 'greyd_forms' ),
				'size_only'       => __( "The file must not be bigger than %s MB.", 'greyd_forms' ),
			),
		);

		if ( ! isset( $tooltips[ $type ] ) ) {
			return '';
		}

		// type has only 1 possibility
		if ( gettype( $tooltips[ $type ] ) === 'string' ) {
			return $tooltips[ $type ];
		}

		if ( $type === 'number' ) {

			if ( isset( $atts['min'] ) && ! empty( $atts['min'] ) && isset( $atts['max'] ) && ! empty( $atts['max'] ) ) {
				$tooltip = sprintf(
					$tooltips[ $type ]['minmax'],
					$atts['min'],
					$atts['max']
				);
			} elseif ( isset( $atts['min'] ) && ! empty( $atts['min'] ) ) {
				$tooltip = sprintf(
					$tooltips[ $type ]['min'],
					$atts['min']
				);
			} elseif ( isset( $atts['max'] ) && ! empty( $atts['max'] ) ) {
				$tooltip = sprintf(
					$tooltips[ $type ]['max'],
					$atts['max']
				);
			} else {
				$tooltip = $tooltips[ $type ]['default'];
			}
		} elseif ( $type === 'file' ) {

			$filetype = isset( $atts['type'] ) ? $atts['type'] : '*';
			if ( isset( $tooltips[ $type ][ $filetype ] ) ) {
				$tooltip = $tooltips[ $type ][ $filetype ];
			} elseif ( $filetype === 'other' ) {
				$filetype = $atts['custom'];
				$tooltip  = sprintf(
					$tooltips[ $type ]['default'],
					$filetype
				);
			}

			if ( isset( $atts['max'] ) && ! empty( $atts['max'] ) ) {
				if ( $tooltip === '' ) {
					$tooltip = sprintf(
						$tooltips[ $type ]['size_only'],
						$atts['max']
					);
				} else {
					$tooltip .= sprintf(
						$tooltips[ $type ]['size'],
						$atts['max']
					);
				}
			}
		}

		return $tooltip;
	}
}
