<?php
/*
	Comments: Plugin File for Form Generator Settings
*/
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Settings( $config );
class Settings {

	private $config;

	public function __construct( $config ) {
		// set config
		$this->config = (object) $config;

		if ( is_admin() ) {
			// Add Submenu Item
			add_action( 'admin_menu', array( $this, 'add_settings_page' ) );

			// Init the Settings
			add_action( 'admin_init', array( $this, 'init_settings_page' ) );
		}
	}

	// Vars
	public $pre = 'greyd_settings_forms';


	// Add Submenu
	public function add_settings_page() {

		add_submenu_page(
			'edit.php?post_type=' . $this->config->plugin_post_type, // Parent Slug
			__( 'Greyd.Forms', 'greyd_forms' ) . ' › ' . __( "Settings", 'greyd_forms' ), // Page Title
			__( "Settings", 'greyd_forms' ), // Menu Title
			'manage_options', // capability
			$this->pre, // Menu slug
			array( $this, 'render_settings_page' ) // callback function
		);

	}



	public function init_settings_page() {

		// DATENSCHUTZ STANDARD

		// Add Section
		add_settings_section(
			$this->pre . '_agb', // id of the section
			__( "Privacy policy", 'greyd_forms' ), // title to be displayed
			array( $this, 'agb_description' ), // add section description function here
			$this->pre // page on which to display the section
		);

		// if default-agb is rendered
		$render_agb = true;
		$render_agb = apply_filters( 'render_agb_default_text', $render_agb );

		if ( $render_agb ) {
			// Standardtext
			register_setting( $this->pre, 'agb_text' );

			add_settings_field(
				'agb_text', // id of the settings field
				__( "Default text:", 'greyd_forms' ), // title
				array( $this, 'render_agb_text' ), // callback function
				$this->pre, // page on which settings display
				$this->pre . '_agb' // section on which to show settings
			);
		} else {
			// add different agb setting
			do_action( 'formsettings_agb_default_text', $this->pre );
		}

		// MAX LENGTH

		// Add Section
		add_settings_section(
			$this->pre . '_maxlength', // id of the section
			__( "Input lengths", 'greyd_forms' ), // title to be displayed
			array( $this, 'maxlength_description' ), // add section description function here
			$this->pre // page on which to display the section
		);

		// Maxlength
		register_setting( $this->pre, 'maxlength_inputs', array( $this, 'validate_maxlength_inputs' ) );

		add_settings_field(
			'maxlength_inputs', // id of the settings field
			__( "Maximum character length:", 'greyd_forms' ), // title
			array( $this, 'render_maxlength_inputs' ), // callback function
			$this->pre, // page on which settings display
			$this->pre . '_maxlength' // section on which to show settings
		);

		// GOOGLE RECAPTCHA

		// Add Section
		add_settings_section(
			$this->pre . '_captcha', // id of the section
			__( 'Google reCAPTCHA', 'greyd_forms' ) . " <span style='font-weight:300;'>V3</span>", // title to be displayed
			array( $this, 'captcha_description' ), // add section description function here
			$this->pre // page on which to display the section
		);

		// Option 2
		register_setting( $this->pre, 'captcha_sitekey' );

		add_settings_field(
			'captcha_sitekey', // id of the settings field
			__( "CAPTCHA site key:", 'greyd_forms' ), // title
			array( $this, 'render_captcha_sitekey' ), // callback function
			$this->pre, // page on which settings display
			$this->pre . '_captcha' // section on which to show settings
		);

		// Option 3
		register_setting( $this->pre, 'captcha_secret' );

		add_settings_field(
			'captcha_secret', // id of the settings field
			__( "Secret key:", 'greyd_forms' ), // title
			array( $this, 'render_captcha_secret' ), // callback function
			$this->pre, // page on which settings display
			$this->pre . '_captcha' // section on which to show settings
		);

		// Option 4
		register_setting( $this->pre, 'captcha_thres' );

		add_settings_field(
			'captcha_thres', // id of the settings field
			__( "Minimum value:", 'greyd_forms' ), // title
			array( $this, 'render_captcha_thres' ), // callback function
			$this->pre, // page on which settings display
			$this->pre . '_captcha' // section on which to show settings
		);

		do_action( 'formsettings_interfaces', $this->pre );

	}


	// RENDER FIELDS


	// AGB Description
	public function agb_description() {
		echo '<p>' . __( "Enter a standard content for the privacy checkboxes here.<br>This content is used when you choose \"use default privacy text\" in the checkbox widget.", 'greyd_forms' ) . '</p>';
	}

	// AGB Text
	public function render_agb_text() {
		$option = esc_attr( get_option( 'agb_text', __( "I hereby accept the terms and conditions.", 'greyd_forms' ) ) );
		// echo "<input type='text' id='agb_text' class='large-text' style='max-width:35em;' name='agb_text' value='$option'>";
		wp_editor(
			htmlspecialchars_decode( $option ),
			'agb_text',
			$settings = array(
				'textarea_name' => 'agb_text',
				'editor_height' => 140,
				'media_buttons' => false,
			)
		);
	}


	// Maxlength Description
	public function maxlength_description() {
		echo '<p>' . __( "This values are used for the validation of all form fields in front-end and backend.", 'greyd_forms' ) . '</p>';
	}

	// Maxlength Inputs
	public function render_maxlength_inputs() {
		$option = esc_attr( get_option( 'maxlength_inputs', 1000 ) );
		echo "<input type='text' id='maxlength_inputs' class='' name='maxlength_inputs' value='$option'>";
		echo "<div style='height:100px;'></div>";
	}


	// Captcha Description
	public function captcha_description() {
		echo '<p>' . __( "Enter your reCAPTCHA information to protect your forms against against bots.", 'greyd_forms' ) .
			 '<br>' . __( "Generate your site key at: ", 'greyd_forms' ) .
			/*Wird der geheime Schlüssel auch dort generiert? Dann sollte es oben "deine Schlüssel" heißen. Ansonsten den Hinweis zu Website Schlüssel runter nehmen und bei Geheimer Schlüssel einen zweiten ergänzen*/
			 "<a href='https://www.google.com/recaptcha' target='_blank'>google.com/recaptcha</a></p>";
	}

	// Captcha Radio Buttons
	public function render_captcha_type() {
		$options = get_option( 'captcha_type' );

		echo '<input type="radio" id="v3" name="captcha_type[version]" value="v3"' . checked( 'v3', $options['version'], false ) . '/>';
		echo '<label for="v3">' . __( "reCAPTCHA V3 (running in the background)", 'greyd_forms' ) . '</label>';
		echo '<br><br>';
		echo '<input type="radio" id="v2" name="captcha_type[version]" value="v2"' . checked( 'v2', $options['version'], false ) . '/>';
		echo '<label for="v2">' . __( "reCAPTCHA V2 (display as CAPTCHA checkbox)", 'greyd_forms' ) . '</label>';

	}

	// Captcha Schlüssel
	public function render_captcha_sitekey() {
		$option = esc_attr( get_option( 'captcha_sitekey', '' ) );
		echo "<input type='text' id='captcha_sitekey' class='large-text' style='max-width:35em;' name='captcha_sitekey' value='$option'>";
	}

	// Captcha Secret
	public function render_captcha_secret() {
		$option = esc_attr( get_option( 'captcha_secret', '' ) );
		echo "<input type='text' id='captcha_secret' class='large-text' style='max-width:35em;' name='captcha_secret' value='$option'>";
	}

	// Captcha Mindestwert
	public function render_captcha_thres() {
		$option = esc_attr( get_option( 'captcha_thres', '' ) );
		echo "<input type='number' id='captcha_thres' name='captcha_thres' value='$option' placeholder='0,5' step='0.05' min='0' max='1'>";
		echo "<small style='display:block;max-width:35em;margin-left:2px;'>" . __( "Google displays a value between 0 (probably a bot) and 1 (probably a human being). By default, users with a value higher than 0.5 are rated as 'save'.", 'greyd_forms' ) . '</small>';
		// evtl. Hinweis was mit als nicht sicher eingestuften Einträgen passiert? Können die das Formular nicht absenden?
	}



	// VALIDATE FIELDS


	// Sanitize input maxlength
	public function validate_maxlength_inputs( $value ) {

		// debug
		// add_settings_error('maxlength_inputs', 'maxlength_inputs', json_encode($value), 'updated');

		$errors = array(
			'empty' => __( "The maximum charachter length cannot not be empty.", 'greyd_forms' ),
			'nan'   => __( "The maximum character length must be a numeric value bigger than 0.", 'greyd_forms' ),
			'max'   => __( "The maximum character length must not be bigger than 100,000.", 'greyd_forms' ),
		);

		// Detect multiple sanitizing passes.
		// Workaround for: https://core.trac.wordpress.org/ticket/21989
		static $cache = null;

		if ( $cache !== null ) {
			return $cache;
		}

		if ( (int) $value === 0 ) {
			add_settings_error( 'maxlength_inputs', 'maxlength_inputs', $errors['nan'], 'error' );
			return 1000;
		}

		if ( (int) $value > 100000 ) {
			add_settings_error( 'maxlength_inputs', 'maxlength_inputs', $errors['max'], 'error' );
			return 100000;
		}

		// Validate the field
		$value = sanitize_text_field( $value );

		// Save result to local cache.
		$cache = $value;
		return $cache;
	}






	public function render_settings_page() {

		// check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		echo "<div class='wrap settings_wrap'>";
		echo "<h1 style='margin-bottom:2em;'>" . get_admin_page_title() . '</h1>';

		settings_errors();

		// echo "<div class='flex-wrap'>";

		// Formular
		echo "<form method='post' action='options.php'>";

		// Render Settings
		settings_fields( $this->pre );
		do_settings_sections( $this->pre );
		submit_button();

		echo '</form>';

		// echo "</div>"; // flex-wrap
		echo '</div>'; // wrap
	}



}
