<?php

/**
 * Greyd.Forms Interface Main Class.
 *
 * This file sets up all the default & custom interface integrations.
 * You can setup your own interface, by following these steps:
 *
 * 1. Use the filter 'greyd_forms_custom_interfaces' to add your interface
 * 2. Add the following directory to your active theme: your_theme/greyd_forms/interfaces/{{slug}}
 * 3. Define the setup of your interface via the config.php inside the directory.
 * 4. Build the whole custom functionality (http handlers, render functions...) via the init.php inside the directory.
 *
 * For code references and examples, have a look at other integrations inside this plugin.
 */

namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Handler;
use \Greyd\Forms\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Greyd_Forms_Interface();

class Greyd_Forms_Interface {

	/**
	 * Class constants
	 */
	const SETTING     = 'greyd_forms_interface_settings';
	const OPTION      = 'interface_settings';
	const ENTRY       = 'greyd_forms_interface_state';
	const AJAX_ACTION = 'interface_ajax';
	const INTERFACES  = array(
		'webhook',
		'mailchimp',
		'mailjet',
		'zapier',
		'hubspot',
		'salesforce',
		'samdock',
		'rapidmail',
		'zoom_webinar',
	);

	/**
	 * Class variables
	 */
	public static $pre        = '';
	public static $interfaces = array();
	public static $settings   = array();
	public static $cache      = null; // used for settings validation
	public static $loaded     = false;

	public function __construct() {

		// init
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_init' ) );
		add_action( 'after_setup_theme', array( $this, 'admin_init' ) );
		add_action( 'greyd_forms_load_interfaces', array( $this, 'load' ) );

		// settings
		add_action( 'formsettings_interfaces', array( $this, 'register_settings' ) );

		// metabox
		add_action( 'render_metabox_interface', array( $this, 'render_metabox' ), 10, 2 );
		add_action( 'formmeta_save', array( $this, 'save_metabox' ) );

		// handler
		add_action( 'greyd_forms_action_after_optin', array( $this, 'handle_after_doi' ), 10, 4 );
		add_action( 'formhandler_after_optout', array( $this, 'handle_optout' ), 10, 1 );

		// admin ajax
		add_action( 'wp_ajax_nopriv_' . self::AJAX_ACTION, array( $this, 'admin_ajax' ) );
		add_action( 'wp_ajax_' . self::AJAX_ACTION, array( $this, 'admin_ajax' ) );

		// display errors
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'formshortcode_errors_before_form', array( $this, 'form_errors' ), 10, 1 );
		add_action( 'greyd_forms_render_optout_warning', array( $this, 'optout_warning' ) );
	}

	/**
	 * Frontend init
	 *
	 * called via @filter 'wp_enqueue_scripts'
	 */
	public function frontend_init() {
		if ( is_admin() ) {
			return;
		}
		// if ( is_admin() || !check_enqueue(['vc_form']) ) return;

		// init settings
		$this->load_interface_init();
	}

	/**
	 * Load interface init.php files
	 *
	 * called via frontend_init() or handle_after_doi()
	 */
	public function load_interface_init() {
		if ( self::$loaded ) {
			return true;
		}

		// init settings
		self::$settings = self::get_option();
		foreach ( self::$settings as $interface => $args ) {

			$exists  = isset( array_flip( self::INTERFACES )[ $interface ] ) ? true : false;
			$enabled = is_array( $args ) && isset( $args['enable'] ) && $args['enable'] === 'on' ? true : false;

			if ( $exists && $enabled ) {
				include_once __DIR__ . '/' . $interface . '/init.php';
			}
		}
		self::$loaded = true;
	}

	/**
	 * Backend init
	 *
	 * called via @filter 'after_setup_theme'
	 */
	public function admin_init() {
		if ( ! is_admin() ) {
			return;
		}

		// init settings
		self::$settings = self::get_option();

		// init interfaces
		self::$interfaces = self::get_interface_config();
		foreach ( self::$interfaces as $interface => $args ) {
			// init class
			include_once __DIR__ . '/' . $interface . '/init.php';
		}
		self::$loaded = true;
	}

	/**
	 * Load interface config.php files
	 *
	 * we need this to build the admin pages
	 */
	public static function get_interface_config() {
		$return = array();
		foreach ( self::INTERFACES as $interface ) {
			$return[ $interface ] = include __DIR__ . '/' . $interface . '/config.php';
		}

		/**
		 * Add custom interface
		 *
		 * use the filter 'greyd_forms_custom_interfaces' to add a custom interface
		 * from inside a child theme.
		 */
		$custom_interfaces = (array) apply_filters( 'greyd_forms_custom_interfaces', array() );
		foreach ( $custom_interfaces as $interface ) {
			$file_path = get_template_directory() . '/greyd_forms/interfaces/' . $interface . '/config.php';
			if ( file_exists( $file_path ) ) {
				$return[ $interface ] = include $file_path;
			}
		}

		/**
		 *  Add/remove interfaces
		 *
		 *  use the filter 'greyd_forms_interfaces' to modify interfaces
		 */
		return apply_filters( 'greyd_forms_interfaces', $return );
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */

	/**
	 * Register the options.
	 */
	public function register_settings( $pre = 'greyd_settings_forms' ) {

		self::$pre = $pre;

		$section = $pre . '_interfaces';

		// add section
		add_settings_section(
			$section, // id of the section
			'', // title to be displayed
			function(){}, // add section description function here
			$pre // page on which to display the section
		);

		// register setting
		register_setting( $pre, self::SETTING, array( $this, 'validate_interface_settings' ) );

		// add setting
		add_settings_field(
			self::SETTING, // id of the settings field
			'', // title
			array( $this, 'render_settings' ), // callback function
			$pre, // page on which settings display
			$section // section on which to show settings
		);

	}

	/**
	 * Render the options.
	 */
	public function render_settings() {

		if ( empty( self::get_interface_config() ) ) {
			return '';
		}

		echo '</td></table>';

		echo '<hr class="greyd_hr">';
		echo '<h1 style="margin-bottom:2em;">' . __( "Interfaces", 'greyd_forms' ) . '</h1>';

		$values = self::$settings;
		// debug( $values , true );

		foreach ( self::$interfaces as $interface => $args ) {

			// vars
			$settings = isset( $args['settings'] ) ? $args['settings'] : false;
			if ( empty( $settings ) ) {
				continue;
			}

			$section   = self::$pre . '_' . $interface;
			$value_pre = self::SETTING . '[' . $interface . ']';
			$title     = isset( $settings['title'] ) ? $settings['title'] : $args['name'];
			$descr     = isset( $settings['description'] ) ? $settings['description'] : '';
			$options   = isset( $settings['options'] ) ? (array) $settings['options'] : array();
			$enabled   = isset( $values[ $interface ] ) && isset( $values[ $interface ]['enable'] ) ? $values[ $interface ]['enable'] : 'off';

			// render
			echo "<table class='form-table'><tr><th style='padding-top:15px;'>";
			if ( ! empty( $title ) ) {
				echo "<h2 style='margin:0;'>$title</h2>";
			}
			// if (!empty($descr)) echo "<p>$descr</p>";

			echo '</th><td>';

			$checked     = $enabled === 'on' ? "checked='checked'" : '';
			$option_name = $value_pre . '[enable]';
			echo "<label for='$option_name'>" .
				"<input type='checkbox' id='$option_name' name='$option_name' $checked onchange='greyd.backend.toggleElemByClass(\"" . $interface . "\")'/>" .
				'<span>' . __( "enable", 'greyd_forms' ) . '</span>' .
				'</label>';

			echo '</td></tr>';

			// render options
			foreach ( $options as $option => $title ) {

				$hidden = $enabled === 'on' ? '' : 'hidden';
				echo "<tr class='toggle_$interface $hidden'>";

				$slug  = $interface . '_' . $option;
				$func  = 'render_setting_' . $slug;
				$value = isset( $values[ $interface ][ $option ] ) ? $values[ $interface ][ $option ] : null;
				// debug($value, true);

				echo "<th>$title</th><td>";
				do_action( $func, $value_pre, $value );
				echo '</td></tr>';

			}
			echo '</table>';
		}
		echo '<table><td>';
	}

	/**
	 * Validate the options.
	 */
	public function validate_interface_settings( $value ) {

		// Detect multiple sanitizing passes.
		// Workaround for: https://core.trac.wordpress.org/ticket/21989
		if ( self::$cache !== null ) {
			// return settings saved in class var
			return self::$settings;
		}
		self::$cache = true;

		// if empty
		if ( empty( $value ) ) {
			return array();
		}

		// save in class var
		self::$settings = $value;

		return $value;
	}

	/**
	 * =================================================================
	 *                          Meta Box
	 * =================================================================
	 */

	/**
	 * Render metabox per interface.
	 */
	public function render_metabox( $post, $form_fields ) {

		/**
		 * Extend form fields with generated metadata
		 *
		 * @since 1.0.8
		 */
		if ( ! is_array( $form_fields ) || count( $form_fields ) < 2 ) {
			$all_fields = array();
		} else {
			$all_fields = array(
				__( "Form Fields", 'greyd_forms' ) => (array) $form_fields,
				__( "Metadata", 'greyd_forms' )       => array(
					'ip_address' => __( "IP address of the user", 'greyd_forms' ),
					'created'    => __( "Time of submission", 'greyd_forms' ),
				),
			);
		}

		$render   = '';
		$settings = (array) self::$settings;
		if ( count( $settings ) > 0 ) {
			foreach ( $settings as $interface => $options ) {
				$return = $this->get_metabox( $interface, $settings[ $interface ], $post, $all_fields );
				if ( $return !== false ) {
					if ( $render !== '' ) {
						$render .= '<hr style="margin:3em 0;">';
					}
					$render .= $return;
				}
			}
		}

		if ( empty( $render ) ) {
			self::interface_empty_text();
		} else {
			echo $render;
		}
	}

	/**
	 * Display meta box per interface.
	 */
	public function get_metabox( $interface, $settings, $post, $all_fields ) {

		$enable = isset( $settings['enable'] ) ? $settings['enable'] : 'off';

		// early exit if it this interface isn't enabled
		if ( $enable !== 'on' ) {
			return false;
		}

		// if one of the options is not set
		$config     = (array) self::get_config( $interface );
		$options    = isset( $config['settings'] ) && isset( $config['settings']['options'] ) ? $config['settings']['options'] : array();
		$incomplete = ! self::is_setting_complete( $options, $settings, $interface );

		// debug( $config );
		$name         = isset( $config['name'] ) ? $config['name'] : $interface;
		$option_name  = self::OPTION . '[' . $interface . ']';
		$meta_default = array(
			'enable'   => array(
				'title'       => sprintf( __( "Enable %s interface?", 'greyd_forms' ), $name ),
				'description' => sprintf( __( "Do you want to send completed forms to %s?", 'greyd_forms' ), $name ),
			),
			'pre_meta' => array(
				'title' => __( "Metadata", 'greyd_forms' ),
			),
			'normal'   => array(
				'title'       => __( "Assign fields", 'greyd_forms' ),
				'description' => sprintf( __( "Select which form field to fill the %s entry.", 'greyd_forms' ), $name ),
				'form_fields' => __( "Form field (name)", 'greyd_forms' ),
				'api_fields'  => sprintf( __( "%s entry", 'greyd_forms' ), $name ),
			),
			'custom'   => array(
				'title'       => __( "Individual fields", 'greyd_forms' ),
				'description' => sprintf( __( "Select which individual fields to pass to %s.", 'greyd_forms' ), $name ),
				'form_fields' => __( "Form field (name)", 'greyd_forms' ),
				'api_fields'  => sprintf( __( "%s entry", 'greyd_forms' ), $name ),
			),
			'meta'     => array(
				'title' => __( "Metadata", 'greyd_forms' ),
			),
		);
		$meta_config  = isset( $config['metabox'] ) ? $config['metabox'] : array();
		$meta_config  = array_replace_recursive( $meta_default, $meta_config );
		$meta_values  = self::get_meta( $post->ID, $interface );
		// debug($meta_values, true);
		/*
		 *  $meta_values = [
		 *      'enable' => ( on | off ),
		 *      'normal' => [
		 *          {{api_field}} => {{frm_field}}
		 *          ..
		 *      ],
		 *      'custom' => [
		 *          {{api_field}} => {{frm_field}}
		 *          ...
		 *      ],
		 *      'meta' => [
		 *          {{api_field}} => {{value}},
		 *          ...
		 *      ]
		 *  ];
		 */

		// check new options
		// if ($interface === 'salesforce') debug( get_post_meta($post->ID, self::OPTION, true), true );
		// debug( get_option(self::SETTING, null), true );

		// get all selected fields
		$chosen_api = $chosen_frm = '';
		if ( isset( $meta_values['normal'] ) && count( $meta_values['normal'] ) > 0 ) {
			$chosen_api .= implode( '', array_keys( $meta_values['normal'] ) );
			$chosen_frm .= implode( '', array_values( $meta_values['normal'] ) );
		}
		if ( isset( $meta_values['custom'] ) && count( $meta_values['custom'] ) > 0 ) {
			$chosen_frm .= implode( '', array_values( $meta_values['custom'] ) );
		}

		// start rendering
		ob_start();
		echo "<table class='form_meta_box'>";

		// HEAD
		echo "<thead><tr><th colspan=2><div>$name</div></th></tr>";

		if ( $incomplete ) {
			echo '</table>';
			echo Helper::render_info_box(
				array(
					'style' => 'info',
					'text'  => self::interface_incomplete_info( $interface, false ),
				)
			);
			$return = ob_get_contents();
			ob_end_clean();
			return $return;
		} else {
			echo '</th></tr>';
		}

		$checked = isset( $meta_values['enable'] ) && $meta_values['enable'] === 'on' ? "checked='checked'" : '';
		echo '<tr>';
			echo '<td>' . strval( $meta_config['enable']['title'] ) .
				 '<small>' . strval( $meta_config['enable']['description'] ) . '</small>' .
				 '</td>';
			echo '<td>';
				echo "<input type='checkbox' name='" . $option_name . "[enable]' value='" . $option_name . "[enable]' $checked onchange='greyd.backend.toggleElemByClass(\"" . $interface . "\")' data-text='" . __( "Yes", 'greyd_forms' ) . "'>";
			echo '</td>';
		echo '</tr></thead>';

		// BODY
		unset( $meta_config['enable'] );
		$hidden = ! empty( $checked ) ? '' : 'hidden';
		echo "<tbody class='toggle_" . $interface . ' ' . $hidden . "'>";

		foreach ( $meta_config as $type => $setup ) {

			// render Help
			if ( $type === 'help' ) {
				echo "<tr><td colspan='2' style='padding-top:1em;'>" .
					( isset( $setup['title'] ) ? $setup['title'] : '' ) .
					( isset( $setup['content'] ) ? $setup['content'] : '' ) .
				'</td></tr>';
				continue;
			}

			// don't render if no fields set
			if ( ! isset( $setup['fields'] ) || $setup['fields'] === false || empty( $setup['fields'] ) ) {
				continue;
			}

			$title       = isset( $setup['title'] ) ? $setup['title'] : '';
			$description = isset( $setup['description'] ) ? $setup['description'] : '';
			echo "<tr><td colspan='2' style='padding-top:1em;'>";
					echo $title;
			if ( ! empty( $description ) ) {
				echo "<small class='max-600'>" . $description . '</small>';
			}
			echo '</td></tr>';

			// Inner Tables
			echo "<tr><td colspan='2' class='inner-table'><table>";

			// Meta Fields
			if ( $type === 'meta' || $type === 'pre_meta' ) {
				$type = 'meta';
				echo '<tbody>';
				foreach ( (array) $setup['fields'] as $field => $atts ) {
					$label = isset( $atts['label'] ) ? $atts['label'] : '';
					$ph    = isset( $atts['value'] ) ? $atts['value'] : '';
					$descr = isset( $atts['description'] ) ? $atts['description'] : '';
					$value = isset( $meta_values[ $type ][ $field ] ) ? $meta_values[ $type ][ $field ] : '';
					$input = isset( $atts['type'] ) ? $atts['type'] : 'text';
					$pre   = $after = '';
					if ( $input === 'textarea' ) {
						$pre   = '<textarea ';
						$after = " placeholder='$ph'>$value</textarea>";
					} elseif ( $input === 'select' ) {
						// debug(json_decode($ph), true);
						$pre   = '<select ';
						$after = '>';
						$ph    = is_string( $ph ) && strpos( $ph, '{' ) !== false ? json_decode( $ph, true ) : $ph;
						if ( isset( $atts['empty_option'] ) && $atts['empty_option'] ) {
							$after .= "<option value=''>" . __( "please select", 'greyd_forms' ) . '</option>';
						}
						foreach ( (array) $ph as $key => $val ) {
							$s      = strval( $key ) === $value ? "selected='selected'" : '';
							$after .= "<option value='$key' $s>$val</option>";
						}
						$after .= '</select>';
					} elseif ( $input === 'checkbox' ) {
						$s     = $value === 'on' ? "checked='checked'" : '';
						$pre   = "<input type='checkbox' $s ";
						$after = "data-text='" . __( "Yes", 'greyd_forms' ) . "' >";
					} else {
						$pre   = "<input type='text' ";
						$after = "placeholder='$ph' value='$value' >";
					}
					echo '<tr><td>';
					if ( ! empty( $label ) ) {
						echo "<label>$label</label><br>";
					}
					echo $pre . "name='" . $option_name . '[' . $type . '][' . $field . "]'" . $after;
					if ( ! empty( $descr ) ) {
						echo "<br><i>$descr</i><br>";
					}
					echo '</td></tr>';
				}
				echo '</tbody>';

				// Normal & Custom Fields
			} else {
				// thead
				echo '<thead><tr>' .
					"<th style='width:20%;'>" . preg_replace( '/\s/', '&nbsp;', $meta_config[ $type ]['form_fields'] ) . '</th>' .
					"<td style='padding:0 .5em;text-align:center;vertical-align:baseline;width:40%;'>→</td>" .
					'<th>' . preg_replace( '/\s/', '&nbsp;', $meta_config[ $type ]['api_fields'] ) . '</th>' .
				'</tr></thead>';

				// tbody
				echo '<tbody>';

				// if no fields with names are defined
				if ( count( $all_fields ) < 2 ) {
					echo "<tr><td colspan='3'><div class='greyd_info_box info'>" .
						'<span class="dashicons dashicons-info"></span><span>' .
						__( "To display fields, please save form.", 'greyd_forms' ) . '</span></div></td></tr>';
				}
				// if form fields are set
				else {

					$values = isset( $meta_values[ $type ] ) ? (array) $meta_values[ $type ] : array();
					$ph     = $type === 'normal' ? array( 'none' => 'none' ) : array( '' => 'none' );
					$values = $ph + $values;

					$i = 0;
					foreach ( $values as $api_field => $form_field ) {

						echo "<tr class='" . ( $i === 0 ? 'hide' : '' ) . "' data-index='$i'><td>";

						echo "<select name='" . $option_name . '[' . $type . '][frm][' . $i . "]' class='api_field frm'>";
						foreach ( $all_fields as $title => $optgroup ) {
							echo "<optgroup label='$title'>";
							foreach ( $optgroup as $val => $option ) {
								$strval = $val && ! empty( $val ) ? esc_attr( $val ) : '';
								if ( $strval === $form_field ) {
									$attr = "selected='selected'";
								} elseif ( empty( $strval ) || strpos( $chosen_frm, $strval ) === false ) {
									$attr = '';
								} else {
									$attr = 'disabled';
								}
								echo "<option value='$strval' $attr >$option</option>";
							}
							echo '</optgroup>';
						}
						echo '</select>';

						echo '</td>' .
							"<td style='padding:0 .5em;text-align:center;vertical-align:baseline;'>→</td>" .
							"<td class='flex-row'>";

						// custom field
						if ( $type === 'custom' ) {
							echo "<input type='text' name='" . $option_name . '[' . $type . '][api][' . $i . "]' value='$api_field'>";
						}

						// normal field
						else {
							$api_options = array( 'none' => __( "please select", 'greyd_forms' ) ) + $setup['fields'];

							echo "<select name='" . $option_name . '[' . $type . '][api][' . $i . "]' class='api_field api' data-init-value='" . $api_field . "'>";
							foreach ( $api_options as $val => $option ) {
								$s = '';

								// check if api-field contains a fieldtype-attribute and remove it
								if ( strpos( $api_field, '::' ) !== false ) {
									$api_field = explode( '::', $api_field )[0];
								}

								$value = empty( $val ) ? '' : strval( $val );
								if ( $api_field === $value ) {
									$attribute = 'selected="selected"';
								} elseif ( strpos( $chosen_api, $value ) !== false ) {
									$attribute = 'disabled';
								} else {
									$attribute = '';
								}
								echo "<option value='$value' $attribute>$option</option>";
							}
							echo '</select>';
						}
						echo "<span class='button remove'>✕</span></td></tr>";

						$i++;
					}

					// add-button
					echo "<tr><td colspan='3'><button class='button greyd add'>+  " . _x( "add field", 'small', 'greyd_forms' ) . '</button></td></tr>';
				}
				echo '</tbody>';
			}

			echo '</table></td></tr>';

		}
		echo '</tbody>';

		echo '</table>';
		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}

	/**
	 * Callback on post save
	 */
	public function save_metabox( $post_args ) {

		$data    = isset( $post_args['data'] ) ? $post_args['data'] : array();
		$post_id = isset( $post_args['post_id'] ) ? $post_args['post_id'] : array();

		$raw_settings       = isset( $data[ self::OPTION ] ) ? $data[ self::OPTION ] : array();
		$interface_settings = array();

		foreach ( $raw_settings as $interface => $args ) {
			// set enable
			$enable                                     = isset( $args['enable'] ) ? 'on' : 'off';
			$interface_settings[ $interface ]['enable'] = $enable;

			// set fields
			$interface_settings[ $interface ]['normal'] = self::get_field_values( $args, 'normal' );
			$interface_settings[ $interface ]['custom'] = self::get_field_values( $args, 'custom' );

			// set meta
			$interface_settings[ $interface ]['meta'] = $raw_settings[ $interface ]['meta'];
		}

		update_post_meta( $post_id, self::OPTION, $interface_settings );

		do_action( 'tp_forms_interface_meta_save', $post_id );
	}

	/**
	 * =================================================================
	 *                          Handler
	 * =================================================================
	 */

	/**
	 * Trigger all the API calls here.
	 *
	 * @action greyd_forms_action_after_optin
	 *
	 * @param array $form_data    Validated form data.
	 * @param int   $form_id      WP_Post ID of the form.
	 * @param int   $entry_id     WP_Post ID of the entry.
	 * @param array $entry_data   Entire data of the entry.
	 */
	public function handle_after_doi( $form_data = array(), $form_id = 0, $entry_id = 0, $entry_data = array() ) {

		$this->load_interface_init();
		// do_action('formhandler_error', $form_data); // debug

		if ( empty( $form_id ) ) {
			return false;
		}

		$postmeta = self::get_meta( $form_id, null );
		if ( is_array( $postmeta ) && count( $postmeta ) > 0 ) {

			$args = array(
				'entry_id' => $entry_id,
				'data'     => $form_data,
			);

			/**
			 * Modify form data with meta infos
			 *
			 * @since 1.0.8
			 */
			if ( ! isset( $args['data']['ip_address'] ) ) {
				$args['data']['ip_address'] = isset( $entry_data['host'] ) ? $entry_data['host'] : '';
			}
			if ( ! isset( $args['data']['created'] ) ) {
				$args['data']['created'] = isset( $entry_data['created'] ) ? $entry_data['created'] : '';
			}

			foreach ( $postmeta as $interface => $atts ) {
				$enable = isset( $atts['enable'] ) ? $atts['enable'] : 'off';
				if ( $enable === 'on' ) {
					$args['postmeta'] = $atts;
					$this->send_data( $interface, $args );
				}
			}
		}
	}

	/**
	 * Send the data to the interface endpoint.
	 */
	public function send_data( $interface, $args ) {
		// do_action('formhandler_error', $args); // debug

		// vars
		$response = false;
		$entry_id = isset( $args['entry_id'] ) ? $args['entry_id'] : null;
		$postmeta = isset( $args['postmeta'] ) ? $args['postmeta'] : null;
		$formdata = isset( $args['data'] ) ? $args['data'] : null;
		if ( empty( $entry_id ) || empty( $postmeta ) || empty( $formdata ) ) {
			return false;
		}

		/**
		 * Filter hook for adding custom action after the doi is triggered
		 *
		 * @filter handle_after_doi_{{interface}}
		 *
		 * @param mixed  $response      The response to return.
		 * @param string $entry_id      The Post ID of the entry.
		 * @param array  $form_data     The user data, keyed by the field name.
		 * @param array  $postmeta      The form post_meta for this specific interface.
		 *
		 * @return mixed    If true is is returned, a default success message is logged to the entry.
		 */
		$response = apply_filters( 'handle_after_doi_' . $interface, $response, $entry_id, $formdata, $postmeta );
		if ( $response === true ) {
			Helper::log_entry_state( $entry_id, sprintf( __( "Successfully sent to %s", 'greyd_forms' ), self::get_config( $interface, 'name' ) ), 'success' );
		} else {
			// do_action('formhandler_error', $response); // debug

			/**
			 * Send the admin an error email.
			 *
			 * @since 1.4.1
			 */
			ob_start();
			$state = Helper::get_entry_state( $entry_id );
			echo sprintf(
				"<p>A submitted Greyd.Form could not be send to the interface <strong>'%s'</strong>.</p><p>Last error state: <b>%s</b>.</p><p>Find request details below:</p>",
				$interface,
				isset( $state['current']['text'] ) ? $state['current']['text'] : 'log in to see details'
			);
			debug( $_REQUEST );
			debug( $formdata );
			$message = ob_get_contents();
			ob_end_clean();

			Handler::send_admin_error_mail( 0, $message, $entry_id );
		}
	}

	/**
	 * Trigger API calls after opt-out.
	 */
	public function handle_optout( $entry_id ) {
		if ( empty( $entry_id ) ) {
			return false; // early exit
		}

		$interfaces = self::get_entry_interface_data( $entry_id );
		if ( is_array( $interfaces ) && count( $interfaces ) > 0 ) {
			foreach ( $interfaces as $interface => $meta ) {
				$optout = self::get_config( $interface, 'optout' );
				if ( $optout === true ) {
					include_once __DIR__ . '/' . $interface . '/init.php';

					/**
					 * Action hook for adding custom action after user opted out
					 *
					 * @action formhandler_optout_{{interface}}
					 *
					 * @param string $entry_id  The Post ID of the entry.
					 * @param array $meta       The entry post_meta of this specific interface.
					 */
					do_action( 'formhandler_optout_' . $interface, $entry_id, $meta );
				}
			}
		}
	}

	/**
	 * Handle ajax.
	 */
	public function admin_ajax() {

		if ( ! check_ajax_referer( self::AJAX_ACTION ) ) {
			wp_die( 'error::wrong nonce' );
		}

		$mode     = $_POST['mode'] ? $_POST['mode'] : '';
		$data     = $_POST['data'] ? $_POST['data'] : '';
		$response = ! empty( $data ) ? json_encode( $data ) : '';
		// wp_die('error::'.$response);

		if ( ! empty( $mode ) && ! empty( $data ) && is_array( $data ) ) {
			do_action( 'tp_forms_interface_ajax_' . $mode, $data );
		}
		wp_die();
	}

	/**
	 * =================================================================
	 *                          Errors
	 * =================================================================
	 */

	/**
	 * Check for errors
	 */
	public function admin_notices() {
		$screen = get_current_screen();

		// only in forms posttype
		if ( 'tp_forms' === $screen->post_type ) {
			$this->settings_errors();

			// if in edit screen
			if ( $screen->parent_base === 'edit' ) {
				global $post;
				$id = isset( $post->ID ) ? $post->ID : null;
				if ( isset( $id ) ) {
					$this->form_errors( $id, false );
				}
			}
		}
	}

	/**
	 * Display admin notice if not all settings are set
	 */
	public function settings_errors() {
		$settings = self::get_option();
		foreach ( $settings as $interface => $args ) {
			$enable = is_array( $args ) && isset( $args['enable'] ) && $args['enable'] === 'on' ? true : false;
			if ( $enable ) {
				$config   = self::get_config( $interface, 'settings', 'options' );
				$complete = self::is_setting_complete( $config, $args, $interface );
				if ( ! $complete ) {
					$text = self::interface_incomplete_text( $interface );
					Helper::render_admin_notice( $text, 'error' );
				}
			}
		}
	}

	/**
	 * Display form errors in front- & backend
	 */
	public function form_errors( $form_id, $frontend = true ) {

		$meta = self::get_meta( $form_id );
		// debug($meta);
		if ( ! is_array( $meta ) || empty( $meta ) ) {
			return false;
		}

		$edit_link = admin_url( 'post.php?post=' . $form_id . '&action=edit' );

		foreach ( $meta as $interface => $args ) {
			$enable = is_array( $args ) && isset( $args['enable'] ) && $args['enable'] === 'on' ? true : false;
			if ( ! $enable ) {
				continue;
			}

			$name = self::get_config( $interface, 'name' );

			// check if fields are set
			if ( ! isset( $args['normal'] ) || empty( $args['normal'] ) ) {

				// if normal fields or custom fields need to be set
				if ( self::get_config( $interface, 'metabox', 'normal' ) !== false || ( ! isset( $args['custom'] ) || empty( $args['custom'] ) ) ) {
					$text = array(
						__( "The form cannot be submitted to %s because no fields have been assigned.", 'greyd_forms' ),
						__( "%1\$sEdit the form%2\$s and set the assignments in the \"Settings\" area at the bottom.", 'greyd_forms' ),
						__( "Set the assignments at the bottom under „Settings“.", 'greyd_forms' ),
					);
					$text = sprintf( $text[0] . ' ' . ( $frontend ? $text[1] : $text[2] ), $name, "<a href='$edit_link'>", '</a>' );
					if ( $frontend ) {
						echo Helper::make_message( $text, 'danger' );
					} else {
						Helper::render_admin_notice( $text, 'error' );
					}
				}
			}
			// check if email is set
			elseif ( ! isset( $args['normal']['email'] ) || empty( $args['normal']['email'] ) ) {
				$text = array(
					__( "Caution: There is no email field associated with %s.", 'greyd_forms' ),
					__( "%1\$sEdit the form%2\$s and assign an email field at the bottom in the \"Settings\" section.", 'greyd_forms' ),
					__( "Assign an email field at the bottom under „Settings“.", 'greyd_forms' ),
				);
				$text = sprintf( $text[0] . ' ' . ( $frontend ? $text[1] : $text[2] ), $name, "<a href='$edit_link'>", '</a>' );
				if ( $frontend ) {
					echo Helper::make_message( $text, 'danger' );
				} else {
					Helper::render_admin_notice( $text, 'error' );
				}
			}
		}
	}

	public function optout_warning() {
		$warnings = array();
		$settings = self::get_option();
		foreach ( $settings as $interface => $args ) {
			$enable = is_array( $args ) && isset( $args['enable'] ) && $args['enable'] === 'on' ? true : false;
			if ( $enable ) {
				$optout = self::get_config( $interface, 'optout' );
				if ( ! $optout ) {
					$warnings[] = self::get_config( $interface, 'name' );
				}
			}
		}
		if ( count( $warnings ) > 0 ) {
			echo Helper::render_info_box(
				array(
					'style' => 'orange',
					'text'  => sprintf( __( "Attention! The opt-out is not supported by these interfaces: %s", 'greyd_forms' ), implode( ', ', $warnings ) ),
				)
			);
		}
	}

	/**
	 * =================================================================
	 *                          Helper
	 * =================================================================
	 */

	/**
	 * Get interface config
	 */
	public static function get_config( $interface = '', $layer1 = '', $layer2 = '' ) {
		$interfaces = count( self::$interfaces ) === 0 ? self::get_interface_config() : self::$interfaces;
		$return     = isset( $interfaces[ $interface ] ) ? $interfaces[ $interface ] : false;
		if ( ! empty( $layer1 ) ) {
			$return = isset( $return[ $layer1 ] ) ? $return[ $layer1 ] : false;
			if ( ! empty( $layer2 ) ) {
				$return = isset( $return[ $layer2 ] ) ? $return[ $layer2 ] : false;
			}
		}
		return $return;
	}

	/**
	 * Get saved interface settings
	 */
	public static function get_option( $interface = '', $option = '' ) {
		$return = count( self::$settings ) === 0 ? (array) get_option( self::SETTING, null ) : self::$settings;

		// apply filters for backward compatibility
		$return = apply_filters( 'tp_forms_interface_settings', $return );

		// dig deeper
		if ( ! empty( $interface ) ) {
			$return = isset( $return[ $interface ] ) ? $return[ $interface ] : false;
			if ( ! empty( $option ) ) {
				$return = isset( $return[ $option ] ) ? $return[ $option ] : false;
			}
		}
		return $return;
	}

	/**
	 * Get post meta setup for a single form
	 */
	public static function get_meta( $post_id, $interface = '', $option = '' ) {
		$return = get_post_meta( $post_id, self::OPTION, true );

		// apply filters for backward compatibility
		$return = apply_filters( 'tp_forms_interface_meta', $return, $post_id );

		// dig deeper
		if ( ! empty( $interface ) ) {
			$return = isset( $return[ $interface ] ) ? $return[ $interface ] : false;
			if ( ! empty( $option ) ) {
				$return = isset( $return[ $option ] ) ? $return[ $option ] : false;
			}
		}
		return $return;
	}

	/**
	 * Text when no interface settings are defined
	 */
	public static function interface_empty_text( $echo = true ) {
		$return = "<i class='info'>" . sprintf(
			__( "Here you can set how the data is passed on to interfaces (such as Salesforce). To do this, you must first activate and set the required interfaces under %1\$sForms > Settings%2\$s.", 'greyd_forms' ),
			'<a href="' . admin_url( 'edit.php?post_type=tp_forms&page=greyd_settings_forms' ) . '">',
			'</a>'
		) . '</i>';
		if ( $echo ) {
			echo $return;
		}
		return $return;
	}

	/**
	 * Info message when interface setting is incomplete.
	 */
	public static function interface_incomplete_info( $interface, $echo = true ) {
		$return = "<small class='info'>" . self::interface_incomplete_text( $interface ) . '</small>';
		if ( $echo ) {
			echo $return;
		}
		return $return;
	}

	/**
	 * Text when interface setting is incomplete.
	 */
	public static function interface_incomplete_text( $interface ) {
		return sprintf(
			__( "The connection to %1\$s has not yet been fully configured. %2\$sPlease set all required options%3\$s", 'greyd_forms' ),
			self::get_config( $interface, 'name' ),
			'<a href="' . admin_url( 'edit.php?post_type=tp_forms&page=greyd_settings_forms' ) . '">',
			'</a>'
		);
	}

	/**
	 * Check if interface is enabled
	 *
	 * @param string $interface Interface Name
	 *
	 * @return bool
	 */
	public static function is_interface_enabled( $interface ) {

		$settings = (array) self::$settings;
		$settings = isset( $settings[ $interface ] ) ? $settings[ $interface ] : false;

		if ( ! $settings ) {
			return false;
		}

		return isset( $settings['enable'] ) && $settings['enable'] === 'on' ? true : false;
	}

	/**
	 * Check if settings for interface are complete
	 */
	public static function is_setting_complete( $config, $settings, $interface ) {
		foreach ( (array) $config as $option => $label ) {
			if ( isset( $settings[ $option ] ) && ! empty( $settings[ $option ] ) ) {
				continue;
			}

			// deprecated api key of hubspot. skip it for now.
			elseif ( ( $interface === 'hubspot' && $option === 'apikey' ) || ( $interface === 'hubspot' && $option === 'auth_token' ) ) {
				continue;
			} else {
				return false;
			}
		}
		return true;
	}

	/**
	 * Get field values from $_POST and convert them to propper array for saving as post meta
	 */
	public static function get_field_values( $args, $type = 'normal' ) {
		$return = array();
		$fields = isset( $args[ $type ] ) ? (array) $args[ $type ] : array();
		if ( isset( $fields['frm'] ) && count( $fields['frm'] ) > 1 && isset( $fields['api'] ) && count( $fields['api'] ) > 1 ) {
			foreach ( $fields['api'] as $i => $api_val ) {
				$frm_val = $fields['frm'][ $i ];
				if ( ! Helper::is_select_empty( $api_val ) && ! Helper::is_select_empty( $frm_val ) ) {
					$return[ $api_val ] = $frm_val;
				}
			}
		}
		return $return;
	}

	/**
	 * Get entry data
	 */
	public static function get_entry_interface_data( $entry_id = '', $interface = '' ) {
		if ( empty( $entry_id ) ) {
			return false;
		}
		$meta = get_post_meta( $entry_id, self::ENTRY, true );
		$meta = ! empty( $meta ) ? (array) $meta : array();
		return isset( $meta[ $interface ] ) ? $meta[ $interface ] : $meta;

	}

	/**
	 * Update entry data
	 */
	public static function update_entry_data( $entry_id, $interface, $data ) {
		$meta               = get_post_meta( $entry_id, self::ENTRY, true );
		$meta               = ! empty( $meta ) ? (array) $meta : array();
		$meta[ $interface ] = $data;
		update_post_meta( $entry_id, self::ENTRY, $meta );
	}

	/**
	 * Send an HTTP request
	 *
	 * @param string $url       URL to send the request to
	 * @param array  $data      Data to send
	 * @param string $method    HTTP method
	 * @param string $format    How $data should be formated before send (supports: 'url', 'JSON')
	 * @param bool   $handle    Whether the response should be run through self::handle_http_response()
	 *
	 * @return array|WP_Error
	 */
	public static function send_http_request( $url = '', $data = array(), $method = 'POST', $format = 'JSON', $handle = false ) {
		if ( empty( $url ) || empty( $data ) ) {
			return false;
		}

		$method = strtoupper( trim( esc_attr( $method ) ) );
		$format = $format === 'url' ? 'http_build_query' : 'json_encode';

		$args = array(
			'method'  => $method,
			'headers' => array(
				'Content-Type' => 'application/json',
				'Accept'       => '*/*',
			),
			'body'    => call_user_func( $format, $data ),
		);

		/**
		 * Filter HTTP request url.
		 *
		 * @filter greyd_forms_interface_remote_request_url
		 *
		 * @param string $url       HTTP request url.
		 * @param array  $args      HTTP request arguments.
		 * @param array  $data      Data to send to the interface.
		 */
		$url = apply_filters( 'greyd_forms_interface_remote_request_url', $url, $args, $data );

		/**
		 * Filter HTTP request arguments.
		 *
		 * @filter greyd_forms_interface_remote_request_arguments
		 *
		 * @param array  $args      HTTP request arguments.
		 * @param string $url       HTTP request url.
		 * @param array  $data      Data to send to the interface.
		 */
		$args = apply_filters( 'greyd_forms_interface_remote_request_arguments', $args, $url, $data );

		$api_response = wp_remote_request( $url, $args );

		return $handle ? self::handle_http_response( $api_response ) : $api_response;
	}

	/**
	 * Handle a HTTP response and get the http default message
	 *
	 * @param mixed $resonse
	 *
	 * @return array [ 'success' => bool, 'text' => {{message}} ]
	 */
	public static function handle_http_response( $response ) {
		if ( empty( $response ) ) {
			return false;
		}

		$return = array(
			'success' => false,
			'text'    => 'Unknown HTTP Response',
		);

		if ( is_wp_error( $response ) ) {
			$return['text'] = $response->get_error_message();
		} else {
			$code = wp_remote_retrieve_response_code( $response );
			$msg  = wp_remote_retrieve_response_message( $response );
			// wp returns empty string if incorrect parameter given
			if ( $code !== '' && $msg !== '' ) {
				$codes   = array(
					0 => 'Unknown',
					1 => 'Info',
					2 => 'Success',
					3 => 'Redirect',
					4 => 'Client Error',
					5 => 'Server Error',
				);
				$subcode = intval( substr( strval( $code ), 0, 1 ) );
				$return  = array(
					'success' => ( $subcode === 1 || $subcode === 2 ? true : false ),
					'text'    => sprintf(
						'%s: %s - %s',
						( isset( $codes[ $subcode ] ) ? $codes[ $subcode ] : $codes[0] ),
						$code,
						$msg
					),
				);
			}
		}
		return $return;
	}
}

/*
 * Backward Compativility PHP < 7.3.0
 * @see https://www.php.net/manual/de/function.array-key-first.php
 */
if ( ! function_exists( 'array_key_first' ) ) {
	function array_key_first( array $arr ) {
		foreach ( $arr as $key => $unused ) {
			return $key;
		}
		return null;
	}
}
