<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Hubspot();

class Hubspot {

	const INTERFACE        = 'hubspot';
	const BASE_URL         = 'https://api.hubapi.com/contacts/v1';
	const HEADERS          = array(
		'Content-Type' => 'application/json',
		'Accept'       => 'application/json',
	);
	const AUTH_CHANGE_TIME = '30.11.2022';

	public function __construct() {

		// settings
		$action = 'render_setting_' . self::INTERFACE . '_';

		add_action( $action . 'auth_token', array( $this, 'render_auth_token' ), 10, 2 );
		add_action( $action . 'lists', array( $this, 'render_lists' ), 10, 2 );

		// admin ajax
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );

		add_filter( 'admin_notices', array( $this, 'display_api_change_notice' ) );
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */

	/**
	 * Render auth token option
	 */
	public function render_auth_token( $pre = '', $value = null ) {
		$option = 'auth_token';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? $value : esc_attr( get_option( $slug, '' ) );

		echo "<input type='password' id='$slug' class='regular-text' name='$slug' value='$value'>";

		echo "<input type='button' style='margin-left: 10px;' class='button small' value='" . __( "Show tokens", 'greyd_forms' ) . "' onclick='hubspot.showToken()' />";
		echo "<input type='button' style='margin-left: 10px;' class='button small' value='" . __( "Insert token", 'greyd_forms' ) . "'  onclick='hubspot.handlePaste()' />";

		// https://developers.hubspot.com/docs/api/migrate-an-api-key-integration-to-a-private-app
		echo "<br><small><a href='" .
			__( 'https://developers.hubspot.com/docs/api/private-apps', 'greyd_forms' ) .
			"' target='_blank'>" .
			__( "Create private app and access Hubspot auth-token", 'greyd_forms' ) .
		'</a></small>';
	}

	/**
	 * render list option
	 */
	public function render_lists( $pre = '', $value = null ) {

		$option = 'lists';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		$lists  = strpos( $value, '{' ) !== false ? json_decode( $value, true ) : $value;
		$token  = Greyd_Forms_Interface::get_option( self::INTERFACE, 'auth_token' );

		// css classes
		$not_ready_class = 'hidden';
		$ready_class     = '';
		if ( empty( $token ) ) {
			$not_ready_class = '';
			$ready_class     = 'hidden';
		}
		$empty_class = 'empty ' . ( empty( $value ) ? '' : 'hidden' );
		$set_class   = 'set ' . ( empty( $value ) ? 'hidden' : '' );
		$info_icon   = "<span class='dashicons dashicons-info'></span>&nbsp;";

		echo "<div id='hubspot'>"; // wrapper

		echo "<input type='hidden' id='$slug' name='$slug' value='$value'>";

		// not ready
		echo "<div class='not_ready $not_ready_class'>";
			echo "<p style='opacity:.5;'>" . $info_icon . __( "Please enter your API key for HubSpot first.", 'greyd_forms' ) . '</p>';
		echo '</div>';

		// ready
		echo "<div class='ready $ready_class'>";
			// loader
			echo "<span style='display:none'><span class='loader'></span></span>";

			// empty
			echo "<div class='$empty_class'>";
				echo "<p style='opacity:.5;'>" . $info_icon . __( "No lists available yet.", 'greyd_forms' ) . '</p>';
				echo "<span class='button getLists' style='margin-top:10px;'>" . __( "Get lists now", 'greyd_forms' ) . '</span>';
			echo '</div>';

			// set
			echo "<div class='$set_class'>";
				echo "<ul class='input_list'>";
		if ( isset( $lists ) && is_array( $lists ) && count( $lists ) > 0 ) {
			if ( count( $lists ) > 1 ) {
				foreach ( (array) $lists as $id => $name ) {
					if ( $id === 'none' ) {
						continue;
					}
					echo "<li><strong>$name</strong> (ID: $id)</li>";
				}
			} else {
				echo '<li>' . array_values( $lists )[0] . '</li>';
			}
		}
				echo '</ul><br>';
				echo "<span class='button getLists $set_class' style='margin-top:10px;'>" . __( "Update lists", 'greyd_forms' ) . '</span>';
			echo '</div>';

			// set
			echo "<div class='_error hidden'>";
				echo Helper::render_info_box(
					array(
						'style' => 'red',
						'text'  => '<span class=text>' . __( "Lists could not be retrieved. Check your API key and try again.", 'greyd_forms' ) . '</span>',
					)
				);
				echo "<span class='button getLists' style='margin-top:10px;'>" . __( "try again", 'greyd_forms' ) . '</span>';
			echo '</div>';

		echo '</div>';

		echo '</div>'; // wrapper end
	}

	/**
	 * admin script
	 */
	public function load_backend() {

		$page = isset( $_GET['page'] ) ? $_GET['page'] : '';

		if ( $page === 'greyd_settings_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_localize_script(
				self::INTERFACE . '_backend_js',
				'local_' . self::INTERFACE,
				array(
					'ajaxurl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
					'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				)
			);
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}

	/**
	 * handle admin ajax
	 */
	public function ajax( $data ) {
		$response = '';

		$token = isset( $data['token'] ) ? $data['token'] : '';

		if ( ! empty( $token ) ) {
			$response = self::get_lists( $token );
			debug( $response );

			// success
			if ( is_array( $response ) ) {
				$response = json_encode( $response );
			}
			// error
			else {
				wp_die( 'error::' . $response );
			}
		}

		wp_die( 'success::' . $response );
	}

	/**
	 * Get lists from API
	 */
	public static function get_lists( $token = '' ) {

		if ( empty( $token ) ) {
			return false;
		}

		$response = __( "Lists could not be retrieved. Check your API key and try again.", 'greyd_forms' );

		$api_response = wp_remote_get(
			self::BASE_URL . '/lists/static?count=250',
			array(
				'headers' => array(
					'Authorization' => "Bearer {$token}",
					'Content-Type'  => 'application/json',
				),
			)
		);

		// success
		if ( is_array( $api_response ) && ! is_wp_error( $api_response ) ) {
			$body  = isset( $api_response['body'] ) ? json_decode( $api_response['body'], true ) : array();
			$lists = isset( $body['lists'] ) ? $body['lists'] : array();
			if ( is_array( $lists ) ) {
				$response = array(
					'none' => __( "no list", 'greyd_forms' ),
				);
				foreach ( (array) $lists as $list ) {
					$id   = isset( $list['listId'] ) ? $list['listId'] : '';
					$name = isset( $list['name'] ) ? $list['name'] : '';
					if ( ! empty( $id ) && ! empty( $name ) ) {
						$response[ $id ] = $name;
					}
				}
			} else {
				$response = isset( $body['message'] ) ? $body['message'] : $response;
			}
		}
		return $response;
	}

	/**
	 * Send form data to hubspot API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		$token = Greyd_Forms_Interface::get_option( self::INTERFACE, 'auth_token' );
		if ( empty( $token ) ) {
			return false; // early exit
		}

		// meta
		$vid    = null;
		$list   = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['list'] ) ? $postmeta['meta']['list'] : '';
		$update = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['update'] ) ? true : false;
		$name   = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

		// save meta in entry
		Greyd_Forms_Interface::update_entry_data(
			$entry_id,
			self::INTERFACE,
			array(
				'list'   => $list,
				'update' => $update,
			)
		);

		// push fields to array
		$interface_data = array();
		$email          = '';
		$fields         = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();
		$fields         = isset( $postmeta['custom'] ) ? array_merge( $fields, (array) $postmeta['custom'] ) : $fields;
		foreach ( $fields as $key => $val ) {
			$value = html_entity_decode( $formdata[ $val ] );

			// if we're updating a contact, we need the email for later
			if ( $update && $key === 'email' ) {
				$email = $formdata[ $val ];
				continue;
			} elseif ( ! empty( $value ) ) {
				$interface_data[] = array(
					'property' => $key,
					'value'    => $value,
				);
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */

		if ( $update ) {
			$response = self::create_or_update_contact( $token, $interface_data, $email );
		} else {
			$response = self::create_contact( $token, $interface_data );
		}

		// error
		if ( $response['success'] === false ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . $response['text'] );
			if ( isset( $response['vid'] ) ) {
				$vid = $response['vid'];
			}
		}
		// success
		else {
			$vid      = $response['text'];
			$response = true;
		}

		// add to list
		if ( ! empty( $list ) && $list !== 'none' ) {
			if ( ! empty( $vid ) ) {

				// safe to entry meta
				Greyd_Forms_Interface::update_entry_data(
					$entry_id,
					self::INTERFACE,
					array(
						'list'  => $list,
						'email' => $email,
						'vid'   => $vid,
					)
				);

				$response = self::add_contact_to_list( $token, $vid, $list );

				// error
				if ( $response['success'] === false ) {
					Helper::log_entry_state( $entry_id, $name . ': ' . $response['text'] );
				}
				// success
				else {
					$listname = Greyd_Forms_Interface::get_option( self::INTERFACE, 'lists' );
					$listname = strpos( $listname, '{' ) !== false ? "'" . json_decode( $listname, true )[ $list ] . "'" : sprintf( "mit der ID '%s'", $list );
					Helper::log_entry_state( $entry_id, sprintf( __( "Contact has been added to the %s list.", 'greyd_forms' ), $listname ), 'success' );
					return true;
				}
			}
		}
		return $response;
	}

	/**
	 * Create a new contact
	 */
	public static function create_contact( $token = '', $properties = array() ) {
		if ( empty( $token ) || empty( $properties ) ) {
			return false;
		}

		$args = array(
			'headers' => self::HEADERS,
			'body'    => json_encode(
				array(
					'properties' => $properties,
				)
			),
		);

		$args['headers']['Authorization'] = "Bearer {$token}";
		$api_response                     = wp_remote_post( self::BASE_URL . '/contact', $args );

		return self::handle_response( $api_response, 'vid' );
	}

	/**
	 * Create or update a contact
	 */
	public static function create_or_update_contact( $token = '', $properties = array(), $email = '' ) {
		if ( empty( $token ) || empty( $properties ) || empty( $email ) ) {
			return false;
		}

		$args = array(
			'headers' => self::HEADERS,
			'body'    => json_encode(
				array(
					'properties' => $properties,
				)
			),
		);

		$args['headers']['Authorization'] = "Bearer {$token}";
		$api_response                     = wp_remote_post( self::BASE_URL . '/contact/createOrUpdate/email/' . $email . '/', $args );

		return self::handle_response( $api_response, 'vid' );
	}

	/**
	 * Add a contact to a list
	 */
	public static function add_contact_to_list( $token = '', $vid = '', $listID = '' ) {
		if ( empty( $token ) || empty( $vid ) || empty( $listID ) ) {
			return false;
		}

		$args = array(
			'headers' => self::HEADERS,
			'body'    => json_encode(
				array(
					'vids' => array( $vid ),
				)
			),
		);

		$args['headers']['Authorization'] = "Bearer {$token}";
		$api_response                     = wp_remote_post( self::BASE_URL . '/lists/' . $listID . '/add/', $args );

		return self::handle_response( $api_response, 'updated' );
	}

	/**
	 * handle HupSpot API response
	 */
	public static function handle_response( $response = array(), $get = null ) {
		$return = array(
			'success' => false,
			'text'    => '',
		);
		if ( is_wp_error( $response ) ) {
			$response = $response->get_error_message();
		} else {
			$response = json_decode( wp_remote_retrieve_body( $response ), true );
		}

		if ( is_array( $response ) ) {

			if ( isset( $response['status'] ) && $response['status'] === 'error' ) {

				if ( isset( $response['errors'] ) ) {
					foreach ( (array) $response['errors'] as $error ) {
						$return['text'] .= $return['text'] === '' ? $error['message'] : ', ' . $error['message'];
					}
				} else {
					$return['text'] = $response['message'];
				}

				/**
				 * @since 2.5.6 add validationResults errors to response message.
				 */
				if ( isset( $response['validationResults'] ) && is_array( $response['validationResults'] ) ) {
					foreach ( $response['validationResults'] as $validationResults ) {
						$msg             = ' ( error: ' . $validationResults['error'] . ' | name: ' . $validationResults['name'] . ' | message: ' . $validationResults['message'] . ')';
						$return['text'] .= $return['text'] === '' ? $msg : ', ' . $msg;
					}
				}

				// contact already exists -> get vid anyway for later
				if ( isset( $response['identityProfile'] ) && isset( $response['identityProfile']['vid'] ) ) {
					$return['vid'] = $response['identityProfile']['vid'];
				}
			} else {
				$return['success'] = true;
				$return['text']    = $get && isset( $response[ $get ] ) ? $response[ $get ] : $response;
			}
		} else {
			$return['text'] = $response;
		}

		return $return;
	}


	/**
	 * Display admin notice to install via wizard.
	 */
	public function display_api_change_notice() {
		$apikey = Greyd_Forms_Interface::get_option( self::INTERFACE, 'apikey' );
		$token  = Greyd_Forms_Interface::get_option( self::INTERFACE, 'auth_token' );

		if ( ! empty( $token ) ) {
			return;
		}

		if ( ! empty( $apikey ) ) {
			if ( strtotime( self::AUTH_CHANGE_TIME ) > strtotime( 'now' ) ) {
				$message      = __( "As of November 30, Hubspot API keys will be eliminated. Migrate to Private App with Access Token now.", 'greyd_forms' );
				$notice_class = 'notice-info';
			} else {
				$message      = __( "As of November 30, Hubspot API keys have been eliminated. Migrate to Private App with Access Token now.", 'greyd_forms' );
				$notice_class = 'notice-error';
			}

			echo "<div class='{$notice_class} notice is-dismissible'>
				<div class='flex' style='padding: 12px 0; flex-direction: row; justify-content: flex-start; align-items: center;'>
				
					<div>
						<p>" . $message . "</p>
						<p>
							<a class='' href='https://developers.hubspot.com/docs/api/migrate-an-api-key-integration-to-a-private-app'>" . __( 'Mehr erfahren', 'core' ) . '</a>
						</p>
					</div>
				</div>
			</div>';
		}
	}
}
