<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Mailchimp_Handler {

	public static $errors = array();
	public static $client;

	public function __construct( $apikey, $dc ) {

		if ( empty( $apikey ) || empty( $dc ) ) {
			return false;
		}

		Helper::include_library(
			array(
				'path' => __DIR__,
				'file' => '/mailchimp-marketing-php/vendor/autoload.php',
				'url'  => 'https://update.greyd.io/libs/mailchimp-marketing-php.zip',
			)
		);

		self::$client = new \MailchimpMarketing\ApiClient();
		self::$client->setConfig(
			array(
				'apiKey' => $apikey,
				'server' => $dc,
			)
		);
	}

	public static function get_lists() {

		$return   = array();
		$response = self::$client->lists->getAllLists();
		$lists    = $response->lists;

		foreach ( $lists as $list ) {
			$return[ $list->id ] = $list->name;
		}

		return $return;
	}

	public static function create_member( $args = array(), $send_activationmail = true ) {
		$return = false;

		// return if no email set
		if ( ! isset( $args ) || ! isset( $args['email'] ) ) {
			return false;
		}

		// set vars
		$email           = isset( $args['email'] ) ? substr( $args['email'], 0, 255 ) : null;
		$list_id         = $args['memberslist_id'];
		$subscriber_hash = md5( strtolower( $email ) );
		$member          = null;
		$status          = '';

		// Get Information about DOI
		$response           = self::$client->lists->getList( $list_id );
		$chimp_double_optin = $response->double_optin;

		// Set Status
		if ( $chimp_double_optin === true ) {
			$status = 'pending';
		} else {
			$status = 'subscribed'; // for Testing
		}

		$member_args = array(
			'email_address' => $email,
			'status_if_new' => $status,
		);
		if ( isset( $args['merge_fields'] ) ) {
			$member_args['merge_fields'] = $args['merge_fields'];
		}

		try {
			$member = self::$client->lists->setListMember( $list_id, $subscriber_hash, $member_args );

		} catch ( \MailchimpMarketing\ApiException $e ) {

			// Catch API client exceptions

			$error_message  = $e->getMessage();
			$error_array    = array();
			$error_array[0] = $error_message;
			$error_array    = json_decode( $error_array[0], true );
			$return         = $error_array;

		} catch ( \ClientErrorResponseException | \GuzzleHttp\Exception\BadResponseException  $e ) {

			$error_message      = $e->getResponse();
			$error_message_body = $error_message->getBody()->getContents();
			$error_array        = json_decode( $error_message_body, true );
			$return             = $error_array;
		}

		return $member !== null ? $member : $return;
	}

	public static function get_member( $list_id, $email ) {

		// return if email or list-id are missing
		if ( ! isset( $list_id ) || ! isset( $email ) ) {
			return false;
		}

		// 1. Get Recipient
		$id     = null;
		$return = sprintf( __( "No recipient with the mail address %s found.", 'greyd_forms' ), $email );

		// Generating a MD5 Hash of the lowercase email
		$subscriber_hash = md5( strtolower( $email ) );
		if ( empty( $subscriber_hash ) ) {
			return false;
		}

		try {
			$response = self::$client->lists->getListMember( $list_id, $subscriber_hash );

		} catch ( \MailchimpMarketing\ApiException $e ) {
			// Catch API client exceptions
			$return = $e->getMessage();
		}
		$return = $response;

		return $return;
	}

	public static function delete_member( $list_id, $subscriber_hash ) {
		if ( empty( $list_id ) || empty( $subscriber_hash ) ) {
			return false;
		}

		$service  = self::$client->lists;
		$response = $service->deleteListMember( $list_id, $subscriber_hash );

		return $response;
	}

	public static function get_merge_fields( $list_id ) {

		// return if list-id is missing
		if ( ! isset( $list_id ) || empty( $list_id ) ) {
			return false;
		}

		try {
			$response = self::$client->lists->getListMergeFields( $list_id );

		} catch ( \MailchimpMarketing\ApiException $e ) {
			// Catch API client exceptions
			$return = $e->getMessage();

		} catch ( \ClientErrorResponseException | \GuzzleHttp\Exception\BadResponseException  $e ) {
			$error_message      = $e->getResponse();
			$error_message_body = $error_message->getBody()->getContents();
			$error_array        = json_decode( $error_message_body, true );
			$return             = $error_array;
		}

		$merge_fields = $response->merge_fields;

		foreach ( $merge_fields as $merge_field ) {
			$return[ $merge_field->tag ] = array(
				'name' => $merge_field->name,
				'type' => $merge_field->type,
			);
		}

		return $return;
	}
}
