<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Mailchimp();

class Mailchimp {

	const INTERFACE = 'mailchimp';

	public function __construct() {

		// settings
		$action = 'render_setting_' . self::INTERFACE . '_';
		add_action( $action . 'apikey', array( $this, 'render_apikey' ), 10, 2 );
		add_action( $action . 'dc', array( $this, 'render_dc' ), 10, 2 );
		add_action( $action . 'lists', array( $this, 'render_lists' ), 10, 2 );

		// admin ajax
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
		add_action( 'formhandler_optout_' . self::INTERFACE, array( $this, 'optout' ), 10, 2 );
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */

	public function render_apikey( $pre = '', $value = null ) {

		$option = 'apikey';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_dc( $pre = '', $value = null ) {

		$option = 'dc';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_lists( $pre = '', $value = null ) {

		$option = 'lists';
		$slug   = $pre . '[' . $option . ']';

		$value    = isset( $value ) ? strval( $value ) : '';
		$lists    = strpos( $value, '{' ) !== false ? json_decode( $value, true ) : $value;
		$settings = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$apikey   = isset( $settings['apikey'] ) ? $settings['apikey'] : '';
		$dc       = isset( $settings['dc'] ) ? $settings['dc'] : '';

		// classes
		$not_ready_class = 'hidden';
		$ready_class     = '';
		if ( empty( $apikey ) || empty( $dc ) ) {
			$not_ready_class = '';
			$ready_class     = 'hidden';
		}
		$empty_class = 'empty ' . ( empty( $value ) ? '' : 'hidden' );
		$set_class   = 'set ' . ( empty( $value ) ? 'hidden' : '' );
		$info_icon   = "<span class='dashicons dashicons-info'></span>&nbsp;";

		echo "<div id='mailchimp'>"; // wrapper

		// hidden input
		echo "<input type='hidden' id='$slug' name='$slug' value='$value'>";

		// not ready
		echo "<div class='not_ready $not_ready_class'>";
		echo "<p style='opacity:.5;'>" . $info_icon . __( "Please enter your API key and the server abbreviation for Mailchimp first.", 'greyd_forms' ) . '</p>';
		echo '</div>';

		// ready
		echo "<div class='ready $ready_class'>";

		// loader
		echo "<span style='display:none'><span class='loader'></span></span>";

		// empty
		echo "<div class='$empty_class'>";
		echo "<p style='opacity:.5;'>" . $info_icon . __( "No lists available yet.", 'greyd_forms' ) . '</p>';
		echo "<span class='button getLists' style='margin-top:10px;'>" . __( "Get lists now", 'greyd_forms' ) . '</span>';
		echo '</div>';

		// set
		echo "<div class='$set_class'>";
		echo "<ul class='input_list'>";

		if ( isset( $lists ) && is_array( $lists ) && count( $lists ) > 0 ) {
			foreach ( $lists as $id => $name ) {
				echo "<li><strong>$name</strong> (ID: $id)</li>";
			}
		}
		echo '</ul><br>';
		echo "<span class='button getLists $set_class' style='margin-top:10px;'>" . __( "Update lists", 'greyd_forms' ) . '</span>';
		echo '</div>';

		// set
		echo "<div class='_error hidden'>";
		echo Helper::render_info_box(
			array(
				'style' => 'red',
				'text'  => '<span class=text>' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) . '</span>',
			)
		);
		echo "<span class='button getLists' style='margin-top:10px;'>" . __( "try again", 'greyd_forms' ) . '</span>';
		echo '</div>';

		echo '</div>';

		echo '</div>'; // wrapper end
	}

	/**
	 * admin script
	 */
	public function load_backend() {

		$page     = isset( $_GET['page'] ) ? $_GET['page'] : '';
		$posttype = get_post_type() ? get_post_type() : get_current_screen()->post_type;

		if ( $page === 'greyd_settings_forms' || $posttype === 'tp_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_localize_script(
				self::INTERFACE . '_backend_js',
				'local_' . self::INTERFACE,
				array(
					'ajaxurl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
					'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				)
			);
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}

	/**
	 * handle admin ajax
	 */
	public function ajax( $data ) {

		$apikey        = isset( $data['apikey'] ) ? $data['apikey'] : '';
		$dc            = isset( $data['dc'] ) ? $data['dc'] : '';
		$selected_list = isset( $data['list'] ) ? $data['list'] : '';

		if ( ! empty( $apikey ) || ! empty( $dc ) ) {

			include_once __DIR__ . '/handle.php';
			$mailchimp = new Mailchimp_Handler( $apikey, $dc );
			$lists     = $mailchimp::get_lists();

			if ( is_array( $lists ) && count( $lists ) > 0 ) {
				wp_die( 'success::' . json_encode( $lists ) );
			}
		}

		if ( ! empty( $selected_list ) ) {

			$options = Greyd_Forms_Interface::get_option( self::INTERFACE );
			$apikey  = isset( $options['apikey'] ) ? $options['apikey'] : '';
			$dc      = isset( $options['dc'] ) ? $options['dc'] : '';

			include_once __DIR__ . '/handle.php';
			$mailchimp    = new Mailchimp_Handler( $apikey, $dc );
			$merge_fields = $mailchimp::get_merge_fields( $selected_list );

			if ( is_array( $merge_fields ) ) {
				wp_die( 'success::' . json_encode( $merge_fields ) );
			}
		}
		wp_die( 'error::' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) );
	}

	/**
	 * Send form data to API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		$options = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$apikey  = isset( $options['apikey'] ) ? $options['apikey'] : '';
		$dc      = isset( $options['dc'] ) ? $options['dc'] : '';
		$lists   = isset( $options['lists'] ) && strpos( $options['lists'], '{' ) !== false ? json_decode( $options['lists'], true ) : '';

		// early exit
		if ( empty( $apikey ) || empty( $dc ) || empty( $lists ) ) {
			return false;
		}

		// meta
		$interface_data = array();
		$list           = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['memberslist'] ) ? $postmeta['meta']['memberslist'] : '';
		$list           = ! empty( $list ) ? $list : $lists[ array_key_first( $lists ) ];
		$name           = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

		// set List-ID as argument for create_member()
		$interface_data['memberslist_id'] = $list;

		// push fields to array
		$fields = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();

		if ( ! isset( $fields['email'] ) ) {
			return false;
		}

		foreach ( $fields as $key => $val ) {

			// Split Merge Tag & Field Type
			if ( strpos( $key, '::' ) !== false ) {
				$exploded = explode( '::', $key );
				$key      = $exploded[0];
				$type     = $exploded[1];
			}

			// validate the fields
			$value = isset( $formdata[ $val ] ) ? $this->validate_field( $key, $formdata[ $val ], $type ) : null;
			// add field to array only if validation is successfull
			if ( $value !== false ) {
				if ( $key == 'email' ) {
					$interface_data[ $key ] = html_entity_decode( $value );
				} else {
					if ( ! isset( $interface_data['merge_fields'] ) ) {
						$interface_data['merge_fields'] = array();
					}
					$interface_data['merge_fields'][ $key ] = html_entity_decode( $value );
				}
			}
		}

		// no merge fields are set
		if ( ! isset( $interface_data['merge_fields'] ) || empty( $interface_data['merge_fields'] ) ) {
			if ( isset( $interface_data['merge_fields'] ) ) {
				unset( $interface_data['merge_fields'] );
			}
			$interface_data['skip_merge_validation'] = true;
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// return when email not set
		if ( empty( $interface_data ) || ! isset( $interface_data['email'] ) ) {
			return $name . ': ' . __( "Email is not defined, entry could not be sent.", 'greyd_forms' );
		}

		// save necessary data for optout in entry
		Greyd_Forms_Interface::update_entry_data(
			$entry_id,
			self::INTERFACE,
			array(
				'memberslist_id' => $list,
				'email'          => $interface_data['email'],
			)
		);

		// handle
		include_once __DIR__ . '/handle.php';
		$mailchimp = new Mailchimp_Handler( $apikey, $dc );

		// args are: memberlist & merge_fields[]
		$response = $mailchimp::create_member( $interface_data );

		// success
		if ( isset( $response->id ) ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was created successfully.", 'greyd_forms' ), 'success' );
			return true;
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
		}
		return $response;
	}

	/**
	 * Validate a field
	 */
	public function validate_field( $name, $value, $type ) {
		// general maxlength
		$maxlength = 255;

		// maxlength validation
		$value = strval( $value );
		if ( strlen( $value ) > $maxlength ) {
			$value = substr( $value, 0, $maxlength );
		}

		if ( $type === 'birthday' ) {
			$value = date( 'm/d', strtotime( $value ) );
		} elseif ( $type === 'true' ) {
			$value = date( 'm/d/Y', strtotime( $value ) );
		}

		return $value;
	}

	/**
	 * format error from mailchimp response
	 */
	public static function get_error( $response ) {

		if ( ! isset( $response ) ) {
			return false;
		}
		$message = '';
		if ( isset( $response['title'] ) ) {
			$message = $response['title'];
		}
		if ( isset( $response['status'] ) ) {
			$message .= ' (' . $response['status'] . ')';
		}
		if ( isset( $response['detail'] ) ) {
			$message .= ': ' . $response['detail'];
		}
		if ( isset( $response['errors'] ) && count( $response['errors'] ) ) {
			$message .= ' [ERRORS: ';
			$message .= implode(
				' | ',
				array_map(
					function( $error ) {
						return implode( ': ', $error );
					},
					$response['errors']
				)
			);
			$message .= ' ]';
		}

		return $message;
	}

	/**
	 * handle opt out
	 */
	public function optout( $entry_id, $meta ) {

		if ( empty( $entry_id ) || empty( $meta ) || ! is_array( $meta ) ) {
			return false;
		}

		// Set Vars
		$list_id = isset( $meta['memberslist_id'] ) ? $meta['memberslist_id'] : '';
		if ( empty( $list_id ) ) {
			return false;
		}
		$options = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$apikey  = isset( $options['apikey'] ) ? $options['apikey'] : '';
		$dc      = isset( $options['dc'] ) ? $options['dc'] : '';
		$email   = isset( $meta['email'] ) ? $meta['email'] : '';

		if ( empty( $apikey ) || empty( $dc ) || empty( $list_id ) ) {
			return false;
		}

		include_once __DIR__ . '/handle.php';
		$mailchimp = new Mailchimp_Handler( $apikey, $dc );

		// Get Member
		$response = $mailchimp::get_member( $list_id, $email );

		// success:
		if ( isset( $response->id ) ) {
			$subscriber_hash = $response->id;
			$response        = $mailchimp::delete_member( $list_id, $subscriber_hash );
			$name            = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

			if ( isset( $response ) ) {
				Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
			} else {
				Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient has been deleted.", 'greyd_forms' ), 'success', false );
			}
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
		}
	}
}
