<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Mailjet();

class Mailjet {

	const INTERFACE = 'mailjet';

	public function __construct() {

		// settings
		$action = 'render_setting_' . self::INTERFACE . '_';
		add_action( $action . 'apikey', array( $this, 'render_apikey' ), 10, 2 );
		add_action( $action . 'apisecret', array( $this, 'render_apisecret' ), 10, 2 );

		// add_action( $action.'dc', array($this, 'render_dc'), 10, 2 );
		add_action( $action . 'lists', array( $this, 'render_lists' ), 10, 2 );

		// // admin ajax
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// // handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
		// add_action( 'formhandler_optout_'.self::INTERFACE, array($this, 'optout'), 10, 2 );
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */
	public function render_apikey( $pre = '', $value = null ) {

		$option = 'apikey';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_apisecret( $pre = '', $value = null ) {

		$option = 'apisecret';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_lists( $pre = '', $value = null ) {

		$option = 'lists';
		$slug   = $pre . '[' . $option . ']';

		$value     = isset( $value ) ? strval( $value ) : '';
		$lists     = strpos( $value, '{' ) !== false ? json_decode( $value, true ) : $value;
		$settings  = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$apikey    = isset( $settings['apikey'] ) ? $settings['apikey'] : '';
		$apisecret = isset( $settings['apisecret'] ) ? $settings['apisecret'] : '';

		// classes
		$not_ready_class = 'hidden';
		$ready_class     = '';
		if ( empty( $apikey ) || empty( $apisecret ) ) {
			$not_ready_class = '';
			$ready_class     = 'hidden';
		}
		$empty_class = 'empty ' . ( empty( $value ) ? '' : 'hidden' );
		$set_class   = 'set ' . ( empty( $value ) ? 'hidden' : '' );
		$info_icon   = "<span class='dashicons dashicons-info'></span>&nbsp;";

		echo "<div id='mailjet'>"; // wrapper

		// hidden input
		echo "<input type='hidden' id='$slug' name='$slug' value='$value'>";

		// not ready
		echo "<div class='not_ready $not_ready_class'>";
		echo "<p style='opacity:.5;'>" . $info_icon . __( "Please enter your API key and the API secret for Mailjet first.", 'greyd_forms' ) . '</p>';
		echo '</div>';

		// ready
		echo "<div class='ready $ready_class'>";

		// loader
		echo "<span style='display:none'><span class='loader'></span></span>";

		// empty
		echo "<div class='$empty_class'>";
		echo "<p style='opacity:.5;'>" . $info_icon . __( "No lists available yet.", 'greyd_forms' ) . '</p>';
		echo "<span class='button getLists' style='margin-top:10px;'>" . __( "Get lists now", 'greyd_forms' ) . '</span>';
		echo '</div>';

		// set
		echo "<div class='$set_class'>";
		echo "<ul class='input_list'>";

		if ( isset( $lists ) && is_array( $lists ) && count( $lists ) > 0 ) {
			foreach ( $lists as $id => $name ) {
				echo "<li><strong>$name</strong> (ID: $id)</li>";
			}
		}
		echo '</ul><br>';
		echo "<span class='button getLists $set_class' style='margin-top:10px;'>" . __( "Update lists", 'greyd_forms' ) . '</span>';
		echo '</div>';

		// set
		echo "<div class='_error hidden'>";
		echo Helper::render_info_box(
			array(
				'style' => 'red',
				'text'  => '<span class=text>' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) . '</span>',
			)
		);
		echo "<span class='button getLists' style='margin-top:10px;'>" . __( "try again", 'greyd_forms' ) . '</span>';
		echo '</div>';

		echo '</div>';

		echo '</div>'; // wrapper end
	}

	/**
	 * admin script
	 */
	public function load_backend() {

		$page     = isset( $_GET['page'] ) ? $_GET['page'] : '';
		$posttype = get_post_type() ? get_post_type() : get_current_screen()->post_type;

		if ( $page === 'greyd_settings_forms' || $posttype === 'tp_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_localize_script(
				self::INTERFACE . '_backend_js',
				'local_' . self::INTERFACE,
				array(
					'ajaxurl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
					'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				)
			);
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}


	/**
	 * handle admin ajax
	 */
	public function ajax( $data ) {

		$apikey        = isset( $data['apikey'] ) ? $data['apikey'] : '';
		$apisecret     = isset( $data['apisecret'] ) ? $data['apisecret'] : '';
		$selected_list = isset( $data['list'] ) ? $data['list'] : '';

		if ( ! empty( $apikey ) || ! empty( $apisecret ) ) {

			include_once __DIR__ . '/handle.php';
			$mailjet = new Mailjet_Handler( $apikey, $apisecret );

			$lists = $mailjet::get_lists();

			if ( is_array( $lists ) && count( $lists ) > 0 ) {
				wp_die( 'success::' . json_encode( $lists ) );
			}
		}

		if ( ! empty( $selected_list ) ) {

			$options   = Greyd_Forms_Interface::get_option( self::INTERFACE );
			$apikey    = isset( $options['apikey'] ) ? $options['apikey'] : '';
			$apisecret = isset( $options['apisecret'] ) ? $options['apisecret'] : '';

			include_once __DIR__ . '/handle.php';
			$mailjet = new Mailjet_Handler( $apikey, $apisecret );

			$contact_properties = $mailjet::get_contact_properties( $selected_list );

			if ( is_array( $contact_properties ) ) {
				wp_die( 'success::' . json_encode( $contact_properties ) );
			}
		}
		wp_die( 'error::' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) );
	}

	/**
	 * Send form data to hubspot API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		$options   = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$apikey    = isset( $options['apikey'] ) ? $options['apikey'] : '';
		$apisecret = isset( $options['apisecret'] ) ? $options['apisecret'] : '';

		$lists = isset( $options['lists'] ) && strpos( $options['lists'], '{' ) !== false ? json_decode( $options['lists'], true ) : '';

		// early exit
		if ( empty( $apikey ) || empty( $apisecret ) || empty( $lists ) ) {
			return false;
		}

		// meta
		$interface_data = array();
		$properties     = array();
		$list           = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['contactlist'] ) ? $postmeta['meta']['contactlist'] : '';
		$list           = ! empty( $list ) ? $list : $lists[ array_key_first( $lists ) ];

		$list_id = intval( $list );

		$name = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

		// push fields to array
		$fields = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();

		if ( ! isset( $fields['email'] ) ) {
			return false;
		}

		foreach ( $fields as $key => $val ) {

			if ( strpos( $key, '::' ) !== false ) {
				$exploded      = explode( '::', $key );
				$key           = $exploded[0];
				$property_name = $exploded[1];
			}

			// validate the fields
			$value = isset( $formdata[ $val ] ) ? $this->validate_field( $key, $formdata[ $val ] ) : null;

			// add field to array only if validation is successfull

			if ( $value !== false ) {
				if ( $key == 'email' ) {
					$interface_data[ $key ] = html_entity_decode( $value );
				} else {
					array_push(
						$properties,
						array(
							'Name'  => $property_name,
							'Value' => $value,
						)
					);
				}
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// return when email not set
		if ( empty( $interface_data ) || ! isset( $interface_data['email'] ) ) {
			return $name . ': ' . __( "Email is not defined, entry could not be sent.", 'greyd_forms' );
		}

		// save necessary data for optout in entry
		// Greyd_Forms_Interface::update_entry_data($entry_id, self::INTERFACE, [ 'contactlist_id' =>  $list, 'email' => $interface_data['email'] ]);

		// handle
		include_once __DIR__ . '/handle.php';
		$mailjet = new Mailjet_Handler( $apikey, $apisecret );

		/**
		 * Unfortunately up to 4 api calls are needed:
		 * (1) check if the contact already exists
		 * (2) create the contact
		 * (3) update the properties
		 * (4) add to the selected list
		 *
		 * @see https://github.com/mailjet/mailjet-apiv3-nodejs/issues/165
		 */

		// (1) check if contact exists
		$contact_exists = $mailjet::get_contact( $interface_data );

		if ( ! $contact_exists->success() ) {

			// (2) create contact
			$response = $mailjet::create_contact( $interface_data );

			// success
			if ( $response !== false && $response->success() ) {
				Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was created successfully.", 'greyd_forms' ), 'success' );
				$data       = $response->getData();
				$contact_id = $data[0]['ID'];
			}
			// error
			else {
				Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
				return false;
			}
		} else {
			$contact_id = $contact_exists->getData()[0]['ID'];
		}

		// (3) update contact properties
		$response = $mailjet::update_contact_properties( $contact_id, $properties );

		// success
		if ( $response !== false && $response->success() ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Contact Properties were geupdated", 'greyd_forms' ), 'success' );
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
			// return false;
		}

		// (4) add contact to list
		$response = $mailjet::add_contact_to_list( $contact_id, $list_id );
		// success
		if ( $response !== false && $response->success() ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Contact has been added to the list.", 'greyd_forms' ), 'success' );
			return true;
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
			// return false;
		}

		return $response;
	}

	/**
	 * validate a field
	 */
	public function validate_field( $name, $value ) {
		// general maxlength
		$maxlength = 255;

		// maxlength validation
		$value = strval( $value );
		if ( strlen( $value ) > $maxlength ) {
			$value = substr( $value, 0, $maxlength );
		}

		return $value;
	}

	/**
	 * format error from mailjet response
	 */
	public static function get_error( $response ) {

		if ( $response === false ) {
			return __( "Unknown error", 'greyd_forms' );
		}

		$status        = $response->getStatus();
		$reason_phrase = $response->getReasonPhrase();

		$message = '';
		if ( isset( $reason_phrase ) ) {
			$message = $reason_phrase;
		}
		if ( isset( $status ) ) {
			$message .= ' (' . $status . ')';
		}

		return $message;
	}
}
