<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Rapidmail_Handler {

	public static $errors = array();
	public static $client;

	public function __construct( $user, $password ) {

		if ( empty( $user ) || empty( $password ) ) {
			return false;
		}

		self::$errors = array(
			409 => __( "Email already exists", 'greyd_forms' ),
		);

		/**
		 * rapidmail APIv3 client
		 *
		 * Official Rapidmail API client library:
		 *
		 * @see https://github.com/rapidmail/rapidmail-apiv3-client-php
		 */
		Helper::include_library(
			array(
				'path' => __DIR__,
				'file' => '/rapidmail-apiv3-client-php/vendor/autoload.php',
				'url'  => 'https://update.greyd.io/libs/rapidmail-apiv3-client-php.zip',
			)
		);

		self::$client = new \Rapidmail\ApiClient\Client( $user, $password );

	}

	public static function get_lists() {

		$return      = array();
		$listService = self::$client->recipientlists();

		foreach ( $listService->query() as $list ) {
			// debug($list);
			$return[ $list['id'] ] = $list['name'];
		}
		return $return;
	}

	public static function create_recipient( $args = array(), $send_activationmail = true ) {
		$return = false;

		// return if no email set
		if ( ! isset( $args ) || ! isset( $args['email'] ) ) {
			return false;
		}

		// set vars
		$email     = isset( $args['email'] ) ? substr( $args['email'], 0, 255 ) : null;
		$sendmail  = $send_activationmail ? 'yes' : 'no';
		$service   = self::$client->recipients();
		$recipient = null;

		try {
			$recipient = $service->create(
				// Dataset: Represents the recipient dataset you're creating
				$args,
				// Flags: Configures system behavior, like sending activationmails
				array(
					'send_activationmail' => $sendmail,
				)
			);

		} catch ( \Rapidmail\ApiClient\Exception\ApiClientException $e ) {

			// Catch API client exceptions
			// echo "Exception raised: " . $e->getMessage();
			$error_message = $e->getMessage();
			$error_array   = array();
			preg_match( '/\{.*\}/', $error_message, $error_array );
			$error_array = json_decode( $error_array[0], true );
			$return      = $error_array;
		}

		return $recipient !== null ? true : $return;
	}

	public static function get_recipient( $list = '', $args = '' ) {

		// return if email or list-id are missing
		if ( ! isset( $args ) || ! isset( $args['email'] ) || ! isset( $list ) || empty( $list ) ) {
			return false;
		}

		// 1. Get Recipient
		$id      = null;
		$return  = sprintf( __( "No recipient with the mail address %s found.", 'greyd_forms' ), $args['email'] );
		$service = self::$client->recipients();
		$filter  = array(
			'recipientlist_id' => intval( $list ), // Recipientlist ID MUST be provided
			'email'            => $args['email'],
			'status'           => array( 'new', 'active', 'bounced', 'abused', 'deleted' ),
		);
		try {
			foreach ( $service->query( $filter ) as $recipient ) {
				if ( isset( $recipient['id'] ) ) {
					$id = $recipient['id'];
				}
			}
		} catch ( \Rapidmail\ApiClient\Exception\ApiClientException $e ) {
			// Catch API client exceptions
			$return = $e->getMessage();
		}
		if ( $id !== null ) {
			$return = intval( $id );
		}
		return $return;
	}

	public static function update_recipient( $id = '', $args = '' ) {

		// return if email or id are missing
		if ( ! isset( $args ) || ! isset( $args['email'] ) || ! isset( $id ) || empty( $id ) ) {
			return false;
		}

		$recipient = null;
		if ( isset( $args['recipientlist_id'] ) ) {
			unset( $args['recipientlist_id'] );
		}
		$service = self::$client->recipients();
		try {
			$recipient = $service->update( $id, $args );

		} catch ( \Rapidmail\ApiClient\Exception\ApiClientException $e ) {
			// Catch API client exceptions
			$error_message = $e->getMessage();
			$error_array   = array();
			preg_match( '/\{.*\}/', $error_message, $error_array );
			$error_array = json_decode( $error_array[0], true );
			$return      = $error_array;
		}

		return $recipient !== null ? true : $return;
	}

	public static function delete_recipient( $id = '' ) {
		if ( empty( $id ) ) {
			return false;
		}

		$service  = self::$client->recipients();
		$response = $service->delete( $id );
		return $response;
	}

}
