<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Rapidmail();

class Rapidmail {

	const INTERFACE = 'rapidmail';

	public function __construct() {

		// settings
		$action = 'render_setting_' . self::INTERFACE . '_';
		add_action( $action . 'user', array( $this, 'render_user' ), 10, 2 );
		add_action( $action . 'password', array( $this, 'render_password' ), 10, 2 );
		add_action( $action . 'lists', array( $this, 'render_lists' ), 10, 2 );

		// admin ajax
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
		add_action( 'formhandler_optout_' . self::INTERFACE, array( $this, 'optout' ), 10, 2 );
	}

	/**
	 * render user field
	 */
	public function render_user( $pre = '', $value = null ) {

		$option = 'user';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	/**
	 * render password field
	 */
	public function render_password( $pre = '', $value = null ) {

		$option = 'password';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
		echo "<br><small><a href='" .
			__( 'https://www.rapidmail.de/hilfe/api-zugang-anlegen-und-dokumentation-der-api', 'greyd_forms' ) .
			"' target='_blank'>" .
			__( "How to create an API key", 'greyd_forms' ) .
		'</a></small>';
	}

	/**
	 * render lists input
	 */
	public function render_lists( $pre = '', $value = null ) {

		$option   = 'lists';
		$slug     = $pre . '[' . $option . ']';
		$value    = isset( $value ) ? strval( $value ) : '';
		$lists    = strpos( $value, '{' ) !== false ? json_decode( $value, true ) : $value;
		$settings = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$user     = isset( $settings['user'] ) ? $settings['user'] : '';
		$password = isset( $settings['password'] ) ? $settings['password'] : '';
		// classes
		$not_ready_class = 'hidden';
		$ready_class     = '';
		if ( empty( $user ) || empty( $password ) ) {
			$not_ready_class = '';
			$ready_class     = 'hidden';
		}
		$empty_class = 'empty ' . ( empty( $value ) ? '' : 'hidden' );
		$set_class   = 'set ' . ( empty( $value ) ? 'hidden' : '' );
		$info_icon   = "<span class='dashicons dashicons-info'></span>&nbsp;";

		echo "<div id='" . self::INTERFACE . "'>"; // wrapper

		echo "<input type='hidden' id='$slug' name='$slug' value='$value'>";

		// not ready
		echo "<div class='not_ready $not_ready_class'>";
			echo "<p style='opacity:.5;'>" . $info_icon . __( "Please enter your login details for Rapidmail first.", 'greyd_forms' ) . '</p>';
		echo '</div>';

		// ready
		echo "<div class='ready $ready_class'>";
			// loader
			echo "<span style='display:none'><span class='loader'></span></span>";

			// empty
			echo "<div class='$empty_class'>";
				echo "<p style='opacity:.5;'>" . $info_icon . __( "No lists available yet.", 'greyd_forms' ) . '</p>';
				echo "<span class='button getLists' style='margin-top:10px;'>" . __( "Get lists now", 'greyd_forms' ) . '</span>';
			echo '</div>';

			// set
			echo "<div class='$set_class'>";
				echo "<ul class='input_list'>";
		if ( isset( $lists ) && is_array( $lists ) && count( $lists ) > 0 ) {
			foreach ( (array) $lists as $id => $name ) {
				echo "<li><strong>$name</strong> (ID: $id)</li>";
			}
		}
				echo '</ul><br>';
				echo "<span class='button getLists $set_class' style='margin-top:10px;'>" . __( "Update lists", 'greyd_forms' ) . '</span>';
			echo '</div>';

			// set
			echo "<div class='_error hidden'>";
				echo Helper::render_info_box(
					array(
						'style' => 'red',
						'text'  => '<span class=text>' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) . '</span>',
					)
				);
				echo "<span class='button getLists' style='margin-top:10px;'>" . __( "try again", 'greyd_forms' ) . '</span>';
			echo '</div>';

		echo '</div>';

		echo '</div>'; // wrapper end
	}

	/**
	 * admin script
	 */
	public function load_backend() {

		$page = isset( $_GET['page'] ) ? $_GET['page'] : '';

		if ( $page === 'greyd_settings_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_localize_script(
				self::INTERFACE . '_backend_js',
				'local_' . self::INTERFACE,
				array(
					'ajaxurl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
					'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				)
			);
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}

	/**
	 * handle admin ajax
	 */
	public function ajax( $data ) {

		$user = isset( $data['user'] ) ? $data['user'] : '';
		$pw   = isset( $data['password'] ) ? $data['password'] : '';
		if ( ! empty( $user ) && ! empty( $pw ) ) {

			include_once __DIR__ . '/handle.php';
			$rapidmail = new Rapidmail_Handler( $user, $pw );
			$lists     = $rapidmail::get_lists();
			if ( is_array( $lists ) && count( $lists ) > 0 ) {
				wp_die( 'success::' . json_encode( $lists ) );
			}
		}
		wp_die( 'error::' . __( "Lists could not be retrieved. Check your login details and try again.", 'greyd_forms' ) );
	}

	/**
	 * Send form data to rapidmail API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {
		// do_action('formhandler_error', $postmeta); // debug

		$options = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$user    = isset( $options['user'] ) ? $options['user'] : '';
		$pw      = isset( $options['password'] ) ? $options['password'] : '';
		$lists   = isset( $options['lists'] ) && strpos( $options['lists'], '{' ) !== false ? json_decode( $options['lists'], true ) : '';

		// early exit
		if ( empty( $user ) || empty( $pw ) || empty( $lists ) ) {
			return false;
		}

		// meta
		$interface_data                     = array();
		$list                               = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['recipientlist'] ) ? $postmeta['meta']['recipientlist'] : '';
		$list                               = ! empty( $list ) ? $list : $lists[ array_key_first( $lists ) ];
		$interface_data['recipientlist_id'] = $list;
		$sendmail                           = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['activationmail'] ) ? true : false;
		$name                               = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

		// push fields to array
		$fields = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();
		if ( ! isset( $fields['email'] ) ) {
			return false;
		}
		foreach ( $fields as $key => $val ) {
			// validate the fields
			$value = isset( $formdata[ $val ] ) ? $this->validate_field( $key, $formdata[ $val ] ) : null;
			// add field to array only if validation is successfull
			if ( isset( $value ) && $value !== false ) {
				$interface_data[ $key ] = html_entity_decode( $value );
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// return when email not set
		if ( empty( $interface_data ) || ! isset( $interface_data['email'] ) ) {
			return $name . ': ' . __( "Email is not defined, entry could not be sent.", 'greyd_forms' );
		}

		// save necessary data for optout in entry
		Greyd_Forms_Interface::update_entry_data(
			$entry_id,
			self::INTERFACE,
			array(
				'list'  => $list,
				'email' => $interface_data['email'],
			)
		);

		// debug
		// do_action('formhandler_error', $postmeta);

		include_once __DIR__ . '/handle.php';
		$rapidmail = new Rapidmail_Handler( $user, $pw );
		$response  = $rapidmail::create_recipient( $interface_data, $sendmail );

		if ( $response === true ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was created successfully.", 'greyd_forms' ), 'success' );
		} elseif ( isset( $response['status'] ) && $response['status'] === 'new' ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was created successfully.", 'greyd_forms' ), 'success' );
			$response = true;
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );

			$update = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['update'] ) ? true : false;
			if ( $update && is_array( $response ) && isset( $response['status'] ) && $response['status'] == 409 ) {
				$response = $rapidmail::get_recipient( $list, $interface_data );

				// success: response == id
				if ( is_int( $response ) ) {
					$recipient_id = $response;
					$response     = $rapidmail::update_recipient( $recipient_id, $interface_data );
					if ( $response !== true ) {
						Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
					} else {
						Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was successfully updated.", 'greyd_forms' ), 'success' );
						return true;
					}
				} else {
					Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
				}
			}
		}
		return $response;
	}

	/**
	 * validate each field for rapidmail
	 *
	 * @see https://developer.rapidmail.wiki/documentation.html?urls.primaryName=Recipients#/Recipients/post_recipients
	 */
	public function validate_field( $name, $value ) {
		// general maxlength
		$maxlength = 255;

		// gender
		if ( $name === 'gender' ) {
			$value = preg_replace( '/[^a-z]+/', '', strtolower( $value ) );

			// try to convert common values to 'male' or 'female'
			$male   = array_flip( array( 'mr', 'mr.', 'männlich', 'mann', 'man', 'men', 'm', 'h', 'herr' ) );
			$female = array_flip( array( 'ms', 'mrs', 'weiblich', 'frau', 'woman', 'women', 'mrs.', 'f', 'w' ) );
			$value  = isset( $male[ $value ] ) ? 'male' : ( isset( $female[ $value ] ) ? 'female' : false );

			// ...otherwise return
			if ( ! $value ) {
				return false;
			}
		}
		// zip code
		elseif ( $name === 'zip' ) {
			$maxlength = 7;
			$value     = preg_replace( '/[^\d]+/', '', $value );
		}
		// extra fields
		elseif ( preg_match( '/extra\d+/', $name ) ) {
			$maxlength = 100;
		}
		// extra big fields
		elseif ( preg_match( '/extrabig\d+/', $name ) ) {
			$maxlength = 65535;
		}

		// maxlength validation
		$value = strval( $value );
		if ( strlen( $value ) > $maxlength ) {
			$value = substr( $value, 0, $maxlength );
		}

		return $value;
	}

	/**
	 * format error from rapidmail response
	 */
	public static function get_error( $response ) {
		if ( ! isset( $response ) ) {
			return false;
		}
		$message = '';
		if ( isset( $response['title'] ) ) {
			$message = $response['title'];
		}
		if ( isset( $response['status'] ) ) {
			$message .= ' (' . $response['status'] . ')';
		}
		if ( isset( $response['detail'] ) ) {
			$message .= ': ' . $response['detail'] . '.';
		}
		return $message;

	}

	/**
	 * handle opt out
	 */
	public function optout( $entry_id, $meta ) {
		if ( empty( $entry_id ) || empty( $meta ) || ! is_array( $meta ) ) {
			return false;
		}

		$list = isset( $meta['list'] ) ? $meta['list'] : '';
		if ( empty( $list ) ) {
			return false;
		}

		$options = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$user    = isset( $options['user'] ) ? $options['user'] : '';
		$pw      = isset( $options['password'] ) ? $options['password'] : '';

		if ( empty( $user ) || empty( $pw ) ) {
			return false;
		}

		include_once __DIR__ . '/handle.php';
		$rapidmail = new Rapidmail_Handler( $user, $pw );

		$email = isset( $meta['email'] ) ? $meta['email'] : '';
		if ( empty( $email ) ) {
			return false;
		}

		$response = $rapidmail::get_recipient( $list, array( 'email' => $email ) );

		// success: response == id
		if ( is_int( $response ) ) {
			$recipient_id = $response;
			$name         = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );
			$response     = $rapidmail::delete_recipient( $recipient_id );
			if ( $response !== true ) {
				Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
			} else {
				Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient has been deleted.", 'greyd_forms' ), 'success', false );
			}
		} else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
		}

	}

}
