<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Salesforce();

class Salesforce {

	const INTERFACE = 'salesforce';

	public function __construct() {

		// settings
		add_action( 'render_setting_' . self::INTERFACE . '_organisation_id', array( $this, 'render_organisation_id' ), 10, 2 );

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );

		// backward compatibility (convert & delete old values)
		add_filter( 'tp_forms_interface_settings', array( $this, 'get_old_settings' ) );
		add_filter( 'tp_forms_interface_meta', array( $this, 'get_old_meta' ), 10, 2 );
		add_action( 'tp_forms_interface_meta_save', array( $this, 'delete_old_meta' ) );
		add_action( 'tp_forms_interface_entry', array( $this, 'convert_old_entry' ) );
	}

	/**
	 * Render Setting: Salesforce Org-ID
	 */
	public function render_organisation_id( $pre = '', $value = null ) {

		$option = 'organisation_id';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? $value : esc_attr( get_option( $slug, '' ) );

		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
		echo "<p style='max-width:35em;margin-left:2px;'><small>" . __( "Here's how to find your Salesforce Organisation ID:", 'greyd_forms' ) .
			 "&nbsp;<a href='https://help.salesforce.com/articleView?id=000325251&type=1&mode=1' target='_blank'>help.salesforce.com</a></small></p>";
	}

	/**
	 * Send form data to salesforce web-to-lead API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		$oid = Greyd_Forms_Interface::get_option( self::INTERFACE, 'organisation_id' );
		if ( empty( $oid ) ) {
			return false; // early exit
		}

		// meta
		$lead_source = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['lead_source'] ) ? $postmeta['meta']['lead_source'] : '';
		$lead_descr  = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['lead_descr'] ) ? $postmeta['meta']['lead_descr'] : '';
		// defaults
		if ( empty( $lead_source ) || empty( $lead_descr ) ) {
			$default = Greyd_Forms_Interface::get_config( self::INTERFACE, 'metabox', 'meta' );
			if ( isset( $default['fields'] ) ) {
				if ( empty( $lead_source ) ) {
					$lead_source = isset( $default['fields']['lead_source']['value'] ) ? $default['fields']['lead_source']['value'] : '';
				}
				if ( empty( $lead_descr ) ) {
					$lead_descr = isset( $default['fields']['lead_descr']['value'] ) ? $default['fields']['lead_descr']['value'] : '';
				}
			}
		}

		$interface_data = array(
			'oid'         => $oid,
			'retURL'      => get_home_url(),
			'lead_source' => $lead_source,
			'description' => $lead_descr,
			// 'debug' => 1, // debug
			// 'debugEmail' => 'support@greyd.de' // debug
		);

		// push fields to array
		$fields = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();
		$fields = isset( $postmeta['custom'] ) ? array_merge( $fields, (array) $postmeta['custom'] ) : $fields;
		foreach ( $fields as $key => $val ) {
			$value = html_entity_decode( $formdata[ $val ] );
			if ( ! empty( $value ) ) {
				$interface_data[ $key ] = $value;
			}
		}
		// do_action('formhandler_error', $interface_data); // debug

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// return when email not set
		if ( empty( $interface_data ) || ! isset( $interface_data['email'] ) ) {
			return $name . ': ' . __( "Email is not defined, entry could not be sent.", 'greyd_forms' );
		}

		$args = array(
			'method'    => 'POST',
			'headers'   => array(
				'Content-type: application/x-www-form-urlencoded',
			),
			'sslverify' => false,
			'body'      => $interface_data,
		);

		$response = wp_remote_post( 'https://webto.salesforce.com/servlet/servlet.WebToLead?encoding=UTF-8', $args );
		if ( $response !== true ) {
			$response = sprintf( __( "Failed to send to %s: ", 'greyd_forms' ), self::INTERFACE ) . $response;
			Helper::log_entry_state( $entry_id, $response );
		} else {
			// safe to entry meta
			Greyd_Forms_Interface::update_entry_data( $entry_id, self::INTERFACE, true );
			return true;
		}
		return $response;
	}

	/**
	 * =================================================================
	 *                          Convert old settings
	 * =================================================================
	 */

	/**
	 * get old salesforce settings
	 */
	public function get_old_settings( $settings ) {

		// init old settings (dev)
		// update_option('salesforce_organisation_id', '00D2p000000PaYD');
		// delete_option(Greyd_Forms_Interface::SETTING);

		// Backward Compatibility with old salesforce option
		$salesforce_option = get_option( 'salesforce_organisation_id', null );
		if ( ! empty( $salesforce_option ) ) {
			$settings['salesforce'] = array(
				'enable'          => 'on',
				'organisation_id' => $salesforce_option,
			);
			update_option( Greyd_Forms_Interface::SETTING, $settings );
			delete_option( 'salesforce_organisation_id' );
		}
		return $settings;
	}

	/**
	 * get old salesforce postmeta
	 */
	public function get_old_meta( $options, $post_id ) {

		// check if old options are set
		$is_lead = get_post_meta( $post_id, 'is_lead', true ) ? 'on' : 'off';
		if ( $is_lead === 'on' ) {
			$lead_fields   = (array) get_post_meta( $post_id, 'lead_fields', true );
			$custom_fields = (array) get_post_meta( $post_id, 'custom_fields', true );
			$lead_source   = get_post_meta( $post_id, 'lead_source', true );
			$lead_descr    = get_post_meta( $post_id, 'lead_descr', true );
			$old           = array(
				'salesforce' => array(
					'enable' => $is_lead,
					'normal' => $lead_fields,
					'custom' => $custom_fields,
					'meta'   => array(
						'lead_source' => $lead_source,
						'lead_descr'  => $lead_descr,
					),
				),
			);
			$options       = array_replace_recursive( $old, $options );
		}
		return $options;
	}

	/**
	 * Delete old salesforce postmeta
	 */
	public function delete_old_meta( $post_id ) {
		delete_post_meta( $post_id, 'is_lead' );
		delete_post_meta( $post_id, 'lead_fields' );
		delete_post_meta( $post_id, 'custom_fields' );
		delete_post_meta( $post_id, 'lead_source' );
		delete_post_meta( $post_id, 'lead_descr' );
	}

	/**
	 * convert old entry metadata
	 */
	public function convert_old_entry( $post_id ) {
		$tp_salesforce = get_post_meta( $post_id, 'tp_salesforce', true );
		if ( $tp_salesforce ) {
			Greyd_Forms_Interface::update_entry_state( $post_id, self::INTERFACE, true );
			delete_post_meta( $post_id, 'tp_salesforce' );
		}
	}

}
