<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Samdock();

class Samdock {

	const INTERFACE = 'samdock';

	const AUTH_URL          = 'https://auth.samdock.app/login';
	const TOKEN_REFRESH_URL = 'https://auth.samdock.app/token/refresh';
	const BASE_URL          = 'https://samdock.app/';

	public function __construct() {

		// settings
		$action = 'render_setting_' . self::INTERFACE . '_';
		add_action( $action . 'email', array( $this, 'render_email' ), 10, 2 );
		add_action( $action . 'password', array( $this, 'render_password' ), 10, 2 );
		add_action( $action . 'authentication', array( $this, 'render_authentication' ), 10, 2 );

		// // admin ajax
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// // handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
		// add_action( 'formhandler_optout_'.self::INTERFACE, array($this, 'optout'), 10, 2 );
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */

	public function render_email( $pre = '', $value = null ) {

		$option = 'email';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_password( $pre = '', $value = null ) {

		$option = 'password';
		$slug   = $pre . '[' . $option . ']';
		$value  = isset( $value ) ? strval( $value ) : '';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_authentication( $pre = '', $value = null ) {

		// vars
		$option = 'authentication';
		$slug   = $pre . '[' . $option . ']';
		$value  = is_string( $value ) && ! empty( $value ) ? json_decode( $value, true ) : $value;

		$email    = Greyd_Forms_Interface::get_option( self::INTERFACE, 'email' );
		$password = Greyd_Forms_Interface::get_option( self::INTERFACE, 'password' );

		// start wrapper
		echo "<div id='samdock'>";
		// render input

		echo "<span style='display:none'><span class='loader'></span></span>";

		echo "<span class='button get_access_token " . ( isset( $value ) ? 'hidden' : '' ) . "' style='margin-right:10px;'>" . __( "generate access token", 'greyd_forms' ) . '</span>';

		if ( empty( $email ) || empty( $password ) ) {
			$value = '';
			echo "<div class='_info nocredentials'>";
			echo Helper::render_info_box(
				array(
					'style' => 'info',
					'text'  => __( "To generate an access token, you need to enter your email & password first.", 'greyd_forms' ),
				)
			);
			echo '</div>';

		}
		// error
		echo "<div class='_error hidden'>";
		echo Helper::render_info_box(
			array(
				'style' => 'red',
				'text'  => '<span class=text>' . __( "Authentification failed", 'greyd_forms' ) . '</span>',
			)
		);
		echo "<br><span class='button get_access_token retry' style='margin-top:10px;'>" . __( "try again", 'greyd_forms' ) . '</span>';
		echo '</div>';

		// success
		echo "<div class='_success " . ( empty( $value ) ? 'hidden' : '' ) . "'>";
		echo Helper::render_info_box(
			array(
				'style' => 'green',
				'text'  => '<span class=text>' . __( "All tokens generated and saved successfully", 'greyd_forms' ) . '</span>',
			)
		);
		echo '</div>';

		echo "<input type='hidden' id='$slug' name='$slug' value='" . ( $value ? json_encode( $value ) : $value ) . "'>";

		// end wrapper
		echo '</div>';
	}

	/**
	 * =================================================================
	 *                          Admin ajax
	 * =================================================================
	 */

	/**
	 * Enqueue script
	 */
	public function load_backend() {

		$page     = isset( $_GET['page'] ) ? $_GET['page'] : '';
		$posttype = get_post_type() ? get_post_type() : get_current_screen()->post_type;

		if ( $page === 'greyd_settings_forms' || $posttype === 'tp_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_localize_script(
				self::INTERFACE . '_backend_js',
				'local_' . self::INTERFACE,
				array(
					'ajaxurl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
					'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				)
			);
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}

	/**
	 * Handle admin ajax for "generate access token"-button
	 */
	public function ajax( $data ) {

		$email    = isset( $data['email'] ) ? $data['email'] : '';
		$password = isset( $data['password'] ) ? $data['password'] : '';
		$value    = '';
		$saved    = false;

		if ( ! empty( $email ) || ! empty( $password ) ) {
			$response = self::request_access_token( $email, $password );
						// error
			if ( isset( $response ) && isset( $response['error'] ) ) {
				$response = self::get_error( $response );
				wp_die( 'error::' . $response );
			}
			// success
			elseif ( is_array( $response ) ) {
				// set value
				$value = $response;

				// update wp option instantly
				$setting                                      = (array) Greyd_Forms_Interface::get_option();
				$setting[ self::INTERFACE ]                   = isset( $setting[ self::INTERFACE ] ) ? $setting[ self::INTERFACE ] : array();
				$setting[ self::INTERFACE ]['authentication'] = json_encode( $value );
				$saved                                        = update_option( Greyd_Forms_Interface::SETTING, $setting ); // returns bool

				if ( $value && $saved ) {
					wp_die( 'success::' . json_encode( $value ) );
				}
			}
		}

		wp_die( 'error::' . __( "Error generating the access token. Check your login details and try again.", 'greyd_forms' ) );
	}

	/**
	 * =================================================================
	 *                          Handler
	 * =================================================================
	 */

	/**
	 * Send form data to samdock
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		$options        = Greyd_Forms_Interface::get_option( self::INTERFACE );
		$authentication = is_string( $options['authentication'] ) && ! empty( $options['authentication'] ) ? json_decode( $options['authentication'], true ) : $options['authentication'];
		$name           = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );

		// early exit
		if ( empty( $authentication['accessToken'] ) || empty( $authentication['refreshToken'] ) ) {
			return false;
		}

		$interface_data = array();
		$fields         = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();

		if ( ! isset( $fields['email'] ) ) {
			return false;
		}

		foreach ( $fields as $key => $val ) {

			// validate the fields
			$value = isset( $formdata[ $val ] ) ? $this->validate_field( $key, $formdata[ $val ] ) : null;
			// add field to array only if validation is successfull
			if ( isset( $value ) && $value !== false ) {
				$interface_data[ $key ] = html_entity_decode( $value );
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// return when email not set
		if ( empty( $interface_data ) || ! isset( $interface_data['email'] ) ) {
			return $name . ': ' . __( "Email is not defined, entry could not be sent.", 'greyd_forms' );
		}

		$response = self::create_contact( $interface_data, $authentication );

		if ( $response['response']['code'] == 401 && $response['body'] == 'Jwt is expired' ) {
			$response = self::refresh_token( $authentication );
			if ( isset( $response['accessToken'] ) && isset( $response['refreshToken'] ) ) {
				$response = self::create_contact( $interface_data, $response );
			} else {
				Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
			}
		}

		// success
		if ( $response && isset( $response['response']['code'] ) && $response['response']['code'] === 201 ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "Recipient was created successfully.", 'greyd_forms' ), 'success' );
			return true;
		}
		// error
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . self::get_error( $response ) );
		}

		return $response;
	}

	public function validate_field( $name, $value ) {
		// general maxlength
		$maxlength = 255;

		// gender
		if ( $name === 'gender' ) {
			$value = preg_replace( '/[^a-z]+/', '', strtolower( $value ) );

			// try to convert common values to 'male' or 'female'
			$male   = array_flip( array( 'mr', 'mr.', 'männlich', 'mann', 'man', 'men', 'm', 'h', 'herr' ) );
			$female = array_flip( array( 'ms', 'mrs', 'weiblich', 'frau', 'woman', 'women', 'mrs.', 'f', 'w' ) );
			$value  = isset( $male[ $value ] ) ? 'male' : ( isset( $female[ $value ] ) ? 'female' : false );

			// ...otherwise return
			if ( ! $value ) {
				return false;
			}
		}

		// maxlength validation
		$value = strval( $value );
		if ( strlen( $value ) > $maxlength ) {
			$value = substr( $value, 0, $maxlength );
		}

		return $value;
	}

	/**
	 * format error from samdock response
	 */
	public static function get_error( $response ) {

		if ( $response === false ) {
			return __( "Unknown error", 'greyd_forms' );
		}

		$status        = isset( $response['error'] ) ? $response['error'] : __( "Unknown error", 'greyd_forms' );
		$reason_phrase = isset( $response['message'] ) ? $response['message'] :

		$message = '';
		if ( isset( $reason_phrase ) ) {
			$message = $reason_phrase;
		}
		if ( isset( $status ) ) {
			$message .= ' (' . $status . ')';
		}

		return $message;
	}

	/**
	 * =================================================================
	 *                          API Calls
	 * =================================================================
	 */

	/**
	 * Request an access token
	 *
	 * @param string $email     email
	 * @param string $password  password
	 *
	 * @return array            Success: array of access_token & refresh_token.
	 *                          Failure: array of error & message
	 */
	public static function request_access_token( $email, $password ) {

		if ( empty( $email ) || empty( $password ) ) {
			$email    = Greyd_Forms_Interface::get_option( self::INTERFACE, 'email' );
			$password = Greyd_Forms_Interface::get_option( self::INTERFACE, 'password' );
		}

		if ( empty( $email ) || empty( $password ) ) {
			return false;
		}

		// request args
		$post_url = add_query_arg(
			array(),
			self::AUTH_URL
		);

		$post_args = array(
			'method'      => 'POST',
			'headers'     => array(
				'Content-Type' => 'application/x-www-form-urlencoded',
			),
			'redirection' => 5,
			'body'        => array(
				'email'    => $email,
				'password' => $password,
			),
		);

		// send request
		$response = wp_remote_post( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		return (array) $body;
	}


	/**
	 * refresh the access token
	 *
	 * @param array $authentication      array of access_token & refresh_token.
	 *
	 * @return array            Success: array of access_token & refresh_token.
	 *                          Failure: array of error & message
	 */
	public static function refresh_token( $authentication ) {

		if ( empty( $authentication['refreshToken'] ) ) {
			$authentication = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
			$authentication = is_string( $authentication ) && ! empty( $authentication ) ? json_decode( $authentication, true ) : $authentication;
		}

		if ( empty( $authentication['refreshToken'] ) ) {
			return false;
		}

		// request args
		$post_url = add_query_arg(
			array(),
			self::TOKEN_REFRESH_URL
		);

		$post_args = array(
			'method'      => 'POST',
			'headers'     => array(
				'Content-Type' => 'application/x-www-form-urlencoded',
			),
			'redirection' => 5,
			'body'        => array(
				'accessToken'  => $authentication['accessToken'],
				'refreshToken' => $authentication['refreshToken'],
			),
		);

		// send request
		$response = wp_remote_post( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );

		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		if ( isset( $body['refreshToken'] ) && isset( $body['refreshToken'] ) ) {

			// update wp option instantly
			$setting                                      = (array) Greyd_Forms_Interface::get_option();
			$setting[ self::INTERFACE ]                   = isset( $setting[ self::INTERFACE ] ) ? $setting[ self::INTERFACE ] : array();
			$setting[ self::INTERFACE ]['authentication'] = json_encode( $body );
			$saved                                        = update_option( Greyd_Forms_Interface::SETTING, $setting ); // returns bool
		}

		return (array) $body;
	}


	public static function create_contact( $interface_data = array(), $authentication = array() ) {
		$return = false;
		$id     = uniqid();

		// return if no authentication or email is set
		if ( ! isset( $authentication ) || ! isset( $interface_data['email'] ) ) {
			return false;
		}

		// set vars
		$email = isset( $interface_data['email'] ) ? substr( $interface_data['email'], 0, 255 ) : null;

		$interface_data['_id'] = $id;

		// request args
		$post_url = add_query_arg(
			array(),
			self::BASE_URL . 'api/contacts/Persons'
		);

		$post_args = array(
			'method'      => 'POST',
			'headers'     => array(
				'Authorization' => 'Bearer ' . $authentication['accessToken'],
				'Content-Type'  => 'application/json',
			),
			'redirection' => 5,
			'body'        => json_encode( $interface_data ),
		);

		// send request
		$response = wp_remote_post( $post_url, $post_args );

		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		return $response !== null ? $response : $return;
	}
}
