<?php
/**
 * This is the custom Webhook interface inside Greyd.Forms.
 *
 * You can send user facing forms to your own HTTP Endpoint. You can
 * optimize this interface by using different filter- & action-hooks
 * inside and outside this file.
 *
 * Usefull files inside this plugin:
 *
 * @see /inc/handle.php                               General handler for all forms.
 * @see /interfaces/class-greyd-forms-interface.php   Main interface API.
 */

namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Webhook();

class Webhook {

	const INTERFACE = 'webhook';

	public function __construct() {

		// settings
		add_action( 'render_setting_' . self::INTERFACE . '_info', array( $this, 'render_info' ), 10, 2 );

		// admin script
		add_action( 'admin_enqueue_scripts', array( $this, 'load_backend' ) );

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
	}

	/**
	 * Render Setting
	 */
	public function render_info( $pre = '', $value = null ) {

		$option = 'info';
		$slug   = $pre . '[' . $option . ']';
		echo "<input type='hidden' id='$slug' class='regular-text' name='$slug' value='true'>";

		echo "<div style='margin-top:-3em;'>" .
			'<p><i>' .
			__( "You can set an individual interface for each form to send all entries of this form.", 'greyd_forms' ) .
			'</i></p>' .
		'</div>';
	}

	/**
	 * Admin script
	 */
	public function load_backend() {

		$posttype = get_post_type() ? get_post_type() : get_current_screen()->post_type;

		if ( $posttype === 'tp_forms' ) {

			$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
			wp_register_script( self::INTERFACE . '_backend_js', $dir . 'assets/backend.js', 'jquery' );
			wp_enqueue_script( self::INTERFACE . '_backend_js' );
		}
	}

	/**
	 * Send form data to custom webhook
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $formdata      The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed    If true is is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {

		// meta
		$name = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );
		$url  = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['url'] ) ? $postmeta['meta']['url'] : '';
		if ( empty( $url ) ) {
			return false;
		}
		$method = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['method'] ) ? strtoupper( trim( esc_attr( $postmeta['meta']['method'] ) ) ) : 'POST';
		$format = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['format'] ) ? $postmeta['meta']['format'] : 'JSON';

		// save meta in entry
		Greyd_Forms_Interface::update_entry_data(
			$entry_id,
			self::INTERFACE,
			array(
				'url'    => $url,
				'method' => $method,
				'format' => $format,
			)
		);

		// get fields
		$interface_data = array();
		$fields         = isset( $postmeta['custom'] ) ? (array) $postmeta['custom'] : array();

		// modify fields
		foreach ( $fields as $key => $val ) {

			$value = isset( $formdata[ $val ] ) ? html_entity_decode( $formdata[ $val ] ) : '';

			// convert checkbox value to bool
			$value = strpos( $value, '[on]' ) !== false ? true : ( strpos( $value, '[off]' ) !== false ? false : $value );

			if ( $value !== '' ) {
				$interface_data[ $key ] = $value;
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		/**
		 * Send the data to the interface
		 *
		 * @see Greyd\Forms\Interfaces\Greyd_Forms_Interface::send_http_request()
		 */
		$response = Greyd_Forms_Interface::send_http_request( $url, $interface_data, $method, $format, true );

		/**
		 * Do something after the interface answered.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $response        HTTP response of the interface
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$response = apply_filters( 'greyd_forms_interface_response_' . self::INTERFACE, $response, $interface_data, $entry_id, $formdata, $postmeta );

		// success
		if ( $response['success'] ) {
			// logging success
			$text = sprintf( __( "Successfully sent to %s", 'greyd_forms' ), $name . ' "' . $url . '"' ) . ' (' . $response['text'] . ')';
			Helper::log_entry_state( $entry_id, $text, 'success' );
			return true;
		}
		// error
		else {
			// logging error
			$text = sprintf( __( "Failed to send to %s", 'greyd_forms' ), $name . ' "' . $url . '"' ) . ' (' . $response['text'] . ')';
			Helper::log_entry_state( $entry_id, $text );
			// do_action('formhandler_error', $text ); // debug
		}

		// return
		return false;
	}

}
