<?php
namespace Greyd\Forms\Interfaces;

use \Greyd\Forms\Helper;
use \Greyd\Forms\Handler;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Zoom_Webinar();

class Zoom_Webinar {

	const DEBUG = false;

	const INTERFACE = 'zoom_webinar';
	const SHORTCODE = 'vc_zoom_webinar';
	const FIELD     = 'occurrence_id';

	const BASE_URL  = 'https://api.zoom.us/v2';
	const TOKEN_URL = 'https://zoom.us/oauth/token';

	public function __construct() {

		// settings
		add_action( 'render_setting_' . self::INTERFACE . '_client_id', array( $this, 'render_client_id' ), 10, 2 );
		add_action( 'render_setting_' . self::INTERFACE . '_client_secret', array( $this, 'render_client_secret' ), 10, 2 );
		add_action( 'render_setting_' . self::INTERFACE . '_authentication', array( $this, 'render_authentication' ), 10, 2 );
		add_action( 'render_setting_' . self::INTERFACE . '_webinars', array( $this, 'render_webinars' ), 10, 2 );

		// everything else is only loaded, when enabled
		if ( ! Greyd_Forms_Interface::is_interface_enabled( self::INTERFACE ) ) {
			return;
		}

		// handler
		add_filter( 'handle_after_doi_' . self::INTERFACE, array( $this, 'send' ), 10, 4 );
		add_action( 'formhandler_optout_' . self::INTERFACE, array( $this, 'optout' ), 10, 2 );

		// add vc modules
		$this->add_modules();
	}

	/**
	 * =================================================================
	 *                          Options
	 * =================================================================
	 */

	public function render_client_id( $pre = '', $value = null ) {

		$option = 'client_id';
		$slug   = $pre . '[' . $option . ']';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_client_secret( $pre = '', $value = null ) {

		$option = 'client_secret';
		$slug   = $pre . '[' . $option . ']';
		echo "<input type='text' id='$slug' class='regular-text' name='$slug' value='$value'>";
	}

	public function render_authentication( $pre = '', $value = null ) {

		// vars
		$option        = 'authentication';
		$slug          = $pre . '[' . $option . ']';
		$value         = is_string( $value ) && ! empty( $value ) ? json_decode( $value, true ) : $value;
		$saved         = true;
		$auth_code_cur = isset( $value['auth_code'] ) ? $value['auth_code'] : null;
		$auth_code_new = isset( $_GET['code'] ) ? $_GET['code'] : null;
		$info_box      = array();
		$settings_url  = admin_url( 'edit.php?post_type=tp_forms&page=greyd_settings_forms' );
		$client_id     = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_id' );
		$client_secret = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_secret' );
		$help_class    = 'hidden';

		// we don't have all the options yet
		if ( ! $client_id || ! $client_secret ) {
			echo Helper::render_info_box(
				array(
					'style' => 'info',
					'text'  => __( "To access the authentication, you must first enter the Client ID & Client Secret and save your changes.", 'greyd_forms' ),
				)
			);
		}

		// we're just coming from the zoom auth page
		elseif ( $auth_code_new && $auth_code_new !== $auth_code_cur ) {

			$value  = null; // reset value
			$bearer = base64_encode( $client_id . ':' . $client_secret );

			$response = $this->create_access_token( $auth_code_new, $bearer );

			// error
			if ( is_string( $response ) ) {
				echo Helper::render_info_box(
					array(
						'style' => 'red',
						'text'  => $response,
					)
				);
			}
			// success
			elseif ( is_array( $response ) ) {
				// set value
				$response['auth_code'] = $auth_code_new;
				$value                 = $response;

				// update wp option instantly
				$setting                            = (array) Greyd_Forms_Interface::get_option();
				$setting['zoom_webinar']            = isset( $setting['zoom_webinar'] ) ? $setting['zoom_webinar'] : array();
				$setting['zoom_webinar'][ $option ] = json_encode( $value );
				$saved                              = update_option( Greyd_Forms_Interface::SETTING, $setting ); // returns bool
			}
		}

		// value saved
		if ( $value && $saved ) {
			echo Helper::render_info_box(
				array(
					'style' => 'green',
					'text'  => __( "All tokens generated and saved successfully", 'greyd_forms' ),
				)
			);
		}

		// render input
		echo "<input type='hidden' id='$slug' name='$slug' value='" . ( $value ? json_encode( $value ) : $value ) . "'>";

		// debug
		if ( self::DEBUG ) {
			echo "<textarea rows='8' disabled='disabled'>";
			print_r( $value );
			echo '</textarea>';
		}

		// display auth link
		$auth_url = 'https://zoom.us/oauth/authorize?response_type=code&client_id=' . $client_id . '&redirect_uri=' . urlencode( $settings_url );
		echo "<br><a class='button' href='$auth_url' style='margin-right:6px;'>" . __( "generate access token", 'greyd_forms' ) . '</a>';

		echo "<a class='button button-ghost" . ( $help_class === 'hidden' ? '' : 'hidden' ) . "' onclick='greyd.backend.toggleElemByClass(\"helpbox\")' style='cursor:pointer;'>" . __( "Help", 'greyd_forms' ) . '</a>';
		echo "<div class='toggle_helpbox $help_class'>
            <div class='greyd_info_box blue' style='margin-top:12px;'>
                <ol style='margin-top:5px;margin-bottom:0;'>
                    <li>" . sprintf(
    __( "To connect to the Zoom Webinar API, you need to create a %1\$sprivate OAuth app%2\$s in the Zoom \"App Marketplace\".", 'greyd_forms' ),
    "<a href='https://marketplace.zoom.us/develop/create' target='_blank'>",
		'</a>'
		) . '</li>
                    <li>' . __( "There you will find the Client ID and Client Secret under “App Credentials”.", 'greyd_forms' ) . '</li>
                    <li>' . sprintf(
						__( "In order for the connection to work properly, you have to enter the following redirect URL in the app: %s", 'greyd_forms' ),
						"<strong>$settings_url</strong>"
					) . '</li>
                    <li>' . __( "In addition, you have to create the authorization „View and manage your webinars“ under „Scopes“.", 'greyd_forms' ) . '</li>
                </ol>
            </div>
        </div>';

	}

	public function render_webinars( $pre = '', $value = null ) {

		$option       = 'webinars';
		$slug         = $pre . '[' . $option . ']';
		$webinars     = ! empty( $value ) ? json_decode( $value, true ) : array();
		$auth         = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
		$auth_arr     = $auth ? json_decode( $auth, true ) : array();
		$access_token = isset( $auth_arr['access_token'] ) ? $auth_arr['access_token'] : null;

		// no access token yet
		if ( ! $access_token ) {
			echo Helper::render_info_box(
				array(
					'style' => 'info',
					'text'  => __( "In order to access the webinars, you must first generate an access token.", 'greyd_forms' ),
				)
			);
		} else {
			$response = $this->list_webinars( $access_token );
			$webinars = array();
			if ( is_string( $response ) ) {
				echo Helper::render_info_box(
					array(
						'style' => 'red',
						'text'  => $response,
					)
				);
			} elseif ( is_array( $response ) && count( $response ) > 0 ) {
				foreach ( $response as $webinar ) {
					$webinars[ $webinar['id'] ] = $webinar['topic'];
				}
			}
			$_value = json_encode( $webinars );

			// update wp option instantly
			if ( $_value !== $value ) {
				$value                              = $_value;
				$setting                            = (array) Greyd_Forms_Interface::get_option();
				$setting['zoom_webinar']            = isset( $setting['zoom_webinar'] ) ? $setting['zoom_webinar'] : array();
				$setting['zoom_webinar'][ $option ] = $value;
				$saved                              = update_option( Greyd_Forms_Interface::SETTING, $setting ); // returns bool
			}

			// display webinars
			if ( ! is_array( $webinars ) || count( $webinars ) === 0 ) {
				echo Helper::render_info_box(
					array(
						'style' => 'info',
						'text'  => __( "No webinars found.", 'greyd_forms' ),
					)
				);
			} else {
				echo '<div>';
					echo "<ul class='input_list'>";
				foreach ( (array) $webinars as $id => $name ) {
					echo "<li><strong>$name</strong> (ID: $id)</li>";
				}
					echo '</ul><br>';
				echo '</div>';
			}
		}

		// render input
		echo "<input type='hidden' id='$slug' name='$slug' value='$value'>";

		// debug
		if ( self::DEBUG ) {
			echo "<textarea rows='3' disabled='disabled'>";
			print_r( $webinars );
			echo '</textarea>';
		}
	}

	/**
	 * =================================================================
	 *                          API Calls
	 * =================================================================
	 */

	/**
	 * Create an access token
	 *
	 * @param string $auth_code authorization code from zoom url param 'code'
	 * @param string $bearer    base64 encoded string of client_id and client_secret
	 *
	 * @return array|string     Success: array of access_token & refresh_token.
	 *                          Failure: string of error message
	 */
	public function create_access_token( $auth_code, $bearer = '' ) {

		$settings_url = admin_url( 'edit.php?post_type=tp_forms&page=greyd_settings_forms' );

		if ( empty( $bearer ) ) {
			$client_id     = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_id' );
			$client_secret = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_secret' );
			$bearer        = base64_encode( $client_id . ':' . $client_secret );
		}

		// request args
		$post_url  = add_query_arg(
			array(
				'grant_type'   => 'authorization_code',
				'code'         => $auth_code,
				'redirect_uri' => urlencode( $settings_url ),
			),
			self::TOKEN_URL
		);
		$post_args = array(
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Basic ' . $bearer,
			),
		);

		// send request
		$response = wp_remote_post( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		return (array) $body;
	}

	/**
	 * List all users webinars
	 *
	 * @param string $refresh_token     defaults to the current saved one.
	 * @param string $bearer            base64 encoded string of client_id and client_secret
	 *
	 * @return array|string     Success: array of access_token & refresh_token.
	 *                          Failure: string of error message
	 */
	public function refresh_access_token( $refresh_token = '', $bearer = '' ) {

		$auth     = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
		$auth_arr = $auth ? json_decode( $auth, true ) : array();

		if ( empty( $refresh_token ) ) {
			$refresh_token = isset( $auth_arr['refresh_token'] ) ? $auth_arr['refresh_token'] : null;
		}

		if ( empty( $bearer ) ) {
			$client_id     = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_id' );
			$client_secret = Greyd_Forms_Interface::get_option( self::INTERFACE, 'client_secret' );
			$bearer        = base64_encode( $client_id . ':' . $client_secret );
		}

		// request args
		$post_url  = add_query_arg(
			array(
				'grant_type'    => 'refresh_token',
				'refresh_token' => $refresh_token,
			),
			self::TOKEN_URL
		);
		$post_args = array(
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Basic ' . $bearer,
			),
		);

		// send request
		$response = wp_remote_post( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		// save in database
		else {
			// set new tokens
			$auth_arr['access_token']  = isset( $body['access_token'] ) ? $body['access_token'] : $auth_arr['access_token'];
			$auth_arr['refresh_token'] = isset( $body['refresh_token'] ) ? $body['refresh_token'] : $auth_arr['refresh_token'];

			// update wp option
			$setting                                   = (array) Greyd_Forms_Interface::get_option();
			$setting['zoom_webinar']                   = isset( $setting['zoom_webinar'] ) ? $setting['zoom_webinar'] : array();
			$setting['zoom_webinar']['authentication'] = json_encode( $auth_arr );
			$saved                                     = update_option( Greyd_Forms_Interface::SETTING, $setting );
			if ( ! $saved ) {
				return "Error updating option '" . Greyd_Forms_Interface::OPTION . "'.";
			}
		}

		return (array) $body;
	}

	/**
	 * List all users webinars
	 *
	 * @param string $access_token      defaults to the current saved one.
	 * @param string $refresh_token     defaults to the current saved one.
	 *
	 * @return array|string     Success: array of webinars.
	 *                          Failure: string of error message
	 */
	public function list_webinars( $access_token = '', $refresh_token = '' ) {

		$auth     = null;
		$auth_arr = array();

		if ( empty( $access_token ) ) {
			$auth         = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
			$auth_arr     = $auth ? json_decode( $auth, true ) : array();
			$access_token = isset( $auth_arr['access_token'] ) ? $auth_arr['access_token'] : null;
		}

		// request args
		$post_url  = self::BASE_URL . '/users/me/webinars';
		$post_args = array(
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Bearer ' . $access_token,
			),
		);

		// send request
		$response = wp_remote_get( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		// invalid access token
		elseif ( isset( $body['code'] ) && $body['code'] == 124 ) {

			if ( empty( $refresh_token ) && isset( $auth_arr['refresh_token'] ) ) {
				$refresh_token = $auth_arr['refresh_token'];
			}

			// refresh access token
			$result = $this->refresh_access_token( $refresh_token );

			// success
			if ( isset( $result['access_token'] ) && isset( $result['refresh_token'] ) ) {
				// recall this function
				return $this->list_webinars( $result['access_token'], $result['refresh_token'] );
			} else {
				return $result;
			}
		}

		// webinars found
		elseif ( isset( $body['webinars'] ) ) {
			return (array) $body['webinars'];
		}

		return (array) $body;
	}

	/**
	 * Add registrant to webinar
	 *
	 * @param string $access_token      Defaults to the current saved one.
	 * @param string $webinar           Webinar ID.
	 * @param array  $userdata          The user data to be sent to the API.
	 * @param string $occurrance_id     ID of an occurrance of the webinar.
	 *
	 * @return array|string     Success: array of registrant data.
	 *                          Failure: string of error message
	 */
	public function add_webinar_registrant( $access_token, $webinar, $userdata, $occurrance_id ) {

		// request args
		$post_url  = self::BASE_URL . '/webinars/' . $webinar . '/registrants';
		$post_args = array(
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => json_encode( $userdata ),
		);

		// add occurrence ID
		if ( ! empty( $occurrance_id ) ) {
			$post_url = add_query_arg( 'occurrence_ids', $occurrance_id, $post_url );
		}

		// send request
		$response = wp_remote_post( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		// invalid access token
		elseif ( isset( $body['code'] ) && $body['code'] == 124 ) {

			// refresh access token
			$result = $this->refresh_access_token();

			// success
			if ( isset( $result['access_token'] ) && isset( $result['refresh_token'] ) ) {
				// recall this function
				return $this->add_webinar_registrant( $result['access_token'], $webinar, $userdata );
			}
			// error
			else {
				return $result;
			}
		}

		// ...otherwise not successfull
		elseif ( ! isset( $body['registrant_id'] ) ) {
			return isset( $body['message'] ) ? strval( $body['message'] ) : sprintf( __( "Errors occurred when the user registered for the webinar (Answer: %s)", 'greyd_forms' ), json_encode( $body ) );
		}

		return (array) $body;
	}

	/**
	 * Update a webinar registrant status
	 *
	 * @param string $access_token      Defaults to the current saved one.
	 * @param string $webinar           Webinar ID.
	 * @param array  $registrant        Must contain 'registrant_id' and 'email'.
	 * @param string $status            Allowed values: approved, cancel, deny.
	 *
	 * @return array|string     Success: array of registrant data.
	 *                          Failure: string of error message
	 */
	public function update_registrant_status( $access_token, $webinar, $registrant, $status = 'cancel' ) {

		// request args
		$post_url  = self::BASE_URL . '/webinars/' . $webinar . '/registrants/status';
		$post_args = array(
			'method'  => 'PUT',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => json_encode(
				array(
					'action'      => strval( $status ),
					'registrants' => array(
						array(
							'id'    => ( isset( $registrant['registrant_id'] ) ? $registrant['registrant_id'] : null ),
							'email' => ( isset( $registrant['email'] ) ? $registrant['email'] : null ),
						),
					),
				)
			),
		);

		// send request
		$response = wp_remote_request( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		// invalid access token
		elseif ( isset( $body['code'] ) && $body['code'] == 124 ) {

			// refresh access token
			$result = $this->refresh_access_token();

			// success
			if ( isset( $result['access_token'] ) && isset( $result['refresh_token'] ) ) {
				// recall this function
				return $this->update_registrant_status( $result['access_token'], $webinar, $registrant, $status );
			}
			// error
			else {
				return $result;
			}
		}

		return (array) $body;
	}

	/**
	 * Get the webinar object
	 *
	 * @param string $webinar_id
	 */
	public function get_webinar( $webinar_id ) {

		$auth         = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
		$auth_arr     = $auth ? json_decode( $auth, true ) : array();
		$access_token = isset( $auth_arr['access_token'] ) ? $auth_arr['access_token'] : null;

		// request args
		$post_url  = self::BASE_URL . '/webinars/' . $webinar_id;
		$post_args = array(
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Accept'        => '*/*',
				'Authorization' => 'Bearer ' . $access_token,
			),
		);

		// send request
		$response = wp_remote_get( $post_url, $post_args );
		if ( is_wp_error( $response ) ) {
			return $response->get_error_message();
		}

		// get response body
		$body = wp_remote_retrieve_body( $response );
		if ( is_wp_error( $body ) ) {
			return $body->get_error_message();
		}

		// decode body
		$body = json_decode( $body, true );

		// error from zoom api
		if ( isset( $body['error'] ) ) {
			return esc_attr( $body['error'] ) . ': ' . esc_attr( $body['reason'] );
		}

		// invalid access token
		elseif ( isset( $body['code'] ) && $body['code'] == 124 ) {

			if ( empty( $refresh_token ) && isset( $auth_arr['refresh_token'] ) ) {
				$refresh_token = $auth_arr['refresh_token'];
			}

			// refresh access token
			$result = $this->refresh_access_token( $refresh_token );

			// success
			if ( isset( $result['access_token'] ) && isset( $result['refresh_token'] ) ) {
				// recall this function
				return $this->list_webinars( $result['access_token'], $result['refresh_token'] );
			} else {
				return $result;
			}
		}

		return (array) $body;
	}

	/**
	 * =================================================================
	 *                          Handle form
	 * =================================================================
	 */

	/**
	 * Send form data to zoom webinar API
	 *
	 * @filter handle_after_doi_{{interface}}
	 *
	 * @param mixed  $response      The response to return.
	 * @param string $entry_id      The Post ID of the entry.
	 * @param array  $form_data     The user data, keyed by the field name.
	 * @param array  $postmeta      The form post_meta for this specific interface.
	 *
	 * @return mixed                If true is returned, a default success message is logged to the entry.
	 */
	public function send( $response, $entry_id, $formdata, $postmeta ) {
		// do_action( 'formhandler_error', $formdata ); // debug

		$name         = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );
		$auth         = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
		$auth_arr     = $auth ? json_decode( $auth, true ) : array();
		$access_token = isset( $auth_arr['access_token'] ) ? $auth_arr['access_token'] : null;

		// early exit
		if ( empty( $auth ) || empty( $access_token ) ) {
			$this->error( $entry_id, __( "No access token was found.", 'greyd_forms' ) );
		}

		// get webinar
		$webinar = isset( $postmeta['meta'] ) && isset( $postmeta['meta']['webinar'] ) ? $postmeta['meta']['webinar'] : '';
		if ( empty( $webinar ) ) {
			$this->error( $entry_id, __( "No webinar to be filled was selected.", 'greyd_forms' ) );
		}

		// get field setup
		$fields = isset( $postmeta['normal'] ) ? (array) $postmeta['normal'] : array();

		// get registrant
		$interface_data = array();
		foreach ( $fields as $key => $val ) {
			if ( isset( $formdata[ $val ] ) && ! empty( $formdata[ $val ] ) ) {

				$value = html_entity_decode( strval( $formdata[ $val ] ) );
				// cut maxlength
				$maxlength = $key === 'email' ? 128 : ( $key === 'first_name' || $key === 'last_name' ? 64 : 255 );
				if ( strlen( $value ) > $maxlength ) {
					$value = substr( $value, 0, $maxlength );
				}

				$interface_data[ $key ] = $value;
			}
		}

		/**
		 * Filter the data send to the interface.
		 *
		 * @filter greyd_forms_interface_response_{{interface}}
		 *
		 * @param array  $interface_data  Data send to the interface.
		 * @param int    $entry_id        WP_Post ID of the entry.
		 * @param array  $fields          Validated form data.
		 * @param array  $postmeta        The form post_meta for this specific interface.
		 */
		$interface_data = apply_filters( 'greyd_forms_interface_data_' . self::INTERFACE, $interface_data, $entry_id, $formdata, $postmeta );

		// check required fields
		if ( ! isset( $interface_data['email'] ) || ! isset( $interface_data['first_name'] ) ) {
			$this->error( $entry_id, __( "In order to register a webinar participant, an email address and a first name must always be given.", 'greyd_forms' ) );
		}

		// get occurrence id
		$occurrance_id = isset( $formdata[ self::FIELD ] ) ? $formdata[ self::FIELD ] : null;

		// send to zoom API
		$response = $this->add_webinar_registrant( $access_token, $webinar, $interface_data, $occurrance_id );

		// error
		if ( is_string( $response ) ) {
			$this->error( $entry_id, $response );
			return false;
		}

		// success
		$registrant_id = isset( $response['registrant_id'] ) ? $response['registrant_id'] : null;

		// save necessary data for optout in entry
		Greyd_Forms_Interface::update_entry_data(
			$entry_id,
			self::INTERFACE,
			array(
				'webinar'       => $webinar,
				'registrant_id' => $registrant_id,
				'email'         => $interface_data['email'],
			)
		);

		// log success
		$msg = sprintf(
			__( "The user was successfully registered for the webinar (Webinar ID: %1\$s | Registrant ID: %2\$s).", 'greyd_forms' ),
			$webinar,
			$registrant_id
		);
		Helper::log_entry_state( $entry_id, $name . ': ' . $msg, 'success' );

		return true;
	}

	/**
	 * Update registrant state
	 *
	 * @action formhandler_optout_{{interface}}
	 *
	 * @param string $entry_id  The Post ID of the entry.
	 * @param array  $meta       The entry post_meta of this specific interface.
	 */
	public function optout( $entry_id, $meta ) {
		if ( empty( $entry_id ) || empty( $meta ) || ! is_array( $meta ) ) {
			return false;
		}

		$name         = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );
		$auth         = Greyd_Forms_Interface::get_option( self::INTERFACE, 'authentication' );
		$auth_arr     = $auth ? json_decode( $auth, true ) : array();
		$access_token = isset( $auth_arr['access_token'] ) ? $auth_arr['access_token'] : null;

		// early exit
		if ( empty( $auth ) || empty( $access_token ) ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "No access token was found.", 'greyd_forms' ) );
			return false;
		}

		// get webinar
		$webinar = isset( $meta['webinar'] ) ? $meta['webinar'] : null;
		if ( empty( $webinar ) ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "No webinar ID was found.", 'greyd_forms' ) );
			return false;
		}

		$response = $this->update_registrant_status( $access_token, $webinar, $meta );

		// error
		if ( is_string( $response ) ) {
			Helper::log_entry_state( $entry_id, $name . ': ' . $response );
			return false;
		}

		// success
		else {
			Helper::log_entry_state( $entry_id, $name . ': ' . __( "The user was successfully deregistered from the webinar.", 'greyd_forms' ), 'success', false );
		}

		return true;
	}

	/**
	 * Error handling (front- & backend)
	 *
	 * @param string $entry_id  Post ID of the entry.
	 * @param string $reason    reason for the error.
	 */
	public function error( $entry_id, $reason ) {

		// log error to entry
		$name = Greyd_Forms_Interface::get_config( self::INTERFACE, 'name' );
		Helper::log_entry_state( $entry_id, $name . ': ' . $reason );

		// get admin message
		$admin_msg = '<strong>' . __( "A user tried to register for the Zoom webinar. The following error occurred:", 'greyd_forms' ) . '</strong>';
		if ( ! empty( $reason ) ) {
			$admin_msg .= '<br><br>' . $reason;
		}

		// send mail to admin
		$form_id = get_post_meta( $entry_id, 'tp_form_id', true );
		if ( ! empty( $form_id ) ) {
			\Greyd\Forms\Handler::send_admin_error_mail( $form_id, $admin_msg );
		}

		// get frontend message
		$front_msg = '<strong>' . __( "The registration for the webinar could not be carried out. Your request has been saved and the admin has been contacted.", 'greyd_forms' ) . '</strong>';
		if ( ! empty( $reason ) ) {
			$front_msg .= ' (' . __( "Reason", 'greyd_forms' ) . ": $reason)";
		}

		// display to frontend
		do_action( 'formhandler_error', $front_msg );
	}

	/**
	 * =================================================================
	 *                          VC modules
	 * =================================================================
	 */
	public function add_modules() {

		// register shortcodes
		add_shortcode( self::SHORTCODE . '_dropdown', array( $this, 'render_zoom_webinar_dropdown' ) );
		add_shortcode( self::SHORTCODE . '_infos', array( $this, 'render_zoom_webinar_infos' ) );

		// allow dropdown field in backend validation
		add_filter( 'formhandler_allow_fields', array( $this, 'allow_dropdown_field' ), 10, 2 );

		// display webinar infos via ajax (frontend)
		if ( ! is_admin() ) {
			$this->load_frontend_script();
		}
		add_action( 'tp_forms_interface_ajax_' . self::INTERFACE, array( $this, 'ajax' ) );

		// add visual composer stuff
		if ( is_admin() && \vc\helper::is_active_plugin( 'js_composer/js_composer.php' ) ) {
			add_action( 'vc_before_init', array( $this, 'add_zoom_webinar_vc' ), 99 );
			add_action( 'admin_notices', array( $this, 'remove_vc_modules' ) );
			add_action( 'vc_frontend_editor_hook_load_edit', array( $this, 'remove_vc_modules' ) );
		}
	}

	/**
	 * Holds all webinars keyed by post_id
	 */
	private $webinars = array();

	/**
	 * Get the current webinar
	 *
	 * @param int $post_id  defaults to current shortcode post_id
	 *
	 * @return mixed        Array on success, null on failure.
	 */
	public function get_webinar_by_post_id( $post_id = null ) {

		if ( empty( $post_id ) ) {
			$post_id = \Greyd\Forms\shortcode::$post_id;
			if ( empty( $post_id ) ) {
				return null;
			}
		}

		if ( isset( $this->webinars[ $post_id ] ) ) {
			$webinar = $this->webinars[ $post_id ];
		} else {
			$meta       = Greyd_Forms_Interface::get_meta( $post_id, self::INTERFACE );
			$enable     = isset( $meta['enable'] ) ? $meta['enable'] : null;
			$webinar_id = isset( $meta['meta'] ) && isset( $meta['meta']['webinar'] ) ? $meta['meta']['webinar'] : null;
			if ( empty( $meta ) || empty( $webinar_id ) || $enable != 'on' ) {
				return;
			}

			$webinar = $this->get_webinar( $webinar_id );
			if ( ! is_array( $webinar ) ) {
				return null;
			}

			$this->webinars[ $post_id ] = $webinar;
		}
		return $webinar;
	}

	/**
	 * Render webinar dropdown
	 *
	 * @param array  $atts      Shortcode attributes.
	 * @param string $content   Content within the shortcode brackets.
	 */
	public function render_zoom_webinar_dropdown( $atts, $content ) {

		// display
		$name    = self::FIELD;
		$label   = isset( $atts['label'] ) ? $atts['label'] : null;
		$title   = \Greyd\Forms\shortcode::$placeholder['title_select'];
		$options = array();

		$webinar = $this->get_webinar_by_post_id();

		// only recurring webinars
		if ( is_array( $webinar ) ) {
			$occurrences = isset( $webinar['occurrences'] ) ? $webinar['occurrences'] : null;
			if ( $webinar['type'] == 9 && is_array( $occurrences ) && count( $occurrences ) > 0 ) {

				foreach ( $occurrences as $occurrence ) {

					if ( $occurrence['status'] === 'available' ) {
						$options[ $occurrence['occurrence_id'] ] = $this->get_frontend_time( $occurrence['start_time'], $webinar['timezone'] );
					}
				}
			}
		}

		// no options set
		$after = '';
		if ( count( $options ) === 0 ) {
			$options = array( '' => __( "no dates available", 'greyd_forms' ) );
			$after   = "<a href='.' title='" . __( "Reload", 'greyd_forms' ) . "'>" . __( "Reload", 'greyd_forms' ) . '</a>';
		}

		// Render
		ob_start();

		echo "<div class='input-outer-wrapper'>
            <div class='input-wrapper custom-select'>";

		if ( ! empty( $label ) ) {
			echo "<label class='label' for='$name'>$label</label><br>";
		}

		echo "<select name='$name' title='$title' >";
		foreach ( $options as $val => $display ) {
			echo "<option value='$val'>$display</option>";
		}
		echo '</select>';

		echo '</div></div>';

		echo $after;

		$return = ob_get_contents();
		ob_end_clean();
		return $return;
	}

	/**
	 * Allow occurrence_id field
	 *
	 * @filter 'formhandler_allow_fields'
	 *
	 * @param array $fields     Inputs to be allowed, keyed by their name.
	 * @param int   $post_id    Post ID of the form.
	 */
	public function allow_dropdown_field( $fields, $post_id ) {

		$fields[ self::FIELD ] = true;

		return $fields;
	}

	/**
	 * Holds the info placeholder tags
	 */
	private $info_tags = array(
		'topic',
		'agenda',
		'type',
		'time',
		'duration',
		'join_link',
		'password',
		'host_email',
	);

	/**
	 * Render webinar infos
	 *
	 * @param array  $atts      Shortcode attributes.
	 * @param string $content   Content within the shortcode brackets.
	 */
	public function render_zoom_webinar_infos( $atts, $content ) {

		$_content = $content;
		foreach ( $this->info_tags as $tag ) {
			$_content = preg_replace( '/_' . $tag . '_/', '<i>' . __( "loading…", 'greyd_forms' ) . '</i>', $_content );
		}
		$post_id = \Greyd\Forms\shortcode::$post_id;

		return "<div class='webinar_infos' data-post_id='$post_id' data-content=" . urlencode( $content ) . ">$_content</div>";
	}

	/**
	 * Load frontend.js
	 *
	 * Used to display webinar infos via ajax
	 */
	public function load_frontend_script() {

		if ( ! Greyd_Forms_Interface::is_interface_enabled( self::INTERFACE ) ) {
			return false;
		}

		$dir = plugin_dir_url( 'greyd_tp_forms/init.php' ) . '/interfaces/' . self::INTERFACE . '/';
		wp_register_script( self::INTERFACE . '_frontend_js', $dir . 'assets/frontend.js', array( 'jquery' ) );
		wp_localize_script(
			self::INTERFACE . '_frontend_js',
			'local_' . self::INTERFACE,
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( Greyd_Forms_Interface::AJAX_ACTION ),
				'action'  => Greyd_Forms_Interface::AJAX_ACTION,
				'error'   => __( "Failed to load the webinar information.", 'greyd_forms' ),
			)
		);
		wp_enqueue_script( self::INTERFACE . '_frontend_js' );
	}

	/**
	 * Handle the ajax from frontend.js
	 *
	 * @param array $data    $_POST['data']
	 *
	 * @return $content      Full html of the module content.
	 */
	public function ajax( $data ) {

		$post_id = isset( $data['post_id'] ) ? intval( $data['post_id'] ) : null;
		$content = isset( $data['content'] ) ? urldecode( $data['content'] ) : null;

		if ( empty( $post_id ) || empty( $content ) ) {
			wp_die( 'return::' . __( "Failed to load the webinar information.", 'greyd_forms' ) );
		} else {
			wp_die( 'return::' . $this->get_zoom_webinar_info_content( $post_id, $content ) );
		}
	}

	/**
	 * Get webinar info content
	 *
	 * @param int    $post_id    WP Post ID.
	 * @param string $content   Content to be modified.
	 *
	 * @return $content          Full html of the module content.
	 */
	public function get_zoom_webinar_info_content( $post_id, $content ) {

		$webinar = $this->get_webinar_by_post_id( $post_id );
		if ( ! is_array( $webinar ) || count( $webinar ) === 0 ) {
			return;
		}

		// vars
		$types = array(
			5 => __( "normal webinar", 'greyd_forms' ),
			6 => __( "recurring webinar without a fixed time", 'greyd_forms' ),
			9 => __( "recurring webinar with a fixed time", 'greyd_forms' ),
		);

		// replace tags
		foreach ( $this->info_tags as $tag ) {
			$value = '';
			if ( isset( $webinar[ $tag ] ) ) {
				$value = $webinar[ $tag ];
				if ( $tag == 'type' ) {
					$value = $types[ $value ];
				} elseif ( $tag == 'duration' ) {
					$hours   = strval( floor( $value / 60 ) );
					$minutes = strval( $value % 60 );
					$value   = $hours . ':' . ( strlen( $minutes ) < 2 ? ( strlen( $minutes ) < 1 ? '00' : '0' . $minutes ) : $minutes ) . 'h';
				}
			} elseif ( $tag == 'join_link' ) {
				$value = "<a href='" . $webinar['join_url'] . "' target='_blank'>" . __( "join meeting", 'greyd_forms' ) . '</a>';
			} elseif ( $tag == 'time' ) {
				// 5: start time
				// 6: none
				// 7: recurrence info
				$type = isset( $webinar['type'] ) ? (int) $webinar['type'] : 5;
				if ( $type == 5 ) {
					if ( isset( $webinar['start_time'] ) ) {
						$value = $this->get_frontend_time( $webinar['start_time'], $webinar['timezone'] );
					}
				} elseif ( $type == 6 ) {
					$value = $types[ $type ];
				} elseif ( $type == 9 ) {
					$recurrence = isset( $webinar['recurrence'] ) ? $webinar['recurrence'] : null;
					if ( is_array( $recurrence ) ) {

						$rec_type = isset( $recurrence['type'] ) ? (int) $recurrence['type'] : 1; // 1: Daily, 2: Weekly, 3: Monthly
						$interval = isset( $recurrence['repeat_interval'] ) ? (int) $recurrence['repeat_interval'] : 1; // interval of type
						$daily    = array(
							1 => __( "Monday", 'greyd_forms' ),
							2 => __( "Tuesday", 'greyd_forms' ),
							3 => __( "Wednesday", 'greyd_forms' ),
							4 => __( "Thursday", 'greyd_forms' ),
							5 => __( "Friday", 'greyd_forms' ),
							6 => __( "Saturday", 'greyd_forms' ),
							7 => __( "Sunday", 'greyd_forms' ),
						);
						$days     = array(
							1 => __( "Monday", 'greyd_forms' ),
							2 => __( "Tuesday", 'greyd_forms' ),
							3 => __( "Wednesday", 'greyd_forms' ),
							4 => __( "Thursday", 'greyd_forms' ),
							5 => __( "Friday", 'greyd_forms' ),
							6 => __( "Saturday", 'greyd_forms' ),
							7 => __( "Sunday", 'greyd_forms' ),
						);
						$weeks    = array(
							1  => __( "first", 'greyd_forms' ),
							2  => __( "second", 'greyd_forms' ),
							3  => __( "third", 'greyd_forms' ),
							4  => __( "fourth", 'greyd_forms' ),
							-1 => __( "last", 'greyd_forms' ),
						);

						// Daily
						if ( $rec_type == 1 ) {
							$value = sprintf( _n( 'every day', 'every %d days', $interval, 'greyd_forms' ), $interval );
						}
						// Weekly
						elseif ( $rec_type == 2 ) {
							$weekly_days = isset( $recurrence['weekly_days'] ) ? strval( recurrence['weekly_days'] ) : false; // 1: Monday, ... 7: Sunday

							// Alle X Wochen...
							$value = sprintf( _n( 'every week', 'every %d weeks', $interval, 'greyd_forms' ), $interval );

							// montags...
							if ( strpos( $weekly_days, ',' ) === false ) {
								$value .= ' ' . $daily[ $weekly_days ];
							}
							// dienstags, mittwochs und samstags
							else {
								$weekly_days = explode( ',', $weekly_days );
								$i           = 0;
								foreach ( $weekly_days as $day ) {
									if ( $i === 0 ) {
										$value .= ' ' . $daily[ $day ];
									} elseif ( $i === count( $weekly_days ) ) {
										$value .= ' ' . __( 'and', 'greyd_forms' ) . ' ' . $daily[ $day ];
									} else {
										$value .= ', ' . $daily[ $day ];
									}
									$i++;
								}
							}
						}
						// Monthly
						elseif ( $rec_type == 3 ) {
							$monthly_day      = isset( $recurrence['monthly_day'] ) ? (int) $recurrence['monthly_day'] : false; // 1-31
							$monthly_week     = isset( $recurrence['monthly_week'] ) ? $recurrence['monthly_week'] : false; // 1: first, ... 4: fourth, -1: last
							$monthly_week_day = isset( $recurrence['monthly_week_day'] ) ? (int) $recurrence['monthly_week_day'] : false; // 1: Monday, ... 7: Sunday

							// Alle X Monate...
							$value = sprintf( _n( 'every month', 'every %d months', $interval, 'greyd_forms' ), $interval );

							if ( $monthly_day ) {
								$value .= ' ' . sprintf( __( 'always on %d.', 'greyd_forms' ), $monthly_day );
							} elseif ( $monthly_week && $monthly_week_day ) {
								$value .= ' ' . sprintf( __( "always on %1\$s of %2\$s week", 'greyd_forms' ), $days[ $monthly_week_day ], $weeks[ $monthly_week ] );
							}
						}

						// Bis zum XX.XX.XX...
						$end_after = isset( $recurrence['end_times'] ) ? (int) $recurrence['end_times'] : false; // end after x times
						$end_date  = isset( $recurrence['end_date_time'] ) ? $recurrence['end_date_time'] : false; // end after this date
						if ( $end_after ) {
							$value .= ' ' . sprintf( __( "%d more times", 'greyd_forms' ), $end_after );
						} elseif ( $end_date ) {
							$value .= ' ' . sprintf( __( "until %s", 'greyd_forms' ), date_i18n( get_option( 'date_format' ), strtotime( $end_date ) ) );
						}
					}
				}
			}

			$content = preg_replace( '/_' . $tag . '_/', $value, $content );
		}
		return $content;
	}

	/**
	 * Return clean time to display in frontend
	 *
	 * @param string $timestring
	 * @param string $timezone
	 */
	public function get_frontend_time( $timestring, $timezone ) {

		$format = get_option( 'date_format' ) . ', ' . get_option( 'time_format' );
		$date   = new \DateTime( $timestring, new \DateTimeZone( $timezone ) );
		$offset = wp_timezone()->getOffset( $date );

		return date_i18n( $format, $date->format( 'U' ) + $offset ) . ' ' . __( "o’clock", 'greyd_forms' );
	}

	/**
	 * Register vc modules
	 */
	public function add_zoom_webinar_vc() {

		if ( ! is_callable( 'vc_map' ) ) {
			return;
		}

		/** Dropdown */
		vc_map(
			array(
				'name'        => __( "Webinar date selection", 'greyd_forms' ),
				'description' => __( "Lets the user choose the date (for repeated webinars)", 'greyd_forms' ),
				'base'        => self::SHORTCODE . '_dropdown',
				'icon'        => plugins_url( '/assets/zoom.svg', __FILE__ ),
				'category'    => __( "Form elements", 'greyd_forms' ),
				'weight'      => 2,
				'params'      => array_merge(
					array(

						// Label
						array(
							'type'             => 'textfield',
							'heading'          => __( 'Label', 'greyd_forms' ),
							'description'      => __( "Displayed above the field.", 'greyd_forms' ),
							'param_name'       => 'label',
							'admin_label'      => true,
							'edit_field_class' => 'vc_col-xs-6',
						),
						array(
							'type'       => 'info_text',
							'text'       => __( "This module automatically renders a dropdown with all available dates. It is only displayed if the selected webinar is a „recurring webinar with fixed times“.", 'greyd_forms' ),
							'param_name' => 'info',
							'styling'    => 'info',
						),
					)
				),
			)
		);

		/** Infos */
		$ph_text = __( "Use the following placeholders for dynamic values in the frontend:", 'greyd_forms' ) . '<br>';
		foreach ( $this->info_tags as $tag ) {
			$ph_text .= '<span class="dtag"><span onclick="composer.insertTagText(this)">_' . $tag . '_</span></span>';
		}
		$ph_text .= '</ul>';
		vc_map(
			array(
				'name'        => __( "Webinar information", 'greyd_forms' ),
				'description' => __( "Shows information about the webinar in the frontend", 'greyd_forms' ),
				'base'        => self::SHORTCODE . '_infos',
				'icon'        => plugins_url( '/assets/zoom.svg', __FILE__ ),
				'category'    => __( "Form elements", 'greyd_forms' ),
				'weight'      => 1,
				'params'      => array_merge(
					array(
						array(
							'type'       => 'seperator',
							'param_name' => 'seperator_content',
							'heading'    => __( "Content", 'greyd_forms' ),
						),
						array(
							'type'             => 'textarea_html',
							'param_name'       => 'content',
							'description'      => $ph_text,
							'value'            => __( "<strong>Register now for the webinar „_topic_“.</strong><br>When: _time_<br>Duration: _duration_", 'greyd_forms' ),
							'edit_field_class' => 'vc_col-xs-12',
						),
					)
				),
			)
		);
	}

	/**
	 * Remove vc module
	 *
	 * the module gets removed if we're either editing another posttype
	 * or if the zoom integration is not activated for this form.
	 */
	public function remove_vc_modules() {

		if ( get_post_type() == 'tp_forms' ) {
			$post_id = isset( $_GET['post'] ) ? intval( $_GET['post'] ) : null;

			if ( $post_id ) {
				$meta    = Greyd_Forms_Interface::get_meta( $post_id, self::INTERFACE );
				$enable  = isset( $meta['enable'] ) ? $meta['enable'] : null;
				$webinar = isset( $meta['meta'] ) && isset( $meta['meta']['webinar'] ) ? $meta['meta']['webinar'] : null;

				// don't remove if enabled
				if ( ! empty( $webinar ) && $enable == 'on' ) {
					return;
				}
			}
		}

		// otherwise -> remove module
		vc_remove_element( self::SHORTCODE . '_infos' );
		vc_remove_element( self::SHORTCODE . '_dropdown' );
	}

}
