<?php
/**
 * Manage core plugin functionality.
 */
namespace Greyd\Forms;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

new Manage( $config );
class Manage {

	private $config;
	private $textdomain;

	public function __construct( $config ) {

		// set config
		$this->config = (object) $config;

		// enable translations
		add_action( 'plugins_loaded', array( $this, 'register_textdomain' ) );
		$this->textdomain = $this->config->plugin_textdomain;

		// enable updates
		add_action( 'after_setup_theme', array( $this, 'enable_updates' ) );

		// display admin notices if necessary
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );
		add_action( 'network_admin_notices', array( $this, 'display_admin_notices' ) );

		// set required versions
		// deprecated - remove in next version
		add_filter( 'greyd_theme_required_versions', array( $this, 'set_required_theme_version' ) );
		add_filter( 'greyd_hub_required_versions', array( $this, 'set_required_hub_version' ) );
		add_filter( 'greyd_blocks_required_versions', array( $this, 'set_required_blocks_version' ) );
	}

	/**
	 * Register textdomain for translations
	 */
	public function register_textdomain() {
		load_plugin_textdomain( 'greyd_forms', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * Enable automatic updates via 'Plugin Update Checker'
	 */
	public function enable_updates() {
		if ( class_exists( 'Puc_v4_Factory' ) && ! $this->is_frozen() ) {
			$myUpdateChecker = \Puc_v4_Factory::buildUpdateChecker(
				$this->config->update_file,     // Full path to the update server manifest file.
				$this->config->plugin_file,     // Full path to the main plugin file or functions.php.
				$this->config->plugin_name      // Insert Plugin slug, usually same as directory name.
			);
		}
	}

	public function is_frozen() {
		$l = get_option( 'gtpl' );
		return $l && ! empty( $l ) && is_array( $l ) && isset( $l['frozen'] ) && $l['frozen'];
	}

	/**
	 * Display admin notices
	 */
	public function display_admin_notices() {

		// show update notice
		if ( $this->plugin_needs_update() && ! ( isset( $_GET['action'] ) && $_GET['action'] === 'do-plugin-upgrade' ) ) {
			echo "<div class='error notice'><p>" . sprintf(
				__( "The plugin %1\$s requires a newer version. Please %2\$supdate the plugin%3\$s to prevent possible errors.", 'greyd_forms' ),
				'<strong>' . $this->config->plugin_name_full . '</strong>',
				"<a href='" . ( is_multisite() ? network_admin_url( 'update-core.php' ) : admin_url( 'update-core.php' ) ) . "'>",
				'</a>'
			) . '</p></div>';
		}

		// hub plugin not active
		if ( ! defined( 'GREYD_VERSION' ) ) {

			// cache is used to not display the notice in other plugins of the suite
			if ( wp_cache_get( 'greyd_hub_plugin_notice_displayed', 'greyd' ) ) return;
			
			?>
			<div class="notice notice-warning">
				<div style="padding:12px">
					<p><strong><?php
						_e("Setup incomplete", "greyd_forms" );
					?></strong></p>
					<p><?php echo sprintf(
						__("The Greyd.Forms plugin only works in conjunction with the %sGreyd plugin%s. Download the plugin and activate it.", "greyd_forms" ),
						'<strong>',
						'</strong>'
					);
					?></p>
					<p>
						<a href="<?php echo esc_url( admin_url( 'plugins.php' ) ) ?>"><?php echo __("To the plugins", "greyd_forms"); ?></a>&nbsp;&nbsp;|&nbsp;&nbsp;<a style="color: inherit;" href="https://update.greyd.io/public/plugins/greyd-plugin/greyd-plugin.zip"><?php echo __("Download plugin", "greyd_forms"); ?></a>
					</p>
				</div>
			</div>
			<?php
			wp_cache_set( 'greyd_hub_plugin_notice_displayed', true, 'greyd' );
		}
	}

	/**
	 * Check if this plugin needs an update
	 *
	 * @return bool Returns true if this plugin needs an update.
	 */
	public function plugin_needs_update() {
		/**
		 * Get required versions
		 * @deprecated
		 * 
		 * @filter greyd_forms_required_versions
		 */
		$requirements = apply_filters( 'greyd_forms_required_versions', array() );

		/**
		 * Get required versions of this plugin
		 * 
		 * @filter greyd_versions
		 */
		$versions = apply_filters( 'greyd_versions', array() );
		foreach( $versions as $n => $v ) {
			if ( isset($v['required'][$this->config->plugin_textdomain]) ) {
				$requirements[$n] = $v['required'][$this->config->plugin_textdomain];
			}
		}

		if ( empty($requirements) || !is_array($requirements) ) return false;
		// debug( 'forms requirements' );
		// debug( $requirements );

		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_true        = true;
		$plugin_data     = get_plugin_data( $this->config->plugin_file );
		$current_version = isset( $plugin_data['Version'] ) && ! empty( $plugin_data['Version'] ) ? $plugin_data['Version'] : '0';
		foreach ( $requirements as $required_version ) {
			if ( $all_true ) {
				$all_true = version_compare( $current_version, $required_version, '>=' );
			}
		}
		return ! $all_true;
	}

	/**
	 * Add required Greyd.Theme Version
	 *
	 * @filter greyd_suite_required_versions
	 */
	public function set_required_theme_version( $return ) {
		$versions = isset( $this->config->required_versions ) ? $this->config->required_versions : array();
		$version  = isset( $versions['greyd_suite'] ) ? $versions['greyd_suite'] : null;
		if ( $version ) {
			$return[ $this->config->plugin_name ] = $version;
		}
		return $return;
	}

	/**
	 * Add required Greyd.Hub Version
	 *
	 * @filter greyd_hub_required_versions
	 */
	public function set_required_hub_version( $return ) {
		$versions = isset( $this->config->required_versions ) ? $this->config->required_versions : array();
		$version  = isset( $versions['greyd_hub'] ) ? $versions['greyd_hub'] : null;
		if ( $version ) {
			$return[ $this->config->plugin_name ] = $version;
		}
		return $return;
	}

	/**
	 * Add required Greyd.Blocks Version
	 *
	 * @filter greyd_blocks_required_versions
	 */
	public function set_required_blocks_version( $return ) {
		$versions = isset( $this->config->required_versions ) ? $this->config->required_versions : array();
		$version  = isset( $versions['greyd_blocks'] ) ? $versions['greyd_blocks'] : null;
		if ( $version ) {
			$return[ $this->config->plugin_name ] = $version;
		}
		return $return;
	}
}
