( function ( wp ) {

	const {
		createElement: el,
	} = wp.element;

	const {
		__,
		_x
	} = wp.i18n;

	const {
		PanelBody,
		ToggleControl,
		Button,
		Icon,
		Tip
	} = wp.components;

	const {
		InnerBlocks,
		InspectorControls
	} = wp.blockEditor;

	/**
	 * Register the conditional content block.
	 */
	wp.blocks.registerBlockType( 'greyd/conditional-content', {
		title: __( 'Conditional Content', 'greyd_hub' ),
		description: __( "Dependency of user roles, time or other parameters", 'greyd_hub' ),
		icon: greyd.tools.getBlockIcon('condition'),
		category: 'greyd-blocks',
		supports: {
			customClassName: false,
		},
		attributes: {
			operator: { type: 'string', default: 'OR' },
			conditions: { type: 'array', default: [] },
			debug: { type: 'bool', default: false },
		},

		edit: function ( props ) {

			const hasChildBlocks = greyd.tools.hasChildBlocks( props.clientId );
			const defaultCondition = {
				type: 'urlparam',
				operator: 'is',
				value: '',
				detail: '',
				custom: '',
			};

			const types = {
				urlparam: { value: 'urlparam', label: _x( "URL parameter", 'small', 'greyd_hub' ) },
				cookie: { value: 'cookie', label: _x( "session cookie", 'small', 'greyd_hub' ) },
				userrole: { value: 'userrole', label: _x( "User role", 'small', 'greyd_hub' ) },
				time: { value: 'time', label: _x( "Time", 'small', 'greyd_hub' ) },
				posttype: { value: 'posttype', label: _x( "Post type", 'small', 'greyd_hub' ) },
				field: { value: 'field', label: _x( "Post type meta field value of the current post", 'small', 'greyd_hub' ) },
				taxonomy: { value: 'taxonomy', label: _x( "Taxonomy of the current post", 'small', 'greyd_hub' ) },
				archivetax: { value: 'archivetax', label: _x( "Taxonomy of the current archive", 'small', 'greyd_hub' ) },
				search: { value: 'search', label: _x( "number of posts found", 'small', 'greyd_hub' ) },
				postIndex: { value: 'postIndex', label: _x( "Index of the current post", 'small', 'greyd_hub' ) },
			};
			const operators = {
				urlparam: {
					is: { value: 'is', label: _x( "has the value:", 'small', 'greyd_hub' ) },
					has: { value: 'has', label: _x( "contains the value:", 'small', 'greyd_hub' ) },
					has_not: { value: 'has_not', label: _x( "does not contain the value:", 'small', 'greyd_hub' ) },
					is_not: { value: 'is_not', label: _x( "has not the value:", 'small', 'greyd_hub' ) },
					not_empty: { value: 'not_empty', label: _x( "is set", 'small', 'greyd_hub' ) },
					empty: { value: 'empty', label: _x( "is not set", 'small', 'greyd_hub' ) },
				},
				userrole: {
					is: { value: 'is', label: _x( "is:", 'small', 'greyd_hub' ) },
					is_not: { value: 'is_not', label: _x( "is not:", 'small', 'greyd_hub' ) },
				},
				posttype: {
					is: { value: 'is', label: _x( "is:", 'small', 'greyd_hub' ) },
					is_not: { value: 'is_not', label: _x( "is not:", 'small', 'greyd_hub' ) },
				},
				field: {
					is: { value: 'is', label: _x( "has the value:", 'small', 'greyd_hub' ) },
					has: { value: 'has', label: _x( "contains the value:", 'small', 'greyd_hub' ) },
					has_not: { value: 'has_not', label: _x( "does not contain the value:", 'small', 'greyd_hub' ) },
					not_empty: { value: 'not_empty', label: _x( "is set", 'small', 'greyd_hub' ) },
					empty: { value: 'empty', label: _x( "is not set", 'small', 'greyd_hub' ) },
				},
				taxonomy: {
					has: { value: 'has', label: _x( "has the taxonomy:", 'small', 'greyd_hub' ) },
					has_not: { value: 'has_not', label: _x( "has not the taxonomy:", 'small', 'greyd_hub' ) },
				},
				archivetax: {
					is: { value: 'is', label: _x( "is archive of the taxonomy:", 'small', 'greyd_hub' ) },
					is_not: { value: 'is_not', label: _x( "is not archive of this taxonomy:", 'small', 'greyd_hub' ) },
				},
				search: {
					multiple: { value: 'multiple', label: _x( "several results found", 'small', 'greyd_hub' ) },
					single: { value: 'single', label: _x( "exactly 1 result found", 'small', 'greyd_hub' ) },
					empty: { value: 'empty', label: _x( "Search results are empty", 'small', 'greyd_hub' ) },
				},
				postIndex: {
					is: { value: 'is', label: _x( "is", 'small', 'greyd_hub' ) },
					is_not: { value: 'is_not', label: _x( "is not", 'small', 'greyd_hub' ) },
					less: { value: 'less', label: _x( "smaller than", 'small', 'greyd_hub' ) },
					greater: { value: 'greater', label: _x( "greater than", 'small', 'greyd_hub' ) },
					even: { value: 'even', label: _x( "Even", 'small', 'greyd_hub' ) },
					odd: { value: 'odd', label: _x( "odd", 'small', 'greyd_hub' ) },
				},
			};
			operators.cookie = operators.urlparam;
			operators.time = operators.userrole;

			const times = {
				'': { value: '', label: __( "Select time", 'greyd_hub' ) },
				'5-12': { value: '5-12', label: _x( "in the morning (from 5am)", 'small', 'greyd_hub' ) },
				'12-17': { value: '12-17', label: _x( "afternoon (from 12 o’clock)", 'small', 'greyd_hub' ) },
				'17-23': { value: '17-23', label: _x( "in the evening (from 5 pm)", 'small', 'greyd_hub' ) },
				'23-24,0-5': { value: '23-24,0-5', label: _x( "at night (11 pm to 5 am)", 'small', 'greyd_hub' ) },
				'custom': { value: 'custom', label: _x( "individual", 'small', 'greyd_hub' ) }
			};

			return [

				// sidebar
				el( InspectorControls, {}, [
					// operator
					el( PanelBody, {
						title: __( "Details of the conditions", 'greyd_hub' ),
						initialOpen: true
					}, [
						el( greyd.components.ButtonGroupControl, {
							value: props.attributes.operator,
							options: [
								{ value: 'OR', label: __( "OR", 'greyd_hub' ) },
								{ value: 'AND', label: __( "AND", 'greyd_hub' ) },
							],
							onChange: ( value ) => { props.setAttributes( { operator: value } ); },
						} ),
						el( Tip, {}, __( "With AND all conditions must be true, with OR at least one.", 'greyd_hub' ) ),
					] ),
					// conditions
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Conditions", 'greyd_hub' ),
						holdsChange: props.attributes.conditions.length > 0,
						initialOpen: true
					}, [
						el( 'div', {
							className: 'components-greyd-controlgroup',
						}, [
							...props.attributes.conditions.map( ( condition, i ) => {

								condition = { ...defaultCondition, ...condition };

								// get all taxonomies & terms
								const postTypes = [], taxonomies = [], terms = [], fields = [];
								
								if ( condition.type == 'taxonomy' || condition.type == 'archivetax' ) {
									for ( const [ slug, taxes ] of Object.entries( greyd.data.all_taxes ) ) {
										if ( taxes.length ) {
											postTypes.push( {
												value: slug,
												label: slug
											} );

											if ( slug == condition.detail ) {
												taxes.forEach( taxonomy => {
													if ( taxonomy.values.length ) {
														taxonomies.push( {
															value: taxonomy.slug,
															label: taxonomy.title
														} );

														if ( taxonomy.slug == condition.custom ) {
															taxonomy.values.forEach( term => {
																terms.push( {
																	value: term.slug,
																	label: term.title
																} );
															} );
														}
													}
												} );
											}
										}
									}
								}
								else if ( condition.type == 'posttype' ) {
									for ( const [ i, slug ] of Object.entries( greyd.data.all_posttypes ) ) {
										postTypes.push( {
											value: slug,
											label: slug
										} );
									}
								}
								else if ( condition.type == 'field' ) {
									greyd.data.post_types.forEach( (pt) => {
										if ( pt.fields ) {
											postTypes.push( {
												value: pt.slug,
												label: pt.slug
											} );
											if ( pt.slug == condition.detail ) {
												pt.fields.forEach( (field) => {
													fields.push( {
														value: field.name,
														label: field.label
													} );
												} );
											}
										}
									} );
								}

								const groupComponents = [];

								// remove
								groupComponents.push( el( Button, {
									className: "components-greyd-controlgroup__remove",
									onClick: () => {
										const newConditions = [ ...props.attributes.conditions ];
										newConditions.splice( i, 1 );
										props.setAttributes( { conditions: newConditions } );
									},
									title: __( "Remove condition", 'greyd_hub' )
								}, el( Icon, { icon: 'no-alt' } ) ) );

								// type
								groupComponents.push( el( greyd.components.OptionsControl, {
									label: __( "Type of condition", 'greyd_hub' ),
									value: condition.type,
									options: Object.values( types ),
									onChange: ( value ) => {
										const newConditions = [ ...props.attributes.conditions ];
										if ( value === 'search' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'multiple' };
										} else if ( value === 'taxonomy' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'has', value: [] };
										} else if ( value == 'archivetax' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'is', value: [] };
										} else if ( value == 'posttype' || value == 'field' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'is', value: [] };
										} else if ( value == 'postIndex' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'is', value: [] };
										} else if ( value == 'postIndex' ) {
											newConditions[ i ] = { ...defaultCondition, type: value, operator: 'is', value: [] };
										} else {
											newConditions[ i ] = { ...defaultCondition, type: value };
										}
										props.setAttributes( { conditions: newConditions } );
									},
								} ) );

								// url-parm / cookie detail
								if ( condition.type == 'urlparam' || condition.type == 'cookie' ) {
									groupComponents.push( el( greyd.components.OptionsControl, {
										// label: __("Parameter", 'greyd_hub'),
										value: condition.detail,
										options: [
											{ value: '', label: __( "Select parameter", 'greyd_hub' ) },
											...greyd.data.url_params.map( param => {
												return {
													label: param.nicename,
													value: param.name
												};
											} ),
											{ value: 'custom', label: __( "individual parameter", 'greyd_hub' ) },
										],
										onChange: ( value ) => {
											const newConditions = [ ...props.attributes.conditions ];
											newConditions[ i ].detail = value;
											props.setAttributes( { conditions: newConditions } );
										},
									} ) );
									if ( condition.detail == 'custom' ) {
										groupComponents.push( el( 'input', {
											value: condition.custom,
											placeholder: __( "individual parameter", 'greyd_hub' ),
											className: 'components-text-control__input components-base-control',
											onInput: ( event ) => {
												const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ index ].custom = event.target.value;
												props.setAttributes( { conditions: newConditions } );
											}
										} ) );
									}
								}

								// taxonomy detail
								if ( condition.type == 'taxonomy' || condition.type == 'archivetax' || condition.type == 'field' ) {

									// posttype
									groupComponents.push( el( greyd.components.SelectCustomControl, {
										value: condition.detail,
										options: [
											{ value: '', label: __( "Select post type", 'greyd_hub' ) },
											...postTypes,
										],
										onChange: ( value ) => {
											const newConditions = [ ...props.attributes.conditions ];
											newConditions[ i ].detail = value;
											newConditions[ i ].custom = '';
											newConditions[ i ].value = [];
											props.setAttributes( { conditions: newConditions } );
										},
									} ) );

									// taxonomy
									if ( !_.isEmpty( condition.detail ) && condition.type !== 'field' ) {
										groupComponents.push( el( greyd.components.SelectCustomControl, {
											value: condition.custom,
											options: [
												{ value: '', label: __( "Select taxonomy", 'greyd_hub' ) },
												...taxonomies,
											],
											onChange: ( value ) => {
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ i ].custom = value;
												newConditions[ i ].value = [];
												props.setAttributes( { conditions: newConditions } );
											},
										} ) );
									}
									// field
									if ( !_.isEmpty( condition.detail ) && condition.type == 'field' ) {
										groupComponents.push( el( greyd.components.SelectCustomControl, {
											value: condition.custom,
											options: [
												{ value: '', label: __( "Select meta field", 'greyd_hub' ) },
												...fields,
											],
											onChange: ( value ) => {
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ i ].custom = value;
												newConditions[ i ].value = [];
												props.setAttributes( { conditions: newConditions } );
											},
										} ) );
									}

								}

								// operator
								groupComponents.push( el( greyd.components.OptionsControl, {
									label: __( "Condition", 'greyd_hub' ),
									value: condition.operator,
									options: Object.values( operators[ condition.type ] ),
									onChange: ( value ) => {
										const newConditions = [ ...props.attributes.conditions ];
										newConditions[ i ].operator = value;
										props.setAttributes( { conditions: newConditions } );
									},
								} ) );

								// value
								if (
									condition.operator !== 'empty'
									&& condition.operator !== 'not_empty'
									&& condition.type !== 'search'
									&& condition.operator !== 'even'
									&& condition.operator !== 'odd'
								) {

									groupComponents.push( el( 'div', { style: { marginBottom: '4px' } }, __( "Value", 'greyd_hub' ) ) );

									if ( condition.type == 'urlparam' || condition.type == 'cookie' ) {
										groupComponents.push( el( 'input', {
											value: condition.value,
											placeholder: __( "Value of the parameter", 'greyd_hub' ),
											className: 'components-text-control__input components-base-control',
											onInput: ( event ) => {
												const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ index ].value = event.target.value;
												props.setAttributes( { conditions: newConditions } );
											}
										} ) );
										if ( condition.detail === 'optin' || condition.detail === 'optout' ) {
											groupComponents.push( el( 'div', {
												// className: 'components-tip',
												style: { marginBottom: '10px', color: '#757575', fontSize: 'small' },
												dangerouslySetInnerHTML: {
													__html: [
														"<ol style=\"margin:1em 0 1em 3em;\">",
														"<li><b>true-just</b>: " + _x( 'die Aktion (Opt-in bzw. Opt-out) wurde mit diesem Seitenaufruf erfolgreich durchgeführt.', 'small', 'core' ) + "</li>",
														"<li><b>true-already</b>: " + _x( 'die Aktion (Opt-in bzw. Opt-out) wurde bereits erfolgreich durchgeführt.', 'small', 'core' ) + "</li>",
														"<li><b>false-found</b>: " + _x( 'es konnte kein entsprechender Eintrag gefunden werden.', 'small', 'core' ) + "</li>",
														"<li><b>false-out</b>: " + _x( 'der Eintrag konnte nicht verifiziert werden (Opt-in), da ihm bereits widersprochen wurde (Opt-out).', 'small', 'core' ) + "</li>",
														"</ol>",
														"<i>" + __( 'Beispiel: Du möchtest einen Inhalt nur anzeigen, wenn der Nutzer seine Identität bestätigt hat (Opt-in). Wähle als Parameter „Formular Opt-in“. Als Bedingung stellst du „enthält den Wert:“ ein und als Wert gibst du „true“ ein. Jetzt trifft die Bedingung auf beide Werte mit dem Zusatz „true“ zu.', 'core' ) + "</i>"
													].join( "" )
												}
											} ) );
										}
									}
									else if ( condition.type == 'userrole' ) {
										groupComponents.push( el( greyd.components.OptionsControl, {
											value: condition.value,
											options: [
												{ value: '', label: _x( "Select user role", 'small', 'greyd_hub' ) },
												{ value: 'none', label: _x( "unknown (not logged in)", 'small', 'greyd_hub' ) },
												...Object.entries( greyd.data.user_roles ).map( param => {
													return {
														label: param[ 1 ].name,
														value: param[ 0 ]
													};
												} ),
											],
											onChange: ( value ) => {
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ i ].value = value;
												props.setAttributes( { conditions: newConditions } );
											},
										} ) );
									}
									else if ( condition.type == 'time' ) {
										groupComponents.push( el( greyd.components.OptionsControl, {
											value: condition.value,
											options: Object.values( times ),
											onChange: ( value ) => {
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ i ].value = value;
												props.setAttributes( { conditions: newConditions } );
											},
										} ) );
										if ( condition.value == 'custom' ) {
											groupComponents.push( el( 'input', {
												value: condition.custom,
												placeholder: __( "individual time of day", 'greyd_hub' ),
												className: 'components-text-control__input components-base-control',
												onInput: ( event ) => {
													const index = parseInt( event.target.closest( '.components-greyd-controlgroup__item' ).dataset.index );
													const newConditions = [ ...props.attributes.conditions ];
													newConditions[ index ].custom = event.target.value;
													props.setAttributes( { conditions: newConditions } );
												}
											} ) );
											groupComponents.push( el( 'div', {
												// className: 'components-tip',
												style: { marginBottom: '10px', color: '#757575', fontSize: 'small' },
												dangerouslySetInnerHTML: {
													__html: __( "Indicate times in full hours or with minutes and separate them with hyphens (e.g. for 12 to 13:30: <code>12-13:30</code>). Separate multiple time periods with commas (e.g., <code>12-13,15-16</code>). Indicate times before and after midnight as individual sections (e.g., for 10 p.m. to 4 a.m.: <code>22-24,0-4</code>).", 'greyd_hub' )
												}
											} ) );
										}
									}
									else if ( condition.type == 'taxonomy' || condition.type == 'archivetax' ) {

										if ( terms.length ) {
											groupComponents.push(
												el( greyd.components.SelectCustomControl, {
													value: condition.value,
													options: terms,
													multiple: condition.type == 'taxonomy',
													// disabled: !terms.length,
													onChange: ( value ) => {
														const newConditions = [ ...props.attributes.conditions ];
														newConditions[ i ].value = value;
														props.setAttributes( { conditions: newConditions } );
													},
												} )
											);
										}
										else {
											groupComponents.push(
												el( wp.components.TextControl, {
													value: condition.value,
													onChange: ( value ) => {
														const newConditions = [ ...props.attributes.conditions ];
														newConditions[ i ].value = value;
														props.setAttributes( { conditions: newConditions } );
													},
												} )
											);
										}

									}
									else if ( condition.type == 'field' ) {

										groupComponents.push(
											el( wp.components.TextControl, {
												value: condition.value,
												onChange: ( value ) => {
													const newConditions = [ ...props.attributes.conditions ];
													newConditions[ i ].value = value;
													props.setAttributes( { conditions: newConditions } );
												},
											} )
										);

									}
									else if ( condition.type == 'posttype' ) {
										groupComponents.push(
											el( greyd.components.SelectCustomControl, {
												value: condition.value,
												options: [
													{ value: '', label: __( "Select post type", 'greyd_hub' ) },
													...postTypes,
												],
												disabled: !postTypes.length,
												onChange: ( value ) => {
													const newConditions = [ ...props.attributes.conditions ];
													newConditions[ i ].value = value;
													props.setAttributes( { conditions: newConditions } );
												},
											} )
										);
									}
									else if ( condition.type == 'postIndex' ) {
										groupComponents.push( el( wp.components.__experimentalNumberControl, {
											value: condition.value,
											isDragEnabled: true,
											isShiftStepEnabled: true,
											min: 0,
											step: 1,
											onChange: ( value ) => {
												const newConditions = [ ...props.attributes.conditions ];
												newConditions[ i ].value = value;
												props.setAttributes( { conditions: newConditions } );
											}
										} ) );
									}
								};

								// info
								if ( condition.type == 'search' ) {
									groupComponents.push( el( 'div', {
										className: 'components-tip',
										style: { marginBottom: '10px' }
									}, __( "The dependency on found posts only works in search & archive templates. Otherwise, this content is not displayed.", 'greyd_hub' ) ) );
								}
								else if ( condition.type == 'taxonomy' ) {
									groupComponents.push( el( 'div', {
										className: 'components-tip',
										style: { marginBottom: '10px' }
									}, __( "The dependency on taxonomies only works in archive templates. Otherwise, this content is not displayed.", 'greyd_hub' ) ) );
								}
								else if ( condition.type == 'archivetax' ) {
									groupComponents.push( el( 'div', {
										className: 'components-tip',
										style: { marginBottom: '10px' }
									}, __( "The dependency on archive taxonomies only works in archive templates. Otherwise, this content is not displayed.", 'greyd_hub' ) ) );
								}
								else if ( condition.type == 'field' ) {
									groupComponents.push( el( 'div', {
										className: 'components-tip',
										style: { marginBottom: '10px' }
									}, __( "The dependency on post type meta fields only works with Greyd post types. Otherwise, this content is not displayed.", 'greyd_hub' ) ) );
								}
								else if ( condition.type == 'postIndex' ) {
									groupComponents.push( el( 'div', {
										className: 'components-tip',
										style: { marginBottom: '10px' }
									}, __( "The dependency on the index of the current post only works in query loops. The first item displayed has the index '1', the second ‚2‘, etc. Otherwise, this value is always '0'.", 'greyd_hub' ) ) );
								}

								return el( 'div', {
									className: 'components-greyd-controlgroup__item',
									'data-index': i,
								}, groupComponents );
							} ),
							el( Button, {
								className: 'components-greyd-controlgroup__add' + ( props.attributes.conditions.length === 0 ? ' group_is_empty' : '' ),
								onClick: ( event ) => {
									props.setAttributes( { conditions: [ ...props.attributes.conditions, defaultCondition ] } );
									// console.log(props.attributes.conditions);
								},
								title: __( "Add condition", 'greyd_hub' )
							}, [
								el( Icon, { icon: 'plus-alt2' } ),
								props.attributes.conditions.length === 0 ? el( 'span', {}, __( "Add condition", 'greyd_hub' ) ) : null
							] )
						] )
					] ),
					// debug
					el( greyd.components.AdvancedPanelBody, {
						title: __( "Debug mode", 'greyd_hub' ),
						holdsChange: props.attributes.debug,
						initialOpen: true
					}, [
						el( ToggleControl, {
							label: __( "Enable debug mode", 'greyd_hub' ),
							checked: props.attributes.debug,
							onChange: ( value ) => { props.setAttributes( { debug: value } ); },
							help: __( "A text is displayed in the frontend that helps to identify problems. Attention: do not use on live sites!", 'greyd_hub' ),
						} ),
					] ),
				] ),

				// preview
				el( 'div', { className: props.className + ' preview-info-wrapper' }, [
					el( 'div', { className: 'preview-info-tag flex' }, [
						greyd.tools.getBlockIcon('condition'),
						el( 'div', { className: 'preview-info-title' },
							el( "ul", {}, props.attributes.conditions.map( ( condition, i ) => {

								condition = { ...defaultCondition, ...condition };
								let prefix = i < 1 ? "" : ( props.attributes.operator === 'AND' ? __( "AND", 'greyd_hub' ) : __( "OR", 'greyd_hub' ) ) + " ";
								let label = types[ condition.type ].label + " ";
								let oprtr = operators[ condition.type ][ condition.operator ].label;
								let value = condition.operator === 'empty' || condition.operator === 'not_empty' ? "" : " '" + condition.value + "'";

								if ( condition.type === 'urlparam' || condition.type === 'cookie' ) {
									if ( condition.detail === 'custom' ) {
										label += "'" + condition.custom + "' ";
									} else {
										label += "'" + condition.detail + "' ";
									}
								}
								else if ( condition.type === 'time' ) {
									if ( condition.value === 'custom' ) value = " '" + condition.custom + "' " + __( "o’clock", 'greyd_hub' );
									else value = " " + times[ condition.value ].label;
								}
								else if ( condition.type === 'search' ) {
									value = "";
								}

								return el( "li", { dangerouslySetInnerHTML: { __html: prefix + "<strong>" + label + "</strong>" + oprtr + value + "." } } );
							} ) )
						),
					] ),
					el( 'div', { className: 'preview-info-content' }, [
						el( InnerBlocks, { renderAppender: hasChildBlocks ? InnerBlocks.DefaultBlockAppender : InnerBlocks.ButtonBlockAppender } )
					] )
				] )
			];
		},
		save: function ( props ) {
			return el( InnerBlocks.Content );
		}
	} );

} )(
	window.wp
);