/**
 * Script for post export and import
 */
(function() { 

	jQuery(function() {

		if (typeof $ === 'undefined') $ = jQuery;

		/**
		 * Compatibility with old class var used by globalcontent
		 */
		post_export = greyd.postExport;

		greyd.postExport.init();
		
		console.log('Post Export Scripts: loaded');
	} );

} )(jQuery);

greyd.postExport = new function() {

	this.init = function() {
		this.importInit();
	}

	/**
	 * Start the export dialog (row action)
	 * 
	 * @param {object} elem 
	 */
	this.openExport = function( elem ) {
		var td          = $(elem).closest("td.title");
		if ( td.find('.filename').length ) {
			var title = td.find('.filename').clone().children().remove().end().text();
		}
		else {
			var title = td.find("strong a").text();
		}
		greyd.backend.confirm( 'post_export', title.trim(), greyd.postExport.export, [ elem ] );
	} 

	/**
	 * Export a post (row action)
	 * 
	 * @param {object} elem 
	 */
	this.export = function( elem ) {

		var mode        = 'post_export';
		var post_id     = $(elem).data('post_id');
		var form        = $('#post_export_form');
		var data        = form.serializeArray().reduce(function(obj, item) {
			obj[item.name] = item.value;
			return obj;
		}, {} );
		data.post_id = post_id;
		console.log( data );
		
		$.post(
			greyd.ajax_url, {
				'action': 'greyd_ajax',
				'_ajax_nonce': greyd.nonce,
				'mode': mode,
				'data': data
			}, 
			function(response) {
				console.log(response);
				
				// successfull
				if ( response.indexOf('success::') > -1 ) {

					// trigger overlay
					greyd.backend.triggerOverlay( true, { "type": "success", "css": mode } );
				
					// download file
					var file    = response.split( 'success::' )[1];
					var link    = $('a#post_export_download');
					var filename = file.match(/\/[^\/]+.zip/) ? file.match(/\/[^\/]+.zip/)[0].replace("/", "") : "";
					if ( link.length === 0 ) {
						$('#wpfooter').after( '<a id="post_export_download"></a>' );
						link = $('a#post_export_download');
					}
					link.attr( {
						"href": file,
						"download": filename
					} );
					link[0].click();
					form[0].reset();
				}
				else if ( response.indexOf('error::') > -1 ) {
					var msg = response.split( 'error::' )[1];
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": msg });
				}
				else {
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": response});
				}
			}
		);
	}

	/**
	 * Open browser upload dialog (on import button click)
	 */
	this.openImport = function() {
		$('#post_import_form input[type="file"]').trigger("click");
	}

	/**
	 * Trigger check_import overlay (on file input click)
	 */
	this.importInit = function() {
		
		var mode        = 'post_import';
		var form        = $('#post_import_form');
		var file_input  = form.find('input[type="file"]');
		
		file_input.on('change', function (e) {

			if ( $(this).val() ) {
				
				// trigger overlay 'check_file'
				greyd.backend.triggerOverlay(true, {"type": "check_file", "css": mode});

				// create formData
				var data = new FormData();
				data.append('action', 'greyd_ajax');
				data.append('_ajax_nonce', greyd.nonce);
				data.append('mode', 'check_post_import');

				// append file data
				var file_data = file_input[0].files[0];
				data.append('data', file_data);

				// send to admin ajax
				$.ajax({
					'type': 'POST',
					'url': greyd.ajax_url,
					'processData': false,
					'contentType': false,
					'cache': false,
					'data': data,
					'error': function( xhr, textStatus, errorThrown ) {
						console.warn(textStatus+": "+errorThrown, xhr);
						file_input.val("");
						greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": errorThrown+": <code>"+xhr.responseText+"</code>" });
					},
					'success': function( response ) {
						file_input.val("");
						
						// successfull
						if ( response.indexOf('success::') > -1 ) {

							// get response: conflicting posts
							var result = response.split( 'success::' )[1];
							result = result.indexOf("[") === 0 ? JSON.parse(result) : result;
							console.log( result );

							greyd.postExport.buildConflictOptions( form, result );
							
							/**
							 * trigger overlay 'confirm'
							 * 
							 * callback: this.import( filename )
							 */
							greyd.backend.confirm( mode, '', greyd.postExport.import, [ file_data.name ] );
						}
						// complete with error
						else if ( response.indexOf('error::') > -1 ) {
							var msg = response.split( 'error::' )[1];
							console.log(msg);
							greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": msg });
						}
						// unknown state
						else {
							console.log(response);
							greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": response});
						}
					}
				});
			}
		});
	}

	/**
	 * Build the conflict selects inside the wrapper
	 * 
	 * @param {object} wrapper  HTML wrapper jQuery object
	 * @param {mixed} posts     Either an array of posts or a string.
	 */
	this.buildConflictOptions = function( wrapper, posts ) {

		var $new        = wrapper.find('.new');
		var $conflicts  = wrapper.find('.conflicts');

		if ( typeof posts === 'object' || Array.isArray(posts) ) {

			var $list   = $conflicts.find('.inner_content');
			var options = decodeURIComponent( $list.data('options') ).replace(/\+/g, ' ');

			$new.hide();
			$list.html("");

			$.each( posts, function(k, post) {

				$list.append( '<div><span>'+post.post_link+'</span><select name="'+post.ID+'">'+options+'</select></div>' );

				if ( post.post_type == 'tp_posttypes' ) {
					$list.find('select[name="'+post.ID+'"] option[value="keep"]').attr('disabled', true);
				}
				else if ( post.ID == 'multioption' ) {
					var $select = $list.find('select[name="'+post.ID+'"]');
					$select.removeAttr('name');
					$select.parent().addClass('multioption');
					$select.on('change', function() {
						var val = $(this).val();
						$list.find('select').each(function(){
							$(this).val( val );
						});
					});
				}
			});
			$conflicts.show();
		}
		else {
			var $post_title = $new.find('.post_title');
			
			$conflicts.hide();
			$post_title.html( posts );
			$new.show();
		}
	}

	/**
	 * Trigger post_import (on check_import overlay confirm)
	 * 
	 * @param {string} filename 
	 */
	this.import = function( filename ) {

		var mode        = 'post_import';
		var form        = $('#post_import_form');
		var form_data   = form.serializeArray().reduce(function(obj, item) {
			obj[item.name] = item.value;
			return obj;
		}, {} );
		var data = {
			filename: filename,
			conflicts: form_data
		};
		console.log( data );

		let isPosttype = false;

		const urlParams = new URLSearchParams(window.location.search);
		if ( urlParams.has('post_type') && urlParams.get('post_type') == 'tp_posttypes' ) {
			isPosttype = true;
		}
		else if ( data.filename && data.filename.indexOf('posts_and_posttype') > -1 ) {
			isPosttype = true;
		}
		
		greyd.postExport.postImport( data, isPosttype );
	}

	/**
	 * Import a post / posttype.
	 * 
	 * @since 1.2.7
	 * 
	 * If @param isPosttype is set, we import the posttype first, and trigger
	 * a second ajax call to import the posts after. This ensures that all
	 * taxonomies etc. are setup correctly.
	 * 
	 * @param {object} data Data containing filename & conflicts
	 * @param {bool} isPosttype Whether we import only the posttype first
	 */
	this.postImport = function( data, isPosttype ) {

		const mode = 'post_import';

		$.post(
			greyd.ajax_url, {
				'action': 'greyd_ajax',
				'_ajax_nonce': greyd.nonce,
				'mode': isPosttype ? 'posttype_import' : 'post_import',
				'data': data
			}, 
			function(response) {
				console.log(response);
				
				// successfull
				if ( response.indexOf('success::') > -1 ) {
					var msg = response.split( 'success::' )[1];

					if ( isPosttype ) {

						// on succes we usually get the new conflicts array
						try {
							data.conflicts = JSON.parse( msg );
						} catch(e) {
							console.error(e);
							greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": "Error parsing JSON. See console for details." });
							return;
						}
						console.log( data );
						greyd.postExport.postImport( data, false );

					} else {

						// use for development
						// greyd.backend.triggerOverlay( true, { "type": "success", "css": mode } );return;
	
						// trigger overlay
						greyd.backend.triggerOverlay( true, { "type": "reload", "css": mode } );
					}
				}
				// complete with error
				else if ( response.indexOf('error::') > -1 ) {
					var msg = response.split( 'error::' )[1];
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": msg });
				}
				// unknown state
				else {
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": response});
				}
			}
		);
	}


	/**
	 * Start the switch theme dialog (row action)
	 * 
	 * @param {object} elem 
	 */
	this.openSwitchTemplateTheme = function( elem ) {

		const td  = $(elem).closest("td.title");
		let title = '';

		if ( td.find('.filename').length ) {
			title = td.find('.filename').clone().children().remove().end().text();
		}
		else {
			title = td.find("strong a").text();
		}

		greyd.backend.confirm( 'switch_template_theme', title.trim(), greyd.postExport.switchTemplateTheme, [ elem ] );
	}

	/**
	 * Switch the theme of a template or template part
	 * 
	 * @param {object} elem 
	 */
	this.switchTemplateTheme = function( elem ) {

		const mode    = 'switch_template_theme';
		const post_id = $(elem).data('post_id');
		
		$.post(
			greyd.ajax_url, {
				'action': 'greyd_ajax',
				'_ajax_nonce': greyd.nonce,
				'mode': mode,
				'data': {
					post_id: post_id,
					switch_references_in_content: document.getElementById('switch_references_in_content').checked
				}
			}, 
			function(response) {
				console.log(response);
				
				// successfull
				if ( response.indexOf('success::') > -1 ) {

					// use for development
					// greyd.backend.triggerOverlay( true, { "type": "success", "css": mode } );return;

					// trigger overlay
					greyd.backend.triggerOverlay( true, { "type": "reload", "css": mode } );
				}
				else if ( response.indexOf('error::') > -1 ) {
					var msg = response.split( 'error::' )[1];
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": msg });
				}
				else {
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": response});
				}
			}
		);
	}


	/**
	 * Start the switch theme dialog (row action)
	 * 
	 * @param {object} elem 
	 */
	this.openRenameTemplate = function( elem ) {

		const td  = $(elem).closest("td.title");
		let title = '';

		if ( td.find('.filename').length ) {
			title = td.find('.filename').clone().children().remove().end().text();
		}
		else {
			title = td.find("strong a").text();
		}

		const post_title = $(elem).data('post_title');
		const post_name  = $(elem).data('post_name');

		$('#rename_template_form input[name="new_post_title"]').val( post_title );
		$('#rename_template_form input[name="new_post_name"]').val( post_name );

		greyd.backend.confirm( 'rename_template', title.trim(), greyd.postExport.renameTemplate, [ elem ] );
	}

	/**
	 * Switch the theme of a template or template part
	 * 
	 * @param {object} elem 
	 */
	this.renameTemplate = function( elem ) {

		const mode    = 'rename_template';
		const post_id = $(elem).data('post_id');
		const data   = {
			post_id: post_id,
			post_title: $('#rename_template_form input[name="new_post_title"]').val(),
			post_name: $('#rename_template_form input[name="new_post_name"]').val()
		};

		console.log( data );
		
		$.post(
			greyd.ajax_url, {
				'action': 'greyd_ajax',
				'_ajax_nonce': greyd.nonce,
				'mode': mode,
				'data': data
			}, 
			function(response) {
				console.log(response);
				
				// successfull
				if ( response.indexOf('success::') > -1 ) {

					// use for development
					// greyd.backend.triggerOverlay( true, { "type": "success", "css": mode } );return;

					// trigger overlay
					greyd.backend.triggerOverlay( true, { "type": "reload", "css": mode } );
				}
				else if ( response.indexOf('error::') > -1 ) {
					var msg = response.split( 'error::' )[1];
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": msg });
				}
				else {
					greyd.backend.triggerOverlay(true, {"type": "fail", "css": mode, "replace": response});
				}
			}
		);
	}
}