<?php
/**
 * @deprecated since 2.0
 * 
 * @todo can be removed in future versions
 *       It's only here tp prevent merge issues.
 */
namespace Greyd;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * disable if plugin wants to run standalone
 * Standalone setup not possible - needs posttypes.
 */
if ( ! class_exists( 'Greyd\Admin' ) ) {
	// reject activation
	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	$plugin_name = get_plugin_data( __FILE__ )['Name'];
	deactivate_plugins( plugin_basename( __FILE__ ) );
	// return reject message
	die( sprintf( '%s can not be activated as standalone Plugin.', $plugin_name ) );
}

new Post_Export( $config );
class Post_Export {

	/**
	 * Holds global config args.
	 */
	private $config;

	/**
	 * Holds all WP_Post objects for either ex- or import.
	 *
	 * @var array
	 */
	public static $posts = array();

	/**
	 * Holds all WP_Media objects for either ex- or import.
	 *
	 * @var array
	 */
	public static $media = array();

	/**
	 * Holds the relative path to the export folder.
	 *
	 * @var string
	 */
	public static $basic_path = null;

	/**
	 * Holds all temporarily created taxonomies
	 *
	 * @since 1.7
	 *
	 * @var array
	 */
	public static $temporary_taxonomies = array();

	/**
	 * Whether logs are echoed.
	 * Usually set via function @see enable_logs();
	 * Logs are especiallly usefull when debugging ajax actions.
	 *
	 * @var bool
	 */
	public static $logs = false;

	/**
	 * Holds the current language code
	 *
	 * @var string
	 */
	public static $language_code = null;

	/**
	 * Constructor
	 */
	public function __construct( $config ) {

		$this->config = (object) $config;

		// Export
		add_action( 'greyd_ajax_mode_post_export', array( $this, 'handle_export' ) );
		add_filter( 'greyd_export_post_meta-dynamic_meta', array( $this, 'prepare_dynamic_meta' ), 10, 3 );

		// Import
		add_action( 'greyd_ajax_mode_check_post_import', array( $this, 'check_import' ) );
		add_action( 'greyd_ajax_mode_post_import', array( $this, 'handle_import' ) );
		add_action( 'greyd_ajax_mode_posttype_import', array( $this, 'handle_posttype_import' ) );
		add_filter( 'greyd_import_post_meta-dynamic_meta', array( $this, 'set_dynamic_meta' ), 10, 2 );

		// UI
		add_filter( 'greyd_overlay_contents', array( $this, 'add_overlay_contents' ) );
		add_filter( 'page_row_actions', array( $this, 'add_export_row_action' ), 10, 2 );
		add_filter( 'post_row_actions', array( $this, 'add_export_row_action' ), 10, 2 );
		add_filter( 'media_row_actions', array( $this, 'add_export_row_action' ), 10, 2 );
		add_action( 'admin_enqueue_scripts', array( $this, 'add_import_page_title_action' ) );
		add_action( 'admin_init', array( $this, 'add_bulk_actions' ) );
		add_action( 'admin_notices', array( $this, 'display_transient_notice' ) );

		// deep debug
		add_action( 'admin_init', array( $this, 'maybe_enable_debug_mode' ) );
	}


	/**
	 * =================================================================
	 *                          EXPORT
	 * =================================================================
	 */

	/**
	 * Handle the ajax export action
	 *
	 * @action 'greyd_ajax_mode_post_export'
	 *
	 * @param array $data   holds the $_POST['data']
	 */
	public function handle_export( $data ) {

		self::enable_logs();

		do_action( 'greyd_post_export_log', "\r\n\r\n" . 'HANDLE EXPORT' . "\r\n", $data );

		$post_id = isset( $data['post_id'] ) ? $data['post_id'] : '';
		$args    = array(
			'append_nested'  => isset( $data['nested'] ) ? true : false,
			'whole_posttype' => isset( $data['whole_posttype'] ) ? true : false,
			'resolve_menus'  => isset( $data['resolve_menus'] ) ? true : false,
			'translations'   => isset( $data['translations'] ) ? true : false,
		);
		if ( ! empty( $post_id ) ) {

			// export post
			$post = self::export_post( $post_id, $args );

			/**
			 * Create the export ZIP-archive.
			 * If posts & media are null, write_export_file() uses the class vars.
			 */
			$posts    = $media = $args['append_nested'] ? null : array();
			$filename = self::write_filename( $post, $args );
			$filepath = $this->write_export_file( $filename, $posts, $media );

			if ( ! $filepath ) {
				self::error( __( "The export file could not be written.", 'greyd_hub' ) );
			}

			self::success( self::convert_content_dir_to_path( $filepath ) );
		}
		self::error( __( "No valid Post ID could found.", 'greyd_hub' ) );
	}

	/**
	 * Get post with all its meta, taxonomies, media etc.
	 *
	 * @param int   $post_id
	 * @param array $args       Arguments.
	 *
	 * @return WP_Post Advanced WP_Post object preparred for export.
	 */
	public static function export_post( $post_id, $args = array() ) {

		$default_args = array(
			'append_nested'  => true,
			'whole_posttype' => false,
			'resolve_menus'  => true,
			'translations'   => false,
		);
		$args         = array_merge( $default_args, $args );

		// reset the class vars
		self::$posts = array();
		self::$media = array();

		// prepare the export based on this post
		$post = self::prepare_post( $post_id, $args );

		// if this is a posttype, enable bulk export
		if (
			class_exists( '\Greyd\Posttypes\Posttype_Helper' )
			&& $args['whole_posttype']
			&& $post->post_type === 'tp_posttypes'
		) {

			$post_type = \Greyd\Posttypes\Posttype_Helper::get_posttype_slug_from_title( $post->post_title );

			do_action( 'greyd_post_export_log', "\r\n" . "EXPORT MULTIPLE POSTS from posttype '" . $post_type . "'\r\n" );

			$post_ids = get_posts(
				array(
					'numberposts' => -1,
					'post_type'   => $post_type,
					'fields'      => 'ids',
				)
			);

			if ( $post_ids && is_array( $post_ids ) ) {
				foreach ( $post_ids as $_post_id ) {
					self::prepare_post( $_post_id, $args );
				}
			}
		}

		return $post;
	}

	/**
	 * Prepare post for export.
	 *
	 * This function attaches all post meta options, taxonomy terms
	 * and other post properties to the WP_Post object. It also gets
	 * all nested templates, forms, images etc. inside the post content,
	 * dynamic meta or set as thumbnail and prepares them for later
	 * replacement.
	 *
	 * @param int   $post_id    WP_Post ID.
	 * @param array $args       Arguments.
	 *
	 * @return object|bool Preparred WP_Post object on success. False on failure.
	 *
	 * This function automatically sets the following class vars.
	 * Use them to export all nested posts at once.
	 *
	 * @var array class::$posts     Array of all preparred post objects.
	 * @var array class::$media     Array of all media files.
	 */
	public static function prepare_post( $post_id, $args = array() ) {

		// return if we're already processed this post
		if ( isset( self::$posts[ $post_id ] ) ) {
			do_action( 'greyd_post_export_log', "\r\n" . "Post '$post_id' already processed" );
			return self::$posts[ $post_id ];
		}

		// arguments
		$default_args = array(
			'append_nested'  => true,
			'whole_posttype' => false,
			'resolve_menus'  => true,
			'translations'   => false,
		);
		$args         = array_merge( $default_args, $args );

		// get WP_Post
		if ( ! $post = get_post( $post_id ) ) {
			return false;
		}

		do_action( 'greyd_post_export_log', "\r\n\r\n|\r\n|  PREPARE POST {$post_id} '{$post->post_title}'\r\n|" );

		// do_action( "greyd_post_export_log", esc_attr($post->post_content) );

		/**
		 * First we append the post object to the class var. We do this to
		 * kind of 'reserve' the position of the post inside the array.
		 */
		self::$posts[ $post_id ] = $post;

		/**
		 * Prepare the $post object.
		 *
		 * After that, it has some additional properties:
		 *
		 * @property array nested   Nested post arrays keyed by post_id:
		 *      {{post_id}} => array(
		 *          @property int    ID
		 *          @property string post_name
		 *          @property string post_type
		 *          @property string front_url
		 *          @property string file_path (only for attachments)
		 *      )
		 * @property WP_Term[] nested_terms   Nested terms keyed by term_id.
		 * @property array meta     All post meta options as arrays.
		 * @property array terms    All post terms sorted by the taxonomy.
		 * @property array media    If this is an attachment:
		 *      @property string name
		 *      @property string path
		 *      @property string url
		 * @property array language Translation information:
		 *      @property string code
		 *      @property string tool
		 *      @property array  post_ids
		 *      @property array  args
		 */
		$post               = self::prepare_nested_posts( $post );
		$post               = self::prepare_nested_terms( $post );
		$post->post_content = self::prepare_strings( $post->post_content, $post_id );
		$post->meta         = self::prepare_meta( $post_id, $args );
		$post->terms        = self::prepare_taxonomy_terms( $post );
		$post->media        = self::prepare_media( $post_id );
		$post->language     = self::prepare_language( $post, $args );

		/**
		 * Resolve menus
		 */
		if ( $args['resolve_menus'] ) {
			$post->post_content = self::resolve_menus( $post->post_content, $post_id );
		}

		/**
		 * Now we update the post in the class var.
		 */
		self::$posts[ $post_id ] = $post;

		/**
		 * Let's save the media to the class var.
		 *
		 * We need this, so write_export_file() can access all the files at once.
		 */
		if ( ! empty( $post->media ) ) {
			self::$media[ $post_id ] = $post->media;
		}

		/**
		 * The post thumbnail always has to be included in the export,
		 * because WP references it with an ID, therefore it needs to
		 * be accessable as a post.
		 */
		if ( $thumbnail_id = get_post_thumbnail_id( $post ) ) {
			self::prepare_post( $thumbnail_id, $args );
		}

		/**
		 * Now we loop through all the nested posts (if the option is set).
		 */
		if ( $args['append_nested'] ) {
			foreach ( $post->nested as $nested_id => $nested_name ) {
				self::prepare_post( $nested_id, $args );
			}
		}

		/**
		 * Now we loop through all translations of this post (if the option is set).
		 */
		if ( $args['translations'] && count( $post->language['post_ids'] ) ) {
			foreach ( $post->language['post_ids'] as $lang => $translated_post_id ) {
				self::prepare_post( $translated_post_id, $args );
			}
		}

		return $post;
	}

	/**
	 * Prepare nested posts in content for export
	 *
	 * @param WP_Post $post
	 *
	 * @return object Preparred with new property: 'nested'
	 */
	public static function prepare_nested_posts( $post ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Prepare nested posts.' );

		$post->nested = array();
		$content      = is_object( $post ) ? $post->post_content : '';

		if ( empty( $content ) ) {
			do_action( 'greyd_post_export_log', '=> post content is empty' );
			return $post;
		}

		// get regex patterns
		$replace_id_patterns = (array) self::regex_nested_posts( $post->ID, $post );

		foreach ( $replace_id_patterns as $key => $pattern ) {

			// doing it wrong
			if (
				! isset( $pattern['search'] ) ||
				! is_array( $pattern['search'] ) ||
				count( $pattern['search'] ) < 2 ||
				! isset( $pattern['replace'] ) ||
				! is_array( $pattern['replace'] )
			) {
				continue;
			}

			$match_regex = '/' . implode( '([\da-z\-\_]+?)', $pattern['search'] ) . '/';
			$regex_group = isset( $pattern['group'] ) ? (int) $pattern['group'] : 2;
			// do_action( "greyd_post_export_log", "  - test regex: ".esc_attr($match_regex) );

			// search for all occurrences
			preg_match_all( $match_regex, $content, $matches );
			$found_posts = isset( $matches[ $regex_group ] ) ? $matches[ $regex_group ] : null;
			if ( ! empty( $found_posts ) ) {

				do_action( 'greyd_post_export_log', "\r\n" . "  Replace '" . $key . "':" );
				foreach ( $found_posts as $name_or_id ) {

					$nested_post = null;
					$nested_id = $name_or_id;

					// IDs
					if ( is_numeric( $name_or_id ) ) {
						$nested_post = get_post( $name_or_id );
					} else {
						if ( isset( $pattern['post_type'] ) ) {
							$args = (object) array(
								'post_name' => $name_or_id,
								'post_type' => $pattern['post_type'],
							);
						} else {
							$args = $name_or_id;
						}
						// get post
						$nested_post = self::get_post_by_name_and_type( $args );
						if ( $nested_post ) {
							$nested_id = $nested_post->ID;
						}
					}

					if ( ! $nested_post ) {
						do_action( 'greyd_post_export_log', "  - post with id or name '$name_or_id' could not be found." );
						continue;
					}

					$search_regex   = '/' . implode( $nested_id, $pattern['search'] ) . '/';
					$replace_string = implode( '{{' . $nested_id . '}}', $pattern['replace'] );

					// replace in $content
					$content = preg_replace( $search_regex, $replace_string, $content );

					// collect in $post->nested
					if ( ! isset( $post->nested[ $nested_id ] ) ) {
						if ( $nested_post ) {


							if ( $nested_post->post_type === 'attachment' ) {
								$post->nested[ $nested_id ] = array(
									'ID'        => $nested_id,
									'post_name' => $nested_post->post_name,
									'post_type' => $nested_post->post_type,
									'front_url' => wp_get_attachment_url( $nested_id ),
									'file_path' => get_attached_file( $nested_id ),
								);

							} else {
								$post->nested[ $nested_id ] = array(
									'ID'        => $nested_id,
									'post_name' => $nested_post->post_name,
									'post_type' => $nested_post->post_type,
									'front_url' => get_permalink( $nested_id ),
								);
							}

							// also replace the front url inside the post content
							$content = str_replace( $post->nested[ $nested_id ]['front_url'], '{{' . $nested_id . '-front-url}}', $content );

							do_action(
								'greyd_post_export_log',
								sprintf(
									"  - nested post '%s' attached for export.\r\n     * ID: %s\r\n     * TYPE: %s\r\n     * URL: %s",
									$nested_post->post_name,
									$nested_id,
									$nested_post->post_type,
									$post->nested[ $nested_id ]['front_url']
								)
							);
						} else {
							$post->nested[ $nested_id ] = null;
						}
					}
				}
			}
		}
		$post->post_content = $content;

		do_action( 'greyd_post_export_log', '=> nested elements were preparred' );

		return $post;
	}

	/**
	 * Prepare nested terms inside the post content
	 *
	 * @since 1.7
	 *
	 * @return object Preparred with new property: 'nested_terms'
	 */
	public static function prepare_nested_terms( $post ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Prepare nested terms.' );

		$post->nested_terms = array();
		$content            = is_object( $post ) ? $post->post_content : '';

		if ( empty( $content ) ) {
			do_action( 'greyd_post_export_log', '=> post content is empty' );
			return $post;
		}

		// register temporary taxonomies to prevent errors
		self::get_dynamic_taxonomies();

		// get patterns
		$replace_id_patterns = (array) self::regex_nested_terms( $post->ID, $post );

		foreach ( $replace_id_patterns as $key => $pattern ) {

			// doing it wrong
			if (
				! isset( $pattern['search'] ) ||
				! is_array( $pattern['search'] ) ||
				count( $pattern['search'] ) < 2 ||
				! isset( $pattern['replace'] ) ||
				! is_array( $pattern['replace'] )
			) {
				continue;
			}

			$match_regex = '/' . implode( '([\da-z\-\_]+?)', $pattern['search'] ) . '/';
			$regex_group = isset( $pattern['group'] ) ? (int) $pattern['group'] : 2;
			// do_action( "greyd_post_export_log", "  - test regex: ".esc_attr($match_regex) );

			// search for all occurrences
			preg_match_all( $match_regex, $content, $matches );
			$found_terms = isset( $matches[ $regex_group ] ) ? $matches[ $regex_group ] : null;
			if ( ! empty( $found_terms ) ) {

				do_action( 'greyd_post_export_log', '  - replace ' . $key . ':' );
				foreach ( $found_terms as $term_id ) {

					// default value for term_ids
					if ( $term_id == 0 || $term_id == -1 ) {
						continue;
					}

					$search_regex   = '/' . implode( $term_id, $pattern['search'] ) . '/';
					$replace_string = implode( '{{' . $term_id . '}}', $pattern['replace'] );

					// replace in $content
					$content = preg_replace( $search_regex, $replace_string, $content );

					// collect in $post->nested_terms
					if ( ! isset( $post->nested_terms[ $term_id ] ) ) {
						$term_object = get_term( $term_id );
						if ( ! $term_object || is_wp_error( $term_object ) ) {
							do_action( 'greyd_post_export_log', "  - term with id '$term_id' could not be found." );
							$post->nested_terms[ $term_id ] = null;
						} else {
							do_action( 'greyd_post_export_log', "  - term with id '$term_id' found.", $term_object );
							$post->nested_terms[ $term_id ] = $term_object;
						}
					}
				}
			}
		}
		$post->post_content = $content;

		do_action( 'greyd_post_export_log', '=> nested terms were preparred' );

		return $post;
	}

	/**
	 * Replace strings for export
	 *
	 * @param string $subject
	 * @param int    $post_id
	 *
	 * @return string $subject
	 */
	public static function prepare_strings( $subject, $post_id, $log = true ) {

		if ( empty( $subject ) ) {
			return $subject;
		}

		if ( $log ) {
			do_action( 'greyd_post_export_log', "\r\n" . 'Prepare strings.' );
		}

		// get patterns
		$replace_strings = (array) self::regex_nested_strings( $subject, $post_id );
		foreach ( $replace_strings as $name => $string ) {
			$subject = str_replace( $string, '{{' . $name . '}}', $subject );
			// if ($log) do_action( "greyd_post_export_log", sprintf("  - '%s' was preparred for export.", $name ) );
		}
		if ( $log ) {
			do_action( 'greyd_post_export_log', '=> strings were preparred' );
		}
		return $subject;
	}

	/**
	 * Prepare meta for consumption
	 *
	 * @param int   $post_id Post ID.
	 * @param array $args Export arguments.
	 *
	 * @return array
	 */
	public static function prepare_meta( $post_id, $args = array() ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Prepare post meta.' );

		$meta          = get_post_meta( $post_id );
		$prepared_meta = array();

		// Transfer all meta
		foreach ( $meta as $meta_key => $meta_array ) {
			foreach ( $meta_array as $meta_value ) {

				// don't prepare blacklisted meta
				if ( in_array( $meta_key, self::blacklisted_meta(), true ) ) {
					continue;
				}
				// skip certain mety keys
				elseif ( self::maybe_skip_meta_option( $meta_key, $meta_value ) ) {
					continue;
				}

				$meta_value = maybe_unserialize( $meta_value );

				/**
				 * Add filter to modify the post meta value before export
				 *
				 * @filter 'greyd_export_post_meta-{{meta_key}}'
				 */
				$meta_value = apply_filters( 'greyd_export_post_meta-' . $meta_key, $meta_value, $post_id, $args );

				$prepared_meta[ $meta_key ][] = $meta_value;
			}
		}

		do_action( 'greyd_post_export_log', '=> post meta prepared' );
		return $prepared_meta;
	}

	/**
	 * Modify dynamic meta value before export
	 *
	 * @filter 'greyd_export_post_meta-dynamic_meta'
	 *
	 * @param mixed $meta_value
	 * @param int   $post_id
	 * @param array $args       Export arguments
	 *
	 * @return mixed $meta_value
	 */
	public function prepare_dynamic_meta( $meta_value, $post_id, $args = array() ) {
		do_action( 'greyd_post_export_log', '  - prepare dynamic meta for export' );

		if ( ! class_exists( '\Greyd\Posttypes\Posttype_Helper' ) ) {
			do_action( 'greyd_post_export_log', '  - Posttype class does not exist.' );
			return $meta_value;
		}

		$post     = get_post( $post_id );
		$posttype = \Greyd\Posttypes\Posttype_Helper::get_dynamic_posttype_by_slug( $post->post_type );
		if ( $posttype && isset( $posttype['fields'] ) ) {
			foreach ( (array) $posttype['fields'] as $field ) {
				$current_value = null;
				if ( is_object( $field ) ) {
					$field = (array) $field;
				}

				// media files
				if ( $field['type'] === 'file' ) {
					$current_value = isset( $meta_value[ $field['name'] ] ) ? $meta_value[ $field['name'] ] : null;
					if ( is_numeric( $current_value ) ) {
						do_action( 'greyd_post_export_log', sprintf( "  - file value for '%s' is an ID (%s) and was prepared for export.", $field['name'], $current_value ) );
						if ( isset( $args['append_nested'] ) && $args['append_nested'] ) {
							do_action( 'greyd_post_export_log', "\r\n---------------------" );
							self::prepare_post( $current_value, $args );
							do_action( 'greyd_post_export_log', "\r\n---------------------\r\n" );
						}
						$current_value = '{{' . $current_value . '}}';
					}
				}
				// url or tinymce editor
				elseif ( $field['type'] === 'url' || $field['type'] === 'text_html' ) {
					$current_value = isset( $meta_value[ $field['name'] ] ) ? $meta_value[ $field['name'] ] : '';
					$current_value = self::prepare_strings( $current_value, $post_id, false );
					do_action( 'greyd_post_export_log', sprintf( "  - content of field '%s' was preparred for export", $field['name'] ) );
				}

				if ( $current_value ) {
					$meta_value[ $field['name'] ] = $current_value;
				}
			}
		}
		return $meta_value;
	}

	/**
	 * Format taxonomy terms for consumption
	 *
	 * @param  int $post_id Post ID.
	 * @return array
	 */
	public static function prepare_taxonomy_terms( $post ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Prepare taxonomy terms.' );

		$taxonomy_terms = array();
		$taxonomies     = get_object_taxonomies( $post );

		// if we haven't found any taxonomies we create them temporarily.
		if ( empty( $taxonomies ) ) {
			do_action( 'greyd_post_export_log', '  - object taxonomies are empty' );
			$taxonomies = self::get_dynamic_taxonomies( $post->post_type );
		}

		if ( empty( $taxonomies ) ) {
			do_action( 'greyd_post_export_log', '=> no taxonomy terms found' );
			return array();
		}

		foreach ( $taxonomies as $taxonomy ) {

			/**
			 * Retrieve post terms directly from the database.
			 *
			 * @since 1.2.7
			 *
			 * WPML attaches a lot of filters to the function wp_get_object_terms(). This results
			 * in terms of the wrong language beeing attached to a post export. This function performs
			 * way more consistent in all tests. Therefore it completely replaced it in this class.
			 *
			 * @deprecated since 1.2.7: $terms = wp_get_object_terms( $post->ID, $taxonomy );
			 */
			$terms = self::get_post_taxonomy_terms( $post->ID, $taxonomy );

			if ( empty( $terms ) ) {
				$taxonomy_terms[ $taxonomy ] = array();
				do_action( 'greyd_post_export_log', "  - no terms found for taxonomy '$taxonomy'." );
			} else {
				$taxonomy_terms[ $taxonomy ] = $terms;
				$count                       = count( $terms );
				do_action(
					'greyd_post_export_log',
					"  - {$count} " . ( $count > 1 ? 'terms' : 'term' ) . " of taxonomy '$taxonomy' prepared:\r\n    - " . implode(
						"\r\n    - ",
						array_map(
							function( $term ) {
								return "{$term->name} (#{$term->term_id})";
							},
							$terms
						)
					)
				);
			}
		}

		do_action( 'greyd_post_export_log', '=> all taxonomy terms prepared' );
		return $taxonomy_terms;
	}

	/**
	 * Format media items for export
	 *
	 * @param  int $post_id Post ID.
	 * @return array        Array with filename as key and url & path as values.
	 */
	public static function prepare_media( $post_id ) {
		$return = array();

		if ( $file_path = get_attached_file( $post_id ) ) {
			$file_url  = wp_get_attachment_url( $post_id );
			$file_name = wp_basename( $file_path );
			$return    = array(
				'name' => $file_name,
				'path' => $file_path,
				'url'  => $file_url,
			);
			do_action( 'greyd_post_export_log', "\r\n" . sprintf( "The file '%s' was added to the post.", $file_name ) );
		}
		return $return;
	}

	/**
	 * Get all necessary language information
	 *
	 * @since 1.6
	 *
	 * @param WP_Post $post     The post object.
	 * @param array   $args       Export arguments.
	 *
	 * @return array $language
	 *      @property string code       The post’s language code (eg. 'en')
	 *      @property string tool       The plugin used to setup the translation.
	 *      @property array  post_ids   All translated post IDs keyed by language code.
	 *      @property array  args       Additional arguments (depends on the tool)
	 */
	public static function prepare_language( $post, $args = array() ) {

		do_action( 'greyd_post_export_log', "\r\n" . 'Prepare post language info.' );

		$language = array(
			'code'     => null,
			'tool'     => null,
			'post_ids' => array(),
			'args'     => array(),
		);

		$tool = self::get_translation_tool();

		switch ( $tool ) {
			case 'wpml':
				$language['tool'] = 'wpml';

				$language_details = self::get_post_language_info( $post );
				if ( $language_details && isset( $language_details['language_code'] ) ) {
					$language['code'] = $language_details['language_code'];
					$language['args'] = $language_details;
					do_action( 'greyd_post_export_log', "  - post has language '{$language['code']}'" );
				}

				// prepare post id's if translations are included
				if ( $args['translations'] ) {
					$language['post_ids'] = self::get_translated_post_ids( $post );
					if ( ! empty( $language['post_ids'] ) ) {
						do_action( 'greyd_post_export_log', '  - translations of this post prepared: ' . implode( ', ', $language['post_ids'] ) );
					}
				}
				break;

			default:
				// $language['code'] = self::get_wp_lang(); // this leads to unwanted errors...
				break;
		}

		do_action( 'greyd_post_export_log', '=> post language info prepared' );
		return $language;
	}

	/**
	 * Resolve menus.
	 *
	 * @param string $subject
	 * @param  int    $post_id Post ID.
	 *
	 * @return string $subject
	 */
	public static function resolve_menus( $subject, $post_id ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Resolve nested menus.' );

		// get patterns
		$replace_menu_patterns = (array) self::regex_nested_menus( $post_id );

		foreach ( $replace_menu_patterns as $name => $pattern ) {

			// doing it wrong
			if (
				! isset( $pattern['search'] ) ||
				! is_array( $pattern['search'] ) ||
				count( $pattern['search'] ) < 2 ||
				! isset( $pattern['replace'] ) ||
				! is_array( $pattern['replace'] )
			) {
				continue;
			}

			$match_regex = '/' . implode( '([\da-z\-\_]+?)', $pattern['search'] ) . '/';
			$regex_group = isset( $pattern['group'] ) ? (int) $pattern['group'] : 2;

			// search for all occurrences
			preg_match_all( $match_regex, $subject, $matches );
			$found_menus = isset( $matches[2] ) ? $matches[2] : null;
			if ( ! empty( $found_menus ) ) {

				do_action( 'greyd_post_export_log', sprintf( "  - %s menu(s) found: '%s'", count( $found_menus ), implode( "', '", $found_menus ) ) );

				foreach ( $found_menus as $menu_name ) {

					if ( $menu_name === 'custom' ) {
						continue;
					}

					$vc_menu  = array();
					$response = wp_get_nav_menus( array( 'slug' => $menu_name ) );
					if ( count( $response ) ) {

						$menu_items = wp_get_nav_menu_items( $response[0] );
						foreach ( $menu_items as $item ) {

							$level   = $item->menu_item_parent == 0 ? '1' : '2';
							$vc_link = array(
								'url:' . self::vc_encode( $item->url ),
								'title:' . self::vc_encode( $item->title ),
								! empty( $item->target ) ? 'target:%20_blank' : '',
								'',
							);
							$vc_link = implode( '|', $vc_link );

							$vc_menu[] = (object) array(
								'menu_link_level' => $level,
								'menu_link'       => $vc_link,
							);
						}

						$search_regex   = '/' . implode( $menu_name, $pattern['search'] ) . '/';
						$replace_string = count( $vc_menu ) ? 'custom" custom_menu="' . urlencode( json_encode( $vc_menu ) ) : $menu_name;
						$replace_string = implode( $replace_string, $pattern['replace'] );

						// replace
						$subject = preg_replace( $search_regex, $replace_string, $subject );

						do_action( 'greyd_post_export_log', sprintf( "  - menu '%s' was resolved.", $menu_name ) );
					}
				}
			}
		}

		do_action( 'greyd_post_export_log', '=> nested menus were resolved' );

		return $subject;
	}

	/**
	 * Handle export bulk action
	 *
	 * set via 'add_bulk_actions'
	 *
	 * @param string $sendback The redirect URL.
	 * @param string $doaction The action being taken.
	 * @param array  $items    Array of IDs of posts.
	 */
	public function handle_export_bulk_action( $sendback, $doaction, $items ) {
		if ( count( $items ) === 0 ) {
			return $sendback;
		}
		if ( $doaction !== 'greyd_export' && $doaction !== 'greyd_export_multilanguage' ) {
			return $sendback;
		}

		$args = array(
			'append_nested'  => true,
			'translations'   => $doaction === 'greyd_export_multilanguage'
		);

		// prepare all post data
		foreach ( $items as $post_id ) {
			$post = self::prepare_post( $post_id, $args );
		}

		$filename = self::write_filename( $items );
		$filepath = $this->write_export_file( $filename );

		if ( $filepath ) {
			$href = self::convert_content_dir_to_path( $filepath );
			// $sendback = add_query_arg( 'download', $href, $sendback );
			$sendback = $href;
		} else {
			// set transient to display admin notice
			set_transient( 'greyd_transient_notice', 'error::' . __( "The export file could not be written.", 'greyd_hub' ) );
		}

		return $sendback;
	}

	/**
	 * Write export as a .zip archive
	 *
	 * @param string $filename  Name of the final archive.
	 * @param array  $posts     Content of posts.json inside archive.
	 *                          Defaults to class var $posts.
	 * @param array  $media     Media files.
	 *                          Defaults to class var $media.
	 *
	 * @return mixed $path      Path to the archive. False on failure.
	 */
	public function write_export_file( $filename, $posts = null, $media = null ) {

		do_action( 'greyd_post_export_log', "\r\n" . "Write export .zip archive '$filename'" );

		$posts_data = $posts ? $posts : self::$posts;
		$media_data = $media ? $media : self::$media;

		// set monthly folder
		$folder = date( 'y-m' );
		$path   = self::get_file_path( $folder );

		// write the temporary posts.json file
		$json_name = 'posts.json';
		$json_path = $path . $json_name;
		$json_file = fopen( $json_path, 'w' );

		if ( ! $json_file ) {
			return false;
		}

		fwrite( $json_file, json_encode( $posts_data, JSON_PRETTY_PRINT ) );
		fclose( $json_file );

		// create a zip archive
		$zip      = new \ZipArchive();
		$zip_name = str_replace( '.zip', '', $filename ) . '.zip';
		$zip_path = $path . $zip_name;

		// delete previous zip archive
		if ( file_exists( $zip_path ) ) {
			unlink( $zip_path );
		}

		// add files to the zip archive
		if ( $zip->open( $zip_path, \ZipArchive::CREATE ) ) {

			// copy the json to the archive
			$zip->addFile( $json_path, $json_name );

			// add media
			$zip->addEmptyDir( 'media' );
			if ( is_array( $media_data ) && count( $media_data ) > 0 ) {
				foreach ( $media_data as $post_id => $_media ) {
					if ( isset( $_media['path'] ) && isset( $_media['name'] ) ) {
						$zip->addFile( $_media['path'], 'media/' . $_media['name'] );
					}
				}
			}

			$zip->close();
		} else {
			return false;
		}

		// delete temporary json file
		unlink( $json_path );

		// return path to file
		return $zip_path;
	}

	/**
	 * =================================================================
	 *                          IMPORT
	 * =================================================================
	 */

	/**
	 * Check the uploaded file
	 *
	 * @action 'greyd_ajax_mode_check_post_import'
	 *
	 * @param array $data   holds the $_FILES['data']
	 */
	public function check_import( $data ) {

		self::enable_logs();

		do_action( 'greyd_post_export_log', "\r\n\r\n" . 'CHECK IMPORT POST DATA' . "\r\n\r\n", $data );

		set_time_limit( 5000 );

		// check for errors
		if ( isset( $data['error'] ) && $data['error'] > 0 ) {
			$file_errors = array(
				1 => sprintf(
					__( "The uploaded file exceeds the server's maximum file limit (max %s MB). The limit is defined in the <u>php.ini</u> file.", 'greyd_hub' ),
					intval( ini_get( 'upload_max_filesize' ) )
				),
				2 => __( "The uploaded file exceeds the allowed file size of the html form.", 'greyd_hub' ),
				3 => __( "The uploaded file was only partially uploaded.", 'greyd_hub' ),
				4 => __( "No file was uploaded.", 'greyd_hub' ),
				6 => __( "Missing a temporary folder.", 'greyd_hub' ),
				7 => __( "Failed to save the file.", 'greyd_hub' ),
				8 => __( "The file was stopped while uploading.", 'greyd_hub' ),
			);
			self::error( $file_errors[ $data['error'] ] );
		}

		// file info
		$filename = $data['name'];
		$filepath = $data['tmp_name'];
		$filetype = $data['type'];

		// check filetype
		if ( $filetype != 'application/zip' && $filetype != 'application/x-zip-compressed' ) {
			self::error( __( "Please select a valid ZIP archive.", 'greyd_hub' ) );
		}
		do_action( 'greyd_post_export_log', '  - file is valid zip.' );

		// create tmp zip
		$new_file = self::get_file_path( 'tmp' ) . $filename;
		$result   = move_uploaded_file( $filepath, $new_file );
		do_action( 'greyd_post_export_log', sprintf( '  - temporary file „%s“ created.', $new_file ) );

		// get post data
		$post_data = self::get_zip_posts_file_contents( $new_file );

		if ( ! is_array( $post_data ) ) {
			self::error( $post_data );
		}

		// get conflicting posts
		$conflicts = self::import_get_conflict_posts_for_backend_form( $post_data );

		if ( $conflicts ) {
			$return = $conflicts;
		}
		// return file name when no conflicts found
		else {
			$return = $filename;
		}

		self::success( $return );
	}

	/**
	 * Handle the ajax import action.
	 *
	 * @action 'greyd_ajax_mode_post_import'
	 *
	 * @param array $data   holds the $_POST['data']
	 */
	public function handle_import( $data ) {

		self::enable_logs();

		do_action( 'greyd_post_export_log', "\r\n\r\n" . 'HANDLE IMPORT' . "\r\n", $data );

		set_time_limit( 5000 );

		$filename = isset( $data['filename'] ) ? $data['filename'] : '';

		if ( empty( $filename ) ) {
			self::error( __( "The file name is empty.", 'greyd_hub' ) );
		}

		// get post data
		$zip_file  = self::get_file_path( 'tmp' ) . $filename;
		$post_data = self::get_zip_posts_file_contents( $zip_file );

		// error
		if ( ! is_array( $post_data ) ) {
			self::error( $post_data );
		}

		// get conflicts with current posts
		$conflicts        = isset( $data['conflicts'] ) ? (array) $data['conflicts'] : array();
		$conflict_actions = self::import_get_conflict_actions_from_backend_form( $conflicts );

		$result = self::import_posts( $post_data, $conflict_actions, $zip_file );

		if ( is_wp_error( $result ) ) {
			self::error( $result->get_error_message() );
		}

		// delete tmp file
		self::delete_tmp_files();

		self::success( sprintf( __( "Post file '%s' has been imported successfully.", 'greyd_hub' ), $filename ) );
	}

	/**
	 * Handle the ajax import action of a posttype.
	 *
	 * @since 1.2.7
	 *
	 * This differs from handle_import() as it's only importing the
	 * first post (which is the posttype) and returns back to the
	 * javascript handler, which then calls handle_import() with a
	 * second AJAX call.
	 * This ensures all the taxonomies etc. are setup before importing
	 * the actual posts - otherwise they could not be set.
	 *
	 * @action 'greyd_ajax_mode_posttype_import'
	 *
	 * @param array $data   holds the $_POST['data']
	 */
	public function handle_posttype_import( $data ) {

		self::enable_logs();

		do_action( 'greyd_post_export_log', "\r\n\r\n" . 'HANDLE IMPORT POSTTYPE' . "\r\n", $data );

		set_time_limit( 5000 );

		$filename = isset( $data['filename'] ) ? $data['filename'] : '';

		if ( empty( $filename ) ) {
			self::error( __( "The file name is empty.", 'greyd_hub' ) );
		}

		// get post data
		$zip_file  = self::get_file_path( 'tmp' ) . $filename;
		$post_data = self::get_zip_posts_file_contents( $zip_file );

		// error
		if ( ! is_array( $post_data ) ) {
			self::error( $post_data );
		}

		// get conflicts with current posts
		$conflicts        = isset( $data['conflicts'] ) ? (array) $data['conflicts'] : array();
		$conflict_actions = self::import_get_conflict_actions_from_backend_form( $conflicts );

		// only insert the first post, as it is the posttype itself
		$first_key     = array_key_first( $post_data );
		$new_post_data = array( $first_key => $post_data[ $first_key ] );

		// import the post
		$result = self::import_posts( $new_post_data, $conflict_actions, $zip_file );

		if ( is_wp_error( $result ) ) {
			self::error( $result->get_error_message() );
		}

		// add the newly created post to the conflict actions
		$new_post_id   = self::$posts[ $first_key ];
		$new_conflicts = array_merge(
			$conflicts,
			array(
				$first_key . '-' . $new_post_id => 'skip',
			)
		);

		// return the new conflicts array to the Javascript handler
		self::success( json_encode( $new_conflicts ) );
	}

	/**
	 * Import posts
	 *
	 * @param WP_Post[] $posts          Preparred post objects.
	 * @param array     $conflict_actions   Array of posts that already exist on the current blog.
	 *                                      Keyed by the same ID as in the @param $posts.
	 *                                  @property post_id: ID of the current post.
	 *                                  @property action: Action to be done (skip|replace|keep)
	 * @param string    $zip_file          Path to imported ZIP archive. (optional)
	 *
	 * @return mixed                    True on success. WP_Error on fatal conflict.
	 */
	public static function import_posts( $posts, $conflict_actions = array(), $zip_file = '' ) {

		// reset the class vars
		self::$posts = array();
		self::$media = array();

		$first_post = null;

		/**
		 * Filter the current posts on import
		 *
		 * @filter 'greyd_import_conflict_actions'
		 *
		 * @param array $conflict_actions   All existing posts with a conflict.
		 * @param array $posts              All posts to be imported.
		 */
		$conflict_actions = (array) apply_filters( 'greyd_import_conflict_actions', $conflict_actions, $posts );

		/**
		 * Loop through posts and insert them
		 */
		foreach ( $posts as $post_id => $post ) {

			do_action( 'greyd_post_export_log', "\r\n----------\r\n" );

			// typecasting $post arrays (eg. via remote requests)
			$post = is_array( $post ) ? (object) $post : $post;
			if ( ! is_object( $post ) ) {
				do_action( 'greyd_post_export_log', sprintf( "  - WP_Post object for post of ID '%s' not set correctly:", $post_id ), $post );
				continue;
			}

			$is_first_post = false;
			if ( ! $first_post ) {
				$first_post    = $post;
				$is_first_post = true;
			}

			do_action( 'greyd_post_export_log', "\r\n" . sprintf( "Insert post '%s'.", $post->post_name ) );

			// handle the post language
			if (
				isset( $post->language )
				&& (
					( is_array($post->language) && isset( $post->language['code'] ) && ! empty( $post->language['code'] ) )
					|| ( is_object($post->language) && isset( $post->language->code ) && ! empty( $post->language->code ) )
				)
			) {

				$post_language_args       = isset( $post->language ) ? (array) $post->language : array();
				$post_language_code       = isset( $post_language_args['code'] ) ? $post_language_args['code'] : '';
				$translation_post_ids     = isset( $post_language_args['post_ids'] ) ? (array) $post_language_args['post_ids'] : array();
				$languages_on_this_site   = self::get_languages_codes();
				$languages_of_this_post   = ! empty( $translation_post_ids ) ? array_keys( $translation_post_ids ) : array();
				$supported_translations   = array_intersect( $languages_on_this_site, $languages_of_this_post );
				$unsupported_translations = array_diff( $languages_of_this_post, $languages_on_this_site );
				$skip_this_translation    = false;

				// switch to the post's language
				$language_switched = self::switch_to_post_lang( $post );

				if ( $language_switched ) {
					do_action( 'greyd_post_export_log', '  - language supported & switched!' );
				} elseif ( count( $supported_translations ) > 0 ) {

					// skip this post if there is a supported translated version
					do_action( 'greyd_post_export_log', '  - There is at least 1 supported language for this post: ' . implode( ', ', $supported_translations ) );
					$skip_this_translation = true;
				} else {
					do_action( 'greyd_post_export_log', '  - There is no supported language for this post.' );

					// check if another translation of this post has already been imported
					if ( ! empty( $translation_post_ids ) ) {
						foreach ( (array) $translation_post_ids as $lang => $translated_post_id ) {
							if ( isset( self::$posts[ $translated_post_id ] ) ) {
								do_action( 'greyd_post_export_log', '  - Another translation of this post has already been imported - we skip this one.' );

								self::$posts[ $post_id ] = $current_post_id;
								$skip_this_translation   = true;
								break;
							}
						}
						do_action( 'greyd_post_export_log', '  - No other translation of this post has been imported - we import this one.' );
					}
				}

				// skip this translation
				if ( $skip_this_translation ) {
					// delete all copies of this post (which is not supported, and a worse translation...)
					if ( isset( $conflict_actions[ $post_id ] ) && isset( $conflict_actions[ $post_id ]['post_id'] ) ) {
						$translated_id = $conflict_actions[ $post_id ]['post_id'];
						$deleted       = wp_trash_post( $translated_id );
						if ( $deleted ) {
							do_action( 'greyd_post_export_log', "  - This version (id: $translated_id ) was trashed, we import a better suited translation." );
						}
					}
					// remove it from the import array to not change it at all
					unset( $posts[ $post_id ] );
					continue;
				}
			}

			/**
			 * Filter the post array before it is imported.
			 *
			 * @filter 'greyd_import_postarr'
			 *
			 * @param array $postarr        Array of post params used for the import.
			 * @param WP_Post $post         Preparred WP_Post object (including meta, taxonomy terms etc.).
			 * @param bool $is_first_post   Whether this is the first post of the import.
			 */
			$postarr = apply_filters(
				'greyd_import_postarr',
				array(
					'post_title'   => $post->post_title,
					'post_name'    => $post->post_name,
					'post_content' => $post->post_content,
					'post_excerpt' => $post->post_excerpt,
					'post_type'    => $post->post_type,
					'post_author'  => isset( $post->post_author ) ? $post->post_author : get_current_user_id(),
					'post_date'    => $post->post_date,
					'post_status'  => $post->post_status,
				),
				$post,
				$is_first_post
			);

			// handle conflict actions
			if ( isset( $conflict_actions[ $post_id ] ) ) {
				$current_post    = (array) $conflict_actions[ $post_id ];
				$current_post_id = $current_post['post_id'];
				$conflict_action = $current_post['action'];

				if ( $conflict_action === 'replace' ) {
					do_action( 'greyd_post_export_log', sprintf( '  - replace existing post with ID: %s.', $current_post_id ) );
					// add @property ID to the array to replace the existing post.
					$postarr['ID'] = $current_post_id;
				} elseif ( $conflict_action === 'skip' ) {
					do_action( 'greyd_post_export_log', sprintf( '  - skip this post and use the existing post: %s.', $current_post_id ) );
					// add the post to the class var for later replacement
					self::$posts[ $post_id ] = $current_post_id;
					// remove it from the import array to not change it at all
					unset( $posts[ $post_id ] );
					continue;
				} elseif ( $conflict_action === 'update' ) {
					do_action( 'greyd_post_export_log', sprintf( '  - update the existing post: %s.', $current_post_id ) );
					// add the post to the class var for later replacement
					self::$posts[ $post_id ] = $current_post_id;
					continue;
				} elseif ( $conflict_action === 'keep' ) {
					do_action( 'greyd_post_export_log', '  - insert post with new ID' );
				}
			}

			// now we insert the post
			do_action( 'greyd_post_export_log', '  - try to insert post with the following data:', array_map(
				function( $value ) {
					return is_string( $value ) ? esc_attr( $value ) : $value;
				},
				$postarr
			) );
			$result = self::create_post( $postarr, $post, $zip_file );

			if ( is_wp_error( $result ) ) {
				return $result;
			} elseif ( $result ) {
				self::$posts[ $post_id ] = $result;

				/**
				 * Set the new post id for all unsupported translations of this post as well
				 *
				 * @since 1.0.9
				 */
				if ( isset( $unsupported_translations ) && ! empty( $unsupported_translations ) ) {
					foreach ( $unsupported_translations as $lang_code ) {
						if ( $lang_code != $post_language_code ) {
							$old_post_id                 = $translation_post_ids[ $lang_code ];
							self::$posts[ $old_post_id ] = $post_id;
							do_action( 'greyd_post_export_log', "  - unsupported translation of this post has been linked with this post (old_post_id: $old_post_id, new_id: $post_id)" );
						}
					}
				}
			}
		}

		/**
		 * Check if first post is a custom posttype.
		 * If it is, we propably need to do some additional action, such as
		 * rewrite the permalinks.
		 */
		if (
			class_exists( '\Greyd\Posttypes\Dynamic_Posttypes' )
			&& $first_post->post_type === 'tp_posttypes'
		) {
			do_action( 'greyd_post_export_log', '  - first post is a posttype: ' . $first_post->post_name );

			// register all dynamic post types & taxonomies
			\Greyd\Posttypes\Dynamic_Posttypes::add_dynamic_posttypes();

			// save transient for rewriting permalink rules
			set_transient( 'flush_rewrite', true );
		}

		do_action( 'greyd_post_export_log', "\r\n----------\r\n\r\n" . 'All posts imported. Now we loop through them.' );

		/**
		 * After we inserted all the posts, we can now do additional actions
		 */
		foreach ( $posts as $old_post_id => $post ) {

			$post = is_array( $post ) ? (object) $post : $post;

			if ( ! isset( self::$posts[ $old_post_id ] ) ) {
				continue;
			}

			$new_post_id = self::$posts[ $old_post_id ];

			do_action( 'greyd_post_export_log', sprintf( "\r\n" . "Check new post '%s' (old id: %s)", $new_post_id, $old_post_id ) );

			// switch to the post's language
			self::switch_to_post_lang( $post );

			// update the post content
			if ( ! empty( $post->post_content ) ) {

				$content = $post->post_content;

				// replace nested posts in post content
				$content = self::replace_nested_posts( $content, $post );

				// switch to the post's language again as we could have been
				// switched during the replacement of nested posts.
				self::switch_to_post_lang( $post );

				// replace nested terms in post content
				$content = self::replace_nested_terms( $content, $post );

				// replace strings in post content
				$content = self::replace_strings( $content, $new_post_id );

				/**
				 * Filter post content before import.
				 *
				 * @filter 'greyd_filter_post_content_before_post_import'
				 *
				 * @param string    $content    The post content.
				 * @param int       $post_id    The post ID.
				 * @param object    $post       The post object.
				 */
				$content = apply_filters( 'greyd_filter_post_content_before_post_import', $content, $new_post_id, $post );

				// update the post content
				$result = wp_update_post(
					array(
						'ID'           => $new_post_id,
						'post_content' => $content,
					),
					true,
					false
				);

				if ( is_wp_error( $result ) ) {
					do_action( 'greyd_post_export_log', '  - post-content could not be updated.' );
				} else {
					do_action( 'greyd_post_export_log', '  - post-content successfully updated.' );
				}
			}

			/**
			 * ------   I M P O R T A N T   ------
			 *
			 * All additonal actions to the post, like adding post-meta options or
			 * setting taxonomy terms need to be done AFTER we called 'wp_update_post'
			 * to update the post-content.
			 * Otherwise those changes are overwritten!
			 */

			// set meta options
			if ( ! empty( $post->meta ) ) {
				self::set_meta( $new_post_id, $post->meta, $post );
			}

			// set terms
			if ( ! empty( $post->terms ) ) {
				self::set_taxonomy_terms( $new_post_id, $post->terms, $post );
			}

			// set translations
			self::set_translations( $new_post_id, $post );

			// replace thumbnail ID
			if ( $thumbnail_id  = get_post_thumbnail_id( $new_post_id ) ) {
				do_action( 'greyd_post_export_log', "\r\n" . sprintf( "Replace thumbnail for post '%s'.", $post->post_name ) );
				$result = false;
				if ( isset( self::$posts[ $thumbnail_id ] ) ) {
					$result = set_post_thumbnail( $new_post_id, self::$posts[ $thumbnail_id ] );
				}
				if ( $result ) {
					do_action( 'greyd_post_export_log', sprintf( "  - thumbnail ID changed from '%s' to '%s'", $thumbnail_id, self::$posts[ $thumbnail_id ] ) );
				} else {
					do_action( 'greyd_post_export_log', sprintf( "  - thumbnail ID '%s' could not be changed.", $thumbnail_id ) );
				}
			}

			/**
			 * Add action to handle additional actions after a post was imported.
			 *
			 * @action 'greyd_after_import_post'
			 */
			do_action( 'greyd_after_import_post', $new_post_id, $post );
		}

		return true;
	}

	/**
	 * Insert a single post
	 *
	 * if $postarr['ID'] is set, the post gets updated. otherwise a new post is created.
	 *
	 * @param array   $postarr  All arguments to be set via wp_insert_post().
	 * @param WP_Post $post     Post object with additional properties.
	 *                          See export_post() for details.
	 * @param string  $zip_file (optional) Full path to the imported ZIP archive.
	 *
	 * @return mixed  Post-ID on success. WP_Error on failure.
	 */
	public static function create_post( $postarr, $post, $zip_file = '' ) {

		// do_action( "greyd_post_export_log", "  - create post with the following attributes: ", $postarr );

		// normal post
		if ( $postarr['post_type'] !== 'attachment' ) {

			// insert post
			$new_post_id = wp_insert_post( $postarr, true, false );

			// error
			if ( is_wp_error( $new_post_id ) ) {
				do_action( 'greyd_post_export_log', "\r\nPost could not be inserted: " . $new_post_id->get_error_message() );
			}
			// success
			else {
				do_action( 'greyd_post_export_log', sprintf( "\r\nPost inserted with the ID '%s'", strval( $new_post_id ) ) );
			}
		}
		// attachment
		else {

			$file  = false;
			$media = isset( $post->media ) ? (array) $post->media : null;

			if ( $media && isset( $media['name'] ) ) {

				$filename    = $media['name'];
				$time_folder = date( 'Y\/m', strtotime( $postarr['post_date'] ) );
				$upload_dir  = wp_upload_dir( $time_folder, true, true );

				if ( ! empty( $zip_file ) ) {
					$file_data = self::get_zip_media_file( $zip_file, $filename );
				} else {
					$file_data = Helper::get_file_contents( $media['url'] );
				}

				if ( wp_mkdir_p( $upload_dir['path'] ) ) {
					$file = $upload_dir['path'] . '/' . $filename;
				} else {
					$file = $upload_dir['basedir'] . '/' . $filename;
				}

				// delete old files if attachment is being replaced
				if ( isset( $postarr['ID'] ) ) {
					do_action( 'greyd_post_export_log', '  - delete old attachment files.' );

					$result = self::delete_current_attachment_files( $postarr['ID'], $file );

					if ( ! $result ) {
						do_action( 'greyd_post_export_log', '  - old attachment files could not be deleted.' );
					} else {
						do_action( 'greyd_post_export_log', '  - old attachment files deleted.' );
					}
				}

				$bytes = file_put_contents( $file, $file_data );
				if ( $bytes === false ) {
					do_action( 'greyd_post_export_log', "  - attachment file '$file' could not be written." );
				} else {
					do_action( 'greyd_post_export_log', "  - attachment file '$file' written (size: {$bytes}b)." );
				}

				// add mime type
				$postarr['post_mime_type'] = wp_check_filetype( $filename, null )['type'];
			}

			// insert post
			$new_post_id = wp_insert_attachment( $postarr, $file );

			// error
			if ( is_wp_error( $new_post_id ) ) {
				do_action( 'greyd_post_export_log', "\r\nMedia file could not be inserted: " . $new_post_id->get_error_message() );
			}
			// success
			else {
				// regenerate attachment metadata
				if ( $file ) {
					do_action( 'greyd_post_export_log', '  - regenerate attachment metadata.' );

					require_once ABSPATH . 'wp-admin/includes/image.php';
					$attach_data = wp_generate_attachment_metadata( $new_post_id, $file );
					$result      = wp_update_attachment_metadata( $new_post_id, $attach_data );
					if ( $result === false ) {
						do_action( 'greyd_post_export_log', '  - attachment metadata could not be updated: ', $attach_data );
					} else {
						do_action( 'greyd_post_export_log', '  - attachment metadata updated: ', $attach_data );
					}
				}

				do_action( 'greyd_post_export_log', sprintf( "\r\nMedia file inserted with the ID '%s'", $new_post_id ) );
			}
		}

		return $new_post_id;
	}

	/**
	 * Delete all current attachment files.
	 *
	 * @see EnableMediaReplace\Replacer->removeCurrent()
	 * @link https://github.com/short-pixel-optimizer/enable-media-replace/blob/master/classes/replacer.php
	 *
	 * @param int    $post_id           The attachment WP_Post ID.
	 * @param string $new_file_path     URL to the new file (does not exist yet).
	 *
	 * @return bool
	 */
	public static function delete_current_attachment_files( $post_id, $new_file_path ) {

		$old_file_path = get_attached_file( $post_id );
		$meta          = wp_get_attachment_metadata( $post_id );
		$backup_sizes  = get_post_meta( $post_id, '_wp_attachment_backup_sizes', true );
		$result        = wp_delete_attachment_files( $post_id, $meta, $backup_sizes, $old_file_path );

		// @todo replace occurences of the new file path on the entire website
		if ( $new_file_path !== $old_file_path ) {

		}

		return $result;
	}

	/**
	 * Replace all nested posts inside a subject, often the post content.
	 *
	 * @param  string $subject  The subject were the strings need to be replaced.
	 * @param  object $post     Preparred post object with the @property 'nested'.
	 *
	 * @return string $content  Content with all nested elements replaced.
	 */
	public static function replace_nested_posts( $subject, $post ) {

		// get $post->nested
		$nested = isset( $post->nested ) ? ( is_array( $post->nested ) || is_object( $post->nested ) ? (array) $post->nested : null ) : null;

		if (
			empty( $nested ) ||
			empty( $subject )
		) {
			do_action( 'greyd_post_export_log', "\r\n" . sprintf( "No nested elements found for post '%s'.", $post->post_name ) );
			return $subject;
		}

		do_action( 'greyd_post_export_log', "\r\n" . sprintf( "Replace nested elements for post '%s'.", $post->post_name ) );

		foreach ( $nested as $nested_id => $nested_postarr ) {

			$replace_string = self::get_nested_post_replacement( $nested_id, $nested_postarr );

			// replace the string
			$subject = str_replace( '{{' . $nested_id . '}}', $replace_string, $subject );

			// replace the front url if post ID was found
			if ( is_numeric( $replace_string ) ) {

				$replace_front_url = $nested_postarr['post_type'] === 'attachment' ? wp_get_attachment_url( $replace_string ) : get_permalink( $replace_string );

				// replace '{{' . $nested_id . '-front-url}}'
				$subject = str_replace( '{{' . $nested_id . '-front-url}}', $replace_front_url, $subject );
			}

			do_action( 'greyd_post_export_log', sprintf( "  - replace '%s' with '%s'.", '{{' . $nested_id . '}}', $replace_string ) );
		}

		do_action( 'greyd_post_export_log', '=> nested elements were replaced' );
		return $subject;
	}

	/**
	 * Get the replacement for the original ID.
	 *
	 * This function looks for an imported or existing post by ID,
	 * name and posttype. If nothing is found, either the original
	 * ID, post_name or frontend url (for attachments) are returned.
	 */
	public static function get_nested_post_replacement( $nested_id, $nested_postarr ) {

		// post imported: use the imported post-ID
		if ( isset( self::$posts[ $nested_id ] ) ) {
			$replace_string = self::$posts[ $nested_id ];
		}
		// no postarr set: use the initial post-ID
		elseif ( ! $nested_postarr || ! is_array( $nested_postarr ) ) {
			$replace_string = $nested_id;
		}
		// post exists: use existing post-ID
		elseif ( $existing_post = self::get_post_by_name_and_type( (object) $nested_postarr ) ) {
			$replace_string = $existing_post->ID;
		}
		// attachments: use the frontend url
		elseif ( $nested_postarr['post_type'] === 'attachment' ) {
			$replace_string = $nested_postarr['front_url'];
		}
		// fallback: use the name
		else {
			$replace_string = $nested_postarr['post_name'];
		}
		return $replace_string;
	}

	/**
	 * Replace all nested terms inside a subject, often the post content.
	 *
	 * @param  string $subject  The subject were the strings need to be replaced.
	 * @param  object $post     Preparred post object with the @property 'nested'.
	 *
	 * @return string $content  Content with all nested elements replaced.
	 */
	public static function replace_nested_terms( $subject, $post ) {

		// get $post->nested_terms
		$nested = isset( $post->nested_terms ) ? ( is_array( $post->nested_terms ) || is_object( $post->nested_terms ) ? (array) $post->nested_terms : null ) : null;

		if (
			empty( $nested ) ||
			empty( $subject )
		) {
			do_action( 'greyd_post_export_log', "\r\n" . sprintf( "No nested elements found for post '%s'.", $post->post_name ) );
			return $subject;
		}

		do_action( 'greyd_post_export_log', "\r\n" . sprintf( "Replace nested terms for post '%s'.", $post->post_name ) );

		foreach ( $nested as $nested_id => $nested_term_object ) {

			if ( ! is_object( $nested_term_object ) ) {
				$nested_term_object = (object) $nested_term_object;
			}

			$term_object = get_term_by( 'slug', $nested_term_object->slug, $nested_term_object->taxonomy );

			if ( $term_object ) {
				$replace_string = $term_object->term_id;
				do_action( 'greyd_post_export_log', "  - term of taxonomy '{$nested_term_object->taxonomy}' with slug '{$nested_term_object->slug}' found.", $term_object );
			} else {
				$replace_string = $nested_id;
				do_action( 'greyd_post_export_log', "  - term of taxonomy '{$nested_term_object->taxonomy}' with slug '{$nested_term_object->slug}' could not be found." );
			}

			// replace the string
			$subject = str_replace( '{{' . $nested_id . '}}', $replace_string, $subject );

			do_action( 'greyd_post_export_log', sprintf( "  - replace '%s' with '%s'.", '{{' . $nested_id . '}}', $replace_string ) );
		}

		do_action( 'greyd_post_export_log', '=> nested elements were replaced' );
		return $subject;
	}
	/**
	 * Replace strings in subject
	 *
	 * @param string $subject
	 * @param int    $post_id
	 *
	 * @return string $subject
	 */
	public static function replace_strings( $subject, $post_id, $log = true ) {

		if ( empty( $subject ) ) {
			return $subject;
		}

		if ( $log ) {
			do_action( 'greyd_post_export_log', "\r\n" . 'Replace strings.' );
		}

		// get patterns
		$replace_strings = (array) self::regex_nested_strings( $subject, $post_id );
		foreach ( $replace_strings as $name => $string ) {
			$subject = str_replace( '{{' . $name . '}}', $string, $subject );
			if ( $log ) {
				do_action( 'greyd_post_export_log', sprintf( "  - '%s' was replaced with '%s'.", $name, $string ) );
			}
		}
		if ( $log ) {
			do_action( 'greyd_post_export_log', '=> strings were replaced' );
		}
		return $subject;
	}

	/**
	 * Given an array of meta, set meta to another post.
	 *
	 * @param int     $post_id  Post ID.
	 * @param array   $meta     Array of meta as key => value.
	 * @param WP_Post $post     The preparred post object (optional).
	 */
	public static function set_meta( $post_id, $meta, $post = null ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Set post meta.' );

		$existing_meta = (array) get_post_meta( $post_id );

		foreach ( (array) $meta as $meta_key => $meta_values ) {

			// don't import blacklisted meta
			if ( in_array( $meta_key, self::blacklisted_meta(), true ) ) {
				continue;
			}
			// skip certain mety keys
			elseif ( self::maybe_skip_meta_option( $meta_key, $meta_values ) ) {
				continue;
			}

			foreach ( (array) $meta_values as $meta_placement => $meta_value ) {
				$has_prev_value = (
						isset( $existing_meta[ $meta_key ] )
						&& is_array( $existing_meta[ $meta_key ] )
						&& array_key_exists( $meta_placement, $existing_meta[ $meta_key ] )
					) ? true : false;
				if ( $has_prev_value ) {
					$prev_value = maybe_unserialize( $existing_meta[ $meta_key ][ $meta_placement ] );
				}

				if ( ! is_array( $meta_value ) ) {
					$meta_value = maybe_unserialize( $meta_value );
				}

				/**
				 * Add filter to modify the post meta value before import
				 *
				 * @filter 'greyd_import_post_meta-{{meta_key}}'
				 */
				$meta_value = apply_filters( 'greyd_import_post_meta-' . $meta_key, $meta_value, $post_id, $post );

				if ( $has_prev_value ) {
					update_post_meta( $post_id, $meta_key, $meta_value, $prev_value );
				} else {
					add_post_meta( $post_id, $meta_key, $meta_value );
				}
			}
		}
		do_action( 'greyd_post_export_log', '=> post meta set' );
	}

	/**
	 * Modify dynamic meta value after import
	 *
	 * @filter 'greyd_import_post_meta-dynamic_meta'
	 *
	 * @param mixed $meta_value
	 * @param int   $post_id
	 *
	 * @return mixed $meta_value
	 */
	public function set_dynamic_meta( $meta_value, $post_id ) {

		$meta_value = (array) $meta_value;

		foreach ( $meta_value as $key => $value ) {
			if ( preg_match( '/\{\{(.+?)\}\}/', $value, $matches ) ) {
				$inner = isset( $matches[1] ) ? $matches[1] : '';

				// ID found
				if ( is_numeric( $inner ) ) {
					$replace_string     = isset( self::$posts[ $inner ] ) ? self::$posts[ $inner ] : $inner;
					$meta_value[ $key ] = preg_replace( '/\{\{(.+?)\}\}/', $replace_string, $value );
					do_action( 'greyd_post_export_log', sprintf( "  - ID '%s' in the field '%s' was replaced with '%s'", $inner, $key, $replace_string ) );
				} else {
					$meta_value[ $key ] = self::replace_strings( $value, $post_id, false );
					do_action( 'greyd_post_export_log', sprintf( "  - All strings in the field '%s' were replaced", $key ) );
				}
			}
		}

		return $meta_value;
	}

	/**
	 * Given an array of terms by taxonomy, set those terms to another post. This function will cleverly merge
	 * terms into the post and create terms that don't exist.
	 *
	 * @param int   $post_id        Post ID.
	 * @param array $taxonomy_terms Array with taxonomy as key and array of terms as values.
	 */
	public static function set_taxonomy_terms( $post_id, $taxonomy_terms, $post ) {
		do_action( 'greyd_post_export_log', "\r\n" . 'Set taxonomy terms.' );

		foreach ( (array) $taxonomy_terms as $taxonomy => $terms ) {

			if ( ! taxonomy_exists( $taxonomy ) ) {
				do_action( 'greyd_post_export_log', "  - taxonomy '{$taxonomy}' doesn't exist." );

				// we temporary register the taxonomy if it is dynamic
				$taxonomies = self::get_dynamic_taxonomies( $post->post_type );

				// register taxonomy if it is dynamic
				if ( in_array( $taxonomy, $taxonomies ) && ! taxonomy_exists( $taxonomy ) ) {
					$result = register_taxonomy( $taxonomy, $post->post_type );
					if ( is_wp_error( $result ) ) {
						do_action( 'greyd_post_export_log', "    - taxonomy '{$taxonomy}' could not be registered: " . $result->get_error_message() );
					} else {
						do_action( 'greyd_post_export_log', "    - taxonomy '{$taxonomy}' registered." );
					}
				}

				// continue if taxonomy still doesn't exist
				if ( ! in_array( $taxonomy, $taxonomies ) && ! taxonomy_exists( $taxonomy ) ) {
					continue;
				}
			}

			$term_ids        = array();
			$term_id_mapping = array();

			foreach ( (array) $terms as $term_array ) {
				if ( ! is_array( $term_array ) ) {
					$term_array = (array) $term_array;
				}

				if ( ! isset( $term_array['slug'] ) ) {
					continue;
				}

				$term = get_term_by( 'slug', $term_array['slug'], $taxonomy );

				if ( empty( $term ) ) {

					$term = wp_insert_term(
						$term_array['name'],
						$taxonomy,
						array(
							'slug'        => $term_array['slug'],
							'description' => isset( $term_array['description'] ) ? $term_array['description'] : '',
						)
					);

					if ( is_wp_error( $term ) ) {
						do_action( 'greyd_post_export_log', "    - term '{$term_array['name']}' of taxonomy '$taxonomy' could not be inserted: " . $term->get_error_message() );
					} else {
						$term_id_mapping[ $term_array['term_id'] ] = $term['term_id'];
						$term_ids[]                                = $term['term_id'];
						do_action( 'greyd_post_export_log', "    - term '{$term_array['name']}' of taxonomy '$taxonomy' inserted with id '{$term['term_id']}'." );
					}
				} else {
					$term_id_mapping[ $term_array['term_id'] ] = $term->term_id;
					$term_ids[]                                = $term->term_id;
					do_action( 'greyd_post_export_log', "    - term '{$term_array['name']}' of taxonomy '$taxonomy' found with id {$term->term_id}." );
				}
			}

			foreach ( (array) $terms as $term_array ) {
				if ( ! is_array( $term_array ) ) {
					$term_array = (array) $term_array;
				}

				if ( empty( $term_array['parent'] ) ) {
					$term = wp_update_term(
						$term_id_mapping[ $term_array['term_id'] ],
						$taxonomy,
						array(
							'parent' => '',
						)
					);
				} elseif ( isset( $term_id_mapping[ $term_array['parent'] ] ) ) {
					$term = wp_update_term(
						$term_id_mapping[ $term_array['term_id'] ],
						$taxonomy,
						array(
							'parent' => $term_id_mapping[ $term_array['parent'] ],
						)
					);
				}
			}

			$new_term_ids = wp_set_object_terms( $post_id, $term_ids, $taxonomy );

			if ( is_wp_error( $new_term_ids ) ) {
				do_action( 'greyd_post_export_log', "  - term ids of taxonomy '$taxonomy' could not be set to post: " . $new_term_ids->get_error_message() );
			} else {
				do_action( 'greyd_post_export_log', "  - term ids '" . implode( ', ', $new_term_ids ) . "' of taxonomy '$taxonomy' set to post." );
			}
		}
		do_action( 'greyd_post_export_log', isset( $new_term_ids ) ? '=> all taxonomy terms set' : '=> no taxonomy terms' );
	}

	/**
	 * Check if other post translations are better suited to be imported on this site.
	 * If yes, we should skip this post and not import it.
	 *
	 * @param WP_Post $post
	 *
	 * @return bool|int     Whether there is a better version. Returns the post id if it was already inserted.
	 */
	public static function better_post_translation_exists( $post ) {

		$language               = isset( $post->language ) ? (array) $post->language : array();
		$translation_post_ids   = isset( $language['post_ids'] ) ? (array) $language['post_ids'] : array();
		$languages_on_this_site = self::get_languages_codes();
		$languages_of_this_post = ! empty( $translation_post_ids ) ? array_keys( $translation_post_ids ) : array();
		$supported_translations = array_intersect( $languages_on_this_site, $languages_of_this_post );

		// check if there is at least 1 equal language
		if ( count( $supported_translations ) > 0 ) {
			do_action( 'greyd_post_export_log', '  - There is at least 1 supported language for this post: ' . implode( ', ', $supported_translations ) );
			return true;
		} elseif ( ! empty( $translation_post_ids ) ) {

			do_action( 'greyd_post_export_log', '  - There is no supported language for this post.' );

			// we check if another translation of this post has already been imported
			foreach ( (array) $translation_post_ids as $lang => $translated_post_id ) {
				if ( isset( self::$posts[ $translated_post_id ] ) ) {
					do_action( 'greyd_post_export_log', '  - Another translation of this post has already been imported - we skip this one.' );
					return true;
				}
			}
			do_action( 'greyd_post_export_log', '  - No other translation of this post has been imported - we import this one.' );
		}
		return false;
	}

	/**
	 * Set the language of a post and link it to it's source post if possible.
	 *
	 * @param int     $post_id      Post ID on this stage.
	 * @param WP_Post $post     Old WP_Post object (Post ID might differ).
	 *
	 * @return bool
	 */
	public static function set_translations( $post_id, $post ) {

		$language = isset( $post->language ) && ! empty( $post->language ) ? (array) $post->language : null;

		if (
			! $language ||
			! isset( $language['args'] ) ||
			! isset( $language['code'] ) ||
			! isset( $language['tool'] )
		) {
			return false;
		}

		do_action( 'greyd_post_export_log', "\r\n" . 'Set translations for the post:', $language );

		$args     = (array) $language['args'];
		$code     = strval( $language['code'] );
		$tool     = strval( $language['tool'] );
		$post_ids = isset( $language['post_ids'] ) ? (array) $language['post_ids'] : null;

		if ( $tool === 'wpml' ) {

			$wpml_element_type = apply_filters( 'wpml_element_type', $post->post_type );
			$wpml_element_trid = apply_filters( 'wpml_element_trid', null, $post_id, $wpml_element_type );

			// get source language
			$source_language_code = isset( $args['source_language_code'] ) ? $args['source_language_code'] : null;
			if ( ! empty( $source_language_code ) ) {

				// get the (original) post-id of the source post...
				$original_source_language_post_id = isset( $post_ids[ $source_language_code ] ) ? $post_ids[ $source_language_code ] : null;
				if ( ! empty( $original_source_language_post_id ) ) {

					// ...get it's post-id on this stage
					$current_source_language_post_id = isset( self::$posts[ $original_source_language_post_id ] ) ? self::$posts[ $original_source_language_post_id ] : null;
					if ( ! empty( $current_source_language_post_id ) ) {

						// get the unique translation-id (trid)
						$wpml_element_trid = apply_filters( 'wpml_element_trid', $wpml_element_trid, $current_source_language_post_id, $wpml_element_type );
					}
				}
			}

			if ( empty( $wpml_element_trid ) ) {
				return false;
			}

			// set the translation
			do_action(
				'wpml_set_element_language_details',
				array(
					'element_id'           => $post_id,
					'element_type'         => $wpml_element_type,
					'trid'                 => $wpml_element_trid,
					'language_code'        => $code,
					'source_language_code' => $source_language_code,
					'check_duplicates'     => false,
				)
			);

			return true;
		}

		return false;
	}


	/**
	 * =================================================================
	 *                          BACKEND UI
	 * =================================================================
	 */

	/**
	 * Add overlay contents
	 *
	 * @filter 'greyd_overlay_contents'
	 *
	 * @param array $contents
	 * @return array $contents
	 */
	public function add_overlay_contents( $contents ) {

		if ( self::is_current_screen_supported() ) {

			$screen = get_current_screen();
			// debug( $screen );

			/**
			 * Export
			 */

			 // default options
			$export_options = array(
				'nested' => array(
					'name'    => 'nested',
					'title'   => __( "Export nested content", 'greyd_hub' ),
					'descr'   => __( "Templates, media, etc. are added to the download so that used images, backgrounds, etc. will be displayed correctly on the target website.", 'greyd_hub' ),
					'checked' => true,
				),
				'menus'  => array(
					'name'  => 'resolve_menus',
					'title' => __( "Resolve menus", 'greyd_hub' ),
					'descr' => __( "All menus will be converted to static links.", 'greyd_hub' ),
					'checked' => true,
				),
			);

			// add option to include posts for posttypes
			if ( $screen->id === 'edit-tp_posttypes' ) {
				$export_options['posttypes'] = array(
					'name'  => 'whole_posttype',
					'title' => __( "Include individual posts", 'greyd_hub' ),
					'descr' => __( "All posts of the post type will be added to the download.", 'greyd_hub' ),
				);
				unset( $export_options['menus'] );
			}

			// remove options for media
			if ( $screen->base === 'upload' ) {
				unset( $export_options['nested'], $export_options['menus'] );
			}

			// remove menu option on blocks installations
			if ( is_greyd_blocks() ) {
				unset( $export_options['menus'] );
			}

			// add option to include translations when wpml is active
			if ( ! empty( self::get_translation_tool() ) ) {
				$export_options['translations'] = array(
					'name'  => 'translations',
					'title' => __( "Include translations", 'greyd_hub' ),
					'descr' => __( "All translations of the post will be added to the download.", 'greyd_hub' ),
				);
			}

			// build the form
			$export_form = "<a id='post_export_download'></a><form id='post_export_form' class='" . ( count( $export_options ) ? 'inner_content' : '' ) . "'>";
			foreach ( $export_options as $option ) {
				$name         = $option['name'];
				$checked      = isset( $option['checked'] ) && $option['checked'] ? "checked='checked'" : '';
				$export_form .= "<label for='$name'>
					<input type='checkbox' id='$name' name='$name' $checked />
					<span>" . $option['title'] . '</span>
					<small>' . $option['descr'] . '</small>
				</label>';
			}
			$export_form .= '</form>';

			// add notices
			if ( $screen->id === 'edit-page' ) {
				$export_form .= Helper::render_info_box(
					array(
						'text'  => __( "Posts in post overviews are not included in the import. Posts and Post Types must be exported separately.", 'greyd_hub' ),
						'style' => 'info',
					)
				);
			}

			// add the contents
			$contents['post_export'] = array(
				'confirm' => array(
					'title'   => __( "Export", 'greyd_hub' ),
					'descr'   => sprintf( __( "Do you want to export \"%s\"?", 'greyd_hub' ), "<b class='replace'></b>" ),
					'content' => $export_form,
					'button'  => __( "export now", 'greyd_hub' ),
				),
				'loading' => array(
					'descr' => __( "Exporting post.", 'greyd_hub' ),
				),
				'success' => array(
					'title' => __( "Export successful", 'greyd_hub' ),
					'descr' => __( "Post has been exported.", 'greyd_hub' ),
				),
				'fail'    => array(
					'title' => __( "Export failed", 'greyd_hub' ),
					'descr' => '<span class="replace">' . __( "The post could not be exported.", 'greyd_hub' ) . '</span>',
				),
			);

			/**
			 * Import
			 */

			// get form options
			$options = '';
			foreach ( array(
				'skip'    => __( "skip", 'greyd_hub' ),
				'replace' => __( "replace", 'greyd_hub' ),
				'keep'    => __( "keep both", 'greyd_hub' ),
			) as $name => $value ) {
				$options .= "<option value='$name'>$value</option>";
			}
			$options = urlencode( $options );

			// add the contents
			$contents['post_import'] = array(
				'check_file' => array(
					'title'   => __( "Please wait", 'greyd_hub' ),
					'descr'   => __( "The file is being validated.", 'greyd_hub' ),
					'content' => '<div class="loading"><div class="loader"></div></div><a href="javascript:window.location.href=window.location.href" class="color_light escape">' . __( "cancel", 'greyd_hub' ) . '</a>',
				),
				'confirm'    => array(
					'title'   => __( "Import", 'greyd_hub' ),
					'content' => "<form id='post_import_form'>
										<input type='file' name='import_file' id='import_file' title='" . __( "Select file", 'greyd_hub' ) . "' accept='zip,application/octet-stream,application/zip,application/x-zip,application/x-zip-compressed' >
										<div class='conflicts'>
											<p>" . __( "<b>Attention:</b> Some content in the file already appears to exist on this site. Choose what to do with it.", 'greyd_hub' ) . "</p>
											<div class='inner_content' data-options='$options'></div>
										</div>
										<div class='new'>
											<p>" . sprintf( __( "No conflicts found. Do you want to import the file \"%s\" now?", 'greyd_hub' ), "<strong class='post_title'></strong>" ) . '</p>
										</div>
									</form>',
					'button'  => __( "import now", 'greyd_hub' ),
				),
				'loading'    => array(
					'descr' => __( "Importing post.", 'greyd_hub' ),
				),
				'reload'     => array(
					'title' => __( "Import successful", 'greyd_hub' ),
					'descr' => __( "Post has been imported.", 'greyd_hub' ),
				),
				'fail'       => array(
					'title' => __( "Import failed", 'greyd_hub' ),
					'descr' => '<span class="replace">' . __( "The file could not be imported.", 'greyd_hub' ) . '</span>',
				),
			);
		}
		return $contents;
	}

	/**
	 * Add export button to row actions
	 *
	 * @param  array   $actions
	 * @param  WP_Post $post
	 * @return array
	 */
	public function add_export_row_action( $actions, $post ) {

		if ( self::is_current_screen_supported() ) {
			$actions['greyd_export'] = "<a style='cursor:pointer;' onclick='greyd.postExport.openExport(this);' data-post_id='" . $post->ID . "'>" . __( "Export", 'greyd_hub' ) . '</a>';
		}

		return $actions;
	}

	/**
	 * Add import button via javascript
	 */
	public function add_import_page_title_action() {

		if ( !self::is_current_screen_supported() || !current_user_can( 'edit_others_posts' ) ) {
			return;
		}

		// get plugin info
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		wp_register_script(
			'greyd-post-export-script',
			plugin_dir_url( __FILE__ ) . 'assets/js/post-export.js',
			array( 'jquery', 'greyd-admin-script' ),
			get_plugin_data( $this->config->plugin_file )['Version']
		);
		wp_enqueue_script( 'greyd-post-export-script' );

		wp_add_inline_script(
			'greyd-post-export-script',
			'jQuery(function() {
				greyd.backend.addPageTitleAction( "⬇&nbsp;' . __( "Import", 'greyd_hub' ) . '", { onclick: "greyd.postExport.openImport();" } );
			});',
			'after'
		);
	}

	/**
	 * Add export bulk actions
	 */
	public function add_bulk_actions() {
		// usual posttypes
		foreach ( self::get_supported_post_types() as $posttype ) {
			add_filter( 'bulk_actions-edit-' . $posttype, array( $this, 'add_export_bulk_action' ) );
			add_filter( 'handle_bulk_actions-edit-' . $posttype, array( $this, 'handle_export_bulk_action' ), 10, 3 );
		}
		// media library
		add_filter( 'bulk_actions-upload', array( $this, 'add_export_bulk_action' ) );
		add_filter( 'handle_bulk_actions-upload', array( $this, 'handle_export_bulk_action' ), 10, 3 );
	}

	/**
	 * Add export to the bulk action dropdown
	 */
	public function add_export_bulk_action( $bulk_actions ) {
		$bulk_actions['greyd_export'] = __( "Export", 'greyd_hub' );

		if ( !empty( self::get_translation_tool() ) ) {
			$bulk_actions['greyd_export_multilanguage'] = __( "Export incl. translations", 'greyd_hub' );
		}
		return $bulk_actions;
	}

	/**
	 * Display an admin notice if the transient 'greyd_transient_notice' is set
	 */
	public function display_transient_notice() {

		// get transient
		$transient = get_transient( 'greyd_transient_notice' );

		if ( $transient ) {
			// cut transient into pieces
			$transient = explode( '::', $transient );
			$mode      = $transient[0];
			$msg       = $transient[1];
			// this is my last resort
			Helper::show_message( $msg, $mode );

			// delete transient
			delete_transient( 'greyd_transient_notice' );
		}
	}


	/**
	 * =================================================================
	 *                          HELPER
	 * =================================================================
	 */

	/**
	 * Get all posts from class var
	 */
	public static function get_all_posts() {
		return self::$posts;
	}

	/**
	 * Get all posts from class var
	 */
	public static function get_all_media() {
		return self::$media;
	}

	/**
	 * Get all supported posttypes for global contents
	 */
	public static function get_supported_post_types() {

		// check cache
		if ( $cache = wp_cache_get( 'get_supported_post_types', 'greyd_post_export' ) ) {
			return $cache;
		}

		if ( class_exists( '\Greyd\Posttypes\Dynamic_Posttypes' ) ) {
			// register all dynamic post types & taxonomies
			\Greyd\Posttypes\Dynamic_Posttypes::add_dynamic_posttypes();
		}


		$include   = array( 'page', 'post', 'attachment' );
		$exclude   = array( 'tp_forms_entry', 'vc_grid_item', 'vc4_templates', 'wp_global_styles', 'wp_template_part', 'wp_navigation' );
		$posttypes = array_keys( get_post_types( array( '_builtin' => false ) ) );

		$supported = array_diff( array_merge( $include, $posttypes ), $exclude );

		// Set cache
		wp_cache_set( 'get_supported_post_types', $supported, 'greyd_post_export' );

		return $supported;
	}

	/**
	 * Check if current screen supports import/export
	 *
	 * @return bool
	 */
	public static function is_current_screen_supported() {

		if ( ! function_exists( 'get_current_screen' ) ) {
			return false;
		}

		$screen = get_current_screen();
		if ( is_object( $screen ) && isset( $screen->base ) ) {
			if ( $screen->base === 'edit' ) {
				$post_types = array_flip( self::get_supported_post_types() );
				if ( isset( $post_types[ $screen->post_type ] ) ) {
					return true;
				}
			} elseif ( $screen->base === 'upload' ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Get path to the export folder. Use this path to write files.
	 *
	 * @param string $folder Folder inside wp-content/backup/posts/
	 *
	 * @return string $path
	 */
	public static function get_file_path( $folder = '' ) {

		// get basic path from class var
		if ( self::$basic_path ) {
			$path = self::$basic_path;
		}
		// init basic path
		else {
			$path = WP_CONTENT_DIR . '/backup';

			if ( ! file_exists( $path ) ) {
				do_action( 'greyd_post_export_log', sprintf( '  - create folder „%s“.', $path ) );
				mkdir( $path, 0755, true );
			}
			$path .= '/posts';
			if ( ! file_exists( $path ) ) {
				do_action( 'greyd_post_export_log', sprintf( '  - create folder „%s“.', $path ) );
				mkdir( $path, 0755, true );
			}
			$path .= '/';

			// save in class var
			self::$basic_path = $path;
		}

		// get directory
		if ( ! empty( $folder ) ) {
			$path .= $folder;
			if ( ! file_exists( $path ) ) {
				do_action( 'greyd_post_export_log', sprintf( '  - create folder „%s“.', $path ) );
				mkdir( $path, 0755, true );
			}
		}

		$path .= '/';
		return $path;
	}

	/**
	 * Convert file path to absolute path. Use this path to download a file.
	 *
	 * @param string $path File path.
	 *
	 * @return string $path
	 */
	public static function convert_content_dir_to_path( $path ) {
		return str_replace(
			WP_CONTENT_DIR,
			WP_CONTENT_URL,
			$path
		);
	}

	/**
	 * Get the „posts.json“ file contents inside an imported zip archive
	 *
	 * @param string $filepath  Relative path to the zip including filename.
	 *
	 * @return mixed            String with error message on failure.
	 *                          Array of contents on success.
	 */
	public static function get_zip_posts_file_contents( $filepath ) {

		if ( ! file_exists( $filepath ) ) {
			__( "The ZIP archive could not be found. It may have been moved or deleted.", 'greyd_hub' );
		} else {
			do_action( 'greyd_post_export_log', sprintf( '  - zip archive „%s“ found.', $filepath ) );
		}

		// open 'posts.json' file inside zip archive
		$zip       = 'zip://' . $filepath . '#posts.json';
		$json_file = Helper::get_file_contents( $zip );

		if ( ! $json_file ) {
			return __( "The ZIP archive does not contain a valid \"posts.json\" file.", 'greyd_hub' );
		} else {
			do_action( 'greyd_post_export_log', sprintf( '  - file „%s“ found.', 'posts.json' ) );
		}

		// decode json data
		$contents = json_decode( $json_file, true );
		if ( $contents === null && json_last_error() !== JSON_ERROR_NONE ) {
			return __( "The post.json file could not be read.", 'greyd_hub' );
		} else {
			do_action( 'greyd_post_export_log', '  - decoded json.' );
		}

		if ( ! is_array( $contents ) ) {
			return __( "The posts.json file does not contain any data.", 'greyd_hub' );
		} else {
			do_action( 'greyd_post_export_log', '  - json contains object.' );
		}

		// convert posts back to objects
		foreach ( $contents as $post_id => $post ) {
			$contents[ $post_id ] = (object) $post;
		}

		return $contents;
	}

	/**
	 * Get the file contents of media file inside imported zip archive
	 *
	 * @param string $filepath  Relative path to the zip including filename.
	 * @param string $medianame Name of the media file.
	 *
	 * @return mixed            String with error message on failure.
	 *                          Array of contents on success.
	 */
	public static function get_zip_media_file( $filepath, $medianame ) {

		if ( ! file_exists( $filepath ) ) {
			__( "The ZIP archive could not be found. It may have been moved or deleted.", 'greyd_hub' );
		} else {
			do_action( 'greyd_post_export_log', sprintf( '  - zip archive „%s“ found.', $filepath ) );
		}

		// open 'posts.json' file inside zip archive
		$zip        = 'zip://' . $filepath . '#media/' . $medianame;
		$media_file = Helper::get_file_contents( $zip );

		if ( ! $media_file ) {
			return sprintf( __( "The file '%s' could not be found in the ZIP archive.", 'greyd_hub' ), 'media/' . $medianame );
		} else {
			do_action( 'greyd_post_export_log', sprintf( '  - file „%s“ found.', 'posts.json' ) );
		}

		return $media_file;
	}

	/**
	 * Returns list of blacklisted meta keys
	 *
	 * @return array
	 */
	public static function blacklisted_meta() {
		/**
		 * @filter 'greyd_export_blacklisted_meta'
		 */
		return apply_filters(
			'greyd_export_blacklisted_meta',
			array(
				'_wp_attached_file',
				'_wp_attachment_metadata',
				'_edit_lock',
				'_edit_last',
				'_wp_old_slug',
				'_wp_old_date',
				'_wpb_vc_js_status',
			)
		);
	}

	/**
	 * Check whether to skip a certain meta key and not im- or export it
	 */
	public static function maybe_skip_meta_option( $meta_key, $meta_value ) {

		// skip empty options
		if ( $meta_value === '' ) {
			// do_action( "greyd_post_export_log","  - skipped empty meta option for '$meta_key'");
			return true;
		}
		// skip oembed meta options
		elseif ( strpos( $meta_key, '_oembed_' ) === 0 ) {
			do_action( 'greyd_post_export_log', "  - skipped oembed option '$meta_key'" );
			return true;
		}
		// skip wpml meta if plugin not active
		elseif ( strpos( $meta_key, '_wpml_' ) === 0 ) {
			if ( self::get_translation_tool() !== 'wpml' ) {
				do_action( 'greyd_post_export_log', "  - skipped wpml option '$meta_key'" );
				return true;
			}
		}
		// skip yoast meta if plugin not active
		elseif ( strpos( $meta_key, '_yoast_' ) === 0 ) {
			if ( ! Helper::is_active_plugin( 'wordpress-seo/wp-seo.php' ) ) {
				do_action( 'greyd_post_export_log', "  - skipped yoast option '$meta_key'" );
				return true;
			}
		}
		/**
		 * @filter 'greyd_export_maybe_skip_meta_option'
		 */
		return apply_filters( 'greyd_export_maybe_skip_meta_option', false, $meta_key, $meta_value );
	}

	/**
	 * Get post by name and post_type
	 *
	 * eg. checks if post already exists.
	 *
	 * @param object|string $post   WP_Post object or post_name
	 *
	 * @return bool|object False on failure, WP_Post on success.
	 */
	public static function get_post_by_name_and_type( $post ) {

		$post_name = is_object( $post ) ? (string) $post->post_name : (string) $post;
		$post_type = is_object( $post ) ? (string) $post->post_type : self::get_supported_post_types();
		$args      = array(
			'name'        => $post_name,
			'post_type'   => $post_type,
			'numberposts' => 1,
			'post_status' => array( 'publish', 'pending', 'draft', 'future', 'private', 'inherit' ),
		);

		// only get post of same language
		if ( self::switch_to_post_lang( $post ) ) {
			$args['suppress_filters'] = false;
		}

		// query
		$result = get_posts( $args );

		if ( is_array( $result ) && isset( $result[0] ) ) {
			do_action( 'greyd_post_export_log', sprintf( "  - %s found with ID '%s'.", $post->post_type, $result[0]->ID ) );
			return $result[0];
		} else {
			do_action( 'greyd_post_export_log', sprintf( "  - Post '%s' not found by name and post type.", $post_name ) );
			return false;
		}
	}

	/**
	 * Create filename from export attributes
	 *
	 * @param WP_Post|array $posts
	 * @param array         $args
	 *
	 * @return string $filename
	 */
	public static function write_filename( $posts, $args = array() ) {

		// vars
		$filename     = array();
		$default_args = array(
			'whole_posttype' => false,
			// we don't need other arguments to create the filename
		);
		$args = array_merge( $default_args, (array) $args );

		// bulk export
		if ( is_array( $posts ) && isset( $posts[0] ) ) {
			$bulk = true;
			$post = $posts[0];
			$post = ! is_object( $post ) ? get_post( $post ) : $post;
		}
		// single export
		elseif ( is_object( $posts ) ) {
			$bulk = false;
			$post = $posts;
			// add post name to filename
			$filename[] = $post->post_name;
		}
		// unknown export
		if ( ! isset( $post ) || ! $post || ! isset( $post->post_type ) ) {
			return 'post-export';
		}

		$post_type     = $post->post_type;
		$default_types = array(
			'post'             => 'post',
			'page'             => 'page',
			'attachment'       => 'media_file',
			'tp_forms'         => 'form',
			'dynamic_template' => 'template',
			'tp_posttypes'     => 'posttype',
			'greyd_popup'      => 'popup',
		);

		// handle default posttypes
		if ( isset( $default_types[ $post_type ] ) ) {

			if ( $args['whole_posttype'] && $post_type == 'tp_posttypes' ) {
				$post_type = 'posts_and_posttype';
			} else {
				$post_type = $default_types[ $post_type ] . ( $bulk ? 's' : '' );
			}
		}
		// handle other posttypes
		elseif ( $bulk ) {
			$post_type_obj = get_post_type_object( $post_type );
			if ( $post_type_obj && isset( $post_type_obj->labels ) ) {
				$post_type = $post_type_obj->labels->name;
			}
		}

		// add post type to filename
		$filename[] = $post_type;

		// add site title to filename
		$filename[] = get_bloginfo( 'name' );

		// cleanup strings
		foreach ( $filename as $k => $string ) {
			$filename[ $k ] = preg_replace( '/[^a-z_]/', '', strtolower( preg_replace( '/-+/', '_', $string ) ) );
		}

		return implode( '-', $filename );
	}

	/**
	 * Return error to frontend
	 */
	public static function error( $message = '' ) {
		if ( self::$logs ) {
			echo "\r\n\r\n";
		}
		wp_die( 'error::' . $message );
	}

	/**
	 * Return success to frontend
	 */
	public static function success( $message = '' ) {
		if ( self::$logs ) {
			echo "\r\n\r\n";
		}
		wp_die( 'success::' . $message );
	}

	/**
	 * Get the patterns to replace post ids in post_content.
	 *
	 * @param int     $post_id      The WP_Post ID.
	 * @param WP_Post $post         The WP_Post Object
	 *
	 * @return array[] with the following structure:
	 *
	 * @property array  search      Regex around the ID to look for.
	 *                              Implodes with: '([\da-z\-\_]+?)'
	 *                              Slashes are automatically set around the regex.
	 * @property array  replace     Strings around the ID to replace by.
	 *                              Implodes with: '{{'.$post_id.'}}'
	 * @property string post_type   (optional) Post type of the found ID.
	 *                              If set and a post_name is found, it looks
	 *                              for the post-ID by post_name & post_type.
	 * @property int    group       (optional) Number of regex-group that contains
	 *                              the ID. Default: 2
	 */
	public static function regex_nested_posts( $post_id = 0, $post = null ) {
		/**
		 * @filter 'greyd_regex_nested_posts'
		 *
		 * @param array   $patterns     Regex pattern arguments.
		 * @param int     $post_id      The WP_Post ID.
		 * @param WP_Post $post         The WP_Post Object
		 */
		return (array) apply_filters(
			'greyd_regex_nested_posts',
			array(
				/**
				 * Find nested templates
				 *  (1) template module:            ' template="{{$name_or_id}}"'
				 *  (2) post overview:              ' post_template="{{$name_or_id}}"'
				 *  (3) dynamic content:            ' dynamic_content="{{id}}::someContent...'
				 *  (4) dynamic post content:       ' dynamic_post_content="{{id}}::someContent...'
				 *  (5) dynamic dynamic content:    ' dynamic_dynamic_content="dynamic_content|dynamic_content|{{id}}"'
				 */
				// (1) + (2)
				'template'                => array(
					'search'    => array( ' (post_)?template=\"', '\"' ),
					'replace'   => array( ' $1template="', '"' ),
					'post_type' => 'dynamic_template',
				),
				// (3) + (4)
				'dynamic_content'         => array(
					'search'    => array( ' dynamic_(post_)?content=\"', '(::|\")' ),
					'replace'   => array( ' dynamic_$1content="', '$3' ),
					'post_type' => 'dynamic_template',
				),
				// (5)
				'dynamic_dynamic_content' => array(
					'search'    => array( ' dynamic_dynamic_content=\"dynamic_content\|dynamic_content\|', '\"' ),
					'replace'   => array( ' dynamic_dynamic_content="dynamic_content|dynamic_content|', '"' ),
					'post_type' => 'dynamic_template',
					'group'     => 1,
				),
				/**
				 * Find nested forms
				 *  (1) form module:    '[vc_form ... id="{{$id}}"]'
				 *  (2) dynamic:        '::id|dropdown_forms|someName|{{$id}}::'
				 */
				// (1)
				'vc_form'                 => array(
					'search'    => array( '\[vc_form([^\[\]]*?) id=\"', '\"' ),
					'replace'   => array( '[vc_form$1 id="', '"' ),
					'post_type' => 'tp_forms',
				),
				// (2)
				'dynamic_form'            => array(
					'search'    => array( '::id\|dropdown_forms\|([^\|]*?)\|', '(::|\")' ),
					'replace'   => array( '::id|dropdown_forms|$1|', '$3' ),
					'post_type' => 'tp_forms',
				),
				/**
				 * Find all nested media files
				 *  (1) normal:         '[vc_icons ... icon="$id"]'
				 *  (2) list icon:      ' vc_list_image="$id"'
				 *  (3) row bg:         ' vc_bg_image="$id"'
				 *  (4) content box bg: ' vc_content_box_image="$id"'
				 *  (5) dynamic:
				 *      - normal:           ::icon|file_picker(_image)|Bild|513::
				 *      - row bg:           ::vc_bg_image|file_picker(_image)|bg_img|516::
				 *      - content box bg:   ::vc_content_box_image|file_picker(_image)|content_box_img|512::
				 *  forms: [vc_form_icon_panel...
				 *      (6) ... icon="508"
				 *      (7) ... icon_hover="509"
				 *      (8) ... icon_select="510"
				 */
				// (1)
				'vc_icons'                => array(
					'search'    => array( '\[vc_icons([^\[\]]*?) icon=\"', '\"' ),
					'replace'   => array( '[vc_icons$1 icon="', '"' ),
					'post_type' => 'attachment',
				),
				// (2) + (3) + (4)
				'vc_image'                => array(
					'search'    => array( ' vc_(list|bg|content_box)_image=\"', '\"' ),
					'replace'   => array( ' vc_$1_image="', '"' ),
					'post_type' => 'attachment',
				),
				// (5)
				'dynamic_image'           => array(
					'search'    => array( '\|file_picker(_image)?\|([^\|]*?)\|', '(::|\")' ),
					'replace'   => array( '|file_picker$1|$2|', '$4' ),
					'post_type' => 'attachment',
					'group'     => 3,
				),
				// (6)
				'icon_panel'              => array(
					'search'    => array( '\[vc_form_icon_panel([^\[\]]*?) icon=\"', '\"' ),
					'replace'   => array( '[vc_form_icon_panel$1 icon="', '"' ),
					'post_type' => 'attachment',
				),
				// (7)
				'icon_panel_hover'        => array(
					'search'    => array( '\[vc_form_icon_panel([^\[\]]*?) icon_hover=\"', '\"' ),
					'replace'   => array( '[vc_form_icon_panel$1 icon_hover="', '"' ),
					'post_type' => 'attachment',
				),
				// (8)
				'icon_panel_select'       => array(
					'search'    => array( '\[vc_form_icon_panel([^\[\]]*?) icon_select=\"', '\"' ),
					'replace'   => array( '[vc_form_icon_panel$1 icon_select="', '"' ),
					'post_type' => 'attachment',
				),
				/**
				 * Find nested popups
				 *  (1) popup button:       [vc_cbutton ...style="popup_open" ... popup="193"...]
				 *  (2) popup contentbox:   [vc_content_box ...vc_content_box_event_type="popup_open" ... vc_content_box_popup="193"...]
				 */
				// (1)
				'popup_button'            => array(
					'search'    => array( '\[vc_cbutton([^\[\]]*?) style=\"popup_open"([^\[\]]*?) popup="', '\"' ),
					'replace'   => array( '[vc_cbutton$1 style="popup_open"$2 popup="', '"' ),
					'post_type' => 'greyd_popup',
					'group'     => 3,
				),
				// (2)
				'popup_contentbox'        => array(
					'search'    => array( '\[vc_content_box([^\[\]]*?) vc_content_box_event_type=\"popup_open"([^\[\]]*?) vc_content_box_popup="', '\"' ),
					'replace'   => array( '[vc_content_box$1 vc_content_box_event_type="popup_open"$2 vc_content_box_popup="', '"' ),
					'post_type' => 'greyd_popup',
					'group'     => 3,
				),
			),
			$post_id,
			$post
		);
	}

	/**
	 * Get all the strings to replace inside post_content.
	 *
	 * @param string $subject   The string to query (usually post content).
	 * @param int    $post_id   The post ID.
	 *
	 * @return string[]
	 *
	 * Default:
	 * @property string upload_url              http://website.de/wp-content/uploads
	 * @property string upload_url_enc          http%3A%2F%2Fwebsite.de%2Fwp-content%2Fuploads
	 * @property string upload_url_enc_twice    http%253A%252F%252Fwebsite.de%252Fwp-content%252Fuploads
	 * @property string site_url                http://website.de/
	 * @property string site_url_enc            http%3A%2F%2Fwebsite.de%2F
	 * @property string site_url_enc_twice      http%253A%252F%252Fwebsite.de%252F
	 */
	public static function regex_nested_strings( $subject, $post_id ) {
		$site_url       = get_site_url();
		$site_url_enc   = urlencode( $site_url );
		$upload_url     = wp_upload_dir()['baseurl'];
		$upload_url_enc = urlencode( $upload_url );

		/**
		 * Replace occurences of different strings
		 *
		 * @filter 'greyd_regex_nested_strings'
		 *
		 * @param string[]  $strings   Array of strings to be replaced.
		 * @param string    $content   The post content.
		 * @param int       $post_id   The post ID.
		 */
		return apply_filters(
			'greyd_regex_nested_strings',
			array(
				'upload_url'           => $upload_url,
				'upload_url_enc'       => $upload_url_enc,
				'upload_url_enc_twice' => urlencode( $upload_url_enc ),
				'site_url'             => $site_url,
				'site_url_enc'         => $site_url_enc,
				'site_url_enc_twice'   => urlencode( $site_url_enc ),
			),
			$subject,
			$post_id
		);
	}

	/**
	 * Get all menus to resolve inside post content
	 *
	 * @param int $post_id    The post ID.
	 *
	 * @return array[] @see regex_nested_posts() for details.
	 */
	public static function regex_nested_menus( $post_id ) {
		/**
		 * Search & replace menus inside post_content.
		 *
		 * @filter 'greyd_regex_nested_menus'
		 *
		 * @param array $patterns   Regex pattern arguments.
		 * @param int   $post_id    The post ID.
		 */
		return apply_filters(
			'greyd_regex_nested_menus',
			array(
				'footer'   => array(
					'search'  => array( '\[vc_footernav(.+?) menu=\"', '\"' ),
					'replace' => array( '[vc_footernav$1 menu="', '"' ),
					'group'   => 2,
				),
				'dropdown' => array(
					'search'  => array( '\[vc_dropdownnav(.+?) menu=\"', '\"' ),
					'replace' => array( '[vc_dropdownnav$1 menu="', '' ),
					'group'   => 2,
				),
			),
			$post_id
		);
	}

	/**
	 * Get the patterns to replace term ids in post_content.
	 *
	 * @param int     $post_id      The WP_Post ID.
	 * @param WP_Post $post         The WP_Post Object
	 *
	 * @return array[] @see regex_nested_posts() for details.
	 */
	public static function regex_nested_terms( $post_id, $post ) {
		/**
		 * @filter 'greyd_regex_nested_terms'
		 *
		 * @param array   $patterns     Regex pattern arguments.
		 * @param int     $post_id      The WP_Post ID.
		 * @param WP_Post $post         The WP_Post Object
		 */
		return (array) apply_filters(
			'greyd_regex_nested_terms',
			array(
				'post_overview_category_filter'  => array(
					'search'  => array( ' ([-_a-z]+)?categoryid=\"', '\"' ),
					'replace' => array( ' $1categoryid="', '"' ),
				),
				'post_overview_tag_filter'       => array(
					'search'  => array( ' ([-_a-z]+)?tagid=\"', '\"' ),
					'replace' => array( ' $1tagid="', '"' ),
				),
				'post_overview_customtax_filter' => array(
					'search'  => array( ' customtax_([-_a-z]+)=\"', '\"' ),
					'replace' => array( ' customtax_$1="', '"' ),
				),
				'post_overview_frontend_filter'  => array(
					'search'  => array( ' _filter=\"([-_a-z]+)" ([-_a-z]+)_initial="', '\"' ),
					'replace' => array( ' _filter="$1" $2_initial="', '"' ),
					'group'   => 3,
				),
			),
			$post_id,
			$post
		);
	}

	/**
	 * Delete all temporary files for import
	 *
	 * usually called after a successfull import
	 */
	public static function delete_tmp_files() {
		$path = self::get_file_path( 'tmp' );
		$dir  = substr( $path, -1 ) === '/' ? substr( $path, 0, -1 ) : $path;

		foreach ( scandir( $dir ) as $item ) {
			if ( $item == '.' || $item == '..' ) {
				continue;
			}
			$file = $dir . DIRECTORY_SEPARATOR . $item;
			if ( ! is_dir( $file ) ) {
				unlink( $file );
			}
		}
		do_action( 'greyd_post_export_log', "\r\n" . sprintf( "All files inside folder '%s' deleted.", $dir ) );
	}

	/**
	 * VC encode
	 */
	public static function vc_encode( $string ) {
		return str_replace( '+', '%20', urlencode( $string ) );
	}

	/**
	 * Toggle debug logs
	 *
	 * @param bool $enable
	 */
	public static function enable_logs( $enable = true ) {
		self::$logs = (bool) $enable;

		add_action( 'greyd_post_export_log', array( '\Greyd\Post_Export', 'log' ), 10, 2 );
	}

	/**
	 * Echo a log
	 */
	public static function log( $message, $var = 'do_not_log' ) {
		if ( self::$logs ) {
			if ( ! empty( $message ) ) {
				echo "\r\n" . $message;
			}
			if ( $var !== 'do_not_log' ) {
				echo "\r\n";
				print_r( $var );
				echo "\r\n";
			}
		}
	}

	/**
	 * Check if there are existing posts in conflict with posts to be impoerted.
	 *
	 * @param WP_Post[] $posts  WP_Posts keyed by ID
	 *
	 * @return WP_Post[]        Returns WP_Posts keyed by the original IDs. Contains
	 *                          post object and full html link object.
	 */
	public static function get_conflicting_posts( $posts ) {

		$conflicts = array();
		foreach ( $posts as $post_id => $post ) {
			if ( $existing_post = self::get_post_by_name_and_type( $post ) ) {
				$conflicts[ $post_id ] = $existing_post;
			}
		}

		/**
		 * @filter 'greyd_import_conflicts'
		 */
		return apply_filters( 'greyd_import_post_conflicts', $conflicts, $posts );
	}

	/**
	 * Get link to post as html object.
	 * Example: <a>Beispiel Seite (Seite)</a>
	 */
	public static function get_post_link_html( $post ) {

		if ( ! is_object( $post ) ) {
			return '';
		}

		if ( $post->post_type === 'attachment' ) {
			$post_title = basename( get_attached_file( $post->ID ) );
			$post_type  = __( "Image/File", 'greyd_hub' );
			$post_url   = wp_get_attachment_url( $post->ID );
		} else {
			$post_title = $post->post_title;
			$post_type  = get_post_type_object( $post->post_type )->labels->singular_name;
			$post_url   = get_edit_post_link( $post );
		}
		$post_title = empty( $post_title ) ? '<i>' . __( "unknown post", 'greyd_hub' ) . '</i>' : $post_title;
		$post_type  = empty( $post_type ) ? '<i>' . __( "unknown post type", 'greyd_hub' ) . '</i>' : $post_type;

		return "<a href='$post_url' target='_blank' title='" . __( "open in new tab", 'greyd_hub' ) . "'>$post_title ($post_type)</a>";
	}

	/**
	 * Get conflicting posts as decoded array to be read and displayed
	 * in the backend 'check-import' overlay-form via backend.js.
	 *
	 * @param WP_Post[] $posts  WP_Posts keyed by ID
	 *
	 * @return string|bool  Decoded string when conflicts found, false otherwise.
	 */
	public static function import_get_conflict_posts_for_backend_form( $posts ) {

		// get conflicting posts
		$conflicts = self::get_conflicting_posts( $posts );

		if ( count( $conflicts ) > 0 ) {
			foreach ( $conflicts as $post_id => $post ) {

				// get the post link to display in the backend
				$conflicts[ $post_id ]->post_link = self::get_post_link_html( $post );
				/**
				 * We add the original ID of the import to the existing post ID.
				 *
				 * In the backend the dropdowns to decide what to do with existing
				 * posts get named by the ID. So their name will be something
				 * like '12-54'
				 *
				 * On the import we have form-data like array('12-54' => 'replace').
				 * We later convert this data via the function
				 * Post_Export::import_get_conflict_actions_from_backend_form()
				 */
				$conflicts[ $post_id ]->ID = $post_id . '-' . $conflicts[ $post_id ]->ID;
			}
			if ( count( $conflicts ) > 4 ) {
				array_unshift(
					$conflicts,
					(object) array(
						'ID'        => 'multioption',
						'post_link' => __( "Multiselect", 'greyd_hub' ),
						'post_type' => '',
					)
				);
			}
			// we don't set keys to keep the order when decoding the array to JS
			return json_encode( array_values( $conflicts ) );
		}
		return false;
	}

	/**
	 * Get all conflicting post actions from the backend 'check-import' overlay-form.
	 *
	 * We have form data from the backend like array( '43-708' => 'skip' )
	 * and we convert this to a proper array that can be used for the function
	 * Post_Export::import_posts() (See function doc for details)
	 *
	 * @return array
	 */
	public static function import_get_conflict_actions_from_backend_form( $conflicts ) {
		$conflict_actions = array();
		foreach ( (array) $conflicts as $ids => $action ) {
			if ( strpos( $ids, '-' ) !== false ) {
				// original format: '43-708' => 'skip'
				$ids                          = explode( '-', $ids );
				$post_id                      = $ids[0];
				$conflict_actions[ $post_id ] = array(
					'post_id' => $ids[1],
					'action'  => $action,
				);
				// new format: 43 => array( 'post_id' => '708', 'action' => 'skip' )
			}
		}
		return $conflict_actions;
	}

	/**
	 * Get the translation tool of this stage.
	 *
	 * @since 1.6 Function supports WPML.
	 */
	public static function get_translation_tool() {
		$tool = null;

		$plugins = get_option( 'active_plugins' );
		if ( in_array( 'sitepress-multilingual-cms/sitepress.php', $plugins ) ) {
			$tool = 'wpml';
		}

		return $tool;
	}

	/**
	 * Get language info of a post
	 *
	 * @param WP_Post $post
	 *
	 * @return array|null
	 */
	public static function get_post_language_info( $post ) {

		$tool = self::get_translation_tool();

		switch ( $tool ) {
			case 'wpml':
				global $wpdb;
				$table_name = "{$wpdb->prefix}icl_translations";
				if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" ) != $table_name ) {
					do_action( 'greyd_post_export_log', "  - database table '{$table_name}' does not exist." );
					break;
				}

				$query  = $wpdb->prepare(
					"SELECT language_code, element_id, trid, source_language_code
					FROM {$table_name}
					WHERE element_id=%d
					AND element_type=%s
					LIMIT 1",
					array( $post->ID, 'post_' . $post->post_type )
				);
				$result = $wpdb->get_results( $query );
				if ( is_array( $result ) && count( $result ) ) {
					return (array) $result[0];
				}
				break;
		}
		return null;
	}

	/**
	 * Get all language codes
	 *
	 * @return array
	 */
	public static function get_languages_codes() {

		$tool = self::get_translation_tool();

		switch ( $tool ) {
			case 'wpml':
				global $wpdb;
				$table_name = "{$wpdb->prefix}icl_languages";
				if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" ) != $table_name ) {
					do_action( 'greyd_post_export_log', "  - database table '{$table_name}' does not exist." );
					break;
				}

				$query  = $wpdb->prepare(
					"SELECT code
					FROM {$table_name}
					WHERE active=%d",
					array( 1 )
				);
				$result = $wpdb->get_results( $query );
				if ( is_array( $result ) && count( $result ) ) {
					return array_map(
						function( $object ) {
							return esc_attr( $object->code );
						},
						$result
					);
				}
				break;
		}

		// default
		return array( self::get_wp_lang() );
	}

	/**
	 * Get language code of the current site (2 chars).
	 */
	public static function get_wp_lang() {
		return explode( '_', get_locale(), 2 )[0];
	}

	/**
	 * Get all translated post IDs
	 *
	 * @param WP_Post $post             The post object (uses ID & post_type)
	 * @param bool    $exclude_current     Whether to include the current $post_id
	 *
	 * @return array
	 */
	public static function get_translated_post_ids( $post, $exclude_current = true ) {
		$return = array();

		$tool = self::get_translation_tool();

		switch ( $tool ) {
			case 'wpml':
				global $wpdb;
				$table_name = "{$wpdb->prefix}icl_translations";
				if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" ) != $table_name ) {
					do_action( 'greyd_post_export_log', "  - database table '{$table_name}' does not exist." );
					break;
				}

				$query  = $wpdb->prepare(
					"SELECT trid
					FROM {$table_name}
					WHERE element_id=%d
					AND element_type=%s
					LIMIT 1",
					array( $post->ID, 'post_' . $post->post_type )
				);
				$result = $wpdb->get_var( $query );

				if ( is_string( $result ) ) {

					$query  = $wpdb->prepare(
						"SELECT language_code, element_id, trid, source_language_code
						FROM {$table_name}
						WHERE trid=%d
						AND element_type=%s",
						array( $result, 'post_' . $post->post_type )
					);
					$result = $wpdb->get_results( $query );

					if ( is_array( $result ) && count( $result ) ) {
						foreach ( $result as $object ) {
							$lang      = isset( $object->language_code ) ? $object->language_code : null;
							$elem_id   = isset( $object->element_id ) ? (int) $object->element_id : null;
							$is_source = empty( $object->source_language_code );

							if ( $lang && $elem_id && ! ( $exclude_current && $post->ID == $elem_id ) ) {
								// log item as first if it's the source
								if ( $is_source ) {
									$return = array_reverse( $return, true );
								}
								$return[ $lang ] = $elem_id;
								if ( $is_source ) {
									$return = array_reverse( $return, true );
								}
							}
						}
					}
				}
				break;
		}

		return $return;
	}

	/**
	 * Switch language if set & supported on current site
	 *
	 * @param WP_Post $post The post object (uses @param array language)
	 *
	 * @return mixed        null:   Post doesn't hold language information.
	 *                      false:  Language is not supported on this site.
	 *                      true:   Language is supported and switched.
	 */
	public static function switch_to_post_lang( $post ) {

		$language  = isset( $post->language ) && ! empty( $post->language ) ? (array) $post->language : array();
		$post_lang = isset( $language['code'] ) ? $language['code'] : null;
		if ( empty( $post_lang ) ) {
			return null;
		}

		// check whether the language is supported on the current site.
		$is_lang_supported = in_array( $post_lang, self::get_languages_codes() );
		if ( ! $is_lang_supported ) {
			do_action( 'greyd_post_export_log', "  - the language '$post_lang' is not part of the supported languages (" . implode( ', ', self::get_languages_codes() ) . ')' );
			return false;
		}

		// we've already switched to this language
		if ( self::$language_code && self::$language_code === $post_lang ) {
			return true;
		}

		$tool = self::get_translation_tool();

		switch ( $tool ) {
			case 'wpml':
				global $sitepress;
				/**
				 * The @var object sitepress is not initialized, when we're trying to call
				 * this function from a different blog without WPML, eg. when the root post
				 * is updated on a single-language site.
				 */
				if ( $sitepress && method_exists( $sitepress, 'switch_lang' ) ) {
					$sitepress->switch_lang( $post_lang );
					self::$language_code = $post_lang;
					do_action( 'greyd_post_export_log', sprintf( "  - switched the language to '%s'.", $post_lang ) );
					return true;
				} else {
					do_action( 'greyd_post_export_log', sprintf( "  - the global \$sitepress object is not initialized, so we could not switch to the language '%s'.", $post_lang ) );
				}
				break;

			default:
				if ( $post_lang === self::get_wp_lang() ) {
					self::$language_code = $post_lang;
					return true;
				}
		}
		return false;
	}

	/**
	 * Register temporary taxonomies to retrieve terms from the database.
	 * We trick WordPress into thinking our taxonomies are valid so we
	 * don't get the 'taxonomy doesn't exist' error.
	 *
	 * @see https://stackoverflow.com/questions/64320279/woocommerce-multisite-network-get-attribute-terms-of-products-from-other-blog
	 *
	 * @param string $post_type
	 *
	 * @return array Array of taxonomy arguments keyed by slug.
	 */
	public static function get_dynamic_taxonomies( $post_type = null ) {

		if (
			! class_exists( '\Greyd\Posttypes\Posttype_Helper' )
			|| ! class_exists( '\Greyd\Posttypes\Dynamic_Posttypes' )
		) {
			return array();
		}

		// register all dynamic post types & taxonomies
		\Greyd\Posttypes\Dynamic_Posttypes::add_dynamic_posttypes();

		// get dynamic taxonomies
		$dynamic_taxonomies = \Greyd\Posttypes\Posttype_Helper::get_dynamic_taxonomies( $post_type );

		// log
		if ( ! empty( $dynamic_taxonomies ) ) {
			do_action( 'greyd_post_export_log', sprintf( "  - found dynamic taxonomies: %s", implode( ', ', array_keys( $dynamic_taxonomies ) ) ) );
		} else {
			do_action( 'greyd_post_export_log', "  - no dynamic taxonomies found." );
		}

		return empty( $dynamic_taxonomies ) ? array() : array_keys( $dynamic_taxonomies );
	}

	/**
	 * Retrieves the terms of the taxonomy that are attached to the post.
	 *
	 * @since 1.1.7
	 * Usually we would use the core function get_the_terms(). However it sometimes returns
	 * terms of completely different taxonomies - without returning an error. To retrieve the
	 * terms directly from the database seems to work more consistent in those cases.
	 *
	 * @since 1.2.7
	 * Update: WPML attaches a lot of filters to the function get_the_terms(). This results
	 * in terms of the wrong language beeing attached to a post export. This function performs
	 * way more consistent in all tests. Therefore it completely replaced it in this class.
	 *
	 * @see get_the_terms()
	 * @see https://developer.wordpress.org/reference/functions/get_the_terms/
	 *
	 * @param int    $post_id    Post ID.
	 * @param string $taxonomy   Taxonomy name.
	 * @return WP_Term[]|null       Array of WP_Term objects on success, null if there are no terms
	 *                              or the post does not exist.
	 */
	public static function get_post_taxonomy_terms( $post_id, string $taxonomy ) {

		if ( ! is_numeric( $post_id ) || ! is_string( $taxonomy ) ) {
			return null;
		}

		global $wpdb;
		$results = $wpdb->get_results(
			"
			SELECT {$wpdb->terms}.term_id, name, slug, term_group, {$wpdb->term_relationships}.term_taxonomy_id, taxonomy, description, parent, count FROM {$wpdb->terms}
				LEFT JOIN {$wpdb->term_relationships} ON
					({$wpdb->terms}.term_id = {$wpdb->term_relationships}.term_taxonomy_id)
				LEFT JOIN {$wpdb->term_taxonomy} ON
					({$wpdb->term_relationships}.term_taxonomy_id = {$wpdb->term_taxonomy}.term_taxonomy_id)
			WHERE {$wpdb->term_relationships}.object_id = {$post_id}
				AND {$wpdb->term_taxonomy}.taxonomy = '{$taxonomy}'
		"
		);
		if ( $results && is_array( $results ) && count( $results ) ) {
			return array_map(
				function( $term ) {
					return new \WP_Term( $term );
				},
				$results
			);
		}
		return null;
	}


	/**
	 * =================================================================
	 *                          DEBUG
	 * =================================================================
	 */

	/**
	 * Enable the debug mode via the URl-parameter 'greyd_export_debug'
	 */
	public function maybe_enable_debug_mode() {

		if ( ! isset( $_GET['greyd_export_debug'] ) ) {
			return;
		}

		self::enable_logs();
		$echo = '';
		ob_start();

		do_action( 'before_greyd_export_debug' );

		if ( isset( $_GET['post'] ) ) {
			$post_id = $_GET['post'];
			$post    = get_post( $post_id );

			// $meta = get_post_meta( $post_id );
			// var_dump( $meta );
			// foreach($meta as $k => $v) {
			// if ( strpos($k, '_oembed_') === 0 ) {
			// delete_post_meta( $post_id, $k );
			// }
			// }

			$response = self::export_post(
				$post_id,
				array(
					'append_nested'  => true,
					'whole_posttype' => true,
					'resolve_menus'  => true,
					'translations'   => true,
				)
			);
			echo "<hr>\r\n\r\n";

			if ( is_object( $response ) && isset( $response->post_content ) ) {
				$response->post_content = esc_attr( $response->post_content );
			}

			var_dump( $response, true );
		} else {
			$posts = \Greyd\Global_Contents\GC_Helper::prepare_global_post_for_import( strval( $_GET['greyd_export_debug'] ) );
			debug( $posts );
		}

		do_action( 'after_greyd_export_debug' );

		$echo = ob_get_contents();
		ob_end_clean();
		if ( ! empty( $echo ) ) {
			echo '<pre>' . $echo . '</pre>';
		}

		wp_die();
	}
}
